/**************************************************************************************
* @file         : VehicleDataClientHandler.h
* @author       : CMC HMI Team
* @addtogroup   : apphmi_navigation
* @brief        :
* @copyright    : (c) 2018-2019 Robert Bosch Car Multimedia GmbH
*                 The reproduction, distribution and utilization of this file as
*                 well as the communication of its contents to others without express
*                 authorization is prohibited. Offenders will be held liable for the
*                 payment of damages. All rights reserved in the event of the grant
*                 of a patent, utility model or design.
**************************************************************************************/


#ifndef VEHICLE_DATACLIENTHANDLER_H
#define VEHICLE_DATACLIENTHANDLER_H


#include "VEHICLE_MAIN_FIProxy.h"
#include "bosch/cm/ai/nissan/hmi/hmidataservice/HmiDataProxy.h"
#include "AppBase/ServiceAvailableIF.h"
#include "Common/Util/HMIModelIncludes.h"
#include "navmid/InfoTypes.h"   // For using enum LanguageCode

#define UNUSED                      0xFF
#define INVALID_LANGUAGE            0
#define SPEED_DEFAULTVALUE          65535
#define SPEEDLOCK_INACTIVE          0
#define SPEEDLOCK_ACTIVE            1
#define KBD_GRC                     121//Country code for Greece defined in languages.dat file
#define KBD_RUS                     147//Country code for Russia defined in languages.dat file
#define KBD_NAM                     166//Country code for US English,Mexican Spanish,Canadian French,AUS English, Brazilian Portuguese defined in languages.dat file
#define KBD_EUR                     167//Country code for other EUR languages except Russian defined in languages.dat file

using namespace ::bosch::cm::ai::nissan::hmi::hmidataservice::HmiData;

struct LanguageISOCodeAndCountryCode
{
   std::string sISOCode;
   uint8 countryCode;

   LanguageISOCodeAndCountryCode() : sISOCode("xx_XXX"), countryCode(UNUSED)
   {}

   LanguageISOCodeAndCountryCode(std::string strISOCode, uint8 iCountryCode): sISOCode(strISOCode), countryCode(iCountryCode)
   {}
};


struct UnitInfoTable
{
   uint8 distanceUnit;
   uint8 fuelUnit;
   uint8 temperatureUnit;

   UnitInfoTable() : distanceUnit(::vehicle_main_fi_types::T_e8_Vehicle_DistanceUnit__Unknown), fuelUnit(::vehicle_main_fi_types::T_e8_Vehicle_FuelConsumptionUnit__Unknown),
      temperatureUnit(::vehicle_main_fi_types::T_e8_Vehicle_TemperatureUnit__Unknown)
   {}
};


class VehicleDataClientHandler: public hmibase::ServiceAvailableIF,
   public StartupSync::PropertyRegistrationIF,
   public ::VEHICLE_MAIN_FI::LanguageCallbackIF,
   public ::VEHICLE_MAIN_FI::LanguageSyncSourceCallbackIF,
   public ::VEHICLE_MAIN_FI::DistanceUnitCallbackIF,
   public ::VEHICLE_MAIN_FI::SpeedCallbackIF,
   public ::VEHICLE_MAIN_FI::SpeedLimitCallbackIF,
   public SpeedLockStatusCallbackIF
{
   public:
      // Member Function Declaration
      VehicleDataClientHandler(const std::string& vehicleMainFiPort, const std::string& hmiDataServicePort);
      virtual ~VehicleDataClientHandler();

      virtual void onAvailable(const ::boost::shared_ptr< ::asf::core::Proxy >& proxy, const ::asf::core::ServiceStateChange& stateChange);
      virtual void onUnavailable(const ::boost::shared_ptr< ::asf::core::Proxy >& proxy, const ::asf::core::ServiceStateChange& stateChange);

      void registerProperties(const ::boost::shared_ptr< asf::core::Proxy >& proxy, \
                              const asf::core::ServiceStateChange& stateChange);
      void deregisterProperties(const boost::shared_ptr< asf::core::Proxy >& proxy, \
                                const asf::core::ServiceStateChange& stateChange);

      void onLanguageStatus(const ::boost::shared_ptr< ::VEHICLE_MAIN_FI::VEHICLE_MAIN_FIProxy >& proxy, \
                            const ::boost::shared_ptr< ::VEHICLE_MAIN_FI::LanguageStatus >& status);
      void onLanguageError(const ::boost::shared_ptr< ::VEHICLE_MAIN_FI::VEHICLE_MAIN_FIProxy >& proxy, \
                           const ::boost::shared_ptr< ::VEHICLE_MAIN_FI::LanguageError >& error);

      void onLanguageSyncSourceError(const ::boost::shared_ptr< ::VEHICLE_MAIN_FI::VEHICLE_MAIN_FIProxy >& proxy, \
                                     const ::boost::shared_ptr< ::VEHICLE_MAIN_FI::LanguageSyncSourceError >& error);
      void onLanguageSyncSourceStatus(const ::boost::shared_ptr< ::VEHICLE_MAIN_FI::VEHICLE_MAIN_FIProxy >& proxy, \
                                      const ::boost::shared_ptr< ::VEHICLE_MAIN_FI::LanguageSyncSourceStatus >& status);

      void onDistanceUnitError(const ::boost::shared_ptr< ::VEHICLE_MAIN_FI::VEHICLE_MAIN_FIProxy >& proxy, \
                               const ::boost::shared_ptr< ::VEHICLE_MAIN_FI::DistanceUnitError >& error);
      void onDistanceUnitStatus(const ::boost::shared_ptr< ::VEHICLE_MAIN_FI::VEHICLE_MAIN_FIProxy >& proxy, \
                                const ::boost::shared_ptr< ::VEHICLE_MAIN_FI::DistanceUnitStatus >& status);
      void onSpeedStatus(const ::boost::shared_ptr< ::VEHICLE_MAIN_FI::VEHICLE_MAIN_FIProxy >& proxy,
                         const ::boost::shared_ptr< ::VEHICLE_MAIN_FI::SpeedStatus >& status);
      void onSpeedError(const ::boost::shared_ptr< ::VEHICLE_MAIN_FI::VEHICLE_MAIN_FIProxy >& proxy,
                        const ::boost::shared_ptr< ::VEHICLE_MAIN_FI::SpeedError >& error);

      void onSpeedLimitError(const ::boost::shared_ptr< ::VEHICLE_MAIN_FI::VEHICLE_MAIN_FIProxy >& proxy,
                             const ::boost::shared_ptr< ::VEHICLE_MAIN_FI::SpeedLimitError >& error);
      void onSpeedLimitStatus(const ::boost::shared_ptr< ::VEHICLE_MAIN_FI::VEHICLE_MAIN_FIProxy >& proxy,
                              const ::boost::shared_ptr< ::VEHICLE_MAIN_FI::SpeedLimitStatus >& status);

      void onSpeedLockStatusUpdate(const ::boost::shared_ptr< HmiDataProxy >& proxy,
                                   const ::boost::shared_ptr< SpeedLockStatusUpdate >& update);
      void onSpeedLockStatusError(const ::boost::shared_ptr< HmiDataProxy >& proxy,
                                  const ::boost::shared_ptr< SpeedLockStatusError >& error);

      //register for GUI ready msg
      //GuiStartupFinishedUpdMsg has to be bind in all the hall component in order to reflect the language changes at the time of bootup/system start up
      COURIER_MSG_MAP_BEGIN(TR_CLASS_APPHMI_NAVIGATION_COURIER_PAYLOAD_MODEL_COMP)
      ON_COURIER_MESSAGE(GuiStartupFinishedUpdMsg)
      COURIER_MSG_MAP_END()

      //GuiStartupFinishedUpdMsg has to be bind in all the hall component in order to reflect the language changes at the time of bootup/system start up
      bool onCourierMessage(const GuiStartupFinishedUpdMsg& msg);

   protected:
      // Member Variables Declaration
      ::boost::shared_ptr< ::VEHICLE_MAIN_FI::VEHICLE_MAIN_FIProxy> _vehicleProxy;
      ::boost::shared_ptr< HmiDataProxy> _hmiDataProxy;

   private:
      // Member Function Declaration
      void readUnitsOnStartUp();
      uint8 readLanguageOnStartUp();
      void updateLanguageTable(std::vector<vehicle_main_fi_types::T_Language_SourceTable> languageSourceTable);
      void sendLanguageUpdate(uint8 languageindex, std::vector<vehicle_main_fi_types::T_Language_SourceTable> languageSourceTable);
      void sendLanguageUpdateToGUI(uint8 languageindex);
      void sendKeyboardCountryCodeUpdate(uint8 languageindex);
      void sendLanguageRelevantForNavigation(uint8 languageIndexSystem, uint8 languageIndexSds, uint8 languageIndexCluster);
      void sendSpeedLockUpdate();
      bool checkSpeedLockState(uint32);
      bool SpeedRestrictionApplicability();
      void setDistanceUnit(uint8 distanceUnit);
      void sendUnitsTableUpdate();
      void currentUnitSettingsStatus(UnitInfoTable unitTable);
      void sendUnitsUpdate();

      bool findISOCodeByLanguageIndex(uint8 languageindex, std::string& matchingISOCode);
      uint8 getLanguageIndexfromKDS();
      navmiddleware::LanguageCode convertLanguageIndexToMiddlewareLanguageIndex(const uint8 languageIndex);

      void initializeLanguageISOCodeOnStartup();
      void initializeLanguageISOCodeMap();
      void initializeLanguageToInternCodeMap();

      // Member Variables Declaration
      std::map< ::vehicle_main_fi_types::T_e8_Language_Code, LanguageISOCodeAndCountryCode> _languageISOCodeMap;
      std::map<vehicle_main_fi_types::T_e8_Language_Code, navmiddleware::LanguageCode> _languageToInternCodeMap;

      uint8 _languageIndex;
      uint8 _countryCode;
      bool _speedLockState;
      bool _hmiDataServiceSpeedLockStatus;
      UnitInfoTable _UnitInfoTable;
};


#endif //VEHICLEDATACLIENTHANDLER_H
