/**************************************************************************************
* @file         : VehicleDataClientHandler.cpp
* @author       : CMC HMI Team
* @addtogroup   : AppHmi_Navigation
* @brief        :
* @copyright    : (c) 2018-2020 Robert Bosch Car Multimedia GmbH
*                 The reproduction, distribution and utilization of this file as
*                 well as the communication of its contents to others without express
*                 authorization is prohibited. Offenders will be held liable for the
*                 payment of damages. All rights reserved in the event of the grant
*                 of a patent, utility model or design.
**************************************************************************************/


#include "AppBase/StartupSync/StartupSync.h"
#include "VehicleDataClientHandler.h"
#include "NaviDataPoolConfig.h"


#ifdef DP_DATAPOOL_ID
#define DP_S_IMPORT_INTERFACE_FI
#include "dp_generic_if.h" //for read kds
#endif

#include "hmi_trace_if.h"

#ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
#define ETG_DEFAULT_TRACE_CLASS           TR_CLASS_APPHMI_NAVIGATION_HALL
#define ETG_I_TRACE_CHANNEL               TR_TTFIS_APPHMI_NAVIGATION
#define ETG_I_TTFIS_CMD_PREFIX            "APPHMI_NAVIGATION_"
#define ETG_I_FILE_PREFIX                 App::Core::VehicleDataClientHandler::
#include "trcGenProj/Header/VehicleDataClientHandler.cpp.trc.h"
#endif // VARIANT_S_FTR_ENABLE_TRC_GEN


using namespace ::asf::core;
using namespace ::VEHICLE_MAIN_FI;
using namespace ::vehicle_main_fi_types;
using namespace navmiddleware;

//
// @Constructor
//
VehicleDataClientHandler::VehicleDataClientHandler(const std::string& vehicleMainFiPort, const std::string& hmiDataServicePort) :
   _vehicleProxy(VEHICLE_MAIN_FIProxy::createProxy(vehicleMainFiPort, *this)),
   _hmiDataProxy(HmiDataProxy::createProxy(hmiDataServicePort, *this)),
   _speedLockState(false),
   _hmiDataServiceSpeedLockStatus(false),
   _languageIndex(UNUSED),
   _countryCode(UNUSED)
{
   ETG_TRACE_USR4(("VehicleDataClientHandler::VehicleDataClientHandler Constructor"));

   if (_vehicleProxy)
   {
      StartupSync::getInstance().registerPropertyRegistrationIF(this, _vehicleProxy->getPortName());
   }

   if (_hmiDataProxy)
   {
      StartupSync::getInstance().registerPropertyRegistrationIF(this, _hmiDataProxy->getPortName());
   }

   //Initialize the ISO Codes in _languageISOCodeMap for corresponding language
   initializeLanguageISOCodeMap();

   //Initialize internally used codes for corresponding FI language
   initializeLanguageToInternCodeMap();
}


//
// @Destructor
//
VehicleDataClientHandler::~VehicleDataClientHandler()
{
   ETG_TRACE_USR4(("VehicleDataClientHandler::VehicleDataClientHandler Destructor"));

   if (!(_languageISOCodeMap.empty()))
   {
      _languageISOCodeMap.clear(); //clear the _languageISOCodeMap
   }
}


void VehicleDataClientHandler::onAvailable(const ::boost::shared_ptr< ::asf::core::Proxy >& proxy, const ::asf::core::ServiceStateChange& stateChange)
{
   ETG_TRACE_USR4(("VehicleDataClientHandler::onAvailable"));
   StartupSync::getInstance().onAvailable(proxy, stateChange);
}


void VehicleDataClientHandler::onUnavailable(const ::boost::shared_ptr< ::asf::core::Proxy >& proxy, const ::asf::core::ServiceStateChange& stateChange)
{
   ETG_TRACE_USR4(("VehicleDataClientHandler::onUnavailable"));
   StartupSync::getInstance().onUnavailable(proxy, stateChange);
}


//
// onAvailable - To Handle Service Availability
// @param[in] proxy
// @param[in] stateChange
// @param[out] None
// @return void
//
void VehicleDataClientHandler::registerProperties(const boost::shared_ptr<asf::core::Proxy>& proxy, \
      const asf::core::ServiceStateChange& stateChange)
{
   ETG_TRACE_USR4(("VehicleDataClientHandler::registerProperties()"));

   if (_vehicleProxy && (proxy == _vehicleProxy))
   {
      //Register for all the required properties update
      //property to Get currently configured language id
      _vehicleProxy->sendLanguageUpReg(*this);
      //property to Set/Get Language Syn source ID. Source can be either HMI or Meter
      _vehicleProxy->sendLanguageSyncSourceUpReg(*this);
      _vehicleProxy->sendDistanceUnitUpReg(*this);
      _vehicleProxy->sendSpeedGet(*this);
      _vehicleProxy->sendSpeedUpReg(*this);
      //property to Get SpeedLimit Status
      _vehicleProxy->sendSpeedLimitUpReg(*this);
   }
   if (_hmiDataProxy && (proxy == _hmiDataProxy))
   {
      _hmiDataProxy->sendSpeedLockStatusGet(*this);
      _hmiDataProxy->sendSpeedLockStatusRegister(*this);
   }
}


//
// onUnavailable - To Handle Service Unavailability
// @param[in] proxy
// @param[in] stateChange
// @param[out] None
// @return void
//
void VehicleDataClientHandler::deregisterProperties(const boost::shared_ptr<asf::core::Proxy>& proxy, \
      const asf::core::ServiceStateChange& stateChange)
{
   ETG_TRACE_USR4(("VehicleDataClientHandler::deregisterProperties()"));

   if (_vehicleProxy && (proxy == _vehicleProxy))
   {
      //Unregister for all the required properties update
      _vehicleProxy->sendRelUpRegAll();
   }
   if (_hmiDataProxy && (proxy == _hmiDataProxy))
   {
      _hmiDataProxy->sendSpeedLockStatusDeregisterAll();
   }
}


//
// onLanguageStatus - ASF framework callback which is triggered on every setLanguage()
// @param[in] proxy
// @param[in] status
// @param[out] None
// @return voidl
//
void VehicleDataClientHandler::onLanguageStatus(const ::boost::shared_ptr< VEHICLE_MAIN_FIProxy >& /*proxy*/,  const ::boost::shared_ptr< LanguageStatus >& status)
{
   ETG_TRACE_USR4(("VehicleDataClientHandler::onLanguageStatus()"));
   std::vector<T_Language_SourceTable> langTable = status->getLangTable();
   uint8 langIndex = status->getLanguage();
   if ((langTable[T_e8_Language_SourceId__Diag].getBHasChanged()) && (langTable[T_e8_Language_SourceId__Diag].getEnLanguage() != INVALID_LANGUAGE))
   {
      //update to GUI
      sendLanguageUpdateToGUI(langTable[T_e8_Language_SourceId__Diag].getEnLanguage());
   }
   else
   {
      sendLanguageUpdate(langIndex, langTable);
      if (_languageIndex != langIndex)
      {
         _languageIndex = langIndex;
         //update to GUI
         sendLanguageUpdateToGUI(langIndex);
      }
   }
   sendKeyboardCountryCodeUpdate(langIndex);
}


//
// onLanguageError - ASF framework callback which is triggered if there is any error in setLanguage()
// @param[in] proxy
// @param[in] error
// @param[out] None
// @return void
//
void VehicleDataClientHandler::onLanguageError(const ::boost::shared_ptr< VEHICLE_MAIN_FIProxy >& /*proxy*/, const ::boost::shared_ptr< LanguageError >& /*error*/)
{
   ETG_TRACE_USR4(("VehicleDataClientHandler::onLanguageError()"));
}


void VehicleDataClientHandler::onSpeedLockStatusUpdate(const ::boost::shared_ptr< HmiDataProxy >& proxy,
      const ::boost::shared_ptr< SpeedLockStatusUpdate >& update)
{
   ETG_TRACE_USR4(("VehicleDataClientHandler::onSpeedLockStatusUpdate()"));

   if (update->hasSpeedLockStatus())
   {
      bool speedLockStatus = update->getSpeedLockStatus();
      ETG_TRACE_USR4(("VehicleDataClientHandler::onSpeedLockStatusUpdate, getSpeedLockStatus() : %d", speedLockStatus));
      if (_hmiDataServiceSpeedLockStatus != speedLockStatus)
      {
         _hmiDataServiceSpeedLockStatus = speedLockStatus;
         POST_MSG((COURIER_MESSAGE_NEW(SpeedLockStatusUpdateMsg)(_speedLockState, _hmiDataServiceSpeedLockStatus)));
      }
   }
}


void VehicleDataClientHandler::onSpeedLockStatusError(const ::boost::shared_ptr< HmiDataProxy >& proxy,
      const ::boost::shared_ptr< SpeedLockStatusError >& error)
{
   ETG_TRACE_USR4(("VehicleDataClientHandler::onSpeedLockStatusError()"));
}


//
// onLanguageSyncSourceStatus - ASF framework callback which is triggered on  setLanguageSyncSource
// @param[in] proxy
// @param[in] status
// @param[out] None
// @return void
//
void VehicleDataClientHandler::onLanguageSyncSourceStatus(const ::boost::shared_ptr< VEHICLE_MAIN_FIProxy >& /*proxy*/, const ::boost::shared_ptr< LanguageSyncSourceStatus >& status)
{
   ETG_TRACE_USR4(("VehicleDataClientHandler::onLanguageSyncSourceStatus()"));

   uint8 synSrcid = status->getLanguage();
   // to be handled
}


//
// onLanguageSyncSourceError - ASF framework callback which is triggered when there is an error in  setLanguageSyncSource
// @param[in] error
// @param[in] status
// @param[out] None
// @return void
//
void VehicleDataClientHandler::onLanguageSyncSourceError(const ::boost::shared_ptr< VEHICLE_MAIN_FIProxy >& /*proxy*/, const ::boost::shared_ptr< LanguageSyncSourceError >& /*error*/)
{
   ETG_TRACE_USR4(("VehicleDataClientHandler::onLanguageSyncSourceError()"));
}


//
// onCourierMessage - To Handle GUI startup message
// when the GUI is ready, request all clients to read language id from dp and then set culture message
// @param[in] GuiStartupFinishedUpdMsg
// @param[out] None
// @return bool
//
bool VehicleDataClientHandler::onCourierMessage(const GuiStartupFinishedUpdMsg& /*msg*/)
{
   ETG_TRACE_USR4(("VehicleDataClientHandler::onCourierMessage(const GuiStartupFinishedUpdMsg()"));

   uint8 languageIndex = 0;
   languageIndex = readLanguageOnStartUp();

   if (languageIndex == INVALID_LANGUAGE)
   {
      languageIndex = getLanguageIndexfromKDS();
   }
   sendLanguageUpdateToGUI(languageIndex);
   sendUnitsTableUpdate();
   sendKeyboardCountryCodeUpdate(languageIndex);
   sendSpeedLockUpdate();

   return false;
}


uint8 VehicleDataClientHandler::readLanguageOnStartUp()
{
   static uint8 languageIndexSystem = NaviDataPoolConfig::getInstance()->getDpVDLanguageIndexSystem();
   ETG_TRACE_USR4(("VehicleDataClientHandler::readLanguageOnStartUp(system %d)", languageIndexSystem));
   initializeLanguageISOCodeOnStartup();
   return languageIndexSystem;
}


void VehicleDataClientHandler::updateLanguageTable(std::vector<vehicle_main_fi_types::T_Language_SourceTable> languageSourceTable)
{
   ETG_TRACE_USR4(("VehicleDataClientHandler::updateLanguageTable()"));

   if (languageSourceTable.size() > 0)
   {
      uint8 languageIndexSystem  = NaviDataPoolConfig::getInstance()->getDpVDLanguageIndexSystem();
      uint8 languageIndexSds     = NaviDataPoolConfig::getInstance()->getDpVDLanguageIndexSds();
      uint8 languageIndexCluster = NaviDataPoolConfig::getInstance()->getDpVDLanguageIndexCluster();

      bool naviLanguageUpdateNeeded = false;
      for (unsigned int i = 0; i < languageSourceTable.size(); ++i)
      {
         switch (languageSourceTable[i].getEnLangSrcId())
         {
            case T_e8_Language_SourceId__System:
            {
               ETG_TRACE_USR4(("VehicleDataClientHandler T_e8_Language_SourceId__System has changed = %x", languageSourceTable[i].getBHasChanged()));

               if (languageSourceTable[i].getBHasChanged())
               {
                  ETG_TRACE_USR4(("VehicleDataClientHandler Language_SourceId__System %d", languageSourceTable[i].getEnLanguage()));

                  if (languageSourceTable[i].getEnLanguage() != languageIndexSystem)
                  {
                     ETG_TRACE_USR4(("VehicleDataClientHandler last stored languageIndexSystem was %d", languageIndexSystem));

                     languageIndexSystem = languageSourceTable[i].getEnLanguage();
                     NaviDataPoolConfig::getInstance()->setDpVDLanguageIndexSystem(languageIndexSystem);
                  }
                  naviLanguageUpdateNeeded = true;
               }
               break;
            }
            case T_e8_Language_SourceId__HMI:
            {
               // not using
               break;
            }
            case T_e8_Language_SourceId__SDS:
            {
               ETG_TRACE_USR4(("VehicleDataClientHandler T_e8_Language_SourceId__SDS has changed = %x", languageSourceTable[i].getBHasChanged()));

               if (languageSourceTable[i].getBHasChanged())
               {
                  ETG_TRACE_USR4(("VehicleDataClientHandler Language_SourceId__SDS %d", languageSourceTable[i].getEnLanguage()));

                  if (languageSourceTable[i].getEnLanguage() != languageIndexSds)
                  {
                     ETG_TRACE_USR4(("VehicleDataClientHandler last stored languageIndexSds was %d", languageIndexSds));

                     languageIndexSds = languageSourceTable[i].getEnLanguage();
                     NaviDataPoolConfig::getInstance()->setDpVDLanguageIndexSds(languageIndexSds);
                  }
                  naviLanguageUpdateNeeded = true;
               }
               break;
            }
            case T_e8_Language_SourceId__Meter:
            {
               ETG_TRACE_USR4(("VehicleDataClientHandler T_e8_Language_SourceId__Meter has changed = %x", languageSourceTable[i].getBHasChanged()));

               if (languageSourceTable[i].getBHasChanged())
               {
                  ETG_TRACE_USR4(("VehicleDataClientHandler Language_SourceId__Meter %d", languageSourceTable[i].getEnLanguage()));

                  if (languageSourceTable[i].getEnLanguage() != languageIndexCluster)
                  {
                     ETG_TRACE_USR4(("VehicleDataClientHandler last stored languageIndexCluster was %d", languageIndexCluster));

                     languageIndexCluster = languageSourceTable[i].getEnLanguage();
                     NaviDataPoolConfig::getInstance()->setDpVDLanguageIndexCluster(languageIndexCluster);
                  }
                  naviLanguageUpdateNeeded = true;
               }
               break;
            }
            default:
            {
               break;
            }
         }
      }

      if (naviLanguageUpdateNeeded)
      {
         sendLanguageRelevantForNavigation(languageIndexSystem, languageIndexSds, languageIndexCluster);
      }
   }
}


//
// sendLanguageUpdate -This function is used to send the language status update to all the registered clients
// @param[in] iLanguageUpdate
// @param[out] None
// @return void
//
void VehicleDataClientHandler::sendLanguageUpdate(uint8 languageIndex, std::vector<T_Language_SourceTable> languageSourceTable)
{
   ETG_TRACE_USR4(("VehicleDataClientHandler::sendLanguageUpdate(%d)", languageIndex));

   //Info: Added languageSourceTable valid check for T_e8_Language_SourceId__System SourceId to avoid the
   //    start up reset because of languageSourceTable data unavailabilty for T_e8_Language_SourceId__System SourceId(-> Because all the applications are reffering the T_e8_Language_SourceId__System SourceId)
   if (languageSourceTable.size() > T_e8_Language_SourceId__System)
   {
      updateLanguageTable(languageSourceTable);
   }
   else
   {
      ETG_TRACE_ERR(("VehicleDataClientHandler:ERROR:languageSourceTable doesn't have the content for T_e8_Language_SourceId__System SourceId"));
   }
}


//
// sendLanguageUpdate - Handle to send the Language update to Framework and Model
// @param[in] isoLangCode
// @param[out] none
// @return bool
//
void VehicleDataClientHandler::sendLanguageUpdateToGUI(uint8 languageIndex)
{
   ETG_TRACE_USR4(("VehicleDataClientHandler::sendLanguageUpdateToGUI(%d)", languageIndex));
   std::map< ::vehicle_main_fi_types::T_e8_Language_Code, LanguageISOCodeAndCountryCode>::iterator itr = \
         _languageISOCodeMap.find((T_e8_Language_Code)languageIndex);
   if (itr != _languageISOCodeMap.end())
   {
      //send Language update to Framework to change all static text
      //COURIER_MESSAGE_NEW(Courier::SetCultureReqMsg)(Courier::ItemId(_languageISOCodeMap[T_e8_Language_Code(languageIndex)].sISOCode.c_str()))->Post();
   }
}


void VehicleDataClientHandler::sendKeyboardCountryCodeUpdate(uint8 languageindex)
{
   ETG_TRACE_USR4(("VehicleDataClientHandler::sendKeyboardCountryCodeUpdate(%d)", languageindex));

   std::map< ::vehicle_main_fi_types::T_e8_Language_Code, LanguageISOCodeAndCountryCode>::iterator itr = \
         _languageISOCodeMap.find((T_e8_Language_Code)languageindex);

   if (itr != _languageISOCodeMap.end())
   {
      ETG_TRACE_USR4(("VehicleDataClientHandler :: country code :%d", itr->second.countryCode));

      if (_countryCode != itr->second.countryCode)
      {
         ETG_TRACE_USR4(("VehicleDataClientHandler :: Country code modified"));
         _countryCode = ((itr->second.countryCode != UNUSED) ? itr->second.countryCode : KBD_EUR);
         POST_MSG((COURIER_MESSAGE_NEW(CountryCodeUpdateMsg)(_countryCode)));
      }
   }
}


void VehicleDataClientHandler::sendLanguageRelevantForNavigation(uint8 languageIndexSystem, uint8 languageIndexSds, uint8 languageIndexCluster)
{
   ETG_TRACE_USR4(("VehicleDataClientHandler::sendLanguageRelevantForNavigation(system %d, sds %d, cluster %d)", languageIndexSystem, languageIndexSds, languageIndexCluster));

   std::string matchingLanguageISOCodeSystem("xx_XX");
   std::string matchingLanguageISOCodeSds("xx_XX");
   std::string matchingLanguageISOCodeCluster("xx_XX");

   findISOCodeByLanguageIndex(languageIndexSystem,  matchingLanguageISOCodeSystem);
   findISOCodeByLanguageIndex(languageIndexSds,     matchingLanguageISOCodeSds);
   findISOCodeByLanguageIndex(languageIndexCluster, matchingLanguageISOCodeCluster);

   navmiddleware::LanguageCode languageNaviRelevantFromSystem  = convertLanguageIndexToMiddlewareLanguageIndex(languageIndexSystem);
   navmiddleware::LanguageCode languageNaviRelevantFromSds     = convertLanguageIndexToMiddlewareLanguageIndex(languageIndexSds);
   navmiddleware::LanguageCode languageNaviRelevantFromCluster = convertLanguageIndexToMiddlewareLanguageIndex(languageIndexCluster);

   POST_MSG((COURIER_MESSAGE_NEW(LanguageRelevantForNavigationUpdateMsg)
             (languageNaviRelevantFromSystem, matchingLanguageISOCodeSystem.c_str(),
              languageNaviRelevantFromSds, matchingLanguageISOCodeSds.c_str(),
              languageNaviRelevantFromCluster, matchingLanguageISOCodeCluster.c_str())));
}


//
// findISOCodeByLanguageIndex - find the matching language ISO code by the input of the language index
// @param[in] languageindex
// @param[out] matchingISOCode
// @return bool
//
bool VehicleDataClientHandler::findISOCodeByLanguageIndex(uint8 languageindex, std::string& matchingISOCode)
{
   ETG_TRACE_USR4(("VehicleDataClientHandler::findISOCodeByLanguageIndex(), Language index: %d", languageindex));

   bool bLanguageIndexIsInList = false;

   std::map< ::vehicle_main_fi_types::T_e8_Language_Code, LanguageISOCodeAndCountryCode>::iterator itr = \
         _languageISOCodeMap.find((T_e8_Language_Code)languageindex);
   if (itr != _languageISOCodeMap.end())
   {
      bLanguageIndexIsInList = true;
      matchingISOCode = _languageISOCodeMap[T_e8_Language_Code(languageindex)].sISOCode.c_str();
      ETG_TRACE_USR4(("VehicleDataClientHandler::findISOCodeByLanguageIndex(), ISO code : %s", matchingISOCode.c_str()));
   }
   return bLanguageIndexIsInList;
}


//
// getLanguageIndexfromKDS - To read language settings from KDS
// @param[in] None
// @param[out] None
// @return uint8 u8Language
//
uint8 VehicleDataClientHandler::getLanguageIndexfromKDS()
{
   ETG_TRACE_USR4(("VehicleDataClientHandler::getLanguageIndexfromKDS()"));

   dp_tclKdsSystemConfiguration1 languageConfig;
   uint8 u8Language = INVALID_LANGUAGE;
   uint8 kdsError = 0;
   kdsError = languageConfig.u8GetDefaultLanguage(u8Language);
   if (kdsError != DP_U8_ELEM_STATUS_VALID)
   {
      u8Language = INVALID_LANGUAGE;
   }
   return u8Language;
}


navmiddleware::LanguageCode VehicleDataClientHandler::convertLanguageIndexToMiddlewareLanguageIndex(const uint8 languageIndex)
{
   ETG_TRACE_USR4(("VehicleDataClientHandler::convertLanguageIndexToMiddlewareLanguageIndex(%d)", languageIndex));

   navmiddleware::LanguageCode internalLanguageCode = LANGUAGE_CODE__UNSUPPORTED;

   std::map< ::vehicle_main_fi_types::T_e8_Language_Code, navmiddleware::LanguageCode>::iterator itr = \
         _languageToInternCodeMap.find((::vehicle_main_fi_types::T_e8_Language_Code)languageIndex);

   if (itr != _languageToInternCodeMap.end())
   {
      internalLanguageCode = itr->second;
   }

   return internalLanguageCode;
}


void VehicleDataClientHandler::initializeLanguageISOCodeOnStartup()
{
   ETG_TRACE_USR4(("VehicleDataClientHandler::initializeLanguageISOCodeOnStartup()"));

   uint8 languageIndexSystem  = NaviDataPoolConfig::getInstance()->getDpVDLanguageIndexSystem();
   uint8 languageIndexSds     = NaviDataPoolConfig::getInstance()->getDpVDLanguageIndexSds();
   uint8 languageIndexCluster = NaviDataPoolConfig::getInstance()->getDpVDLanguageIndexCluster();

   ETG_TRACE_USR4(("VehicleDataClientHandler::initializeLanguageISOCodeOnStartup(system %d, sds %d, cluster %d)", languageIndexSystem, languageIndexSds, languageIndexCluster));

   sendLanguageRelevantForNavigation(languageIndexSystem, languageIndexSds, languageIndexCluster);
}


//
// initializeLanguageISOCodeMap - Handle to Initialize the ISO Codes in _languageISOCodeMap
//                            for corresponding language
// @param[in] None
// @param[out] None
// @return void
//
void VehicleDataClientHandler::initializeLanguageISOCodeMap()
{
   ETG_TRACE_USR4(("VehicleDataClientHandler::initializeLanguageISOCodeMap()"));

   _languageISOCodeMap[T_e8_Language_Code__Unknown]                                                = LanguageISOCodeAndCountryCode(std::string("xx_XX"), UNUSED);
   _languageISOCodeMap[T_e8_Language_Code__Albanese]                                               = LanguageISOCodeAndCountryCode(std::string("sq_AL"), UNUSED);
   _languageISOCodeMap[T_e8_Language_Code__Arabic]                                                 = LanguageISOCodeAndCountryCode(std::string("ar_AR"), UNUSED);
   _languageISOCodeMap[T_e8_Language_Code__Bulgarian]                                              = LanguageISOCodeAndCountryCode(std::string("bg_BG"), UNUSED);
   _languageISOCodeMap[T_e8_Language_Code__Chinese_Cantonese_Simplified_Chinese_character]         = LanguageISOCodeAndCountryCode(std::string("zh_CN"), UNUSED);
   _languageISOCodeMap[T_e8_Language_Code__Chinese_Cantonese_Traditional_Chinese_character]        = LanguageISOCodeAndCountryCode(std::string("zh_HK"), UNUSED);
   _languageISOCodeMap[T_e8_Language_Code__Chinese_Mandarin_Simplified_Chinese_character]          = LanguageISOCodeAndCountryCode(std::string("zh_CN"), UNUSED);
   _languageISOCodeMap[T_e8_Language_Code__Croatian]                                               = LanguageISOCodeAndCountryCode(std::string("hr_HR"), UNUSED);
   _languageISOCodeMap[T_e8_Language_Code__Czech]                                                  = LanguageISOCodeAndCountryCode(std::string("cs_CZ"), KBD_EUR);
   _languageISOCodeMap[T_e8_Language_Code__Danish]                                                 = LanguageISOCodeAndCountryCode(std::string("da_DK"), KBD_EUR);
   _languageISOCodeMap[T_e8_Language_Code__Dutch]                                                  = LanguageISOCodeAndCountryCode(std::string("nl_NL"), KBD_EUR);
   _languageISOCodeMap[T_e8_Language_Code__English_Australian]                                     = LanguageISOCodeAndCountryCode(std::string("en_GB"), KBD_NAM); // Fix for [NCG3D-21694]
   _languageISOCodeMap[T_e8_Language_Code__English_Canadian]                                       = LanguageISOCodeAndCountryCode(std::string("en_CA"), UNUSED);
   _languageISOCodeMap[T_e8_Language_Code__English_India]                                          = LanguageISOCodeAndCountryCode(std::string("en_GB"), UNUSED);
   _languageISOCodeMap[T_e8_Language_Code__English_UK]                                             = LanguageISOCodeAndCountryCode(std::string("en_GB"), KBD_EUR);
   _languageISOCodeMap[T_e8_Language_Code__English_US]                                             = LanguageISOCodeAndCountryCode(std::string("en_US"), KBD_NAM);
   _languageISOCodeMap[T_e8_Language_Code__English_US_for_JPN]                                     = LanguageISOCodeAndCountryCode(std::string("en_US"), UNUSED);
   _languageISOCodeMap[T_e8_Language_Code__English_US_for_PRC]                                     = LanguageISOCodeAndCountryCode(std::string("en_GB"), UNUSED);
   _languageISOCodeMap[T_e8_Language_Code__Estonian]                                               = LanguageISOCodeAndCountryCode(std::string("et_EE"), UNUSED);
   _languageISOCodeMap[T_e8_Language_Code__Finnish]                                                = LanguageISOCodeAndCountryCode(std::string("fi_FI"), UNUSED);
   _languageISOCodeMap[T_e8_Language_Code__Flemish]                                                = LanguageISOCodeAndCountryCode(std::string("nl_NL"), UNUSED);
   _languageISOCodeMap[T_e8_Language_Code__French]                                                 = LanguageISOCodeAndCountryCode(std::string("fr_FR"), KBD_EUR);
   _languageISOCodeMap[T_e8_Language_Code__French_Canadian]                                        = LanguageISOCodeAndCountryCode(std::string("fr_CA"), KBD_EUR);
   _languageISOCodeMap[T_e8_Language_Code__German]                                                 = LanguageISOCodeAndCountryCode(std::string("de_DE"), KBD_EUR);
   _languageISOCodeMap[T_e8_Language_Code__Greek]                                                  = LanguageISOCodeAndCountryCode(std::string("el_GR"), KBD_GRC);
   _languageISOCodeMap[T_e8_Language_Code__Hebrew]                                                 = LanguageISOCodeAndCountryCode(std::string("he_IL"), UNUSED);
   _languageISOCodeMap[T_e8_Language_Code__Hindi]                                                  = LanguageISOCodeAndCountryCode(std::string("hi_IN"), UNUSED);
   _languageISOCodeMap[T_e8_Language_Code__Hungarian]                                              = LanguageISOCodeAndCountryCode(std::string("hu_HU"), KBD_EUR);
   _languageISOCodeMap[T_e8_Language_Code__Indonesian]                                             = LanguageISOCodeAndCountryCode(std::string("id_ID"), UNUSED);
   _languageISOCodeMap[T_e8_Language_Code__Italian]                                                = LanguageISOCodeAndCountryCode(std::string("it_IT"), KBD_EUR);
   _languageISOCodeMap[T_e8_Language_Code__Japanese]                                               = LanguageISOCodeAndCountryCode(std::string("ja_JA"), UNUSED);
   _languageISOCodeMap[T_e8_Language_Code__Korean]                                                 = LanguageISOCodeAndCountryCode(std::string("ko_KO"), UNUSED);
   _languageISOCodeMap[T_e8_Language_Code__Latvian]                                                = LanguageISOCodeAndCountryCode(std::string("lv_LV"), UNUSED);
   _languageISOCodeMap[T_e8_Language_Code__Lithuanian]                                             = LanguageISOCodeAndCountryCode(std::string("lt_LT"), UNUSED);
   _languageISOCodeMap[T_e8_Language_Code__Malay]                                                  = LanguageISOCodeAndCountryCode(std::string("xx_XX"), UNUSED);
   _languageISOCodeMap[T_e8_Language_Code__Norwegian]                                              = LanguageISOCodeAndCountryCode(std::string("nb_NO"), KBD_EUR);
   _languageISOCodeMap[T_e8_Language_Code__Persian_Farsi_Iranian]                                  = LanguageISOCodeAndCountryCode(std::string("fa_FA"), UNUSED);
   _languageISOCodeMap[T_e8_Language_Code__Philippines]                                            = LanguageISOCodeAndCountryCode(std::string("xx_XX"), UNUSED);
   _languageISOCodeMap[T_e8_Language_Code__Polish]                                                 = LanguageISOCodeAndCountryCode(std::string("pl_PL"), KBD_EUR);
   _languageISOCodeMap[T_e8_Language_Code__Portuguese]                                             = LanguageISOCodeAndCountryCode(std::string("pt_PT"), KBD_EUR);
   _languageISOCodeMap[T_e8_Language_Code__Portuguese_Brazilian]                                   = LanguageISOCodeAndCountryCode(std::string("pt_BR"), KBD_NAM);
   _languageISOCodeMap[T_e8_Language_Code__Romanian]                                               = LanguageISOCodeAndCountryCode(std::string("ro_RO"), UNUSED);
   _languageISOCodeMap[T_e8_Language_Code__Russian]                                                = LanguageISOCodeAndCountryCode(std::string("ru_RU"), KBD_RUS);
   _languageISOCodeMap[T_e8_Language_Code__Serbian]                                                = LanguageISOCodeAndCountryCode(std::string("sr_RS"), UNUSED);
   _languageISOCodeMap[T_e8_Language_Code__Slovakian]                                              = LanguageISOCodeAndCountryCode(std::string("sk_SK"), UNUSED);
   _languageISOCodeMap[T_e8_Language_Code__Slovenian]                                              = LanguageISOCodeAndCountryCode(std::string("sl_SL"), UNUSED);
   _languageISOCodeMap[T_e8_Language_Code__Spanish]                                                = LanguageISOCodeAndCountryCode(std::string("es_ES"), KBD_EUR);
   _languageISOCodeMap[T_e8_Language_Code__Spanish_Latin_American]                                 = LanguageISOCodeAndCountryCode(std::string("es_MX"), KBD_EUR);
   _languageISOCodeMap[T_e8_Language_Code__Spanish_Mexican]                                        = LanguageISOCodeAndCountryCode(std::string("es_MX"), KBD_EUR);
   _languageISOCodeMap[T_e8_Language_Code__Swedish]                                                = LanguageISOCodeAndCountryCode(std::string("sv_SE"), KBD_EUR);
   _languageISOCodeMap[T_e8_Language_Code__Taiwanese]                                              = LanguageISOCodeAndCountryCode(std::string("zh_TW"), UNUSED);
   _languageISOCodeMap[T_e8_Language_Code__Thai]                                                   = LanguageISOCodeAndCountryCode(std::string("th_TH"), UNUSED);
   _languageISOCodeMap[T_e8_Language_Code__Turkish]                                                = LanguageISOCodeAndCountryCode(std::string("tr_TR"), UNUSED);
   _languageISOCodeMap[T_e8_Language_Code__Ukrainian]                                              = LanguageISOCodeAndCountryCode(std::string("uk_UK"), UNUSED);
   _languageISOCodeMap[T_e8_Language_Code__Vietnamese]                                             = LanguageISOCodeAndCountryCode(std::string("vi_VN"), UNUSED);
   _languageISOCodeMap[T_e8_Language_Code__UnSupported]                                            = LanguageISOCodeAndCountryCode(std::string("xx_XX"), UNUSED);
   _languageISOCodeMap[T_e8_Language_Code__Unused]                                                 = LanguageISOCodeAndCountryCode(std::string("xx_XX"), UNUSED);
}


void VehicleDataClientHandler::initializeLanguageToInternCodeMap()
{
   ETG_TRACE_USR4(("VehicleDataClientHandler::initializeLanguageToInternCodeMap()"));

   _languageToInternCodeMap[T_e8_Language_Code__Unknown] = LANGUAGE_CODE__UNKNOWN;
   _languageToInternCodeMap[T_e8_Language_Code__Albanese] = LANGUAGE_CODE__ALBANESE;
   _languageToInternCodeMap[T_e8_Language_Code__Arabic] = LANGUAGE_CODE__ARABIC;
   _languageToInternCodeMap[T_e8_Language_Code__Bulgarian] = LANGUAGE_CODE__BULGARIEN;
   _languageToInternCodeMap[T_e8_Language_Code__Chinese_Cantonese_Simplified_Chinese_character] = LANGUAGE_CODE__CHINESE_CANTONESE_SIMPLIFIED_CHINESE_CHARACTER;
   _languageToInternCodeMap[T_e8_Language_Code__Chinese_Cantonese_Traditional_Chinese_character] = LANGUAGE_CODE__CHINESE_CANTONESE_TRADITIONAL_CHINESE_CHARACTER;
   _languageToInternCodeMap[T_e8_Language_Code__Chinese_Mandarin_Simplified_Chinese_character] = LANGUAGE_CODE__CHINESE_MANDARIN_SIMPLIFIED_CHINESE_CHARACTER;
   _languageToInternCodeMap[T_e8_Language_Code__Croatian] = LANGUAGE_CODE__CROATIAN;
   _languageToInternCodeMap[T_e8_Language_Code__Czech] = LANGUAGE_CODE__CZECH;
   _languageToInternCodeMap[T_e8_Language_Code__Danish] = LANGUAGE_CODE__DANISH;
   _languageToInternCodeMap[T_e8_Language_Code__Dutch] = LANGUAGE_CODE__DUTCH;
   _languageToInternCodeMap[T_e8_Language_Code__English_Australian] = LANGUAGE_CODE__ENGLISH_AUSTRALIAN;
   _languageToInternCodeMap[T_e8_Language_Code__English_Canadian] = LANGUAGE_CODE__ENGLISH_CANADIAN;
   _languageToInternCodeMap[T_e8_Language_Code__English_India] = LANGUAGE_CODE__ENGLISH_INDIA;
   _languageToInternCodeMap[T_e8_Language_Code__English_UK] = LANGUAGE_CODE__ENGLISH_UK;
   _languageToInternCodeMap[T_e8_Language_Code__English_US] = LANGUAGE_CODE__ENGLISH_US;
   _languageToInternCodeMap[T_e8_Language_Code__English_US_for_JPN] = LANGUAGE_CODE__ENGLISH_US_FOR_JPN;
   _languageToInternCodeMap[T_e8_Language_Code__English_US_for_PRC] = LANGUAGE_CODE__ENGLISH_US_FOR_PRC;
   _languageToInternCodeMap[T_e8_Language_Code__Estonian] = LANGUAGE_CODE__ESTONIAN;
   _languageToInternCodeMap[T_e8_Language_Code__Finnish] = LANGUAGE_CODE__FINNISH;
   _languageToInternCodeMap[T_e8_Language_Code__Flemish] = LANGUAGE_CODE__FLEMISH;
   _languageToInternCodeMap[T_e8_Language_Code__French] = LANGUAGE_CODE__FRENCH;
   _languageToInternCodeMap[T_e8_Language_Code__French_Canadian] = LANGUAGE_CODE__FRENCH_CANADIAN;
   _languageToInternCodeMap[T_e8_Language_Code__German] = LANGUAGE_CODE__GERMAN;
   _languageToInternCodeMap[T_e8_Language_Code__Greek] = LANGUAGE_CODE__GREEK;
   _languageToInternCodeMap[T_e8_Language_Code__Hebrew] = LANGUAGE_CODE__HEBREW;
   _languageToInternCodeMap[T_e8_Language_Code__Hindi] = LANGUAGE_CODE__HINDI;
   _languageToInternCodeMap[T_e8_Language_Code__Hungarian] = LANGUAGE_CODE__HUNGARIAN;
   _languageToInternCodeMap[T_e8_Language_Code__Indonesian] = LANGUAGE_CODE__INDONESIAN;
   _languageToInternCodeMap[T_e8_Language_Code__Italian] = LANGUAGE_CODE__ITALIAN;
   _languageToInternCodeMap[T_e8_Language_Code__Japanese] = LANGUAGE_CODE__JAPANESE;
   _languageToInternCodeMap[T_e8_Language_Code__Korean] = LANGUAGE_CODE__KOREAN;
   _languageToInternCodeMap[T_e8_Language_Code__Latvian] = LANGUAGE_CODE__LATVIAN;
   _languageToInternCodeMap[T_e8_Language_Code__Lithuanian] = LANGUAGE_CODE__LITHUANIAN;
   _languageToInternCodeMap[T_e8_Language_Code__Malay] = LANGUAGE_CODE__MALAY;
   _languageToInternCodeMap[T_e8_Language_Code__Norwegian] = LANGUAGE_CODE__NORWEGIAN;
   _languageToInternCodeMap[T_e8_Language_Code__Persian_Farsi_Iranian] = LANGUAGE_CODE__PERSIAN_FARSI_IRANIAN;
   _languageToInternCodeMap[T_e8_Language_Code__Philippines] = LANGUAGE_CODE__PHILIPPINES;
   _languageToInternCodeMap[T_e8_Language_Code__Polish] = LANGUAGE_CODE__POLISH;
   _languageToInternCodeMap[T_e8_Language_Code__Portuguese] = LANGUAGE_CODE__PORTUGUESE;
   _languageToInternCodeMap[T_e8_Language_Code__Portuguese_Brazilian] = LANGUAGE_CODE__PORTUGUESE_BRAZILIAN;
   _languageToInternCodeMap[T_e8_Language_Code__Romanian] = LANGUAGE_CODE__ROMANIAN;
   _languageToInternCodeMap[T_e8_Language_Code__Russian] = LANGUAGE_CODE__RUSSIAN;
   _languageToInternCodeMap[T_e8_Language_Code__Serbian] = LANGUAGE_CODE__SERBIAN;
   _languageToInternCodeMap[T_e8_Language_Code__Slovakian] = LANGUAGE_CODE__SLOVAKIAN;
   _languageToInternCodeMap[T_e8_Language_Code__Slovenian] = LANGUAGE_CODE__SLOVENIAN;
   _languageToInternCodeMap[T_e8_Language_Code__Spanish] = LANGUAGE_CODE__SPANISH;
   _languageToInternCodeMap[T_e8_Language_Code__Spanish_Latin_American] = LANGUAGE_CODE__SPANISH_LATIN_AMERICAN;
   _languageToInternCodeMap[T_e8_Language_Code__Spanish_Mexican] = LANGUAGE_CODE__SPANISH_MEXICAN;
   _languageToInternCodeMap[T_e8_Language_Code__Swedish] = LANGUAGE_CODE__SWEDISH;
   _languageToInternCodeMap[T_e8_Language_Code__Taiwanese] = LANGUAGE_CODE__TAIWANESE;
   _languageToInternCodeMap[T_e8_Language_Code__Thai] = LANGUAGE_CODE__THAI;
   _languageToInternCodeMap[T_e8_Language_Code__Turkish] = LANGUAGE_CODE__TURKISH;
   _languageToInternCodeMap[T_e8_Language_Code__Ukrainian] = LANGUAGE_CODE__UKRAINIAN;
   _languageToInternCodeMap[T_e8_Language_Code__Vietnamese] = LANGUAGE_CODE__VIETNAMESE;
   _languageToInternCodeMap[T_e8_Language_Code__UnSupported] = LANGUAGE_CODE__UNSUPPORTED;
   _languageToInternCodeMap[T_e8_Language_Code__Unused] = LANGUAGE_CODE__UNUSED;
}


void VehicleDataClientHandler::onDistanceUnitStatus(const ::boost::shared_ptr< VEHICLE_MAIN_FIProxy >& proxy, const ::boost::shared_ptr< DistanceUnitStatus >& status)
{
   if (_vehicleProxy && (proxy == _vehicleProxy))
   {
      _UnitInfoTable.distanceUnit = status->getE8DistanceUnit();;
      sendUnitsUpdate();
   }
}


void VehicleDataClientHandler::onDistanceUnitError(const ::boost::shared_ptr< VEHICLE_MAIN_FIProxy >& /*proxy*/, const ::boost::shared_ptr< DistanceUnitError >& /*error*/)
{
}


void VehicleDataClientHandler::setDistanceUnit(uint8 distanceUnit)
{
   if (_vehicleProxy)
   {
      _vehicleProxy->sendDistanceUnitSet(*this, (T_e8_Vehicle_DistanceUnit)distanceUnit);
   }
}


void VehicleDataClientHandler::sendUnitsUpdate()
{
   //Update status to all registered(external) client(s)
   currentUnitSettingsStatus(_UnitInfoTable);
}


void VehicleDataClientHandler::currentUnitSettingsStatus(UnitInfoTable unitTable)
{
   ETG_TRACE_USR4(("VDVehicleClientHandler:currentUnitSettingsStatus: DistanceUnit=%d, FuelUnit=%d, TempUnit=%d",
                   unitTable.distanceUnit, unitTable.fuelUnit, unitTable.temperatureUnit));
   // send UnitSettingsStatus update to model for any add ons
   POST_MSG((COURIER_MESSAGE_NEW(UnitSettingsStatusUpdateMsg)(unitTable.distanceUnit, unitTable.fuelUnit, unitTable.temperatureUnit)));
}


/*
* sendUnitsTableUpdate - Handle to send the Units table update to all the registered clients
* @param[in] none
* @param[out] none
* @return void
*/
void VehicleDataClientHandler::sendUnitsTableUpdate()
{
   ETG_TRACE_USR4(("VehicleDataHandler::sendUnitsTableUpdate"));
   readUnitsOnStartUp();
}


void VehicleDataClientHandler::readUnitsOnStartUp()
{
   ETG_TRACE_USR4(("VDVehicleClientHandler::readUnitsOnStartUp()"));
   dp_tclKdsSystemConfiguration1 unitConfigSystem;
   dp_tclKdsVehicleInformation unitConfigVehicle;
   UnitInfoTable unitTable;
   unitConfigSystem.u8GetDefaultDistanceUnits(unitTable.distanceUnit);
   unitConfigSystem.u8GetDefaultTemperatureUnit(unitTable.temperatureUnit);
   unitConfigVehicle.u8GetFuelConsumptionInfo(unitTable.fuelUnit);
   POST_MSG((COURIER_MESSAGE_NEW(UnitSettingsStatusUpdateMsg)(unitTable.distanceUnit, unitTable.fuelUnit, unitTable.temperatureUnit)));
   ETG_TRACE_USR4(("VDVehicleClientHandler::readUnitsOnStartUp() :: distanceUnit = %d, fuelUnit = %d, temperatureUnit= %d",
                   unitTable.distanceUnit, unitTable.fuelUnit, unitTable.temperatureUnit));
}


void VehicleDataClientHandler::onSpeedLimitError(const ::boost::shared_ptr< VEHICLE_MAIN_FIProxy >& /*proxy*/, const ::boost::shared_ptr< SpeedLimitError >& /*error*/)
{
}


void VehicleDataClientHandler::onSpeedLimitStatus(const ::boost::shared_ptr< VEHICLE_MAIN_FIProxy >& proxy, const ::boost::shared_ptr< SpeedLimitStatus >& status)
{
   if (_vehicleProxy && (proxy == _vehicleProxy))
   {
      ETG_TRACE_USR4(("VehicleDataHandler::onSpeedLimitStatus(SpeedLimit1 %d, SpeedLimit2 %d, Display %d, DisplayUnit %d, OverSpeed %d, FreeSpeed %d)",
                      status->getSpeedLimit1(), status->getSpeedLimit2(), status->getDisplay(), status->getDisplayUnit(), status->getOverSpeed(), status->getFreeSpeed()));

      // Update status if CAN data valid
      if (status->getDisplay() != 255)
      {
         POST_MSG((COURIER_MESSAGE_NEW(SpeedLimitStatusUpdateMsg)(status->getSpeedLimit1(), status->getDisplay(), static_cast< enSpeedUnit >(status->getDisplayUnit()), status->getFreeSpeed(), status->getOverSpeed())));
      }
   }
}


void VehicleDataClientHandler::onSpeedError(const ::boost::shared_ptr< VEHICLE_MAIN_FIProxy >& /*proxy*/, const ::boost::shared_ptr< SpeedError >& /*error*/)
{
   ETG_TRACE_ERR(("VehicleDataClientHandler::onSpeedError()"));
}


void VehicleDataClientHandler::onSpeedStatus(const ::boost::shared_ptr< VEHICLE_MAIN_FIProxy >& proxy, const ::boost::shared_ptr< SpeedStatus >& status)
{
   if (_vehicleProxy && (proxy == _vehicleProxy))
   {
      if (status->getSpeedValue() != SPEED_DEFAULTVALUE)
      {
         if (SpeedRestrictionApplicability() == true)
         {
            bool speedLockState = checkSpeedLockState(status->getSpeedValue());
            ETG_TRACE_USR4(("VehicleDataClientHandler::onSpeedStatus speed value : %d speedLockState:%d SpeedLockState from Data Service : %d", status->getSpeedValue(), speedLockState, _hmiDataServiceSpeedLockStatus));
            if (_speedLockState != speedLockState)
            {
               _speedLockState = speedLockState;
               POST_MSG((COURIER_MESSAGE_NEW(SpeedLockStatusUpdateMsg)(_speedLockState, _hmiDataServiceSpeedLockStatus)));
            }
         }
      }
   }
}


bool VehicleDataClientHandler:: SpeedRestrictionApplicability()
{
   uint8 OEMType = 0xFF;
   uint8 SpeedLockRestrictions = 0xFF;
   if ((DP_S32_NO_ERR != DP_s32GetConfigItem("SystemConfiguration1", "SpeedLockRestrictions", &SpeedLockRestrictions, 1)))
   {
      ETG_TRACE_ERR(("VehicleDataClientHandler:readSSpeedLockRestrictionsFromKDS Error"));
      return false;
   }
   ETG_TRACE_USR4(("VehicleDataClientHandler:SpeedLockRestrictions Applicability = %d", SpeedLockRestrictions));
   if (SpeedLockRestrictions == 1)
   {
      return true;
   }
   return true;
}


bool VehicleDataClientHandler:: checkSpeedLockState(uint32 vehiclespeed)
{
   uint32 thresholdSpeed = 0;
   uint8 OEMType = 0xFF;
   uint8 thresholdMinSpeed = 0xFF;
   uint8 thresholdMaxSpeed = 0xFF;
   bool currentSpeedLockStatus = false;
   if (DP_S32_NO_ERR != DP_s32GetConfigItem("SystemConfiguration1", "SpeedLockMinThresholdForUseRestrictionKmH", &thresholdMinSpeed, 1))
   {
      ETG_TRACE_ERR(("VehicleDataClientHandler:readSpeedLockMinThresholdFromKDS Error"));
      currentSpeedLockStatus = SPEEDLOCK_INACTIVE;
   }
   if (DP_S32_NO_ERR != DP_s32GetConfigItem("SystemConfiguration1", "SpeedLockMaxThresholdForUseRestrictionKmH", &thresholdMaxSpeed, 1))
   {
      ETG_TRACE_ERR(("VehicleDataClientHandler:readSpeedLockMaxThresholdFromKDS Error"));
      currentSpeedLockStatus = SPEEDLOCK_INACTIVE;
   }
   ETG_TRACE_USR4(("VehicleDataClientHandler:Speed Threshold Values Recieved Min = %d, Max = %d", thresholdMinSpeed, thresholdMaxSpeed));

   // As provided in vehicle main fi "SpeedValue = Actual value * 100" Actual value = speedvalue received from VCAN
   vehiclespeed = vehiclespeed / 100;

   if (thresholdMaxSpeed == thresholdMinSpeed) // If the min and max threshold values are same in KDS
   {
      if (((vehiclespeed) > thresholdMaxSpeed))
      {
         currentSpeedLockStatus = SPEEDLOCK_ACTIVE;
      }
      else
      {
         currentSpeedLockStatus = SPEEDLOCK_INACTIVE;
      }
   }
   else if (((vehiclespeed) >= thresholdMaxSpeed)) // If the current vehicle speed is greater than the maximum threshold value
   {
      currentSpeedLockStatus = SPEEDLOCK_ACTIVE;
   }
   else if (((vehiclespeed) <= thresholdMinSpeed)) // If the current vehicle speed is lesser than the minimum threshold value
   {
      currentSpeedLockStatus = SPEEDLOCK_INACTIVE;
   }
   else // Else retain the current status of the SpeedLock
   {
      currentSpeedLockStatus = _speedLockState;
   }

   return currentSpeedLockStatus;
}


void VehicleDataClientHandler::sendSpeedLockUpdate()
{
   ETG_TRACE_USR4(("VehicleDataClientHandler::sendSpeedLockUpdate : Send the current speed lock at the start up "));
   POST_MSG((COURIER_MESSAGE_NEW(SpeedLockStatusUpdateMsg)(_speedLockState, _hmiDataServiceSpeedLockStatus)));
}
