/**************************************************************************************
* @file         : SpmClientHandler.cpp
* @author       :
* @addtogroup   : AppHmi_Navigation
* @brief        :
* @copyright    : (c) 2018-2018 Robert Bosch Car Multimedia GmbH
*                 The reproduction, distribution and utilization of this file as
*                 well as the communication of its contents to others without express
*                 authorization is prohibited. Offenders will be held liable for the
*                 payment of damages. All rights reserved in the event of the grant
*                 of a patent, utility model or design.
**************************************************************************************/

#include <hmibase/hall_std_if.h>
#include "SpmClientHandler.h"

#define NAV_USER_POI_USB_REGISTER   4
#define NAV_USER_POI_USB_UNREGISTER 1
#define NAV_USER_POI_DEVICE_FOLDER  "/myPOIs"

#ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
#define ETG_DEFAULT_TRACE_CLASS TR_CLASS_APPHMI_NAVIGATION_HALL
#define ETG_I_TRACE_CHANNEL TR_TTFIS_APPHMI_NAVIGATION
#define ETG_I_TTFIS_CMD_PREFIX "APPHMI_NAVIGATION_"
#define ETG_I_FILE_PREFIX App::Core::SpmClientHandler::
#include "trcGenProj/Header/SpmClientHandler.cpp.trc.h"
#endif


using namespace ::asf::core;
using namespace SPM_CORE_FI;
using namespace ::spm_core_fi_types;
using namespace fcswupdatesrv::FcSwUpdateSrv;

/** ccaservice.h */
// static const tU16 CCA_C_U16_SRV_MAPUPDATE       =    (tU16)0x0280;
static const tU16 CCA_C_U16_APP_FC_SW_UPDATE       = (tU16)0x00BC;
// static const tU16 CCA_C_U16_APP_FC_MAPUPDATE    =    (tU16)0x0131;
static const tU16 AMT_C_U16_APPID_INVALID          = (tU16)0xFFFF;
static const tU16 CCA_NAVIGATION_APPID             = (tU16)0x0099; // CCA_C_U16_APP_HMI_03

static const bool doClear = true;
static const ::std::string CENTRAL_LOCK_NAME = "UPDATE_LOCK_CENTRAL";


namespace App {
namespace Core {

SpmClientHandler::SpmClientHandler(const std::string& spmFiPort, const std::string& fcswupdatePort)
   : _serviceAvailable(false)
   , _fcswupdateProxyAvailable(false)
   , _spmCoreFiProxy(SPM_CORE_FIProxy::createProxy(spmFiPort, *this))
   , _fcswupdateProxy(FcSwUpdateSrvProxy::createProxy(fcswupdatePort, *this))
   , _mapUpdateLockOwner(AMT_C_U16_APPID_INVALID)
   , _isRequestingLock(false)
{
   if (_spmCoreFiProxy.get())
   {
      StartupSync::getInstance().registerPropertyRegistrationIF(this, _spmCoreFiProxy->getPortName());
   }
   if (_fcswupdateProxy.get())
   {
      StartupSync::getInstance().registerPropertyRegistrationIF(this, _fcswupdateProxy->getPortName());
   }
}


SpmClientHandler::~SpmClientHandler()
{
}


void SpmClientHandler::onAvailable(const ::boost::shared_ptr< ::asf::core::Proxy >& proxy, const ::asf::core::ServiceStateChange& stateChange)
{
   ETG_TRACE_USR4(("SpmClientHandler::onAvailable"));
   StartupSync::getInstance().onAvailable(proxy, stateChange);
}


void SpmClientHandler::onUnavailable(const ::boost::shared_ptr< ::asf::core::Proxy >& proxy, const ::asf::core::ServiceStateChange& stateChange)
{
   ETG_TRACE_USR4(("SpmClientHandler::onUnavailable"));
   StartupSync::getInstance().onUnavailable(proxy, stateChange);
}


void SpmClientHandler::registerProperties(const ::boost::shared_ptr< asf::core::Proxy >& proxy, const asf::core::ServiceStateChange& stateChange)
{
   if (proxy == _spmCoreFiProxy)
   {
      ETG_TRACE_USR4(("SpmClientHandler::registerProperties() _spmCoreFiProxy"));
      _serviceAvailable = true;
      _spmCoreFiProxy->sendDownloadLockStatesUpReg(*this);
      _spmCoreFiProxy->sendUpdateLockStatesUpReg(*this);
      _spmCoreFiProxy->sendSystemStateUpReg(*this);
   }

   if (proxy == _fcswupdateProxy)
   {
      _fcswupdateProxyAvailable = true;
      ETG_TRACE_USR4(("SpmClientHandler::registerProperties() _fcswupdateProxy"));
      if (_isRequestingLock)
      {
         _isRequestingLock = false;
      }
   }
}


void SpmClientHandler::deregisterProperties(const ::boost::shared_ptr< asf::core::Proxy >& proxy, const asf::core::ServiceStateChange& stateChange)
{
   if (proxy == _spmCoreFiProxy)
   {
      ETG_TRACE_USR4(("SpmClientHandler::deregisterProperties() _spmCoreFiProxy"));
      _serviceAvailable = false;
      _spmCoreFiProxy->sendDownloadLockStatesRelUpRegAll();
      _spmCoreFiProxy->sendUpdateLockStatesRelUpRegAll();
      _spmCoreFiProxy->sendSystemStateRelUpRegAll();
   }

   if (proxy == _fcswupdateProxy)
   {
      _fcswupdateProxyAvailable = false;
      ETG_TRACE_USR4(("SpmClientHandler::deregisterProperties() _fcswupdateProxy"));
   }
}


void SpmClientHandler::onSystemStateStatus(const ::boost::shared_ptr< ::SPM_CORE_FI::SPM_CORE_FIProxy >&  /*proxy*/, const ::boost::shared_ptr< ::SPM_CORE_FI::SystemStateStatus >& status)
{
   spm_core_fi_types::T_SPM_e32_SYSTEM_STATES systemState = status->getSystemState();
   ETG_TRACE_USR4(("SpmClientHandler::onSystemStateStatus(%d)", systemState));
   POST_MSG((COURIER_MESSAGE_NEW(SpmSystemStateChangedUpdMsg)(getRelevantSpmSystemState(systemState))));

   if (NAV_USER_POI_USB_REGISTER == systemState)
   {
      PrmNotificationHandler::vIntializeFuncUnit(this);
      vRegisterForUSBMedia();
   }
   else if (NAV_USER_POI_USB_UNREGISTER == systemState)
   {
      vUnRegisterForUSBMedia();
   }
}


void SpmClientHandler::vPRMHandler(const PrmNotificationHandler::DeviceInfo& oData)
{
   tU16 notiType = oData.type;
   ETG_TRACE_USR4(("SpmClientHandler::vPRMHandler(), Notification type : %d", notiType));

   if (OSAL_C_U16_NOTI_MEDIA_STATE == notiType)
   {
      vNotifyStateChange(oData.status, oData.devicePath);
   }
   else
   {
      ETG_TRACE_USR4(("SpmClientHandler::vPRMHandler(), OSAL_C_U16_NOTI_MEDIA_CHANGE and OSAL_C_U16_NOTI_MODE_CHANGE are not handled"));
   }
}


tVoid SpmClientHandler::vNotifyStateChange(const tU16& status, const std::string& szUsbPath)
{
   bool devicestatus = false;
   std::string myPOIsPath = "";
   ETG_TRACE_USR4(("SpmClientHandler::vNotifyStateChange(), Status : %d, USB path : %s", status, szUsbPath.c_str()));

   if (OSAL_C_U16_MEDIA_READY == status)
   {
      devicestatus = true;
      myPOIsPath = szUsbPath;
      myPOIsPath += NAV_USER_POI_DEVICE_FOLDER;
      bool isSrcAvailable = vUSBSourceAvailableCheck(myPOIsPath);
      if(false == isSrcAvailable)
      {
         myPOIsPath = "";
      }
   }
   POST_MSG((COURIER_MESSAGE_NEW(UPOIDeviceStatusUpdMsg)(devicestatus, myPOIsPath.c_str())));
}


bool SpmClientHandler::vUSBSourceAvailableCheck(const std::string& szUsbPath)
{
   bool status = false;
   OSAL_trIOCtrlDir* prIOCtrlDir = OSAL_NULL;
   ETG_TRACE_USR4(("SpmClientHandler::vUSBSourceAvailableCheck(), USB path: %s", szUsbPath.c_str()));
   OSAL_tIODescriptor fd = OSAL_IOOpen(szUsbPath.c_str(), OSAL_EN_READWRITE);
   if (fd != OSAL_ERROR)
   {
      if (OSAL_NULL != (prIOCtrlDir = OSALUTIL_prOpenDir(szUsbPath.c_str())))
      {
         status = true;
         OSALUTIL_s32CloseDir(prIOCtrlDir);
      }
      else
      {
         ETG_TRACE_USR4(("SpmClientHandler::vUSBSourceAvailableCheck(), Fail"));
      }
      OSAL_s32IOClose(fd);
   }
   return status;
}


void SpmClientHandler::vRegisterForUSBMedia()
{
   ETG_TRACE_USR4(("SpmClientHandler::vRegisterForUSBMedia()"));

   OSAL_tIODescriptor fd = OSAL_IOOpen(OSAL_C_STRING_DEVICE_PRM, OSAL_EN_READONLY);
   if (fd != OSAL_ERROR)
   {
      OSAL_trNotifyDataExt2 oNotifyData;
      oNotifyData.u16AppID = OSAL_C_U16_NAVI_APPID;
      oNotifyData.ResourceName = NAV_USER_POI_MEDIA_SOURCE_PATH;
      oNotifyData.u16NotificationType = OSAL_C_U16_NOTI_MEDIA_CHANGE | OSAL_C_U16_NOTI_MEDIA_STATE | OSAL_C_U16_NOTI_MODE_CHANGE;
      oNotifyData.pCallbackExt2 = (OSALCALLBACKFUNCEXT2)PrmNotificationHandler::vUSBNotificationHandler;

      int s32Status = OSAL_s32IOControl(fd, OSAL_C_S32_IOCTRL_PRM_REG_NOTIFICATION_EXT2, (uintptr_t)&oNotifyData);
      ETG_TRACE_USR4(("SpmClientHandler::vRegisterForUSBMedia() OSAL_s32IOControl status %d", s32Status));
      OSAL_s32IOClose(fd);
   }
}


void SpmClientHandler::vUnRegisterForUSBMedia()
{
   ETG_TRACE_USR4(("SpmClientHandler::vUnRegisterForUSBMedia()"));

   OSAL_tIODescriptor fd = OSAL_IOOpen(OSAL_C_STRING_DEVICE_PRM, OSAL_EN_READONLY);
   if (fd != OSAL_ERROR)
   {
      OSAL_trNotifyDataExt2 oNotifyData;
      oNotifyData.u16AppID = OSAL_C_U16_NAVI_APPID;
      oNotifyData.ResourceName = NAV_USER_POI_MEDIA_SOURCE_PATH;
      oNotifyData.u16NotificationType = OSAL_C_U16_NOTI_MEDIA_CHANGE | OSAL_C_U16_NOTI_MEDIA_STATE | OSAL_C_U16_NOTI_MODE_CHANGE;

      int s32Status = OSAL_s32IOControl(fd, OSAL_C_S32_IOCTRL_PRM_UNREG_NOTIFICATION_EXT2, (uintptr_t)&oNotifyData);
      ETG_TRACE_USR4(("SpmClientHandler::vUnRegisterForUSBMedia() OSAL_s32IOControl status %d", s32Status));
      OSAL_s32IOClose(fd);
   }
}


void SpmClientHandler::onSystemStateError(const ::boost::shared_ptr< ::SPM_CORE_FI::SPM_CORE_FIProxy >& /*proxy*/, const ::boost::shared_ptr< ::SPM_CORE_FI::SystemStateError >& error)
{
   ETG_TRACE_USR4(("SpmClientHandler::onSystemStateError()"));
}


void SpmClientHandler::onOnOffTriggerError(const ::boost::shared_ptr< SPM_CORE_FIProxy >& /*proxy*/,
      const ::boost::shared_ptr< OnOffTriggerError >& /*error*/)
{
   ETG_TRACE_USR4(("SpmClientHandler::onOnOffTriggerError()"));
}


void SpmClientHandler::onOnOffTriggerResult(const ::boost::shared_ptr< SPM_CORE_FIProxy >& /*proxy*/,
      const ::boost::shared_ptr< OnOffTriggerResult >& /*result*/)
{
   ETG_TRACE_USR4(("SpmClientHandler::onOnOffTriggerResult()"));
}


void SpmClientHandler::onDownloadLockStatesError(const ::boost::shared_ptr< SPM_CORE_FIProxy >& /*proxy*/,
      const ::boost::shared_ptr< DownloadLockStatesError >& /*error*/)
{
   ETG_TRACE_USR4(("SpmClientHandler::onDownloadLockStatesError()"));
}


void SpmClientHandler::onDownloadLockStatesStatus(const ::boost::shared_ptr< SPM_CORE_FIProxy >& /*proxy*/,
      const ::boost::shared_ptr< DownloadLockStatesStatus >& status)
{
   ETG_TRACE_USR4(("SpmClientHandler::onDownloadLockStatesStatus()"));
   const ::boost::ptr_vector< ::spm_core_fi_types::T_SPM_RestrictedDlLock >& listRestrictedLock = status->getListRestrictedLock();
   bool isSWUpdating = false;
   std::stringstream traceOut;
   for (unsigned int i = 0; i < status->getListRestrictedLock().size(); ++i)
   {
      traceOut << "{ ";
      traceOut << status->getListRestrictedLock()[i].getStrRestrictedLockArea() << ", ";
      traceOut << status->getListRestrictedLock()[i].getU16AppIdLockOwner() << ", ";
      traceOut << status->getListRestrictedLock()[i].getU16AppIdLockRequester();
      traceOut << " }" << std::endl;

      if (CCA_C_U16_APP_FC_SW_UPDATE == status->getListRestrictedLock()[i].getU16AppIdLockOwner())
      {
         isSWUpdating = true;
      }
   }

   ETG_TRACE_USR4(("%s", traceOut.str().c_str()));

   // POST_MSG((COURIER_MESSAGE_NEW(MapDownloadLockStateChangedUpdMsg)(isSWUpdating)));
}


bool SpmClientHandler::onCourierMessage(const NAVDATAUPDATE_SpmStartMapUpdateLockReqMsg& oMsg)
{
   ETG_TRACE_USR4(("SpmClientHandler::onCourierMessage(NAVDATAUPDATE_SpmStartMapUpdateLockReqMsg))"));
   COURIER_UNUSED(oMsg);
   bool isCentralLockOwnedbyNavi = (CCA_NAVIGATION_APPID == _mapUpdateLockOwner);

   if (_spmCoreFiProxy && _serviceAvailable)
   {
      if (!_eLockState.empty() && ((_eLockState.front() == T_e8_UpdateLockState__SPM_e8_Locking) || (_eLockState.front() == T_e8_UpdateLockState__SPM_e8_Locked)
                                   || (_eLockState.front() == T_e8_UpdateLockState__SPM_e8_Reserved)) && (!isCentralLockOwnedbyNavi))
      {
         POST_MSG((COURIER_MESSAGE_NEW(NAVDATAUPDATE_SpmStartMapUpdateLockResMsg)(false)));
      }
      else if (!_eLockState.empty() && isCentralLockOwnedbyNavi && (T_e8_UpdateLockState__SPM_e8_Locked == _eLockState.front()))
      {
         // Navi is currently owning the CENTRAL_LOCK and can perform operations directly without asking SPM again
         // So just simply send response to HMIModel
         POST_MSG((COURIER_MESSAGE_NEW(NAVDATAUPDATE_SpmStartMapUpdateLockResMsg)(true)));
         _isRequestingLock = false;
      }
      else
      {
         _isRequestingLock = true;
         _spmCoreFiProxy->sendUpdateLockStart(*this, CENTRAL_LOCK_NAME, ::spm_core_fi_types::T_e8_UpdateLockCommand__SPM_e8_Reserve);
      }
   }
   else
   {
      POST_MSG((COURIER_MESSAGE_NEW(NAVDATAUPDATE_SpmStartMapUpdateLockResMsg)(false)));
      _isRequestingLock = false;
      ETG_TRACE_ERR(("_spmCoreFiProxy is not available!"));
   }

   return true;
}


bool SpmClientHandler::onCourierMessage(const NAVDATAUPDATE_SpmMapUpdateUnLockReqMsg& oMsg)
{
   COURIER_UNUSED(oMsg);
   ETG_TRACE_USR4(("SpmClientHandler::onCourierMessage(NAVDATAUPDATE_SpmMapUpdateUnLockReqMsg))"));
   releaseMapUpdateLock();
   return true;
}


void SpmClientHandler::releaseMapUpdateLock()
{
   ETG_TRACE_USR4(("SpmClientHandler::releaseMapUpdateLock()"));
   const bool hasMapUpdateLock = (CCA_NAVIGATION_APPID == _mapUpdateLockOwner);

   if (_spmCoreFiProxy && _serviceAvailable && hasMapUpdateLock)
   {
      _spmCoreFiProxy->sendUpdateLockStart(*this, CENTRAL_LOCK_NAME, ::spm_core_fi_types::T_e8_UpdateLockCommand__SPM_e8_Use);
   }
   else
   {
      ETG_TRACE_ERR(("_spmCoreFiProxy is not available or not having MapUploadLock!"));
   }
}


void SpmClientHandler::onUpdateLockError(const ::boost::shared_ptr< SPM_CORE_FI::SPM_CORE_FIProxy >& /*proxy*/, const ::boost::shared_ptr< SPM_CORE_FI::UpdateLockError >& /*error*/)
{
   ETG_TRACE_USR4(("SpmClientHandler::onUpdateLockError()"));

   if (_isRequestingLock)
   {
      POST_MSG((COURIER_MESSAGE_NEW(NAVDATAUPDATE_SpmStartMapUpdateLockResMsg)(false)));
      _isRequestingLock = false;
   }
}


void SpmClientHandler::onUpdateLockResult(const ::boost::shared_ptr< SPM_CORE_FI::SPM_CORE_FIProxy >& /*proxy*/, const ::boost::shared_ptr< SPM_CORE_FI::UpdateLockResult >& result)
{
   tS32 s32ReturnVal = result->getS32ReturnVal();
   ETG_TRACE_USR4(("SpmClientHandler::onUpdateLockResult returnVal:%d", s32ReturnVal));
}


void SpmClientHandler::onUpdateLockStatesError(const ::boost::shared_ptr< SPM_CORE_FI::SPM_CORE_FIProxy >& /*proxy*/, const ::boost::shared_ptr< SPM_CORE_FI::UpdateLockStatesError >& /*error*/)
{
   ETG_TRACE_USR4(("SpmClientHandler::onUpdateLockStatesError()"));
}


void SpmClientHandler::onUpdateLockStatesStatus(const ::boost::shared_ptr< SPM_CORE_FI::SPM_CORE_FIProxy >& /*proxy*/, const ::boost::shared_ptr< SPM_CORE_FI::UpdateLockStatesStatus >& status)
{
   ETG_TRACE_USR4(("SpmClientHandler::onUpdateLockStatesStatus()"));
   ::boost::ptr_vector< ::spm_core_fi_types::T_SPM_UpdateLockState > spmData = status->getLLocks();
   ::boost::ptr_vector< ::spm_core_fi_types::T_SPM_UpdateLockState >::iterator pos;
   bool UPDATE_LOCK_CENTRAL_FOUND = false;
   pos = spmData.begin();

   while ((pos != spmData.end()) && (!UPDATE_LOCK_CENTRAL_FOUND))
   {
      ::spm_core_fi_types::T_SPM_UpdateLockState& updateLockStateData = *pos;
      std::string lockName = updateLockStateData.getStrLockName();
      if (CENTRAL_LOCK_NAME == lockName)
      {
         UPDATE_LOCK_CENTRAL_FOUND = true;
         _mapUpdateLockOwner = updateLockStateData.getU16LockOwner();
         _eLockState.clear();
         _eLockState.push_back(updateLockStateData.getELockState());
         const std::vector<tU16> userList = updateLockStateData.getLu16LockUsers();
         const bool isNaviUsingCentralLock = searchElemByUser(CCA_NAVIGATION_APPID, userList);
         const bool isNaviCentralLockOwner = (_mapUpdateLockOwner == CCA_NAVIGATION_APPID ? true : false);
         ETG_TRACE_USR4(("SpmClientHandler::onUpdateLockStatesStatus(ownerAppId:%u state:%u)", _mapUpdateLockOwner, ETG_CENUM(::spm_core_fi_types::T_e8_UpdateLockState, _eLockState.front())));
         printUserListInfo(userList);

         switch (_eLockState.front())
         {
            case ::spm_core_fi_types::T_e8_UpdateLockState__SPM_e8_Used:
            {
               if ((_mapUpdateLockOwner == 0) && (!isNaviUsingCentralLock))
               {
                  _spmCoreFiProxy->sendUpdateLockStart(*this, CENTRAL_LOCK_NAME, ::spm_core_fi_types::T_e8_UpdateLockCommand__SPM_e8_Use);
               }
               break;
            }
            case ::spm_core_fi_types::T_e8_UpdateLockState__SPM_e8_Locked:
            {
               if ((isNaviCentralLockOwner) && (userList.empty()))
               {
                  if (_isRequestingLock)
                  {
                     POST_MSG((COURIER_MESSAGE_NEW(NAVDATAUPDATE_SpmStartMapUpdateLockResMsg)(true)));
                     _isRequestingLock = false;
                  }
                  else
                  {
                     _spmCoreFiProxy->sendUpdateLockStart(*this, CENTRAL_LOCK_NAME, ::spm_core_fi_types::T_e8_UpdateLockCommand__SPM_e8_Use);
                  }
               }
               break;
            }
            case ::spm_core_fi_types::T_e8_UpdateLockState__SPM_e8_Reserved:
            {
               if ((!isNaviCentralLockOwner) && (isNaviUsingCentralLock))
               {
                  _spmCoreFiProxy->sendUpdateLockStart(*this, CENTRAL_LOCK_NAME, ::spm_core_fi_types::T_e8_UpdateLockCommand__SPM_e8_Release);
               }
               else if ((isNaviCentralLockOwner) && (isNaviUsingCentralLock))
               {
                  _spmCoreFiProxy->sendUpdateLockStart(*this, CENTRAL_LOCK_NAME, ::spm_core_fi_types::T_e8_UpdateLockCommand__SPM_e8_Lock);
               }
               break;
            }
            case ::spm_core_fi_types::T_e8_UpdateLockState__SPM_e8_Locking:
            {
               if ((!isNaviCentralLockOwner) && (isNaviUsingCentralLock))
               {
                  _spmCoreFiProxy->sendUpdateLockStart(*this, CENTRAL_LOCK_NAME, ::spm_core_fi_types::T_e8_UpdateLockCommand__SPM_e8_Release);
               }
               break;
            }
            default:
               ETG_TRACE_USR4(("SpmClientHandler::onUpdateLockStatesStatus(state:%u), UpdateLockState not handled", ETG_CENUM(::spm_core_fi_types::T_e8_UpdateLockState, _eLockState.front())));
               break;
         }
      }
      ++pos;
   }
   if (!UPDATE_LOCK_CENTRAL_FOUND)
   {
      ETG_TRACE_USR4(("SpmClientHandler::onUpdateLockStatesStatus(),UPDATE_LOCK_CENTRAL is not in the list, adding navigation as lock user"));
      _spmCoreFiProxy->sendUpdateLockStart(*this, CENTRAL_LOCK_NAME, ::spm_core_fi_types::T_e8_UpdateLockCommand__SPM_e8_Use);
   }
}


ENUM_CONVERT_START(SpmClientHandler::getRelevantSpmSystemState, T_SPM_e32_SYSTEM_STATES, enSpmSystemState)
ENUM_CONVERT_ENTRY(T_SPM_e32_SYSTEM_STATES__SPM_SYSTEM_BACKGROUND, SPM_SYSTEM_STATE__BACKGROUND)
ENUM_CONVERT_ENTRY(T_SPM_e32_SYSTEM_STATES__SPM_SYSTEM_PREPARE_SHUTDOWN, SPM_SYSTEM_STATE__PREPARE_SHUTDOWN)
ENUM_CONVERT_ENTRY(T_SPM_e32_SYSTEM_STATES__SPM_SYSTEM_SHUTDOWN, SPM_SYSTEM_STATE__SHUTDOWN)
ENUM_CONVERT_ENTRY(T_SPM_e32_SYSTEM_STATES__SPM_SYSTEM_SUSPEND, SPM_SYSTEM_STATE__SUSPEND)
ENUM_CONVERT_ENTRY(T_SPM_e32_SYSTEM_STATES__SPM_SYSTEM_STANDBY, SPM_SYSTEM_STATE__STANDBY)
ENUM_CONVERT_ENTRY(T_SPM_e32_SYSTEM_STATES__SPM_SYSTEM_OFF, SPM_SYSTEM_STATE__OFF)
ENUM_CONVERT_ENTRY(T_SPM_e32_SYSTEM_STATES__SPM_SYSTEM_DOWNLOAD, SPM_SYSTEM_STATE__DOWNLOAD)
ENUM_CONVERT_ENTRY(T_SPM_e32_SYSTEM_STATES__SPM_SYSTEM_ON, SPM_SYSTEM_STATE__ON)
ENUM_CONVERT_ENTRY(T_SPM_e32_SYSTEM_STATES__SPM_SYSTEM_DOOR_OPEN, SPM_SYSTEM_STATE__DOOR_OPEN)
ENUM_CONVERT_ENTRY(T_SPM_e32_SYSTEM_STATES__SPM_SYSTEM_IGNITION, SPM_SYSTEM_STATE__IGNITION)
ENUM_CONVERT_ENTRY(T_SPM_e32_SYSTEM_STATES__SPM_SYSTEM_DIAGNOSIS, SPM_SYSTEM_STATE__DIAGNOSIS)
ENUM_CONVERT_ENTRY(T_SPM_e32_SYSTEM_STATES__SPM_SYSTEM_STATE_PROFILE, SPM_SYSTEM_STATE__PROFILE)
ENUM_CONVERT_ENTRY(T_SPM_e32_SYSTEM_STATES__SPM_SYSTEM_MMI_STANDBY_PWR_SAVE_1, SPM_SYSTEM_STATE__MMI_STANDBY_PWR_SAVE_1)
ENUM_CONVERT_ENTRY(T_SPM_e32_SYSTEM_STATES__SPM_SYSTEM_MMI_STANDBY_PWR_SAVE_2, SPM_SYSTEM_STATE__MMI_STANDBY_PWR_SAVE_2)
ENUM_CONVERT_ENTRY(T_SPM_e32_SYSTEM_STATES__SPM_SYSTEM_MMI_STANDBY, SPM_SYSTEM_STATE__MMI_STANDBY)
ENUM_CONVERT_ENTRY(T_SPM_e32_SYSTEM_STATES__SPM_SYSTEM_MMI_ON, SPM_SYSTEM_STATE__MMI_ON)
ENUM_CONVERT_ENTRY(T_SPM_e32_SYSTEM_STATES__SPM_SYSTEM_MMI_STANDBY_RESTRICTED, SPM_SYSTEM_STATE__MMI_STANDBY_RESCTRICTED)
ENUM_CONVERT_ENTRY(T_SPM_e32_SYSTEM_STATES__SPM_SYSTEM_MMI_ON_DIAG, SPM_SYSTEM_STATE__MMI_ON_DIAG)
ENUM_CONVERT_ENTRY(T_SPM_e32_SYSTEM_STATES__SPM_SYSTEM_MMI_ON_TEL, SPM_SYSTEM_STATE__MMI_ON_TEL)
ENUM_CONVERT_ENTRY(T_SPM_e32_SYSTEM_STATES__SPM_SYSTEM_MMI_ON_SWDL, SPM_SYSTEM_STATE__MMI_ON_SWDL)
ENUM_CONVERT_ENTRY(T_SPM_e32_SYSTEM_STATES__SPM_SYSTEM_MMI_STANDBY_CUSTOMER_SWDL, SPM_SYSTEM_STATE__MMI_STANDBY_CUSTOMER_SWDL)
ENUM_CONVERT_ENTRY(T_SPM_e32_SYSTEM_STATES__SPM_SYSTEM_MMI_STANDBY_PWR_SAVE, SPM_SYSTEM_STATE__MMI_STANDBY_PWR_SAVE)
ENUM_CONVERT_ENTRY(T_SPM_e32_SYSTEM_STATES__SPM_SYSTEM_OVERTEMP, SPM_SYSTEM_STATE__OVER_TMP)
ENUM_CONVERT_ENTRY(T_SPM_e32_SYSTEM_STATES__SPM_SYSTEM_SAFE, SPM_SYSTEM_STATE__SAFE)
ENUM_CONVERT_ENTRY(T_SPM_e32_SYSTEM_STATES__SPM_SYSTEM_STATE_MAX, SPM_SYSTEM_STATE__STATE_MAX)
ENUM_CONVERT_END(SPM_SYSTEM_STATE__STATE_MAX)


bool SpmClientHandler::searchElemByUser(tU16 appId, const std::vector<tU16>& userList)
{
   ETG_TRACE_USR4(("SpmClientHandler::searchElemByUser(Id:%u)", appId));
   for (std::vector<tU16>::const_iterator iter = userList.begin(); iter != userList.end(); ++iter)
   {
      if (appId == *iter)
      {
         return true;
      }
   }
   return false;
}


/*bool SpmClientHandler::onCourierMessage(const SetSpmSubstateCustomerDownloadActivationReqMsg& oMsg)
{
   ETG_TRACE_USR4(("SpmClientHandler::onCourierMessage(SetSpmSubstateCustomerDownloadActivationReqMsg %d)", oMsg.GetIsCustomerDownloadActive()));
   if (_spmCoreFiProxy && _serviceAvailable) // safety
   {
      _spmCoreFiProxy->sendOnOffTriggerStart(*this, T_SPM_e32_SubStateType__SPM_U32_SUBSTATE_CUSTOMER_DOWNLOAD, oMsg.GetIsCustomerDownloadActive());
   }
   else
   {
      ETG_TRACE_ERR(("spmCoreFiProxy is not available. Could not set SubState."));
   }

   return true;
}*/


void SpmClientHandler::printUserListInfo(const std::vector<tU16>& userList)
{
   for (std::vector<tU16>::const_iterator Iter = userList.begin(); Iter != userList.end(); ++Iter)
   {
      ETG_TRACE_USR4(("user:0x%x", *Iter));
   }
}


} // namespace Core
} // namespace App
