/**************************************************************************************
* @file         : NavigationHall.cpp
* @author       :
* @addtogroup   : AppHmi_Navigation
* @brief        :
* @copyright    : (c) 2018-2020 Robert Bosch Car Multimedia GmbH
*                 The reproduction, distribution and utilization of this file as
*                 well as the communication of its contents to others without express
*                 authorization is prohibited. Offenders will be held liable for the
*                 payment of damages. All rights reserved in the event of the grant
*                 of a patent, utility model or design.
**************************************************************************************/

#include "hall_std_if.h"
#include "App/Core/NavigationHall.h"
#include "unistd.h"

#ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
#define ETG_DEFAULT_TRACE_CLASS           TR_CLASS_APPHMI_NAVIGATION_HALL
#define ETG_I_TRACE_CHANNEL               TR_TTFIS_APPHMI_NAVIGATION
#define ETG_I_TTFIS_CMD_PREFIX            "APPHMI_NAVIGATION_"
#define ETG_I_FILE_PREFIX                 App::Core::NavigationHall::
#include "trcGenProj/Header/NavigationHall.cpp.trc.h"
#endif

#define FREE_OBJ(x) if(x!=NULL){delete x; x=NULL;}

using namespace ::App::Core;
using namespace ::CourierTunnelService::CourierMessageReceiver;

static const ::std::string VDDIM_FI_PORT = "vdDimFiPort";
static const ::std::string BTTEL_FI_PORT = "btTelFiPort";
static const ::std::string BTSET_FI_PORT = "btSetFiPort";
static const ::std::string COMMANDDINTERFACE_FI_PORT = "commandInterfacePort";
static const ::std::string MASCF_FI_PORT = "mascfFiPort";
static const ::std::string VEHICLE_MAIN_FI_PORT = "vehicleMainFiPort";
static const ::std::string CLOCK_FI_PORT = "clockFiPort";
static const ::std::string SPM_FI_PORT = "spmFiPort";
static const ::std::string FCSWUPDATE_PORT = "fcSwUpdatePort";
static const ::std::string APPLICATION_SWITCH_PORT = "applicationSwitchPort";
static const ::std::string APP_HMI_NAVIGATION_SERVICE_PORT = "appHmiNavigationServicePort";
static const ::std::string HMI_DATA_SERVICE_PORT = "hmiDataServicePort";

namespace App {
namespace Core {

DEFINE_CLASS_LOGGER_AND_LEVEL("App/Core", NavigationHall, Info);

NavigationHall::NavigationHall()
   : HallComponentBase("", "App.Core.AppHmi_Navigation", "/org/genivi/NodeStateManager/LifeCycleConsumer/AppHmi_Navigation")
{
   ETG_I_REGISTER_FILE();
   ETG_TRACE_USR4(("NavigationHall:: Navigation Hall constructor \n"));

   _btPhoneClientHandler = new BTPhoneClientHandler(BTTEL_FI_PORT, BTSET_FI_PORT);
   _audioClientHandler = new AudioClientHandler(COMMANDDINTERFACE_FI_PORT, MASCF_FI_PORT);
   _vehicleDataClientHandler = new VehicleDataClientHandler(VEHICLE_MAIN_FI_PORT, HMI_DATA_SERVICE_PORT);
   _vehicleDimmingClientHandler = new VehicleDimmingClientHandler(VDDIM_FI_PORT);
   _clockClientHandler = new ClockClientHandler(CLOCK_FI_PORT);
   _spmClientHandler = new SpmClientHandler(SPM_FI_PORT, FCSWUPDATE_PORT);
   _applicationSwitchClientComponent = ApplicationSwitchClientComponent::poGetInstance();
   _audioSourceHandler = AudioSourceHandler::getInstance();
   _naviApplicationSwitchClientHandler = new NaviApplicationSwitchClientHandler(APPLICATION_SWITCH_PORT);
   DiagnosisServiceClientHandler::s_Intialize(APP_HMI_NAVIGATION_SERVICE_PORT); //port name needs to be same as port name given in cmc file
   _diagnosticsClientHandler = DiagnosticsClientHandler::getInstance();
   _clusterDataClientHandler = ClusterDataClientHandler::getInstance();

   if (_applicationSwitchClientComponent != NULL)
   {
      _applicationSwitchClientComponent->setApplicationID(APPID_APPHMI_NAVIGATION);
   }
}


NavigationHall::~NavigationHall()
{
   FREE_OBJ(_btPhoneClientHandler)
   FREE_OBJ(_audioClientHandler)
   FREE_OBJ(_vehicleDataClientHandler)
   FREE_OBJ(_vehicleDimmingClientHandler)
   FREE_OBJ(_clockClientHandler)
   FREE_OBJ(_spmClientHandler);
   FREE_OBJ(_naviApplicationSwitchClientHandler);
   DiagnosisServiceClientHandler::s_Destrory();
   DiagnosticsClientHandler::destroyInstance();
   ClusterDataClientHandler::destroyInstance();
}


bool NavigationHall::onCourierMessage(const ApplicationStateUpdMsg& oMsg)
{
   ETG_TRACE_USR4(("NavigationHall::onCourierMessage(ApplicationStateUpdMsg(%d))", oMsg.GetState()));

   _modelComponent.onCourierMessage(oMsg);
   if (NULL != _naviApplicationSwitchClientHandler)
   {
      _naviApplicationSwitchClientHandler->onCourierMessage(oMsg);
   }

   return true;
}


void NavigationHall::onSavePersistency()
{
   ETG_TRACE_USR1(("NavigationHall::onSavePersistency()"));

   _modelComponent.savePersistency();
}


bool NavigationHall::OnAppModeChange(hmibase::app::base::lcm::ShutdownType newState, hmibase::app::base::lcm::ShutdownType oldState)
{
   ETG_TRACE_USR1(("NavigationHall::OnAppModeChange(newState: %d, oldState: %d)", newState, oldState));

   if ((hmibase::app::base::lcm::SHUTDOWN_NORMAL == newState) || (hmibase::app::base::lcm::SHUTDOWN_FAST == newState))
   {
      _modelComponent.deinitialize();
   }
   else if ((hmibase::app::base::lcm::SHUTDOWN_RUNUP != oldState) && (hmibase::app::base::lcm::SHUTDOWN_RUNUP == newState))
   {
      // For normal startup we get Courier::StartupMsg from the framework and call _modelComponent.initialize().
      // In case the HMI was requested to shut down but woken up again, we handle it here and call restart of modelComponent.
      _modelComponent.reinitialize();
   }
   else
   {
   }
   return true;
}


void NavigationHall::onExpired(asf::core::Timer& /*timer*/, boost::shared_ptr<asf::core::TimerPayload> /*data*/)
{
}


/**
* reqPrepareResponse: To know the response of the respective APP when Restore factory setting is pressed.
* @param[in]: None
* @param[in] :None
*/
void NavigationHall::reqPrepareResponse(const PrepareResDiagMsgData& /*oPrepareResDiagMsgData*/)
{
   ETG_TRACE_USR4(("NavigationHall:: reqPrepareResponse"));
}


/**
 * reqExecuteResponse: To Execute the response when restore factory setting has been executed..
 * @param[in]: None
 * @param[in] :None
 */
void NavigationHall::reqExecuteResponse(const ExecuteResDiagMsgData& /*oExecuteResDiagMsgData*/)
{
   ETG_TRACE_USR4(("NavigationHall:: reqExecuteResponse"));
}


/**
 * reqExecuteResponse: To Finalize the response when restore factory setting has been executed.
 * @param[in]: None
 * @param[in] :None
 */
void NavigationHall:: reqFinalizeResponse(const FinalizeResDiagMsgData& /*oFinalizeResDiagMsgData*/)
{
   ETG_TRACE_USR4(("NavigationHall:: reqFinalizeResponse"));
}


} // namespace Core
} // namespace App
