/**************************************************************************************
* @file         : BTPhoneClientHandler.cpp
* @author       :
* @addtogroup   : AppHmi_Navigation
* @brief        :
* @copyright    : (c) 2018-2022 Robert Bosch Car Multimedia GmbH
*                 The reproduction, distribution and utilization of this file as
*                 well as the communication of its contents to others without express
*                 authorization is prohibited. Offenders will be held liable for the
*                 payment of damages. All rights reserved in the event of the grant
*                 of a patent, utility model or design.
**************************************************************************************/

#include <hmibase/hall_std_if.h>
#include "BTPhoneClientHandler.h"

#ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
#define ETG_DEFAULT_TRACE_CLASS           TR_CLASS_APPHMI_NAVIGATION_HALL
#define ETG_I_TRACE_CHANNEL               TR_TTFIS_APPHMI_NAVIGATION
#define ETG_I_TTFIS_CMD_PREFIX            "APPHMI_NAVIGATION_"
#define ETG_I_FILE_PREFIX                 App::Core::BTPhoneClientHandler::
#include "trcGenProj/Header/BTPhoneClientHandler.cpp.trc.h"
#endif

using namespace ::asf::core;
using namespace ::MOST_Tel_FI;
using namespace ::MOST_BTSet_FI;
using namespace ::most_Tel_fi_types;

namespace App {
namespace Core {

BTPhoneClientHandler::BTPhoneClientHandler(const std::string& btTelFiPort, const std::string& btSetFiPort)
   : _btTelProxy(MOST_Tel_FIProxy::createProxy(btTelFiPort, *this))
   , _btSetProxy(MOST_BTSet_FIProxy::createProxy(btSetFiPort, *this))
   , _isServiceAvailable(false)
   , _isPANConnected(false)
   , _isPANConnectionRequired(false)
   , _panProfileDeviceHandle(0)
   , _queuedDeviceHandleForPanRequest(0)
   , _activeDeviceHandle(0)
   , _connectedDeviceHandle()
{
   if (_btTelProxy.get())
   {
      ETG_TRACE_USR4(("BTPhoneClientHandler::BTPhoneClientHandler(), registerPropertyRegistrationIF for Phone"));
      StartupSync::getInstance().registerPropertyRegistrationIF(this, _btTelProxy->getPortName());
   }
   if (_btSetProxy.get())
   {
      ETG_TRACE_USR4(("BTPhoneClientHandler::BTPhoneClientHandler(), registerPropertyRegistrationIF for BT"));
      StartupSync::getInstance().registerPropertyRegistrationIF(this, _btSetProxy->getPortName());
   }
}


BTPhoneClientHandler::~BTPhoneClientHandler()
{
}


void BTPhoneClientHandler::onAvailable(const ::boost::shared_ptr< ::asf::core::Proxy >& proxy, const ::asf::core::ServiceStateChange& stateChange)
{
   ETG_TRACE_USR4(("BTPhoneClientHandler::onAvailable()"));
   StartupSync::getInstance().onAvailable(proxy, stateChange);
}


void BTPhoneClientHandler::onUnavailable(const ::boost::shared_ptr< ::asf::core::Proxy >& proxy, const ::asf::core::ServiceStateChange& stateChange)
{
   ETG_TRACE_USR4(("BTPhoneClientHandler::onUnavailable()"));
   StartupSync::getInstance().onUnavailable(proxy, stateChange);
}


void BTPhoneClientHandler::registerProperties(const ::boost::shared_ptr< Proxy >& proxy, const ServiceStateChange& stateChange)
{
   if(proxy == _btTelProxy)
   {
      ETG_TRACE_USR4(("BTPhoneClientHandler::registerProperties(Phone)"));
      _btTelProxy->sendActivePhoneDeviceUpReg(*this);
   }
   else if(proxy == _btSetProxy)
   {
      ETG_TRACE_USR4(("BTPhoneClientHandler::registerProperties(BT)"));
      _btSetProxy->sendDeviceListExtendedUpReg(*this);
      _btSetProxy->sendServiceAvailableUpReg(*this);
   }
}


void BTPhoneClientHandler::deregisterProperties(const ::boost::shared_ptr< Proxy >& proxy, const ServiceStateChange& stateChange)
{
   if(proxy == _btTelProxy)
   {
      ETG_TRACE_USR4(("BTPhoneClientHandler::deregisterProperties(Phone)"));
      _btTelProxy->sendActivePhoneDeviceRelUpRegAll();
   }
   else if(proxy == _btSetProxy)
   {
      ETG_TRACE_USR4(("BTPhoneClientHandler::deregisterProperties(BT)"));
      _btSetProxy->sendDeviceListExtendedRelUpRegAll();
      _btSetProxy->sendServiceAvailableRelUpRegAll();
   }
}


void BTPhoneClientHandler::onDialError(const ::boost::shared_ptr< MOST_Tel_FIProxy >& /*proxy*/, const boost::shared_ptr< DialError >& /*error*/)
{
   ETG_TRACE_USR4(("BTPhoneClientHandler::onDialError()"));
}


void BTPhoneClientHandler::onDialResult(const ::boost::shared_ptr< MOST_Tel_FIProxy >& /*proxy*/, const boost::shared_ptr< DialResult >& result)
{
   ETG_TRACE_USR4(("BTPhoneClientHandler::onDialResult(), callInstance : %d, callStatus : %d, isUsingVehicleAudio : %d",
                   result->getU16CallInstance(),
                   result->getE8CallStatus(),
                   result->getBUsingVehicleAudio()));
}


void BTPhoneClientHandler::onDeviceListExtendedError(const ::boost::shared_ptr< MOST_BTSet_FIProxy >& /*proxy*/,
      const boost::shared_ptr< DeviceListExtendedError >& /*error*/)
{
   ETG_TRACE_USR4(("BTPhoneClientHandler::onDeviceListExtendedError()"));
}


void BTPhoneClientHandler::onDeviceListExtendedStatus(const ::boost::shared_ptr< MOST_BTSet_FIProxy >& /*proxy*/,
      const boost::shared_ptr< DeviceListExtendedStatus >& status)
{
   bool isPANConnected = false;
   bool isDeviceConnected = false;

   // Reset device handles to 0
   _panProfileDeviceHandle = 0;
   for(uint8 connectedIndex = 0; connectedIndex < MAX_NUM_CONNECTED_DEVICES; connectedIndex++)
   {
      _connectedDeviceHandle[connectedIndex] = 0;
   }

   // Store updated connected device handles
   for (uint8 connectedIndex = 0, pairedIndex = 0;
         (connectedIndex < MAX_NUM_CONNECTED_DEVICES) && (pairedIndex < status->getU8NumPairedDevices()); ++pairedIndex)
   {
      if(true == status->getODeviceListExtendedResult()[pairedIndex].getBDeviceConnectedStatus())
      {
         if(!isDeviceConnected)
         {
            isDeviceConnected = true;
         }
         _connectedDeviceHandle[connectedIndex] = status->getODeviceListExtendedResult()[pairedIndex].getU8DeviceHandle();
         connectedIndex++;

         if(!isPANConnected)
         {
            if(most_BTSet_fi_types_Extended::T_e8_BTConnectionStatus__e8STATUS_CONNECTED ==
                  status->getODeviceListExtendedResult()[pairedIndex].getODeviceProfileConnectionStatus().getE8PAN())
            {
               isPANConnected = true;
               _panProfileDeviceHandle = status->getODeviceListExtendedResult()[pairedIndex].getU8DeviceHandle();
               _queuedDeviceHandleForPanRequest = 0;
            }
         }
      }
   }
   ETG_TRACE_USR4(("BTPhoneClientHandler::onDeviceListExtendedStatus(), Connected devices(%d, %d)", _connectedDeviceHandle[0], _connectedDeviceHandle[1]));
   sendBTConnectedDeviceUpdateMsg(isDeviceConnected);
   if(_isPANConnected != isPANConnected)
   {
      _isPANConnected = isPANConnected;
      ETG_TRACE_USR4(("BTPhoneClientHandler::onDeviceListExtendedStatus(), PAN profile device handle : %d, bPAN : %d", _panProfileDeviceHandle, _isPANConnected));

      if((true == _isPANConnectionRequired))
      {
         POST_MSG((COURIER_MESSAGE_NEW(PANConnectionStatusMsg)(_isPANConnected)));
      }
   }
}


bool BTPhoneClientHandler::onCourierMessage(const BTPhoneDialReqMsg& oMsg)
{
   ETG_TRACE_USR4(("BTPhoneClientHandler::onCourierMessage(BTPhoneDialReqMsg), Dialed number : %d", oMsg.GetTelNumber().GetCString()));
   if ((true == _isServiceAvailable) && (0 != _activeDeviceHandle))
   {
      _btTelProxy->sendDialStart(*this, oMsg.GetTelNumber().GetCString(), (T_e8_TelEchoCancellationNoiseReductionSetting)0);
   }
   else
   {
      ETG_TRACE_USR4(("BTPhoneClientHandler::onCourierMessage(BTPhoneDialReqMsg), Dial failed due to service unavailable/device not connected"));
   }
   return true;
}


void BTPhoneClientHandler::sendBTActiveDeviceUpdateMsg(bool isAvailable)
{
   static bool isActiveDeviceAvailable = false;
   if(isActiveDeviceAvailable != isAvailable)
   {
      ETG_TRACE_USR4(("BTPhoneClientHandler::sendBTActiveDeviceUpdateMsg(%d)", isAvailable));
      isActiveDeviceAvailable = isAvailable;
      POST_MSG((COURIER_MESSAGE_NEW(BTActiveDeviceUpdateMsg)(isAvailable)));
   }
}


void BTPhoneClientHandler::sendBTConnectedDeviceUpdateMsg(bool isConnected)
{
   static bool isConnectedDeviceAvailable = false;
   if(isConnectedDeviceAvailable != isConnected)
   {
      ETG_TRACE_USR4(("BTPhoneClientHandler::sendBTConnectedDeviceUpdateMsg(%d)", isConnected));
      isConnectedDeviceAvailable = isConnected;
      POST_MSG((COURIER_MESSAGE_NEW(BTConnectedDeviceUpdateMsg)(isConnected)));
   }
}


void BTPhoneClientHandler::onDisconnectBluetoothProfileError(const ::boost::shared_ptr< ::MOST_BTSet_FI::MOST_BTSet_FIProxy >& /*proxy*/,
      const ::boost::shared_ptr< ::MOST_BTSet_FI::DisconnectBluetoothProfileError >& /*error*/)
{
   ETG_TRACE_ERR(("BTPhoneClientHandler::onDisconnectBluetoothProfileError()"));
}


void BTPhoneClientHandler::onDisconnectBluetoothProfileResult(const ::boost::shared_ptr< ::MOST_BTSet_FI::MOST_BTSet_FIProxy >& /* proxy*/,
      const ::boost::shared_ptr< ::MOST_BTSet_FI::DisconnectBluetoothProfileResult >& result)
{
   ETG_TRACE_USR2(("BTPhoneClientHandler::onDisconnectBluetoothProfileResult(), Device handle : %d, bHFP : %d, bAVP : %d, bPAN : %d, bPBAP : %d",
         result->getU8DeviceHandle(),
         result->getOProfileDisconnectionStatus().getBHFP(),
         result->getOProfileDisconnectionStatus().getBAVP(),
         result->getOProfileDisconnectionStatus().getBPAN(),
         result->getOProfileDisconnectionStatus().getBPBAP()));
}


void BTPhoneClientHandler::onConnectBluetoothProfileError(const ::boost::shared_ptr< ::MOST_BTSet_FI::MOST_BTSet_FIProxy >& /*proxy*/,
      const ::boost::shared_ptr< ::MOST_BTSet_FI::ConnectBluetoothProfileError >& error)
{
   ETG_TRACE_ERR(("BTPhoneClientHandler::onConnectBluetoothProfileError()"));

   if((true == _isPANConnectionRequired) && (false == _isPANConnected) && (0 != _queuedDeviceHandleForPanRequest))
   {
      setPANProfileON(_queuedDeviceHandleForPanRequest);
      _queuedDeviceHandleForPanRequest = 0;
   }
}


void BTPhoneClientHandler::onConnectBluetoothProfileResult(const ::boost::shared_ptr< ::MOST_BTSet_FI::MOST_BTSet_FIProxy >& /*proxy*/,
      const ::boost::shared_ptr< ::MOST_BTSet_FI::ConnectBluetoothProfileResult >& result)
{
   ETG_TRACE_USR2(("BTPhoneClientHandler::onConnectBluetoothProfileResult(), Device handle : %d, bHFP : %d, bAVP : %d, bPAN : %d, bPBAP : %d",
         result->getU8DeviceHandle(),
         result->getOProfileConnectionStatus().getBHFP(),
         result->getOProfileConnectionStatus().getBAVP(),
         result->getOProfileConnectionStatus().getBPAN(),
         result->getOProfileConnectionStatus().getBPBAP()));

   if((true == result->getOProfileConnectionStatus().getBPAN()) && (false == _isPANConnected))
   {
      _isPANConnected = true;
      _panProfileDeviceHandle = result->getU8DeviceHandle();
      _queuedDeviceHandleForPanRequest = 0;
      if(true == _isPANConnectionRequired)
      {
         POST_MSG((COURIER_MESSAGE_NEW(PANConnectionStatusMsg)(true)));
      }
   }
   else if((false == result->getOProfileConnectionStatus().getBPAN()) && (false == _isPANConnected)
           && (true == _isPANConnectionRequired) && (0 != _queuedDeviceHandleForPanRequest))
   {
      setPANProfileON(_queuedDeviceHandleForPanRequest);
      _queuedDeviceHandleForPanRequest = 0;
   }
}


void BTPhoneClientHandler::onActivePhoneDeviceError(const ::boost::shared_ptr< MOST_Tel_FIProxy >& proxy,
      const ::boost::shared_ptr< ActivePhoneDeviceError >& error)
{
   ETG_TRACE_ERR(("BTPhoneClientHandler::onActivePhoneDeviceError"));
}


void BTPhoneClientHandler::onActivePhoneDeviceStatus(const ::boost::shared_ptr< MOST_Tel_FIProxy >& proxy,
      const ::boost::shared_ptr< ActivePhoneDeviceStatus >& status)
{
   _activeDeviceHandle = status->getU8DeviceHandle();
   ETG_TRACE_USR4(("BTPhoneClientHandler::onActivePhoneDeviceStatus(), Active device handle : %d", _activeDeviceHandle));

   bool isActiveDeviceAvailable = (0 != _activeDeviceHandle) ? true : false;
   sendBTActiveDeviceUpdateMsg(isActiveDeviceAvailable);
}


void BTPhoneClientHandler::onServiceAvailableError(const ::boost::shared_ptr< MOST_BTSet_FIProxy >& proxy, const ::boost::shared_ptr< ServiceAvailableError >& error)
{
   ETG_TRACE_USR4(("BTPhoneClientHandler::onServiceAvailableError"));
}


void BTPhoneClientHandler::onServiceAvailableStatus(const ::boost::shared_ptr< MOST_BTSet_FIProxy >& proxy, const ::boost::shared_ptr< ServiceAvailableStatus >& status)
{
   ETG_TRACE_USR4(("BTPhoneClientHandler::onServiceAvailableStatus(), Status = %d", status->getE8ServiceAvailable()));

   _isServiceAvailable = (status->getE8ServiceAvailable() == most_BTSet_fi_types_Extended::T_e8_BTServiceAvailable__e8SERVICE_AVAILABLE) ? true : false;

   if(false == _isServiceAvailable)
   {
      sendBTActiveDeviceUpdateMsg(false);
      sendBTConnectedDeviceUpdateMsg(false);
   }
}


bool BTPhoneClientHandler::onCourierMessage(const StartStopBtTetheringReqMsg& oMsg)
{
   ETG_TRACE_USR4(("BTPhoneClientHandler::onCourierMessage(StartStopBtTetheringReqMsg), isTurnOnPAN : %d", oMsg.GetTurnOnPAN()));

   _isPANConnectionRequired = oMsg.GetTurnOnPAN();
   if(true == _isPANConnectionRequired)
   {
      int index = 0;
      if(0 != _connectedDeviceHandle[index])
      {
         setPANProfileON(_connectedDeviceHandle[index]);
         index++;
         if(0 != _connectedDeviceHandle[index])
         {
            _queuedDeviceHandleForPanRequest = _connectedDeviceHandle[index];
            ETG_TRACE_USR4(("BTPhoneClientHandler::onCourierMessage(StartStopBtTetheringReqMsg), Queued PAN connection request to device handle : %d", _queuedDeviceHandleForPanRequest));
         }
      }
   }
   else
   {
      setPANProfileOFF();
   }
   return true;
}


void BTPhoneClientHandler::setPANProfileON(uint8 deviceHandle)
{
   if(false == _isPANConnected)
   {
      if (NULL != _btSetProxy.get())
      {
         ETG_TRACE_USR4(("BTPhoneClientHandler::setPANProfileON(), Device handle : %d", deviceHandle));
         ::most_BTSet_fi_types_Extended::T_BTServiceType serviceType;
         serviceType.setBPAN(true);
         _btSetProxy->sendConnectBluetoothProfileStart(*this, deviceHandle, (const ::most_BTSet_fi_types_Extended::T_BTServiceType)serviceType);
      }
   }
   else
   {
      ETG_TRACE_USR4(("BTPhoneClientHandler::setPANProfileON(), PAN already connected in device handle : %d", _panProfileDeviceHandle));
      POST_MSG((COURIER_MESSAGE_NEW(PANConnectionStatusMsg)(true)));
   }
}


void BTPhoneClientHandler::setPANProfileOFF()
{
   if(true == _isPANConnected)
   {
      if ((NULL != _btSetProxy.get()) && (0 != _panProfileDeviceHandle))
      {
         ETG_TRACE_USR4(("BTPhoneClientHandler::setPANProfileOFF(), Device handle : %d", _panProfileDeviceHandle));
         ::most_BTSet_fi_types_Extended::T_BTServiceType serviceType;
         serviceType.setBPAN(false);
         _btSetProxy->sendDisconnectBluetoothProfileStart(*this, _panProfileDeviceHandle, (const ::most_BTSet_fi_types_Extended::T_BTServiceType)serviceType);

         _panProfileDeviceHandle = 0;
         _isPANConnected = false;
      }
   }
}


} // namespace Core
} // namespace App
