/**************************************************************************************
* @file         : AudioSourceHandler.cpp
* @author       : RBEI/ECG5-INF4CV_NavigationTeam
* @addtogroup   : AppHmi_navigation
* @brief        :
* @copyright    : (c) 2017-2019 Robert Bosch Car Multimedia GmbH
*                 The reproduction, distribution and utilization of this file as
*                 well as the communication of its contents to others without express
*                 authorization is prohibited. Offenders will be held liable for the
*                 payment of damages. All rights reserved in the event of the grant
*                 of a patent, utility model or design.
**************************************************************************************/
#include "hall_std_if.h"
#include "App/Core/AudioSource/AudioSourceHandler.h"

#include "AppHmi_MasterBase/AudioInterface/AudioDefines.h"

#include "AppBase/ScreenBrokerClient/ScreenBrokerClient.h"


#ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
#define ETG_DEFAULT_TRACE_CLASS           TR_CLASS_APPHMI_NAVIGATION_HALL
#define ETG_I_TRACE_CHANNEL               TR_TTFIS_APPHMI_NAVIGATION
#define ETG_I_TTFIS_CMD_PREFIX            "APPHMI_NAVIGATION_"
#define ETG_I_FILE_PREFIX                 App::Core::AudioSourceHandler::
#include "trcGenProj/Header/AudioSourceHandler.cpp.trc.h"
#endif

using namespace MASTERAUDIOSERVICE_INTERFACE::AudioSourceChange;
using namespace MASTERAUDIOSERVICE_INTERFACE::SoundProperties;
using namespace org::genivi::audiomanager::CommandInterface;

namespace App {
namespace Core {

AudioSourceHandler* AudioSourceHandler::_audioSourceHandler = NULL;

AudioSourceHandler* AudioSourceHandler::getInstance()
{
   if (_audioSourceHandler == 0)
   {
      _audioSourceHandler = new AudioSourceHandler();
   }

   return _audioSourceHandler;
}


/**
 * @Destructor
 */
AudioSourceHandler::~AudioSourceHandler()
{
   if (_audioSourceChangeProxy != NULL)
   {
      _audioSourceChangeProxy.reset();
   }

   if (_soundPropertiesProxy != NULL)
   {
      _soundPropertiesProxy.reset();
   }

   if (_commandIFProxy != NULL)
   {
      _commandIFProxy.reset();
   }

   if (_audioSourceHandler != NULL)
   {
      delete _audioSourceHandler;
      _audioSourceHandler = NULL;
   }
}


AudioSourceHandler::AudioSourceHandler()
   : _audioSourceChangeProxy(AudioSourceChangeProxy::createProxy("audioSourceChangePort", *this))
   , _soundPropertiesProxy(SoundPropertiesProxy::createProxy("soundPropertiesPort", *this))
   , _commandIFProxy(::CommandInterfaceProxy::createProxy("commandInterfacePort", *this))
   , m_u8CurrentDeviceTag(0)
{
   StartupSync::getInstance().registerPropertyRegistrationIF(this, "audioSourceChangePort");
   StartupSync::getInstance().registerPropertyRegistrationIF(this, "soundPropertiesPort");
   StartupSync::getInstance().registerPropertyRegistrationIF(this, "commandInterfacePort");

   ETG_I_REGISTER_FILE();
   _audioSourceHandler = this;
   m_u32RequestedSubSrcId = 0;
}


void AudioSourceHandler::onAvailable(const ::boost::shared_ptr< ::asf::core::Proxy >& proxy, const ::asf::core::ServiceStateChange& stateChange)
{
   ETG_TRACE_USR4(("AudioSourceHandler::onAvailable"));
   StartupSync::getInstance().onAvailable(proxy, stateChange);
}


void AudioSourceHandler::onUnavailable(const ::boost::shared_ptr< ::asf::core::Proxy >& proxy, const ::asf::core::ServiceStateChange& stateChange)
{
   ETG_TRACE_USR4(("AudioSourceHandler::onUnavailable"));
   StartupSync::getInstance().onUnavailable(proxy, stateChange);
}


/**
 * registerProperties - Trigger property registration to hmi_master,  called from NavigationHall class
 * @param[in] proxy
 * @parm[in] stateChange - state change service for corrosponding  proxy
 * @return void
 */
void AudioSourceHandler::registerProperties(const ::boost::shared_ptr< asf::core::Proxy >& proxy, const asf::core::ServiceStateChange& statechange)
{
   if (_audioSourceChangeProxy && _audioSourceChangeProxy == proxy)
   {
      _audioSourceChangeProxy->sendSourceListChangedRegister(*this);
      _audioSourceChangeProxy->sendGetSourceListRequest(*this, 4); //4 is group id for navigation i.e GROUP_OTHERS
      _audioSourceChangeProxy->sendActiveSourceListRegister(*this);
      _audioSourceChangeProxy->sendActiveSourceRegister(*this);
      _audioSourceChangeProxy->sendActiveSourceGet(*this);
   }
   else if (_soundPropertiesProxy && _soundPropertiesProxy == proxy)
   {
      _soundPropertiesProxy->sendVolumeRegister(*this);
      _soundPropertiesProxy->sendVolumeGet(*this);
      _soundPropertiesProxy->sendMuteStateRegister(*this);
      _soundPropertiesProxy->sendMuteStateGet(*this);
      _soundPropertiesProxy->sendMainSinkSoundPropertyChangedRegister(*this);
   }
   else
   {
      //do nothing.
   }
}


/**
 * deregisterProperties - Trigger property deregistration to hmi_master,  called from NavigationHall class
 * @param[in] proxy
 * @parm[in] stateChange - state change service for corrosponding  proxy
 * @return void
 */
void AudioSourceHandler::deregisterProperties(const ::boost::shared_ptr< asf::core::Proxy >& proxy, const asf::core::ServiceStateChange& stateChange)
{
   if (_audioSourceChangeProxy && _audioSourceChangeProxy == proxy)
   {
      _audioSourceChangeProxy->sendDeregisterAll();
   }
   else if (_soundPropertiesProxy && _soundPropertiesProxy == proxy)
   {
      _soundPropertiesProxy->sendVolumeDeregisterAll();
      _soundPropertiesProxy->sendMuteStateDeregisterAll();
      _soundPropertiesProxy->sendMainSinkSoundPropertyChangedDeregisterAll();
   }
   else
   {
      //do nothing.
   }
}


/**
 * onActivateSourceError - Notification from HMI master when source requested for activation is unsuccessful
 * @param[in] proxy
 * @parm[in] error
 * @return void
 */
void AudioSourceHandler::onActivateSourceError(const ::boost::shared_ptr< MASTERAUDIOSERVICE_INTERFACE::AudioSourceChange::AudioSourceChangeProxy >& proxy,
      const ::boost::shared_ptr< MASTERAUDIOSERVICE_INTERFACE::AudioSourceChange::ActivateSourceError >& error)
{
   ETG_TRACE_FATAL(("AudioSourceHandler:: onActivateSourceError() \n"));
}


/**
 * onActivateSourceResponse - Notification from HMI master when source requested for activation is successful
 * @param[in] proxy
 * @parm[in] response
 * @return void
 */
void AudioSourceHandler::onActivateSourceResponse(const ::boost::shared_ptr< MASTERAUDIOSERVICE_INTERFACE::AudioSourceChange::AudioSourceChangeProxy >& proxy,
      const ::boost::shared_ptr< MASTERAUDIOSERVICE_INTERFACE::AudioSourceChange::ActivateSourceResponse >& /*response*/)
{
   ETG_TRACE_USR4(("AudioSourceHandler:: onActivateSourceResponse() \n"));
   //current active device
}


/**
 * onDeactivateSourceError - Notification from HMI master when request for source deactivation is not successful
 * @param[in] proxy
 * @parm[in] error
 * @return void
 */
void AudioSourceHandler::onDeactivateSourceError(const ::boost::shared_ptr< MASTERAUDIOSERVICE_INTERFACE::AudioSourceChange::AudioSourceChangeProxy >& /*proxy*/,
      const ::boost::shared_ptr< MASTERAUDIOSERVICE_INTERFACE::AudioSourceChange::DeactivateSourceError >& /*error*/)
{
   ETG_TRACE_FATAL(("AudioSourceHandler:: onDeactivateSourceError() \n"));
}


/**
 * onDeactivateSourceResponse - Notification from HMI master when request for source deactivation is successful
 * @param[in] proxy
 * @parm[in] response
 * @return void
 */
void AudioSourceHandler::onDeactivateSourceResponse(const ::boost::shared_ptr< MASTERAUDIOSERVICE_INTERFACE::AudioSourceChange::AudioSourceChangeProxy >& /*proxy*/,
      const ::boost::shared_ptr< MASTERAUDIOSERVICE_INTERFACE::AudioSourceChange::DeactivateSourceResponse >& /*response*/)
{
   ETG_TRACE_USR4(("AudioSourceHandler:: onDeactivateSourceResponse() \n"));
}


/**
 * onSourceListChangedError - Error notification from HMI master when sourceListChange error happens
 * @param[in] proxy
 * @parm[in] error
 * @return void
 */
void AudioSourceHandler::onSourceListChangedError(const ::boost::shared_ptr< MASTERAUDIOSERVICE_INTERFACE::AudioSourceChange::AudioSourceChangeProxy >& /*proxy*/,
      const ::boost::shared_ptr< MASTERAUDIOSERVICE_INTERFACE::AudioSourceChange::SourceListChangedError >& /*error*/)
{
   ETG_TRACE_USR4(("AudioSourceHandler:: onSourceListChangedError() \n"));
}


/**
 * onSourceListChangedSignal - Notification from HMI master when SourceListChange is obtained
 * @param[in] proxy
 * @parm[in] signal
 * @return void
 */
void AudioSourceHandler::onSourceListChangedSignal(const ::boost::shared_ptr< MASTERAUDIOSERVICE_INTERFACE::AudioSourceChange::AudioSourceChangeProxy >& /*proxy*/,
      const ::boost::shared_ptr< MASTERAUDIOSERVICE_INTERFACE::AudioSourceChange::SourceListChangedSignal >& signal)
{
   ETG_TRACE_USR4(("AudioSourceHandler:: onSourceListChangedSignal() \n"));
   if (_audioSourceChangeProxy && signal->getGroupId() == GROUP_OTHERS)
   {
      ETG_TRACE_USR4(("sendGetSourceListRequest for navigation"));
      //Request for the changed source list
      //_audioSourceChangeProxy->sendGetSourceListRequest(*this, GROUP_OTHERS); //TODO:need to check group navigation id
   }
}


/**
 * onGetSourceListError - Notification from HMI master when SourceListChange request is unsuccessful
 * @param[in] proxy
 * @parm[in] error
 * @return void
 */
void AudioSourceHandler::onGetSourceListError(const ::boost::shared_ptr< MASTERAUDIOSERVICE_INTERFACE::AudioSourceChange::AudioSourceChangeProxy >& /*proxy*/,
      const ::boost::shared_ptr< MASTERAUDIOSERVICE_INTERFACE::AudioSourceChange::GetSourceListError >& /*error*/)
{
   ETG_TRACE_USR4(("AudioSourceHandler:: onGetSourceListError() \n"));
}


/**
 * onGetSourceListResponse - Notification from HMI master when request for source list is successful
 * @param[in] proxy
 * @parm[in] signal
 * @return void
 */
void AudioSourceHandler::onGetSourceListResponse(const ::boost::shared_ptr< MASTERAUDIOSERVICE_INTERFACE::AudioSourceChange::AudioSourceChangeProxy >& proxy,
      const ::boost::shared_ptr< MASTERAUDIOSERVICE_INTERFACE::AudioSourceChange::GetSourceListResponse >& response)
{
   ETG_TRACE_USR4(("AudioSourceHandler:: onGetSourceListResponse() \n"));

   if (_audioSourceChangeProxy == proxy)
   {
      const ::std::vector< sourceDetails >& SourceList = response->getSources();
      ::std::vector< sourceDetails >::const_iterator it = SourceList.begin();

      ETG_TRACE_USR4(("onGetSourceListResponse Sourcelistsize:%d", SourceList.size()));

      while (it != SourceList.end())
      {
         ETG_TRACE_USR4(("onGetSourceListResponse"));
         if ((it->getSrcId() == SRC_NAVI_SPEECH) && (static_cast<int32>(it->getSubSrcId()) == m_u32RequestedSubSrcId) && (it->getAvailability() == 1))
         {
            sourceData sourceInfo(it->getSrcId(), it->getSubSrcId(), COCKPIT_SINK_ID, 0);
            act_t token = _audioSourceChangeProxy->sendActivateSourceRequest(*this, sourceInfo, true);
            ETG_TRACE_USR4(("onGetSourceListResponse Acknowledgement token : %d", token));
            break;
         }

         ++it;
      } // End of while loop
   }
}


/**
 * vRequestSourceActivation - helper function tused by Navigation classes to request a source activation
 * @param[in] SrcID, SubSrcID - source id and sub source id
 * @return void
 */
void AudioSourceHandler::requestSourceActivation(int32 SrcID, int32 SubSrcID)
{
   ETG_TRACE_USR4(("vRequestSourceActivation SrcID %d SubSrcID %d", SrcID, SubSrcID));
   m_u32RequestedSubSrcId = SubSrcID;
   _audioSourceChangeProxy->sendGetSourceListRequest(*this, GROUP_OTHERS); //TODO:need to check group navigation id
}


/**
 * onActiveSourceError - Notification from HMI master when  active source property update is unsuccessful
 * @param[in] proxy, error
 * @return void
 */
void AudioSourceHandler::onActiveSourceError(const ::boost::shared_ptr< MASTERAUDIOSERVICE_INTERFACE::AudioSourceChange::AudioSourceChangeProxy >& /*proxy*/,
      const ::boost::shared_ptr< MASTERAUDIOSERVICE_INTERFACE::AudioSourceChange::ActiveSourceError >& /*error*/)
{
   ETG_TRACE_USR4(("AudioSourceHandler:: onActiveSourceError() "));
}


/**
 * onActiveSourceUpdate - Notification from HMI master when  active source property update is successful
 * @param[in] proxy, update
 * @return void
 */
void AudioSourceHandler::onActiveSourceUpdate(const ::boost::shared_ptr< MASTERAUDIOSERVICE_INTERFACE::AudioSourceChange::AudioSourceChangeProxy >& /*proxy*/,
      const ::boost::shared_ptr< MASTERAUDIOSERVICE_INTERFACE::AudioSourceChange::ActiveSourceUpdate >& update)
{
   ETG_TRACE_USR4(("AudioSourceHandler:: onActiveSourceUpdate():update->getActiveSource().getSrcId():%d", update->getActiveSource().getSrcId()));
   ETG_TRACE_USR4(("AudioSourceHandler:: onActiveSourceUpdate():update->getActiveSource().getSubSrcId():%d", update->getActiveSource().getSubSrcId()));
   //if source id other than navigationplayed then set the subsrc as 255
   if (update->getActiveSource().getSrcId() == SRC_NAVI_SPEECH)
   {
      setCurrentDeviceTag(update->getActiveSource().getSubSrcId());
   }
   else
   {
      setCurrentDeviceTag(0);
   }
}


inline void AudioSourceHandler::setCurrentDeviceTag(int8 u8CurrentDeviceTag)
{
   m_u8CurrentDeviceTag = u8CurrentDeviceTag;
}


inline int8 AudioSourceHandler::getCurrentDeviceTag()
{
   return m_u8CurrentDeviceTag;
}


/**
* onSetMainSinkSoundPropertyError - Error handling for SetMainSinkSoundProperty
* @param[in] proxy
* @param[in] error
* @param[out] none
* @return void
*/
void AudioSourceHandler::onSetMainSinkSoundPropertyError(const ::boost::shared_ptr< MASTERAUDIOSERVICE_INTERFACE::SoundProperties::SoundPropertiesProxy  >& /*proxy*/,
      const ::boost::shared_ptr< MASTERAUDIOSERVICE_INTERFACE::SoundProperties::SetMainSinkSoundPropertyError >& /*error*/)
{
   ETG_TRACE_USR4(("AudioSourceHandler:: onSetMainSinkSoundPropertyError() \n"));
}


/**
* onSetMainSinkSoundPropertyResponse - response for SetMainSinkSoundProperty
* @param[in] proxy
* @param[in] error
* @param[out] none
* @return void
*/
void AudioSourceHandler::onSetMainSinkSoundPropertyResponse(const ::boost::shared_ptr< MASTERAUDIOSERVICE_INTERFACE::SoundProperties::SoundPropertiesProxy  >& /*proxy*/,
      const ::boost::shared_ptr< MASTERAUDIOSERVICE_INTERFACE::SoundProperties::SetMainSinkSoundPropertyResponse >& /*response*/)
{
   ETG_TRACE_USR4(("AudioSourceHandler:: onSetMainSinkSoundPropertyResponse() \n"));
}


/**
* onMainSinkSoundPropertyChangedError - Error handling for MainSinkSoundPropertyChanged
* @param[in] proxy
* @param[in] error
* @param[out] none
* @return void
*/
void AudioSourceHandler::onMainSinkSoundPropertyChangedError(const ::boost::shared_ptr<MASTERAUDIOSERVICE_INTERFACE::SoundProperties::SoundPropertiesProxy  >& /*proxy*/,
      const ::boost::shared_ptr<MASTERAUDIOSERVICE_INTERFACE::SoundProperties::MainSinkSoundPropertyChangedError >& /*error*/)
{
   ETG_TRACE_USR4(("AudioSourceHandler:: onMainSinkSoundPropertyChangedError() \n"));
}


/**
 * onMainSinkSoundPropertyChangedSignal - Status/Signal handling for MainSinkSoundPropertyChanged
 * @param[in] proxy
 * @param[in] error
 * @param[out] none
 * @return void
 */
void AudioSourceHandler::onMainSinkSoundPropertyChangedSignal(const ::boost::shared_ptr<MASTERAUDIOSERVICE_INTERFACE::SoundProperties::SoundPropertiesProxy  >& /*proxy*/,
      const ::boost::shared_ptr<MASTERAUDIOSERVICE_INTERFACE::SoundProperties::MainSinkSoundPropertyChangedSignal >& signal)
{
   ETG_TRACE_USR4(("AudioSourceHandler:: onMainSinkSoundPropertyChangedSignal() \n"));
}


/**
* onSetVolumeError - Error handling for SetVolume
* @param[in] proxy
* @param[in] error
* @param[out] none
* @return void
*/
void AudioSourceHandler::onSetVolumeError(const ::boost::shared_ptr< AUDIOMANAGER_COMMANDINTERFACE::CommandInterfaceProxy >& /*proxy*/,
      const ::boost::shared_ptr< AUDIOMANAGER_COMMANDINTERFACE::SetVolumeError >& /*error*/)
{
   ETG_TRACE_USR4(("AudioSourceHandler:: onSetVolumeError() \n"));
}


/**
 * onSetVolumeResponse - Response handling for SetVolume
 * @param[in] proxy
 * @param[in] error
 * @param[out] none
 * @return void
 */
void AudioSourceHandler::onSetVolumeResponse(const ::boost::shared_ptr< AUDIOMANAGER_COMMANDINTERFACE::CommandInterfaceProxy >& /*proxy*/,
      const ::boost::shared_ptr< AUDIOMANAGER_COMMANDINTERFACE::SetVolumeResponse >& /*response*/)
{
   ETG_TRACE_USR4(("AudioSourceHandler:: onSetVolumeResponse() \n"));
}


/**
 * onVolumeError - This status is received from HMI_Master if there are any errors in volume.
 * @param[in] proxy
 * @parm[in] status
 * @return void
 */
void AudioSourceHandler::onVolumeError(const ::boost::shared_ptr<MASTERAUDIOSERVICE_INTERFACE::SoundProperties::SoundPropertiesProxy >& /*proxy*/,
                                       const ::boost::shared_ptr<MASTERAUDIOSERVICE_INTERFACE::SoundProperties::VolumeError >& /*error*/)
{
   ETG_TRACE_USR4(("AudioSourceHandler:: onVolumeError() \n"));
}


/**
 * onVolumeUpdate - This status is received from HMI_Master if there is any change in property volume.
 * @param[in] proxy
 * @parm[in] status
 * @return void
 */
void AudioSourceHandler::onVolumeUpdate(const ::boost::shared_ptr<MASTERAUDIOSERVICE_INTERFACE::SoundProperties::SoundPropertiesProxy >& /*proxy*/,
                                        const ::boost::shared_ptr<MASTERAUDIOSERVICE_INTERFACE::SoundProperties::VolumeUpdate >& update)
{
   ETG_TRACE_USR4(("AudioSourceHandler:: onVolumeUpdate() \n"));
}


/**
 * onMuteStateError - This status is received from HMI_Master if there are any errors in muteState.
 * @param[in] proxy
 * @parm[in] status
 * @return void
 */
void AudioSourceHandler::onMuteStateError(const ::boost::shared_ptr< MASTERAUDIOSERVICE_INTERFACE::SoundProperties::SoundPropertiesProxy >& /*proxy*/,
      const ::boost::shared_ptr< MASTERAUDIOSERVICE_INTERFACE::SoundProperties::MuteStateError >& /*error*/)
{
   ETG_TRACE_USR4(("AudioSourceHandler:: onMuteStateError() \n"));
}


/**
 * onMuteStateUpdate - This status is received from HMI_Master if there is any change in property mutestate.
 * @param[in] proxy
 * @parm[in] status
 * @return void
 */
void AudioSourceHandler::onMuteStateUpdate(const ::boost::shared_ptr< MASTERAUDIOSERVICE_INTERFACE::SoundProperties::SoundPropertiesProxy >& /*proxy*/,
      const ::boost::shared_ptr< MASTERAUDIOSERVICE_INTERFACE::SoundProperties::MuteStateUpdate >& update)
{
   ETG_TRACE_USR4(("AudioSourceHandler:: onMuteStateUpdate() \n"));
}


/**
 * onSetMuteStateError - called if SetMuteState method is updated with error from navigation player
 *
 * @param[in] proxy
 * @parm[in] error
 * @return void
 */
void AudioSourceHandler::onSetMuteStateError(const ::boost::shared_ptr< MASTERAUDIOSERVICE_INTERFACE::SoundProperties::SoundPropertiesProxy >& /*proxy*/,
      const ::boost::shared_ptr< MASTERAUDIOSERVICE_INTERFACE::SoundProperties::SetMuteStateError >& /*error*/)
{
   ETG_TRACE_USR4(("AudioSourceHandler:: onSetMuteStateError() \n"));
}


/**
 * onSetMuteStateResponse - Receives info about the set MuteState response and updates the same on video main scene
 *
 * @param[in] proxy
 * @parm[in] onMuteStateUpdate
 * @return void
 */
void AudioSourceHandler::onSetMuteStateResponse(const ::boost::shared_ptr< MASTERAUDIOSERVICE_INTERFACE::SoundProperties::SoundPropertiesProxy >& /*proxy*/,
      const ::boost::shared_ptr< MASTERAUDIOSERVICE_INTERFACE::SoundProperties::SetMuteStateResponse >& response)
{
   ETG_TRACE_USR4(("AudioSourceHandler:: onSetMuteStateResponse() \n"));
}


/**
 * onActiveSourceListError - This status is received from HMI_Master if there are any errors in ActiveSourceList request.
 *
 * @param[in] proxy
 * @parm[in] error
 * @return void
 */
void AudioSourceHandler::onActiveSourceListError(const ::boost::shared_ptr< MASTERAUDIOSERVICE_INTERFACE::AudioSourceChange::AudioSourceChangeProxy >& proxy,
      const ::boost::shared_ptr< MASTERAUDIOSERVICE_INTERFACE::AudioSourceChange::ActiveSourceListError >& error)
{
   ETG_TRACE_USR4(("AudioSourceHandler:: onActiveSourceListError() \n"));
}


/**
 * onActiveSourceListUpdate - This status is received from HMI_Master on ActiveSourceList request.
 *
 * @param[in] proxy
 * @parm[in] status
 * @return void
 */
void AudioSourceHandler::onActiveSourceListUpdate(const ::boost::shared_ptr< MASTERAUDIOSERVICE_INTERFACE::AudioSourceChange::AudioSourceChangeProxy >& proxy,
      const ::boost::shared_ptr< MASTERAUDIOSERVICE_INTERFACE::AudioSourceChange::ActiveSourceListUpdate >& signal)
{
   ETG_TRACE_USR4(("AudioSourceHandler:: onActiveSourceListUpdate() \n"));
}


} // namespace Core
} // namespace App
