/**************************************************************************************
* @file         : AudioClientHandler.cpp
* @author       : ECG5-Atchuta Sashank Kappagantu
* @addtogroup   : AppHmi_Navigation
* @brief        :
* @copyright    : (c) -2018 Robert Bosch Car Multimedia GmbH
*                 The reproduction, distribution and utilization of this file as
*                 well as the communication of its contents to others without express
*                 authorization is prohibited. Offenders will be held liable for the
*                 payment of damages. All rights reserved in the event of the grant
*                 of a patent, utility model or design.
**************************************************************************************/

#include "hall_std_if.h"
#include "audiomanagertypes.h"                             // System/sound property type enums and #defines
#include "AppHmi_MasterBase/AudioInterface/AudioDefines.h" // Defines for sink IDs
#include "AudioClientHandler.h"

#ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
#define ETG_DEFAULT_TRACE_CLASS TR_CLASS_APPHMI_NAVIGATION_HALL
#define ETG_I_TRACE_CHANNEL TR_TTFIS_APPHMI_NAVIGATION
#define ETG_I_TTFIS_CMD_PREFIX "APPHMI_NAVIGATION_"
#define ETG_I_FILE_PREFIX App::Core::AudioClientHandler::
#include "trcGenProj/Header/AudioClientHandler.cpp.trc.h"
#endif

const uint16 NAVI_SOUND_PROPERTY_MIN_VALUE = 5;
const uint16 NAVI_SOUND_PROPERTY_MAX_VALUE = 40;
const uint16 VOLUME_VALUE_INVALID = 0xFF;

using namespace ::asf::core;
using namespace ::mascf_main_fi;
using namespace COMMAND_INTERFACE;
using namespace ::bosch::cm::ai::hmi::masteraudioservice::AudioSourceChange;
using namespace ::bosch::cm::ai::hmi::masteraudioservice::SoundProperties;

namespace App {
namespace Core {

AudioClientHandler::AudioClientHandler(const std::string& commandInterfacePort, const std::string& mascfFiPort)
   : _commandIFProxy(CommandInterfaceProxy::createProxy(commandInterfacePort, *this))
   , _mascfFiProxy(Mascf_main_fiProxy::createProxy(mascfFiPort, *this))
   , _volumeLevelCockpit(VOLUME_VALUE_INVALID)
   , _volumeLevelPrivate(VOLUME_VALUE_INVALID)
   , _sinkId(SINK_INVALID)
{
   _audioSourceChangeProxy = (::AudioSourceChangeProxy::createProxy("audioSourceChangePort", *this));
   _soundPropertiesProxy = (::SoundPropertiesProxy::createProxy("soundPropertiesPort", *this));
   if (_commandIFProxy.get())
   {
      StartupSync::getInstance().registerPropertyRegistrationIF(this, _commandIFProxy->getPortName());
   }
   if (_mascfFiProxy.get())
   {
      StartupSync::getInstance().registerPropertyRegistrationIF(this, _mascfFiProxy->getPortName());
   }
   if (_audioSourceChangeProxy.get())
   {
      StartupSync::getInstance().registerPropertyRegistrationIF(this, _audioSourceChangeProxy->getPortName());
   }
   if (_soundPropertiesProxy.get())
   {
      StartupSync::getInstance().registerPropertyRegistrationIF(this, _soundPropertiesProxy->getPortName());
   }
}


AudioClientHandler::~AudioClientHandler()
{
}


void AudioClientHandler::onAvailable(const ::boost::shared_ptr< ::asf::core::Proxy >& proxy, const ::asf::core::ServiceStateChange& stateChange)
{
   ETG_TRACE_USR4(("AudioClientHandler::onAvailable"));
   StartupSync::getInstance().onAvailable(proxy, stateChange);
}


void AudioClientHandler::onUnavailable(const ::boost::shared_ptr< ::asf::core::Proxy >& proxy, const ::asf::core::ServiceStateChange& stateChange)
{
   ETG_TRACE_USR4(("AudioClientHandler::onUnavailable"));
   StartupSync::getInstance().onUnavailable(proxy, stateChange);
}


void AudioClientHandler::registerProperties(const ::boost::shared_ptr< Proxy >& proxy, const ServiceStateChange& stateChange)
{
   if (_soundPropertiesProxy == proxy)
   {
      ETG_TRACE_USR4(("AudioClientHandler::registerProperties()"));

      _soundPropertiesProxy->sendSystemPropertyListRegister(*this);           // Property update when RequestSystemPropertyList() is invoked
      _soundPropertiesProxy->sendSystemPropertyListGet(*this);                // Invoked to get systemPropertyList on startup

      _soundPropertiesProxy->sendSoundPropertyListRegister(*this);            // Property update when RequestMainSinkSoundPropertyList(sinkId) is invoked
      _soundPropertiesProxy->sendSoundPropertyListGet(*this);                 // Invoked to get soundPropertyList on startup

      // Invoked to get navigation guidance volume value on startup, in cockpit and private speaker sink
      _soundPropertiesProxy->sendRequestMainSinkSoundPropertyListRequest(COCKPIT_SINK_ID);
      _soundPropertiesProxy->sendRequestMainSinkSoundPropertyListRequest(PRIVATE_SPEAKER_SINK_ID);

      _soundPropertiesProxy->sendMainSinkSoundPropertyChangedRegister(*this); // Received when there is a change in sound property
      _soundPropertiesProxy->sendSystemPropertyChangedRegister(*this);        // Received when there is a change in system property
   }
}


void AudioClientHandler::deregisterProperties(const ::boost::shared_ptr< Proxy >& proxy, const ServiceStateChange& stateChange)
{
   if (_soundPropertiesProxy == proxy)
   {
      ETG_TRACE_USR4(("AudioClientHandler::deregisterProperties()"));

      _soundPropertiesProxy->sendSystemPropertyListDeregisterAll();
      _soundPropertiesProxy->sendSoundPropertyListDeregisterAll();
      _soundPropertiesProxy->sendMainSinkSoundPropertyChangedDeregisterAll();
      _soundPropertiesProxy->sendSystemPropertyChangedDeregisterAll();
   }
}


void AudioClientHandler::onSetMainSinkSoundPropertyError(const ::boost::shared_ptr< MASTERAUDIOSERVICE_INTERFACE::SoundProperties::SoundPropertiesProxy  >& /*proxy*/,
      const ::boost::shared_ptr< MASTERAUDIOSERVICE_INTERFACE::SoundProperties::SetMainSinkSoundPropertyError >& /*error*/)
{
   ETG_TRACE_ERR(("AudioClientHandler::onSetMainSinkSoundPropertyError()"));
}


void AudioClientHandler::onSetMainSinkSoundPropertyResponse(const ::boost::shared_ptr< MASTERAUDIOSERVICE_INTERFACE::SoundProperties::SoundPropertiesProxy  >& /*proxy*/,
      const ::boost::shared_ptr< MASTERAUDIOSERVICE_INTERFACE::SoundProperties::SetMainSinkSoundPropertyResponse >& /*response*/)
{
   ETG_TRACE_USR4(("AudioClientHandler::onSetMainSinkSoundPropertyResponse()"));
}


void AudioClientHandler::onMainSinkSoundPropertyChangedError(const ::boost::shared_ptr<MASTERAUDIOSERVICE_INTERFACE::SoundProperties::SoundPropertiesProxy  >& /*proxy*/,
      const ::boost::shared_ptr<MASTERAUDIOSERVICE_INTERFACE::SoundProperties::MainSinkSoundPropertyChangedError >& /*error*/)

{
   ETG_TRACE_USR4(("AudioClientHandler::onMainSinkSoundPropertyChangedError()"));
}


void AudioClientHandler::onMainSinkSoundPropertyChangedSignal(const ::boost::shared_ptr<MASTERAUDIOSERVICE_INTERFACE::SoundProperties::SoundPropertiesProxy  >& proxy,
      const ::boost::shared_ptr<MASTERAUDIOSERVICE_INTERFACE::SoundProperties::MainSinkSoundPropertyChangedSignal >& signal)
{
   if (_soundPropertiesProxy == proxy)
   {
      int sinkId       = signal->getSinkId();
      int propertyType = signal->getPropertyType();
      if (((sinkId == COCKPIT_SINK_ID) || (sinkId == PRIVATE_SPEAKER_SINK_ID)) && (propertyType == am::MSP_VOLUME_NAVIGATION))
      {
         int value = signal->getValue();
         ETG_TRACE_USR4(("AudioClientHandler::onMainSinkSoundPropertyChangedSignal(), sinkId : %d, value : %d", sinkId, value));
         sendGuidanceVolumeUpdateMsg(sinkId, value);
      }
   }
}


void AudioClientHandler::onSoundPropertyListError(const ::boost::shared_ptr< MASTERAUDIOSERVICE_INTERFACE::SoundProperties::SoundPropertiesProxy >& proxy,
      const ::boost::shared_ptr< MASTERAUDIOSERVICE_INTERFACE::SoundProperties::SoundPropertyListError >& error)
{
   ETG_TRACE_USR4(("AudioClientHandler::onSoundPropertyListError()"));
}


void AudioClientHandler::onSoundPropertyListUpdate(const ::boost::shared_ptr< MASTERAUDIOSERVICE_INTERFACE::SoundProperties::SoundPropertiesProxy >& proxy,
      const ::boost::shared_ptr< MASTERAUDIOSERVICE_INTERFACE::SoundProperties::SoundPropertyListUpdate >& update)
{
   if (_soundPropertiesProxy == proxy)
   {
      const SoundPropertyData& soundPropertyData = update->getSoundPropertyList();
      uint16 sinkId = soundPropertyData.getSinkId();
      if ((COCKPIT_SINK_ID == sinkId) || (PRIVATE_SPEAKER_SINK_ID == sinkId))
      {
         std::vector< SoundProperty > audioPropertyVector = soundPropertyData.getSoundData();
         std::vector< SoundProperty >::iterator itr = audioPropertyVector.begin();
         for (; itr < audioPropertyVector.end(); ++itr)
         {
            int propertyType = itr->getPropertyType();
            if (am::MSP_VOLUME_NAVIGATION == propertyType)
            {
               int value = itr->getValue();
               ETG_TRACE_USR4(("AudioClientHandler::onSoundPropertyListUpdate(), sinkId : %d, value : %d", sinkId, value));
               sendGuidanceVolumeUpdateMsg(sinkId, value);
            }
         }
      }
   }
}


void AudioClientHandler::onSystemPropertyListError(const ::boost::shared_ptr< MASTERAUDIOSERVICE_INTERFACE::SoundProperties::SoundPropertiesProxy >& proxy,
      const ::boost::shared_ptr< MASTERAUDIOSERVICE_INTERFACE::SoundProperties::SystemPropertyListError >& error)
{
   ETG_TRACE_USR4(("AudioClientHandler::onSystemPropertyListError()"));
}


void AudioClientHandler::onSystemPropertyListUpdate(const ::boost::shared_ptr< MASTERAUDIOSERVICE_INTERFACE::SoundProperties::SoundPropertiesProxy >& proxy,
      const ::boost::shared_ptr< MASTERAUDIOSERVICE_INTERFACE::SoundProperties::SystemPropertyListUpdate >& update)
{
   if (_soundPropertiesProxy == proxy)
   {
      std::vector< SystemProperty > audioPropertyVector = update->getSystemPropertyList();
      std::vector< SystemProperty >::iterator itr = audioPropertyVector.begin();
      for (; itr < audioPropertyVector.end(); ++itr)
      {
         int propertyType = itr->getPropertyType();
         if (am::SYP_PRIVATE_MODE == propertyType)
         {
            ETG_TRACE_USR4(("AudioClientHandler::onSystemPropertyListUpdate(), propertyType : %d", propertyType));
            onPrivateModeUpdate(itr->getValue());
         }
      }
   }
}


void AudioClientHandler::onSystemPropertyChangedError(const ::boost::shared_ptr<MASTERAUDIOSERVICE_INTERFACE::SoundProperties::SoundPropertiesProxy >& proxy,
      const ::boost::shared_ptr<MASTERAUDIOSERVICE_INTERFACE::SoundProperties::SystemPropertyChangedError >& error)
{
   ETG_TRACE_USR4(("AudioClientHandler::onSystemPropertyChangedError()"));
}


void AudioClientHandler::onSystemPropertyChangedSignal(const ::boost::shared_ptr<MASTERAUDIOSERVICE_INTERFACE::SoundProperties::SoundPropertiesProxy >& proxy,
      const ::boost::shared_ptr<MASTERAUDIOSERVICE_INTERFACE::SoundProperties::SystemPropertyChangedSignal >& signal)
{
   if (_soundPropertiesProxy == proxy)
   {
      int propertyType = signal->getPropertyType();
      if (am::SYP_PRIVATE_MODE == propertyType)
      {
         ETG_TRACE_USR4(("AudioClientHandler::onSystemPropertyChangedSignal(), propertyType : %d", propertyType));
         onPrivateModeUpdate(signal->getValue());
      }
   }
}


void AudioClientHandler::onPrivateModeUpdate(int privateMode)
{
   ETG_TRACE_USR4(("AudioClientHandler::onPrivateModeUpdate(), privateMode : %d", privateMode));

   uint16 sinkId = (AM_SYP_PRIVATE_MODE_3 == privateMode) ? PRIVATE_SPEAKER_SINK_ID : COCKPIT_SINK_ID;
   if (_sinkId != sinkId)
   {
      _sinkId = sinkId;
      int volume = (COCKPIT_SINK_ID == _sinkId) ? _volumeLevelCockpit : _volumeLevelPrivate;
      sendGuidanceVolumeUpdateMsg(_sinkId, volume, true);
   }
}


bool AudioClientHandler::onCourierMessage(const SetGuidanceVolumeReqMsg& oMsg)
{
   ETG_TRACE_USR4(("AudioClientHandler::onCourierMessage(SetGuidanceVolumeReqMsg(%d))", oMsg.GetValue()));
   int value = checkVolumeValue(oMsg.GetValue());
   sendGuidanceVolumeToAudioManager(value);
   return true;
}


void AudioClientHandler::sendGuidanceVolumeToAudioManager(int value)
{
   if ((_soundPropertiesProxy) && (SINK_INVALID != _sinkId))
   {
      ETG_TRACE_USR4(("AudioClientHandler::sendGuidanceVolumeToAudioManager(), sinkId : %d, value : %d", _sinkId, value));
      _soundPropertiesProxy->sendSetMainSinkSoundPropertyRequest(*this, am::MSP_VOLUME_NAVIGATION, value, _sinkId);
   }
}


void AudioClientHandler::sendGuidanceVolumeUpdateMsg(uint16 sinkId, int value, bool forceUpdate)
{
   ETG_TRACE_USR4(("AudioClientHandler::sendGuidanceVolumeUpdateMsg(), sinkId : %d", sinkId));

   bool isUpdateRequired = forceUpdate;
   if (false == forceUpdate)
   {
      if (COCKPIT_SINK_ID == sinkId)
      {
         if (_volumeLevelCockpit != value)
         {
            _volumeLevelCockpit = value;
            isUpdateRequired = true;
         }
      }
      else if (PRIVATE_SPEAKER_SINK_ID == sinkId)
      {
         if (_volumeLevelPrivate != value)
         {
            _volumeLevelPrivate = value;
            isUpdateRequired = true;
         }
      }
   }
   if ((true == isUpdateRequired) && (VOLUME_VALUE_INVALID != value) && (_sinkId == sinkId))
   {
      int newVolumeValue = checkVolumeValue(value);
      if (newVolumeValue == value)  // If volume value is within min-max bounds
      {
         ETG_TRACE_USR4(("AudioClientHandler::sendGuidanceVolumeUpdateMsg(), value : %d", value));
         POST_MSG((COURIER_MESSAGE_NEW(GuidanceVolumeUpdateMsg)(value)));
      }
      else   // If volume value is reset to min-max bounds
      {
         sendGuidanceVolumeToAudioManager(newVolumeValue);
      }
   }
}


int AudioClientHandler::checkVolumeValue(int value)
{
   int retVal = value;
   retVal = (retVal < NAVI_SOUND_PROPERTY_MIN_VALUE) ? NAVI_SOUND_PROPERTY_MIN_VALUE : retVal;
   retVal = (retVal > NAVI_SOUND_PROPERTY_MAX_VALUE) ? NAVI_SOUND_PROPERTY_MAX_VALUE : retVal;
   return retVal;
}


void AudioClientHandler::onBeepFileError(const ::boost::shared_ptr< Mascf_main_fiProxy >& /*proxy*/,
      const ::boost::shared_ptr< ::mascf_main_fi::BeepFileError >& /*response*/)
{
   ETG_TRACE_ERR(("AudioClientHandler::onBeepFileError()"));
}


void AudioClientHandler::onBeepFileResult(const ::boost::shared_ptr< Mascf_main_fiProxy >& /*proxy*/,
      const ::boost::shared_ptr< ::mascf_main_fi::BeepFileResult >& /*response*/)
{
   ETG_TRACE_USR4(("AudioClientHandler::onBeepFileResult()"));
}


void AudioClientHandler::sendAudioBeepFile(::std::string beepFile)
{
   ETG_TRACE_USR4(("AudioClientHandler::sendAudioBeepFile()"));

   const int16 DEFAULT_SYSTEM_VOLUME = -32;   // System default volume
   if (!beepFile.empty())
   {
      ::masc_main_fi_types::T_BeepConfig beepConfig;
      beepConfig.setE8VolType(::masc_main_fi_types::T_e8_Aud_Beep_Vol_Type__FC_AUDIO_BEEP_VOL_TYPE_USER_SPECIFIC);
      beepConfig.setS16VoldB(DEFAULT_SYSTEM_VOLUME);//Volume-32dB set for Beep file
      _mascfFiProxy->sendBeepFileStart(*this, beepFile, beepConfig);
   }
   else
   {
      ETG_TRACE_USR4(("Beep file path empty"));
   }
}


bool AudioClientHandler::onCourierMessage(const PlayUserBeepReqMsg& oMsg)
{
   ETG_TRACE_USR4(("AudioClientHandler::onCourierMessage(PlayUserBeepReqMsg(%s))", oMsg.GetBeepFilePath().GetCString()));
   sendAudioBeepFile(oMsg.GetBeepFilePath().GetCString());
   return true;
}


} // namespace App
} // namespace Core
