/* ***************************************************************************************
* FILE:          HMIModelTabImplementation.cpp
* SW-COMPONENT:  HMI-BASE
*  DESCRIPTION:  HMIModelTabImplementation.cpp is part of HMI-Base reference/demo/test applications
*    COPYRIGHT:  (c) 2018-2020 Robert Bosch Car Multimedia GmbH
*
* The reproduction, distribution and utilization of this file as well as the
* communication of its contents to others without express authorization is
* prohibited. Offenders will be held liable for the payment of damages.
* All rights reserved in the event of the grant of a patent, utility model or design.
*
*21.09.17 Rev. 0.1 Ruchi (RBEI/ECG5)
* 		  		   Initial Revision.
*************************************************************************************** */

#include "gui_std_if.h"

#include "util/StrUtf8.h"
#include "AppHmi_MediaStateMachine.h"
#include "HMIModelTabImplementation.h"
#include "Core/ApplicationSwitchClientHandler/ApplicationSwitchClientHandler.h"
#include "hmi_trace_if.h"
#include "CgiExtensions/ImageLoader.h"
#include "MPlay_fi_types.h"

#ifdef WINDOWS_SIMULATION
#include "DemoApp.h"
#endif

#define ETG_DEFAULT_TRACE_CLASS TR_CLASS_APPHMI_MEDIA_HALL
#ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
#include "trcGenProj/Header/HMIModelTabImplementation.cpp.trc.h"
#endif

namespace App {
namespace Core {

HMIModelTabImplementation* HMIModelTabImplementation::_theTabModelInstance = 0;
static const char* const DATA_CONTEXT_TITLE_MENU_LIST_ITEM = "MediaMenuTitle";
static const char* const DATA_CONTEXT_TITLE_MENU_ARTIST_ITEM = "MediaMenuArtist";
static const char* const DATA_CONTEXT_TITLE_MENU_ALBUM_ITEM = "MediaMenuAlbum";
static const char* const DATA_CONTEXT_TITLE_MENU_GENRE_ITEM = "MediaMenuGenre";
static const char* const DATA_CONTEXT_TITLE_MENU_FOLDER_ITEM = "MediaMenuFolder";
static const char* const DATA_CONTEXT_TITLE_MENU_PLAYLIST_ITEM = "MediaMenuPlaylist";
static const char* const DATA_CONTEXT_TITLE_MENU_AUDIOBOOK_ITEM = "MediaMenuAudiobook";
static const char* const DATA_CONTEXT_TITLE_MENU_PODCAST_ITEM = "MediaMenuPodcast";
static const char* const DATA_CONTEXT_TITLE_MENU_VIDEO_ITEM = "MediaMenuVideo";

//MediaList Switch Mapping
/****************************************************************************************/
//Layer_TitleListEntry = 0
//Layer_ArtistListEntry = 1
//Layer_AlbumListEntry = 2
//Layer_GenreListEntry = 3
//Layer_FolderListEntry = 4
//Layer_PlaylistListEntry = 5
//Layer_AudiobkListEntry = 6
//Layer_PodcastListEntry = 7
//Layer_VideoListEntry = 8
//Layer_GenreArtistEntry = 9
//Layer_GenreArtistAlbum = 10
//Layer_GenreArtistAlbumSongListEntry = 11
//Layer_PlaylistSongEntry = 12
//LayerArtistAlbumTitleEntry = 13
//Layer_VideoEpisodeListEntry = 14
//Layer_PodcastEpisodeEntry = 15
//Layer_AudioBkEpisodeEntry = 16
//Layer_AllTitleArtistEntry = 17
//Layer_GenreAllAlbums = 18
//Layer_GenreAllAlbumTitleEntry = 19
//Layer_GenreAllTitlesEntry = 20
//Layer_FolderFileListEntry = 21
//Layer_ArtistAlbumEntry = 22
//Layer_AlbumTitleEntry = 23
/****************************************************************************************/
/* ***********************************************************
* Function 		: getInstance
* Description 	: function to create the HMIModelTabImplementation singleton instance
* Parameters 	: void
* Return value	: _theTabModelInstance - HMIModelTabImplementation object instance
* ***********************************************************/
HMIModelTabImplementation& HMIModelTabImplementation::getInstance()
{
   if (_theTabModelInstance == 0)
   {
      _theTabModelInstance = new HMIModelTabImplementation();
   }

   // assert(_theTabModelInstance);
   return *_theTabModelInstance;
}


/* ***********************************************************
* Function 		: removeInstance
* Description 	: function to delete the singleton instance
* Parameters 	: void
* Return value	: void
* ***********************************************************/
void HMIModelTabImplementation::removeInstance()
{
   if (_theTabModelInstance)
   {
      delete _theTabModelInstance;
      _theTabModelInstance = 0;
   }
}


/* ***********************************************************
* Function 		: HMIModelTabImplementation
* Description 	: constructor
* Parameters 	: void
* Return value	: none
* ***********************************************************/
HMIModelTabImplementation::HMIModelTabImplementation()
{
   g_SM_ListType = SM_MEDIA_LISTTYPE_INVALID;
   m_u32Previouslisttype = 0;
   m_u8startindex = 0;
   m_u8PageDirection = 0;
   isdevicetagchanged = true;

   ListRegistry::s_getInstance().addListImplementation(LISTID_MEDIA_HOR, this);
}


/* ***********************************************************
* Function 		: ~HMIModelTabImplementation
* Description 	: destructor
* Parameters 	: void
* Return value	: none
* ***********************************************************/
HMIModelTabImplementation::~HMIModelTabImplementation()
{
   ListRegistry::s_getInstance().removeListImplementation(LISTID_MEDIA_HOR);
}


bool HMIModelTabImplementation::Init()
{
   return true;
}


/* ***********************************************************
* Function 		: onCourierMessage
* Description 	: Incomming startup courier message for HMI data model
* Parameters 	: StartupMsg &oMsg
* Return value	: TRUE if it consumes the message else FALSE
* ***********************************************************/
bool HMIModelTabImplementation::onCourierMessage(const Courier::StartupMsg& /*oMsg*/)
{
   (*_dbSetTabActive).mTitleTabActiveStateChange = true;
   _dbSetTabActive.MarkAllItemsModified();
   _dbSetTabActive.SendUpdate();
   return true;
}


/* ***********************************************************
* Function 		: onCourierMessage
* Description 	:
* Parameters 	: ListDateProviderReqMsg &oMsg
* Return value	: TRUE if it consumes the message else FALSE
* ***********************************************************/
bool HMIModelTabImplementation::onCourierMessage(const ListDateProviderReqMsg& oMsg)
{
   ETG_TRACE_USR4(("HMIModelTabImplementation::onCourierMessage::onCourierMessage:ListDateProviderReqMsg listid: %d,startindex: %d, windowelementsize: %d", oMsg.GetListId(), oMsg.GetStartIndex(), oMsg.GetWindowElementSize()));

   return ListRegistry::s_getInstance().updateList(oMsg.GetListId(), oMsg.GetStartIndex(), oMsg.GetWindowElementSize());
}


/* ***********************************************************
* Function 		: getListDataProvider
* Description 	: Updates the List data Based on the LIST_IDs
* Parameters 	: ListDateProviderReqMsg &oMsg
* Return value	: tSharedPtrDataProvider
* ***********************************************************/
tSharedPtrDataProvider HMIModelTabImplementation::getListDataProvider(const ListDateProviderReqMsg& oMsg)
{
   ETG_TRACE_USR1(("HMIModelTabImplementation::getListDataProvider"));

   ListDataProviderBuilder listBuilder(LISTID_MEDIA_HOR, "MediaMenuTitle");
   uint8 u8activeRegion = ApplicationSwitchClientHandler::poGetInstance()->getActiveRegion();

   unsigned item = 0;
   unsigned row = 0;

   unsigned hdlRow[10];

   for (unsigned page = 0; page < 3; ++page)
   {
      for (row = 0; row < 3; ++row)
      {
         if (item == TAB_TITLE)
         {
            listBuilder.AddItem(item, 0, DATA_CONTEXT_TITLE_MENU_LIST_ITEM);
         }
         else if (item == TAB_ARTIST)
         {
            listBuilder.AddItem(item, 0, DATA_CONTEXT_TITLE_MENU_ARTIST_ITEM);
         }
         else if (item == TAB_ALBUM)
         {
            listBuilder.AddItem(item, 0, DATA_CONTEXT_TITLE_MENU_ALBUM_ITEM);
         }
         else if (item == TAB_GENRE)
         {
            listBuilder.AddItem(item, 0, DATA_CONTEXT_TITLE_MENU_GENRE_ITEM);
         }
         else if (item == TAB_FOLDER)
         {
            listBuilder.AddItem(item, 0, DATA_CONTEXT_TITLE_MENU_FOLDER_ITEM);
         }
         else if (item == TAB_PLAYLIST)
         {
            listBuilder.AddItem(item, 0, DATA_CONTEXT_TITLE_MENU_PLAYLIST_ITEM);
         }
         else if (item == TAB_AUDIOBOOK)
         {
            listBuilder.AddItem(item, 0, DATA_CONTEXT_TITLE_MENU_AUDIOBOOK_ITEM);
         }
         else if (item == TAB_PODCAST)
         {
            listBuilder.AddItem(item, 0, DATA_CONTEXT_TITLE_MENU_PODCAST_ITEM);
         }
         else if (item == TAB_VIDEO)
         {
            listBuilder.AddItem(item, 0, DATA_CONTEXT_TITLE_MENU_VIDEO_ITEM);
         }
         else
         {
            ETG_TRACE_USR1(("HMIModelTabImplementation::getListDataProvider INVALID TAB SELECTION"));
         }

         item++;
      }
#ifdef VARIANT_S_FTR_ENABLE_VIDEO_PLAYBACK
      (*_dbTabVideoSource).mTabVideoSourceVisibility = true;
      if (u8activeRegion == REGION_COCKPIT)
      {
         (*_dbTabVideoSource).mTabVideoSourceEnable = true;
         ETG_TRACE_USR1(("HMIModelTabImplementation::getListDataProvider Enable Video Tab"));
      }
      else
      {
         (*_dbTabVideoSource).mTabVideoSourceEnable = false;
         ETG_TRACE_USR1(("HMIModelTabImplementation::getListDataProvider Disable Video Tab"));
      }
      _dbTabVideoSource.MarkAllItemsModified();
      _dbTabVideoSource.SendUpdate(true);
#else
      (*_dbTabVideoSource).mTabVideoSourceVisibility = false;//video disabled in smart
      _dbTabVideoSource.MarkAllItemsModified();
      _dbTabVideoSource.SendUpdate(true);
#endif
   }

   //ETG_TRACE_USR4(("Tab:Onstartup:%d,g_SM_ListType::%d", onStartup, g_SM_ListType));
   //set the tile as selected when the devicetag is changed
   ETG_TRACE_USR4(("HMIModelTabImplementation::getListDataProvider:isdevicetagchanged %d ", isdevicetagchanged));
   ETG_TRACE_USR4(("HMIModelTabImplementation::getListDataProvider:Activemediasource %d ", g_SM_ActiveMediaSource));
   ETG_TRACE_USR4(("HMIModelTabImplementation::getListDataProvider:Listtype %d ", g_SM_ListType));

   if (isdevicetagchanged)
   {
      if (g_SM_ActiveMediaSource == MPlay_fi_types::T_e8_MPlayDeviceType__e8DTY_BLUETOOTH)//If BT select the second page and disable other tabs
      {
         setTabStatus(false);
         POST_MSG(COURIER_MESSAGE_NEW(ListChangeMsg)(LISTID_MEDIA_HOR, ListChangeSet, EN_MEDIA_INDEX3));
         m_u8startindex = 3;
      }
      else
      {
         setTabStatus(true);
         if (g_SM_ListType == SM_MEDIA_LISTTYPE_VIDEO)
         {
            POST_MSG(COURIER_MESSAGE_NEW(ListChangeMsg)(LISTID_MEDIA_HOR, ListChangeSet, EN_MEDIA_INDEX6));
            setVideoTabStateVisibility();
            m_u8startindex = 6;
         }
         else
         {
            POST_MSG(COURIER_MESSAGE_NEW(ListChangeMsg)(LISTID_MEDIA_HOR, ListChangeSet, EN_MEDIA_INDEX0));

            //Clearing Title List - workaround to stop overlap for the device change  - to be deleted later

            //Clearing Album List - workaround to overcome overlap issue - to be deleted later
            /*ListDataProviderBuilder listBuilder_Album(LISTID_MEDIA_ALBUM);
            tSharedPtrDataProvider dataProvider_Album = listBuilder_Album.CreateDataProvider();
            POST_MSG((COURIER_MESSAGE_NEW(ListDateProviderResMsg)(dataProvider_Album)));*/

            setTitleTabStateVisibility();
            m_u8startindex = 0;
         }
      }
      isdevicetagchanged = false;
   }
   else
   {
      if (g_SM_ActiveMediaSource == MPlay_fi_types::T_e8_MPlayDeviceType__e8DTY_BLUETOOTH)
      {
         setTabStatus(false);
         POST_MSG(COURIER_MESSAGE_NEW(ListChangeMsg)(LISTID_MEDIA_HOR, ListChangeSet, EN_MEDIA_INDEX3));
         m_u8startindex = 3;
      }
      else
      {
         if (g_SM_ListType == SM_MEDIA_LISTTYPE_VIDEO)
         {
            setTabStatus(true);
            POST_MSG(COURIER_MESSAGE_NEW(ListChangeMsg)(LISTID_MEDIA_HOR, ListChangeSet, EN_MEDIA_INDEX6));
            setVideoTabStateVisibility();
            m_u8startindex = 6;
         }
         else
         {
            setTabStatus(true);
            switch (m_u8startindex)
            {
               case EN_MEDIA_INDEX0:
                  POST_MSG(COURIER_MESSAGE_NEW(ListChangeMsg)(LISTID_MEDIA_HOR, ListChangeSet, EN_MEDIA_INDEX0));
                  break;
               case EN_MEDIA_INDEX3:
                  POST_MSG(COURIER_MESSAGE_NEW(ListChangeMsg)(LISTID_MEDIA_HOR, ListChangeSet, EN_MEDIA_INDEX3));
                  break;
               case EN_MEDIA_INDEX6:
                  POST_MSG(COURIER_MESSAGE_NEW(ListChangeMsg)(LISTID_MEDIA_HOR, ListChangeSet, EN_MEDIA_INDEX6));
                  break;
               default:
                  break;
            }
         }
      }
   }
   ETG_TRACE_USR4(("HMIModelTabImplementation::getListDataProvider startindex : %d ", m_u8startindex));

   if (g_SM_ActiveMediaSource != MPlay_fi_types::T_e8_MPlayDeviceType__e8DTY_BLUETOOTH)
   {
      //changing mask of prev and next button based on page index
      switch (m_u8startindex)
      {
         case EN_MEDIA_INDEX0:
            //Disable the previous button and enable next button
            (*_tabNextPrevIndex).mPrevIndex = false;
            (*_tabNextPrevIndex).mNextIndex = true;

            (*_tabNextPrevIndex).mEnableTouchPrev = false;
            (*_tabNextPrevIndex).mEnableTouchNext = true;
            break;
         case EN_MEDIA_INDEX3:
            //Enable both Next and Prev Button
            (*_tabNextPrevIndex).mPrevIndex = true;
            (*_tabNextPrevIndex).mNextIndex = true;

            (*_tabNextPrevIndex).mEnableTouchPrev = true;
            (*_tabNextPrevIndex).mEnableTouchNext = true;
            break;
         case EN_MEDIA_INDEX6:
            //Disable the Nextybutton and enable Previous button
            (*_tabNextPrevIndex).mPrevIndex = true;
            (*_tabNextPrevIndex).mNextIndex = false;

            (*_tabNextPrevIndex).mEnableTouchPrev = true;
            (*_tabNextPrevIndex).mEnableTouchNext = false;
            break;
         default:
            break;
      }
      _tabNextPrevIndex.MarkAllItemsModified();
      _tabNextPrevIndex.SendUpdate();
   }
   else
   {
      //case EN_MEDIA_INDEX3:
      //(*_tabNextPrevIndex).mNextPrevIndex = 1;

      //Disable both Next and Prev Button when ActiveDevice is BT
      (*_tabNextPrevIndex).mPrevIndex = 0;
      (*_tabNextPrevIndex).mNextIndex = 0;

      (*_tabNextPrevIndex).mEnableTouchPrev = false;
      (*_tabNextPrevIndex).mEnableTouchNext = false;
      //break;

      _tabNextPrevIndex.MarkAllItemsModified();
      _tabNextPrevIndex.SendUpdate();
   }
   /*if ((*_folderTabSwitchIndex).mSwitchIndex)
   	POST_MSG(COURIER_MESSAGE_NEW(ListChangeMsg)(LISTID_MEDIA_HOR, ListChangeSet, STARTINDEXFORFOLDERTAB));*/
   tSharedPtrDataProvider dataProvider = listBuilder.CreateDataProvider();
   dataProvider->setListChangeSetIndex(m_u8startindex);
   dataProvider->setCacheOnOff(false);
   return dataProvider;
}


/* ***********************************************************
* Function 		: IsDeviceTagChanged
* Description 	: stores the device change status if the device is changed
* Parameters 	: devicetagchangestate
* Return value	: void
* ***********************************************************/
void HMIModelTabImplementation::IsDeviceTagChanged(bool devicetagchangestate)
{
   isdevicetagchanged = devicetagchangestate;
   ETG_TRACE_USR4(("HMIModelTabImplementation::IsDeviceTagChanged devicetagchanged: %d ", isdevicetagchanged));
}


/* ***********************************************************
* Function 		: resetVerticalStartindex
* Description 	: reset the vertical start index
* Return value	: void
* ***********************************************************/
void HMIModelTabImplementation::resetVerticalStartindex()
{
   uint32_t  _uint32Verticalstartindex ;
   _uint32Verticalstartindex = HMIModelListImplementation::getInstance().getVerticalStartIndex();
   ETG_TRACE_USR4(("HMIModelTabImplementation::resetVerticalStartindex verticalstartindex:%d", _uint32Verticalstartindex));
   _uint32Verticalstartindex = 0;
   HMIModelListImplementation::getInstance().setVerticalStartIndex(_uint32Verticalstartindex);
}


/* ***********************************************************
* Function 		: onCourierMessage
* Description 	: Courier message received when Previous button is selected
* Parameters 	: oMsg
* Return value	: bool
* ***********************************************************/
bool HMIModelTabImplementation::onCourierMessage(const HorTabButtonPageUpReqMsg& oMsg)
{
   ETG_TRACE_USR1(("HMIModelTabImplementation::PageUp"));
   g_SM_InitialTabLoading = 0;
   resetVerticalStartindex();

   COURIER_MESSAGE_NEW(ListChangeMsg)(LISTID_MEDIA_HOR, ListChangePageUp, 1)->Post();
   m_u8PageDirection = EN_MEDIATAB_PAGEUP;
   return true;
}


/* ***********************************************************
* Function 		: ChangeHorListTabButtonRegMsg
* Description 	: Courier message received when Next button is selected
* Parameters 	: oMsg
* Return value	: bool
* ***********************************************************/
bool HMIModelTabImplementation::onCourierMessage(const HorTabButtonPageDownReqMsg& oMsg)
{
   ETG_TRACE_USR1(("HMIModelTabImplementation::PageDown"));
   g_SM_InitialTabLoading = 0;
   resetVerticalStartindex();

   COURIER_MESSAGE_NEW(ListChangeMsg)(LISTID_MEDIA_HOR, ListChangePageDown, 1)->Post();
   m_u8PageDirection = EN_MEDIATAB_DOWN;
   return true;
}


bool HMIModelTabImplementation::onCourierMessage(const BTBrowsePressedUpdMsg& oMsg)
{
   ETG_TRACE_USR1(("HMIModelTabImplementation::BTBrowsePressed"));

   resetVerticalStartindex();
   g_SM_ListType = SM_MEDIA_LISTTYPE_FOLDER;
   tclMediaClientHandler::getInstance().initStatemachine();
   tclMediaClientHandler::getInstance().clearFolderBrowseHistory();
   vOnRequestTabList(g_SM_ListType);
   HMIModelListImplementation::getInstance().clearListBrowsingHistory();
   HMIModelListImplementation::getInstance().updateCurrentListId(LISTID_MEDIA_FOLDER);
   HMIModelListImplementation::getInstance().IsFolderBrowse(true);
   HMIModelListImplementation::getInstance().updateIsNewListRequest(true);
   HMIModelListImplementation::getInstance().updateListSize();
   HMIModelListImplementation::getInstance().updateRequestedListHandle(0);
   HMIModelListImplementation::getInstance().setIsPlaySongFromListRequested(false);
   return true;
}


bool HMIModelTabImplementation::onCourierMessage(const VideoBrowsePressedUpdMsg& oMsg)
{
   ETG_TRACE_USR1(("HMIModelTabImplementation::onCourierMessage:VideoBrowsePressedUpdMsg  g_SM_ListType:%d", g_SM_ListType));
   uint8 u8activeRegion = ApplicationSwitchClientHandler::poGetInstance()->getActiveRegion();

   if (g_SM_ListType == SM_MEDIA_LISTTYPE_INVALID)
   {
      g_SM_ListType = SM_MEDIA_LISTTYPE_VIDEO;
      tclMediaClientHandler::getInstance().initStatemachine();
      tclMediaClientHandler::getInstance().clearFolderBrowseHistory();
      if (u8activeRegion == REGION_COCKPIT)
      {
         vOnRequestTabList(g_SM_ListType);
      }
      HMIModelListImplementation::getInstance().clearListBrowsingHistory();
      HMIModelListImplementation::getInstance().updateCurrentListId(LISTID_MEDIA_VIDEO);
      HMIModelListImplementation::getInstance().IsFolderBrowse(false);
      HMIModelListImplementation::getInstance().updateIsNewListRequest(true);
      HMIModelListImplementation::getInstance().updateListSize();
      HMIModelListImplementation::getInstance().updateRequestedListHandle(0);
      isdevicetagchanged = true;
   }
   else
   {
      HMIModelListImplementation::getInstance().updateIsNewListRequest(true);
      isdevicetagchanged = false;
   }

   return true;
}


bool HMIModelTabImplementation::onCourierMessage(const BrowseBtnPressedUpdMsg& oMsg)
{
   ETG_TRACE_USR4(("HMIModelTabImplementation::onCourierMessage:BrowseButtonPressed g_SM_ListType:%d", g_SM_ListType));

   resetVerticalStartindex();

   if (g_SM_ListType == SM_MEDIA_LISTTYPE_INVALID)
   {
      g_SM_ListType = SM_MEDIA_LISTTYPE_SONG;
      tclMediaClientHandler::getInstance().initStatemachine();
      tclMediaClientHandler::getInstance().clearFolderBrowseHistory();
      vOnRequestTabList(g_SM_ListType);
      HMIModelListImplementation::getInstance().clearListBrowsingHistory();
      HMIModelListImplementation::getInstance().updateCurrentListId(LISTID_MEDIA_TITLE);
      HMIModelListImplementation::getInstance().IsFolderBrowse(false);
      HMIModelListImplementation::getInstance().updateIsNewListRequest(true);
      HMIModelListImplementation::getInstance().updateListSize();
      HMIModelListImplementation::getInstance().updateRequestedListHandle(0);
      HMIModelListImplementation::getInstance().setIsPlaySongFromListRequested(false);
      isdevicetagchanged = true;
   }
   else
   {
      HMIModelListImplementation::getInstance().updateIsNewListRequest(true);
      isdevicetagchanged = false;
   }
   return true;
}


/* ***********************************************************
* Function 		: onCourierMessage
* Description 	: Courier message received when there is change in the list
* Parameters 	: oMsg
* Return value	: void
* ***********************************************************/
bool HMIModelTabImplementation::onCourierMessage(const ListChangedUpdMsg& oMsg)
{
   ETG_TRACE_USR1(("HMIModelTabImplementation::onCourierMessage:ListChangedUpdMsg"));
   ETG_TRACE_USR4(("HMIModelTabImplementation::onCourierMessage:ListChangedUpdMsg listid:%d", oMsg.GetListId()));

   if (oMsg.GetListId() == LISTID_MEDIA_HOR)
   {
      m_u8startindex = oMsg.GetStartIndex();
      ETG_TRACE_USR4(("HMIModelTabImplementation::onCourierMessage:ListChangedUpdMsg m_u8startindex:%d", m_u8startindex));

      ETG_TRACE_USR4(("HMIModelTabImplementation::onCourierMessage:ListChangedUpdMsg g_SM_InitialTabLoading  %d", g_SM_InitialTabLoading));
      //setTabStatus(true);
      ETG_TRACE_USR4(("HMIModelTabImplementation::onCourierMessage:ListChangedUpdMsg m_u8PageDirection %d", m_u8PageDirection));

      if (g_SM_ActiveMediaSource != MPlay_fi_types::T_e8_MPlayDeviceType__e8DTY_BLUETOOTH && g_SM_InitialTabLoading == 0)
      {
         setTabStatus(true); //enabling the genre and playlist tabs
         if (m_u8PageDirection)//EN_MEDIATAB_DOWN
         {
            switch (m_u8startindex)
            {
               case EN_MEDIA_INDEX3://m_u8startindex == 3
               {
                  ETG_TRACE_USR1(("HMIModelTabImplementation::PageDown::Request sent for GENRE"));
                  g_SM_ListType = SM_MEDIA_LISTTYPE_GENRE;

                  setGenreTabStateVisibility();
                  tclMediaClientHandler::getInstance().initStatemachine();
                  tclMediaClientHandler::getInstance().clearFolderBrowseHistory();
                  HMIModelListImplementation::getInstance().updateRequestedListHandle(0);
                  HMIModelListImplementation::getInstance().updateIsNewListRequest(true);
                  HMIModelListImplementation::getInstance().clearListBrowsingHistory();
                  HMIModelListImplementation::getInstance().updateCurrentListId(LISTID_MEDIA_GENRE);
                  HMIModelListImplementation::getInstance().updateListSize();
                  HMIModelListImplementation::getInstance().IsHorListTabSelected(true);
                  HMIModelListImplementation::getInstance().IsFolderBrowse(false);
                  HMIModelListImplementation::getInstance().IsAllSelected(false);
                  HMIModelListImplementation::getInstance().IsNewList(true);
                  HMIModelListImplementation::getInstance().setIsPlaySongFromListRequested(false);
                  vOnRequestTabList(g_SM_ListType);
                  break;
               }
               case EN_MEDIA_INDEX6:
               {
                  ETG_TRACE_USR1(("HMIModelTabImplementation::PageDown::Request sent for AUDIOBOOK "));
                  g_SM_ListType = SM_MEDIA_LISTTYPE_AUDIOBOOK;

                  setAudiobookTabStateVisibility();
                  tclMediaClientHandler::getInstance().initStatemachine();
                  tclMediaClientHandler::getInstance().clearFolderBrowseHistory();
                  HMIModelListImplementation::getInstance().updateRequestedListHandle(0);
                  HMIModelListImplementation::getInstance().updateIsNewListRequest(true);
                  HMIModelListImplementation::getInstance().clearListBrowsingHistory();
                  HMIModelListImplementation::getInstance().updateCurrentListId(LISTID_MEDIA_AUDIOBOOK);
                  HMIModelListImplementation::getInstance().updateListSize();
                  HMIModelListImplementation::getInstance().IsHorListTabSelected(true);
                  HMIModelListImplementation::getInstance().IsFolderBrowse(false);
                  HMIModelListImplementation::getInstance().IsAllSelected(false);
                  HMIModelListImplementation::getInstance().IsNewList(true);
                  HMIModelListImplementation::getInstance().setIsPlaySongFromListRequested(false);
                  vOnRequestTabList(g_SM_ListType);
                  break;
               }
               default:
                  break;
            }
         }
         else//EN_MEDIATAB_PAGEUP
         {
            switch (m_u8startindex)
            {
               case EN_MEDIA_INDEX3:
               {
                  ETG_TRACE_USR1(("HMIModelTabImplementation::PageDown::Request sent for PLAYLIST"));
                  g_SM_ListType = SM_MEDIA_LISTTYPE_PLAYLIST;

                  setPlaylistTabStateVisibility();
                  tclMediaClientHandler::getInstance().initStatemachine();
                  tclMediaClientHandler::getInstance().clearFolderBrowseHistory();
                  HMIModelListImplementation::getInstance().updateRequestedListHandle(0);
                  HMIModelListImplementation::getInstance().updateIsNewListRequest(true);
                  HMIModelListImplementation::getInstance().clearListBrowsingHistory();
                  HMIModelListImplementation::getInstance().updateCurrentListId(LISTID_MEDIA_PLAYLIST);
                  HMIModelListImplementation::getInstance().updateListSize();
                  HMIModelListImplementation::getInstance().IsHorListTabSelected(true);
                  HMIModelListImplementation::getInstance().IsFolderBrowse(false);
                  HMIModelListImplementation::getInstance().IsAllSelected(false);
                  HMIModelListImplementation::getInstance().IsNewList(true);
                  HMIModelListImplementation::getInstance().setIsPlaySongFromListRequested(false);
                  vOnRequestTabList(g_SM_ListType);
                  break;
               }
               case EN_MEDIA_INDEX0:
               {
                  ETG_TRACE_USR1(("HMIModelTabImplementation::PageDown::Request sent for ALBUM"));
                  g_SM_ListType = SM_MEDIA_LISTTYPE_ALBUM;

                  setAlbumTabStateVisibility();
                  tclMediaClientHandler::getInstance().initStatemachine();
                  tclMediaClientHandler::getInstance().clearFolderBrowseHistory();
                  HMIModelListImplementation::getInstance().updateRequestedListHandle(0);
                  HMIModelListImplementation::getInstance().updateIsNewListRequest(true);
                  HMIModelListImplementation::getInstance().clearListBrowsingHistory();
                  HMIModelListImplementation::getInstance().updateCurrentListId(LISTID_MEDIA_ALBUM);
                  HMIModelListImplementation::getInstance().updateListSize();
                  HMIModelListImplementation::getInstance().IsHorListTabSelected(true);
                  HMIModelListImplementation::getInstance().IsFolderBrowse(false);
                  HMIModelListImplementation::getInstance().IsAllSelected(false);
                  HMIModelListImplementation::getInstance().IsNewList(true);
                  HMIModelListImplementation::getInstance().setIsPlaySongFromListRequested(false);
                  vOnRequestTabList(g_SM_ListType);
                  break;
               }
               default:
                  break;
            }
         }
      }
      /*      else
            {
               if ((g_SM_ActiveMediaSource == MPlay_fi_types::T_e8_MPlayDeviceType__e8DTY_BLUETOOTH) && (g_SM_InitialTabLoading == 0))
               {
                  //request for Folder list
                  g_SM_ListType = SM_MEDIA_LISTTYPE_FOLDER;
                  vOnRequestTabList(g_SM_ListType);
                  //disable Genre and Playlist
                  setTabStatus(false);
               }

               //(*_dbSetTabActive).mFolderTabActiveStateChange = true; //set folder tab as active
               //_dbSetTabActive.MarkAllItemsModified();
               //_dbSetTabActive.SendUpdate();
            }*/

      ListRegistry::s_getInstance().updateList(LISTID_MEDIA_HOR);
      ETG_TRACE_USR4(("m_u8startindex %d ", m_u8startindex));
      return true;
   }
   return false;
}


/* ***********************************************************
* Function 		: ChangeHorListTabButtonRegMsg
* Description 	: Updates the current selected tab button
* Parameters 	: oMsg
* Return value	: void
* ***********************************************************/
bool HMIModelTabImplementation::onCourierMessage(const ChangeHorListTabButtonRegMsg& oMsg)
{
   Courier::UInt32 SelectMediaTab = oMsg.GetSelectMediaTab();

   ETG_TRACE_USR4(("HMIModelTabImplementation::onCourierMessage:ChangeHorListTabButtonRegMsg rowvalue:%d", SelectMediaTab));
   g_SM_InitialTabLoading = 0;
   resetVerticalStartindex();

   tclMediaClientHandler::getInstance().initStatemachine();
   tclMediaClientHandler::getInstance().clearFolderBrowseHistory();
   HMIModelListImplementation::getInstance().updateRequestedListHandle(0);
   HMIModelListImplementation::getInstance().updateIsNewListRequest(true);
   HMIModelListImplementation::getInstance().IsHorListTabSelected(true);
   HMIModelListImplementation::getInstance().IsAllSelected(false);
   HMIModelListImplementation::getInstance().IsNewList(true);
   HMIModelListImplementation::getInstance().clearListBrowsingHistory();
   HMIModelListImplementation::getInstance().setIsPlaySongFromListRequested(false);

   switch (SelectMediaTab)
   {
      case TAB_TITLE:
      {
         g_SM_ListType = SM_MEDIA_LISTTYPE_SONG;
         HMIModelListImplementation::getInstance().updateCurrentListId(LISTID_MEDIA_TITLE);
         HMIModelListImplementation::getInstance().updateListSize();
         HMIModelListImplementation::getInstance().IsFolderBrowse(false);
         vOnRequestTabList(g_SM_ListType);
         break;
      }
      case TAB_ARTIST:
      {
         g_SM_ListType = SM_MEDIA_LISTTYPE_ARTIST;
         HMIModelListImplementation::getInstance().updateCurrentListId(LISTID_MEDIA_ARTIST);
         HMIModelListImplementation::getInstance().updateListSize();
         HMIModelListImplementation::getInstance().IsFolderBrowse(false);
         vOnRequestTabList(g_SM_ListType);
         break;
      }
      case TAB_ALBUM:
      {
         g_SM_ListType = SM_MEDIA_LISTTYPE_ALBUM;
         HMIModelListImplementation::getInstance().updateCurrentListId(LISTID_MEDIA_ALBUM);
         HMIModelListImplementation::getInstance().updateListSize();
         HMIModelListImplementation::getInstance().IsFolderBrowse(false);
         vOnRequestTabList(g_SM_ListType);
         break;
      }
      case TAB_GENRE:
      {
         g_SM_ListType = SM_MEDIA_LISTTYPE_GENRE;
         HMIModelListImplementation::getInstance().updateCurrentListId(LISTID_MEDIA_GENRE);
         HMIModelListImplementation::getInstance().updateListSize();
         HMIModelListImplementation::getInstance().IsFolderBrowse(false);
         vOnRequestTabList(g_SM_ListType);
         break;
      }
      case TAB_FOLDER:
      {
         g_SM_ListType = SM_MEDIA_LISTTYPE_FOLDER;
         HMIModelListImplementation::getInstance().updateCurrentListId(LISTID_MEDIA_FOLDER);
         HMIModelListImplementation::getInstance().updateListSize();
         HMIModelListImplementation::getInstance().IsFolderBrowse(true);
         vOnRequestTabList(g_SM_ListType);
         break;
      }
      case TAB_PLAYLIST:
      {
         g_SM_ListType = SM_MEDIA_LISTTYPE_PLAYLIST;
         HMIModelListImplementation::getInstance().updateCurrentListId(LISTID_MEDIA_PLAYLIST);
         HMIModelListImplementation::getInstance().updateListSize();
         HMIModelListImplementation::getInstance().IsFolderBrowse(false);
         vOnRequestTabList(g_SM_ListType);
         break;
      }
      case TAB_AUDIOBOOK:
      {
         g_SM_ListType = SM_MEDIA_LISTTYPE_AUDIOBOOK;
         HMIModelListImplementation::getInstance().updateCurrentListId(LISTID_MEDIA_AUDIOBOOK);
         HMIModelListImplementation::getInstance().updateListSize();
         HMIModelListImplementation::getInstance().IsFolderBrowse(false);
         vOnRequestTabList(g_SM_ListType);
         break;
      }
      case TAB_PODCAST:
      {
         g_SM_ListType = SM_MEDIA_LISTTYPE_PODCAST;
         HMIModelListImplementation::getInstance().updateCurrentListId(LISTID_MEDIA_PODCAST);
         HMIModelListImplementation::getInstance().updateListSize();
         HMIModelListImplementation::getInstance().IsFolderBrowse(false);
         vOnRequestTabList(g_SM_ListType);
         break;
      }
      case TAB_VIDEO:
      {
         g_SM_ListType = SM_MEDIA_LISTTYPE_VIDEO;
         HMIModelListImplementation::getInstance().updateCurrentListId(LISTID_MEDIA_VIDEO);
         HMIModelListImplementation::getInstance().updateListSize();
         HMIModelListImplementation::getInstance().IsFolderBrowse(false);
         vOnRequestTabList(g_SM_ListType);
         break;
      }
      default:
         break;
   }
   //ListRegistry::s_getInstance().updateList(LISTID_MEDIA_HOR);
   setActiveStateVisibility(SelectMediaTab);

   return true;
}


/* ***********************************************************
* Function 		: setTitleTabStateVisibility
* Description 	: sets the visibilty based on the tab selection
* Parameters 	: void
* Return value	: void
* ***********************************************************/
void HMIModelTabImplementation::setTitleTabStateVisibility()
{
   ETG_TRACE_USR1(("HMIModelTabImplementation::setTitleTabStateVisibility"));

   (*_dbSetTabActive).mTitleTabActiveStateChange = true;
   (*_dbSetTabActive).mArtistTabActiveStateChange = false;
   (*_dbSetTabActive).mGenreTabActiveStateChange = false;
   (*_dbSetTabActive).mAlbumTabActiveStateChange = false;
   (*_dbSetTabActive).mFolderTabActiveStateChange = false;
   (*_dbSetTabActive).mPlaylistTabActiveStateChange = false;
   (*_dbSetTabActive).mAudioBkTabActiveStateChange = false;
   (*_dbSetTabActive).mPodcastTabActiveStateChange = false;
   (*_dbSetTabActive).mVideoTabActiveStateChange = false;

   _dbSetTabActive.MarkAllItemsModified();
   _dbSetTabActive.SendUpdate();
}


/* ***********************************************************
* Function 		: setArtistTabStateVisibility
* Description 	: sets the visibilty based on the tab selection
* Parameters 	: void
* Return value	: void
* ***********************************************************/
void HMIModelTabImplementation::setArtistTabStateVisibility()
{
   ETG_TRACE_USR1(("HMIModelTabImplementation::setArtistTabStateVisibility"));

   (*_dbSetTabActive).mTitleTabActiveStateChange = false;
   (*_dbSetTabActive).mArtistTabActiveStateChange = true;
   (*_dbSetTabActive).mAlbumTabActiveStateChange = false;
   (*_dbSetTabActive).mGenreTabActiveStateChange = false;
   (*_dbSetTabActive).mFolderTabActiveStateChange = false;
   (*_dbSetTabActive).mPlaylistTabActiveStateChange = false;
   (*_dbSetTabActive).mAudioBkTabActiveStateChange = false;
   (*_dbSetTabActive).mPodcastTabActiveStateChange = false;
   (*_dbSetTabActive).mVideoTabActiveStateChange = false;

   _dbSetTabActive.MarkAllItemsModified();
   _dbSetTabActive.SendUpdate();
}


/* ***********************************************************
* Function 		: setAlbumTabStateVisibility
* Description 	: sets the visibilty based on the tab selection
* Parameters 	: void
* Return value	: void
* ***********************************************************/
void HMIModelTabImplementation::setAlbumTabStateVisibility()
{
   ETG_TRACE_USR1(("HMIModelTabImplementation::setAlbumTabStateVisibility"));

   (*_dbSetTabActive).mTitleTabActiveStateChange = false;
   (*_dbSetTabActive).mArtistTabActiveStateChange = false;
   (*_dbSetTabActive).mAlbumTabActiveStateChange = true;
   (*_dbSetTabActive).mGenreTabActiveStateChange = false;
   (*_dbSetTabActive).mFolderTabActiveStateChange = false;
   (*_dbSetTabActive).mPlaylistTabActiveStateChange = false;
   (*_dbSetTabActive).mAudioBkTabActiveStateChange = false;
   (*_dbSetTabActive).mPodcastTabActiveStateChange = false;
   (*_dbSetTabActive).mVideoTabActiveStateChange = false;

   _dbSetTabActive.MarkAllItemsModified();
   _dbSetTabActive.SendUpdate();
}


/* ***********************************************************
* Function 		: setGenreTabStateVisibility
* Description 	: sets the visibilty based on the tab selection
* Parameters 	: void
* Return value	: void
* ***********************************************************/
void HMIModelTabImplementation::setGenreTabStateVisibility()
{
   ETG_TRACE_USR1(("HMIModelTabImplementation::setGenreTabStateVisibility"));

   (*_dbSetTabActive).mTitleTabActiveStateChange = false;
   (*_dbSetTabActive).mArtistTabActiveStateChange = false;
   (*_dbSetTabActive).mAlbumTabActiveStateChange = false;
   (*_dbSetTabActive).mGenreTabActiveStateChange = true;
   (*_dbSetTabActive).mFolderTabActiveStateChange = false;
   (*_dbSetTabActive).mPlaylistTabActiveStateChange = false;
   (*_dbSetTabActive).mAudioBkTabActiveStateChange = false;
   (*_dbSetTabActive).mPodcastTabActiveStateChange = false;
   (*_dbSetTabActive).mVideoTabActiveStateChange = false;

   _dbSetTabActive.MarkAllItemsModified();
   _dbSetTabActive.SendUpdate();
}


/* ***********************************************************
* Function 		: setFolderTabStateVisibility
* Description 	: sets the visibilty based on the tab selection
* Parameters 	: void
* Return value	: void
* ***********************************************************/
void HMIModelTabImplementation::setFolderTabStateVisibility()
{
   ETG_TRACE_USR1(("HMIModelTabImplementation::setFolderTabStateVisibility"));

   (*_dbSetTabActive).mTitleTabActiveStateChange = false;
   (*_dbSetTabActive).mArtistTabActiveStateChange = false;
   (*_dbSetTabActive).mAlbumTabActiveStateChange = false;
   (*_dbSetTabActive).mGenreTabActiveStateChange = false;
   (*_dbSetTabActive).mFolderTabActiveStateChange = true;
   (*_dbSetTabActive).mPlaylistTabActiveStateChange = false;
   (*_dbSetTabActive).mAudioBkTabActiveStateChange = false;
   (*_dbSetTabActive).mPodcastTabActiveStateChange = false;
   (*_dbSetTabActive).mVideoTabActiveStateChange = false;

   _dbSetTabActive.MarkAllItemsModified();
   _dbSetTabActive.SendUpdate();
}


/* ***********************************************************
* Function 		: setPlaylistTabStateVisibility
* Description 	: sets the visibilty based on the tab selection
* Parameters 	: void
* Return value	: void
* ***********************************************************/
void HMIModelTabImplementation::setPlaylistTabStateVisibility()
{
   ETG_TRACE_USR1(("HMIModelTabImplementation::setPlaylistTabStateVisibility"));

   (*_dbSetTabActive).mTitleTabActiveStateChange = false;
   (*_dbSetTabActive).mArtistTabActiveStateChange = false;
   (*_dbSetTabActive).mAlbumTabActiveStateChange = false;
   (*_dbSetTabActive).mGenreTabActiveStateChange = false;
   (*_dbSetTabActive).mFolderTabActiveStateChange = false;
   (*_dbSetTabActive).mPlaylistTabActiveStateChange = true;
   (*_dbSetTabActive).mAudioBkTabActiveStateChange = false;
   (*_dbSetTabActive).mPodcastTabActiveStateChange = false;
   (*_dbSetTabActive).mVideoTabActiveStateChange = false;

   _dbSetTabActive.MarkAllItemsModified();
   _dbSetTabActive.SendUpdate();
}


/* ***********************************************************
* Function 		: setAudiobookTabStateVisibility
* Description 	: sets the visibilty based on the tab selection
* Parameters 	: void
* Return value	: void
* ***********************************************************/
void HMIModelTabImplementation::setAudiobookTabStateVisibility()
{
   ETG_TRACE_USR1(("HMIModelTabImplementation::setAudiobookTabStateVisibility"));

   (*_dbSetTabActive).mTitleTabActiveStateChange = false;
   (*_dbSetTabActive).mArtistTabActiveStateChange = false;
   (*_dbSetTabActive).mAlbumTabActiveStateChange = false;
   (*_dbSetTabActive).mGenreTabActiveStateChange = false;
   (*_dbSetTabActive).mFolderTabActiveStateChange = false;
   (*_dbSetTabActive).mPlaylistTabActiveStateChange = false;
   (*_dbSetTabActive).mAudioBkTabActiveStateChange = true;
   (*_dbSetTabActive).mPodcastTabActiveStateChange = false;
   (*_dbSetTabActive).mVideoTabActiveStateChange = false;

   _dbSetTabActive.MarkAllItemsModified();
   _dbSetTabActive.SendUpdate();
}


/* ***********************************************************
* Function 		: setPodcastTabStateVisibility
* Description 	: sets the visibilty based on the tab selection
* Parameters 	: void
* Return value	: void
* ***********************************************************/
void HMIModelTabImplementation::setPodcastTabStateVisibility()
{
   ETG_TRACE_USR1(("HMIModelTabImplementation::setPodcastTabStateVisibility"));

   (*_dbSetTabActive).mTitleTabActiveStateChange = false;
   (*_dbSetTabActive).mArtistTabActiveStateChange = false;
   (*_dbSetTabActive).mAlbumTabActiveStateChange = false;
   (*_dbSetTabActive).mGenreTabActiveStateChange = false;
   (*_dbSetTabActive).mFolderTabActiveStateChange = false;
   (*_dbSetTabActive).mPlaylistTabActiveStateChange = false;
   (*_dbSetTabActive).mAudioBkTabActiveStateChange = false;
   (*_dbSetTabActive).mPodcastTabActiveStateChange = true;
   (*_dbSetTabActive).mVideoTabActiveStateChange = false;

   _dbSetTabActive.MarkAllItemsModified();
   _dbSetTabActive.SendUpdate();
}


/* ***********************************************************
* Function 		: setVideoTabStateVisibility
* Description 	: sets the visibilty based on the tab selection
* Parameters 	: void
* Return value	: void
* ***********************************************************/
void HMIModelTabImplementation::setVideoTabStateVisibility()
{
   ETG_TRACE_USR1(("HMIModelTabImplementation::setVideoTabStateVisibility"));

   (*_dbSetTabActive).mTitleTabActiveStateChange = false;
   (*_dbSetTabActive).mArtistTabActiveStateChange = false;
   (*_dbSetTabActive).mAlbumTabActiveStateChange = false;
   (*_dbSetTabActive).mGenreTabActiveStateChange = false;
   (*_dbSetTabActive).mFolderTabActiveStateChange = false;
   (*_dbSetTabActive).mPlaylistTabActiveStateChange = false;
   (*_dbSetTabActive).mAudioBkTabActiveStateChange = false;
   (*_dbSetTabActive).mPodcastTabActiveStateChange = false;
   (*_dbSetTabActive).mVideoTabActiveStateChange = true;

   _dbSetTabActive.MarkAllItemsModified();
   _dbSetTabActive.SendUpdate();
}


/* ***********************************************************
* Function 		: setTabStatus
* Description 	: sets the visibilty of some tabs based on the active source
* Parameters 	: CurrentStatus
* Return value	: void
* ***********************************************************/
void HMIModelTabImplementation::setTabStatus(bool CurrentStatus)
{
   ETG_TRACE_USR4(("HMIModelTabImplementation::setTabStatus CurrentStatus:%d", CurrentStatus));
   if (CurrentStatus == true)
   {
      (*_dbSetTabDisable).mGenreTabDisabledStateChange = true;
      (*_dbSetTabDisable).mPlaylistTabDisabledStateChange = true;
   }
   else
   {
      (*_dbSetTabDisable).mGenreTabDisabledStateChange = false;
      (*_dbSetTabDisable).mPlaylistTabDisabledStateChange = false;
   }
   _dbSetTabDisable.MarkAllItemsModified();
   _dbSetTabDisable.SendUpdate();
}


/* ***********************************************************
* Function 		: setActiveStateVisibility
* Description 	: sets the visibilty based on the tab selection
* Parameters 	: void
* Return value	: void
* ***********************************************************/
void HMIModelTabImplementation::setActiveStateVisibility(uint32_t SelectMediaTab)
{
   ETG_TRACE_USR4(("HMIModelTabImplementation::setActiveStateVisibility mediatabselected:%d", SelectMediaTab));
   setTabStatus(true);
   switch (SelectMediaTab)
   {
      case TAB_TITLE:
      {
         setTitleTabStateVisibility();
         break;
      }
      case TAB_ARTIST:
      {
         setArtistTabStateVisibility();
         break;
      }
      case TAB_ALBUM:
      {
         setAlbumTabStateVisibility();
         break;
      }
      case TAB_GENRE:
      {
         setGenreTabStateVisibility();
         break;
      }
      case TAB_FOLDER:
      {
         setFolderTabStateVisibility();
         break;
      }
      case TAB_PLAYLIST:
      {
         setPlaylistTabStateVisibility();
         break;
      }
      case TAB_AUDIOBOOK:
      {
         setAudiobookTabStateVisibility();
         break;
      }
      case TAB_PODCAST:
      {
         setPodcastTabStateVisibility();
         break;
      }
      case TAB_VIDEO:
      {
         setVideoTabStateVisibility();
         break;
      }
      default:
         break;
   }

   ListRegistry::s_getInstance().updateList(LISTID_MEDIA_HOR);
}


/************************************************************************
*FUNCTION			:vOnRequestTabList
*DESCRIPTION		:update LastBrowseListAppReqMsg to BL when Filter tab is pressed.
*PARAMETER			:u32Listtype
*RETURNVALUE		:None
************************************************************************/

void HMIModelTabImplementation::vOnRequestTabList(int32_t u32Listtype)
{
   ETG_TRACE_USR4(("HMIModelTabImplementation::vOnRequestTabList listtype:%d", u32Listtype));

   ::Courier::Message* poBrowseTabselection = NULL;
   poBrowseTabselection = COURIER_MESSAGE_NEW(::BrowseListReqMsg)(u32Listtype);
   if (poBrowseTabselection != NULL)
   {
      poBrowseTabselection->Post();
   }

   HMIModelListImplementation::getInstance().SetCurrentListType(u32Listtype);
}


}//end of Core
}//end of App
