/* ***************************************************************************************
* FILE:          HMIModelTabImplementation.cpp
* SW-COMPONENT:  HMI-BASE
*  DESCRIPTION:  HMIModelTabImplementation.cpp is part of HMI-Base reference/demo/test applications
*    COPYRIGHT:  (c) 2015-2016 Robert Bosch Car Multimedia GmbH
*
* The reproduction, distribution and utilization of this file as well as the
* communication of its contents to others without express authorization is
* prohibited. Offenders will be held liable for the payment of damages.
* All rights reserved in the event of the grant of a patent, utility model or design.
*
*21.09.17 Rev. 0.1 Sonia (RBEI/ECG5)
* 		  		   Initial Revision.
*************************************************************************************** */

#ifndef VARIANT_S_FTR_ENABLE_UNITTEST
#include "gui_std_if.h"

#include "util/StrUtf8.h"
#include "AppHmi_MediaStateMachine.h"
#endif
#include "HMIModelListImplementation.h"
#ifndef VARIANT_S_FTR_ENABLE_UNITTEST
#include "hmi_trace_if.h"
#include "CgiExtensions/ImageLoader.h"
#include "MPlay_fi_types.h"
#endif
#include <string>
#ifdef VARIANT_S_FTR_ENABLE_UNITTEST
#include "utest/stubs/EnumData.h"
#endif

#ifdef WINDOWS_SIMULATION
#include "DemoApp.h"
#endif

#define ETG_DEFAULT_TRACE_CLASS TR_CLASS_APPHMI_MEDIA_HALL
#ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
#include "trcGenProj/Header/HMIModelListImplementation.cpp.trc.h"
#endif

#define ICON_TITLE 0
#define ICON_ARTIST 1
#define ICON_ALBUM 2
#define ICON_GENRE 3
#define ICON_PLAYLIST 4
#define ICON_AUDIOBOOK 5
#define ICON_PODCAST 6
#define ICON_VIDEO 7
#define ICON_PODCAST_EPISODE 8
#define ICON_AUDIOBOOK_EPISODE 9
#define ICON_INVALID 255
#define SLIDERSIZE 50

namespace App {
namespace Core {

HMIModelListImplementation* HMIModelListImplementation::_theModelListImplementationInstance = 0;

#ifndef VARIANT_S_FTR_ENABLE_UNITTEST
enAnimationReqType HMIModelListImplementation::_waitAnimationState = ANIMATION_STOP ;
#endif

//static const char* const DATA_CONTEXT_TITLE_MENU_LIST_ITEM = "MediaMenuTitle";RY = 2;
static uint32 ListSelectionmap[][2] =
{
   /*List ID when 'Item' is Selected*/      /*List ID When 'All' is Selected*/
   {INVALID_VALUE_MEDIA,            		INVALID_VALUE_MEDIA},
   {LISTID_MEDIA_ALBUMTITLE, 	    		INVALID_VALUE_MEDIA},				/* LTY_ALBUM */
   {PLAY_SONG_IN_LIST, 						INVALID_VALUE_MEDIA},				/* LTY_SONG */
   {PLAY_SONG_IN_LIST, 						INVALID_VALUE_MEDIA},				/* LTY_ALBUM_SONG */
   {LISTID_MEDIA_ARTISTALBUM, 				INVALID_VALUE_MEDIA},				/* LTY_ARTIST */
   {LISTID_MEDIA_FOLDERFILE, 				INVALID_VALUE_MEDIA},				/* LTY_FOLDER */
   {LISTID_MEDIA_ALLALBUMGENRE, 				INVALID_VALUE_MEDIA},				/* LTY_GENRE*/
   {LISTID_MEDIA_ARTISTALBUMTITLE,  		LISTID_MEDIA_ALLTITLEARTIST},		/* LTY_ARTIST_ALBUM */
   {LISTID_MEDIA_PLAYLISTSONG, 				INVALID_VALUE_MEDIA},				/* LTY_PLAYLIST */
   {LISTID_MEDIA_VIDEOEPISODE, 				INVALID_VALUE_MEDIA},				/* LTY_VIDEO */
   {PLAY_SONG_IN_LIST, 						INVALID_VALUE_MEDIA},				/* LTY_ARTIST_ALBUM_SONG*/
   {LISTID_MEDIA_PODCASTEPISODE, 			INVALID_VALUE_MEDIA},				/* LTY_PODCAST */
   { INVALID_VALUE_MEDIA, 			       INVALID_VALUE_MEDIA },		/* LTY_GENRE_ARTIST */
   { INVALID_VALUE_MEDIA, 		           INVALID_VALUE_MEDIA },		/* LTY_GENRE_ARTIST_ALBUM */
   {PLAY_SONG_IN_LIST, 						INVALID_VALUE_MEDIA},				/* LTY_GENRE_ARTIST_ALBUM_SONG */
   {PLAY_SONG_IN_LIST,	 					INVALID_VALUE_MEDIA},				/* LTY_PLAYLISTSONG */
   {PLAY_SONG_IN_LIST,	 					INVALID_VALUE_MEDIA},				/* LTY_VIDEO_EPISODE */
   {LISTID_MEDIA_AUDIOBOOKKEPISODE, 		INVALID_VALUE_MEDIA},				/* LTY_AUDIOBOOK */
   {PLAY_SONG_IN_LIST,	 					INVALID_VALUE_MEDIA},				/* LTY_PODCASTEPISODE */
   {PLAY_SONG_IN_LIST,	 					INVALID_VALUE_MEDIA},				/* LTY_AUDIOBOOKEPISODE */
   {INVALID_VALUE_MEDIA, 					PLAY_SONG_IN_LIST},					/* LTY_ALLTITLE_ARTIST */
   {LISTID_MEDIA_GENREALLALBUMTITLE, 					LISTID_MEDIA_GENREALLTITLE},	/* LTY_ALLALBUM_GENRE */
   {PLAY_SONG_IN_LIST, 					INVALID_VALUE_MEDIA},					/* LTY_GENREALLALBUMTITLE */
   {INVALID_VALUE_MEDIA, 					PLAY_SONG_IN_LIST},					/* LTY_GENREALLTITLE */
   {LISTID_MEDIA_FOLDERFILE, 				INVALID_VALUE_MEDIA},				/* LTY_FOLDER_FILE */
};


/* **********************************************************
* Function 		: getInstance
* Description 	: function to create the HMIModelTabImplementation singleton instance
* Parameters 	: void
* Return value	: _theTabModelInstance - HMIModelTabImplementation object instance
* ***********************************************************/
HMIModelListImplementation& HMIModelListImplementation::getInstance()
{
   if (_theModelListImplementationInstance == 0)
   {
      _theModelListImplementationInstance = new HMIModelListImplementation();
   }

   return *_theModelListImplementationInstance;
}


/* ***********************************************************
* Function 		: removeInstance
* Description 	: function to delete the singleton instance
* Parameters 	: void
* Return value	: void
* ***********************************************************/
void HMIModelListImplementation::removeInstance()
{
   if (_theModelListImplementationInstance)
   {
      delete _theModelListImplementationInstance;
      _theModelListImplementationInstance = 0;
   }
}


/* ***********************************************************
* Function 		: HMIModelListImplementation
* Description 	: constructor
* Parameters 	: void
* Return value	: none
* ***********************************************************/
HMIModelListImplementation::HMIModelListImplementation()
{
#ifndef VARIANT_S_FTR_ENABLE_UNITTEST
   ETG_TRACE_USR1(("HMIModelListImplementation::HMIModelListImplementation constructor"));
   m_currentListType = SM_MEDIA_LISTTYPE_SONG;
   _uint32ListID = 0;
   _uint32WindowSize = 20;
   _uint32StartIndex = 0;
   m_uint32ListSize = 0;
   m_uint32ListType = 0;
   _isHorListTabSelected = false;
   _isNewListEntry = false;
   _listBrowsingHistory.clear();
   _isFolderBrowse = false;
   _isRootFolder = false;
   _isGenreAlbum = false;
   _isSongPlayedFromFolder = false;
   _currentListHandle = 0;
   _isNewListRequest = false;
   _requestedListHandle = 0;
   _currentListId = 0;
   _prevListId = 0;
   _isAllSelected = 0;
   _isPlaySongFromListRequested = false;
   m_u8startindex = 0;

   _currentListSize.clear();
   ListRegistry::s_getInstance().addListImplementation(LISTID_MEDIA_ALBUM, this);
   ListRegistry::s_getInstance().addListImplementation(LISTID_MEDIA_TITLE, this);
   ListRegistry::s_getInstance().addListImplementation(LISTID_MEDIA_ALBUMTITLE, this);
   ListRegistry::s_getInstance().addListImplementation(LISTID_MEDIA_ARTIST, this);
   ListRegistry::s_getInstance().addListImplementation(LISTID_MEDIA_GENRE, this);
   ListRegistry::s_getInstance().addListImplementation(LISTID_MEDIA_ARTISTALBUM, this);
   ListRegistry::s_getInstance().addListImplementation(LISTID_MEDIA_PLAYLIST, this);
   ListRegistry::s_getInstance().addListImplementation(LISTID_MEDIA_VIDEO, this);
   ListRegistry::s_getInstance().addListImplementation(LISTID_MEDIA_ARTISTALBUMTITLE, this);
   ListRegistry::s_getInstance().addListImplementation(LISTID_MEDIA_PODCAST, this);
   //ListRegistry::s_getInstance().addListImplementation(LISTID_MEDIA_GENREARTIST, this);
   //ListRegistry::s_getInstance().addListImplementation(LISTID_MEDIA_GENREARTISTALBUM, this);
   //ListRegistry::s_getInstance().addListImplementation(LISTID_MEDIA_GENREARTISTALBUMSONG, this);
   ListRegistry::s_getInstance().addListImplementation(LISTID_MEDIA_PLAYLISTSONG, this);
   ListRegistry::s_getInstance().addListImplementation(LISTID_MEDIA_VIDEOEPISODE, this);
   ListRegistry::s_getInstance().addListImplementation(LISTID_MEDIA_AUDIOBOOK, this);
   ListRegistry::s_getInstance().addListImplementation(LISTID_MEDIA_PODCASTEPISODE, this);
   ListRegistry::s_getInstance().addListImplementation(LISTID_MEDIA_AUDIOBOOKKEPISODE, this);
   ListRegistry::s_getInstance().addListImplementation(LISTID_MEDIA_ALLTITLEARTIST, this);
   ListRegistry::s_getInstance().addListImplementation(LISTID_MEDIA_ALLALBUMGENRE, this);
   ListRegistry::s_getInstance().addListImplementation(LISTID_MEDIA_GENREALLALBUMTITLE, this);
   ListRegistry::s_getInstance().addListImplementation(LISTID_MEDIA_GENREALLTITLE, this);
   ListRegistry::s_getInstance().addListImplementation(LISTID_MEDIA_FOLDER, this);
   ListRegistry::s_getInstance().addListImplementation(LISTID_MEDIA_FOLDERFILE, this);

   vCreateMapListIDtoListType();
   initVerticalListSize();
#endif
}


#ifndef VARIANT_S_FTR_ENABLE_UNITTEST
void HMIModelListImplementation::initVerticalListSize()
{
   _verticalListsize.DefaultBackButtonSize = 0.0f;
   _verticalListsize.DefaultAllAlbumAndTitleButtonSize = 0.0f;
   _verticalListsize.BackButtonSize = 66.0f;
   _verticalListsize.AllAlbumAndTitleButtonSize = 66.0f;
   _verticalListsize.ListheightWithoutButtons = 330.0f;
   _verticalListsize.ListheightWithBackButton = 264.0f;
   _verticalListsize.ListheightWithBackAndAllButton = 198.0f;
   _verticalListsize.ListWidth = 760.0f;
   _verticalListsize.ScrollBarWidth = 35.0f;
}


#endif

/* ***********************************************************
* Function 		: ~HMIModelTabImplementation
* Description 	: destructor
* Parameters 	: void
* Return value	: none
* ***********************************************************/
HMIModelListImplementation::~HMIModelListImplementation()
{
#ifndef VARIANT_S_FTR_ENABLE_UNITTEST
   _currentListSize.clear();
#endif
   removeInstance();
#ifndef VARIANT_S_FTR_ENABLE_UNITTEST
   ListRegistry::s_getInstance().removeListImplementation(LISTID_MEDIA_ALBUM);
   ListRegistry::s_getInstance().removeListImplementation(LISTID_MEDIA_TITLE);
   ListRegistry::s_getInstance().removeListImplementation(LISTID_MEDIA_ALBUMTITLE);
   ListRegistry::s_getInstance().removeListImplementation(LISTID_MEDIA_ARTIST);
   ListRegistry::s_getInstance().removeListImplementation(LISTID_MEDIA_GENRE);
   ListRegistry::s_getInstance().removeListImplementation(LISTID_MEDIA_ARTISTALBUM);
   ListRegistry::s_getInstance().removeListImplementation(LISTID_MEDIA_PLAYLIST);
   ListRegistry::s_getInstance().removeListImplementation(LISTID_MEDIA_VIDEO);
   ListRegistry::s_getInstance().removeListImplementation(LISTID_MEDIA_ARTISTALBUMTITLE);
   ListRegistry::s_getInstance().removeListImplementation(LISTID_MEDIA_PODCAST);
   //ListRegistry::s_getInstance().removeListImplementation(LISTID_MEDIA_GENREARTIST);
   //ListRegistry::s_getInstance().removeListImplementation(LISTID_MEDIA_GENREARTISTALBUM);
   //ListRegistry::s_getInstance().removeListImplementation(LISTID_MEDIA_GENREARTISTALBUMSONG);
   ListRegistry::s_getInstance().removeListImplementation(LISTID_MEDIA_PLAYLISTSONG);
   ListRegistry::s_getInstance().removeListImplementation(LISTID_MEDIA_VIDEOEPISODE);
   ListRegistry::s_getInstance().removeListImplementation(LISTID_MEDIA_AUDIOBOOK);
   ListRegistry::s_getInstance().removeListImplementation(LISTID_MEDIA_PODCASTEPISODE);
   ListRegistry::s_getInstance().removeListImplementation(LISTID_MEDIA_AUDIOBOOKKEPISODE);
   ListRegistry::s_getInstance().removeListImplementation(LISTID_MEDIA_ALLTITLEARTIST);
   ListRegistry::s_getInstance().removeListImplementation(LISTID_MEDIA_ALLALBUMGENRE);
   ListRegistry::s_getInstance().removeListImplementation(LISTID_MEDIA_GENREALLALBUMTITLE);
   ListRegistry::s_getInstance().removeListImplementation(LISTID_MEDIA_GENREALLTITLE);
   ListRegistry::s_getInstance().removeListImplementation(LISTID_MEDIA_FOLDER);
   ListRegistry::s_getInstance().removeListImplementation(LISTID_MEDIA_FOLDERFILE);
#endif
}


#ifndef VARIANT_S_FTR_ENABLE_UNITTEST
/* ***********************************************************
* Function 		: onCourierMessage
* Description 	: Incomming startup courier message for HMI data model
* Parameters 	: StartupMsg &oMsg
* Return value	: TRUE if it consumes the message else FALSE
* ***********************************************************/
bool HMIModelListImplementation::onCourierMessage(const Courier::StartupMsg& /*oMsg*/)
{
   g_SM_CurrentView = Media_MediaPlayerScenes_M_Album;

   //initialising the list type and added to the map
   createInstanceIfNull(SM_MEDIA_LISTTYPE_SONG);
   //creating map of the listtypes as this is called once once in the beginning
   createListTypeMap();
   return true;
}


void HMIModelListImplementation::createListTypeMap()
{
   ETG_TRACE_USR1(("HMIModelListImplementation::createListTypeMap creating map of the listtypes"));
   uint32_t listType[] = { SM_MEDIA_LISTTYPE_SONG, SM_MEDIA_LISTTYPE_ARTIST, SM_MEDIA_LISTTYPE_ALBUM, SM_MEDIA_LISTTYPE_ALBUM_SONG,
                           SM_MEDIA_LISTTYPE_GENRE, SM_MEDIA_LISTTYPE_ARTIST_ALBUM, SM_MEDIA_LISTTYPE_PLAYLIST, SM_MEDIA_LISTTYPE_VIDEO,
                           SM_MEDIA_LISTTYPE_ARTIST_ALBUM_SONG, SM_MEDIA_LISTTYPE_PODCAST, SM_MEDIA_LISTTYPE_GENRE_ARTIST, SM_MEDIA_LISTTYPE_GENRE_ARTIST_ALBUM,
                           /*SM_MEDIA_LISTTYPE_GENRE_ARTIST_ALBUM_SONG,*/ SM_MEDIA_LISTTYPE_PLAYLIST_SONG, SM_MEDIA_LISTTYPE_VIDEO_EPISODE, SM_MEDIA_LISTTYPE_AUDIOBOOK,
                           SM_MEDIA_LISTTYPE_PODCAST_EPISODE, SM_MEDIA_LISTTYPE_AUDIOBOOKTITLE_CHAPTER, SM_MEDIA_LISTTYPE_ARTIST_SONG, SM_MEDIA_LISTTYPE_GENRE_ALBUM,
                           SM_MEDIA_LISTTYPE_GENRE_ALBUM_SONG, /*SM_MEDIA_LISTTYPE_GENRE_ARTIST_SONG,*/ SM_MEDIA_LISTTYPE_GENRE_SONG, SM_MEDIA_LISTTYPE_FOLDER, SM_MEDIA_LISTTYPE_FOLDER_FILE
                         };

   int iListTypeSize = sizeof(listType) / sizeof(listType[0]);
   ETG_TRACE_USR4(("HMIModelListImplementation::createListTypeMap listtype sizeof(listType) %d", iListTypeSize));
   for (int index = 0; index < iListTypeSize; index++)
   {
      createInstanceIfNull(listType[index]);
   }
}


/* ***********************************************************
* Function 		: onCourierMessage
* Description 	:
* Parameters 	: ListDateProviderReqMsg &oMsg
* Return value	: TRUE if it consumes the message else FALSE
* ***********************************************************/
bool HMIModelListImplementation::onCourierMessage(const ListDateProviderReqMsg& oMsg)
{
   ETG_TRACE_USR4(("HMIModelListImplementation::onCourierMessage:ListDateProviderReqMsg lsitid %d, StartIndex %d, windowelementsize %d", oMsg.GetListId(), oMsg.GetStartIndex(), oMsg.GetWindowElementSize()));

   return ListRegistry::s_getInstance().updateList(oMsg.GetListId(), oMsg.GetStartIndex(), oMsg.GetWindowElementSize());
}


bool HMIModelListImplementation::onCourierMessage(const ListChangedUpdMsg& oMsg)
{
   ETG_TRACE_USR4(("HMIModelListImplementation::onCourierMessage:ListChangedUpdMsg lsitid:%d", oMsg.GetListId()));

   if (oMsg.GetListId() == LISTID_MEDIA_TITLE)
   {
      m_u8startindex = oMsg.GetStartIndex();
      ETG_TRACE_USR4(("HMIModelListImplementation::onCourierMessage:ListChangedUpdMsg statrtindex:%d", m_u8startindex));
      return true;
   }

   return false;
}


/* ***********************************************************
* Function 		: getListDataProvider
* Description 	: Updates the List data Based on the LIST_IDs
* Parameters 	: ListDateProviderReqMsg &oMsg
* Return value	: tSharedPtrDataProvider
* ***********************************************************/
tSharedPtrDataProvider HMIModelListImplementation::getListDataProvider(const ListDateProviderReqMsg& oMsg)
{
   ETG_TRACE_USR4(("HMIModelListImplementation::getListDataProvider list id %u, startindex %d, count %u", oMsg.GetListId(), oMsg.GetStartIndex(), oMsg.GetWindowElementSize()));
   if (oMsg.GetListId() == LISTID_MEDIA_TITLE)
   {
      if (_uint32StartIndex != oMsg.GetStartIndex())
      {
         _isNewListRequest = true;
      }
      else
      {
         _isNewListRequest = false;
      }
      _uint32ListID = oMsg.GetListId();
      _uint32WindowSize = oMsg.GetWindowElementSize();
      _uint32StartIndex = oMsg.GetStartIndex();
      //m_uint32ListType = m_ListIDtoListTypeMap[_uint32ListID];
      m_uint32ListType = m_currentListType;
      ETG_TRACE_USR4(("HMIModelListImplementation::getListDataProvider listType id %d", m_uint32ListType));

      _listMap[m_uint32ListType]->vSetCurrentStartIndex(_uint32StartIndex);
      _listMap[m_uint32ListType]->vSetCurrentWindowSize(_uint32WindowSize);

      //send the request to the middleware for the listdata
      if ((_listMap[m_uint32ListType]->u32GetListSize() > 0))// && (_isNewListRequest))
      {
         _isNewListRequest = false;
         if ((m_uint32ListType == LIST_TYPE_FOLDER) || (m_uint32ListType == LIST_TYPE_FILE))
         {
            tclMediaClientHandler::getInstance().requestFolderSliceList(_uint32StartIndex, _uint32WindowSize);
         }
         else
         {
            tclMediaClientHandler::getInstance().requestMediaPlayerSliceList(_uint32StartIndex, _uint32WindowSize);
         }
      }
      else
      {
         tclMediaClientHandler::getInstance().vClearListDataOnSizeEmpty();
      }
   }
   return tSharedPtrDataProvider();
}


/* ***********************************************************
* Function 		: UpdateFolderList
* Description 	: Updates the folder List data
* Parameters 	: _sFolderListItemText
* Return value	: void
* ***********************************************************/
void HMIModelListImplementation::UpdateFolderList(std::vector<std::string>  _sFolderListItemText)
{
   int32_t _uint32Listsize = 0;
   int32_t _uint32startIndex = 0;
   int32_t _uint32windowSize = 0;

   if (_listMap.count(SM_MEDIA_LISTTYPE_FOLDER))
   {
      _uint32startIndex = _listMap[SM_MEDIA_LISTTYPE_FOLDER]->u32GetCurrentStartIndex();
      _uint32windowSize = _listMap[SM_MEDIA_LISTTYPE_FOLDER]->u32GetCurrentWindowSize();
      _uint32Listsize = _listMap[SM_MEDIA_LISTTYPE_FOLDER]->u32GetListSize();
   }
   else
   {
      ETG_TRACE_USR1(("HMIModelListImplementation::UpdateFolderList :List is not added in _ListMap"));
   }
   ETG_TRACE_USR4(("HMIModelListImplementation::UpdateFolderList startIndex :%d ,windowSize: %d, Listsize: %d ", _uint32startIndex, _uint32windowSize, _uint32Listsize));

   ListDataProviderBuilder listBuilder(LISTID_MEDIA_TITLE);
   FolderListDataItemData data;
   std::vector<std::string>::iterator itr = _sFolderListItemText.begin();

   for (uint32_t dataindex = 0; (itr != _sFolderListItemText.end()); itr++, dataindex++)
   {
      //Icon Display
      bool b_FolderIcon = tclMediaClientHandler::getInstance().bCheckFolder(dataindex);

      if (b_FolderIcon)
      {
         data.mIconImageIndexVal = 0;
      }
      else
      {
         data.mIconImageIndexVal = 1;
      }

      listBuilder.AddItem(
         static_cast<int>(dataindex),
         _currentListHandle,
         "Layer_FolderListEntry_1_1").AddData(_sFolderListItemText[dataindex].c_str())
      .AddDataBindingUpdater<FolderListDataItemDataBindingSource>(data);
   }

   tSharedPtrDataProvider dataProvider = listBuilder.CreateDataProvider(_uint32startIndex, _uint32Listsize);
   updateVerticalListSize(_verticalListsize.DefaultBackButtonSize, _verticalListsize.DefaultAllAlbumAndTitleButtonSize, _verticalListsize.ListheightWithoutButtons, _verticalListsize.ListWidth, _verticalListsize.ScrollBarWidth);
   updateBackAndAllButtonVisibility(false, false);
   updateAllButtonState(false, false, "");
   dataProvider->setCacheOnOff(false);
   ETG_TRACE_USR4(("HMIModelListImplementation::UpdateFolderList startindex :%d", m_u8startindex));
   dataProvider->setListChangeSetIndex(m_u8startindex);

   POST_MSG((COURIER_MESSAGE_NEW(ListDateProviderResMsg)(dataProvider)));
   updateListVisibility(true);
   setWaitAnimationState(ANIMATION_STOP);
}


/* ***********************************************************
* Function 		: UpdateFolderFileList
* Description 	: Updates the folder File List data
* Parameters 	: _sFolderListItemText
* Return value	: void
* ***********************************************************/
void HMIModelListImplementation::UpdateFolderFileList(std::vector<std::string>  _sFolderFileListItemText)
{
   int32_t _uint32Listsize = 0;
   int32_t _uint32startIndex = 0;
   int32_t _uint32windowSize = 0;
   if (_listMap.count(SM_MEDIA_LISTTYPE_FOLDER_FILE))
   {
      _uint32startIndex = _listMap[SM_MEDIA_LISTTYPE_FOLDER_FILE]->u32GetCurrentStartIndex();
      _uint32windowSize = _listMap[SM_MEDIA_LISTTYPE_FOLDER_FILE]->u32GetCurrentWindowSize();
      _uint32Listsize = _listMap[SM_MEDIA_LISTTYPE_FOLDER_FILE]->u32GetListSize();
   }
   else
   {
      ETG_TRACE_USR1(("HMIModelListImplementation::UpdateFolderFileList List is not added in _ListMap"));
   }
   ETG_TRACE_USR4(("HMIModelListImplementation::UpdateFolderFileList startIndex :%d ,windowSize: %d, Listsize: %d ", _uint32startIndex, _uint32windowSize, _uint32Listsize));
   ListDataProviderBuilder listBuilder(LISTID_MEDIA_TITLE);
   FolderFileListDataItemData data;
   std::vector<std::string>::iterator itr = _sFolderFileListItemText.begin();

   for (uint32_t dataindex = 0; (itr != _sFolderFileListItemText.end()); itr++, dataindex++)
   {
      //Icon Display
      bool b_FolderIcon = tclMediaClientHandler::getInstance().bCheckFolder(dataindex);

      if (b_FolderIcon)
      {
         data.mFolderFileIconImageIndex = 0;
      }
      else
      {
         data.mFolderFileIconImageIndex = 1;
      }

      listBuilder.AddItem(
         static_cast<int>(dataindex),
         _currentListHandle,
         "Layer_FolderFileListEntry_1").AddData(_sFolderFileListItemText[dataindex].c_str())
      .AddDataBindingUpdater<FolderFileListDataItemDataBindingSource>(data);
   }

   tSharedPtrDataProvider dataProvider = listBuilder.CreateDataProvider(_uint32startIndex, _uint32Listsize);
   if (_isRootFolder)
   {
      updateVerticalListSize(_verticalListsize.DefaultBackButtonSize, _verticalListsize.DefaultAllAlbumAndTitleButtonSize, _verticalListsize.ListheightWithoutButtons, _verticalListsize.ListWidth, _verticalListsize.ScrollBarWidth);
      updateBackAndAllButtonVisibility(false, false);
      updateAllButtonState(false, false, "");
   }
   else
   {
      updateVerticalListSize(_verticalListsize.BackButtonSize, _verticalListsize.DefaultAllAlbumAndTitleButtonSize, _verticalListsize.ListheightWithBackButton, _verticalListsize.ListWidth, _verticalListsize.ScrollBarWidth);
      updateBackAndAllButtonVisibility(true, false);
      updateAllButtonState(false, false, "");
   }
   dataProvider->setCacheOnOff(false);
   ETG_TRACE_USR4(("HMIModelListImplementation::UpdateFolderFileList startindex :%d", m_u8startindex));

   dataProvider->setListChangeSetIndex(m_u8startindex);

   POST_MSG((COURIER_MESSAGE_NEW(ListDateProviderResMsg)(dataProvider)));
   updateListVisibility(true);
   setWaitAnimationState(ANIMATION_STOP);
}


/* ***********************************************************
* Function 		: UpdateCurrentListSize
* Description 	: Updates the CurrentListSize with the data received from clienthandler
* Parameters 	: u32currentListSize,u32currentListType
* Return value	: void
* ***********************************************************/
void HMIModelListImplementation::UpdateCurrentListSize(uint32_t u32currentListSize, uint32_t u32currentListType)
{
   if (_listMap.count(u32currentListType))
   {
      _listMap[u32currentListType]->vSetListSize(u32currentListSize);
      ETG_TRACE_USR4(("HMIModelListImplementation::UpdateCurrentListSize u32currentListType:%d u32currentListSize::%d", u32currentListType, _listMap[u32currentListType]->u32GetListSize()));
   }
   else
   {
      ETG_TRACE_USR1(("HMIModelListImplementation::UpdateCurrentListSize List is not added in _ListMap"));
   }
}


/* ***********************************************************
* Function 		: ButtonReactionMsg
* Description 	: gets called whenever there is a button selection
* Parameters 	: oMsg
* Return value	: void
* ***********************************************************/
bool HMIModelListImplementation::onCourierMessage(const ButtonReactionMsg& oMsg)
{
   ETG_TRACE_USR1(("HMIModelListImplementation::ButtonReactionMsg"));

   ListProviderEventInfo info;
   if (enRelease == oMsg.GetEnReaction())
   {
      if (::Courier::ViewId("Media#MediaPlayerScenes#M_Album") == oMsg.GetView())
      {
         if ((Courier::Identifier("Button_Back/ButtonWidget") == oMsg.GetSender()))
         {
            vBackButtonListOperation();
         }
         else if ((Courier::Identifier("Button_All/ButtonWidget") == oMsg.GetSender()))
         {
            HandleButtonAllRequest();
         }
         else if ((ListProviderEventInfo::GetItemIdentifierInfo(oMsg.GetSender(), info)) && (info.getListId() == LISTID_MEDIA_TITLE))
         {
            unsigned int listId = info.getListId();     // the list id for generic access
            unsigned int hdlRow = info.getHdlRow();     // normaly the index
            unsigned int hdlCol = info.getHdlCol();     // if more than 1 active element in one list row, e.g. Button in a button line

            ETG_TRACE_USR4(("HMIModelListImplementation::ButtonReactionMsg rowvalue:%d", hdlRow));

            //For each ButtonReactionMsg for buttons which are part of list items we post a ButtonListItemMsg with detailed info about the list item.
            ButtonListItemUpdMsg* msg = COURIER_MESSAGE_NEW(ButtonListItemUpdMsg)(listId, hdlRow, hdlCol, oMsg.GetEnReaction());
            if (msg != NULL)
            {
               msg->Post();
            }
            return true;
         }
      }
   }
   return false;
}


/* ***********************************************************
* Function 		: ButtonListItemUpdMsg
* Description 	: gets called on press of list item
* Parameters 	: oMsg
* Return value	: bool
* ***********************************************************/
bool HMIModelListImplementation::onCourierMessage(const ButtonListItemUpdMsg& oMsg)
{
   ETG_TRACE_USR1(("HMIModelListImplementation::ButtonListItemUpdMsg"));

   if (oMsg.GetReaction() == enRelease)
   {
      unsigned int listId = oMsg.GetListId();
      unsigned int hdlRow = oMsg.GetHdl();
      unsigned int hdlCol = oMsg.GetSubHdl();
      ETG_TRACE_USR4(("HMIModelListImplementation::ButtonListItemUpdMsg listId:%d, hdlrow:%d, hdlcol:%d", listId, hdlRow, hdlCol));

      if (listId != LISTID_MEDIA_HOR && listId != LIST_ID_MEDIA_SOURCE_LIST && listId != LIST_ID_EXTERNAL_MEDIA_VER && listId != LISTID_EXTERNAL_MEDIA_HOR)
      {
         if ((_requestedListHandle != hdlCol) || (_isSongPlayedFromFolder) || (_isPlaySongFromListRequested))
         {
            listId = getNextListId();
            ETG_TRACE_USR3(("HMIModelListImplementation::ButtonListItemUpdMsg listid:%d", listId));

            if (listId == PLAY_SONG_IN_LIST)
            {
               _isNewListEntry = false;
               tclMediaClientHandler::getInstance().vPlayitemRequest(hdlRow);
            }
            else
            {
               m_u8startindex = 0;
               //if (!_isWaitingForListCompletion)
               {
                  ETG_TRACE_USR4(("HMIModelListImplementation::ButtonListItemUpdMsg: _requestedListHandle: %d", _requestedListHandle));
                  ETG_TRACE_USR4(("HMIModelListImplementation::ButtonListItemUpdMsg: hdlCol : %d", hdlCol));
                  if ((_requestedListHandle != hdlCol) || (_isSongPlayedFromFolder))
                  {
                     _requestedListHandle = hdlCol;
                     _isSongPlayedFromFolder = false;
                     _isNewListEntry = true;
                     _listBrowsingHistory.push_back(listId);
                     tclMediaClientHandler::getInstance().initStatemachine();
                     browseListSetting(listId, hdlRow);   //To play the item selected in the corresponding list
                  }
               }
            }
         }
      }
   }
   return false;
}


/* ***********************************************************
* Function 		: vBackButtonListOperation
* Description 	: Setting visibily of other lists on buttonreaction of BackButton
* Parameters 	: oMsg
* Return value	: void
* ***********************************************************/
void HMIModelListImplementation::vBackButtonListOperation()
{
   uint32 listId = 0;
   _isNewListEntry = true;
   _requestedListHandle = 0;
   _isPlaySongFromListRequested = false;
   ETG_TRACE_USR4(("HMIModelListImplementation::vBackButtonListOperation _requestedListHandle:%d", _requestedListHandle));
   ETG_TRACE_USR4(("HMIModelListImplementation::vBackButtonListOperation size:%d", _listBrowsingHistory.size()));

   if (!_listBrowsingHistory.empty())
   {
      ETG_TRACE_USR4(("HMIModelListImplementation::vBackButtonListOperation startindex:%d", m_u8startindex));
      m_u8startindex = 0;

      _listBrowsingHistory.pop_back();
      listId = _listBrowsingHistory.back();

      _currentListId = _listBrowsingHistory.back();
      ETG_TRACE_USR4(("HMIModelListImplementation::vBackButtonListOperation listid:%d ,currentlistid:%d", listId, _currentListId));

      updateListSize();
      if ((_currentListId == LISTID_MEDIA_ARTISTALBUM) || (_currentListId == LISTID_MEDIA_ALLALBUMGENRE)) /*||(_currentListId == LISTID_MEDIA_GENREARTIST) || (_currentListId == LISTID_MEDIA_GENREARTISTALBUM)*/
      {
         _isAllSelected = false;
      }
      if ((_listBrowsingHistory.size() == 1))
      {
         _isRootFolder = true;
      }
      else
      {
         _isRootFolder = false;
      }
      if (!(_isFolderBrowse))
      {
         tclMediaClientHandler::getInstance().vClearListHandle();
         SetCurrentListType(m_ListIDtoListTypeMap[listId]);
         tclMediaClientHandler::getInstance().getListHandle(m_ListIDtoListTypeMap[listId]);
         setWaitAnimationState(ANIMATION_START);
         tclMediaClientHandler::getInstance().requestNewListEntry();
         ::Courier::Message* poBrowseTabselection = NULL;
         poBrowseTabselection = COURIER_MESSAGE_NEW(::BrowseListReqMsg)(m_ListIDtoListTypeMap[listId]);
         if (poBrowseTabselection != NULL)
         {
            poBrowseTabselection->Post();
         }
      }
      else
      {
         SetCurrentListType(SM_MEDIA_LISTTYPE_FOLDER_FILE);
         tclMediaClientHandler::getInstance().populateListFromHistory();
         setWaitAnimationState(ANIMATION_START);
         tclMediaClientHandler::getInstance().requestNewListEntry();
         tclMediaClientHandler::getInstance().vHandleParentLevelFolderListRequest();
      }
   }
}


void HMIModelListImplementation::HandleButtonAllRequest()
{
   ETG_TRACE_USR1(("HMIModelListImplementation::HandleButtonAllRequest"));
   _isAllSelected = true;
   _isNewListEntry = true;
   _isPlaySongFromListRequested = false;
   m_u8startindex = 0;

   unsigned int listId = getNextListId();
   ETG_TRACE_USR4(("HMIModelListImplementation::HandleButtonAllRequest listid:%d", listId));

   _listBrowsingHistory.push_back(listId);
   updateListSize();
   tclMediaClientHandler::getInstance().initStatemachine();
   ::Courier::Message* poBrowseTabselection = NULL;
   poBrowseTabselection = COURIER_MESSAGE_NEW(::BrowseListReqMsg)(m_ListIDtoListTypeMap[listId]);
   if (poBrowseTabselection != NULL)
   {
      poBrowseTabselection->Post();
   }
}


void HMIModelListImplementation::updateVerticalListSize(float backButtonSize, float allAlbumAndTitleButtonSize, float verticalListSize, float verticalListWidth, float scrollBarWidth)
{
   float sliderSize = 0.0f;
   ETG_TRACE_USR4(("HMIModelListImplementation::updateVerticalListSize:backButtonSize:%f ,allAlbumAndTitleButtonSize:%f,verticalListSize: %f", backButtonSize, allAlbumAndTitleButtonSize, verticalListSize));
   if (verticalListSize > 0)
   {
      sliderSize = (SLIDERSIZE / verticalListSize);
   }
   ETG_TRACE_USR4(("HMIModelListImplementation::updateVerticalListSize: sliderSize %f ", sliderSize));
   (*m_BrowserScreenListSize).mPreviousButtonSize.SetX(verticalListWidth);
   (*m_BrowserScreenListSize).mPreviousButtonSize.SetY(backButtonSize);
   (*m_BrowserScreenListSize).mAllAlbumAndTitleButtonSize.SetX(verticalListWidth);
   (*m_BrowserScreenListSize).mAllAlbumAndTitleButtonSize.SetY(allAlbumAndTitleButtonSize);
   (*m_BrowserScreenListSize).mVerticalListSize.SetX(verticalListWidth);
   (*m_BrowserScreenListSize).mVerticalListSize.SetY(verticalListSize);
   (*m_BrowserScreenListSize).mScrollBarSize.SetX(scrollBarWidth);
   (*m_BrowserScreenListSize).mScrollBarSize.SetY(verticalListSize);
   (*m_BrowserScreenListSize).mScrollBarSliderSize = sliderSize;
   m_BrowserScreenListSize.MarkAllItemsModified();
   m_BrowserScreenListSize.SendUpdate();
}


void HMIModelListImplementation::updateBackAndAllButtonVisibility(bool backBtnVisibility, bool allBtnVisibility)
{
   ETG_TRACE_USR4(("HMIModelListImplementation::updateBackAndAllButtonVisibility:%d %d ", backBtnVisibility, allBtnVisibility));
   (*m_BrowserScreenListSize).mPreviousButtonVisibility = backBtnVisibility;
   (*m_BrowserScreenListSize).mAllAlbumAndTitleButtonVisibility = allBtnVisibility;
   m_BrowserScreenListSize.MarkAllItemsModified();
   m_BrowserScreenListSize.SendUpdate();
}


void HMIModelListImplementation::updateListVisibility(bool listVisibility)
{
   ETG_TRACE_USR4(("HMIModelListImplementation::updateListVisibility:listvisibility:%d ", listVisibility));
   (*m_BrowserScreenListSize).mListVisibility = listVisibility;
   m_BrowserScreenListSize.MarkAllItemsModified();
   m_BrowserScreenListSize.SendUpdate();
}


void HMIModelListImplementation::updateAllButtonState(bool titleIconVisibility, bool albumIconVisibility, Candera::String text)
{
   ETG_TRACE_USR4(("HMIModelListImplementation::updateAllButtonState:titleiconvisibility:%d,albumiconvisibility: %d ", titleIconVisibility, albumIconVisibility));
   ETG_TRACE_USR4(("HMIModelListImplementation::updateAllButtonState:text:%s ", text));
   (*m_BrowserScreenListSize).mTitleIconVisibility = titleIconVisibility;
   (*m_BrowserScreenListSize).mAlbumIconVisibility = albumIconVisibility;
   (*m_BrowserScreenListSize).mAllAlbumAndTitleText = text;
   m_BrowserScreenListSize.MarkAllItemsModified();
   m_BrowserScreenListSize.SendUpdate();
}


void HMIModelListImplementation::updateIsNewListRequest(bool status)
{
   _isNewListRequest = status;
   ETG_TRACE_USR4(("HMIModelListImplementation::updateIsNewListRequest _isNewListRequest:%d ", _isNewListRequest));
}


void HMIModelListImplementation::populateEmptyList(bool isPreviousButtonAvailable)
{
   ETG_TRACE_USR4(("HMIModelListImplementation::populateEmptyList isPreviousButtonAvailable:%d ", isPreviousButtonAvailable));

   if (isPreviousButtonAvailable)
   {
      updateVerticalListSize(_verticalListsize.BackButtonSize, _verticalListsize.DefaultAllAlbumAndTitleButtonSize, 0.0f, 0.0f, 0.0f);
      updateBackAndAllButtonVisibility(true, false);
      updateListVisibility(false);
      updateAllButtonState(false, false, "");
   }
   else
   {
      updateVerticalListSize(_verticalListsize.DefaultBackButtonSize, _verticalListsize.DefaultAllAlbumAndTitleButtonSize, 0.0f, 0.0f, 0.0f);
      updateBackAndAllButtonVisibility(false, false);
      updateListVisibility(false);
      updateAllButtonState(false, false, "");
   }
}


void HMIModelListImplementation::updateListSize()
{
   ETG_TRACE_USR4(("HMIModelListImplementation::updateListSize currentlistid:%d ", _currentListId));

   if ((_currentListId == LISTID_MEDIA_TITLE) || (_currentListId == LISTID_MEDIA_ALBUM) || (_currentListId == LISTID_MEDIA_ARTIST) || (_currentListId == LISTID_MEDIA_GENRE)
         || (_currentListId == LISTID_MEDIA_FOLDER) || (_currentListId == LISTID_MEDIA_PLAYLIST)  || (_currentListId == LISTID_MEDIA_PODCAST)
         || (_currentListId == LISTID_MEDIA_AUDIOBOOK) || (_currentListId == LISTID_MEDIA_VIDEO))
   {
      updateVerticalListSize(_verticalListsize.DefaultBackButtonSize, _verticalListsize.DefaultAllAlbumAndTitleButtonSize, _verticalListsize.ListheightWithoutButtons, _verticalListsize.ListWidth, _verticalListsize.ScrollBarWidth);
      updateBackAndAllButtonVisibility(false, false);
      updateListVisibility(false);
   }
   else if ((_currentListId == LISTID_MEDIA_ARTISTALBUM) || (_currentListId == LISTID_MEDIA_ALLALBUMGENRE))    //|| (_currentListId == LISTID_MEDIA_GENREARTIST) || (_currentListId == LISTID_MEDIA_GENREARTISTALBUM))
   {
      updateVerticalListSize(_verticalListsize.BackButtonSize, _verticalListsize.AllAlbumAndTitleButtonSize, _verticalListsize.ListheightWithBackAndAllButton, _verticalListsize.ListWidth, _verticalListsize.ScrollBarWidth);
      updateBackAndAllButtonVisibility(false, false);
      updateListVisibility(false);
   }
   else
   {
      updateVerticalListSize(_verticalListsize.BackButtonSize, _verticalListsize.DefaultAllAlbumAndTitleButtonSize, _verticalListsize.ListheightWithBackButton, _verticalListsize.ListWidth, _verticalListsize.ScrollBarWidth);
      updateBackAndAllButtonVisibility(false, false);
      updateListVisibility(false);
   }
}


void HMIModelListImplementation::updateListHandle(uint32 listHandle)
{
   _currentListHandle = listHandle;
   ETG_TRACE_USR4(("HMIModelListImplementation::updateListHandle currentlisthandle:%d ", _currentListHandle));
}


void HMIModelListImplementation::updateRequestedListHandle(uint32 requstedListHandle)
{
   _requestedListHandle = requstedListHandle;
   ETG_TRACE_USR4(("HMIModelListImplementation::updateRequestedListHandle requestedlisthandle:%d ", _requestedListHandle));
}


void HMIModelListImplementation::updateCurrentListId(uint32 currentListId)
{
   ETG_TRACE_USR4(("HMIModelListImplementation::updateCurrentListId:%d", currentListId));
   _currentListId = currentListId;
   _listBrowsingHistory.push_back(currentListId);
}


void HMIModelListImplementation::IsHorListTabSelected(bool status)
{
   ETG_TRACE_USR4(("HMIModelListImplementation::IsHorListTabSelected:%d", status));
   _isHorListTabSelected = status;
}


void HMIModelListImplementation::IsAllSelected(bool status)
{
   ETG_TRACE_USR4(("HMIModelListImplementation::IsAllSelected:%d", status));
   _isAllSelected = status;
}


void HMIModelListImplementation::clearListBrowsingHistory()
{
   ETG_TRACE_USR1(("HMIModelListImplementation::clearListBrowsingHistory"));
   _listBrowsingHistory.clear();
}


#endif


uint32 HMIModelListImplementation::getNextListId()
{
   uint32 nextListId = 0;
#ifndef VARIANT_S_FTR_ENABLE_UNITTEST
   ETG_TRACE_USR4(("HMIModelListImplementation::getNextListId currentListId: %d", _currentListId));
#endif
   if ((_currentListId >= LISTID_MEDIA_HOR) && (_currentListId <= LISTID_MEDIA_FOLDERFILE))
   {
      nextListId = ListSelectionmap[_currentListId % LISTID_MEDIA_HOR][_isAllSelected];
#ifndef VARIANT_S_FTR_ENABLE_UNITTEST
      ETG_TRACE_USR4(("HMIModelListImplementation::getNextListId nextListId:%d", nextListId));
#endif
      if (nextListId != PLAY_SONG_IN_LIST)
      {
         _prevListId = _currentListId;
         _currentListId = nextListId;
         _isPlaySongFromListRequested = false;
      }
      else
      {
         _isPlaySongFromListRequested = true;
      }
   }
#ifndef VARIANT_S_FTR_ENABLE_UNITTEST
   ETG_TRACE_USR4(("HMIModelListImplementation::getNextListId: nextListId:%d", nextListId));
   ETG_TRACE_USR4(("HMIModelListImplementation::getNextListId: _currentListId: %d", _currentListId));
#endif
   return nextListId;
}


bool HMIModelListImplementation::IsSongListId(uint32 listId)
{
   bool status = false;
#ifndef VARIANT_S_FTR_ENABLE_UNITTEST
   ETG_TRACE_USR4(("HMIModelListImplementation::IsSongListId ListId:%d", listId));
#endif


   if ((listId == LISTID_MEDIA_TITLE) || (listId == LISTID_MEDIA_ALBUMTITLE) || (listId == LISTID_MEDIA_ARTISTALBUMTITLE) || (listId == LISTID_MEDIA_GENREARTISTALBUMSONG)
         || (listId == LISTID_MEDIA_ALLTITLEARTIST) || (listId == LISTID_MEDIA_GENREALLALBUMTITLE) || (listId == LISTID_MEDIA_GENREALLTITLE))
   {
      status = true;
   }
#ifndef VARIANT_S_FTR_ENABLE_UNITTEST
   ETG_TRACE_USR4(("HMIModelListImplementation::IsSongListId status:%d", status));
#endif

   return status;
}


#ifdef VARIANT_S_FTR_ENABLE_UNITTEST

void HMIModelListImplementation::setCurrentListId(uint32 currentListId)
{
   _currentListId = currentListId;
}


void HMIModelListImplementation::setIsAllSelected(bool isAllSelected)
{
   _isAllSelected = isAllSelected;
}


#endif


#ifndef VARIANT_S_FTR_ENABLE_UNITTEST
void HMIModelListImplementation::IsFolderBrowse(bool status)
{
   ETG_TRACE_USR4(("HMIModelListImplementation::IsFolderBrowse status: %d", status));
   _isFolderBrowse = status;
}


void HMIModelListImplementation::IsNewList(bool status)
{
   ETG_TRACE_USR4(("HMIModelListImplementation::IsNewList status: %d", status));
   _isNewListEntry = status;
}


bool HMIModelListImplementation::getIsNewList()
{
   ETG_TRACE_USR4(("HMIModelListImplementation::getIsNewList newListentry:%d", _isNewListEntry));

   return _isNewListEntry;
}


uint32_t HMIModelListImplementation::getVerticalStartIndex()
{
   ETG_TRACE_USR4(("HMIModelListImplementation::getVerticalStartindex startindex:%d", m_u8startindex));
   return m_u8startindex;
}


void HMIModelListImplementation::setVerticalStartIndex(uint32_t startIndex)
{
   ETG_TRACE_USR4(("HMIModelListImplementation::setVerticalStartIndex start index%d", startIndex));

   m_u8startindex = startIndex;
}


void HMIModelListImplementation::IsSongPlayedFromFolder()
{
   ETG_TRACE_USR1(("HMIModelListImplementation::IsSongPlayedFromFolder"));
   _isSongPlayedFromFolder = true;
   _listBrowsingHistory.pop_back();
}


void HMIModelListImplementation::browseListSetting(uint32_t listId, uint32_t hdlRow)//naga changed
{
   ETG_TRACE_USR4(("HMIModelListImplementation::browseListSetting listId:%d ,hdlrow:%d", listId, hdlRow));
   switch (listId)
   {
      case LISTID_MEDIA_FOLDERFILE:
      {
         _isRootFolder = false;

         if (_prevListId == LISTID_MEDIA_FOLDER)
         {
            tclMediaClientHandler::getInstance().HandleBrowseFolderSubListRequest(SM_MEDIA_LISTTYPE_FOLDER, SM_MEDIA_LISTTYPE_FOLDER_FILE, hdlRow);
         }
         else
         {
            tclMediaClientHandler::getInstance().HandleBrowseFolderSubListRequest(SM_MEDIA_LISTTYPE_FOLDER_FILE, SM_MEDIA_LISTTYPE_FOLDER_FILE, hdlRow);
         }
         break;
      }
      default:
      {
         updateListSize();
         SetCurrentListType(m_ListIDtoListTypeMap[listId]);
         tclMediaClientHandler::getInstance().BrowseListItemSelect(m_ListIDtoListTypeMap[listId], hdlRow);
         break;
      }
   }
}


void HMIModelListImplementation::ClearListType()
{
   ETG_TRACE_USR1(("HMIModelListImplementation::ClearListType"));
   g_SM_ListType = SM_MEDIA_LISTTYPE_INVALID;
}


/* ***********************************************************
* Function 		: calculateWindowSize
* Description 	: Calculates the windowsize
* Parameters 	: windowstart,windowsize,ListSize
* Return value	: void
* ***********************************************************/
//uint32_t HMIModelListImplementation::calculateWindowSize(uint32_t windowstart, uint32_t windowsize, uint32_t ListSize)
//{
//	uint32_t lastItemIndex_List = windowstart + windowsize ;
//	if ((lastItemIndex_List == 0) || (lastItemIndex_List > ListSize))
//	{
//		lastItemIndex_List = ListSize;
//	}
//	return lastItemIndex_List;
//}
/* ***********************************************************
* Function 		: SetCurrentListType
* Description 	: Updating the current list type
* Parameters 	: oMsg
* Return value	: void
* ***********************************************************/
void HMIModelListImplementation::SetCurrentListType(uint32_t currentListType)
{
   ETG_TRACE_USR4(("HMIModelListImplementation::currentListType currentListType: %d", currentListType));
   m_currentListType = currentListType;
   createInstanceIfNull(m_currentListType);
}


/**
* vCreateInstanceIfNull - Creats a new database for list type
* @param[in] u8ListType
* @parm[out] none
* @return void
*/
void HMIModelListImplementation::createInstanceIfNull(uint32 listType)
{
   ETG_TRACE_USR4(("HMIModelListImplementation::createInstanceIfNull listType %d", listType));

   std::map<uint32, ListUtilityDatabase*>::iterator Iter_DB = _listMap.find(listType);
   if (Iter_DB == _listMap.end())
   {
      ListUtilityDatabase* listUtilityDatabase = new ListUtilityDatabase();
      insertNewUtilityDatabaseToListMap(listType, listUtilityDatabase);
   }
   else
   {
      ETG_TRACE_USR1(("HMIModelListImplementation::vCreateInstanceIfNull ListUtilityDatabase is already created"));
   }
}


/**
*  insertNewUtilityDatabaseToListMap - inserts the listUtilityDatabase for a 'listType' to the '_listMap'
*  @param [in] listType
*  @param [in] listUtilityDatabase
*  @return void
*/
void HMIModelListImplementation::insertNewUtilityDatabaseToListMap(uint32 listType, ListUtilityDatabase* listUtilityDatabase)
{
   ETG_TRACE_USR4(("HMIModelListImplementation::insertNewUtilityDatabaseToListMap listtype:%d", listType));

   if (listUtilityDatabase)
   {
      _listMap.insert(std::pair<uint32, ListUtilityDatabase* >(listType, listUtilityDatabase));
   }
   else
   {
   }
}


/* ***********************************************************
* Function 		: OneItemListDataProviderToGui
* Description 	: Updating the list according to current list type
* Parameters 	: _ListItemText,u32CurrentListType
* Return value	: void
* ***********************************************************/
void HMIModelListImplementation::OneItemListDataProviderToGui(std::vector<std::string> _ListItemText, uint32_t u32CurrentListType)
{
   ETG_TRACE_USR4(("HMIModelListImplementation::OneItemListDataProviderToGui CurrentListType = %d", u32CurrentListType));

   uint32 dataIcon = updateDataIconInList(u32CurrentListType);

   if (dataIcon != ICON_INVALID)
   {
      int32_t _uint32Listsize = 0;
      int32_t _uint32startIndex = 0;
      int32_t _uint32windowSize = 0;
      if (_listMap.count(u32CurrentListType))
      {
         _uint32startIndex = _listMap[u32CurrentListType]->u32GetCurrentStartIndex();
         _uint32windowSize = _listMap[u32CurrentListType]->u32GetCurrentWindowSize();
         _uint32Listsize = _listMap[u32CurrentListType]->u32GetListSize();
      }
      else
      {
         ETG_TRACE_USR1(("HMIModelListImplementation::OneItemListDataProviderToGui List is not added in _ListMap"));
      }
      ETG_TRACE_USR4(("HMIModelListImplementation::OneItemListDataProviderToGui startindex: %d,windowsize :%d, listsize:%d", _uint32startIndex, _uint32windowSize, _uint32Listsize));

      ListDataProviderBuilder listBuilder(LISTID_MEDIA_TITLE);

      std::vector<std::string>::iterator itr = _ListItemText.begin();

      for (uint32_t dataindex = 0; (itr != _ListItemText.end()); itr++, dataindex++)
      {
         listBuilder.AddItem(
            static_cast<int>(dataindex),
            _currentListHandle,
            "Layer_Two_Item").AddData(_ListItemText[dataindex].c_str()).AddData(dataIcon);
      }

      tSharedPtrDataProvider dataProvider = listBuilder.CreateDataProvider(_uint32startIndex, _uint32Listsize);
      updateSizeAndVisibilityDuringListBuild(u32CurrentListType);
      dataProvider->setCacheOnOff(false);
      ETG_TRACE_USR4(("HMIModelListImplementation::OneItemListDataProviderToGui startindex%d", m_u8startindex));
      dataProvider->setListChangeSetIndex(m_u8startindex);

      POST_MSG((COURIER_MESSAGE_NEW(ListDateProviderResMsg)(dataProvider)));
      updateListVisibility(true);
      setWaitAnimationState(ANIMATION_STOP);
   }
}


/* ***********************************************************
* Function 		: TwoItemListDataProviderToGui
* Description 	: Updating the list according to current list type
* Parameters 	: _ListItemText,u32CurrentListType
* Return value	: void
* ***********************************************************/
void HMIModelListImplementation::TwoItemListDataProviderToGui(std::vector<media_TwoTextListInfo> _ListItemText, uint32_t u32CurrentListType)
{
   ETG_TRACE_USR4(("HMIModelListImplementation::TwoItemListDataProviderToGui u32CurrentListType = %d", u32CurrentListType));

   uint32 dataIcon = updateDataIconInList(u32CurrentListType);

   if (dataIcon != ICON_INVALID)
   {
      int32_t _uint32Listsize = 0;
      int32_t _uint32startIndex = 0;
      int32_t _uint32windowSize = 0;
      if (_listMap.count(u32CurrentListType))
      {
         _uint32startIndex = _listMap[u32CurrentListType]->u32GetCurrentStartIndex();
         _uint32windowSize = _listMap[u32CurrentListType]->u32GetCurrentWindowSize();
         _uint32Listsize = _listMap[u32CurrentListType]->u32GetListSize();
      }
      else
      {
         ETG_TRACE_USR1(("HMIModelListImplementation::TwoItemListDataProviderToGui List is not added in _ListMap"));
      }
      ETG_TRACE_USR4(("HMIModelListImplementation::TwoItemListDataProviderToGui: startindex: %d,windowsize :%d, listsize:%d", _uint32startIndex, _uint32windowSize, _uint32Listsize));

      ListDataProviderBuilder listBuilder(LISTID_MEDIA_TITLE);

      std::vector<media_TwoTextListInfo>::iterator itr = _ListItemText.begin();

      for (uint32_t dataindex = 0; itr != _ListItemText.end(); itr++, dataindex++)
      {
         listBuilder.AddItem(
            static_cast<int>(dataindex),
            _currentListHandle,
            "Layer_Three_Item")
         .AddData(_ListItemText[dataindex].szFirstText.c_str())
         .AddData(_ListItemText[dataindex].szSecondText.c_str())
         .AddData(dataIcon);
      }

      tSharedPtrDataProvider dataProvider = listBuilder.CreateDataProvider(_uint32startIndex, _uint32Listsize);
      updateSizeAndVisibilityDuringListBuild(u32CurrentListType);
      dataProvider->setCacheOnOff(false);
      dataProvider->setListChangeSetIndex(m_u8startindex);
      ETG_TRACE_USR4(("HMIModelListImplementation::TwoItemListDataProviderToGui m_u8startindex:%d", m_u8startindex));

      POST_MSG((COURIER_MESSAGE_NEW(ListDateProviderResMsg)(dataProvider)));
      updateListVisibility(true);
      setWaitAnimationState(ANIMATION_STOP);
   }
}


void HMIModelListImplementation::updateSizeAndVisibilityDuringListBuild(uint32_t u32CurrentListType)
{
   ETG_TRACE_USR4(("HMIModelListImplementation::updateSizeAndVisibilityDuringListBuild currentlisttype: %d", u32CurrentListType));

   if ((u32CurrentListType == SM_MEDIA_LISTTYPE_SONG) || (u32CurrentListType == SM_MEDIA_LISTTYPE_ARTIST) || (u32CurrentListType == SM_MEDIA_LISTTYPE_ALBUM)
         || (u32CurrentListType == SM_MEDIA_LISTTYPE_GENRE) || (u32CurrentListType == SM_MEDIA_LISTTYPE_FOLDER) || (u32CurrentListType == SM_MEDIA_LISTTYPE_PLAYLIST)
         || (u32CurrentListType == SM_MEDIA_LISTTYPE_PODCAST) || (u32CurrentListType == SM_MEDIA_LISTTYPE_AUDIOBOOK) || (u32CurrentListType == SM_MEDIA_LISTTYPE_VIDEO))
   {
      updateVerticalListSize(_verticalListsize.DefaultBackButtonSize, _verticalListsize.DefaultAllAlbumAndTitleButtonSize, _verticalListsize.ListheightWithoutButtons, _verticalListsize.ListWidth, _verticalListsize.ScrollBarWidth);
      updateBackAndAllButtonVisibility(false, false);
      //updateListVisibility(true);
      updateAllButtonState(false, false, "");
   }
   /* else if (u32CurrentListType == SM_MEDIA_LISTTYPE_GENRE_ARTIST)
    {
       Candera::String strAlbum = HMIModelComponent::getInstance().GetListAllAlbumId();
       updateVerticalListSize(_verticalListsize.BackButtonSize, _verticalListsize.AllAlbumAndTitleButtonSize, _verticalListsize.ListheightWithBackAndAllButton, _verticalListsize.ListWidth, _verticalListsize.ScrollBarWidth);
       updateBackAndAllButtonVisibility(true, true);
       //updateListVisibility(true);
       updateAllButtonState(false, true, strAlbum);
    }*/
   else if (/*(u32CurrentListType == SM_MEDIA_LISTTYPE_GENRE_ARTIST_ALBUM)||*/(u32CurrentListType == SM_MEDIA_LISTTYPE_ARTIST_ALBUM) || (u32CurrentListType == SM_MEDIA_LISTTYPE_GENRE_ALBUM))
   {
      Candera::String strTitle = HMIModelComponent::getInstance().GetListAllTitleId();
      updateVerticalListSize(_verticalListsize.BackButtonSize, _verticalListsize.AllAlbumAndTitleButtonSize, _verticalListsize.ListheightWithBackAndAllButton, _verticalListsize.ListWidth, _verticalListsize.ScrollBarWidth);
      updateBackAndAllButtonVisibility(true, true);
      //updateListVisibility(true);
      updateAllButtonState(true, false, strTitle);
   }
   else
   {
      updateVerticalListSize(_verticalListsize.BackButtonSize, _verticalListsize.DefaultAllAlbumAndTitleButtonSize, _verticalListsize.ListheightWithBackButton, _verticalListsize.ListWidth, _verticalListsize.ScrollBarWidth);
      updateBackAndAllButtonVisibility(true, false);
      //updateListVisibility(true);
      updateAllButtonState(false, false, "");
   }
}


void HMIModelListImplementation::setWaitAnimationState(enAnimationReqType state)
{
   if (((ANIMATION_STOP == state) && (ANIMATION_START == _waitAnimationState)) || (ANIMATION_START == state))
   {
      _waitAnimationState = state;
      Courier::Message* pMsg = COURIER_MESSAGE_NEW(WaitAnimationReqMsg)(_waitAnimationState);
      if (pMsg)
      {
         pMsg->Post();
      }
   }
}


void HMIModelListImplementation::setIsPlaySongFromListRequested(bool state)
{
   ETG_TRACE_USR4(("HMIModelListImplementation::setIsPlaySongFromListRequested: %d", state));
   _isPlaySongFromListRequested = state;
}


#endif


uint32 HMIModelListImplementation::updateDataIconInList(uint32_t u32CurrentListType)
{
#ifndef VARIANT_S_FTR_ENABLE_UNITTEST
   ETG_TRACE_USR4(("HMIModelListImplementation::updateDataIconInList currentlisttype: %d", u32CurrentListType));
#endif
   uint32 returnDataIcon = ICON_INVALID;
   switch (u32CurrentListType)
   {
      case SM_MEDIA_LISTTYPE_SONG:
      case SM_MEDIA_LISTTYPE_ALBUM_SONG:
      case SM_MEDIA_LISTTYPE_PLAYLIST_SONG:
      case SM_MEDIA_LISTTYPE_ARTIST_SONG:
      case SM_MEDIA_LISTTYPE_GENRE_ALBUM_SONG:
      case SM_MEDIA_LISTTYPE_ARTIST_ALBUM_SONG:
      /* case SM_MEDIA_LISTTYPE_GENRE_ARTIST_SONG:
       case SM_MEDIA_LISTTYPE_GENRE_ARTIST_ALBUM_SONG:*/
      case SM_MEDIA_LISTTYPE_GENRE_SONG:
      {
         returnDataIcon = ICON_TITLE;
         break;
      }
      case SM_MEDIA_LISTTYPE_ALBUM:
      case SM_MEDIA_LISTTYPE_GENRE_ALBUM:
      case SM_MEDIA_LISTTYPE_ARTIST_ALBUM:
      case SM_MEDIA_LISTTYPE_GENRE_ARTIST_ALBUM:
      {
         returnDataIcon = ICON_ALBUM;
         break;
      }
      case SM_MEDIA_LISTTYPE_ARTIST:
      case SM_MEDIA_LISTTYPE_GENRE_ARTIST:
      {
         returnDataIcon = ICON_ARTIST;
         break;
      }
      case SM_MEDIA_LISTTYPE_GENRE:
      {
         returnDataIcon = ICON_GENRE;
         break;
      }
      case SM_MEDIA_LISTTYPE_PLAYLIST:
      {
         returnDataIcon = ICON_PLAYLIST;
         break;
      }
      case SM_MEDIA_LISTTYPE_PODCAST:
      {
         returnDataIcon = ICON_PODCAST;
         break;
      }
      case SM_MEDIA_LISTTYPE_PODCAST_EPISODE:
      {
         returnDataIcon = ICON_PODCAST_EPISODE;
         break;
      }
      case SM_MEDIA_LISTTYPE_AUDIOBOOK:
      {
         returnDataIcon = ICON_AUDIOBOOK;
         break;
      }
      case SM_MEDIA_LISTTYPE_AUDIOBOOKTITLE_CHAPTER:
      {
         returnDataIcon = ICON_AUDIOBOOK_EPISODE;
         break;
      }
      case SM_MEDIA_LISTTYPE_VIDEO:
      case SM_MEDIA_LISTTYPE_VIDEO_EPISODE:
      {
         returnDataIcon = ICON_VIDEO;
         break;
      }
      default:
      {
         break;
      }
   }
#ifndef VARIANT_S_FTR_ENABLE_UNITTEST
   ETG_TRACE_USR4(("HMIModelListImplementation::updateDataIconInList returndataicon: %d", returnDataIcon));
#endif
   return returnDataIcon;
}


#ifndef VARIANT_S_FTR_ENABLE_UNITTEST
/**************************************************************************************
*FUNCTION		: vCreateMapListTypetoMetaDataTag
*DESCRIPTION	: creating the Map of listtype to Metadata
*PARAMETER		: NA
*RETURNVALUE	: NA
*HISTORY		:
**************************************************************************************/
void HMIModelListImplementation::vCreateMapListIDtoListType()
{
   ETG_TRACE_USR1(("HMIModelListImplementation::vCreateMapListIDtoListType"));

   m_ListIDtoListTypeMap[LISTID_MEDIA_ALBUM] = SM_MEDIA_LISTTYPE_ALBUM;
   m_ListIDtoListTypeMap[LISTID_MEDIA_TITLE] = SM_MEDIA_LISTTYPE_SONG;
   m_ListIDtoListTypeMap[LISTID_MEDIA_ALBUMTITLE] = SM_MEDIA_LISTTYPE_ALBUM_SONG;
   m_ListIDtoListTypeMap[LISTID_MEDIA_ARTIST] = SM_MEDIA_LISTTYPE_ARTIST;
   m_ListIDtoListTypeMap[LISTID_MEDIA_GENRE] = SM_MEDIA_LISTTYPE_GENRE;
   m_ListIDtoListTypeMap[LISTID_MEDIA_ARTISTALBUM] = SM_MEDIA_LISTTYPE_ARTIST_ALBUM;
   m_ListIDtoListTypeMap[LISTID_MEDIA_PLAYLIST] = SM_MEDIA_LISTTYPE_PLAYLIST;
   m_ListIDtoListTypeMap[LISTID_MEDIA_VIDEO] = SM_MEDIA_LISTTYPE_VIDEO;
   m_ListIDtoListTypeMap[LISTID_MEDIA_ARTISTALBUMTITLE] = SM_MEDIA_LISTTYPE_ARTIST_ALBUM_SONG;
   m_ListIDtoListTypeMap[LISTID_MEDIA_PODCAST] = SM_MEDIA_LISTTYPE_PODCAST;
   /*  m_ListIDtoListTypeMap[LISTID_MEDIA_GENREARTIST] = SM_MEDIA_LISTTYPE_GENRE_ARTIST;
     m_ListIDtoListTypeMap[LISTID_MEDIA_GENREARTISTALBUM] = SM_MEDIA_LISTTYPE_GENRE_ARTIST_ALBUM;
     m_ListIDtoListTypeMap[LISTID_MEDIA_GENREARTISTALBUMSONG] = SM_MEDIA_LISTTYPE_ARTIST_ALBUM_SONG;*/
   m_ListIDtoListTypeMap[LISTID_MEDIA_PLAYLISTSONG] = SM_MEDIA_LISTTYPE_PLAYLIST_SONG;
   m_ListIDtoListTypeMap[LISTID_MEDIA_VIDEOEPISODE] = SM_MEDIA_LISTTYPE_VIDEO_EPISODE;
   m_ListIDtoListTypeMap[LISTID_MEDIA_PODCASTEPISODE] = SM_MEDIA_LISTTYPE_PODCAST_EPISODE;
   m_ListIDtoListTypeMap[LISTID_MEDIA_AUDIOBOOK] = SM_MEDIA_LISTTYPE_AUDIOBOOK;
   m_ListIDtoListTypeMap[LISTID_MEDIA_AUDIOBOOKKEPISODE] = SM_MEDIA_LISTTYPE_AUDIOBOOKTITLE_CHAPTER;
   m_ListIDtoListTypeMap[LISTID_MEDIA_ALLTITLEARTIST] = SM_MEDIA_LISTTYPE_ARTIST_SONG;
   m_ListIDtoListTypeMap[LISTID_MEDIA_ALLALBUMGENRE] = SM_MEDIA_LISTTYPE_GENRE_ALBUM;
   m_ListIDtoListTypeMap[LISTID_MEDIA_GENREALLALBUMTITLE] = SM_MEDIA_LISTTYPE_GENRE_ALBUM_SONG;
   m_ListIDtoListTypeMap[LISTID_MEDIA_GENREALLTITLE] = SM_MEDIA_LISTTYPE_GENRE_SONG;
   m_ListIDtoListTypeMap[LISTID_MEDIA_FOLDER] = SM_MEDIA_LISTTYPE_FOLDER;
   m_ListIDtoListTypeMap[LISTID_MEDIA_FOLDERFILE] = SM_MEDIA_LISTTYPE_FOLDER_FILE;
}


#endif
}//end of Core
}//end of App
