/**************************************************************************************
 * @file         : UnSupportedPopupHandler.cpp
 * @author       : RBEI/ECH2-INF4CV_MediaTeam
 * @addtogroup   : AppHmi_media
 * @brief        :
 * @copyright    : (c) 2018-2020 Robert Bosch Car Multimedia GmbH
 *                 The reproduction, distribution and utilization of this file as
 *                 well as the communication of its contents to others without express
 *                 authorization is prohibited. Offenders will be held liable for the
 *                 payment of damages. All rights reserved in the event of the grant
 *                 of a patent, utility model or design.
 *30.07.20 Rev. 0.1 Naga Suman(RBEI/ECH2)
 * 		  			Initial Revision.
 **************************************************************************************/
#include "hall_std_if.h"
#include <fstream>
#include <boost/algorithm/string.hpp>
#include "Core/UnSupportedPopupHandler/UnSupportedPopupHandler.h"
#include "Core/MediaDefines.h"
#include "ProjectBaseTypes.h"

#ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
#define ETG_DEFAULT_TRACE_CLASS           TR_CLASS_APPHMI_MEDIA_HALL
#define ETG_I_TRACE_CHANNEL               TR_TTFIS_APPHMI_MEDIA
#define ETG_I_TTFIS_CMD_PREFIX            "APPHMI_MEDIA_"
#define ETG_I_FILE_PREFIX                 App::Core::UnSupportedPopupHandler::
#include "trcGenProj/Header/UnSupportedPopupHandler.cpp.trc.h"
#endif

using namespace mplay_devicemanager_fi;

namespace App {
namespace Core {

UnSupportedPopupHandler* UnSupportedPopupHandler::_unSupportedPopupHandler = NULL;
UnSupportedPopupHandler& UnSupportedPopupHandler::getInstance()
{
   if (_unSupportedPopupHandler == 0)
   {
      _unSupportedPopupHandler = new UnSupportedPopupHandler();
   }
   return *_unSupportedPopupHandler;
}


void UnSupportedPopupHandler::removeInstance()
{
   if (_unSupportedPopupHandler)
   {
      delete _unSupportedPopupHandler;
      _unSupportedPopupHandler = 0;
   }
}


/************************************************************************
*FUNCTION		: UnSupportedPopupHandler
*DESCRIPTION	: Constructor. Initializes the member variables.
*PARAMETER		:
*PARAMETER		:
*RETURNVALUE	: NA
************************************************************************/
UnSupportedPopupHandler::UnSupportedPopupHandler()
{
   ETG_TRACE_USR1(("UnSupportedPopupHandler::UnSupportedPopupHandler Constructor called"));
   _deviceManagerProxy = ::mplay_devicemanager_fi::Mplay_devicemanager_fiProxy::createProxy("mPlayDeviceManagerFiPort", *this);
   _whiteListDeviceInfo.clear();
   _deviceInfoVector.clear();
   m_unSupportedDeviceType = ::mplay_devicemanager_fi_types::T_e8_Device__DTY_UNKNOWN;
   m_unSupportedReason = ::mplay_devicemanager_fi_types::T_e8_DVM_UnsupportedReason__OK;
   m_MediaDeviceDetails = (struct stMediaDeviceInfo)
   {
      0, 0, 0, 0, 0, false
   };
   if (_deviceManagerProxy.get())
   {
      ETG_TRACE_USR4(("UnSupportedPopupHandler::UnSupportedPopupHandler StartupSync _deviceManagerProxy"));
      StartupSync::getInstance().registerPropertyRegistrationIF(this, _deviceManagerProxy->getPortName());
   }
   readFromUsbSupportedDevicesFile();
}


/************************************************************************
*FUNCTION		: UnSupportedPopupHandler
*DESCRIPTION	: Destructor.
*PARAMETER		:
*RETURNVALUE	: NA
************************************************************************/
UnSupportedPopupHandler::~UnSupportedPopupHandler()
{
   _deviceManagerProxy.reset();
}


void UnSupportedPopupHandler::registerProperties(const ::boost::shared_ptr< asf::core::Proxy >& proxy, const asf::core::ServiceStateChange& /*stateChange*/)
{
   ETG_TRACE_COMP(("UnSupportedPopupHandler::registerProperties"));
   if (_deviceManagerProxy && _deviceManagerProxy == proxy)
   {
      _deviceManagerProxy->sendDevManagerNotifyConnectionUpReg(*this);
   }
}


/**
* deregisterProperties - Trigger property deregistration to hmi_master,  called from MediaHall class
* @param[in] proxy
* @parm[in] stateChange - state change service for corrosponding  proxy
* @return void
*/
void UnSupportedPopupHandler::deregisterProperties(const ::boost::shared_ptr< asf::core::Proxy >& proxy, const asf::core::ServiceStateChange& /*stateChange*/)
{
   ETG_TRACE_COMP(("UnSupportedPopupHandler::deregisterProperties"));
   if (_deviceManagerProxy && _deviceManagerProxy == proxy)
   {
      _deviceManagerProxy->sendDevManagerNotifyConnectionRelUpRegAll();
   }
}


void UnSupportedPopupHandler::onUnavailable(const boost::shared_ptr<asf::core::Proxy>& proxy, const asf::core::ServiceStateChange& /*stateChange*/)
{
   ETG_TRACE_COMP(("UnSupportedPopupHandler::onUnavailable entered"));
   if (_deviceManagerProxy && _deviceManagerProxy == proxy)
   {
      _deviceManagerProxy->sendDevManagerNotifyConnectionRelUpRegAll();
   }
}


void UnSupportedPopupHandler::onAvailable(const ::boost::shared_ptr< asf::core::Proxy >& proxy, const asf::core::ServiceStateChange& /*stateChange*/)
{
   ETG_TRACE_COMP(("UnSupportedPopupHandler::onAvailable entered"));
   if (_deviceManagerProxy && _deviceManagerProxy == proxy)
   {
      _deviceManagerProxy->sendDevManagerNotifyConnectionUpReg(*this);
   }
}


/**
 * onDevManagerNotifyConnectionError - called if onDevManagerNotifyConnectionError property is updated with error
 * @param[in] proxy
 * @parm[in] error
 * @return void
 */
void UnSupportedPopupHandler::onDevManagerNotifyConnectionError(const ::boost::shared_ptr<mplay_devicemanager_fi::Mplay_devicemanager_fiProxy >& proxy, const ::boost::shared_ptr< ::mplay_devicemanager_fi::DevManagerNotifyConnectionError >& error)
{
   ETG_TRACE_USR4(("UnSupportedPopupHandler: onDevManagerNotifyConnectionError Received"));
}


/**
 * onDevManagerNotifyConnectionStatus - called if onDevManagerNotifyConnectionStatus property is updated for the detection of Molex Hub
 * @param[in] proxy
 * @parm[in] status
 * @return void
 */
void UnSupportedPopupHandler::onDevManagerNotifyConnectionStatus(const ::boost::shared_ptr<mplay_devicemanager_fi::Mplay_devicemanager_fiProxy >& proxy, const ::boost::shared_ptr< ::mplay_devicemanager_fi::DevManagerNotifyConnectionStatus >& status)
{
   ETG_TRACE_COMP(("UnSupportedPopupHandler: onDevManagerNotifyConnectionStatus Received"));
   uint16 vendorID;
   uint16 productID;
   stMediaDeviceInfo mediaDeviceInfoDetails;
   ETG_TRACE_USR4(("UnSupportedPopupHandler::onDevManagerNotifyConnectionStatus status->getDevicetype().size() %d", status->getDevicetype().size()));
   for (unsigned int index = 0; index < status->getDevicetype().size(); index++)
   {
      ::mplay_devicemanager_fi_types::T_e8_Device deviceType = status->getDevicetype()[index].getDeviceType();
      ::mplay_devicemanager_fi_types::T_e8_DVM_UnsupportedReason unSupportedReason = status->getDevicetype()[index].getUnsupportedReason();
      ::mplay_devicemanager_fi_types::T_e8_DeviceConnectStatus deviceConnectStatus = status->getDevicetype()[index].getDeviceConnectStatus();
      productID = status->getDevicetype()[index].getProductID();
      vendorID = status->getDevicetype()[index].getVendorID();

      ETG_TRACE_COMP(("UnSupportedPopupHandler::onDevManagerNotifyConnectionStatus DeviceType :%d, UnsupportedReason :%d, DeviceConnectStatus :%d, productID :%d, vendorID :%d", deviceType, unSupportedReason, deviceConnectStatus, productID, vendorID));

      if (isProductIDNotAvailableInMediaDeviceInfoVector(status->getDevicetype()[index].getProductID(), status->getDevicetype()[index].getDeviceConnectStatus()))
      {
         mediaDeviceInfoDetails.vendorID = vendorID;
         mediaDeviceInfoDetails.productID = productID;
         mediaDeviceInfoDetails.unSupportedReason = status->getDevicetype()[index].getUnsupportedReason();
         mediaDeviceInfoDetails.devType = status->getDevicetype()[index].getDeviceType();
         mediaDeviceInfoDetails.connectionStatus = status->getDevicetype()[index].getDeviceConnectStatus();
         mediaDeviceInfoDetails.isNewMediaDevice = true;
         _deviceInfoVector.push_back(mediaDeviceInfoDetails);
         ETG_TRACE_COMP(("UnSupportedPopupHandler:onDevManagerNotifyConnectionStatus After inserting in Vector :%d, mediaDeviceInfoDetails.UnsupportedReason :%d, mediaDeviceInfoDetails.DeviceConnectStatus :%d, mediaDeviceInfoDetails.productID :%d, mediaDeviceInfoDetails.vendorID :%d, mediaDeviceInfoDetails.isMediaDeviceInfoaAvailableInVector %d", mediaDeviceInfoDetails.devType, mediaDeviceInfoDetails.unSupportedReason, mediaDeviceInfoDetails.connectionStatus, mediaDeviceInfoDetails.productID, mediaDeviceInfoDetails.vendorID, mediaDeviceInfoDetails.isNewMediaDevice));
      }
   }
   removeMediaDeviceInfoOnUserDisconnection();
   performUnSupportedMediaDevicesOperation();
}


void UnSupportedPopupHandler::performUnSupportedMediaDevicesOperation()
{
   uint16 unSupportedReason =  ::mplay_devicemanager_fi_types::T_e8_DVM_UnsupportedReason__OK;
   uint16 mediaDeviceType = ::mplay_devicemanager_fi_types::T_e8_Device__DTY_UNKNOWN;
   ETG_TRACE_USR4(("UnSupportedPopupHandler:performUnSupportedMediaDevicesOperation deviceInfoMap.size() %d", _deviceInfoVector.size()));
   for (unsigned int itr = 0; itr < _deviceInfoVector.size(); itr++)
   {
      if (_deviceInfoVector[itr].isNewMediaDevice == true)
      {
         unSupportedReason = _deviceInfoVector[itr].unSupportedReason;
         mediaDeviceType = _deviceInfoVector[itr].devType;
         _deviceInfoVector[itr].isNewMediaDevice = false;
         ETG_TRACE_USR4(("UnSupportedPopupHandler:performUnSupportedMediaDevicesOperation unSupportedReason %d", unSupportedReason));
         ETG_TRACE_USR4(("UnSupportedPopupHandler:performUnSupportedMediaDevicesOperation mediaDeviceType %d", mediaDeviceType));
         ETG_TRACE_USR4(("UnSupportedPopupHandler:performUnSupportedMediaDevicesOperation _deviceInfoVector[itr].isNewMediaDevice %d", _deviceInfoVector[itr].isNewMediaDevice));
         bool isMediaDeviceInfoAvaialbleInWhitelistMap =  ignoreSupportedMediaDevices(_deviceInfoVector[itr].vendorID, _deviceInfoVector[itr].productID);
         ETG_TRACE_USR4(("UnSupportedPopupHandler:performUnSupportedMediaDevicesOperation isMediaDeviceInfoAvaialbleInWhitelistMap %d", isMediaDeviceInfoAvaialbleInWhitelistMap));
         if (isMediaDeviceInfoAvaialbleInWhitelistMap == false)
         {
            triggerUnsupportedUSBToastPopup(mediaDeviceType, unSupportedReason, isMediaDeviceInfoAvaialbleInWhitelistMap);
         }
      }
   }
}


void UnSupportedPopupHandler::removeMediaDeviceInfoOnUserDisconnection()
{
   ETG_TRACE_USR4(("UnSupportedPopupHandler:removeMediaDeviceInfoOnUserDisconnection deviceInfoMap.size() %d", _deviceInfoVector.size()));
   for (unsigned int itr = 0; itr < _deviceInfoVector.size(); itr++)
   {
      if (_deviceInfoVector[itr].isNewMediaDevice == false
            && _deviceInfoVector[itr].connectionStatus ==::mplay_devicemanager_fi_types::T_e8_DeviceConnectStatus__USB_DEV_REMOVED_BY_USR)
      {
         ETG_TRACE_USR4(("UnSupportedPopupHandler:removeMediaDeviceInfoOnUserDisconnection ERASE isNewMediaDevice %d", _deviceInfoVector[itr].isNewMediaDevice));
         ETG_TRACE_USR4(("UnSupportedPopupHandler:removeMediaDeviceInfoOnUserDisconnection ERASE connectionstatus %d", _deviceInfoVector[itr].connectionStatus));
         ETG_TRACE_USR4(("UnSupportedPopupHandler:removeMediaDeviceInfoOnUserDisconnection ERASE vendor id %d", _deviceInfoVector[itr].vendorID));
         ETG_TRACE_USR4(("UnSupportedPopupHandler:removeMediaDeviceInfoOnUserDisconnection ERASE product id %d", _deviceInfoVector[itr].productID));
         ETG_TRACE_USR4(("UnSupportedPopupHandler:removeMediaDeviceInfoOnUserDisconnection ERASE unsupportreason id %d", _deviceInfoVector[itr].unSupportedReason));
         ETG_TRACE_USR4(("UnSupportedPopupHandler:removeMediaDeviceInfoOnUserDisconnection ERASE devtype id %d", _deviceInfoVector[itr].devType));
         _deviceInfoVector.erase(_deviceInfoVector.begin() + itr);
      }
   }
}


/**
 * triggerUnsupportedUSBToastPopup - triggers unsupported usb toast popup (when usb and hub not supported.)
 * @param[in] deviceType, unsupportedReason and deviceConnectStatus
 * @ return void
 */
void UnSupportedPopupHandler::triggerUnsupportedUSBToastPopup(uint16 deviceType, uint16 unsupportedReason, bool isSupportedDeviceConnected)
{
   ETG_TRACE_USR4(("UnSupportedPopupHandler::triggerUnsupportedUSBToastPopupUSB deviceType %d", deviceType));
   ETG_TRACE_USR4(("UnSupportedPopupHandler::triggerUnsupportedUSBToastPopupUSB unsupportedReason %d", unsupportedReason));
   ETG_TRACE_USR4(("UnSupportedPopupHandler::triggerUnsupportedUSBToastPopupUSB isSupportedDeviceConnected %d", isSupportedDeviceConnected));
   if (isSupportedDeviceConnected == false)
   {
      performUnsupportedPopUpOperation(deviceType, unsupportedReason);
   }
}


void UnSupportedPopupHandler::performUnsupportedPopUpOperation(uint16 deviceType, uint16 unsupportedReason)
{
   ETG_TRACE_USR4(("UnSupportedPopupHandler::performUnsupportedPopUpOperation deviceType %d", deviceType));
   ETG_TRACE_USR4(("UnSupportedPopupHandler::performUnsupportedPopUpOperation unsupportedReason %d", unsupportedReason));
   m_unSupportedDeviceType = deviceType;
   m_unSupportedReason = unsupportedReason;
   if (deviceType == ::mplay_devicemanager_fi_types::T_e8_Device__DTY_NOT_SUPPORTED)
   {
      ETG_TRACE_USR4(("UnSupportedPopupHandler::performUnsupportedPopUpOperation USB not supported"));
      // usb not supported
      POST_MSG((COURIER_MESSAGE_NEW(DeActivatePopUpMsg)(Media_MediaPlayer_Pop_Up_system_Pfo_MediaMessage)));
      POST_MSG((COURIER_MESSAGE_NEW(ActivatePopUpMsg)(Media_MediaPlayer_Pop_Up_system_Pfo_MediaUnsupportedPopupMessage)));
   }
   else if ((deviceType == ::mplay_devicemanager_fi_types::T_e8_Device__DTY_HUB)
            && (unsupportedReason == ::mplay_devicemanager_fi_types::T_e8_DVM_UnsupportedReason__HubButNotCustomerHub))
   {
      // HUB not supported
      ETG_TRACE_USR4(("UnSupportedPopupHandler::performUnsupportedPopUpOperation HUB not supported"));
      POST_MSG((COURIER_MESSAGE_NEW(DeActivatePopUpMsg)(Media_MediaPlayer_Pop_Up_system_Pfo_MediaMessage)));
      POST_MSG((COURIER_MESSAGE_NEW(ActivatePopUpMsg)(Media_MediaPlayer_Pop_Up_system_Pfo_MediaUnsupportedPopupMessageHUB)));
   }
}


/**
 * ignoreSupportedMediaDevices - ignores unsupported usb toast popup which is present in Whitelist file
 * @param[in] vendorId,  productId and isUnsupportedReasonOKReceived
 * @ return bool
 */
bool UnSupportedPopupHandler::ignoreSupportedMediaDevices(uint16 vendorId, uint16 productId)
{
   ETG_TRACE_USR1(("UnSupportedPopupHandler::ignoreSupportedMediaDevices"));
   std::multimap<uint16, uint16>::iterator itr = _whiteListDeviceInfo.begin();
   bool isMediaDeviceAvailable = false;
   for (; itr != _whiteListDeviceInfo.end(); itr++)
   {
      ETG_TRACE_USR1(("UnSupportedPopupHandler::ignoreSupportedMediaDevices:From media whitelist Map _vendorId= %d, productId= %d", itr->first, itr->second));
      ETG_TRACE_USR1(("UnSupportedPopupHandler::ignoreSupportedMediaDevices:From local vector _vendorId= %d, productId= %d", vendorId, productId));

      if ((itr->first == vendorId) && (itr->second == productId))
      {
         isMediaDeviceAvailable = true;
         break;
      }
   }
   ETG_TRACE_USR1(("UnSupportedPopupHandler::ignoreSupportedMediaDevices isMediaDeviceAvailable %d", isMediaDeviceAvailable));
   return isMediaDeviceAvailable;
}


bool UnSupportedPopupHandler::isProductIDNotAvailableInMediaDeviceInfoVector(uint16 productid, uint16 connectionStatus)
{
   ETG_TRACE_USR1(("UnSupportedPopupHandler::isProductIDNotAvailableInMediaDeviceInfoVector productid %d", productid));
   ETG_TRACE_USR1(("UnSupportedPopupHandler::isProductIDNotAvailableInMediaDeviceInfoVector _deviceInfoVector.size() %d", _deviceInfoVector.size()));
   bool isProductIDNotAvailable = true;
   for (unsigned int itr = 0 ; itr < _deviceInfoVector.size(); itr++)
   {
      if (_deviceInfoVector[itr].productID == productid)
      {
         isProductIDNotAvailable = false;
         _deviceInfoVector[itr].connectionStatus = connectionStatus;
         break;
      }
   }
   ETG_TRACE_USR1(("UnSupportedPopupHandler::isProductIDNotAvailableInMediaDeviceInfoVector isProductIDNotAvailable %d", isProductIDNotAvailable));
   return  isProductIDNotAvailable;
}


/**
 * readFromUsbSupportedDevicesFile - Read VendorId and ProductId from Whitelist file
 * @param[in]      Nil
 * @return         void
 */
void UnSupportedPopupHandler::readFromUsbSupportedDevicesFile()
{
   const std::string filePath = "/opt/bosch/whitelist/usb_supported_devices.txt";
   std::ifstream file;
   std::string supported_device;
   std::string details;
   uint16 vid;
   uint16 pid;
   std::vector<std::string>device_details;
   file.open(filePath.c_str());
   if (file.is_open())
   {
      while (file)
      {
         getline(file, supported_device);
         ETG_TRACE_USR4(("UnSupportedPopupHandler::readFromUsbSupportedDevicesFile:supported devices are:%s", supported_device.c_str()));
         if (!supported_device.empty())
         {
            int count = 0;
            //count is used to ignore the first column in the file
            std::stringstream ss(supported_device);
            while (getline(ss, details, ','))
            {
               if (!details.empty() && count != 0)
               {
                  //ignore the extra whitespaces in the file and push_back into vector
                  boost::trim(details);
                  device_details.push_back(details);
               }
               count = 1;
            }
            if (device_details.size() < 2)
            {
               ETG_TRACE_USR4(("UnSupportedPopupHandler::readFromUsbSupportedDevicesFile:Something wrong with file content.."));
            }
            else
            {
               std::stringstream v;
               std::stringstream p;
               v << std::hex << device_details.front();
               p << std::hex << device_details.back();
               v >> vid;
               p >> pid;
               _whiteListDeviceInfo.insert(std::make_pair(vid, pid));
               device_details.clear();
            }
         }
      }
      std::multimap<uint16, uint16>::iterator itr = _whiteListDeviceInfo.begin();
      for (itr = _whiteListDeviceInfo.begin(); itr != _whiteListDeviceInfo.end(); itr++)
      {
         ETG_TRACE_USR4(("UnSupportedPopupHandler::readFromUsbSupportedDevicesFile:_whiteListDeviceInfo_vendorid: %d, _whiteListDeviceInfo_product_id: %d", itr->first, itr->second));
      }
      file.close();
   }
}


void UnSupportedPopupHandler::callUnsupportedPopUpOnGuiStartUp()
{
   ETG_TRACE_USR1(("UnSupportedPopupHandler::callUnsupportedPopUpOnGuiStartUp"));
   performUnsupportedPopUpOperation(m_unSupportedDeviceType, m_unSupportedReason);
}


}//end of namespace Core
}//end of namespace App
