/**************************************************************************************
 * @file         : MediaPlaybackStatusUpdater.cpp
 * @author       : RBEI/ECH2-INF4CV_MediaTeam
 * @addtogroup   : AppHmi_media
 * @brief        :
 * @copyright    : (c) 2018-2020 Robert Bosch Car Multimedia GmbH
 *                 The reproduction, distribution and utilization of this file as
 *                 well as the communication of its contents to others without express
 *                 authorization is prohibited. Offenders will be held liable for the
 *                 payment of damages. All rights reserved in the event of the grant
 *                 of a patent, utility model or design.
 * 		  			Initial Revision.
 **************************************************************************************/

#include "hall_std_if.h"
#include "Core/MediaPlaybackStatusUpdater/MediaPlaybackStatusUpdater.h"
#include "AppHmi_MasterBase/AudioInterface/AudioDefines.h"
#include <vector>
#include "Core/MediaClientHandler/MediaClientHandler.h"
#include "Core/MediaSourceListHandler/MediaSourceListHandler.h"
#include "Core/ExternalMediaDefines.h"
#include "Core/StateTransitionHandler/StateTransitionHandler.h"

#ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
#define ETG_DEFAULT_TRACE_CLASS           TR_CLASS_APPHMI_MEDIA_HALL
#define ETG_I_TRACE_CHANNEL               TR_TTFIS_APPHMI_MEDIA
#define ETG_I_TTFIS_CMD_PREFIX            "APPHMI_MEDIA_"
#define ETG_I_FILE_PREFIX                 App::Core::MediaPlaybackStatusUpdater::
#include "trcGenProj/Header/MediaPlaybackStatusUpdater.cpp.trc.h"
#endif

using namespace ::VEHICLE_MAIN_FI;
using namespace MASTERAUDIOSERVICE_INTERFACE::AudioSourceChange;

namespace App {
namespace Core {

/**
	* Description     : Constructor of class MediaPlaybackStatusUpdater
	*/
MediaPlaybackStatusUpdater::MediaPlaybackStatusUpdater(): _vehicleProxy(::VEHICLE_MAIN_FIProxy::createProxy("vehicleMainFiPort", *this)),
   _mediaPlayerProxy(::mplay_MediaPlayer_FI::Mplay_MediaPlayer_FIProxy::createProxy("mediaPlayerFiPort", *this)),
   _audioSourceChangeProxy(AudioSourceChangeProxy::createProxy("audioSourceChangePort", *this))
{
   ETG_TRACE_USR1(("MediaPlaybackStatusUpdater::MediaPlaybackStatusUpdater constructor \n"));
   if (_vehicleProxy)
   {
      StartupSync::getInstance().registerPropertyRegistrationIF(this, _vehicleProxy->getPortName());
   }
   if (_mediaPlayerProxy.get())
   {
      StartupSync::getInstance().registerPropertyRegistrationIF(this, _mediaPlayerProxy->getPortName());
   }
   _applicationSwitchClientComponent = ApplicationSwitchClientComponent::poGetInstance();
   _isSeekUpdated = false;
   _playbackCANstate = EN_SNA;
   _currentElapsedTime = EN_NOT_AVAILABLE;
   _shuffleCANStatus = EN_SNA;
   _repeatCANStatus = EN_SNA;
   _currentTrackNum = EN_NOT_AVAILABLE ;
}


MediaPlaybackStatusUpdater::~MediaPlaybackStatusUpdater()
{
   ETG_TRACE_USR1(("ExternalMediaListDBManager::MediaPlaybackStatusUpdater Destructor \n"));
}


void MediaPlaybackStatusUpdater::onUnavailable(const boost::shared_ptr<asf::core::Proxy>& proxy, const asf::core::ServiceStateChange& /*stateChange*/)
{
   ETG_TRACE_COMP(("MediaPlaybackStatusUpdater::onUnavailable entered"));

   if ((proxy == _mediaPlayerProxy) && (_mediaPlayerProxy.get()))
   {
      _mediaPlayerProxy->sendMediaPlayerDeviceConnectionsRelUpRegAll();
      _mediaPlayerProxy->sendPlaybackStateRelUpRegAll();
      _mediaPlayerProxy->sendNowPlayingRelUpRegAll();
      _mediaPlayerProxy->sendPlaybackModeRelUpRegAll();
      _mediaPlayerProxy->sendRepeatModeRelUpRegAll();
      _mediaPlayerProxy->sendPlaytimeRelUpRegAll();
   }
   if ((proxy == _vehicleProxy) && (_vehicleProxy.get()))
   {
      _vehicleProxy->sendSourceCommandRelUpRegAll();
   }
   if (_audioSourceChangeProxy && _audioSourceChangeProxy == proxy)
   {
      _audioSourceChangeProxy->sendDeregisterAll();
   }
}


void MediaPlaybackStatusUpdater::onAvailable(const ::boost::shared_ptr< asf::core::Proxy >& proxy, const asf::core::ServiceStateChange& /*stateChange*/)
{
   ETG_TRACE_COMP(("MediaPlaybackStatusUpdater::onAvailable entered"));

   if ((proxy == _mediaPlayerProxy) && (_mediaPlayerProxy.get()))
   {
      _mediaPlayerProxy->sendMediaPlayerDeviceConnectionsUpReg(*this);
      _mediaPlayerProxy->sendNowPlayingUpReg(*this);
      _mediaPlayerProxy->sendPlaybackModeUpReg(*this);
      _mediaPlayerProxy->sendPlaybackStateUpReg(*this);
      _mediaPlayerProxy->sendRepeatModeUpReg(*this);
      _mediaPlayerProxy->sendPlaytimeUpReg(*this);
   }
   if ((proxy == _vehicleProxy) && (_vehicleProxy.get()))
   {
      _vehicleProxy->sendDeviceStatusUSB1SrcStart(*this, EN_USB_AVAIL);
      _vehicleProxy->sendDeviceStatusUSB2SrcStart(*this, EN_USB_AVAIL);
      _vehicleProxy->sendSourceCommandUpReg(*this);
      _vehicleProxy->sendDeviceStatusBluetoothAvailStart(*this, EN_BLUETOOTH_AVAIL);
   }
   if (_audioSourceChangeProxy && _audioSourceChangeProxy == proxy)
   {
      _audioSourceChangeProxy->sendActiveSourceRegister(*this);
   }
}


void MediaPlaybackStatusUpdater::registerProperties(const ::boost::shared_ptr< asf::core::Proxy >& proxy, const asf::core::ServiceStateChange& /*stateChange*/)
{
   ETG_TRACE_COMP(("MediaPlaybackStatusUpdater::registerProperties entered"));
}


void MediaPlaybackStatusUpdater::deregisterProperties(const ::boost::shared_ptr< asf::core::Proxy >& proxy, const asf::core::ServiceStateChange& /*stateChange*/)
{
   ETG_TRACE_COMP(("MediaPlaybackStatusUpdater::deregisterProperties entered"));
}


void MediaPlaybackStatusUpdater:: onMediaStatusCurrTrackNumError(const ::boost::shared_ptr< ::VEHICLE_MAIN_FI::VEHICLE_MAIN_FIProxy >& proxy, const ::boost::shared_ptr< ::VEHICLE_MAIN_FI::MediaStatusCurrTrackNumError >& error)
{
   ETG_TRACE_COMP(("MediaPlaybackStatusUpdater::onMediaStatusCurrTrackNumError received"));
}


void MediaPlaybackStatusUpdater:: onMediaStatusCurrTrackNumResult(const ::boost::shared_ptr< ::VEHICLE_MAIN_FI::VEHICLE_MAIN_FIProxy >& proxy, const ::boost::shared_ptr< ::VEHICLE_MAIN_FI::MediaStatusCurrTrackNumResult >& result)
{
   ETG_TRACE_COMP(("MediaPlaybackStatusUpdater::onMediaStatusCurrTrackNumResult received"));
}


void MediaPlaybackStatusUpdater:: onDeviceStatusUSB1PresentError(const ::boost::shared_ptr< ::VEHICLE_MAIN_FI::VEHICLE_MAIN_FIProxy >& proxy, const ::boost::shared_ptr< ::VEHICLE_MAIN_FI::DeviceStatusUSB1PresentError >& error)
{
   ETG_TRACE_COMP(("MediaPlaybackStatusUpdater::onDeviceStatusUSB1PresentError received"));
}


void MediaPlaybackStatusUpdater:: onDeviceStatusUSB1PresentResult(const ::boost::shared_ptr< ::VEHICLE_MAIN_FI::VEHICLE_MAIN_FIProxy >& proxy, const ::boost::shared_ptr< ::VEHICLE_MAIN_FI::DeviceStatusUSB1PresentResult >& result)
{
   ETG_TRACE_COMP(("MediaPlaybackStatusUpdater::onDeviceStatusUSB1PresentResult received"));
}


void MediaPlaybackStatusUpdater:: onDeviceStatusUSB1SrcError(const ::boost::shared_ptr< ::VEHICLE_MAIN_FI::VEHICLE_MAIN_FIProxy >& proxy, const ::boost::shared_ptr< ::VEHICLE_MAIN_FI::DeviceStatusUSB1SrcError >& error)
{
   ETG_TRACE_COMP(("MediaPlaybackStatusUpdater::onDeviceStatusUSB1SrcError received"));
}


void MediaPlaybackStatusUpdater:: onDeviceStatusUSB1SrcResult(const ::boost::shared_ptr< ::VEHICLE_MAIN_FI::VEHICLE_MAIN_FIProxy >& proxy, const ::boost::shared_ptr< ::VEHICLE_MAIN_FI::DeviceStatusUSB1SrcResult >& result)
{
   ETG_TRACE_COMP(("MediaPlaybackStatusUpdater::onDeviceStatusUSB1SrcResult received"));
}


void MediaPlaybackStatusUpdater:: onDeviceStatusBluetoothAvailError(const ::boost::shared_ptr< ::VEHICLE_MAIN_FI::VEHICLE_MAIN_FIProxy >& proxy, const ::boost::shared_ptr< ::VEHICLE_MAIN_FI::DeviceStatusBluetoothAvailError >& error)
{
   ETG_TRACE_COMP(("MediaPlaybackStatusUpdater::onDeviceStatusBluetoothAvailError received"));
}


void MediaPlaybackStatusUpdater:: onDeviceStatusBluetoothAvailResult(const ::boost::shared_ptr< ::VEHICLE_MAIN_FI::VEHICLE_MAIN_FIProxy >& proxy, const ::boost::shared_ptr< ::VEHICLE_MAIN_FI::DeviceStatusBluetoothAvailResult >& result)
{
   ETG_TRACE_COMP(("MediaPlaybackStatusUpdater::onDeviceStatusBluetoothAvailResult received"));
}


void MediaPlaybackStatusUpdater:: onDeviceStatusUSB2PresentError(const ::boost::shared_ptr< ::VEHICLE_MAIN_FI::VEHICLE_MAIN_FIProxy >& proxy, const ::boost::shared_ptr< ::VEHICLE_MAIN_FI::DeviceStatusUSB2PresentError >& error)
{
   ETG_TRACE_COMP(("MediaPlaybackStatusUpdater::onDeviceStatusUSB2PresentError received"));
}


void MediaPlaybackStatusUpdater:: onDeviceStatusUSB2PresentResult(const ::boost::shared_ptr< ::VEHICLE_MAIN_FI::VEHICLE_MAIN_FIProxy >& proxy, const ::boost::shared_ptr< ::VEHICLE_MAIN_FI::DeviceStatusUSB2PresentResult >& result)
{
   ETG_TRACE_COMP(("MediaPlaybackStatusUpdater::onDeviceStatusUSB2PresentResult received"));
}


void MediaPlaybackStatusUpdater:: onDeviceStatusUSB2SrcError(const ::boost::shared_ptr< ::VEHICLE_MAIN_FI::VEHICLE_MAIN_FIProxy >& proxy, const ::boost::shared_ptr< ::VEHICLE_MAIN_FI::DeviceStatusUSB2SrcError >& error)
{
   ETG_TRACE_COMP(("MediaPlaybackStatusUpdater::onDeviceStatusUSB2SrcError received"));
}


void MediaPlaybackStatusUpdater:: onDeviceStatusUSB2SrcResult(const ::boost::shared_ptr< ::VEHICLE_MAIN_FI::VEHICLE_MAIN_FIProxy >& proxy, const ::boost::shared_ptr< ::VEHICLE_MAIN_FI::DeviceStatusUSB2SrcResult >& result)
{
   ETG_TRACE_COMP(("MediaPlaybackStatusUpdater::onDeviceStatusUSB2SrcResult received"));
}


void MediaPlaybackStatusUpdater:: onMediaStatusMixStateError(const ::boost::shared_ptr< ::VEHICLE_MAIN_FI::VEHICLE_MAIN_FIProxy >& proxy, const ::boost::shared_ptr< ::VEHICLE_MAIN_FI::MediaStatusMixStateError >& error)
{
   ETG_TRACE_COMP(("MediaPlaybackStatusUpdater::onMediaStatusMixStateError received"));
}


void MediaPlaybackStatusUpdater:: onMediaStatusMixStateResult(const ::boost::shared_ptr< ::VEHICLE_MAIN_FI::VEHICLE_MAIN_FIProxy >& proxy, const ::boost::shared_ptr< ::VEHICLE_MAIN_FI::MediaStatusMixStateResult >& result)
{
   ETG_TRACE_COMP(("MediaPlaybackStatusUpdater::onMediaStatusMixStateResult received"));
}


void MediaPlaybackStatusUpdater:: onMediaStatusPlayStatusError(const ::boost::shared_ptr< ::VEHICLE_MAIN_FI::VEHICLE_MAIN_FIProxy >& proxy, const ::boost::shared_ptr< ::VEHICLE_MAIN_FI::MediaStatusPlayStatusError >& error)
{
   ETG_TRACE_COMP(("MediaPlaybackStatusUpdater::onMediaStatusPlayStatusError received"));
}


void MediaPlaybackStatusUpdater:: onMediaStatusPlayStatusResult(const ::boost::shared_ptr< ::VEHICLE_MAIN_FI::VEHICLE_MAIN_FIProxy >& proxy, const ::boost::shared_ptr< ::VEHICLE_MAIN_FI::MediaStatusPlayStatusResult >& result)
{
   ETG_TRACE_COMP(("MediaPlaybackStatusUpdater::onMediaStatusPlayStatusResult received"));
}


void MediaPlaybackStatusUpdater:: onMediaStatusRepeatStatusError(const ::boost::shared_ptr< ::VEHICLE_MAIN_FI::VEHICLE_MAIN_FIProxy >& proxy, const ::boost::shared_ptr< ::VEHICLE_MAIN_FI::MediaStatusRepeatStatusError >& error)
{
   ETG_TRACE_COMP(("MediaPlaybackStatusUpdater::onMediaStatusRepeatStatusError received"));
}


void MediaPlaybackStatusUpdater:: onMediaStatusRepeatStatusResult(const ::boost::shared_ptr< ::VEHICLE_MAIN_FI::VEHICLE_MAIN_FIProxy >& proxy, const ::boost::shared_ptr< ::VEHICLE_MAIN_FI::MediaStatusRepeatStatusResult >& result)
{
   ETG_TRACE_COMP(("MediaPlaybackStatusUpdater::onMediaStatusRepeatStatusResult received"));
}


void MediaPlaybackStatusUpdater:: onMediaStatusSeekStatusError(const ::boost::shared_ptr< ::VEHICLE_MAIN_FI::VEHICLE_MAIN_FIProxy >& proxy, const ::boost::shared_ptr< ::VEHICLE_MAIN_FI::MediaStatusSeekStatusError >& error)
{
   ETG_TRACE_COMP(("MediaPlaybackStatusUpdater::onMediaStatusSeekStatusError received"));
}


void MediaPlaybackStatusUpdater:: onMediaStatusSeekStatusResult(const ::boost::shared_ptr< ::VEHICLE_MAIN_FI::VEHICLE_MAIN_FIProxy >& proxy, const ::boost::shared_ptr< ::VEHICLE_MAIN_FI::MediaStatusSeekStatusResult >& result)
{
   ETG_TRACE_COMP(("MediaPlaybackStatusUpdater::onMediaStatusSeekStatusResult received"));
}


void MediaPlaybackStatusUpdater:: onMediaStatusTrackTimeError(const ::boost::shared_ptr< ::VEHICLE_MAIN_FI::VEHICLE_MAIN_FIProxy >& proxy, const ::boost::shared_ptr< ::VEHICLE_MAIN_FI::MediaStatusTrackTimeError >& error)
{
   ETG_TRACE_COMP(("MediaPlaybackStatusUpdater::onMediaStatusTrackTimeError received"));
}


void MediaPlaybackStatusUpdater:: onMediaStatusTrackTimeResult(const ::boost::shared_ptr< ::VEHICLE_MAIN_FI::VEHICLE_MAIN_FIProxy >& proxy, const ::boost::shared_ptr< ::VEHICLE_MAIN_FI::MediaStatusTrackTimeResult >& result)
{
   ETG_TRACE_COMP(("MediaPlaybackStatusUpdater::onMediaStatusTrackTimeResult received"));
}


void MediaPlaybackStatusUpdater::onNowPlayingError(const ::boost::shared_ptr< ::mplay_MediaPlayer_FI::Mplay_MediaPlayer_FIProxy >& proxy, const ::boost::shared_ptr< ::mplay_MediaPlayer_FI::NowPlayingError >& error)
{
   ETG_TRACE_COMP(("MediaPlaybackStatusUpdater::onNowPlayingError received."));
}


void MediaPlaybackStatusUpdater::onNowPlayingStatus(const ::boost::shared_ptr< ::mplay_MediaPlayer_FI::Mplay_MediaPlayer_FIProxy >& proxy, const ::boost::shared_ptr< ::mplay_MediaPlayer_FI::NowPlayingStatus >& status)
{
   _currentTrackNum = status->getOMediaObject().getU16ExtendedTrackNumber();
   ETG_TRACE_USR4(("MediaPlaybackStatusUpdater::onNowPlayingStatus currentTrackNum	= %d", _currentTrackNum));
   if (MediaSourceHandling::getInstance().isInternalMediaSourceActiveInSink(COCKPIT_SINK_ID))
   {
      _vehicleProxy->sendMediaStatusCurrTrackNumStart(*this, _currentTrackNum);
   }
}


void MediaPlaybackStatusUpdater::onPlaybackStateStatus(const ::boost::shared_ptr< ::mplay_MediaPlayer_FI::Mplay_MediaPlayer_FIProxy >& proxy, const ::boost::shared_ptr< ::mplay_MediaPlayer_FI::PlaybackStateStatus >& status)
{
   ETG_TRACE_COMP(("MediaPlaybackStatusUpdater::onPlaybackStateStatus"));

   tU8 playbackstate = status->getE8State();
   ETG_TRACE_USR4(("MediaPlaybackStatusUpdater::onPlaybackStateStatus playbackstate :%d ", playbackstate));
   if (MediaSourceHandling::getInstance().isInternalMediaSourceActiveInSink(COCKPIT_SINK_ID))
   {
      switch (playbackstate)
      {
         case MPlay_fi_types::T_e8_MPlayState__e8PBS_PLAYING:
         {
            if (_isSeekUpdated == true)
            {
               _vehicleProxy->sendMediaStatusSeekStatusStart(*this, CAN_INACTIVE);
               _isSeekUpdated = false;
            }
            _vehicleProxy->sendMediaStatusPlayStatusStart(*this, CAN_PLAY_VALUE);
            _playbackCANstate = CAN_PLAY_VALUE;
         }
         break;
         case MPlay_fi_types::T_e8_MPlayState__e8PBS_PAUSED:
         {
            _vehicleProxy->sendMediaStatusPlayStatusStart(*this, CAN_PAUSE_VALUE);
            _playbackCANstate = CAN_PAUSE_VALUE;
         }
         break;
         case MPlay_fi_types::T_e8_MPlayState__e8PBS_STOPPED:
         {
            _vehicleProxy->sendMediaStatusPlayStatusStart(*this, CAN_PAUSE_VALUE);
            _playbackCANstate = CAN_PAUSE_VALUE;
         }
         break;
         case MPlay_fi_types::T_e8_MPlayState__e8PBS_FFWD:
         {
            _vehicleProxy->sendMediaStatusSeekStatusStart(*this, CAN_FAST_FORWARD);
            _isSeekUpdated = true;
         }
         break;
         case MPlay_fi_types::T_e8_MPlayState__e8PBS_FREV:
         {
            _vehicleProxy->sendMediaStatusSeekStatusStart(*this, CAN_FAST_REWIND);
            _isSeekUpdated = true;
         }
         break;
         default:
         {
            ETG_TRACE_USR4(("MediaPlaybackStatusUpdater::onPlaybackStateStatus: Default Case"));
         }
         break;
      }
   }
}


void MediaPlaybackStatusUpdater::onPlaybackStateError(const ::boost::shared_ptr< ::mplay_MediaPlayer_FI::Mplay_MediaPlayer_FIProxy >& proxy, const ::boost::shared_ptr< ::mplay_MediaPlayer_FI::PlaybackStateError >& error)
{
   ETG_TRACE_COMP(("MediaPlaybackStatusUpdater::onPlaybackStateError received"));
}


void MediaPlaybackStatusUpdater::onPlaytimeStatus(const ::boost::shared_ptr< ::mplay_MediaPlayer_FI::Mplay_MediaPlayer_FIProxy >& proxy, const ::boost::shared_ptr< ::mplay_MediaPlayer_FI::PlaytimeStatus >& status)
{
   ETG_TRACE_USR4(("MediaPlaybackStatusUpdater::onPlaytimeStatus received: with ElapsedTime :%d TotalTime:%d", status->getU32ElapsedPlaytime(), status->getU32TotalPlaytime()));
   _currentElapsedTime = status->getU32ElapsedPlaytime();
   ETG_TRACE_USR4(("MediaPlaybackStatusUpdater onPlaytimeStatus currentElapsedTime %d", _currentElapsedTime));
   if (MediaSourceHandling::getInstance().isInternalMediaSourceActiveInSink(COCKPIT_SINK_ID))
   {
      _vehicleProxy->sendMediaStatusTrackTimeStart(*this, _currentElapsedTime);
   }
}


void MediaPlaybackStatusUpdater::onPlaytimeError(const ::boost::shared_ptr< ::mplay_MediaPlayer_FI::Mplay_MediaPlayer_FIProxy >& proxy, const ::boost::shared_ptr< ::mplay_MediaPlayer_FI::PlaytimeError >& error)
{
   ETG_TRACE_COMP(("MediaPlaybackStatusUpdater::onPlaytimeError received"));
}


void MediaPlaybackStatusUpdater::onMediaPlayerDeviceConnectionsStatus(const ::boost::shared_ptr< ::mplay_MediaPlayer_FI::Mplay_MediaPlayer_FIProxy >& /*proxy*/, const boost::shared_ptr< ::mplay_MediaPlayer_FI::MediaPlayerDeviceConnectionsStatus >& status)
{
   ETG_TRACE_COMP(("MediaPlaybackStatusUpdater::onMediaPlayerDeviceConnectionsStatus"));

   ::MPlay_fi_types::T_MPlayDeviceInfo oDeviceInfo = status->getODeviceInfo();
   MPlay_fi_types::T_MPlayDeviceInfo::iterator itr = oDeviceInfo.begin();
   std::string sysPath = "";

   uint8 currListIndex = 0;
   bool isCurrIndexDeviceConnected = false;
   // New device list update info loop
   for (; currListIndex < status->getODeviceInfo().size(); ++currListIndex)
   {
      //Identify the device type
      if (status->getODeviceInfo().at(currListIndex).getE8DeviceType() == MPlay_fi_types::T_e8_MPlayDeviceType__e8DTY_USB)
      {
         sysPath = status->getODeviceInfo().at(currListIndex).getSDeviceSysPath();
         isCurrIndexDeviceConnected = status->getODeviceInfo().at(currListIndex).getBDeviceConnected();
         ETG_TRACE_USR4(("MediaPlaybackStatusUpdater::onMediaPlayerDeviceConnectionsStatus hub syspath %s", sysPath.c_str()));
         ETG_TRACE_USR4(("MediaPlaybackStatusUpdater::onMediaPlayerDeviceConnectionsStatus isCurrIndexDeviceConnected %d", isCurrIndexDeviceConnected));
         if ((isCurrIndexDeviceConnected) && (tclMediaClientHandler::getInstance().isDeviceConnectedInEvenPortNumber(sysPath)))
         {
            _vehicleProxy->sendDeviceStatusUSB2PresentStart(*this, EN_USB_PRESENT);
         }
         else
         {
            _vehicleProxy->sendDeviceStatusUSB2PresentStart(*this, EN_USB_NOT_PRESENT);
         }

         if ((isCurrIndexDeviceConnected) && (tclMediaClientHandler::getInstance().isDeviceConnectedInEvenPortNumber(sysPath)) == false)
         {
            _vehicleProxy->sendDeviceStatusUSB1PresentStart(*this, EN_USB_PRESENT);
         }
         else
         {
            _vehicleProxy->sendDeviceStatusUSB1PresentStart(*this, EN_USB_NOT_PRESENT);
         }
      }
   }
}


void MediaPlaybackStatusUpdater::onMediaPlayerDeviceConnectionsError(const ::boost::shared_ptr< ::mplay_MediaPlayer_FI::Mplay_MediaPlayer_FIProxy >& /*proxy*/, const boost::shared_ptr< ::mplay_MediaPlayer_FI::MediaPlayerDeviceConnectionsError >& /*error*/)
{
   ETG_TRACE_COMP(("MediaPlaybackStatusUpdater::onMediaPlayerDeviceConnectionsError received."));
}


void MediaPlaybackStatusUpdater::onRepeatModeStatus(const ::boost::shared_ptr< ::mplay_MediaPlayer_FI::Mplay_MediaPlayer_FIProxy >& /*proxy*/, const boost::shared_ptr< ::mplay_MediaPlayer_FI::RepeatModeStatus >& status)
{
   tU8 repeatStatus = status->getE8Repeat();
   ETG_TRACE_USR4(("MediaPlaybackStatusUpdater::onRepeatModeStatus RepeatMode %d", repeatStatus));

   if (MediaSourceHandling::getInstance().isInternalMediaSourceActiveInSink(COCKPIT_SINK_ID))
   {
      switch (repeatStatus)
      {
         case MPlay_fi_types::T_e8_MPlayRepeat__e8RPT_NONE:
         {
            _vehicleProxy->sendMediaStatusRepeatStatusStart(*this, CAN_REPEAT_OFF);
            _repeatCANStatus = CAN_REPEAT_OFF;
         }
         break;
         case MPlay_fi_types::T_e8_MPlayRepeat__e8RPT_ONE:
         {
            _vehicleProxy->sendMediaStatusRepeatStatusStart(*this, CAN_REPEAT_TRACK);
            _repeatCANStatus = CAN_REPEAT_TRACK;
         }
         break;
         case MPlay_fi_types::T_e8_MPlayRepeat__e8RPT_LIST:
         {
            _vehicleProxy->sendMediaStatusRepeatStatusStart(*this, CAN_REPEAT_LIST);
            _repeatCANStatus = CAN_REPEAT_LIST;
         }
         break;
      }
   }
}


void MediaPlaybackStatusUpdater::onRepeatModeError(const ::boost::shared_ptr< ::mplay_MediaPlayer_FI::Mplay_MediaPlayer_FIProxy >& /*proxy*/, const boost::shared_ptr< ::mplay_MediaPlayer_FI::RepeatModeError >& /*error*/)
{
   ETG_TRACE_COMP(("MediaPlaybackStatusUpdater::onRepeatModeError received"));
}


void MediaPlaybackStatusUpdater::onPlaybackModeStatus(const ::boost::shared_ptr< ::mplay_MediaPlayer_FI::Mplay_MediaPlayer_FIProxy >& /*proxy*/, const boost::shared_ptr< ::mplay_MediaPlayer_FI::PlaybackModeStatus >& status)
{
   tU8 shuffleStatus = status->getE8Mode();

   ETG_TRACE_USR4(("MediaPlaybackStatusUpdater::onPlaybackModeStatus RandomMode %d", shuffleStatus));
   if (MediaSourceHandling::getInstance().isInternalMediaSourceActiveInSink(COCKPIT_SINK_ID))
   {
      switch (shuffleStatus)
      {
         case MPlay_fi_types::T_e8_MPlayMode__e8PBM_NORMAL:
         {
            _vehicleProxy->sendMediaStatusMixStateStart(*this, CAN_SHUFFLE_OFF);
            _shuffleCANStatus = CAN_SHUFFLE_OFF;
         }
         break;
         case MPlay_fi_types::T_e8_MPlayMode__e8PBM_RANDOM:
         {
            _vehicleProxy->sendMediaStatusMixStateStart(*this, CAN_SHUFFLE_ON);
            _shuffleCANStatus = CAN_SHUFFLE_ON ;
         }
         break;
      }
   }
}


void MediaPlaybackStatusUpdater::onPlaybackModeError(const ::boost::shared_ptr< ::mplay_MediaPlayer_FI::Mplay_MediaPlayer_FIProxy >& /*proxy*/, const boost::shared_ptr< ::mplay_MediaPlayer_FI::PlaybackModeError >& /*error*/)
{
   ETG_TRACE_COMP(("MediaPlaybackStatusUpdater::onPlaybackModeError received"));
}


void MediaPlaybackStatusUpdater::onActiveSourceError(const ::boost::shared_ptr< MASTERAUDIOSERVICE_INTERFACE::AudioSourceChange::AudioSourceChangeProxy >& /*proxy*/, const ::boost::shared_ptr< MASTERAUDIOSERVICE_INTERFACE::AudioSourceChange::ActiveSourceError >& /*error*/)
{
   ETG_TRACE_COMP(("MediaPlaybackStatusUpdater::onActiveSourceError received"));
}


void MediaPlaybackStatusUpdater::onActiveSourceUpdate(const ::boost::shared_ptr< MASTERAUDIOSERVICE_INTERFACE::AudioSourceChange::AudioSourceChangeProxy >& /*proxy*/,
      const ::boost::shared_ptr< MASTERAUDIOSERVICE_INTERFACE::AudioSourceChange::ActiveSourceUpdate >& update)
{
   ETG_TRACE_COMP(("MediaPlaybackStatusUpdater::onActiveSourceUpdate received"));
   ETG_TRACE_USR4(("MediaPlaybackStatusUpdater::onActiveSourceUpdate srcID :%d", update->getActiveSource().getSrcId()));
   ETG_TRACE_USR4(("MediaPlaybackStatusUpdater::onActiveSourceUpdate _playbackstate :%d", _playbackCANstate));
   ETG_TRACE_USR4(("MediaPlaybackStatusUpdater::onActiveSourceUpdate _currentElapsedTime :%d", _currentElapsedTime));
   ETG_TRACE_USR4(("MediaPlaybackStatusUpdater::onActiveSourceUpdate _shuffleCANStatus :%d", _shuffleCANStatus));
   ETG_TRACE_USR4(("MediaPlaybackStatusUpdater::onActiveSourceUpdate _repeatCANStatus :%d", _repeatCANStatus));
   ETG_TRACE_USR4(("MediaPlaybackStatusUpdater::onActiveSourceUpdate _currentTrackNum :%d", _currentTrackNum));

   uint8 u8activeRegion = ApplicationSwitchClientHandler::poGetInstance()->getActiveRegion();

   if (MediaSourceHandling::getInstance().IsTunerSourceRange(update->getActiveSource().getSrcId()) && update->getActiveSource().getSinkId() == COCKPIT_SINK_ID)
   {
      _vehicleProxy->sendMediaStatusMixStateStart(*this, EN_SNA);
      _vehicleProxy->sendMediaStatusRepeatStatusStart(*this, EN_SNA);
      _vehicleProxy->sendMediaStatusTrackTimeStart(*this, EN_NOT_AVAILABLE);
      _vehicleProxy->sendMediaStatusPlayStatusStart(*this, EN_SNA);
      _vehicleProxy->sendMediaStatusCurrTrackNumStart(*this, EN_NOT_AVAILABLE);
   }
   else if ((MediaSourceHandling::getInstance().IsMediaSourceRange(update->getActiveSource().getSrcId()))
            && (update->getActiveSource().getSinkId() == COCKPIT_SINK_ID))
   {
      if (MediaSourceHandling::getInstance().isInternalMediaSourceActiveInSink(COCKPIT_SINK_ID))
      {
         _vehicleProxy->sendMediaStatusMixStateStart(*this, _shuffleCANStatus);
         _vehicleProxy->sendMediaStatusRepeatStatusStart(*this, _repeatCANStatus);
         _vehicleProxy->sendMediaStatusTrackTimeStart(*this, _currentElapsedTime);
         _vehicleProxy->sendMediaStatusPlayStatusStart(*this, _playbackCANstate);
         _vehicleProxy->sendMediaStatusCurrTrackNumStart(*this, _currentTrackNum);
      }
   }
}


void MediaPlaybackStatusUpdater::onSourceCommandError(const ::boost::shared_ptr< ::VEHICLE_MAIN_FI::VEHICLE_MAIN_FIProxy >& proxy, const ::boost::shared_ptr< ::VEHICLE_MAIN_FI::SourceCommandError >& error)
{
   ETG_TRACE_COMP(("MediaPlaybackStatusUpdater::onSourceCommandError received"));
}


void MediaPlaybackStatusUpdater::onSourceCommandStatus(const ::boost::shared_ptr< ::VEHICLE_MAIN_FI::VEHICLE_MAIN_FIProxy >& proxy, const ::boost::shared_ptr< ::VEHICLE_MAIN_FI::SourceCommandStatus >& Status)
{
   tU8 sourceCommandValue = Status->getSourceCommandValue();
   ETG_TRACE_USR4(("MediaPlaybackStatusUpdater::onSourceCommandStatus sourceCommandValue %d", sourceCommandValue));
   int srcId = SRC_INVALID;
   uint8 u8activeRegion = ApplicationSwitchClientHandler::poGetInstance()->getActiveRegion();

   switch (sourceCommandValue)
   {
      case CAN_SOURCE_ID_USB1:
      {
         int devicetagUSB1 = tclMediaClientHandler::getInstance().getUSB1DeviceTag();
         if (devicetagUSB1 != 0)
         {
            if (u8activeRegion == REGION_COCKPIT && _applicationSwitchClientComponent != NULL)
            {
               _applicationSwitchClientComponent->requestContextSwitchApplication(2, APPID_APPHMI_UNKNOWN, enActivityIDs__eActivityID_NONE, APPID_APPHMI_MEDIA, (bosch::cm::ai::hmi::hmimasterservice::ApplicationSwitch::enActivityIDs)enActivityIDs__eActivityID_MEDIA_MCP_USB1);
            }
            else
            {
               srcId = tclMediaClientHandler::getInstance().getSourceId(FI_EN_E8DTY_USB);
               MediaSourceHandling::getInstance().requestSourceActivation(srcId, devicetagUSB1, u8activeRegion);
               StateTransitionHandler::getInstance().setStateTransitionReason(ST_TRANSITION_REASON_USER_SRC_REQUEST);
            }
         }
      }
      break;
      case CAN_SOURCE_ID_USB2:
      {
         int devicetagUSB2 = tclMediaClientHandler::getInstance().getUSB2DeviceTag();
         if (devicetagUSB2 != 0)
         {
            if (u8activeRegion == REGION_COCKPIT && _applicationSwitchClientComponent != NULL)
            {
               _applicationSwitchClientComponent->requestContextSwitchApplication(2, APPID_APPHMI_UNKNOWN, enActivityIDs__eActivityID_NONE, APPID_APPHMI_MEDIA, (bosch::cm::ai::hmi::hmimasterservice::ApplicationSwitch::enActivityIDs)enActivityIDs__eActivityID_MEDIA_MCP_USB2);
            }
            else
            {
               srcId = tclMediaClientHandler::getInstance().getSourceId(FI_EN_E8DTY_USB);
               MediaSourceHandling::getInstance().requestSourceActivation(srcId, devicetagUSB2, u8activeRegion);
               StateTransitionHandler::getInstance().setStateTransitionReason(ST_TRANSITION_REASON_USER_SRC_REQUEST);
            }
         }
      }
      break;
      case CAN_SOURCE_ID_BLUETOOTH1:
      {
         int devicetagBT = tclMediaClientHandler::getInstance().getDeviceTag(FI_EN_E8DTY_BLUETOOTH);
         if (devicetagBT != 0)
         {
            if (u8activeRegion == REGION_COCKPIT && _applicationSwitchClientComponent != NULL)
            {
               _applicationSwitchClientComponent->requestContextSwitchApplication(2, APPID_APPHMI_UNKNOWN, enActivityIDs__eActivityID_NONE, APPID_APPHMI_MEDIA, (bosch::cm::ai::hmi::hmimasterservice::ApplicationSwitch::enActivityIDs)enActivityIDs__eActivityID_MEDIA_MCP_BTSTREAMING);
            }
            else
            {
               srcId = tclMediaClientHandler::getInstance().getSourceId(FI_EN_E8DTY_BLUETOOTH);
               MediaSourceHandling::getInstance().requestSourceActivation(srcId, devicetagBT, u8activeRegion);
               StateTransitionHandler::getInstance().setStateTransitionReason(ST_TRANSITION_REASON_USER_SRC_REQUEST);
            }
         }
      }
      break;
      default:
      {
         ETG_TRACE_USR1(("MediaPlaybackStatusUpdater::onSourceCommandStatus Default case option"));
      }
      break;
   }
}


}// namespace Core
}// namespace App
