/**************************************************************************************
 * @file         : ListInfoDatabase.cpp
 * @author       : RBEI/ECH2-INF4CV_MediaTeam
 * @addtogroup   : AppHmi_media
 * @brief        :
 * @copyright    : (c) 2018-2020 Robert Bosch Car Multimedia GmbH
 *                 The reproduction, distribution and utilization of this file as
 *                 well as the communication of its contents to others without express
 *                 authorization is prohibited. Offenders will be held liable for the
 *                 payment of damages. All rights reserved in the event of the grant
 *                 of a patent, utility model or design.
 * 		  			Initial Revision.
 **************************************************************************************/


#include "hall_std_if.h"
#include "Core/ListHandling/ListInfoDatabase/ListInfoDatabase.h"
#ifndef VARIANT_S_FTR_ENABLE_UNITTEST
#include "AppHmi_MasterBase/AudioInterface/AudioDefines.h"
#include <map>
#include <vector>
#include <iterator>

#ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
#define ETG_DEFAULT_TRACE_CLASS           TR_CLASS_APPHMI_MEDIA_HALL
#define ETG_I_TRACE_CHANNEL               TR_TTFIS_APPHMI_MEDIA
#define ETG_I_TTFIS_CMD_PREFIX            "APPHMI_MEDIA_"
#define ETG_I_FILE_PREFIX                 App::Core::ListInfoDatabase::
#include "trcGenProj/Header/ListInfoDatabase.cpp.trc.h"
#endif


#endif

#ifdef VARIANT_S_FTR_ENABLE_UNITTEST
#include "utest/stubs/Apphmi_Media_Types_Common.h"
#include "utest/stubs/etg.h"
#endif


namespace App {
namespace Core {


ListInfoDatabase::ListInfoDatabase()
{
   ETG_TRACE_USR1(("ListInfoDatabase constructor \n"));

   stListDetails = (struct stListInfo)
   {
      0, EN_LIST_STATE_UNKNOWN, 0, {0, 0}
   };

   m_activeListType = EnumConst::entListType_Invalid;
}


ListInfoDatabase::~ListInfoDatabase()
{
   ETG_TRACE_USR1(("ListInfoDatabase Destructor \n"));
   resetMembers();
}


void ListInfoDatabase::resetMembers()
{
   ETG_TRACE_USR1(("ListInfoDatabase resetMembers \n"));
   m_activeListType = EnumConst::entListType_Invalid;
   resetListDatabase();
   stListDetails = (struct stListInfo)
   {
      0, EN_LIST_STATE_UNKNOWN, 0, {0, 0}
   };
}


bool ListInfoDatabase::isListInfoAvailable(EnumConst::entListType listType)
{
   ETG_TRACE_USR4(("ListInfoDatabase::isListInfoAvailable listType:%d", ETG_CENUM(entListType, listType)));

   std::map<EnumConst::entListType, stListInfo >::iterator mapIter = _listInfoMap.find(listType);

   bool bStatus = true;
   if (mapIter == _listInfoMap.end())
   {
      bStatus = false;
   }
   else
   {
      ETG_TRACE_USR1(("ListInfoDatabaseHandler setActiveListType DB already available"));
   }
   ETG_TRACE_USR4(("ListInfoDatabase::isListInfoAvailable bStatus:%d", bStatus));

   return bStatus;
}


void ListInfoDatabase::createListDatabase(EnumConst::entListType listType)
{
   ETG_TRACE_USR4(("ListInfoDatabase::createListDatabase listtype:%d", ETG_CENUM(entListType, listType)));

   stListInfo   stInfoDetails = (struct stListInfo)
   {
      0, EN_LIST_STATE_UNKNOWN, 0, {0, 0}
   };

   stInfoDetails.listState = EN_LIST_STATE_UNKNOWN;
   stInfoDetails.totalListSize = 0;
   _listInfoMap.insert(std::pair<EnumConst::entListType, stListInfo>(listType, stInfoDetails));
}


void ListInfoDatabase::resetListDatabase()
{
   ETG_TRACE_USR4(("ListInfoDatabase::resetListDatabase "));
   _listInfoMap.clear();
}


#ifndef VARIANT_S_FTR_ENABLE_UNITTEST
void ListInfoDatabase::updateListSize(EnumConst::entListType listType, int totalListSize)
{
   ETG_TRACE_USR4(("ListInfoDatabase::updateListSize listtype  %d  totalListSize  %d", ETG_CENUM(entListType, listType), totalListSize));

   if (isListInfoAvailable(listType))
   {
      _listInfoMap[listType].totalListSize = totalListSize;
   }
   else
   {
      ETG_TRACE_USR1(("ListInfoDatabase updateListSize DB not available"));
   }
}


enListState ListInfoDatabase::getListState(EnumConst::entListType listType)
{
   enListState currentListState = EN_LIST_STATE_UNKNOWN;

   if (isListInfoAvailable(listType))
   {
      currentListState = _listInfoMap[listType].listState;
   }
   ETG_TRACE_USR4(("ListInfoDatabase::getListState listtype:%d  listState  %d", ETG_CENUM(entListType, listType), ETG_CENUM(enListState, currentListState)));
   return currentListState;
}


bool ListInfoDatabase::isListActive()
{
   std::map<EnumConst::entListType, stListInfo>::iterator itr = _listInfoMap.begin();

   bool bListActive = false;

   for (; itr != _listInfoMap.end(); ++itr)
   {
      if (itr->second.listState == EN_LIST_STATE_ACTIVE)
      {
         bListActive = true;
         m_activeListType = itr->first;
      }
   }
   ETG_TRACE_USR4(("ListInfoDatabase::isListAvailable bListActive  %d", bListActive));
   return bListActive;
}


EnumConst::entListType ListInfoDatabase::getListTypeActive()
{
   ETG_TRACE_USR4(("ListInfoDatabase::isListActive m_activeListType  %d", ETG_CENUM(entListType, m_activeListType)));
   return m_activeListType;
}


int ListInfoDatabase::getCurrentListSize(EnumConst::entListType iActiveListType)
{
   int totalListSize = 0;
   if (isListInfoAvailable(iActiveListType))
   {
      totalListSize = _listInfoMap[iActiveListType].totalListSize;
   }

   ETG_TRACE_USR4(("ListInfoDatabase::getListState iActiveListType:%d  totalListSize  %d", ETG_CENUM(entListType, iActiveListType), totalListSize));
   return totalListSize;
}


void ListInfoDatabase::updateListState(EnumConst::entListType iActiveListType, enListState currentListState)
{
   ETG_TRACE_USR4(("ListInfoDatabase::updateListState iActiveListType  currentListState  %d    %d", ETG_CENUM(entListType, iActiveListType), ETG_CENUM(enListState, currentListState)));
   if (isListInfoAvailable(iActiveListType))
   {
      _listInfoMap[iActiveListType].listState = currentListState;

      if ((currentListState == EN_LIST_STATE_ACTIVE) || (currentListState == EN_LIST_STATE_PROCESSING))
      {
         updateOtherListStatesInactive(iActiveListType);
      }
   }
   else
   {
      ETG_TRACE_USR4(("ListInfoDatabase::updateListState listtype not available"));
   }
}


void ListInfoDatabase::updateOtherListStatesInactive(EnumConst::entListType iActiveListType)
{
   std::map<EnumConst::entListType, stListInfo> ::iterator itr = _listInfoMap.begin();

   ETG_TRACE_USR4(("ListInfoDatabase::updateOtherListStatesInactive size of _listInfoMap   %d ", _listInfoMap.size()));
   for (; itr != _listInfoMap.end(); ++itr)
   {
      if (itr->first != iActiveListType)
      {
         itr->second.listState = EN_LIST_STATE_INACTIVE;
      }
   }
}


void ListInfoDatabase::updateListStateOnDeviceRemoval()
{
   ETG_TRACE_USR4(("ListInfoDatabase::updateListStateOnDeviceRemoval size of _listInfoMap   %d ", _listInfoMap.size()));
   if (_listInfoMap.size() != 0)
   {
      std::map<EnumConst::entListType, stListInfo> ::iterator itr = _listInfoMap.begin();
      for (; itr != _listInfoMap.end(); ++itr)
      {
         itr->second.listState = EN_LIST_STATE_UNKNOWN;
      }
   }
}


void ListInfoDatabase::updateListStateOnBrowseExit()
{
   ETG_TRACE_USR4(("ListInfoDatabase::updateListStateOnBrowseExit size of _listInfoMap   %d  m_activeListType   %d", _listInfoMap.size(), ETG_CENUM(entListType, m_activeListType)));

   if (_listInfoMap.size() != 0)
   {
      std::map<EnumConst::entListType, stListInfo> ::iterator itr = _listInfoMap.begin();
      for (; itr != _listInfoMap.end(); ++itr)
      {
         if (itr->second.listState == EN_LIST_STATE_ACTIVE)
         {
            itr->second.listState = EN_LIST_STATE_INACTIVE;
         }
      }
   }
}


void ListInfoDatabase::updateAbsoluteIndex(EnumConst::entListType iActiveListType, std::vector<uint16>& _absoluteIndex)
{
   ETG_TRACE_USR4(("ListInfoDatabase::updateAbsoluteIndex iActiveListType  %d , vector size  %d ", ETG_CENUM(entListType, iActiveListType), _absoluteIndex.size()));
   if (isListInfoAvailable(iActiveListType))
   {
      for (int index = 0; index < _absoluteIndex.size(); ++index)
      {
         _listInfoMap[iActiveListType].absoluteIndex.push_back(_absoluteIndex[index]);
      }
   }
}


int ListInfoDatabase::getAbsoluteIndex(EnumConst::entListType iActiveListType, int selectedrow)
{
   int _absoluteIndex = 0;

   if (isListInfoAvailable(iActiveListType))
   {
      if (_listInfoMap[iActiveListType].absoluteIndex.size() > selectedrow)
      {
         int index = selectedrow > _listInfoMap[iActiveListType].startIndex ? selectedrow - _listInfoMap[iActiveListType].startIndex : selectedrow;
         _absoluteIndex = _listInfoMap[iActiveListType].absoluteIndex[index];
      }
   }

   ETG_TRACE_USR4(("ListInfoDatabase::getListState iActiveListType:%d  _absoluteIndex  %d", ETG_CENUM(entListType, iActiveListType), _absoluteIndex));
   return _absoluteIndex;
}


void ListInfoDatabase::updateStartIndex(EnumConst::entListType iActiveListType, uint16 startIndex)
{
   ETG_TRACE_USR4(("ListInfoDatabase::updateStartIndex startIndex  %d ", startIndex));
   _listInfoMap[iActiveListType].startIndex = startIndex;
}


#endif
}// namespace Core
}// namespace App
