/**************************************************************************************
 * @file         : VerticalListGuiResponseHandler.cpp
 * @author       : RBEI/ECH2-INF4CV_MediaTeam
 * @addtogroup   : AppHmi_media
 * @brief        :
 * @copyright    : (c) 2018-2020 Robert Bosch Car Multimedia GmbH
 *                 The reproduction, distribution and utilization of this file as
 *                 well as the communication of its contents to others without express
 *                 authorization is prohibited. Offenders will be held liable for the
 *                 payment of damages. All rights reserved in the event of the grant
 *                 of a patent, utility model or design.
 * 		  			Initial Revision.
 **************************************************************************************/

#include "hall_std_if.h"
#include "Core/ListHandling/ListGuiResponseHandler/VerticalListGuiResponseHandler.h"
#include "DataModel/HMIModelListImplementation.h"
#include "DataModel/HMIModelComponent.h"
#include "AppHmi_MasterBase/AudioInterface/AudioDefines.h"
#include "CgiExtensions/ImageLoader.h"
#include "MPlay_fi_types.h"
#include "AvdeccPluginConstant.h"
#include <vector>

#ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
#define ETG_DEFAULT_TRACE_CLASS           TR_CLASS_APPHMI_MEDIA_HALL
#define ETG_I_TRACE_CHANNEL               TR_TTFIS_APPHMI_MEDIA
#define ETG_I_TTFIS_CMD_PREFIX            "APPHMI_MEDIA_"
#define ETG_I_FILE_PREFIX                 App::Core::VerticalListGuiResponseHandler::
#include "trcGenProj/Header/VerticalListGuiResponseHandler.cpp.trc.h"
#endif

#define SLIDERSIZE 50

namespace App {
namespace Core {

VerticalListGuiResponseHandler::VerticalListGuiResponseHandler(ListContainer* _container, ListConfiguration* _listConfiguration)
{
   ETG_TRACE_USR1(("VerticalListGuiResponseHandler constructor \n"));
   vCreateMapListTypeInfo();
   _listDataContainer = _container;
   _listConfig = _listConfiguration;
   m_startIndex = 0;
   m_totalListsize = 0;
   initVerticalListSize();
}


VerticalListGuiResponseHandler::~VerticalListGuiResponseHandler()
{
   ETG_TRACE_USR1(("VerticalListGuiResponseHandler Destructor \n"));
}


void VerticalListGuiResponseHandler::initVerticalListSize()
{
   ETG_TRACE_USR1(("VerticalListGuiResponseHandler initVerticalListSize \n"));

   _verticalListsize.DefaultBackButtonSize = 0.0f;
   _verticalListsize.DefaultAllAlbumAndTitleButtonSize = 0.0f;
   _verticalListsize.BackButtonSize = 66.0f;
   _verticalListsize.AllAlbumAndTitleButtonSize = 66.0f;
   _verticalListsize.ListheightWithoutButtons = 330.0f;
   _verticalListsize.ListheightWithBackButton = 264.0f;
   _verticalListsize.ListheightWithBackAndAllButton = 198.0f;
   _verticalListsize.ListWidth = 760.0f;
   _verticalListsize.ScrollBarWidth = 35.0f;
}


void VerticalListGuiResponseHandler::resetMembers()
{
   ETG_TRACE_USR1(("VerticalListGuiResponseHandler resetMembers \n"));
   m_startIndex = 0;
   m_totalListsize = 0;
}


/* ***********************************************************
* Function 		: updateListContent
* Description 	: Updates the List data Based on the List Type
* Parameters 	:
* Return value	: void
* ***********************************************************/
void VerticalListGuiResponseHandler::updateListContent(EnumConst::entListType listType, uint16 verticalStartIndex, uint32 totalListSize)
{
   int iNumberofText = _listConfig->getNumberOfTextInList(listType);
   ETG_TRACE_USR4(("VerticalListGuiResponseHandler::updateListContent listtype   iNumberofText   %d   %d ", ETG_CENUM(entListType, listType), iNumberofText));
   m_startIndex = verticalStartIndex;
   m_totalListsize = totalListSize;

   if (iNumberofText == 1)
   {
      if (listType == EnumConst::entListType_Video_Languages)
      {
         oneFieldListCreationForVideoLanguage(listType);
      }
      else
      {
         oneFieldListCreation(listType);
      }
   }
   else if (iNumberofText == 2)
   {
      twoFieldListCreation(listType);
   }

   // _verticalListobj->onListLoadingComplete();
}


void VerticalListGuiResponseHandler::oneFieldListCreation(EnumConst::entListType listType)
{
   uint32 dataIcon = _listConfig->getIconTypeForList(listType);
   int _currentListtype = static_cast<int>(listType);

   ListDataProviderBuilder listBuilder(LIST_ID_EXTERNAL_MEDIA_VER);

   std::vector<std::string> textToFillList = _listDataContainer->getFirstTextOfList();
   std::vector<std::string>::iterator itr = textToFillList.begin();
   FolderListDataItemData data;

   for (int index = 0; (itr != textToFillList.end()); itr++, index++)
   {
      if (listType == EnumConst::entListType_Folder)
      {
         data.mIconImageIndexVal = _listDataContainer->checkFileType(index);

         listBuilder.AddItem(
            static_cast<int>(index),
            _currentListtype,
            _listTypeInfo[listType]).AddData(textToFillList[index].c_str()).AddDataBindingUpdater<FolderListDataItemDataBindingSource>(data);
      }
      else
      {
         listBuilder.AddItem(
            static_cast<int>(index),
            _currentListtype,
            _listTypeInfo[listType]).AddData(textToFillList[index].c_str()).AddData(dataIcon);
      }
   }

   tSharedPtrDataProvider dataProvider = listBuilder.CreateDataProvider(m_startIndex, m_totalListsize);
   updateSizeAndVisibilityDuringListBuild(listType);
   dataProvider->setCacheOnOff(false);
   dataProvider->setListChangeSetIndex(m_startIndex);
   ETG_TRACE_USR4(("HMIModelListImplementation::oneFieldListCreation m_totalListsize%d m_startIndex: %d", m_totalListsize, m_startIndex));

   POST_MSG((COURIER_MESSAGE_NEW(ListDateProviderResMsg)(dataProvider)));
   HMIModelListImplementation::getInstance().setWaitAnimationState(ANIMATION_STOP);
}


void VerticalListGuiResponseHandler::oneFieldListCreationForVideoLanguage(EnumConst::entListType listType)
{
   int _currentListtype = static_cast<int>(listType);
   ETG_TRACE_COMP(("VerticalListGuiResponseHandler::oneFieldListCreationForVideoLanguage: _currentListtype %d", _currentListtype));

   ListDataProviderBuilder listBuilder(LIST_ID_EXTERNAL_MEDIA_VER);

   std::vector<std::string> textToFillList = _listDataContainer->getFirstTextOfList();
   std::vector<std::string>::iterator itr1 = textToFillList.begin();
   std::vector<bool> languageSelStatus = _listDataContainer->getIsLanguageSelected();
   std::vector<bool>::iterator itr2 = languageSelStatus.begin();

   listBuilder.AddItem(0, _currentListtype, "Layer_AllLanguages").AddData(_listDataContainer->isAllLanguageSelected());
   for (int index = 1; (itr1 != textToFillList.end()), (itr2 != languageSelStatus.end()); itr1++, itr2++, index++)
   {
      listBuilder.AddItem(
         static_cast<int>(index),
         _currentListtype,
         _listTypeInfo[listType]).AddData(textToFillList[index - 1].c_str()).AddData(languageSelStatus[index - 1]);
   }

   tSharedPtrDataProvider dataProvider = listBuilder.CreateDataProvider(m_startIndex, m_totalListsize + 1);
   updateSizeAndVisibilityDuringListBuild(listType);
   dataProvider->setCacheOnOff(false);
   dataProvider->setListChangeSetIndex(m_startIndex);
   ETG_TRACE_USR4(("HMIModelListImplementation::oneFieldListCreationForVideoLanguage m_totalListsize%d m_startIndex:%d", m_totalListsize, m_startIndex));

   POST_MSG((COURIER_MESSAGE_NEW(ListDateProviderResMsg)(dataProvider)));
   HMIModelListImplementation::getInstance().setWaitAnimationState(ANIMATION_STOP);
}


void VerticalListGuiResponseHandler::twoFieldListCreation(EnumConst::entListType listType)
{
   uint32 dataIcon = _listConfig->getIconTypeForList(listType);
   int _currentListtype = static_cast<int>(listType);

   ListDataProviderBuilder listBuilder(LIST_ID_EXTERNAL_MEDIA_VER);

   std::vector<std::string> firstTextToFillList = _listDataContainer->getFirstTextOfList();
   std::vector<std::string> secondTextToFillList = _listDataContainer->getSecondTextOfList();
   std::vector<std::string>::iterator itr = firstTextToFillList.begin();

   for (int index = 0; (itr != firstTextToFillList.end()); itr++, index++)
   {
      listBuilder.AddItem(
         static_cast<int>(index),
         _currentListtype,
         _listTypeInfo[listType])
      .AddData(firstTextToFillList[index].c_str())
      .AddData(secondTextToFillList[index].c_str())
      .AddData(dataIcon);
   }

   tSharedPtrDataProvider dataProvider = listBuilder.CreateDataProvider(m_startIndex, m_totalListsize);
   updateSizeAndVisibilityDuringListBuild(listType);
   dataProvider->setCacheOnOff(false);
   dataProvider->setListChangeSetIndex(m_startIndex);
   ETG_TRACE_USR4(("VerticalListGuiResponseHandler::twoFieldListCreation m_totalListsize%d m_startIndex:%d", m_totalListsize, m_startIndex));

   POST_MSG((COURIER_MESSAGE_NEW(ListDateProviderResMsg)(dataProvider)));
   HMIModelListImplementation::getInstance().setWaitAnimationState(ANIMATION_STOP);
}


void VerticalListGuiResponseHandler::updateSizeAndVisibilityDuringListBuild(EnumConst::entListType listType)
{
   bool isPrevButtonAvailable = _listConfig->getPreviousButtonAvilability(listType);
   bool isAllButtonAvailable = _listConfig->getAllButtonAvilability(listType);
   std::string _folderPath = _listDataContainer->getFolderPath();

   if (listType == EnumConst::entListType_Folder)
   {
      if (_folderPath == "/")
      {
         updateVerticalListSize(_verticalListsize.DefaultBackButtonSize, _verticalListsize.DefaultAllAlbumAndTitleButtonSize, _verticalListsize.ListheightWithoutButtons, _verticalListsize.ListWidth, _verticalListsize.ScrollBarWidth);

         updateBackAndAllButtonVisibility(false, false);
         updateListVisibility(true);
         updateTitleAndAlbumIconVisibility(false, false);
         updateButtonText("");
      }
      else
      {
         updateVerticalListSize(_verticalListsize.BackButtonSize, _verticalListsize.DefaultAllAlbumAndTitleButtonSize, _verticalListsize.ListheightWithBackButton, _verticalListsize.ListWidth, _verticalListsize.ScrollBarWidth);

         updateBackAndAllButtonVisibility(true, false);
         updateListVisibility(true);
         updateTitleAndAlbumIconVisibility(false, false);
         updateButtonText("");
      }
   }
   else
   {
      if (isPrevButtonAvailable == false && isAllButtonAvailable == false)
      {
         updateVerticalListSize(_verticalListsize.DefaultBackButtonSize, _verticalListsize.DefaultAllAlbumAndTitleButtonSize, _verticalListsize.ListheightWithoutButtons, _verticalListsize.ListWidth, _verticalListsize.ScrollBarWidth);

         updateBackAndAllButtonVisibility(false, false);
         updateListVisibility(true);
         updateTitleAndAlbumIconVisibility(false, false);
         updateButtonText("");
      }
      else if (isPrevButtonAvailable == true && isAllButtonAvailable == true)
      {
         Candera::String strTitle = HMIModelComponent::getInstance().GetListAllTitleId();
         updateVerticalListSize(_verticalListsize.BackButtonSize, _verticalListsize.AllAlbumAndTitleButtonSize, _verticalListsize.ListheightWithBackAndAllButton, _verticalListsize.ListWidth, _verticalListsize.ScrollBarWidth);

         updateBackAndAllButtonVisibility(true, true);
         updateListVisibility(true);
         updateTitleAndAlbumIconVisibility(true, false);
         updateButtonText(strTitle);
      }
      else
      {
         updateVerticalListSize(_verticalListsize.BackButtonSize, _verticalListsize.DefaultAllAlbumAndTitleButtonSize, _verticalListsize.ListheightWithBackButton, _verticalListsize.ListWidth, _verticalListsize.ScrollBarWidth);

         updateBackAndAllButtonVisibility(true, false);
         updateListVisibility(true);
         updateTitleAndAlbumIconVisibility(false, false);
         updateButtonText("");
      }
   }

   m_ExternalBrowseScreenListSize.MarkAllItemsModified();
   m_ExternalBrowseScreenListSize.SendUpdate();
}


void VerticalListGuiResponseHandler::updateVerticalListSize(float backButtonSize, float allAlbumAndTitleButtonSize, float verticalListSize, float verticalListWidth, float scrollBarWidth)
{
   float sliderSize = 0.0f;
   ETG_TRACE_USR4(("VerticalListGuiResponseHandler::updateVerticalListSize:backButtonSize:%f ,allAlbumAndTitleButtonSize:%f,verticalListSize: %f", backButtonSize, allAlbumAndTitleButtonSize, verticalListSize));
   if (verticalListSize > 0)
   {
      sliderSize = (SLIDERSIZE / verticalListSize);
   }
   ETG_TRACE_USR4(("HMIModelListImplementation::updateVerticalListSize: sliderSize %f ", sliderSize));
   (*m_ExternalBrowseScreenListSize).mPreviousButtonSize.SetX(verticalListWidth);
   (*m_ExternalBrowseScreenListSize).mPreviousButtonSize.SetY(backButtonSize);
   (*m_ExternalBrowseScreenListSize).mAllAlbumAndTitleButtonSize.SetX(verticalListWidth);
   (*m_ExternalBrowseScreenListSize).mAllAlbumAndTitleButtonSize.SetY(allAlbumAndTitleButtonSize);
   (*m_ExternalBrowseScreenListSize).mVerticalListSize.SetX(verticalListWidth);
   (*m_ExternalBrowseScreenListSize).mVerticalListSize.SetY(verticalListSize);
   (*m_ExternalBrowseScreenListSize).mScrollBarSize.SetX(scrollBarWidth);
   (*m_ExternalBrowseScreenListSize).mScrollBarSize.SetY(verticalListSize);
   (*m_ExternalBrowseScreenListSize).mScrollBarSliderSize = sliderSize;
   m_ExternalBrowseScreenListSize.MarkAllItemsModified();
   m_ExternalBrowseScreenListSize.SendUpdate();
}


/* ***********************************************************
* Function 		: updateEmptyListData
* Description 	: To update the empty list
* Parameters 	:
* Return value	: void
* ***********************************************************/
void VerticalListGuiResponseHandler::updateEmptyListData(EnumConst::entListType listType)
{
   ListDataProviderBuilder listBuilder(LIST_ID_EXTERNAL_MEDIA_VER);
   tSharedPtrDataProvider dataProvider = listBuilder.CreateDataProvider();

   bool PrevButtonAvailability = _listConfig->getPreviousButtonAvilability(listType);
   ETG_TRACE_USR4(("VerticalListGuiResponseHandler::updateEmptyListData PrevButtonAvailability  %d", PrevButtonAvailability));

   if (PrevButtonAvailability == true)
   {
      updateVerticalListSize(_verticalListsize.BackButtonSize, _verticalListsize.DefaultAllAlbumAndTitleButtonSize, 0.0f, 0.0f, 0.0f);

      updateBackAndAllButtonVisibility(true, false);
      updateListVisibility(false);
      updateTitleAndAlbumIconVisibility(false, false);
      (*m_ExternalBrowseScreenListSize).mAllAlbumAndTitleText = "";
   }
   else
   {
      updateVerticalListSize(_verticalListsize.DefaultBackButtonSize, _verticalListsize.DefaultAllAlbumAndTitleButtonSize, 0.0f, 0.0f, 0.0f);
      updateBackAndAllButtonVisibility(false, false);
      updateListVisibility(false);
      updateTitleAndAlbumIconVisibility(false, false);
      updateButtonText("");
   }

   POST_MSG((COURIER_MESSAGE_NEW(ListDateProviderResMsg)(dataProvider)));
   HMIModelListImplementation::getInstance().setWaitAnimationState(ANIMATION_STOP);
   updateListInfoText(true);
}


void VerticalListGuiResponseHandler::updateListInfoText(bool listState)
{
   ETG_TRACE_USR4(("VerticalListGuiResponseHandler::updateListInfoText listState  %d", listState));
   (*m_BrowseListInfoText).mListInfoTextVisiblity = listState;
   m_BrowseListInfoText.MarkAllItemsModified();
   m_BrowseListInfoText.SendUpdate(false);
}


void VerticalListGuiResponseHandler::updateBackAndAllButtonVisibility(bool backBtnVisibility, bool allBtnVisibility)
{
   (*m_ExternalBrowseScreenListSize).mPreviousButtonVisibility = backBtnVisibility;
   (*m_ExternalBrowseScreenListSize).mAllAlbumAndTitleButtonVisibility = allBtnVisibility;

   m_ExternalBrowseScreenListSize.MarkAllItemsModified();
   m_ExternalBrowseScreenListSize.SendUpdate();
}


void VerticalListGuiResponseHandler::updateListVisibility(bool listVisibility)
{
   (*m_ExternalBrowseScreenListSize).mListVisibility = listVisibility;
   m_ExternalBrowseScreenListSize.MarkAllItemsModified();
   m_ExternalBrowseScreenListSize.SendUpdate();
}


void VerticalListGuiResponseHandler::updateTitleAndAlbumIconVisibility(bool titleIconVisibility, bool albumIconVisibility)
{
   (*m_ExternalBrowseScreenListSize).mTitleIconVisibility = titleIconVisibility;
   (*m_ExternalBrowseScreenListSize).mAlbumIconVisibility = albumIconVisibility;
   m_ExternalBrowseScreenListSize.MarkAllItemsModified();
   m_ExternalBrowseScreenListSize.SendUpdate();
}


void VerticalListGuiResponseHandler::updateButtonText(Candera::String text)
{
   (*m_ExternalBrowseScreenListSize).mAllAlbumAndTitleText = text;
   m_ExternalBrowseScreenListSize.MarkAllItemsModified();
   m_ExternalBrowseScreenListSize.SendUpdate();
}


void VerticalListGuiResponseHandler::vCreateMapListTypeInfo()
{
   _listTypeInfo.insert(pair<EnumConst::entListType, const char*>(EnumConst::entListType_Song, "Layer_Three_Item"));
   _listTypeInfo.insert(pair<EnumConst::entListType, const char*>(EnumConst::entListType_Album, "Layer_Two_Item"));
   _listTypeInfo.insert(pair<EnumConst::entListType, const char*>(EnumConst::entListType_Album_Song, "Layer_Three_Item"));
   _listTypeInfo.insert(pair<EnumConst::entListType, const char*>(EnumConst::entListType_Artist, "Layer_Two_Item"));
   _listTypeInfo.insert(pair<EnumConst::entListType, const char*>(EnumConst::entListType_Artist_Album, "Layer_Three_Item"));
   _listTypeInfo.insert(pair<EnumConst::entListType, const char*>(EnumConst::entListType_Artist_Album_Song, "Layer_Three_Item"));
   _listTypeInfo.insert(pair<EnumConst::entListType, const char*>(EnumConst::entListType_Artist_Song, "Layer_Three_Item"));
   _listTypeInfo.insert(pair<EnumConst::entListType, const char*>(EnumConst::entListType_Genre, "Layer_Two_Item"));
   _listTypeInfo.insert(pair<EnumConst::entListType, const char*>(EnumConst::entListType_Genre_Album, "Layer_Three_Item"));
   _listTypeInfo.insert(pair<EnumConst::entListType, const char*>(EnumConst::entListType_Genre_Album_Song, "Layer_Three_Item"));
   _listTypeInfo.insert(pair<EnumConst::entListType, const char*>(EnumConst::entListType_Genre_Song, "Layer_Three_Item"));
   _listTypeInfo.insert(pair<EnumConst::entListType, const char*>(EnumConst::entListType_Playlist, "Layer_Two_Item"));
   _listTypeInfo.insert(pair<EnumConst::entListType, const char*>(EnumConst::entListType_Playlist_song, "Layer_Three_Item"));
   _listTypeInfo.insert(pair<EnumConst::entListType, const char*>(EnumConst::entListType_Folder, "Layer_FolderListEntry_1_1"));
   _listTypeInfo.insert(pair<EnumConst::entListType, const char*>(EnumConst::entListType_Folder_file, "Layer_Three_Item"));
   _listTypeInfo.insert(pair<EnumConst::entListType, const char*>(EnumConst::entListType_Video, "Layer_Video_Item"));
   _listTypeInfo.insert(pair<EnumConst::entListType, const char*>(EnumConst::entListType_Video_Languages, "Layer_IndividualLanguages"));
   _listTypeInfo.insert(pair<EnumConst::entListType, const char*>(EnumConst::entListType_Video_episode, "Layer_Three_Item"));
}


}// namespace Core
}// namespace App
