/**************************************************************************************
 * @file         : EMPListInterfaceHandler.cpp
 * @author       : RBEI/ECH2-INF4CV_MediaTeam
 * @addtogroup   : AppHmi_media
 * @brief        :
 * @copyright    : (c) 2018-2020 Robert Bosch Car Multimedia GmbH
 *                 The reproduction, distribution and utilization of this file as
 *                 well as the communication of its contents to others without express
 *                 authorization is prohibited. Offenders will be held liable for the
 *                 payment of damages. All rights reserved in the event of the grant
 *                 of a patent, utility model or design.
 * 		  			Initial Revision.
 **************************************************************************************/

#include "hall_std_if.h"
#include "Core/ListHandling/EMPListInterfaceHandler/EMPListInterfaceHandler.h"
#include "Core/ListHandling/ListInfoDatabaseHandler/ListInfoDatabaseHandler.h"
#include "Core/ListHandling/ListContainer/ListContainer.h"
#include "Core/StateTransitionHandler/StateTransitionHandler.h"
#include "Core/ApplicationSwitchClientHandler/ApplicationSwitchClientHandler.h"
#include "Core/SourceSwitch/MediaSourceHandling.h"
#include "AvdeccPluginConstant.h"
#include <vector>
#include <string>
#include <stdint.h>

#ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
#define ETG_DEFAULT_TRACE_CLASS           TR_CLASS_APPHMI_MEDIA_HALL
#define ETG_I_TRACE_CHANNEL               TR_TTFIS_APPHMI_MEDIA
#define ETG_I_TTFIS_CMD_PREFIX            "APPHMI_MEDIA_"
#define ETG_I_FILE_PREFIX                 App::Core::EMPListInterfaceHandler::
#include "trcGenProj/Header/EMPListInterfaceHandler.cpp.trc.h"
#endif

using namespace ::bosch::cm::di::HmiEmpListInterface;

namespace App {
namespace Core {

EMPListInterfaceHandler::EMPListInterfaceHandler(VerticalListManager* _obj, ListContainer* _container,
      ListInfoDatabaseHandler* _listInfoDBHandlerObj, EnumConst::entSourceType currentSourceType)
   : _hmiEmpListInterfaceProxy(HmiEmpListInterfaceProxy::createProxy("HmiEmpListInterfacePort", *this))
{
   ETG_TRACE_USR1(("EMPListInterfaceHandler constructor \n"));
   StartupSync::getInstance().registerPropertyRegistrationIF(this, "HmiEmpListInterfacePort");

   _verticalListManager = _obj;
   _listDataContainer = _container;
   _listInfoDBHandler = _listInfoDBHandlerObj;
   m_requestedlistType = EnumConst::entListType_Invalid;
   m_currentSourceType = currentSourceType;
   _startIndex = 0;
}


EMPListInterfaceHandler::~EMPListInterfaceHandler()
{
   ETG_TRACE_USR1(("EMPListInterfaceHandler Destructor \n"));
   resetMembers();
}


void EMPListInterfaceHandler::resetMembers()
{
   ETG_TRACE_USR1(("EMPListInterfaceHandler resetMembers \n"));
   m_requestedlistType = EnumConst::entListType_Invalid;
   m_currentSourceType = EnumConst::entSourceType_DEFAULT;
}


void EMPListInterfaceHandler::registerProperties(const ::boost::shared_ptr< asf::core::Proxy >& proxy, const asf::core::ServiceStateChange& /*stateChange*/)
{
   ETG_TRACE_COMP(("EMPListInterfaceHandler::registerProperties"));
}


void EMPListInterfaceHandler::deregisterProperties(const ::boost::shared_ptr< asf::core::Proxy >& proxy, const asf::core::ServiceStateChange& /*stateChange*/)
{
   ETG_TRACE_COMP(("EMPListInterfaceHandler::deregisterProperties"));
}


void EMPListInterfaceHandler::onUnavailable(const boost::shared_ptr<asf::core::Proxy>& proxy, const asf::core::ServiceStateChange& /*stateChange*/)
{
   ETG_TRACE_COMP(("EMPListInterfaceHandler::onUnavailable"));
   if (_hmiEmpListInterfaceProxy && _hmiEmpListInterfaceProxy == proxy)
   {
      _hmiEmpListInterfaceProxy->sendListSizeDeregisterAll();
   }
}


void EMPListInterfaceHandler::onAvailable(const ::boost::shared_ptr< asf::core::Proxy >& proxy, const asf::core::ServiceStateChange& /*stateChange*/)
{
   ETG_TRACE_COMP(("ExternalMediaListDBManager::onAvailable"));
   if (_hmiEmpListInterfaceProxy && _hmiEmpListInterfaceProxy == proxy)
   {
      _hmiEmpListInterfaceProxy->sendListSizeRegister(*this);
   }
}


void EMPListInterfaceHandler::onReqCreateMediaPlayerListError(const ::boost::shared_ptr< HmiEmpListInterfaceProxy >& proxy, const ::boost::shared_ptr< ReqCreateMediaPlayerListError >& error)
{
   ETG_TRACE_USR1(("EMPListInterfaceHandler::EMPListInterfaceHandler: onReqCreateMediaPlayerListError"));
}


void EMPListInterfaceHandler::onReqCreateMediaPlayerListResponse(const ::boost::shared_ptr< HmiEmpListInterfaceProxy >& proxy, const ::boost::shared_ptr< ReqCreateMediaPlayerListResponse >& response)
{
   bool bListresp = response->getResponse();
   ETG_TRACE_USR4(("EMPListInterfaceHandler::EMPListInterfaceHandler:bListresp %d", bListresp));
}


void EMPListInterfaceHandler::onReqCreateMediaPlayerFolderListError(const ::boost::shared_ptr< HmiEmpListInterfaceProxy >& proxy, const ::boost::shared_ptr< ReqCreateMediaPlayerFolderListError >& error)
{
   ETG_TRACE_USR1(("EMPListInterfaceHandler::EMPListInterfaceHandler: onReqCreateMediaPlayerFolderListError"));
}


void EMPListInterfaceHandler::onReqCreateMediaPlayerFolderListResponse(const ::boost::shared_ptr< HmiEmpListInterfaceProxy >& proxy, const ::boost::shared_ptr< ReqCreateMediaPlayerFolderListResponse >& response)
{
   uint32 _currentListSize = response->getListCount();
   EnumConst::entSourceType srcType = getCurrentSourceType();
   ETG_TRACE_USR4(("EMPListInterfaceHandler::onReqCreateMediaPlayerFolderListResponse:_currentListSize  m_requestedlistType %d  %d", _currentListSize, m_requestedlistType));
   if (m_currentSourceType == srcType)
   {
      _listInfoDBHandler->updateCurrentListSize(m_requestedlistType, _currentListSize);
      _verticalListManager->onListCreateResponse();
   }
}


void EMPListInterfaceHandler::onSetMediaVideoLanguagesListError(const ::boost::shared_ptr< HmiEmpListInterfaceProxy >& proxy,
      const ::boost::shared_ptr< SetMediaVideoLanguagesListError >& error)
{
   ETG_TRACE_COMP(("EMPListInterfaceHandler::onSetMediaVideoLanguagesListError"));
}


void EMPListInterfaceHandler::onSetMediaVideoLanguagesListResponse(const ::boost::shared_ptr< HmiEmpListInterfaceProxy >& proxy,
      const ::boost::shared_ptr< SetMediaVideoLanguagesListResponse >& update)
{
   ETG_TRACE_COMP(("EMPListInterfaceHandler::onSetMediaVideoLanguagesListResponse"));
}


void EMPListInterfaceHandler::createListRequest(EnumConst::entListType listType, uint16 absoluteIndex1, uint16 absoluteIndex2)
{
   ETG_TRACE_USR4(("EMPListInterfaceHandler::createListRequest listType   absoluteIndex1 absoluteIndex2 %d  %d %d", ETG_CENUM(entListType, listType), absoluteIndex1, absoluteIndex2));

   uint8 u8ActiveRegion = ApplicationSwitchClientHandler::poGetInstance()->getActiveRegion();
   ETG_TRACE_USR4(("EMPListInterfaceHandler::createListRequest u8ActiveRegion %d, ", u8ActiveRegion));
   EnumConst::entSourceType srcType = getCurrentSourceType();
   ETG_TRACE_USR4(("EMPListInterfaceHandler::createListRequest srcType   %d", ETG_CENUM(entSourceType, srcType)));

   m_requestedlistType = listType;
   bosch::cm::di::HmiEmpListInterface::entListType currentListType = static_cast <bosch::cm::di::HmiEmpListInterface::entListType>(listType);
   bosch::cm::di::HmiEmpListInterface::entSourceType currentSourceType = static_cast <bosch::cm::di::HmiEmpListInterface::entSourceType>(m_currentSourceType);
   if (m_currentSourceType == srcType)
   {
      _hmiEmpListInterfaceProxy->sendReqCreateMediaPlayerListRequest(*this, currentListType, currentSourceType, u8ActiveRegion, absoluteIndex1/*, absoluteIndex2*/);
   }
}


void EMPListInterfaceHandler::createListRequest(EnumConst::entListType listType, std::string folderPath)
{
   ETG_TRACE_COMP(("EMPListInterfaceHandler::createListRequest"));
   uint8 u8ActiveRegion = ApplicationSwitchClientHandler::poGetInstance()->getActiveRegion();
   ETG_TRACE_USR4(("EMPListInterfaceHandler::createListRequest u8ActiveRegion %d, ", u8ActiveRegion));
   EnumConst::entSourceType srcType = getCurrentSourceType();
   ETG_TRACE_USR4(("EMPListInterfaceHandler::createListRequest srcType   %d", ETG_CENUM(entSourceType, srcType)));
   m_requestedlistType = listType;
   bosch::cm::di::HmiEmpListInterface::entSourceType currentSourceType = static_cast <bosch::cm::di::HmiEmpListInterface::entSourceType>(m_currentSourceType);
   if (m_currentSourceType == srcType)
   {
      _hmiEmpListInterfaceProxy->sendReqCreateMediaPlayerFolderListRequest(*this, currentSourceType, u8ActiveRegion, folderPath);
   }
}


void EMPListInterfaceHandler::onListSizeError(const ::boost::shared_ptr< HmiEmpListInterfaceProxy >& proxy, const ::boost::shared_ptr< ListSizeError >& error)
{
   ETG_TRACE_USR1(("EMPListInterfaceHandler::onListSizeError Entered"));
   _verticalListManager->onListErrorResponse();
}


void EMPListInterfaceHandler::onListSizeUpdate(const ::boost::shared_ptr< HmiEmpListInterfaceProxy >& proxy, const ::boost::shared_ptr< ListSizeUpdate >& update)
{
   uint32 _currentListSize = update->getListSize().getCount();
   EnumConst::entSourceType srcType = getCurrentSourceType();
   ETG_TRACE_USR4(("EMPListInterfaceHandler::onListSizeUpdate srcType   %d", ETG_CENUM(entSourceType, srcType)));
   ETG_TRACE_USR4(("EMPListInterfaceHandler::onListSizeUpdate m_requestedlistType  _currentListSize   %d   %d", ETG_CENUM(entListType, m_requestedlistType), _currentListSize));
   if (m_currentSourceType == srcType)
   {
      _listInfoDBHandler->updateCurrentListSize(m_requestedlistType, _currentListSize);
      _verticalListManager->onListCreateResponse();
   }
}


void EMPListInterfaceHandler::sliceListRequest(EnumConst::entListType listType, uint16 startIndex, uint16 windowSize, uint16 absoluteIndex1, uint16 absoluteIndex2)
{
   uint8 u8ActiveRegion = ApplicationSwitchClientHandler::poGetInstance()->getActiveRegion();
   _startIndex = startIndex;
   ETG_TRACE_USR4(("EMPListInterfaceHandler::sliceListRequest listType  %d startIndex  %d windowSize  %d  u8ActiveRegion  %d ", ETG_CENUM(entListType, listType), startIndex, windowSize, u8ActiveRegion));
   EnumConst::entSourceType srcType = getCurrentSourceType();
   ETG_TRACE_USR4(("EMPListInterfaceHandler::sliceListRequest srcType   %d", ETG_CENUM(entSourceType, srcType)));

   if (m_currentSourceType == srcType)
   {
      _hmiEmpListInterfaceProxy->sendReqMediaPlayerListSliceRequest(*this, (bosch::cm::di::HmiEmpListInterface::entListType)listType, (bosch::cm::di::HmiEmpListInterface::entSourceType)m_currentSourceType, u8ActiveRegion, startIndex, windowSize/*, absoluteIndex1, absoluteIndex2*/);
   }
}


void EMPListInterfaceHandler::sliceListRequest(uint16 startIndex, uint16 windowSize, std::string folderPath)
{
   ETG_TRACE_COMP(("EMPListInterfaceHandler::sliceListRequest"));
   uint8 u8ActiveRegion = ApplicationSwitchClientHandler::poGetInstance()->getActiveRegion();
   _startIndex = startIndex;
   ETG_TRACE_USR4(("EMPListInterfaceHandler::sliceListRequest startIndex  %d windowSize  %d  u8ActiveRegion  %d ", startIndex, windowSize, u8ActiveRegion));
   EnumConst::entSourceType srcType = getCurrentSourceType();
   ETG_TRACE_USR4(("EMPListInterfaceHandler::sliceListRequest srcType   %d", ETG_CENUM(entSourceType, srcType)));

   if (m_currentSourceType == srcType)
   {
      _hmiEmpListInterfaceProxy->sendReqMediaPlayerFolderListSliceRequest(*this, (bosch::cm::di::HmiEmpListInterface::entSourceType)m_currentSourceType, u8ActiveRegion, startIndex, windowSize, folderPath);
   }
}


void EMPListInterfaceHandler::onReqMediaPlayerListSliceError(const ::boost::shared_ptr< HmiEmpListInterfaceProxy >& proxy, const ::boost::shared_ptr< ReqMediaPlayerListSliceError >& error)
{
   ETG_TRACE_USR1(("EMPListInterfaceHandler::onReqMediaPlayerListSliceError Entered"));
   _verticalListManager->onListErrorResponse();
}


void EMPListInterfaceHandler::onReqMediaPlayerListSliceResponse(const ::boost::shared_ptr< HmiEmpListInterfaceProxy >& proxy, const ::boost::shared_ptr< ReqMediaPlayerListSliceResponse >& response)
{
   ETG_TRACE_USR1(("EMPListInterfaceHandler::onReqMediaPlayerListSliceResponse Entered"));
   ::std::vector<bosch::cm::di::HmiEmpListInterface::ListContainerStruct> listContainer = response->getMArrayListContainer();

   EnumConst::entSourceType srcType = getCurrentSourceType();
   ETG_TRACE_USR4(("EMPListInterfaceHandler::onReqMediaPlayerListSliceResponse srcType   %d", ETG_CENUM(entSourceType, srcType)));

   if (m_currentSourceType == srcType)
   {
      updateFirstTextToContainer(listContainer);
      updateSecondTextToContainer(listContainer);
      updateAbsoluteIndexToContainer(listContainer);
      updateLanguageSelectionStatusToContainer(listContainer);
      updateAbsoluteIndexToDBHandler(listContainer);
      updateFileTypeToContainer(listContainer);
      _listInfoDBHandler->updateStartIndex(m_requestedlistType, _startIndex);
      _verticalListManager->onListSliceResponse();
   }
}


void EMPListInterfaceHandler::onReqMediaPlayerFolderListSliceError(const ::boost::shared_ptr< HmiEmpListInterfaceProxy >& proxy, const ::boost::shared_ptr< ReqMediaPlayerFolderListSliceError >& error)
{
   ETG_TRACE_USR1(("EMPListInterfaceHandler::onReqMediaPlayerFolderListSliceError Entered"));
   _verticalListManager->onListErrorResponse();
}


void EMPListInterfaceHandler::onReqMediaPlayerFolderListSliceResponse(const ::boost::shared_ptr< HmiEmpListInterfaceProxy >& proxy, const ::boost::shared_ptr< ReqMediaPlayerFolderListSliceResponse >& response)
{
   ETG_TRACE_USR1(("EMPListInterfaceHandler::onReqMediaPlayerFolderListSliceResponse Entered"));
   ::std::vector<bosch::cm::di::HmiEmpListInterface::ListContainerStruct> listContainer = response->getMArrayListContainer();

   EnumConst::entSourceType srcType = getCurrentSourceType();
   ETG_TRACE_USR4(("EMPListInterfaceHandler::onReqMediaPlayerFolderListSliceResponse srcType   %d", ETG_CENUM(entSourceType, srcType)));

   if (m_currentSourceType == srcType)
   {
      updateFirstTextToContainer(listContainer);
      updateSecondTextToContainer(listContainer);
      updateAbsoluteIndexToContainer(listContainer);
      updateLanguageSelectionStatusToContainer(listContainer);
      updateAbsoluteIndexToDBHandler(listContainer);
      updateFileTypeToContainer(listContainer);
      _listInfoDBHandler->updateStartIndex(m_requestedlistType, _startIndex);
      _verticalListManager->onListSliceResponse();
   }
}


void EMPListInterfaceHandler::updateFirstTextToContainer(std::vector<bosch::cm::di::HmiEmpListInterface::ListContainerStruct>& listContainer)
{
   ETG_TRACE_USR4(("EMPListInterfaceHandler::updateFirstTextToContainer vector size  %d", listContainer.size()));

   std::vector<string> metadataList;

   for (int index = 0; index < listContainer.size(); ++index)
   {
      metadataList.push_back(listContainer[index].getText1());
   }
   _listDataContainer->setFirstTextOfList(metadataList);
}


void EMPListInterfaceHandler::updateSecondTextToContainer(std::vector<bosch::cm::di::HmiEmpListInterface::ListContainerStruct>& listContainer)
{
   ETG_TRACE_USR4(("EMPListInterfaceHandler::updateSecondTextToContainer vector size  %d", listContainer.size()));

   std::vector<string> metadataList;

   for (int index = 0; index < listContainer.size(); ++index)
   {
      metadataList.push_back(listContainer[index].getText2());
   }
   _listDataContainer->setSecondTextOfList(metadataList);
}


void EMPListInterfaceHandler::updateAbsoluteIndexToContainer(std::vector<bosch::cm::di::HmiEmpListInterface::ListContainerStruct>& listContainer)
{
   ETG_TRACE_USR4(("EMPListInterfaceHandler::updateAbsoluteIndexToContainer vector size  %d", listContainer.size()));

   std::vector<uint16> absoluteIndex;

   for (int index = 0; index < listContainer.size(); ++index)
   {
      absoluteIndex.push_back(listContainer[index].getAbsoluteIndex());
   }
   _listDataContainer->setAbsoluteIndex(absoluteIndex);
}


void EMPListInterfaceHandler::updateLanguageSelectionStatusToContainer(std::vector<bosch::cm::di::HmiEmpListInterface::ListContainerStruct>& listContainer)
{
   ETG_TRACE_COMP(("EMPListInterfaceHandler::updateLanguageSelectionStatusToContainer vector size  %d", listContainer.size()));

   std::vector<bool> isSelectedLanguage;

   for (int index = 0; index < listContainer.size(); ++index)
   {
      isSelectedLanguage.push_back(listContainer[index].getIsSelectedLanguage());
   }
   _listDataContainer->setIsLanguageSelected(isSelectedLanguage);
}


void EMPListInterfaceHandler::updateFileTypeToContainer(std::vector<bosch::cm::di::HmiEmpListInterface::ListContainerStruct>& listContainer)
{
   ETG_TRACE_COMP(("EMPListInterfaceHandler::updateFileTypeToContainer vector size  %d", listContainer.size()));

   std::vector<bosch::cm::di::HmiEmpListInterface::entFileType> fileType;

   for (int index = 0; index < listContainer.size(); ++index)
   {
      fileType.push_back(listContainer[index].getFileType());
   }
   _listDataContainer->setFileType(fileType);
}


EnumConst::entSourceType EMPListInterfaceHandler::getCurrentSourceType()
{
   int iActiveSrcId = MediaSourceHandling::getInstance().getCurrentRegionSrcId();

   EnumConst::entSourceType srcType = EnumConst::entSourceType_DEFAULT;

   if (iActiveSrcId == SRC_CAMERAPORT1_USB_AUDIO || iActiveSrcId == SRC_CAMERAPORT1_USB_VIDEO)
   {
      srcType = EnumConst::entSourceType_CMP;
   }
   else if (iActiveSrcId == SRC_CMR1_USB1_VIDEO || iActiveSrcId == SRC_CMR1_USB1_AUDIO)
   {
      srcType = EnumConst::entSourceType_CMR;
   }
   ETG_TRACE_USR4(("EMPListInterfaceHandler::onCourierMessage iActiveSrcId  %d   srcType %d", iActiveSrcId, ETG_CENUM(entSourceType, srcType)));

   return srcType;
}


void EMPListInterfaceHandler::updateAbsoluteIndexToDBHandler(std::vector<bosch::cm::di::HmiEmpListInterface::ListContainerStruct>& listContainer)
{
   std::vector<uint16> absoluteIndex;
   for (int index = 0; index < listContainer.size(); ++index)
   {
      absoluteIndex.push_back(listContainer[index].getAbsoluteIndex());
      _listInfoDBHandler->updateAbsoluteIndex(m_requestedlistType, absoluteIndex);
   }
}


void EMPListInterfaceHandler::updateSelectedLanguageList()
{
   ::entSourceType srcType = static_cast<bosch::cm::di::HmiEmpListInterface::entSourceType>(getCurrentSourceType());
   ETG_TRACE_USR1(("EMPListInterfaceHandler::updateSelectedLanguageList: srcType %d", ETG_CENUM(bosch::cm::di::HmiEmpListInterface::entSourceType, srcType)));
   uint8 region = ApplicationSwitchClientHandler::poGetInstance()->getActiveRegion();
   std::vector<uint16> selectedLanguagesId = _listDataContainer->getSelectedLanguageId();
   _hmiEmpListInterfaceProxy->sendSetMediaVideoLanguagesListRequest(*this, srcType, region, selectedLanguagesId);
}


}// namespace Core
}// namespace App
