/**************************************************************************************
* @file         : HmiStateHandler.cpp
* @author       : RBEI/ECG5-INF4CV_MediaTeam
* @addtogroup   : AppHmi_media
* @brief        : HmiStateHandler is to handle HALL implementation for
*                 SPM_CORE_FI service
* @copyright    : (c) 2018-2020 Robert Bosch Car Multimedia GmbH
*                 The reproduction, distribution and utilization of this file as
*                 well as the communication of its contents to others without express
*                 authorization is prohibited. Offenders will be held liable for the
*                 payment of damages. All rights reserved in the event of the grant
*                 of a patent, utility model or design.
**************************************************************************************/
#include "hall_std_if.h"
#include "HmiStateHandler.h"
#include "spm_core_fi_typesConst.h"

#ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
#define ETG_DEFAULT_TRACE_CLASS         TR_CLASS_APPHMI_MEDIA_HALL
#define ETG_I_TRACE_CHANNEL             TR_TTFIS_APPHMI_MEDIA
#define ETG_I_TTFIS_CMD_PREFIX          "APPHMI_MEDIA_"
#define ETG_I_FILE_PREFIX               App::Core::HmiStateHandler::
#include "trcGenProj/Header/HmiStateHandler.cpp.trc.h"
#endif


using namespace ::SPM_CORE_FI;
using namespace ::spm_core_fi_types;

namespace App {
namespace Core {

HmiStateHandler::HmiStateHandler()
{
   _spmProxy = (::SPM_CORE_FIProxy::createProxy("spmFiPort", *this));
   ETG_TRACE_COMP(("HmiStateHandler Constructor \n"));
   if (_spmProxy.get())
   {
      ETG_TRACE_USR1(("HmiStateHandler::HmiStateHandler StartupSync _spmProxy"));
      StartupSync::getInstance().registerPropertyRegistrationIF(this, _spmProxy->getPortName());
   }
}


HmiStateHandler::~HmiStateHandler()
{
   ETG_TRACE_COMP(("HmiStateHandler Destructor \n"));
}


void HmiStateHandler::onAvailable(const ::boost::shared_ptr< asf::core::Proxy >& proxy, const ServiceStateChange& stateChange)
{
   ETG_TRACE_COMP(("HmiStateHandler::onAvailable"));

   if (_spmProxy && _spmProxy == proxy)
   {
      _spmProxy->sendHmiStateUpReg(*this);
   }
}


void HmiStateHandler::onUnavailable(const ::boost::shared_ptr< asf::core::Proxy >& proxy, const ServiceStateChange& stateChange)
{
   ETG_TRACE_COMP(("HmiStateHandler::onUnavailable"));

   if (_spmProxy && _spmProxy == proxy)
   {
      _spmProxy->sendHmiStateRelUpRegAll();
   }
}


void HmiStateHandler::registerProperties(const ::boost::shared_ptr< asf::core::Proxy >& proxy, const asf::core::ServiceStateChange& /*stateChange*/)
{
   ETG_TRACE_COMP(("HmiStateHandler::registerProperties"));
}


void HmiStateHandler::deregisterProperties(const ::boost::shared_ptr< asf::core::Proxy >& proxy, const asf::core::ServiceStateChange& /*stateChange*/)
{
   ETG_TRACE_COMP(("HmiStateHandler::deregisterProperties"));
}


/************************************************************************
*FUNCTION		: onHmiStateError
*DESCRIPTION	: To update HMI state
*PARAMETER		: NA
*PARAMETER		: NA
*RETURNVALUE	: void
*SWFL			: [Task 1363977]
************************************************************************/
void HmiStateHandler::onHmiStateError(const ::boost::shared_ptr< SPM_CORE_FIProxy >& proxy, const boost::shared_ptr< HmiStateError >& error)
{
   ETG_TRACE_ERR(("HmiStateHandler: onHmiStateError"));
}


/************************************************************************
*FUNCTION		: onHmiStateStatus
*DESCRIPTION	: To update HMI state
*PARAMETER		: NA
*PARAMETER		: NA
*RETURNVALUE	: void
*SWFL			: [Task 1363977]
************************************************************************/
void HmiStateHandler::onHmiStateStatus(const ::boost::shared_ptr< SPM_CORE_FIProxy >& proxy, const boost::shared_ptr< HmiStateStatus >& status)
{
   ETG_TRACE_COMP(("HmiStateHandler: onHmiStateStatus: State: %d", ETG_CENUM(T_SPM_e32_HMI_STATE, status->getHmiState())));
   notifyHmiStateStatusListeners(status->getHmiState());
}


void  HmiStateHandler::addHmiStateStatusObserver(IHmiStateStatusObserver& imp)
{
   ETG_TRACE_COMP(("HmiStateHandler::addHmiStateStatusObserver is called"));

   std::vector< IHmiStateStatusObserver* >::const_iterator itr = std::find(_hmiStateStatusListener.begin(), _hmiStateStatusListener.end(), (&imp));
   if (itr == _hmiStateStatusListener.end())
   {
      _hmiStateStatusListener.push_back((&imp));
   }
}


void  HmiStateHandler::removeHmiStateStatusObserver(IHmiStateStatusObserver& imp)
{
   ETG_TRACE_COMP(("HmiStateHandler::removeHmiStateStatusObserver is called"));

   std::vector< IHmiStateStatusObserver* >::iterator itr = std::find(_hmiStateStatusListener.begin(), _hmiStateStatusListener.end(), (&imp));
   if (itr != _hmiStateStatusListener.end())
   {
      (void)_hmiStateStatusListener.erase(itr);
   }
}


void HmiStateHandler::notifyHmiStateStatusListeners(uint8 currentHmiState)
{
   ETG_TRACE_COMP(("HmiStateHandler::notifyHmiStateStatusListeners: currentHmiState = %d", ETG_CENUM(T_SPM_e32_HMI_STATE, currentHmiState)));

   for (std::vector<IHmiStateStatusObserver* >::const_iterator itr = _hmiStateStatusListener.begin(); (itr != _hmiStateStatusListener.end()); ++itr)
   {
      if (NULL != (*itr))
      {
         (*itr)->onHmiStateUpdate(currentHmiState);
      }
   }
}


}// namespace Core
}// namespace App
