/**************************************************************************************
 * @file         : ExternalMediaSourceHandler.cpp
 * @author       : RBEI/ECH2-INF4CV_MediaTeam
 * @addtogroup   : AppHmi_media
 * @brief        :
 * @copyright    : (c) 2018-2020 Robert Bosch Car Multimedia GmbH
 *                 The reproduction, distribution and utilization of this file as
 *                 well as the communication of its contents to others without express
 *                 authorization is prohibited. Offenders will be held liable for the
 *                 payment of damages. All rights reserved in the event of the grant
 *                 of a patent, utility model or design.
 * 		  			Initial Revision.
 **************************************************************************************/

#include "hall_std_if.h"
#include "Core/ExternalMediaSourceHandler/ExternalMediaSourceHandler.h"
#ifndef VARIANT_S_FTR_ENABLE_UNITTEST
#include "Core/MediaSourceListHandler/MediaSourceListHandler.h"
#include "AppHmi_MasterBase/AudioInterface/AudioDefines.h"
#include "Core/SourceSwitch/MediaSourceHandling.h"

#include "Core/ExternalMediaPlaybackHandler/ExternalMediaPlaybackHandler.h"

#ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
#define ETG_DEFAULT_TRACE_CLASS           TR_CLASS_APPHMI_MEDIA_HALL
#define ETG_I_TRACE_CHANNEL               TR_TTFIS_APPHMI_MEDIA
#define ETG_I_TTFIS_CMD_PREFIX            "APPHMI_MEDIA_"
#define ETG_I_FILE_PREFIX                 App::Core::ExternalMediaSourceHandler::
#include "trcGenProj/Header/ExternalMediaSourceHandler.cpp.trc.h"
#endif

#ifdef DP_DATAPOOL_ID
#define DP_S_IMPORT_INTERFACE_FI
#include "dp_generic_if.h"
#endif

using namespace MASTERAUDIOSERVICE_INTERFACE::AudioSourceChange;
#endif

namespace App {
namespace Core {


/**
	* Description     : Constructor of class ExternalMediaSourceHandler
	*/
#ifndef VARIANT_S_FTR_ENABLE_UNITTEST
ExternalMediaSourceHandler::ExternalMediaSourceHandler()
   : _audioSourceChangeProxy(AudioSourceChangeProxy::createProxy("audioSourceChangePort", *this))
{
   ETG_TRACE_USR1(("ExternalMediaSourceHandler::ExternalMediaSourceHandler constructor \n"));
   StartupSync::getInstance().registerPropertyRegistrationIF(this, "audioSourceChangePort");

   m_camportUsbSourceAvailObservor.clear();
   m_camportUsbSourceAllocationObservor.clear();
   m_routerSourceAllocObservor.clear();
   m_hdmiSourceAvailObservor.clear();
   m_routerSourceAvailObservor.clear();
   m_disconnectedSource = SRC_INVALID;
   m_bCabinHdmiActiveState = false;
   m_activeSourceInfo.clear();
   _cabinSource = SRC_INVALID;
   _cabinSourcePlaybackState = EXT_MEDIA_PLAY;
}


#else
ExternalMediaSourceHandler::ExternalMediaSourceHandler()
{
   m_activeSourceInfo.clear();
}


#endif


/**
* Description     : Destructor of class ExternalMediaSourceHandler
*/
ExternalMediaSourceHandler::~ExternalMediaSourceHandler()
{
#ifndef VARIANT_S_FTR_ENABLE_UNITTEST
   m_camportUsbSourceAvailObservor.clear();
   m_camportUsbSourceAllocationObservor.clear();
   m_hdmiSourceAvailObservor.clear();
   m_routerSourceAllocObservor.clear();
   _audioSourceChangeProxy.reset();
   m_prevAudioSrcList.clear();
   m_routerSourceAvailObservor.clear();
   _cabinSource = SRC_INVALID;
   _cabinSourcePlaybackState = EXT_MEDIA_PLAY;
#endif
   m_activeSourceInfo.clear();
}


/**
	* Description     : Request from Framework to start registering for properties. This request means GUI is ready now.
	*
	* @param[in]      : proxy : reference to the base class of all proxies
	* @param[in]      : stateChange : represents a state transition of a service
	* @return         : void
	*/
#ifndef VARIANT_S_FTR_ENABLE_UNITTEST
void ExternalMediaSourceHandler::registerProperties(const ::boost::shared_ptr< asf::core::Proxy >& proxy, const asf::core::ServiceStateChange& /*stateChange*/)
{
   ETG_TRACE_COMP(("ExternalMediaSourceHandler::registerProperties"));
}


/**
	* Description     : Request from Framework to start de-registering for properties.
	*
	* @param[in]      : proxy : reference to the base class of all proxies
	* @param[in]      : stateChange : represents a state transition of a service
	* @return         : void
	*/
void ExternalMediaSourceHandler::deregisterProperties(const ::boost::shared_ptr< asf::core::Proxy >& proxy, const asf::core::ServiceStateChange& /*stateChange*/)
{
   ETG_TRACE_COMP(("ExternalMediaSourceHandler::deregisterProperties"));
}


void ExternalMediaSourceHandler::onUnavailable(const boost::shared_ptr<asf::core::Proxy>& proxy, const asf::core::ServiceStateChange& /*stateChange*/)
{
   ETG_TRACE_COMP(("ExternalMediaClientHandler::onUnavailable"));
   if (_audioSourceChangeProxy && _audioSourceChangeProxy == proxy)
   {
      _audioSourceChangeProxy->sendDeregisterAll();
   }
}


void ExternalMediaSourceHandler::onAvailable(const ::boost::shared_ptr< asf::core::Proxy >& proxy, const asf::core::ServiceStateChange& /*stateChange*/)
{
   ETG_TRACE_COMP(("ExternalMediaSourceHandler::onAvailable"));
   if (_audioSourceChangeProxy && _audioSourceChangeProxy == proxy)
   {
      _audioSourceChangeProxy->sendActiveSourceListRegister(*this);
   }
}


#endif


bool ExternalMediaSourceHandler::isExternalMediaSource(int srcid)
{
   ETG_TRACE_USR4(("ExternalMediaSourceHandler::IsExternalMediaSource:sourceid %d ", srcid));
   if (((srcid >= SRC_CAMERAPORT1_CAM1) && (srcid <= SRC_CAMERAPORT4_HDMI)) || (srcid == SRC_CMR1_USB1_VIDEO) || (srcid == SRC_CMR1_USB1_AUDIO) || (srcid == SRC_CAMERAPORT1_USB_AUDIO))
   {
      return true;
   }
   else
   {
      return false;
   }
}


void ExternalMediaSourceHandler::updateActiveExternalSourceInfoToMap(int iSrcId, int iSinkId, int iConnectionState) //write UT
{
   if (isExternalMediaSource(iSrcId))
   {
      stActiveSourceInfo stActiveInfoDetails;
      stActiveInfoDetails.srcId = iSrcId;
      stActiveInfoDetails.connectionState = iConnectionState;
      m_activeSourceInfo.insert(std::pair<int, stActiveSourceInfo>(iSinkId, stActiveInfoDetails));
   }
}


bool ExternalMediaSourceHandler::isCurrentSrcExternalUSB(int iSrcId)
{
   bool bSourceAvail = false;

   if ((iSrcId == SRC_CAMERAPORT1_USB_VIDEO) || (iSrcId == SRC_CAMERAPORT1_USB_AUDIO) || (iSrcId == SRC_CMR1_USB1_VIDEO) || (iSrcId == SRC_CMR1_USB1_AUDIO))
   {
      bSourceAvail = true;
   }
   ETG_TRACE_USR4(("ExternalMediaSourceHandler::isCurrentSrcExternalUSB CamportSourceAvail %d   iSrcId  %d", bSourceAvail, iSrcId));
   return bSourceAvail;
}


bool ExternalMediaSourceHandler::isCurrentSrcHdmi(int iSrcId)
{
   bool bSourceAvail = false;

   if ((iSrcId == SRC_CAMERAPORT1_HDMI) || (iSrcId == SRC_CAMERAPORT2_HDMI) || (iSrcId == SRC_CAMERAPORT3_HDMI))
   {
      bSourceAvail = true;
   }
   ETG_TRACE_USR4(("ExternalMediaSourceHandler::isCurrentSrcHdmi HdmiSourceAvail %d   iSrcId  %d", bSourceAvail, iSrcId));
   return bSourceAvail;
}


bool ExternalMediaSourceHandler::isCamportHDMISourceAvailableInCabinSink(int iDisconnSrcId, int sinkId)
{
   bool bSourceAvail = false;
   if (!m_activeSourceInfo.empty())
   {
      std::map<int, stActiveSourceInfo>::iterator itr = m_activeSourceInfo.begin();
      for (; itr != m_activeSourceInfo.end(); ++itr)
      {
         if (itr->first == sinkId)
         {
            if (itr->second.srcId == iDisconnSrcId || itr->second.srcId == iDisconnSrcId || itr->second.srcId == iDisconnSrcId)
            {
               bSourceAvail = true;
            }
         }
      }
   }
   ETG_TRACE_USR4(("ExternalMediaSourceHandler::isCamportHDMISourceAvailableInCabinSink CamportHDMISourceAvail:%d sinkId %d  iDisconnSrcId %d", bSourceAvail, sinkId, iDisconnSrcId));
   return bSourceAvail;
}


bool ExternalMediaSourceHandler::isCamportUsbSourceAvailableInCabinSink(int sinkId)
{
   bool bSourceAvail = false;

   if (!m_activeSourceInfo.empty())
   {
      std::map<int, stActiveSourceInfo>::iterator itr = m_activeSourceInfo.begin();

      for (; itr != m_activeSourceInfo.end(); ++itr)
      {
         if (itr->first == sinkId)
         {
            if (itr->second.srcId == SRC_CAMERAPORT1_USB_VIDEO || itr->second.srcId == SRC_CAMERAPORT1_USB_AUDIO)
            {
               bSourceAvail = true;
            }
         }
      }
   }

   ETG_TRACE_USR4(("ExternalMediaSourceHandler::isCamportUsbSourceAvailableInCabinSink  CamportSourceAvail %d", bSourceAvail));
   return bSourceAvail;
}


bool ExternalMediaSourceHandler::isRouterSourceAvailableInCabinSink(int sinkId)
{
   bool bSourceAvail = false;
   if (!m_activeSourceInfo.empty())
   {
      std::map<int, stActiveSourceInfo>::iterator itr = m_activeSourceInfo.begin();
      for (; itr != m_activeSourceInfo.end(); ++itr)
      {
         if (itr->first == sinkId)
         {
            if (itr->second.srcId == SRC_CMR1_USB1_VIDEO || itr->second.srcId == SRC_CMR1_USB1_AUDIO)
            {
               bSourceAvail = true;
            }
         }
      }
   }
   ETG_TRACE_USR4(("ExternalMediaSourceHandler::isRouterSourceAvailableInCabinSink CamportSourceAvail:%d sinkId %d", bSourceAvail, sinkId));
   return bSourceAvail;
}


#ifndef VARIANT_S_FTR_ENABLE_UNITTEST
void ExternalMediaSourceHandler::updateCamportUsbSourceAvailabilityObservors(int iSourceAvailability)
{
   ETG_TRACE_USR4(("ExternalMediaSourceHandler::updateCamportUsbSourceAvailabilityObservors iSourceAvailability %d", iSourceAvailability));
   for (::std::vector< ICamportUsbSourceAvailabilityObservor* >::const_iterator itr = m_camportUsbSourceAvailObservor.begin(); (itr != m_camportUsbSourceAvailObservor.end()); ++itr)
   {
      if (NULL != (*itr))
      {
         (*itr)->onCamportUsbSourceAvailabilityUpdate(iSourceAvailability);
      }
   }
}


void ExternalMediaSourceHandler::updateHDMISourceAvailabilityObservors(int srcId, int iSourceAvailability)
{
   ETG_TRACE_USR4(("ExternalMediaSourceHandler::updateHDMISourceAvailabilityObservors iSourceAvailability %d", iSourceAvailability));
   for (::std::vector< IhdmiSourceAvailabilityObservor* >::const_iterator itr = m_hdmiSourceAvailObservor.begin(); (itr != m_hdmiSourceAvailObservor.end()); ++itr)
   {
      if (NULL != (*itr))
      {
         (*itr)->onHdmiSourceAvailabilityUpdate(srcId, iSourceAvailability);
      }
   }
}


void ExternalMediaSourceHandler::addCamportUsbSourceAvailabilityObservor(ICamportUsbSourceAvailabilityObservor& imp)
{
   ETG_TRACE_USR1(("ExternalMediaSourceHandler::addCamportUsbSourceAvailabilityObservor entered"));
   ::std::vector< ICamportUsbSourceAvailabilityObservor* >::const_iterator itr = ::std::find(m_camportUsbSourceAvailObservor.begin(), m_camportUsbSourceAvailObservor.end(), (&imp));
   if (itr == m_camportUsbSourceAvailObservor.end())
   {
      m_camportUsbSourceAvailObservor.push_back((&imp));
   }
}


void ExternalMediaSourceHandler::removeCamportUsbSourceAvailabilityObservor(ICamportUsbSourceAvailabilityObservor& imp)
{
   ETG_TRACE_USR1(("ExternalMediaSourceHandler::removeCamportUsbSourceAvailabilityObservor entered"));
   ::std::vector< ICamportUsbSourceAvailabilityObservor* >::iterator itr = ::std::find(m_camportUsbSourceAvailObservor.begin(), m_camportUsbSourceAvailObservor.end(), (&imp));
   if (itr != m_camportUsbSourceAvailObservor.end())
   {
      (void)m_camportUsbSourceAvailObservor.erase(itr);
   }
}


/* onActiveSourceListError - This status is received from HMI_Master if there are any errors in ActiveSourceList request.
*
* @param[in] proxy
* @parm[in] error
* @return void
*/
void ExternalMediaSourceHandler::onActiveSourceListError(const ::boost::shared_ptr< MASTERAUDIOSERVICE_INTERFACE::AudioSourceChange::AudioSourceChangeProxy >& proxy,
      const ::boost::shared_ptr< MASTERAUDIOSERVICE_INTERFACE::AudioSourceChange::ActiveSourceListError >& error)
{
   ETG_TRACE_COMP(("ExternalMediaSourceHandler::onActiveSourceListError\n"));
}


/**
* onActiveSourceListUpdate - This status is received from HMI_Master on ActiveSourceList request.
*
* @param[in] proxy
* @parm[in] status
* @return void
*/
void ExternalMediaSourceHandler::onActiveSourceListUpdate(const ::boost::shared_ptr< MASTERAUDIOSERVICE_INTERFACE::AudioSourceChange::AudioSourceChangeProxy >& proxy,
      const ::boost::shared_ptr< MASTERAUDIOSERVICE_INTERFACE::AudioSourceChange::ActiveSourceListUpdate >& signal)
{
   ETG_TRACE_COMP(("ExternalMediaSourceHandler::onActiveSourceListUpdate()"));

   if (signal != NULL)
   {
      std::vector<sourceData>AudioSourceList = signal->getActiveSourceList();
      if (!AudioSourceList.empty())
      {
         std::vector<sourceData>::const_iterator iter = AudioSourceList.begin(); // stack top is list bottom
         std::vector<sourceData>::const_iterator mapIter = AudioSourceList.begin();//to Update ExternalSourceInfoMap

         m_activeSourceInfo.clear();
         while (mapIter != AudioSourceList.end())
         {
            ETG_TRACE_COMP(("ExternalMediaSourceHandler::onActiveSourceListUpdate UpdateMap SrcID  %d Sink Id   %d   ConnectionState %d", mapIter->getSrcId(), mapIter->getSinkId(), mapIter->getConnectionState()));
            updateActiveExternalSourceInfoToMap(mapIter->getSrcId(), mapIter->getSinkId(), mapIter->getConnectionState());
            mapIter++;
         }

         checkExternalSourceDisconnected();

         while (iter != AudioSourceList.end())
         {
            ETG_TRACE_COMP(("ExternalMediaSourceHandler::onActiveSourceListUpdate SrcID  %d Sink Id   %d   ConnectionState %d", iter->getSrcId(), iter->getSinkId(), iter->getConnectionState()));
            updateActiveExternalSourceInfoToMap(iter->getSrcId(), iter->getSinkId(), iter->getConnectionState());

            if (isCurrentSrcExternalUSB(iter->getSrcId()) || isCurrentSrcHdmi(iter->getSrcId()))
            {
               updateExternalSourceStateToObservers(iter->getSrcId(), iter->getSinkId(), iter->getConnectionState());
            }
            if (((iter->getSrcId() == SRC_MIC1_ANNOUNCEMENT) && ((iter->getSinkId() == CABIN_A_SINK_ID) || (iter->getSinkId() == CABIN_B_SINK_ID)) && (iter->getConnectionState() == STATE_CONNECTED)) && (isMic1AvailableInPrevSourceList(CABIN_A_SINK_ID) == false))
            {
               if (MediaSourceHandling::getInstance().isExternalMediaSourceActiveInSink(CABIN_A_SINK_ID) == true)
               {
                  _cabinSource = MediaSourceHandling::getInstance().getSinkSrcId(CABIN_A_SINK_ID);
                  _cabinSourcePlaybackState = ExternalMediaPlaybackHandler::getInstance()->getPlaybackState();
               }
            }
            if (((iter->getSrcId() != SRC_MIC1_ANNOUNCEMENT) && (iter->getSinkId() == CABIN_A_SINK_ID)) && (isMic1AvailableInPrevSourceList(CABIN_A_SINK_ID) == true))
            {
               if (MediaSourceHandling::getInstance().isExternalMediaSourceActiveInSink(CABIN_A_SINK_ID) == false)
               {
                  _cabinSource = SRC_INVALID;
                  _cabinSourcePlaybackState = EXT_MEDIA_PLAY;
               }
            }
            ETG_TRACE_USR1(("ExternalMediaSourceHandler::onActiveSourceListUpdate: _cabinSource %d, _cabinSourcePlaybackState %d", _cabinSource, _cabinSourcePlaybackState));
            iter++;
         }
         checkExternalSourceDisconnected();
         m_prevAudioSrcList = AudioSourceList;
      }
   }
}


bool ExternalMediaSourceHandler::isMic1AvailableInPrevSourceList(int sinkId)
{
   bool isMic1Available = false;
   std::vector<sourceData>::const_iterator itr = m_prevAudioSrcList.begin();
   while (itr != m_prevAudioSrcList.end())
   {
      if ((itr->getSrcId() == SRC_MIC1_ANNOUNCEMENT) && (itr->getSinkId() == sinkId))
      {
         isMic1Available = true;
      }
      itr++;
   }
   ETG_TRACE_USR4(("ExternalMediaSourceHandler::isMic1AvailableInPrevSourceList: sinkId  %d, isMic1Available %d", sinkId, isMic1Available));
   return isMic1Available;
}


bool ExternalMediaSourceHandler::isMediaSrcAvailableInPreviousSrcList(int sinkId)
{
   bool isMediaSrcAvailable = false;
   std::vector<sourceData>::const_iterator itr = m_prevAudioSrcList.begin();
   while (itr != m_prevAudioSrcList.end())
   {
      if (itr->getSinkId() == sinkId)
      {
         if (MediaSourceHandling::getInstance().IsMediaSourceRange(itr->getSrcId()))
         {
            isMediaSrcAvailable = true;
         }
      }
      itr++;
   }
   ETG_TRACE_USR4(("ExternalMediaSourceHandler::isMediaSrcAvailableInPreviousSrcList: sinkId  %d, isMediaSrcAvailable %d", sinkId, isMediaSrcAvailable));
   return isMediaSrcAvailable;
}


uint8 ExternalMediaSourceHandler::getCabinSrcPlaybackState()
{
   ETG_TRACE_USR1(("ExternalMediaSourceHandler::getCabinSrcPlaybackState: _cabinSourcePlaybackState %d", _cabinSourcePlaybackState));
   return _cabinSourcePlaybackState;
}


void ExternalMediaSourceHandler::setCabinSrcPlaybackState(uint8 cabinPlaybackState)
{
   _cabinSourcePlaybackState = cabinPlaybackState;
   ETG_TRACE_USR1(("ExternalMediaSourceHandler::setCabinSrcPlaybackState: _cabinSourcePlaybackState %d", _cabinSourcePlaybackState));
}


int ExternalMediaSourceHandler::getCabinSrc()
{
   ETG_TRACE_USR1(("ExternalMediaSourceHandler::getCabinSrc: _cabinSource %d", _cabinSource));
   return _cabinSource;
}


void ExternalMediaSourceHandler::setCabinSrc(int cabinSrc)
{
   _cabinSource = cabinSrc;
   ETG_TRACE_USR1(("ExternalMediaSourceHandler::setCabinSrc: _cabinSource %d", _cabinSource));
}


void ExternalMediaSourceHandler::updateExternalSourceStateToObservers(int iSrcId, int iSinkId, int iConnectionState)
{
   int iSinkIdOfSrc = iSinkId;
   int connectionState = iConnectionState;
   ETG_TRACE_USR4(("ExternalMediaSourceHandler::updateExternalSourceStateToObservers iSrcId   %d    iSinkIdOfSrc  %d  connectionState %d ", iSrcId, iSinkIdOfSrc, connectionState));

   if (iSrcId != SRC_INVALID)
   {
      int prevStateOfExternalSource = getPreviousAudioStateOfExternalSource(iSrcId, iSinkIdOfSrc);
      ETG_TRACE_USR4(("ExternalMediaSourceHandler::updateExternalSourceStateToObservers prevStateOfExternalSource  %d", prevStateOfExternalSource));

      if (connectionState == STATE_CONNECTED  &&  prevStateOfExternalSource  != STATE_CONNECTED)
      {
         if (iSrcId == SRC_CAMERAPORT1_USB_VIDEO || iSrcId == SRC_CAMERAPORT1_USB_AUDIO)
         {
            updateCamportUsbSourceAllocObservors(iSrcId, iSinkIdOfSrc, STATE_CONNECTED);
         }
         else if (iSrcId == SRC_CMR1_USB1_VIDEO || iSrcId == SRC_CMR1_USB1_AUDIO)
         {
            updateRouterSourceAllocObservors(iSrcId, iSinkIdOfSrc, STATE_CONNECTED);
         }
         else if (iSrcId == SRC_CAMERAPORT1_HDMI || iSrcId == SRC_CAMERAPORT2_HDMI || iSrcId == SRC_CAMERAPORT3_HDMI)
         {
            updateHdmiSourceAllocObservors(iSrcId, iSinkIdOfSrc, STATE_CONNECTED);
         }
      }
      else if (connectionState == STATE_SUSPENDED && prevStateOfExternalSource  != STATE_SUSPENDED)
      {
         if (iSrcId == SRC_CAMERAPORT1_USB_VIDEO || iSrcId == SRC_CAMERAPORT1_USB_AUDIO)
         {
            updateCamportUsbSourceAllocObservors(iSrcId, iSinkIdOfSrc, STATE_SUSPENDED);
         }
         else if (iSrcId == SRC_CMR1_USB1_VIDEO || iSrcId == SRC_CMR1_USB1_AUDIO)
         {
            updateRouterSourceAllocObservors(iSrcId, iSinkIdOfSrc, STATE_SUSPENDED);
         }
      }
   }
}


bool ExternalMediaSourceHandler::isCamportUsbSourceAvailableInPrevSourceList(int sinkId)
{
   bool bSourceAvail = false;
   std::vector<sourceData>::const_iterator itr = m_prevAudioSrcList.begin();

   while (itr != m_prevAudioSrcList.end())
   {
      if (itr->getSrcId() == SRC_CAMERAPORT1_USB_VIDEO || itr->getSrcId() == SRC_CAMERAPORT1_USB_AUDIO)
      {
         if (itr->getSinkId() == sinkId)
         {
            m_disconnectedSource = itr->getSrcId();
            bSourceAvail = true;
         }
      }

      itr++;
   }

   ETG_TRACE_USR4(("ExternalMediaSourceHandler::isCamportUsbSourceAvailableInPrevSourceList CamportSourceAvail %d   sinkId  %d", bSourceAvail, sinkId));
   return bSourceAvail;
}


bool ExternalMediaSourceHandler::isRouterSourceAvailableInPrevSourceList(int sinkId)
{
   bool bSourceAvail = false;
   std::vector<sourceData>::const_iterator itr = m_prevAudioSrcList.begin();
   while (itr != m_prevAudioSrcList.end())
   {
      if (itr->getSrcId() == SRC_CMR1_USB1_VIDEO || itr->getSrcId() == SRC_CMR1_USB1_AUDIO)
      {
         if (itr->getSinkId() == sinkId)
         {
            m_disconnectedSource = itr->getSrcId();
            bSourceAvail = true;
         }
      }
      itr++;
   }
   ETG_TRACE_USR4(("ExternalMediaSourceHandler::isRouterSourceAvailableInPrevSourceList CamportSourceAvail %d   sinkId  %d", bSourceAvail, sinkId));
   return bSourceAvail;
}


bool ExternalMediaSourceHandler::isCamportHDMISourceAvailableInPrevSourceList(int sinkId)
{
   bool bSourceAvail = false;
   std::vector<sourceData>::const_iterator itr = m_prevAudioSrcList.begin();
   while (itr != m_prevAudioSrcList.end())
   {
      if (itr->getSrcId() == SRC_CAMERAPORT1_HDMI || itr->getSrcId() == SRC_CAMERAPORT2_HDMI || itr->getSrcId() == SRC_CAMERAPORT3_HDMI)
      {
         if (itr->getSinkId() == sinkId)
         {
            m_disconnectedSource = itr->getSrcId();
            bSourceAvail = true;
         }
      }
      itr++;
   }
   ETG_TRACE_USR4(("ExternalMediaSourceHandler::isCamportHDMISourceAvailableInPrevSourceList CamportHDMISourceAvail %d  m_disconnectedSource %d sinkId  %d", bSourceAvail, m_disconnectedSource, sinkId));
   return bSourceAvail;
}


void ExternalMediaSourceHandler::checkExternalSourceDisconnected()
{
   if (!isCamportUsbSourceAvailableInCabinSink(CABIN_A_SINK_ID))
   {
      bool isCamportUSBSrcAvailBefore = isCamportUsbSourceAvailableInPrevSourceList(CABIN_A_SINK_ID);
      ETG_TRACE_USR4(("ExternalMediaSourceHandler::checkExternalSourceDisconnected CABIN A isCamportUSBSrcAvailBefore %d  ", isCamportUSBSrcAvailBefore));
      if (isCamportUSBSrcAvailBefore)
      {
         updateCamportUsbSourceAllocObservors(m_disconnectedSource, CABIN_A_SINK_ID, STATE_DISCONNECTED);
         MediaSourceHandling::getInstance().vResetAudioSrcMapofCabin(m_disconnectedSource);
      }
   }

   if (!isCamportUsbSourceAvailableInCabinSink(CABIN_B_SINK_ID))
   {
      bool isCamportUSBSrcAvailBefore = isCamportUsbSourceAvailableInPrevSourceList(CABIN_B_SINK_ID);
      ETG_TRACE_USR4(("ExternalMediaSourceHandler::checkExternalSourceDisconnected CABIN B isCamportUSBSrcAvailBefore %d  ", isCamportUSBSrcAvailBefore));
      if (isCamportUSBSrcAvailBefore)
      {
         updateCamportUsbSourceAllocObservors(m_disconnectedSource, CABIN_B_SINK_ID, STATE_DISCONNECTED);
         MediaSourceHandling::getInstance().vResetAudioSrcMapofCabin(m_disconnectedSource);
      }
   }
   if (!isRouterSourceAvailableInCabinSink(CABIN_A_SINK_ID))
   {
      bool isRouterSrcAvailBefore = isRouterSourceAvailableInPrevSourceList(CABIN_A_SINK_ID);
      ETG_TRACE_USR4(("ExternalMediaSourceHandler::checkExternalSourceDisconnected CABIN A isRouterSrcAvailBefore %d  ", isRouterSrcAvailBefore));
      if (isRouterSrcAvailBefore)
      {
         updateRouterSourceAllocObservors(m_disconnectedSource, CABIN_A_SINK_ID, STATE_DISCONNECTED);
         MediaSourceHandling::getInstance().vResetAudioSrcMapofCabin(m_disconnectedSource);
      }
   }
   if (!isRouterSourceAvailableInCabinSink(CABIN_B_SINK_ID))
   {
      bool isRouterSrcAvailBefore = isRouterSourceAvailableInPrevSourceList(CABIN_B_SINK_ID);
      ETG_TRACE_USR4(("ExternalMediaSourceHandler::checkExternalSourceDisconnected CABIN B isRouterSrcAvailBefore %d  ", isRouterSrcAvailBefore));
      if (isRouterSrcAvailBefore)
      {
         updateRouterSourceAllocObservors(m_disconnectedSource, CABIN_B_SINK_ID, STATE_DISCONNECTED);
         MediaSourceHandling::getInstance().vResetAudioSrcMapofCabin(m_disconnectedSource);
      }
   }
   if (isCamportHDMISourceAvailableInPrevSourceList(CABIN_A_SINK_ID))
   {
      bool isCamportHDMISrcAvail = isCamportHDMISourceAvailableInCabinSink(m_disconnectedSource, CABIN_A_SINK_ID);
      ETG_TRACE_USR4(("ExternalMediaSourceHandler::checkExternalSourceDisconnected CABIN A isCamportHDMISrcAvail %d  ", isCamportHDMISrcAvail));
      if (!isCamportHDMISrcAvail)
      {
         updateHdmiSourceAllocObservors(m_disconnectedSource, CABIN_A_SINK_ID, STATE_DISCONNECTED);
         MediaSourceHandling::getInstance().vResetAudioSrcMapofCabin(m_disconnectedSource);
      }
   }
   if (isCamportHDMISourceAvailableInPrevSourceList(CABIN_B_SINK_ID))
   {
      bool isCamportHDMISrcAvail = isCamportHDMISourceAvailableInCabinSink(m_disconnectedSource, CABIN_B_SINK_ID);
      ETG_TRACE_USR4(("ExternalMediaSourceHandler::checkExternalSourceDisconnected CABIN B isCamportHDMISrcAvailBefore %d  ", isCamportHDMISrcAvail));
      if (!isCamportHDMISrcAvail)
      {
         updateHdmiSourceAllocObservors(m_disconnectedSource, CABIN_B_SINK_ID, STATE_DISCONNECTED);
         MediaSourceHandling::getInstance().vResetAudioSrcMapofCabin(m_disconnectedSource);
      }
   }
   m_disconnectedSource = SRC_INVALID;

   ETG_TRACE_USR1(("ExternalMediaSourceHandler::checkExternalSourceDisconnected Exited"));
}


void ExternalMediaSourceHandler::addHdmiSourceAllocInfoObservor(IHdmiSourceAllocationObservor& imp)
{
   ETG_TRACE_USR1(("ExternalMediaSourceHandler::addHdmiSourceAllocInfoObservor entered"));
   ::std::vector< IHdmiSourceAllocationObservor* >::const_iterator itr = ::std::find(m_hdmiSourceAllocObservor.begin(), m_hdmiSourceAllocObservor.end(), (&imp));

   if (itr == m_hdmiSourceAllocObservor.end())
   {
      m_hdmiSourceAllocObservor.push_back((&imp));
   }
}


void ExternalMediaSourceHandler::removeHdmiSourceAllocInfoObservor(IHdmiSourceAllocationObservor& imp)
{
   ETG_TRACE_USR1(("ExternalMediaSourceHandler::removeHdmiSourceAllocInfoObservor entered"));
   ::std::vector< IHdmiSourceAllocationObservor* >::iterator itr = ::std::find(m_hdmiSourceAllocObservor.begin(), m_hdmiSourceAllocObservor.end(), (&imp));

   if (itr != m_hdmiSourceAllocObservor.end())
   {
      (void)m_hdmiSourceAllocObservor.erase(itr);
   }
}


void ExternalMediaSourceHandler::updateHdmiSourceAllocObservors(int iSrcId, int iSinkId, int connectionState)
{
   ETG_TRACE_USR4(("ExternalMediaSourceHandler::updateHdmiSourceAllocObservors iSrcId  %d  iSinkId %d   connectionState  %d", iSrcId, iSinkId, connectionState));
   for (::std::vector< IHdmiSourceAllocationObservor* >::const_iterator itr = m_hdmiSourceAllocObservor.begin(); (itr != m_hdmiSourceAllocObservor.end()); ++itr)
   {
      if (NULL != (*itr))
      {
         (*itr)->onHdmiSourceAllocationUpdate(iSrcId, iSinkId, connectionState);
      }
   }
}


/*bool ExternalMediaSourceHandler::getCurrentHdmiActiveState()
{
   ETG_TRACE_USR4(("ExternalMediaSourceHandler::getCurrentHdmiActiveState() m_bCabinHdmiActiveState  %d", m_bCabinHdmiActiveState));
   return m_bCabinHdmiActiveState;
}*/


/*int ExternalMediaSourceHandler::getConnectionStateofExternalSource()
{
   int iConnectionState = STATE_UNKNOWN;

   if (!m_activeSourceInfo.empty())
   {
      std::map<int, stActiveSourceInfo>::iterator itr = m_activeSourceInfo.begin();

      for (; itr != m_activeSourceInfo.end(); ++itr)
      {
         if (itr->first == m_ActiveCamportUSBSource)
         {
            iConnectionState = itr->second.connectionState;
            break;
         }
      }
   }
   ETG_TRACE_USR4(("ExternalMediaSourceHandler::iConnectionState iConnectionState %d   m_ActiveCamportUSBSource  %d", iConnectionState, m_ActiveCamportUSBSource));
   return iConnectionState;
}


int ExternalMediaSourceHandler::getSinkId()
{
   int iSinkId = SINK_INVALID;
   if (!m_activeSourceInfo.empty())
   {
      std::map<int, stActiveSourceInfo>::iterator itr = m_activeSourceInfo.begin();

      for (; itr != m_activeSourceInfo.end(); ++itr)
      {
         if (itr->first == m_ActiveCamportUSBSource)
         {
            iSinkId = itr->second.sinkId;
            break;
         }
      }
   }
   ETG_TRACE_USR4(("ExternalMediaSourceHandler::getSinkId iSinkId %d m_ActiveCamportUSBSource %d", iSinkId, m_ActiveCamportUSBSource));
   return iSinkId;
}*/


int ExternalMediaSourceHandler::getPreviousAudioStateOfExternalSource(int iSrcId, int iSinkId)
{
   ETG_TRACE_USR4(("ExternalMediaSourceHandler::getPreviousAudioStateOfExternalSource() iSrcId %d  iSinkId %d   ", iSrcId, iSinkId));
   int iPrevSrcConnectionState = STATE_UNKNOWN;
   std::vector<sourceData>::const_iterator prevIter = m_prevAudioSrcList.begin();

   while (prevIter != m_prevAudioSrcList.end())
   {
      if (prevIter->getSrcId() == iSrcId && prevIter->getSinkId() == iSinkId)
      {
         iPrevSrcConnectionState = prevIter->getConnectionState();
         break;
      }
      prevIter++;
   }
   ETG_TRACE_USR4(("MediaSourceHandling::getPreviousAudioStateOfExternalSource() iPrevSrcConnectionState  %d", iPrevSrcConnectionState));
   return iPrevSrcConnectionState;
}


void ExternalMediaSourceHandler::updateCamportUsbSourceAllocObservors(int iSrcId, int iSinkId, int connectionState)
{
   ETG_TRACE_USR4(("ExternalMediaSourceHandler::updateCamportUsbSourceAllocObservors iSrcId  %d  iSinkId %d   connectionState  %d", iSrcId, iSinkId, connectionState));
   for (::std::vector< ICamportUsbSourceAllocationObservor* >::const_iterator itr = m_camportUsbSourceAllocationObservor.begin(); (itr != m_camportUsbSourceAllocationObservor.end()); ++itr)
   {
      if (NULL != (*itr))
      {
         (*itr)->onCamportUsbSourceAllocationUpdate(iSrcId, iSinkId, connectionState);
      }
   }
}


void ExternalMediaSourceHandler::updateRouterSourceAllocObservors(int iSrcId, int iSinkId, int connectionState)
{
   ETG_TRACE_USR4(("ExternalMediaSourceHandler::updateRouterSourceAllocObservors iSrcId  %d  iSinkId %d   connectionState  %d", iSrcId, iSinkId, connectionState));
   for (::std::vector< IRouterSourceAllocationObservor* >::const_iterator itr = m_routerSourceAllocObservor.begin(); (itr != m_routerSourceAllocObservor.end()); ++itr)
   {
      if (NULL != (*itr))
      {
         (*itr)->onRouterSourceAllocationUpdate(iSrcId, iSinkId, connectionState);
      }
   }
}


void ExternalMediaSourceHandler::addCamportUsbSourceAllocObservor(ICamportUsbSourceAllocationObservor& imp)
{
   ETG_TRACE_USR1(("ExternalMediaSourceHandler::addCamportUsbSourceAllocObservor entered"));
   ::std::vector< ICamportUsbSourceAllocationObservor* >::const_iterator itr = ::std::find(m_camportUsbSourceAllocationObservor.begin(), m_camportUsbSourceAllocationObservor.end(), (&imp));
   if (itr == m_camportUsbSourceAllocationObservor.end())
   {
      m_camportUsbSourceAllocationObservor.push_back((&imp));
   }
}


void ExternalMediaSourceHandler::removeCamportUsbSourceAllocObservor(ICamportUsbSourceAllocationObservor& imp)
{
   ETG_TRACE_USR1(("ExternalMediaSourceHandler::removeCamportUsbSourceAllocObservor entered"));
   ::std::vector< ICamportUsbSourceAllocationObservor* >::iterator itr = ::std::find(m_camportUsbSourceAllocationObservor.begin(), m_camportUsbSourceAllocationObservor.end(), (&imp));
   if (itr != m_camportUsbSourceAllocationObservor.end())
   {
      (void)m_camportUsbSourceAllocationObservor.erase(itr);
   }
}


void ExternalMediaSourceHandler::addRouterSourceAllocationInfoObservor(IRouterSourceAllocationObservor& imp)
{
   ETG_TRACE_USR1(("ExternalMediaSourceHandler::addRouterSourceAllocationInfoObservor entered"));
   ::std::vector< IRouterSourceAllocationObservor* >::const_iterator itr = ::std::find(m_routerSourceAllocObservor.begin(), m_routerSourceAllocObservor.end(), (&imp));
   if (itr == m_routerSourceAllocObservor.end())
   {
      m_routerSourceAllocObservor.push_back((&imp));
   }
}


void ExternalMediaSourceHandler::removeRouterSourceAllocationInfoObservor(IRouterSourceAllocationObservor& imp)
{
   ETG_TRACE_USR1(("ExternalMediaSourceHandler::removeRouterSourceAllocationInfoObservor entered"));
   ::std::vector< IRouterSourceAllocationObservor* >::iterator itr = ::std::find(m_routerSourceAllocObservor.begin(), m_routerSourceAllocObservor.end(), (&imp));
   if (itr != m_routerSourceAllocObservor.end())
   {
      (void)m_routerSourceAllocObservor.erase(itr);
   }
}


bool ExternalMediaSourceHandler::onCourierMessage(const ExternalCamportVideoSourceAvailabilityMsg& oMsg)
{
   ETG_TRACE_USR4(("ExternalMediaSourceHandler::ExternalCamportVideoSourceAvailabilityMsg entered   %d", oMsg.GetConnectionStatus()));
   updateCamportUsbSourceAvailabilityObservors(oMsg.GetConnectionStatus());
   return true;
}


bool ExternalMediaSourceHandler::onCourierMessage(const ExternalCamportAudioSourceAvailabilityMsg& oMsg)
{
   ETG_TRACE_USR4(("ExternalMediaSourceHandler::ExternalCamportAudioSourceAvailabilityMsg entered   %d", oMsg.GetConnectionStatus()));
   updateCamportUsbSourceAvailabilityObservors(oMsg.GetConnectionStatus());
   return true;
}


bool ExternalMediaSourceHandler::onCourierMessage(const ExternalHDMISourceAvailabilityMsg& oMsg)
{
   ETG_TRACE_USR4(("ExternalMediaSourceHandler::ExternalHDMISourceAvailabilityMsg entered   %d", oMsg.GetConnectionStatus()));
   updateHDMISourceAvailabilityObservors(oMsg.GetSrcId(), oMsg.GetConnectionStatus());
   return true;
}


bool ExternalMediaSourceHandler::onCourierMessage(const SourceAllocationMsg& oMsg)
{
   ETG_TRACE_USR4(("ExternalMediaSourceHandler::ExternalCamportVideoSourceAvailabilityMsg entered  srcId %d   sinkId  %d  Connectionstate  %d", oMsg.GetSrcId(), oMsg.GetSinkId(), oMsg.GetConnectionState()));
   updateCamportUsbSourceAllocObservors(oMsg.GetSrcId(), oMsg.GetSinkId(), oMsg.GetConnectionState());
   return true;
}


void ExternalMediaSourceHandler::addHdmiSourceAvailabilityInfoObservor(IhdmiSourceAvailabilityObservor& imp)
{
   std::vector< IhdmiSourceAvailabilityObservor* >::const_iterator itr = std::find(m_hdmiSourceAvailObservor.begin(), m_hdmiSourceAvailObservor.end(), (&imp));
   if (itr == m_hdmiSourceAvailObservor.end())
   {
      m_hdmiSourceAvailObservor.push_back((&imp));
   }
}


void ExternalMediaSourceHandler::removeHdmiSourceAvailabilityInfoObservor(IhdmiSourceAvailabilityObservor& imp)
{
   std::vector< IhdmiSourceAvailabilityObservor* >::iterator itr = std::find(m_hdmiSourceAvailObservor.begin(), m_hdmiSourceAvailObservor.end(), (&imp));
   if (itr != m_hdmiSourceAvailObservor.end())
   {
      (void)m_hdmiSourceAvailObservor.erase(itr);
   }
}


void ExternalMediaSourceHandler::addRouterSourceAvailabilityInfoObservor(IRouterSourceAvailabilityObservor& imp)
{
   std::vector< IRouterSourceAvailabilityObservor* >::const_iterator itr = std::find(m_routerSourceAvailObservor.begin(), m_routerSourceAvailObservor.end(), (&imp));
   if (itr == m_routerSourceAvailObservor.end())
   {
      m_routerSourceAvailObservor.push_back((&imp));
   }
}


void ExternalMediaSourceHandler::removeRouterSourceAvailabilityInfoObservor(IRouterSourceAvailabilityObservor& imp)
{
   std::vector< IRouterSourceAvailabilityObservor* >::iterator itr = std::find(m_routerSourceAvailObservor.begin(), m_routerSourceAvailObservor.end(), (&imp));
   if (itr != m_routerSourceAvailObservor.end())
   {
      (void)m_routerSourceAvailObservor.erase(itr);
   }
}


void ExternalMediaSourceHandler::updateRouterUsbConnectionState(int usbConnected)
{
   ETG_TRACE_USR4(("ExternalMediaSourceHandler::updateRouterUsbConnectionState usbConnected %d", usbConnected));
   for (::std::vector< IRouterSourceAvailabilityObservor* >::const_iterator itr = m_routerSourceAvailObservor.begin(); (itr != m_routerSourceAvailObservor.end()); ++itr)
   {
      if (NULL != (*itr))
      {
         (*itr)->onRouterSourceAvailabilityUpdate(usbConnected);
      }
   }
}


bool ExternalMediaSourceHandler::onCourierMessage(const ExternalRouterSourceAvailabilityMsg& oMsg)
{
   ETG_TRACE_USR4(("ExternalMediaClientHandler::ExternalRouterSourceAvailabilityMsg entered   %d", oMsg.GetConnectionStatus()));
   updateRouterUsbConnectionState(oMsg.GetConnectionStatus());
   return true;
}


int ExternalMediaSourceHandler::getCabinSrcID(EnumConst::entSourceType sourceType)
{
   int currentSourceid = SRC_INVALID;

   if (sourceType == EnumConst::entSourceType_CMP)
   {
      currentSourceid = SRC_CAMERAPORT1_USB_VIDEO;
   }
   else if (sourceType == EnumConst::entSourceType_CMR)
   {
      currentSourceid = SRC_CMR1_USB1_VIDEO;
   }
   ETG_TRACE_USR4(("ExternalMediaClientHandler::getCabinSrcID currentSourceid  %d", currentSourceid));
   return currentSourceid;
}


void ExternalMediaSourceHandler::updateCamportSourceConnectionState(int srcId, int connectionState)
{
   if (srcId == SRC_CAMERAPORT1_USB_VIDEO || srcId == SRC_CAMERAPORT1_USB_AUDIO)
   {
      if (connectionState == EN_EXTERNAL_MEDIA_CONNECTED || connectionState == EN_EXTERNAL_MEDIA_DISCONNECTED)//connected/disconnected
      {
         updateCamportUsbSourceAvailabilityObservors(connectionState);
      }
   }
   else if (srcId == SRC_CAMERAPORT1_HDMI || srcId == SRC_CAMERAPORT2_HDMI || srcId == SRC_CAMERAPORT3_HDMI)
   {
      if (connectionState == EN_EXTERNAL_MEDIA_CONNECTED || connectionState == EN_EXTERNAL_MEDIA_DISCONNECTED)
      {
         updateHDMISourceAvailabilityObservors(srcId, connectionState);
      }
   }
}


#endif
}// namespace Core
}// namespace App
