/**************************************************************************************
* @file         : ExternalMediaSourceConnectionStateHandler.cpp
* @author       : RBEI/ECG5-INF4CV_MediaTeam
* @addtogroup   : AppHmi_media
* @brief        :
* @copyright    : (c) 2018-2020 Robert Bosch Car Multimedia GmbH
*                 The reproduction, distribution and utilization of this file as
*                 well as the communication of its contents to others without express
*                 authorization is prohibited. Offenders will be held liable for the
*                 payment of damages. All rights reserved in the event of the grant
*                 of a patent, utility model or design.
**************************************************************************************/

#include "hall_std_if.h"
#include "AppHmi_MasterBase/AudioInterface/AudioDefines.h"
#include "Core/ExternalMediaSourceConnectionStateHandler/ExternalMediaSourceConnectionStateHandler.h"

#ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
#define ETG_DEFAULT_TRACE_CLASS           TR_CLASS_APPHMI_MEDIA_HALL
#define ETG_I_TRACE_CHANNEL               TR_TTFIS_APPHMI_MEDIA
#define ETG_I_TTFIS_CMD_PREFIX            "APPHMI_MEDIA_"
#define ETG_I_FILE_PREFIX                 App::Core::ExternalMediaSourceConnectionStateHandler::
#include "trcGenProj/Header/ExternalMediaSourceConnectionStateHandler.cpp.trc.h"
#endif

namespace App {
namespace Core {

/**
	* Description     : Constructor of class ExternalMediaSourceConnectionStateHandler
	*/
ExternalMediaSourceConnectionStateHandler::ExternalMediaSourceConnectionStateHandler(): _externalMediaSourceHandler(NULL),
   _cabinSystemStateHandler(NULL),
   _systemInfoMap(NULL),
   _externalMediaSourceConnectionStatus(NULL)
{
   ETG_TRACE_COMP(("ExternalMediaSourceConnectionStateHandler::ExternalMediaSourceConnectionStateHandler constructor \n"));
   _usbCamportAvailablility = EN_EXTERNAL_MEDIA_DISCONNECTED;
   _hdmiAvailability = EN_EXTERNAL_MEDIA_DISCONNECTED;
   _routerUSBAvailability = EN_EXTERNAL_MEDIA_DISCONNECTED;
   _hdmiDeviceInfo.clear();
   initialiseHdmiDevToSysCategoryMap();
}


ExternalMediaSourceConnectionStateHandler::~ExternalMediaSourceConnectionStateHandler()
{
   ETG_TRACE_COMP(("ExternalMediaSourceConnectionStateHandler::ExternalMediaSourceConnectionStateHandler Destructor \n"));
   _hdmiDeviceInfo.clear();
   if (_externalMediaSourceHandler != NULL)
   {
      _externalMediaSourceHandler->removeHdmiSourceAvailabilityInfoObservor((*this));
      _externalMediaSourceHandler->removeCamportUsbSourceAvailabilityObservor((*this));
      _externalMediaSourceHandler->removeRouterSourceAvailabilityInfoObservor((*this));
   }
   if (_cabinSystemStateHandler != NULL)
   {
      _cabinSystemStateHandler->removeCabinSystemInfoUpdateObserver((*this));
   }
}


void ExternalMediaSourceConnectionStateHandler::setExternalMediaSourceHandlerInstance(ExternalMediaSourceHandler* _externalMediaSourceHandlerInstance)
{
   if (_externalMediaSourceHandlerInstance != NULL)
   {
      _externalMediaSourceHandler = _externalMediaSourceHandlerInstance;
      _externalMediaSourceHandler->addHdmiSourceAvailabilityInfoObservor((*this));
      _externalMediaSourceHandler->addCamportUsbSourceAvailabilityObservor((*this));
      _externalMediaSourceHandler->addRouterSourceAvailabilityInfoObservor((*this));
   }
   else
   {
      ETG_TRACE_ERR(("ExternalMediaSourceConnectionStateHandler::_externalMediaSourceHandler is NULL"));
   }
}


void ExternalMediaSourceConnectionStateHandler::setCabinSystemStateHandlerInstance(CabinSystemStateHandler* _cabinSystemStateHandlerInstance)
{
   if (_cabinSystemStateHandlerInstance != NULL)
   {
      _cabinSystemStateHandler = _cabinSystemStateHandlerInstance;
      _cabinSystemStateHandler->addCabinSystemInfoUpdateObserver((*this));
   }
   else
   {
      ETG_TRACE_ERR(("ExternalMediaSourceConnectionStateHandler::_cabinSystemStateHandler is NULL"));
   }
}


void ExternalMediaSourceConnectionStateHandler::setSystemInfoMapInstance(SystemInfoMap* _systemInfoMapInstance)
{
   if (_systemInfoMapInstance != NULL)
   {
      _systemInfoMap = _systemInfoMapInstance;
   }
   else
   {
      ETG_TRACE_ERR(("ExternalMediaSourceConnectionStateHandler::_systemInfoMapInstance is NULL"));
   }
}


void ExternalMediaSourceConnectionStateHandler::setExternalMediaSourceConnectionStatusInstance(ExternalMediaSourceConnectionStatus* _externalMediaSourceConnectionStatusInstance)
{
   if (_externalMediaSourceConnectionStatusInstance != NULL)
   {
      _externalMediaSourceConnectionStatus = _externalMediaSourceConnectionStatusInstance;
   }
   else
   {
      ETG_TRACE_USR4(("ExternalMediaSourceConnectionStateHandler::_externalMediaSourceConnectionStatusInstance is NULL"));
   }
}


void  ExternalMediaSourceConnectionStateHandler::addExternalSourceAvailabilityStatusObserver(IExternalSourceAvailabilityStatusObserver& imp)
{
   ETG_TRACE_COMP(("ExternalMediaSourceConnectionStateHandler::addExternalSourceAvailabilityStatusObserver is called"));

   std::vector< IExternalSourceAvailabilityStatusObserver* >::const_iterator itr = std::find(_externalSourceAvailabilityStatusObserver.begin(), _externalSourceAvailabilityStatusObserver.end(), (&imp));
   if (itr == _externalSourceAvailabilityStatusObserver.end())
   {
      _externalSourceAvailabilityStatusObserver.push_back((&imp));
   }
}


void  ExternalMediaSourceConnectionStateHandler::removeExternalSourceAvailabilityStatusObserver(IExternalSourceAvailabilityStatusObserver& imp)
{
   ETG_TRACE_COMP(("ExternalMediaSourceConnectionStateHandler::removeExternalSourceAvailabilityStatusObserver is called"));

   std::vector< IExternalSourceAvailabilityStatusObserver* >::iterator itr = std::find(_externalSourceAvailabilityStatusObserver.begin(), _externalSourceAvailabilityStatusObserver.end(), (&imp));
   if (itr != _externalSourceAvailabilityStatusObserver.end())
   {
      (void)_externalSourceAvailabilityStatusObserver.erase(itr);
   }
}


void ExternalMediaSourceConnectionStateHandler::initialiseHdmiDevToSysCategoryMap()
{
   ETG_TRACE_COMP(("ExternalMediaSourceConnectionStateHandler::initialiseHdmiDevToSysCategoryMap"));
   _hdmiDeviceInfo[SRC_CAMERAPORT1_HDMI].status = EN_EXTERNAL_MEDIA_DISCONNECTED ;
   _hdmiDeviceInfo[SRC_CAMERAPORT1_HDMI].systemCategory = SYS_CATEGORY_CMP1;
   _hdmiDeviceInfo[SRC_CAMERAPORT2_HDMI].status = EN_EXTERNAL_MEDIA_DISCONNECTED ;
   _hdmiDeviceInfo[SRC_CAMERAPORT2_HDMI].systemCategory = SYS_CATEGORY_CMP2;
   _hdmiDeviceInfo[SRC_CAMERAPORT3_HDMI].status = EN_EXTERNAL_MEDIA_DISCONNECTED ;
   _hdmiDeviceInfo[SRC_CAMERAPORT3_HDMI].systemCategory = SYS_CATEGORY_CMP3;
}


void ExternalMediaSourceConnectionStateHandler::onCamportUsbSourceAvailabilityUpdate(int usbCamportAvailable)
{
   ETG_TRACE_COMP(("ExternalMediaSourceConnectionStateHandler::onCamportUsbSourceAvailabilityUpdate usbCamportAvailable  %d", usbCamportAvailable));
   _usbCamportAvailablility = usbCamportAvailable;
   updateSourceAvailabilityStatusToMap();
}


void ExternalMediaSourceConnectionStateHandler::onHdmiSourceAvailabilityUpdate(int srcId, int hdmiavailable)
{
   ETG_TRACE_COMP(("ExternalMediaSourceConnectionStateHandler::onHdmiSourceAvailabilityUpdate srcId %d, hdmiavailable %d", srcId, hdmiavailable));
   std::map <int, HdmiSrcStatusInfo>::iterator itr = _hdmiDeviceInfo.find(srcId);
   if (itr != _hdmiDeviceInfo.end())
   {
      itr -> second.status = hdmiavailable;
   }
   updateSourceAvailabilityStatusToMap();
}


void ExternalMediaSourceConnectionStateHandler::onRouterSourceAvailabilityUpdate(int routerSrcAvailable)
{
   ETG_TRACE_COMP(("ExternalMediaSourceConnectionStateHandler::onRouterSourceAvailabilityUpdate routerSrcAvailable  %d", routerSrcAvailable));
   _routerUSBAvailability = routerSrcAvailable;
   updateSourceAvailabilityStatusToMap();
}


void ExternalMediaSourceConnectionStateHandler::onCabinSystemUpdate()
{
   ETG_TRACE_COMP(("ExternalMediaSourceConnectionStateHandler::onCabinSystemUpdate"));
   updateSourceAvailabilityStatusToMap();
}


void ExternalMediaSourceConnectionStateHandler::updateSourceAvailabilityStatusToMap()
{
   ETG_TRACE_COMP(("ExternalMediaSourceConnectionStateHandler::updateSourceAvailabilityStatusToMap"));
   if (_usbCamportAvailablility == EN_EXTERNAL_MEDIA_CONNECTED && (_systemInfoMap->getSystemStatus(SYS_CATEGORY_CMA) == SYSTEM_STATUS_CONNECTED) && (_systemInfoMap->getSystemStatus(SYS_CATEGORY_CMP1) == SYSTEM_STATUS_CONNECTED))
   {
      _externalMediaSourceConnectionStatus->setExternalSourceConnectionStatus(SRC_CAMERAPORT1_USB_VIDEO, true);
   }
   else
   {
      _externalMediaSourceConnectionStatus->setExternalSourceConnectionStatus(SRC_CAMERAPORT1_USB_VIDEO, false);
   }
   if ((_routerUSBAvailability == /*EN_ROUTER_USB_CONNECTED*/ EN_EXTERNAL_MEDIA_CONNECTED && (_systemInfoMap->getSystemStatus(SYS_CATEGORY_CMA) == SYSTEM_STATUS_CONNECTED) && (_systemInfoMap->getSystemStatus(SYS_CATEGORY_CMR) == SYSTEM_STATUS_CONNECTED)) || (_routerUSBAvailability != EN_EXTERNAL_MEDIA_CONNECTED))
   {
      _externalMediaSourceConnectionStatus->setExternalSourceConnectionStatus(SRC_CMR1_USB1_VIDEO, false); // Temporarily modified as part of 1606014 - for disabling Router USB
   }

   if (_hdmiDeviceInfo[SRC_CAMERAPORT1_HDMI].status != EN_EXTERNAL_MEDIA_DISCONNECTED && (_systemInfoMap->getSystemStatus(SYS_CATEGORY_CMA) == SYSTEM_STATUS_CONNECTED)
         && (_systemInfoMap->getSystemStatus(_hdmiDeviceInfo[SRC_CAMERAPORT1_HDMI].systemCategory) == SYSTEM_STATUS_CONNECTED))
   {
      _externalMediaSourceConnectionStatus->setExternalSourceConnectionStatus(SRC_CAMERAPORT1_HDMI, true);
   }
   else
   {
      _externalMediaSourceConnectionStatus->setExternalSourceConnectionStatus(SRC_CAMERAPORT1_HDMI, false);
   }
   if (_hdmiDeviceInfo[SRC_CAMERAPORT2_HDMI].status != EN_EXTERNAL_MEDIA_DISCONNECTED && (_systemInfoMap->getSystemStatus(SYS_CATEGORY_CMA) == SYSTEM_STATUS_CONNECTED)
         && (_systemInfoMap->getSystemStatus(_hdmiDeviceInfo[SRC_CAMERAPORT2_HDMI].systemCategory) == SYSTEM_STATUS_CONNECTED)
         && (_systemInfoMap->getSystemStatus(_hdmiDeviceInfo[SRC_CAMERAPORT1_HDMI].systemCategory) == SYSTEM_STATUS_CONNECTED))
   {
      _externalMediaSourceConnectionStatus->setExternalSourceConnectionStatus(SRC_CAMERAPORT2_HDMI, true);
   }
   else
   {
      _externalMediaSourceConnectionStatus->setExternalSourceConnectionStatus(SRC_CAMERAPORT2_HDMI, false);
   }
   if (_hdmiDeviceInfo[SRC_CAMERAPORT3_HDMI].status != EN_EXTERNAL_MEDIA_DISCONNECTED && (_systemInfoMap->getSystemStatus(SYS_CATEGORY_CMA) == SYSTEM_STATUS_CONNECTED)
         && (_systemInfoMap->getSystemStatus(_hdmiDeviceInfo[SRC_CAMERAPORT3_HDMI].systemCategory) == SYSTEM_STATUS_CONNECTED)
         && (_systemInfoMap->getSystemStatus(_hdmiDeviceInfo[SRC_CAMERAPORT1_HDMI].systemCategory) == SYSTEM_STATUS_CONNECTED))
   {
      _externalMediaSourceConnectionStatus->setExternalSourceConnectionStatus(SRC_CAMERAPORT3_HDMI, true);
   }
   else
   {
      _externalMediaSourceConnectionStatus->setExternalSourceConnectionStatus(SRC_CAMERAPORT3_HDMI, false);
   }
   notifyExternalSourceAvailabilityStatusListeners();
}


void ExternalMediaSourceConnectionStateHandler::notifyExternalSourceAvailabilityStatusListeners()
{
   ETG_TRACE_COMP(("ExternalMediaSourceConnectionStateHandler::notifyExternalSourceAvailabilityStatusListeners is called"));

   for (std::vector<IExternalSourceAvailabilityStatusObserver* >::const_iterator itr = _externalSourceAvailabilityStatusObserver.begin(); (itr != _externalSourceAvailabilityStatusObserver.end()); ++itr)
   {
      if (NULL != (*itr))
      {
         (*itr)->onExternalSourceAvailabilityUpdate();
      }
   }
}


}//end of namespace Core
}//end of namespace App
