/**************************************************************************************
 * @file         : ExternalMediaListManager.cpp
 * @author       : RBEI/ECH2-INF4CV_MediaTeam
 * @addtogroup   : AppHmi_media
 * @brief        :
 * @copyright    : (c) 2018-2020 Robert Bosch Car Multimedia GmbH
 *                 The reproduction, distribution and utilization of this file as
 *                 well as the communication of its contents to others without express
 *                 authorization is prohibited. Offenders will be held liable for the
 *                 payment of damages. All rights reserved in the event of the grant
 *                 of a patent, utility model or design.
 * 		  			Initial Revision.
 **************************************************************************************/

#include "hall_std_if.h"
#include "Core/ExternalMediaListManager/ExternalMediaListManager.h"
#include "Core/ListHandling/ListGuiRequestHandler/ListGuiRequestHandler.h"
#include "Core/SourceSwitch/MediaSourceHandling.h"
#include "AvdeccPluginConstant.h"
#include "AppHmi_MediaStateMachine.h"
#include "AppHmi_MediaDatabinding.h"
#include <vector>
#include <string>
#include "Core/ExternalMediaDefines.h"
#include <stdint.h>

#ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
#define ETG_DEFAULT_TRACE_CLASS           TR_CLASS_APPHMI_MEDIA_HALL
#define ETG_I_TRACE_CHANNEL               TR_TTFIS_APPHMI_MEDIA
#define ETG_I_TTFIS_CMD_PREFIX            "APPHMI_MEDIA_"
#define ETG_I_FILE_PREFIX                 App::Core::ExternalMediaListManager::
#include "trcGenProj/Header/ExternalMediaListManager.cpp.trc.h"
#endif


namespace App {
namespace Core {

ExternalMediaListManager::ExternalMediaListManager(ExternalMediaSourceHandler* _externalMediaSourceHandlerInstance, PlayableContentMap* _playableContentMapInstance)
   :  _verticalListManagercmp(NULL),
      _horizontalListManagercmp(NULL),
      _listGuiRequestHandlercmp(NULL),
      _verticalListManagercmr(NULL),
      _horizontalListManagercmr(NULL),
      _listGuiRequestHandlercmr(NULL),
      _playableContentMap(NULL),
      _externalMediaSourceHandler(NULL)
{
   ETG_TRACE_USR1(("ExternalMediaListManager constructor \n"));
   initializeInstances();
   _externalMediaSourceHandler = _externalMediaSourceHandlerInstance;
   _playableContentMap = _playableContentMapInstance;
   _currentSourceType = EnumConst::entSourceType_DEFAULT;

   ListRegistry::s_getInstance().addListImplementation(LISTID_EXTERNAL_MEDIA_HOR, this);
   ListRegistry::s_getInstance().addListImplementation(LIST_ID_EXTERNAL_MEDIA_VER, this);

   if (_externalMediaSourceHandler != NULL)
   {
      _externalMediaSourceHandler->addCamportUsbSourceAllocObservor((*this));
      _externalMediaSourceHandler->addRouterSourceAllocationInfoObservor((*this));
      _externalMediaSourceHandler->addCamportUsbSourceAvailabilityObservor((*this));
      _externalMediaSourceHandler->addRouterSourceAvailabilityInfoObservor((*this));
   }
}


ExternalMediaListManager::~ExternalMediaListManager()
{
   ETG_TRACE_USR1(("ExternalMediaListManager Destructor \n"));

   ListRegistry::s_getInstance().removeListImplementation(LISTID_EXTERNAL_MEDIA_HOR);
   ListRegistry::s_getInstance().removeListImplementation(LIST_ID_EXTERNAL_MEDIA_VER);

   if (_externalMediaSourceHandler != NULL)
   {
      _externalMediaSourceHandler->removeCamportUsbSourceAllocObservor((*this));
      _externalMediaSourceHandler->removeRouterSourceAllocationInfoObservor((*this));
      _externalMediaSourceHandler->removeCamportUsbSourceAvailabilityObservor((*this));
      _externalMediaSourceHandler->removeRouterSourceAvailabilityInfoObservor((*this));

      delete _externalMediaSourceHandler;
      _externalMediaSourceHandler = NULL;
   }
   if (_playableContentMap != NULL)
   {
      delete _playableContentMap;
      _playableContentMap = NULL;
   }
   if (_horizontalListManagercmp != NULL)
   {
      delete _horizontalListManagercmp;
      _horizontalListManagercmp = NULL;
   }
   if (_horizontalListManagercmr != NULL)
   {
      delete _horizontalListManagercmr;
      _horizontalListManagercmr = NULL;
   }
   if (_verticalListManagercmp != NULL)
   {
      delete _verticalListManagercmp;
      _verticalListManagercmp = NULL;
   }
   if (_verticalListManagercmr != NULL)
   {
      delete _verticalListManagercmr;
      _verticalListManagercmr = NULL;
   }
   if (_listGuiRequestHandlercmp != NULL)
   {
      delete _listGuiRequestHandlercmp;
      _listGuiRequestHandlercmp = NULL;
   }
   if (_listGuiRequestHandlercmr != NULL)
   {
      delete _listGuiRequestHandlercmr;
      _listGuiRequestHandlercmr = NULL;
   }
}


void ExternalMediaListManager::initializeInstances()
{
}


void ExternalMediaListManager::onCamportUsbSourceAllocationUpdate(int iSourceId, int iSinkId, int connectionState)
{
   ETG_TRACE_USR4(("ExternalMediaListManager::onCamportUsbSourceAllocationUpdate  iSinkId %d  iSourceId %d  connectionState %d ", iSinkId, iSourceId, connectionState));

   if (connectionState == STATE_CONNECTED)
   {
      _currentSourceType = EnumConst::entSourceType_CMP;

      if (NULL == _horizontalListManagercmp)
      {
         ETG_TRACE_USR1(("ExternalMediaListManager::onCamportUsbSourceAllocationUpdate is _horizontalListManagercmp"));
         _horizontalListManagercmp = new HorizontalListManager(_externalMediaSourceHandler, _currentSourceType, _playableContentMap);
      }
      if (NULL == _verticalListManagercmp)
      {
         ETG_TRACE_USR1(("ExternalMediaListManager::onCamportUsbSourceAllocationUpdate is _verticalListManagercmp"));
         _verticalListManagercmp = new VerticalListManager(_horizontalListManagercmp, _externalMediaSourceHandler, _currentSourceType);
      }
      if (NULL == _listGuiRequestHandlercmp)
      {
         _listGuiRequestHandlercmp = new ListGuiRequestHandler(_horizontalListManagercmp, _verticalListManagercmp);
      }
      if (NULL != _verticalListManagercmp)
      {
         _verticalListManagercmp->onSourceAllocationListUpdate();
      }
   }
}


void ExternalMediaListManager::onRouterSourceAllocationUpdate(int iSourceId, int iSinkId, int connectionState)
{
   ETG_TRACE_USR4(("ExternalMediaListManager::onRouterSourceAllocationUpdate  iSinkId %d  iSourceId %d  connectionState %d ", iSinkId, iSourceId, connectionState));

   if (connectionState == STATE_CONNECTED)
   {
      _currentSourceType = EnumConst::entSourceType_CMR;

      if (NULL == _horizontalListManagercmr)
      {
         _horizontalListManagercmr = new HorizontalListManager(_externalMediaSourceHandler, _currentSourceType, _playableContentMap);
      }
      if (NULL == _verticalListManagercmr)
      {
         _verticalListManagercmr = new VerticalListManager(_horizontalListManagercmr, _externalMediaSourceHandler, _currentSourceType);
      }
      if (NULL == _listGuiRequestHandlercmr)
      {
         _listGuiRequestHandlercmr = new ListGuiRequestHandler(_horizontalListManagercmr, _verticalListManagercmr);
      }
      if (NULL != _verticalListManagercmr)
      {
         _verticalListManagercmr->onSourceAllocationListUpdate();
      }
   }
}


void ExternalMediaListManager::onCamportUsbSourceAvailabilityUpdate(int iUSBCamportAvailability)
{
   ETG_TRACE_USR4(("ExternalMediaListManager::onCamportUsbSourceAvailabilityUpdate iUSBCamportAvailability  %d", iUSBCamportAvailability));

   if (iUSBCamportAvailability == EN_EXTERNAL_MEDIA_DISCONNECTED)
   {
      if (_verticalListManagercmp != NULL)
      {
         delete _verticalListManagercmp;
         _verticalListManagercmp = NULL;
      }
      if (_horizontalListManagercmp)
      {
         delete _horizontalListManagercmp;
         _horizontalListManagercmp = NULL;
      }
      if (_listGuiRequestHandlercmp != NULL)
      {
         delete _listGuiRequestHandlercmp;
         _listGuiRequestHandlercmp = NULL;
      }
   }
}


void ExternalMediaListManager::onRouterSourceAvailabilityUpdate(int usbAvailabilityInRouter)
{
   ETG_TRACE_USR4(("ExternalMediaListManager::onRouterSourceAvailabilityUpdate usbAvailabilityInRouter  %d", usbAvailabilityInRouter));
   if (usbAvailabilityInRouter == EN_EXTERNAL_MEDIA_DISCONNECTED)
   {
      if (_verticalListManagercmr != NULL)
      {
         delete _verticalListManagercmr;
         _verticalListManagercmr = NULL;
      }
      if (_horizontalListManagercmr != NULL)
      {
         delete _horizontalListManagercmr;
         _horizontalListManagercmr = NULL;
      }
      if (_listGuiRequestHandlercmr != NULL)
      {
         delete _listGuiRequestHandlercmr;
         _listGuiRequestHandlercmr = NULL;
      }
   }
}


/***********************************************************
* Function 		: getListDataProvider
* Description 	: Updates the List data Based on the LIST_IDs
* Parameters 	: ListDateProviderReqMsg &oMsg
* Return value	: tSharedPtrDataProvider
* ***********************************************************/
tSharedPtrDataProvider ExternalMediaListManager::getListDataProvider(const ListDateProviderReqMsg& oMsg)
{
   ETG_TRACE_USR4(("ExternalMediaListManager::getListDataProvider list id %u, startindex %d, count %u", oMsg.GetListId(), oMsg.GetStartIndex(), oMsg.GetWindowElementSize()));

   tSharedPtrDataProvider dataProvider;

   if (_currentSourceType == getCurrentActiveSourceType())
   {
      if (_currentSourceType == EnumConst::entSourceType_CMP && (_listGuiRequestHandlercmp))
      {
         _listGuiRequestHandlercmp->getListDataProvider(oMsg);
      }
      else if (_currentSourceType == EnumConst::entSourceType_CMR && (_listGuiRequestHandlercmr))
      {
         _listGuiRequestHandlercmr->getListDataProvider(oMsg);
      }
   }

   return dataProvider;
}


/* ***********************************************************
* Function 		: onCourierMessage
* Description 	: Courier message received when there is change in the list
* Parameters 	: oMsg
* Return value	: void
* ***********************************************************/
bool ExternalMediaListManager::onCourierMessage(const ListChangedUpdMsg& oMsg)
{
   ETG_TRACE_USR4(("ExternalMediaListManager::onCourierMessage:ListChangedUpdMsg listid:%d", oMsg.GetListId()));

   if (_currentSourceType == getCurrentActiveSourceType())
   {
      if (_currentSourceType == EnumConst::entSourceType_CMP && (_listGuiRequestHandlercmp))
      {
         _listGuiRequestHandlercmp->onCourierMessage(oMsg);
      }
      else if (_currentSourceType == EnumConst::entSourceType_CMR && (_listGuiRequestHandlercmr))
      {
         _listGuiRequestHandlercmr->onCourierMessage(oMsg);
      }
   }
   return false;
}


bool ExternalMediaListManager::onCourierMessage(const ExternalMediaListItemPressUpdMsg& oMsg)
{
   ETG_TRACE_USR4(("ExternalMediaListManager::ExternalMediaListItemPressUpdMsg Hdl: %d", oMsg.GetHdl()));

   if (_currentSourceType == getCurrentActiveSourceType())
   {
      if (_currentSourceType == EnumConst::entSourceType_CMP && (_listGuiRequestHandlercmp))
      {
         _listGuiRequestHandlercmp->onCourierMessage(oMsg);
      }
      else if (_currentSourceType == EnumConst::entSourceType_CMR && (_listGuiRequestHandlercmr))
      {
         _listGuiRequestHandlercmr->onCourierMessage(oMsg);
      }
   }
   return false;
}


bool ExternalMediaListManager::onCourierMessage(const ExternalMediaTabBtnPressUpdMsg& oMsg)
{
   ETG_TRACE_USR4(("ExternalMediaListManager::ExternalMediaTabBtnPressUpdMsg SelectMediaTab: %d", oMsg.GetSelectMediaTab()));

   if (_currentSourceType == getCurrentActiveSourceType())
   {
      if (_currentSourceType == EnumConst::entSourceType_CMP && (_listGuiRequestHandlercmp))
      {
         _listGuiRequestHandlercmp->onCourierMessage(oMsg);
      }
      else if (_currentSourceType == EnumConst::entSourceType_CMR && (_listGuiRequestHandlercmr))
      {
         _listGuiRequestHandlercmr->onCourierMessage(oMsg);
      }
   }

   return false;
}


bool ExternalMediaListManager::onCourierMessage(const ExternalMediaTabListPageUpReqMsg& oMsg)
{
   ETG_TRACE_USR1(("ExternalMediaListManager::ExternalMediaTabListPageUpReqMsg"));

   if (_currentSourceType == getCurrentActiveSourceType())
   {
      if (_currentSourceType == EnumConst::entSourceType_CMP && (_listGuiRequestHandlercmp))
      {
         _listGuiRequestHandlercmp->onCourierMessage(oMsg);
      }
      else if (_currentSourceType == EnumConst::entSourceType_CMR && (_listGuiRequestHandlercmr))
      {
         _listGuiRequestHandlercmr->onCourierMessage(oMsg);
      }
   }

   return false;
}


bool ExternalMediaListManager::onCourierMessage(const ExternalMediaTabListPageDownReqMsg& oMsg)
{
   ETG_TRACE_USR1(("ExternalMediaListManager::ExternalMediaTabListPageDownReqMsg:"));

   if (_currentSourceType == getCurrentActiveSourceType())
   {
      if (_currentSourceType == EnumConst::entSourceType_CMP && (_listGuiRequestHandlercmp))
      {
         _listGuiRequestHandlercmp->onCourierMessage(oMsg);
      }
      else if (_currentSourceType == EnumConst::entSourceType_CMR && (_listGuiRequestHandlercmr))
      {
         _listGuiRequestHandlercmr->onCourierMessage(oMsg);
      }
   }

   return false;
}


bool ExternalMediaListManager::onCourierMessage(const ExternalMediaBackButtonPressUpdMsg& oMsg)
{
   ETG_TRACE_USR1(("ExternalMediaListManager::ExternalMediaBackButtonPressUpdMsg"));

   if (_currentSourceType == getCurrentActiveSourceType())
   {
      if (_currentSourceType == EnumConst::entSourceType_CMP && (_listGuiRequestHandlercmp))
      {
         _listGuiRequestHandlercmp->onCourierMessage(oMsg);
      }
      else if (_currentSourceType == EnumConst::entSourceType_CMR && (_listGuiRequestHandlercmr))
      {
         _listGuiRequestHandlercmr->onCourierMessage(oMsg);
      }
   }

   return false;
}


bool ExternalMediaListManager::onCourierMessage(const ExternalBrowseScreenExitUpdMsg& oMsg)
{
   ETG_TRACE_COMP(("ExternalMediaListManager::ExternalBrowseScreenExitUpdMsg"));

   if (_currentSourceType == getCurrentActiveSourceType())
   {
      if (_currentSourceType == EnumConst::entSourceType_CMP && (_listGuiRequestHandlercmp))
      {
         _listGuiRequestHandlercmp->onCourierMessage(oMsg);
      }
      else if (_currentSourceType == EnumConst::entSourceType_CMR && (_listGuiRequestHandlercmr))
      {
         _listGuiRequestHandlercmr->onCourierMessage(oMsg);
      }
   }

   return false;
}


bool ExternalMediaListManager::onCourierMessage(const ExternalMediaAllButtonPressUpdMsg& oMsg)
{
   ETG_TRACE_USR1(("ExternalMediaListManager::ExternalMediaAllButtonPressUpdMsg"));

   if (_currentSourceType == getCurrentActiveSourceType())
   {
      if (_currentSourceType == EnumConst::entSourceType_CMP && (_listGuiRequestHandlercmp))
      {
         _listGuiRequestHandlercmp->onCourierMessage(oMsg);
      }
      else if (_currentSourceType == EnumConst::entSourceType_CMR && (_listGuiRequestHandlercmr))
      {
         _listGuiRequestHandlercmr->onCourierMessage(oMsg);
      }
   }

   return false;
}


bool ExternalMediaListManager::onCourierMessage(const CurrentMediaStateUpdMsg& msg)
{
   ETG_TRACE_USR1(("ExternalMediaListManager::CurrentMediaStateUpdMsg"));

   if (_currentSourceType == getCurrentActiveSourceType())
   {
      if (_currentSourceType == EnumConst::entSourceType_CMP && (_verticalListManagercmp))
      {
         _verticalListManagercmp->onCourierMessage(msg);
      }
      else if (_currentSourceType == EnumConst::entSourceType_CMR && (_verticalListManagercmr))
      {
         _verticalListManagercmr->onCourierMessage(msg);
      }
   }
   return false;
}


bool ExternalMediaListManager::onCourierMessage(const MockCreateListandSliceListResMsg& msg)
{
   ETG_TRACE_USR1(("ExternalMediaListManager::MockCreateListandSliceListResMsg"));
   if (_currentSourceType == getCurrentActiveSourceType())
   {
      if (_currentSourceType == EnumConst::entSourceType_CMP && (_verticalListManagercmp))
      {
         _verticalListManagercmp->onCourierMessage(msg);
      }
      else if (_currentSourceType == EnumConst::entSourceType_CMR && (_verticalListManagercmr))
      {
         _verticalListManagercmr->onCourierMessage(msg);
      }
   }
   return false;
}


int ExternalMediaListManager::getCurrentActiveSourceType()
{
   int currentActiveSrcId = MediaSourceHandling::getInstance().getCurrentRegionSrcId();
   int activeSourceType = EnumConst::entSourceType_DEFAULT;

   if (currentActiveSrcId == SRC_CAMERAPORT1_USB_VIDEO || currentActiveSrcId == SRC_CAMERAPORT1_USB_AUDIO)
   {
      activeSourceType =  EnumConst::entSourceType_CMP;
   }
   else if (currentActiveSrcId == SRC_CMR1_USB1_AUDIO || currentActiveSrcId == SRC_CMR1_USB1_VIDEO)
   {
      activeSourceType = EnumConst::entSourceType_CMR;
   }
   ETG_TRACE_USR4(("ExternalMediaListManager::ExternalMediaAllButtonPressUpdMsg  currentActiveSrcId  %d   activeSourceType   %d", currentActiveSrcId, activeSourceType));

   return activeSourceType;
}


}// namespace Core
}// namespace App
