/**************************************************************************************
 * @file         : ExternalMediaClientHandler.cpp
 * @author       : RBEI/ECG5-INF4CV_MediaTeam
 * @addtogroup   : AppHmi_media
 * @brief        :
 * @copyright    : (c) 2018-2020 Robert Bosch Car Multimedia GmbH
 *                 The reproduction, distribution and utilization of this file as
 *                 well as the communication of its contents to others without express
 *                 authorization is prohibited. Offenders will be held liable for the
 *                 payment of damages. All rights reserved in the event of the grant
 *                 of a patent, utility model or design.
 *04.09.18 Rev. 0.1 Ruchi/SUMAN (RBEI/ECH2)
 * 		  			Initial Revision.
 **************************************************************************************/

#include "hall_std_if.h"
#include "Core/ExternalMediaClientHandler/ExternalMediaClientHandler.h"
#include "Core/Utils/MediaUtils.h"
#include "Core/MediaDefines.h"
#include "../ApplicationSwitchClientHandler/ApplicationSwitchClientHandler.h"
#include "Core/MediaSourceListHandler/MediaSourceListHandler.h"
#include "PluginConstants.h"
#include "MediaCameraPort.h"
#include "MediaRouter.h"
#include "Core/MediaClientHandler/MediaClientHandler.h"
#include "Core/StateTransitionHandler/StateTransitionHandler.h"
#include "Core/Utils/DbDownloadMap.h"
#include "App/Core/HmiTranslation_TextIds.h"
#include "App/datapool/MediaDataPoolConfig.h"

#include "EMPData.h"
#include <bits/stdc++.h>
#ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
#define ETG_DEFAULT_TRACE_CLASS           TR_CLASS_APPHMI_MEDIA_HALL
#define ETG_I_TRACE_CHANNEL               TR_TTFIS_APPHMI_MEDIA
#define ETG_I_TTFIS_CMD_PREFIX            "APPHMI_MEDIA_"
#define ETG_I_FILE_PREFIX                 App::Core::ExternalMediaClientHandler::
#include "trcGenProj/Header/ExternalMediaClientHandler.cpp.trc.h"
#endif

#define CTRLBLOCK_NAME_MEDIA_SIGNAL_SOURCE_CHANGE "SignalSourceChange"
#define CTRLBLOCK_ROUTER_SIGNAL_SOURCE_CHANGE "RouterSignalSourceChange"

#ifdef DP_DATAPOOL_ID
#define DP_S_IMPORT_INTERFACE_FI
#include "dp_generic_if.h"
#endif

using namespace ::bosch::cm::di::HmiEmpListInterface;

namespace App {
namespace Core {
DEFINE_CLASS_LOGGER_AND_LEVEL("App/Core/ExternalMediaClientHandler", ExternalMediaClientHandler, Info);


/**
	* Description     : Destructor of class ExternalMediaClientHandler
	*/
ExternalMediaClientHandler::~ExternalMediaClientHandler()
{
   ETG_I_UNREGISTER_FILE();
   m_u8CabinAReg = 0;
   m_u8CabinBReg = 0;
   m_isRouterSourceActive = false;
   if (_externalMediaSourceHandler != NULL)
   {
      _externalMediaSourceHandler->removeHdmiSourceAvailabilityInfoObservor((*this));
      _externalMediaSourceHandler->removeCamportUsbSourceAllocObservor((*this));
      _externalMediaSourceHandler->removeCamportUsbSourceAvailabilityObservor((*this));
      _externalMediaSourceHandler->removeRouterSourceAllocationInfoObservor((*this));
      _externalMediaSourceHandler->removeRouterSourceAvailabilityInfoObservor((*this));
      _externalMediaSourceHandler->removeHdmiSourceAllocInfoObservor((*this));
   }

   if (_externalMediaListDBManager != NULL)
   {
      _externalMediaListDBManager->deregisterExternalDBDownloadObservor((*this));
   }
   if (_hmiStateHandler != NULL)
   {
      _hmiStateHandler->removeHmiStateStatusObserver((*this));
   }
   clearInstance();
   _hdmiSrcAvailability.clear();
   _hdmiDevTypSrcMap.clear();
}


/**
	* Description     : Constructor of class ExternalMediaClientHandler
	*/
ExternalMediaClientHandler::ExternalMediaClientHandler()
   : _hmiEmpListInterfaceProxy(HmiEmpListInterfaceProxy::createProxy("HmiEmpListInterfacePort", *this))
   , _externalMediaListDBManager(NULL)
   , _playableContentMap(NULL)
   , _systemInfoMap(NULL)
   , _playableContentInterfaceHandler(NULL)
   , _hmiStateHandler(NULL)
{
   StartupSync::getInstance().registerPropertyRegistrationIF(this, "HmiEmpListInterfacePort");
   _camportUSBSourceActivationHandler = ExternalMediaSourceActivationFactory::getExternalActivationSourceHandlerObject(FI_EN_E8DTY_USB_CAMERAPORT);
   _camportRouterSourceActivationHandler = ExternalMediaSourceActivationFactory::getExternalActivationSourceHandlerObject(FI_EN_E8DTY_ROUTER_USB);
   _camportUSBPluginClientHandler = PluginClientFactory::getPluginClient(SRC_CAMERAPORT1_USB_VIDEO);
   _routerPluginClientHandler = PluginClientFactory::getPluginClient(SRC_CMR1_USB1_VIDEO);
   _externalMediaSourceHandler = NULL;
   _hdmiSrcAvailability.clear();
   _hdmiDevTypSrcMap.clear();
   initialiseHdmiSrcAvailabilityMap();
   initialiseHdmiDeviceNumberMap();
   m_iUsbCamportAvailable = EN_EXTERNAL_MEDIA_DISCONNECTED;
   m_isRouterUsbConnected = EN_EXTERNAL_MEDIA_DISCONNECTED ;
   _currentMediaState = MEDIA_STATE_INVALID;
   m_u8CabinAReg = 0;
   m_u8CabinBReg = 0;

   u16ControlVal = 0;
   name = " ";
   m_u8CabinARegHDMI = 0;
   m_u8CabinBRegHDMI = 0;
   m_u8ActiveRegion = 2;

   m_bGlassAvailable = false;
   m_isRouterSourceActive = false;
   m_currentTrackIndex = 0;
   m_isCamportUSBSourceActive = false;
   _isHmiOffWithActiveMediaSrc = false;
   _canSigFFReqOngoing = false;
   _canSigFRReqOngoing = false;
   ETG_I_REGISTER_FILE();
}


void ExternalMediaClientHandler::clearInstance()
{
   ETG_TRACE_USR1(("ExternalMediaClientHandler clearInstance \n"));
   if (NULL != _externalMediaListDBManager)
   {
      delete _externalMediaListDBManager;
      _externalMediaListDBManager = NULL;
   }
   if (NULL != _playableContentInterfaceHandler)
   {
      delete _playableContentInterfaceHandler;
      _playableContentInterfaceHandler = NULL;
   }
}


void ExternalMediaClientHandler::initialiseHdmiSrcAvailabilityMap()
{
   ETG_TRACE_USR1(("MediaSourceHandling::initialiseHdmiSrcAvailabilityMap is entered \n"));

   _hdmiSrcAvailability[SRC_CAMERAPORT1_HDMI] = HdmiSourceInfo(EN_EXTERNAL_MEDIA_DISCONNECTED, false);
   _hdmiSrcAvailability[SRC_CAMERAPORT2_HDMI] = HdmiSourceInfo(EN_EXTERNAL_MEDIA_DISCONNECTED, false);
   _hdmiSrcAvailability[SRC_CAMERAPORT3_HDMI] = HdmiSourceInfo(EN_EXTERNAL_MEDIA_DISCONNECTED, false);
}


void ExternalMediaClientHandler::initialiseHdmiDeviceNumberMap()
{
   ETG_TRACE_USR1(("MediaSourceHandling::initialiseHdmiSrcAvailabilityMap is entered \n"));
   _hdmiDevTypSrcMap.insert(pair<uint8, int>(FI_EN_E8DTY_USB_CAMERAPORT_HDMI, SRC_CAMERAPORT1_HDMI));
   _hdmiDevTypSrcMap.insert(pair<uint8, int>(FI_EN_E8DTY_USB_CAMERAPORT_HDMI_2, SRC_CAMERAPORT2_HDMI));
   _hdmiDevTypSrcMap.insert(pair<uint8, int>(FI_EN_E8DTY_USB_CAMERAPORT_HDMI_3, SRC_CAMERAPORT3_HDMI));
}


/**
	* Description     : Request from Framework to start registering for properties. This request means GUI is ready now.
	*
	* @param[in]      : proxy : reference to the base class of all proxies
	* @param[in]      : stateChange : represents a state transition of a service
	* @return         : void
	*/
void ExternalMediaClientHandler::registerProperties(const ::boost::shared_ptr< asf::core::Proxy >& proxy, const asf::core::ServiceStateChange& /*stateChange*/)
{
   ETG_TRACE_COMP(("ExternalMediaClientHandler::registerProperties"));
}


/**
	* Description     : Request from Framework to start de-registering for properties.
	*
	* @param[in]      : proxy : reference to the base class of all proxies
	* @param[in]      : stateChange : represents a state transition of a service
	* @return         : void
	*/
void ExternalMediaClientHandler::deregisterProperties(const ::boost::shared_ptr< asf::core::Proxy >& proxy, const asf::core::ServiceStateChange& /*stateChange*/)
{
   ETG_TRACE_COMP(("ExternalMediaClientHandler::deregisterProperties"));
}


void ExternalMediaClientHandler::onUnavailable(const boost::shared_ptr<asf::core::Proxy>& proxy, const asf::core::ServiceStateChange& /*stateChange*/)
{
   ETG_TRACE_COMP(("ExternalMediaClientHandler::onUnavailable"));
   if (_hmiEmpListInterfaceProxy && _hmiEmpListInterfaceProxy == proxy)
   {
      _hmiEmpListInterfaceProxy->sendSig_AlbumArtStateDeregisterAll();
      _hmiEmpListInterfaceProxy->sendSig_CurrentTrackInfoDeregisterAll();
   }
}


void ExternalMediaClientHandler::onAvailable(const ::boost::shared_ptr< asf::core::Proxy >& proxy, const asf::core::ServiceStateChange& /*stateChange*/)
{
   ETG_TRACE_COMP(("ExternalMediaClientHandler::onAvailable"));
   if (_hmiEmpListInterfaceProxy && _hmiEmpListInterfaceProxy == proxy)
   {
      _hmiEmpListInterfaceProxy->sendSig_AlbumArtStateRegister(*this);
      _hmiEmpListInterfaceProxy->sendSig_CurrentTrackInfoRegister(*this);
   }
}


/*
Traceability
Function Name : SelectDeviceSourceMsg
SYSFL ID's:SYSFL-4483, SYSFL-4484,SYSFL-4487
*/

void ExternalMediaClientHandler::onSig_AlbumArtStateError(const ::boost::shared_ptr< HmiEmpListInterfaceProxy >& proxy, const ::boost::shared_ptr< Sig_AlbumArtStateError >& error)
{
   ETG_TRACE_USR1(("ExternalMediaClientHandler::onSig_AlbumArtStateError on:eneterd"));
   int srcID = MediaSourceHandling::getInstance().getCurrentRegionSrcId();
   ETG_TRACE_USR4(("ExternalMediaClientHandler::onSig_AlbumArtStateError:srcID  %d", srcID));

   // Check the requested source id
   switch (srcID)
   {
      case SRC_CAMERAPORT1_USB_VIDEO:
      case SRC_CAMERAPORT1_USB_AUDIO:
      case SRC_CMR1_USB1_VIDEO:
      case SRC_CMR1_USB1_AUDIO:
      {
         HMIModelComponent::getInstance().bOnUpdateAlbumArtDefaultforExternalMediaSources(srcID);

         break;
      }

      default:
      {
         ETG_TRACE_USR4(("ExternalMediaClientHandler::onCourierMessage:srcID: INVALID DEVICE"));
         break;
      }
   }// End of Switch
}


void ExternalMediaClientHandler::onSig_AlbumArtStateSignal(const ::boost::shared_ptr< HmiEmpListInterfaceProxy >& proxy, const ::boost::shared_ptr< Sig_AlbumArtStateSignal >& signal)
{
   ETG_TRACE_USR1(("ExternalMediaClientHandler::onSig_AlbumArtStateSignal:entered"));
   uint8 downloadState = static_cast<uint8>(signal->getDownloadState());
   uint16 songId = signal->getSongId();
   ETG_TRACE_USR1(("ExternalMediaClientHandler::onSig_AlbumArtStateSignal:downloadState  %d", downloadState));
   ETG_TRACE_USR1(("ExternalMediaClientHandler::onSig_AlbumArtStateSignal:songId %d", songId));
   //ETG_TRACE_USR1(("ExternalMediaClientHandler::onSig_AlbumArtStateSignal:m_currentTrackIndex %d", m_currentTrackIndex));
   int srcID = MediaSourceHandling::getInstance().getCurrentRegionSrcId();
   if (downloadState == entDownloadState_Downloadsuccess /*&& songId == m_currentTrackIndex*/)
   {
      if (srcID == SRC_CAMERAPORT1_USB_VIDEO || srcID == SRC_CAMERAPORT1_USB_AUDIO)
      {
         HMIModelComponent::getInstance().vOnUpdateAlbumArtForUsbCamport();
      }
      else if (srcID == SRC_CMR1_USB1_VIDEO || srcID == SRC_CMR1_USB1_AUDIO)
      {
         HMIModelComponent::getInstance().vOnUpdateAlbumArtForUsbRouter();
      }
   }
   else
   {
      ETG_TRACE_USR1(("ExternalMediaClientHandler::onSig_AlbumArtStateSignal Album for song is not avaliable "));

      // Check the requested source id
      switch (srcID)
      {
         case SRC_CAMERAPORT1_USB_VIDEO:
         case SRC_CAMERAPORT1_USB_AUDIO:
         case SRC_CMR1_USB1_VIDEO:
         case SRC_CMR1_USB1_AUDIO:
         {
            HMIModelComponent::getInstance().bOnUpdateAlbumArtDefaultforExternalMediaSources(srcID);

            break;
         }

         default:
         {
            ETG_TRACE_USR4(("ExternalMediaClientHandler::onCourierMessage:srcID: INVALID DEVICE"));
            break;
         }
      }// End of Switch
   }
}


void ExternalMediaClientHandler::onSig_CurrentTrackInfoError(const ::boost::shared_ptr< HmiEmpListInterfaceProxy >& proxy, const ::boost::shared_ptr< Sig_CurrentTrackInfoError >& error)
{
   ETG_TRACE_USR1(("ExternalMediaClientHandler::onsig_CurrentTrackInfoError on:eneterd"));
}


void ExternalMediaClientHandler::onSig_CurrentTrackInfoSignal(const ::boost::shared_ptr< HmiEmpListInterfaceProxy >& proxy, const ::boost::shared_ptr< Sig_CurrentTrackInfoSignal >& signal)
{
   uint8 listtype = static_cast<uint8>(signal->getListType());
   uint16 songId = signal->getSongId();
   std::string path = signal->getFolderPath();
   EnumConst::entSourceType sourceType = static_cast<EnumConst::entSourceType>(signal->getSourceType());
   ETG_TRACE_USR4(("ExternalMediaClientHandler::onsig_CurrentTrackInfoSignal:listtype %d songId %d sourceType:%d", listtype, songId, ETG_CENUM(entSourceType, sourceType)));
   int currentActiveSrcId = _externalMediaSourceHandler->getCabinSrcID(sourceType);
   IPluginClientAppl* l_iPluginClientAppl = PluginClientFactory::getPluginClient(currentActiveSrcId);
   if (l_iPluginClientAppl != NULL)
   {
      l_iPluginClientAppl->updateCurrentPlayingListType(listtype, songId);
      l_iPluginClientAppl->updateCurentPlayingFolderPath(path);
      l_iPluginClientAppl->updateCurrentPlayingSrcType(sourceType);
   }
}


bool ExternalMediaClientHandler::onCourierMessage(const PluginUpdateRespMsg& oMsg)
{
   ETG_TRACE_USR1(("ExternalMediaClientHandler::onCourierMessage:PluginUpdateRespMsg"));
   const Candera::String& pluginName = oMsg.GetPluginName();
   const Candera::String& controlBlockName = oMsg.GetCtrlBlockName();

   const boost::shared_ptr<EventDataUtility>& dataUtility = oMsg.GetPluginData();
   if (NULL != dataUtility.get())
   {
      const std::vector<EventDataItem*>& dataItems = dataUtility->getData();
      ETG_TRACE_USR4(("ExternalMediaClientHandler:onCourierMessage:PluginUpdateRespMsg dataItems %d", dataItems.size()));
      ETG_TRACE_USR4(("ExternalMediaClientHandler:onCourierMessage:PluginUpdateRespMsg plugin name %s", pluginName.GetCString()));

      if (dataItems.size() > 1)
      {
         if (0 == strcmp(PLUGIN_NAME_ROUTER, pluginName.GetCString()))
         {
            if (0 == strcmp(CTRLBLOCK_ROUTER_SIGNAL_SOURCE_CHANGE, controlBlockName.GetCString()))
            {
               EventDataItem* eventDataItem = dataItems.at(0);
               EventDataItem* eventDataItemControlVal = dataItems.at(1);
               EventDataItem* eventDataItemPosition = dataItems.at(1);
               EventDataItem* trackIndex = dataItems.at(2);
               EventDataItem* folderPath = dataItems.at(3);
               EventDataItem* listType = dataItems.at(4);
               EventDataItem* subSrcType = dataItems.at(5);
               EventDataItem* region = dataItems.at(6);
               if (_routerPluginClientHandler != NULL)
               {
                  _routerPluginClientHandler->onMessagePluginUpdateResp(eventDataItem, eventDataItemControlVal, eventDataItemPosition, trackIndex, folderPath, listType, subSrcType, region);
               }
            }
            else
            {
               EventDataItem* eventDataItem = dataItems.at(0);
               EventDataItem* eventDataItemControlVal = dataItems.at(1);
               EventDataItem* eventDataItemPosition = dataItems.at(1);
               EventDataItem* trackIndex = dataItems.at(1);
               EventDataItem* folderPath = NULL;
               EventDataItem* listType = dataItems.at(1);
               EventDataItem* subSrcType = dataItems.at(1);
               EventDataItem* region = dataItems.at(1);
               if (_routerPluginClientHandler != NULL)
               {
                  _routerPluginClientHandler->onMessagePluginUpdateResp(eventDataItem, eventDataItemControlVal, eventDataItemPosition, trackIndex, folderPath, listType, subSrcType, region);
               }
            }
         }
         else if (0 == strcmp(PLUGIN_NAME_MEDIA, pluginName.GetCString()))
         {
            if (0 == strcmp(CTRLBLOCK_NAME_MEDIA_SIGNAL_SOURCE_CHANGE, controlBlockName.GetCString()))
            {
               EventDataItem* eventDataItem = dataItems.at(0);
               EventDataItem* eventDataItemControlVal = dataItems.at(1);
               EventDataItem* eventDataItemPosition = dataItems.at(2);
               EventDataItem* trackIndex = dataItems.at(3);
               EventDataItem* folderPath = dataItems.at(4);
               EventDataItem* listType = dataItems.at(5);
               EventDataItem* subSrcType = dataItems.at(6);
               EventDataItem* region = dataItems.at(7);
               if (_camportUSBPluginClientHandler != NULL)
               {
                  _camportUSBPluginClientHandler->onMessagePluginUpdateResp(eventDataItem, eventDataItemControlVal, eventDataItemPosition, trackIndex, folderPath, listType, subSrcType, region);
               }
            }
            else
            {
               EventDataItem* eventDataItem = dataItems.at(0);
               EventDataItem* eventDataItemControlVal = dataItems.at(1);
               EventDataItem* eventDataItemPosition = dataItems.at(2);
               EventDataItem* trackIndex = dataItems.at(1);
               EventDataItem* folderPath = NULL;
               EventDataItem* listType = dataItems.at(1);
               EventDataItem* subSrcType = dataItems.at(1);
               EventDataItem* region = dataItems.at(1);
               if (_camportUSBPluginClientHandler != NULL)
               {
                  _camportUSBPluginClientHandler->onMessagePluginUpdateResp(eventDataItem, eventDataItemControlVal, eventDataItemPosition, trackIndex, folderPath, listType, subSrcType, region);
               }
            }
         }
      }
   }
   return true;
}


bool ExternalMediaClientHandler::onCourierMessage(const SelectDeviceSourceMsg& oMsg)
{
   ETG_TRACE_USR4(("ExternalMediaClientHandler::onCourierMessage:SelectDeviceSourceMsg DeviceSrc: %d", oMsg.GetDeviceSrc()));

   //// Requested Source Information
   tU8 u8RequestedSrc = oMsg.GetDeviceSrc();
   ETG_TRACE_USR4(("ExternalMediaClientHandler::onCourierMessage:SelectDeviceSourceMsg:u8RequestedSrc: %d", u8RequestedSrc));
   bool isValidRequestedSrc = false;
   uint8 u8activeRegion = ApplicationSwitchClientHandler::poGetInstance()->getActiveRegion();
   ETG_TRACE_USR4(("ExternalMediaClientHandler::onCourierMessage:SelectDeviceSourceMsg u8activeRegion  %d ", u8activeRegion));

   // Check the requested device type if current source is not the requested source
   switch (u8RequestedSrc)
   {
      case FI_EN_E8DTY_USB_CAMERAPORT:
      {
         ETG_TRACE_USR4(("ExternalMediaClientHandler::onCourierMessage:SelectDeviceSourceMsg:m_iUsbCamportAvailable: %d", m_iUsbCamportAvailable));
         onSourceSelection(EnumConst::entSourceType_CMP);

         break;
      }
      case FI_EN_E8DTY_USB_CAMERAPORT_HDMI:
      case FI_EN_E8DTY_USB_CAMERAPORT_HDMI_2:
      case FI_EN_E8DTY_USB_CAMERAPORT_HDMI_3:
      {
         ETG_TRACE_USR4(("ExternalMediaClientHandler::onCourierMessage:SelectDeviceSourceMsg:HDMI Selected"));
         onHdmiSourceSelection(u8RequestedSrc);

         break;
      }
      case FI_EN_E8DTY_ROUTER_USB:
      {
         ETG_TRACE_USR4(("ExternalMediaClientHandler::onCourierMessage:SelectDeviceSourceMsg:m_isRouterUsbConnected: %d", m_isRouterUsbConnected));

         onSourceSelection(EnumConst::entSourceType_CMR);

         break;
      }
      default:
      {
         ETG_TRACE_USR4(("ExternalMediaClientHandler::onCourierMessage:SelectDeviceSourceMsg: INVALID DEVICE"));
         break;
      }
   }// End of Switch
   return true;
}


void ExternalMediaClientHandler::onHdmiSourceSelection(uint8 deviceType)
{
   int srcId = getHdmiSrcId(deviceType);
   bool bIsHDMISourceActive = false;
   bIsHDMISourceActive = getHdmiSourceActiveState(srcId);
   ETG_TRACE_USR1(("ExternalMediaClientHandler::onHdmiSourceSelection: deviceType = %d, srcId = %d, bIsHDMISourceActive: %d", deviceType, srcId, bIsHDMISourceActive));
   if ((_systemInfoMap->getSystemStatus(SYS_CATEGORY_CMGA) == SYSTEM_STATUS_UNAVAILABLE || _systemInfoMap->getSystemStatus(SYS_CATEGORY_CMGA) == SYSTEM_STATUS_DISCONNECTED) && getHdmiSourceAvailability(srcId) == EN_EXTERNAL_MEDIA_CONNECTED)
   {
      POST_MSG((COURIER_MESSAGE_NEW(Popup_Pfo_MediaGlassUnavailableMessage_Open)()));
   }
   else if (getHdmiSourceAvailability(srcId) == EN_EXTERNAL_MEDIA_UNSUPPORTED)
   {
      POST_MSG((COURIER_MESSAGE_NEW(Popup_Pfo_SignalRangeError_Open)()));
   }
   else
   {
      handleSourceSelectionOnDownloadComplete(deviceType, bIsHDMISourceActive);
   }
}


void ExternalMediaClientHandler::onCabinTrackCommandChangeRequest(uint8 cabinMediaTrackCommandValue)
{
   ETG_TRACE_COMP(("ExternalMediaClientHandler::onCabinTrackCommandChangeRequest: cabinMediaTrackCommandValue = %d", cabinMediaTrackCommandValue));
   int iActiveSource = MediaUtils::getInstance().getActiveSource(REGION_CABIN_A);
   switch (cabinMediaTrackCommandValue)
   {
      case CAN_CabinTrackCommand_Pause:
      {
         if (ExternalMediaPlaybackHandler::getInstance()->getPlaybackState() != EXT_MEDIA_PAUSE)
         {
            ExternalMediaPlaybackHandler::getInstance()->setOperationType(EN_MEDIA_PAUSE);
            ExternalMediaPlaybackHandler::getInstance()->onPlaybackActionRequest(EN_MEDIA_PAUSE, iActiveSource);
         }
      }
      break;
      case CAN_CabinTrackCommand_Play:
      {
         if (ExternalMediaPlaybackHandler::getInstance()->getPlaybackState() != EXT_MEDIA_PLAY)
         {
            ExternalMediaPlaybackHandler::getInstance()->setOperationType(EN_MEDIA_PLAY);
            ExternalMediaPlaybackHandler::getInstance()->onPlaybackActionRequest(EN_MEDIA_PLAY, iActiveSource);
         }
      }
      break;
      case CAN_CabinTrackCommand_Next:
      {
         ExternalMediaPlaybackHandler::getInstance()->setOperationType(EN_MEDIA_NEXT);
         ExternalMediaPlaybackHandler::getInstance()->onPlaybackActionRequest(EN_MEDIA_NEXT, iActiveSource);
      }
      break;
      case CAN_CabinTrackCommand_Previous:
      {
         ExternalMediaPlaybackHandler::getInstance()->setOperationType(EN_MEDIA_PREV);
         ExternalMediaPlaybackHandler::getInstance()->onPlaybackActionRequest(EN_MEDIA_PREV, iActiveSource);
      }
      break;
      case CAN_CabinTrackCommand_FastForward:
      {
         ExternalMediaPlaybackHandler::getInstance()->setOperationType(EN_MEDIA_FASTFORWARD);
         ExternalMediaPlaybackHandler::getInstance()->onPlaybackActionRequest(EN_MEDIA_FASTFORWARD, iActiveSource);
         _canSigFFReqOngoing = true;
      }
      break;
      case CAN_CabinTrackCommand_FastBackward:
      {
         ExternalMediaPlaybackHandler::getInstance()->setOperationType(EN_MEDIA_FASTREWIND);
         ExternalMediaPlaybackHandler::getInstance()->onPlaybackActionRequest(EN_MEDIA_FASTREWIND, iActiveSource);
         _canSigFRReqOngoing = true;
      }
      break;
      case TrackCommand_RandomRepeatOFF:
      {
         if ((ExternalMediaPlaybackHandler::getInstance()->getRepeatState() == EXT_REPEAT_TRACK) || (ExternalMediaPlaybackHandler::getInstance()->getRepeatState() == EXT_REPEAT_LIST))
         {
            ExternalMediaPlaybackHandler::getInstance()->setOperationType(EN_MEDIA_REPEAT_OFF);
            ExternalMediaPlaybackHandler::getInstance()->onPlaybackActionRequest(EN_MEDIA_REPEAT_OFF, iActiveSource);
         }
         else if (ExternalMediaPlaybackHandler::getInstance()->getShuffleState() == EXT_SHUFFLE_LIST)
         {
            ExternalMediaPlaybackHandler::getInstance()->setOperationType(EN_MEDIA_SHUFFLE_OFF);
            ExternalMediaPlaybackHandler::getInstance()->onPlaybackActionRequest(EN_MEDIA_SHUFFLE_OFF, iActiveSource);
         }
      }
      break;
      case CAN_CabinTrackCommand_RepeatTrack:
      {
         ExternalMediaPlaybackHandler::getInstance()->setOperationType(EN_MEDIA_REPEAT_TRACK);
         ExternalMediaPlaybackHandler::getInstance()->onPlaybackActionRequest(EN_MEDIA_REPEAT_TRACK, iActiveSource);
      }
      break;
      case CAN_CabinTrackCommand_RepeatAll:
      {
         ExternalMediaPlaybackHandler::getInstance()->setOperationType(EN_MEDIA_REPEAT_LIST);
         ExternalMediaPlaybackHandler::getInstance()->onPlaybackActionRequest(EN_MEDIA_REPEAT_LIST, iActiveSource);
      }
      break;
      case TrackCommand_RandomAll:
      {
         ExternalMediaPlaybackHandler::getInstance()->setOperationType(EN_MEDIA_SHUFFLE_LIST);
         ExternalMediaPlaybackHandler::getInstance()->onPlaybackActionRequest(EN_MEDIA_SHUFFLE_LIST, iActiveSource);
      }
      break;
      case CAN_CabinTrackCommand_NotAvailable:
      {
         if (_canSigFFReqOngoing == true)
         {
            ExternalMediaPlaybackHandler::getInstance()->setOperationType(EN_MEDIA_FASTFORWARD_STOP);
            ExternalMediaPlaybackHandler::getInstance()->onPlaybackActionRequest(EN_MEDIA_FASTFORWARD_STOP, iActiveSource);
            _canSigFFReqOngoing = false;
         }
         else if (_canSigFRReqOngoing == true)
         {
            ExternalMediaPlaybackHandler::getInstance()->setOperationType(EN_MEDIA_FASTREWIND_STOP);
            ExternalMediaPlaybackHandler::getInstance()->onPlaybackActionRequest(EN_MEDIA_FASTREWIND_STOP, iActiveSource);
            _canSigFRReqOngoing = false;
         }
      }
      break;
      default:
      {
         ETG_TRACE_USR1(("ExternalMediaClientHandler:onCabinTrackCommandChangeRequest Default case option"));
      }
      break;
   }
}


void ExternalMediaClientHandler::onExternalSrcActivationWhenInCockpit(EnumConst::entSourceType srcType, int deviceType)
{
   ETG_TRACE_COMP(("ExternalMediaClientHandler::onExternalSrcActivationWhenInCockpit: srcType %d, deviceType %d", ETG_CENUM(entSourceType, srcType), deviceType));
   IExternalMediaSourceActivationHandler* iExternalMediaSourceActivationHandler = ExternalMediaSourceActivationFactory::getExternalActivationSourceHandlerObject(deviceType);
   if (DbDownloadMap::getInstance().getDbDownloadState(srcType) == EnumConst::entDownloadState_Downloadsuccess)
   {
      if (_playableContentMap->getPlayableAudioContent(srcType) == true ||
            (_playableContentMap->getPlayableVideoContent(srcType) == true && (_systemInfoMap->getSystemStatus(SYS_CATEGORY_CMGA) != SYSTEM_STATUS_UNAVAILABLE && _systemInfoMap->getSystemStatus(SYS_CATEGORY_CMGA) != SYSTEM_STATUS_DISCONNECTED)))
      {
         iExternalMediaSourceActivationHandler->requestSourceActivation(REGION_CABIN_A);
      }
   }
}


void ExternalMediaClientHandler::onExternalSrcActivationWhenInCockpit(int deviceType)
{
   int srcId = getHdmiSrcId(deviceType);
   ETG_TRACE_USR1(("ExternalMediaClientHandler::onExternalSrcActivationWhenInCockpit: deviceType %d, srcId %d", deviceType, srcId));
   IExternalMediaSourceActivationHandler* iExternalMediaSourceActivationHandler = ExternalMediaSourceActivationFactory::getExternalActivationSourceHandlerObject(deviceType);
   if ((_systemInfoMap->getSystemStatus(SYS_CATEGORY_CMGA) != SYSTEM_STATUS_UNAVAILABLE && _systemInfoMap->getSystemStatus(SYS_CATEGORY_CMGA) != SYSTEM_STATUS_DISCONNECTED) && getHdmiSourceAvailability(srcId) == EN_EXTERNAL_MEDIA_CONNECTED)
   {
      MediaSourceHandling::getInstance().requestSourceActivation(srcId, -1, REGION_CABIN_A);;
   }
}


/*
Traceability
Function Name : MediaOperationMsgReq
SYSFL ID's:SYSFL-4617,
SYSFL-4618,SYSFL-4619,SYSFL-4620,SYSFL-4621,SYSFL-4622,[SWFL-2960]: Random mode , 462617 [SWFL-2962]: Repeat mode
*/

void ExternalMediaClientHandler::vRequestExternalMediaSourcePlayBack(int32 iAudioConnStatus, int srcID)
{
   ETG_TRACE_USR4(("ExternalMediaClientHandler::vRequestExternalMediaSourcePlayBack  iAudioConnStatus  %d ", iAudioConnStatus));
   ETG_TRACE_USR4(("ExternalMediaClientHandler::vRequestExternalMediaSourcePlayBack  srcID  %d ", srcID));

   IPluginClientAppl* l_iPluginClientAppl = PluginClientFactory::getPluginClient(srcID);
   if (iAudioConnStatus == STATE_SUSPENDED)
   {
      if (l_iPluginClientAppl != NULL)
      {
         l_iPluginClientAppl->sendPluginRequest(EXT_MEDIA_PAUSE, srcID);
      }
   }
   if (iAudioConnStatus == STATE_CONNECTED)
   {
      if (ExternalMediaPlaybackHandler::getInstance()->getPlaybackState() == EXT_MEDIA_PLAY)
      {
         if (l_iPluginClientAppl != NULL)
         {
            l_iPluginClientAppl->sendPluginRequest(EXT_MEDIA_PLAY, srcID);
         }
      }
   }
}


void ExternalMediaClientHandler::vRequestStateTransitionForExternalSource(bool bIsExternalSourceActive)
{
   ETG_TRACE_USR4(("ExternalMediaClientHandler::vRequestStateTransitionForExternalSource  bIsExternalSourceActive  %d ", bIsExternalSourceActive));
   if (bIsExternalSourceActive == true)
   {
      StateTransitionHandler::getInstance().activateMediaState(MEDIA_STATE_NOWPLAYING);
   }
   else
   {
      StateTransitionHandler::getInstance().setStateTransitionReason(ST_TRANSITION_REASON_USER_SRC_REQUEST);
   }
}


void ExternalMediaClientHandler::setExternalMediaListDBManagerInstance(ExternalMediaListDBManager* _externalMediaListDBManagerInstance)
{
   if (_externalMediaListDBManagerInstance != NULL)
   {
      _externalMediaListDBManager = _externalMediaListDBManagerInstance;
      _externalMediaListDBManager->registerExternalDBDownloadObservor((*this));
   }
}


void ExternalMediaClientHandler::setPlayableContentMapInstance(PlayableContentMap* _playableContentMapInstance)
{
   if (_playableContentMapInstance != NULL)
   {
      _playableContentMap = _playableContentMapInstance;
      _camportUSBSourceActivationHandler->setPlayableMapInstance(_playableContentMap);
      _camportRouterSourceActivationHandler->setPlayableMapInstance(_playableContentMap);
   }
   else
   {
      ETG_TRACE_USR1(("ExternalMediaClientHandler::setPlayableContentMapInstance _playableContentMapInstance is NULL"));
   }
}


void ExternalMediaClientHandler::setPlayableContentInterfaceHandler(PlayableContentInterfaceHandler* _playableContentInterfaceHandlerIntance)
{
   if (_playableContentInterfaceHandlerIntance != NULL)
   {
      _playableContentInterfaceHandler = _playableContentInterfaceHandlerIntance;
      if (_camportUSBPluginClientHandler != NULL)
      {
         _camportUSBPluginClientHandler->setPlayableContentInterfaceHandlerInstance(_playableContentInterfaceHandler);
      }
      if (_routerPluginClientHandler != NULL)
      {
         _routerPluginClientHandler->setPlayableContentInterfaceHandlerInstance(_playableContentInterfaceHandler);
      }
   }
   else
   {
      ETG_TRACE_USR1(("ExternalMediaClientHandler::setPlayableContentInterfaceHandlerInstance _playableContentInterfaceHandlerIntance is NULL"));
   }
}


void ExternalMediaClientHandler::setHmiStateHandlerInstance(HmiStateHandler* _hmiStateHandlerInstance)
{
   if (_hmiStateHandlerInstance != NULL)
   {
      _hmiStateHandler = _hmiStateHandlerInstance;
      _hmiStateHandler->addHmiStateStatusObserver((*this));
   }
   else
   {
      ETG_TRACE_ERR(("ExternalMediaClientHandler::_hmiStateHandler is NULL"));
   }
}


bool ExternalMediaClientHandler::onCourierMessage(const DBDownloadStateUpdateMsg& oMsg)
{
   ETG_TRACE_USR1(("ExternalMediaClientHandler::DBDownloadStateUpdateMsg"));
   onExternalDBDownloadUpdate((EnumConst::entDownloadState(oMsg.GetDownloadState())), (EnumConst::entSourceType(oMsg.GetSourceType())));
   DbDownloadMap::getInstance().setDbDownloadState(EnumConst::entSourceType(oMsg.GetSourceType()), EnumConst::entDownloadState(oMsg.GetDownloadState()));
   return false;
}


void ExternalMediaClientHandler::onExternalDBDownloadUpdate(uint8 downloadState, EnumConst::entSourceType sourceType)
{
   ETG_TRACE_USR4(("ExternalMediaClientHandler::onExternalDBDownloadUpdate downloadState:%d sourceType:%d", downloadState, sourceType));
   int iActiveCabinSrcId = MediaSourceHandling::getInstance().getCurrentRegionSrcId();
   EnumConst::entDownloadState downloadStatus = static_cast<EnumConst::entDownloadState>(downloadState);
   ETG_TRACE_USR4(("ExternalMediaClientHandler::onExternalDBDownloadUpdate downloadStatus:%d", ETG_CENUM(entDownloadState, downloadStatus)));
   if (downloadStatus == EnumConst::entDownloadState_Downloadsuccess)
   {
      handleDBDownloadStateUpdate(sourceType);
   }
   else
   {
      if ((downloadStatus == EnumConst::entDownloadState_DownloadError) && (StateTransitionHandler::getInstance().getcurrentMediaState() != MEDIA_STATE_SOURCESELECTION))
      {
         if (sourceType == EnumConst::entSourceType_CMP)
         {
            (*m_DbDownloadFailureData).mExternalDeviceType = "CMP";
         }
         else
         {
            (*m_DbDownloadFailureData).mExternalDeviceType = "CMR";
         }
         m_DbDownloadFailureData.MarkItemModified(ItemKey::DBDownloadError::ExternalDeviceTypeItem);
         m_DbDownloadFailureData.SendUpdate();

         POST_MSG((COURIER_MESSAGE_NEW(ActivatePopUpMsg)(Media_MediaPlayer_Pop_Up_system_Pfo_MediaDB_Download_Failure)));//trigger download failure popup
      }

      //Transit to source selection screen
      if (StateTransitionHandler::getInstance().getcurrentMediaState() == MEDIA_STATE_NOWPLAYING)
      {
         if (((iActiveCabinSrcId == SRC_CAMERAPORT1_USB_VIDEO || iActiveCabinSrcId == SRC_CAMERAPORT1_USB_AUDIO) && (sourceType == EnumConst::entSourceType_CMP))
               || ((iActiveCabinSrcId == SRC_CMR1_USB1_VIDEO || iActiveCabinSrcId == SRC_CMR1_USB1_AUDIO) && (sourceType == EnumConst::entSourceType_CMR)))
         {
            MediaSourceHandling::getInstance().vRequestSourceDeactivation(iActiveCabinSrcId, -1, ApplicationSwitchClientHandler::poGetInstance()->getActiveRegion());
            if (sourceType == EnumConst::entSourceType_CMR)
            {
               StateTransitionHandler::getInstance().stateTransit_onSourceSelectionContextUpdate();
            }
         }
      }
   }
}


void ExternalMediaClientHandler::handleDBDownloadStateUpdate(EnumConst::entSourceType sourceType)
{
   if (sourceType == EnumConst::entSourceType_CMP)
   {
      //Trigger Popup to show "Camera port USB DB download successful"
      (*m_DbDownloadSuccessfulData).mDbDownloadDevice = Candera::String(TextId_0x13B2);
   }
   else
   {
      //Trigger Popup to show " Router USB DB download successful"
      (*m_DbDownloadSuccessfulData).mDbDownloadDevice = Candera::String(TextId_0x13AF);
   }
   m_DbDownloadSuccessfulData.MarkItemModified(ItemKey::ExternalDeviceConnected::DbDownloadDeviceItem);
   m_DbDownloadSuccessfulData.SendUpdate();

   if (sourceType == EnumConst::entSourceType_CMP) //Temporarily added as part of 1606014 - for disabling Router USB
   {
      POST_MSG((COURIER_MESSAGE_NEW(ActivatePopUpMsg)(Media_MediaPlayer_Pop_Up_system_Pfo_MediaDB_Download_Successful)));//popuptrigger
   }
}


void ExternalMediaClientHandler::setExternalMediaSourceHandlerInstance(ExternalMediaSourceHandler* _externalMediaSourceHandlerInstance)
{
   if (_externalMediaSourceHandlerInstance != NULL)
   {
      _externalMediaSourceHandler = _externalMediaSourceHandlerInstance;
      _externalMediaSourceHandler->addHdmiSourceAvailabilityInfoObservor((*this));
      _externalMediaSourceHandler->addCamportUsbSourceAvailabilityObservor((*this));
      _externalMediaSourceHandler->addCamportUsbSourceAllocObservor((*this));
      _externalMediaSourceHandler->addRouterSourceAllocationInfoObservor((*this));
      _externalMediaSourceHandler->addRouterSourceAvailabilityInfoObservor((*this));
      _externalMediaSourceHandler->addHdmiSourceAllocInfoObservor((*this));

      if (_routerPluginClientHandler != NULL)
      {
         _routerPluginClientHandler->setExternalMediaSrcHandlerInstance(_externalMediaSourceHandler);
      }

      if (_camportUSBPluginClientHandler != NULL)
      {
         _camportUSBPluginClientHandler->setExternalMediaSrcHandlerInstance(_externalMediaSourceHandler);
      }
   }
   else
   {
      ETG_TRACE_USR1(("ExternalMediaClientHandler::_externalMediaSourceHandler is NULL"));
   }
}


void ExternalMediaClientHandler::onHdmiSourceAvailabilityUpdate(int srcId, int hdmiavailable)
{
   ETG_TRACE_USR4(("ExternalMediaClientHandler::onHdmiSourceAvailabilityUpdate:srcId %d, hdmiavailable %d", srcId, hdmiavailable));
   std::map <int, HdmiSourceInfo>::iterator itr = _hdmiSrcAvailability.find(srcId);
   int iActiveCabinSrcId = MediaSourceHandling::getInstance().getCurrentRegionSrcId();
   if (itr != _hdmiSrcAvailability.end())
   {
      itr ->second.sourceAvailability = hdmiavailable;
      if ((hdmiavailable == EN_EXTERNAL_MEDIA_UNSUPPORTED) && (iActiveCabinSrcId == srcId))
      {
         POST_MSG((COURIER_MESSAGE_NEW(Popup_Pfo_SignalRangeError_Open)()));
         StateTransitionHandler::getInstance().setStateTransitionReason(ST_TRANSITION_REASON_DEVICE_REMOVAL);
         StateTransitionHandler::getInstance().stateTransit_onActiveSourceDeviceDisconnection();
      }
   }
}


void ExternalMediaClientHandler::onHdmiSourceAllocationUpdate(int iSourceId, int iSinkId, int ConnectionState)
{
   ETG_TRACE_USR4(("ExternalMediaClientHandler::onHdmiSourceAllocationUpdate  iSinkId %d  iSourceId %d  connectionState %d ", iSinkId, iSourceId, ConnectionState));

   if (ConnectionState == STATE_CONNECTED)
   {
      setHdmiActiveState(iSourceId, true);
   }
   else if (ConnectionState == STATE_DISCONNECTED)
   {
      setHdmiActiveState(iSourceId, false);
      MediaSourceHandling::getInstance().vResetAudioSrcMapofCabin(iSourceId);
   }
}


void ExternalMediaClientHandler::onCamportUsbSourceAvailabilityUpdate(int iUSBCamportAvailability)
{
   ETG_TRACE_USR4(("ExternalMediaClientHandler::onCamportUsbSourceAvailabilityUpdate iUSBCamportAvailability  %d", iUSBCamportAvailability));
   m_iUsbCamportAvailable = iUSBCamportAvailability;

   if (m_iUsbCamportAvailable == EN_EXTERNAL_MEDIA_DISCONNECTED)
   {
      m_isCamportUSBSourceActive = false;
      //    IExternalMediaSourceActivationHandler* iExternalMediaSourceActivationHandler = ExternalMediaSourceActivationFactory::getExternalActivationSourceHandlerObject(FI_EN_E8DTY_USB_CAMERAPORT);
      //    iExternalMediaSourceActivationHandler->reset();
      _playableContentMap->setPlayableAudioContent(EnumConst::entSourceType_CMP, false);
      _playableContentMap->setPlayableVideoContent(EnumConst::entSourceType_CMP, false);
   }
}


void ExternalMediaClientHandler::onCamportUsbSourceAllocationUpdate(int iSourceId, int iSinkId, int connectionState)
{
   ETG_TRACE_USR4(("ExternalMediaClientHandler::onCamportUsbSourceAllocationUpdate  iSinkId %d  iSourceId %d  connectionState %d, _isHmiOffWithActiveMediaSrc %d", iSinkId, iSourceId, connectionState, _isHmiOffWithActiveMediaSrc));
   IPluginClientAppl* l_iPluginClientAppl = PluginClientFactory::getPluginClient(iSourceId);
   _externalMediaSourceHandler->setCabinSrc(iSourceId);

   int iActiveCabinSrcId = MediaSourceHandling::getInstance().getCurrentRegionSrcId();
   int prevCabinSrcid = MediaSourceHandling::getInstance().getPrevSrcId();
   if (iSinkId == CABIN_A_SINK_ID ||  iSinkId == CABIN_B_SINK_ID)
   {
      if (connectionState == STATE_CONNECTED)
      {
         m_isCamportUSBSourceActive = true;
         _playableContentInterfaceHandler->onCRCStatusReceived(EnumConst::entSourceType_CMP, EN_CRC_MATCH);
         ExternalMediaPlaybackHandler::getInstance()->updatePlaybackState();
         ExternalMediaPlaybackHandler::getInstance()->updateOperationType();
         if (iSourceId == SRC_CAMERAPORT1_USB_VIDEO)
         {
            if (_isHmiOffWithActiveMediaSrc == true)
            {
               if (l_iPluginClientAppl != NULL)
               {
                  ExternalMediaPlaybackHandler::getInstance()->setPlaybackState(MEDIA_VIDEOSOURCERESTORE);
                  ExternalMediaPlaybackHandler::getInstance()->setOperationType(EN_MEDIA_PAUSE);
                  l_iPluginClientAppl->sendPluginRequest(MEDIA_VIDEOSOURCERESTORE, iSourceId);
                  _isHmiOffWithActiveMediaSrc = false;
               }
            }
            else
            {
               if (l_iPluginClientAppl != NULL)
               {
                  l_iPluginClientAppl->sendPluginRequest(getExternalSrcPlaybackState(iSourceId), iSourceId);
               }
            }
         }
         else
         {
            if (l_iPluginClientAppl != NULL)
            {
               l_iPluginClientAppl->sendPluginRequest(getExternalSrcPlaybackState(iSourceId), iSourceId);
            }
         }
      }
      else if (connectionState == STATE_DISCONNECTED || connectionState == STATE_SUSPENDED)
      {
         if (l_iPluginClientAppl != NULL)
         {
            l_iPluginClientAppl->sendPluginRequest(EXT_MEDIA_PAUSE, iSourceId);
         }

         if (connectionState == STATE_DISCONNECTED)
         {
            m_isCamportUSBSourceActive = false;
            if (iActiveCabinSrcId == SRC_CAMERAPORT1_USB_VIDEO || iActiveCabinSrcId == SRC_CAMERAPORT1_USB_AUDIO)
            {
               MediaSourceHandling::getInstance().vResetAudioSrcMapofCabin(iActiveCabinSrcId);
            }
         }
      }
      else
      {
         ETG_TRACE_USR4(("ExternalMediaClientHandler::onCamportUsbSourceAllocationUpdate Default "));
      }
   }
}


uint8 ExternalMediaClientHandler::getExternalSrcPlaybackState(int sourceId)
{
   uint8 externalSrcPlaybackState = EXT_MEDIA_PLAY;
   int iActiveCabinSrcId = MediaSourceHandling::getInstance().getCurrentRegionSrcId();
   int prevCabinSrcid = MediaSourceHandling::getInstance().getPrevSrcId();

   ETG_TRACE_USR4(("getExternalSrcPlaybackState::prevCabinSrcid:%d iActiveCabinSrcId:%d ", prevCabinSrcid, iActiveCabinSrcId));
   ETG_TRACE_USR1(("ExternalMediaClientHandler::getExternalSrcPlaybackState sourceId : %d", sourceId));
   ETG_TRACE_USR1(("ExternalMediaClientHandler::getCabinSrc %d", _externalMediaSourceHandler->getCabinSrc()));

   if (sourceId == _externalMediaSourceHandler->getCabinSrc())
   {
      externalSrcPlaybackState = ExternalMediaPlaybackHandler::getInstance()->getlastPlaybackState();
      ETG_TRACE_USR1(("ExternalMediaClientHandler::getExternalSrcPlaybackState :%d ", externalSrcPlaybackState));
      _externalMediaSourceHandler->setCabinSrc(SRC_INVALID);

      if (externalSrcPlaybackState == EXT_MEDIA_PAUSE)
      {
         ExternalMediaPlaybackHandler::getInstance()->setPlaybackState(EXT_MEDIA_PAUSE);
         ExternalMediaPlaybackHandler::getInstance()->setOperationType(EN_MEDIA_PAUSE);
         externalSrcPlaybackState = EXT_MEDIA_PAUSE;
      }
      else if (externalSrcPlaybackState == EXT_MEDIA_PLAY)
      {
         if (iActiveCabinSrcId != prevCabinSrcid && prevCabinSrcid != SRC_INVALID)
         {
            externalSrcPlaybackState = EXT_MEDIA_SOURCEACTIVE;
         }
         else
         {
            externalSrcPlaybackState = EXT_MEDIA_PLAY;
         }
      }
   }
   ETG_TRACE_USR1(("ExternalMediaClientHandler::getExternalSrcPlaybackState : sourceId %d,externalSrcPlaybackState %d", sourceId, externalSrcPlaybackState));
   return externalSrcPlaybackState;
}


int ExternalMediaClientHandler::getHdmiSrcId(uint8 deviceType)
{
   int srcId = SRC_INVALID;
   std::map <uint8, int>::iterator itr = _hdmiDevTypSrcMap.find(deviceType);
   if (itr != _hdmiDevTypSrcMap.end())
   {
      srcId  = itr -> second;
   }
   ETG_TRACE_USR1(("ExternalMediaClientHandler::getHdmiSrcId:deviceType %d, srcId %d", deviceType, srcId));
   return srcId;
}


void ExternalMediaClientHandler::handleSourceSelectionOnDownloadComplete(int deviceType, bool isSourceActive)
{
   uint8 activeRegion = ApplicationSwitchClientHandler::poGetInstance()->getActiveRegion();
   ETG_TRACE_USR4(("ExternalMediaClientHandler::handleSourceSelectionOnDownloadComplete activeRegion :%d deviceType:%d", activeRegion, deviceType));
   HMIModelComponent::getInstance().vUpdateDeviceTypeToSM(deviceType);

   vRequestStateTransitionForExternalSource(isSourceActive);

   if (deviceType == FI_EN_E8DTY_USB_CAMERAPORT)
   {
      IExternalMediaSourceActivationHandler* iExternalMediaSourceActivationHandler = ExternalMediaSourceActivationFactory::getExternalActivationSourceHandlerObject(deviceType);
      if (_playableContentMap->getPlayableVideoContent(EnumConst::entSourceType_CMP) == true && (_systemInfoMap->getSystemStatus(SYS_CATEGORY_CMGA) == SYSTEM_STATUS_UNAVAILABLE || _systemInfoMap->getSystemStatus(SYS_CATEGORY_CMGA) == SYSTEM_STATUS_DISCONNECTED))
      {
         POST_MSG((COURIER_MESSAGE_NEW(Popup_Pfo_MediaGlassUnavailableMessage_Open)()));
      }
      else
      {
         iExternalMediaSourceActivationHandler->requestSourceActivation(activeRegion);
      }
   }
   else if (deviceType == FI_EN_E8DTY_ROUTER_USB)
   {
      IExternalMediaSourceActivationHandler* iExternalMediaSourceActivationHandler = ExternalMediaSourceActivationFactory::getExternalActivationSourceHandlerObject(deviceType);
      if (_playableContentMap->getPlayableVideoContent(EnumConst::entSourceType_CMR) == true && (_systemInfoMap->getSystemStatus(SYS_CATEGORY_CMGA) == SYSTEM_STATUS_UNAVAILABLE || _systemInfoMap->getSystemStatus(SYS_CATEGORY_CMGA) == SYSTEM_STATUS_DISCONNECTED))
      {
         POST_MSG((COURIER_MESSAGE_NEW(Popup_Pfo_MediaGlassUnavailableMessage_Open)()));
      }
      else
      {
         iExternalMediaSourceActivationHandler->requestSourceActivation(activeRegion);
      }
   }
   else if (deviceType == FI_EN_E8DTY_USB_CAMERAPORT_HDMI || deviceType == FI_EN_E8DTY_USB_CAMERAPORT_HDMI_2 || deviceType == FI_EN_E8DTY_USB_CAMERAPORT_HDMI_3)
   {
      int srcId = getHdmiSrcId(deviceType);
      MediaSourceHandling::getInstance().requestSourceActivation(srcId, -1, activeRegion);
   }

   if ((deviceType == FI_EN_E8DTY_USB_CAMERAPORT) || (deviceType == FI_EN_E8DTY_ROUTER_USB))
   {
      HMIModelComponent::getInstance().vUpdateUsbAlbumIconAndTextVisibility();
   }
}


void ExternalMediaClientHandler::onSourceSelection(EnumConst::entSourceType sourceType)
{
   ETG_TRACE_USR4(("ExternalMediaClientHandler::onSourceSelection sourceType:%d isCamportSourceActive:%d isRouterSourceActive:%d",
                   sourceType, m_isCamportUSBSourceActive, m_isRouterSourceActive));

   if (DbDownloadMap::getInstance().getDbDownloadState(sourceType) != EnumConst::entDownloadState_Downloadsuccess)
   {
      if (DbDownloadMap::getInstance().getDbDownloadState(sourceType) == EnumConst::entDownloadState_DownloadError) //triggerdownload
      {
         bosch::cm::di::HmiEmpListInterface::entSourceType currentSourceType = static_cast <bosch::cm::di::HmiEmpListInterface::entSourceType>(sourceType);
         _hmiEmpListInterfaceProxy->sendReqDBDownloadRequest(*this, currentSourceType);
      }
      else
      {
         std::string strPopupText = Candera::String(TextId_0x1BF6).GetCString(); // to be used when text id is released
         //std::string strPopupText = Candera::String(TextId_0x1A1A).GetCString(); // for testing..remove later
         std::string strSourceType = "";
         if (sourceType == EnumConst::entSourceType_CMP)
         {
            strSourceType = "CMP";
         }
         else
         {
            strSourceType = "CMR";
         }
         (*m_DbInPreparationData).mDeviceType = (replaceInTranslationTextID(strPopupText, strSourceType, "%1")).c_str();
         m_DbInPreparationData.MarkItemModified(ItemKey::DBInPreparation::DeviceTypeItem);
         m_DbInPreparationData.SendUpdate();

         POST_MSG((COURIER_MESSAGE_NEW(::PopupReqMsg)(hmibase::popups::Show, Courier::ViewId("Media#MediaPlayer_Pop_Up#Pfo_MediaDB_Download_Error"))));//popuptrigger
      }
   }
   else
   {
      if ((_playableContentMap->getPlayableAudioContent(sourceType) == false)
            && (_playableContentMap->getPlayableVideoContent(sourceType) == false))
      {
         ETG_TRACE_USR1(("ExternalMediaClientHandler::onCourierMessage:SelectDeviceSourceMsg Trigger No Playable Content Popup"));
         POST_MSG((COURIER_MESSAGE_NEW(Popup_Pfo_NoPlayableContentPopup_Open)()));
      }
      else
      {
         bool isSourceActive = (sourceType == EnumConst::entSourceType_CMP) ? m_isCamportUSBSourceActive  : m_isRouterSourceActive;
         int deviceType = (sourceType == EnumConst::entSourceType_CMP) ? FI_EN_E8DTY_USB_CAMERAPORT : FI_EN_E8DTY_ROUTER_USB;
         handleSourceSelectionOnDownloadComplete(deviceType, isSourceActive);
      }
   }
}


std::string ExternalMediaClientHandler::replaceInTranslationTextID(std::string TranslationStr, std::string StrForReplace, std::string strToSearch)
{
   if (TranslationStr.find(strToSearch) != std::string::npos)
   {
      TranslationStr.replace(TranslationStr.find(strToSearch), strToSearch.size(), StrForReplace);
   }
   ETG_TRACE_USR4(("ExternalMediaClientHandler:onCourierMessage:PluginUpdateRespMsg plugin name %s", TranslationStr.c_str()));
   return TranslationStr;
}


void ExternalMediaClientHandler::onReqDBDownloadError(const ::boost::shared_ptr< HmiEmpListInterfaceProxy >& proxy, const ::boost::shared_ptr< ReqDBDownloadError >& error)
{
   ETG_TRACE_USR1(("ExternalMediaClientHandler:: onReqDBDownloadError"));
}


void ExternalMediaClientHandler::onReqDBDownloadResponse(const ::boost::shared_ptr< HmiEmpListInterfaceProxy >& proxy, const ::boost::shared_ptr< ReqDBDownloadResponse >& response)
{
   bool bDownloadResponse = response->getResponse();
   ETG_TRACE_USR1(("ExternalMediaClientHandler:: onReqDBDownloadResponse : %d", bDownloadResponse));
}


void ExternalMediaClientHandler::onRouterSourceAvailabilityUpdate(int usbAvailabilityInRouter)
{
   ETG_TRACE_USR4(("ExternalMediaClientHandler::onRouterSourceAvailabilityUpdate usbAvailabilityInRouter  %d", usbAvailabilityInRouter));
   if (usbAvailabilityInRouter == EN_EXTERNAL_MEDIA_DISCONNECTED)
   {
      m_isRouterSourceActive = false;
      DbDownloadMap::getInstance().setDbDownloadState(EnumConst::entSourceType_CMR, EnumConst::entDownloadState_Default);
      DbDownloadMap::getInstance().setDbDownloadRequested(EnumConst::entSourceType_CMR, false);
      _playableContentMap->setPlayableAudioContent(EnumConst::entSourceType_CMR, false);
      _playableContentMap->setPlayableVideoContent(EnumConst::entSourceType_CMR, false);
   }
}


void ExternalMediaClientHandler::onRouterSourceAllocationUpdate(int iSourceId, int iSinkId, int connectionState)
{
   ETG_TRACE_USR4(("ExternalMediaClientHandler::onRouterSourceAllocationUpdate  iSinkId %d  iSourceId %d  connectionState %d ", iSinkId, iSourceId, connectionState));
   int iActiveCabinSrcId = MediaSourceHandling::getInstance().getCurrentRegionSrcId();
   int prevCabinSrcid = MediaSourceHandling::getInstance().getPrevSrcId();
   ETG_TRACE_USR4(("ExternalMediaClientHandler::onRouterSourceAllocationUpdate iActiveCabinSrcId %d ", iActiveCabinSrcId));
   IPluginClientAppl* l_iPluginClientAppl = PluginClientFactory::getPluginClient(iSourceId);
   if (iSinkId == CABIN_A_SINK_ID || iSinkId == CABIN_B_SINK_ID)
   {
      if (connectionState == STATE_CONNECTED)
      {
         m_isRouterSourceActive = true;
         _playableContentInterfaceHandler->onCRCStatusReceived(EnumConst::entSourceType_CMR, EN_CRC_MATCH);
         ExternalMediaPlaybackHandler::getInstance()->updatePlaybackState();
         ExternalMediaPlaybackHandler::getInstance()->updateOperationType();
         if (iSourceId == SRC_CMR1_USB1_VIDEO)
         {
            if (_isHmiOffWithActiveMediaSrc == true)
            {
               if (l_iPluginClientAppl != NULL)
               {
                  ExternalMediaPlaybackHandler::getInstance()->setPlaybackState(MEDIA_VIDEOSOURCERESTORE);
                  ExternalMediaPlaybackHandler::getInstance()->setOperationType(EN_MEDIA_PAUSE);
                  l_iPluginClientAppl->sendPluginRequest(MEDIA_VIDEOSOURCERESTORE, iSourceId);
                  _isHmiOffWithActiveMediaSrc = false;
               }
            }
            else
            {
               if (l_iPluginClientAppl != NULL)
               {
                  ETG_TRACE_USR4(("ExternalMediaClientHandler::prevCabinSrcid:%d iActiveCabinSrcId:%d ", prevCabinSrcid, iActiveCabinSrcId));
                  if (iActiveCabinSrcId != prevCabinSrcid)
                  {
                     l_iPluginClientAppl->sendPluginRequest(getExternalSrcPlaybackState(iSourceId), iSourceId);
                  }
                  else
                  {
                     l_iPluginClientAppl->sendPluginRequest((ExternalMediaPlaybackHandler::getInstance()->getPlaybackState()), iSourceId);
                  }
               }
            }
         }
         else
         {
            if (l_iPluginClientAppl != NULL)
            {
               l_iPluginClientAppl->sendPluginRequest(getExternalSrcPlaybackState(iSourceId), iSourceId);
            }
         }
      }
      else if ((connectionState == STATE_DISCONNECTED) || (connectionState == STATE_SUSPENDED))
      {
         /* if (l_iPluginClientAppl != NULL)
          {
             l_iPluginClientAppl->sendPluginRequest(EXT_MEDIA_PAUSE, iSourceId);  // pause request is blocked for bug 2054844 which will be revisited when router source is enabled
          }*/

         if (connectionState == STATE_DISCONNECTED)
         {
            m_isRouterSourceActive = false;
            if (iActiveCabinSrcId == SRC_CMR1_USB1_VIDEO  || iActiveCabinSrcId == SRC_CMR1_USB1_AUDIO)
            {
               MediaSourceHandling::getInstance().vResetAudioSrcMapofCabin(iActiveCabinSrcId);
            }
         }
      }
      else
      {
         ETG_TRACE_USR4(("ExternalMediaClientHandler::onRouterSourceAllocationUpdate Default "));
      }
   }
}


bool ExternalMediaClientHandler::getCurrentRouterActiveState()
{
   ETG_TRACE_USR4(("ExternalMediaClientHandler::getCurrentRouterActiveState  m_isRouterSourceActive %d  ", m_isRouterSourceActive));
   return m_isRouterSourceActive;
}


void ExternalMediaClientHandler::setSystemInfoMapInstance(SystemInfoMap* _systemInfoMapInstance)
{
   if (_systemInfoMapInstance != NULL)
   {
      _systemInfoMap = _systemInfoMapInstance;
   }
   else
   {
      ETG_TRACE_USR4(("ExternalMediaClientHandler::_systemInfoMapInstance is NULL"));
   }
}


bool ExternalMediaClientHandler::onCourierBindingItemChange_USBCameraPortPlaybackActionItem(const Courier::Request& request)
{
   uint8 u8activeRegion = ApplicationSwitchClientHandler::poGetInstance()->getActiveRegion();
   ETG_TRACE_USR4(("ExternalMediaClientHandler::onCourierBindingItemChange_USBCameraPortPlaybackActionItem in external u8activeRegion  %d ", u8activeRegion));
   int srcID = MediaSourceHandling::getInstance().getCurrentRegionSrcId();
   ETG_TRACE_USR4(("ExternalMediaClientHandler::onCourierBindingItemChange_USBCameraPortPlaybackActionItem getCurrentRegionSrcId: srcID %d", srcID));
   if ((u8activeRegion == 0) && (srcID == SRC_CAMERAPORT1_USB_VIDEO || srcID == SRC_CAMERAPORT1_USB_AUDIO))
   {
      m_USBCameraPortMediaPlayBackData.SetValue(request.ItemKey(), request.GetItemValue());
      IPluginClientAppl* l_iPluginClientAppl = PluginClientFactory::getPluginClient(srcID);
      unsigned int seektime = static_cast<unsigned int>((*m_USBCameraPortMediaPlayBackData).mSliderElapsedTime);
      if (l_iPluginClientAppl != NULL)
      {
         double m_Totalslidervalue = l_iPluginClientAppl->getSliderValue();
         ETG_TRACE_USR4(("ExternalMediaClientHandler::onCourierBindingItemChange_USBCameraPortPlaybackActionItem  seektime %d ,m_Totalslidervalue   %d", seektime, m_Totalslidervalue));
         float seekValue = (float)seektime / (float)m_Totalslidervalue;
         ETG_TRACE_USR4(("ExternalMediaClientHandler::onCourierBindingItemChange_USBCameraPortPlaybackActionItem seekValue %f", seekValue));
         int seekPercent = (seekValue * 100);
         ETG_TRACE_USR4(("ExternalMediaClientHandler::onCourierBindingItemChange_USBCameraPortPlaybackActionItem seekPercent %d", seekPercent));
         l_iPluginClientAppl->updateSeekToStreamType(seekPercent);
      }
   }
   return true;
}


bool ExternalMediaClientHandler::onCourierBindingItemChange_USBRouterPlaybackActionItem(const Courier::Request& request)
{
   int srcID = MediaSourceHandling::getInstance().getCurrentRegionSrcId();
   ETG_TRACE_USR4(("ExternalMediaClientHandler::onCourierBindingItemChange_USBRouterPlaybackActionItem getCurrentRegionSrcId: srcID %d", srcID));
   if (srcID == SRC_CMR1_USB1_VIDEO || srcID == SRC_CMR1_USB1_AUDIO)
   {
      m_USBRouterMediaPlayBackData.SetValue(request.ItemKey(), request.GetItemValue());
      IPluginClientAppl* l_iPluginClientAppl = PluginClientFactory::getPluginClient(srcID);
      unsigned int seektime = static_cast<unsigned int>((*m_USBRouterMediaPlayBackData).mSliderElapsedTime);
      if (l_iPluginClientAppl != NULL)
      {
         double m_Totalslidervalue = l_iPluginClientAppl->getSliderValue();
         ETG_TRACE_USR4(("ExternalMediaClientHandler::onCourierBindingItemChange_USBRouterPlaybackActionItem  seektime %d ,m_Totalslidervalue   %d", seektime, m_Totalslidervalue));
         float seekValue = (float)seektime / (float)m_Totalslidervalue;
         ETG_TRACE_USR4(("ExternalMediaClientHandler::onCourierBindingItemChange_USBRouterPlaybackActionItem seekValue %f", seekValue));
         int seekPercent = (seekValue * 100);
         ETG_TRACE_USR4(("ExternalMediaClientHandler::onCourierBindingItemChange_USBRouterPlaybackActionItem seekPercent %d", seekPercent));
         l_iPluginClientAppl->updateSeekToStreamType(seekPercent);
      }
   }
   return true;
}


bool ExternalMediaClientHandler::onCourierMessage(const CurrentMediaStateUpdMsg& msg)
{
   _currentMediaState = msg.GetCurrentState();
   ETG_TRACE_USR4(("ExternalMediaClientHandler::CurrentMediaStateUpdMsg  _currentMediaState %d", _currentMediaState));

   return false;
}


int ExternalMediaClientHandler::getHdmiSourceAvailability(int srcId)
{
   int SrcAvailability = EN_EXTERNAL_MEDIA_DISCONNECTED;
   std::map < int, HdmiSourceInfo>::iterator  itr = _hdmiSrcAvailability.find(srcId);

   if (itr != _hdmiSrcAvailability.end())
   {
      SrcAvailability = itr->second.sourceAvailability;
   }
   ETG_TRACE_USR4(("ExternalMediaClientHandler::getHdmiSourceAvailability  srcId %d    SrcAvailability   %d", srcId, SrcAvailability));
   return SrcAvailability;
}


bool ExternalMediaClientHandler::getHdmiSourceActiveState(int srcId)
{
   bool activeState = false;
   std::map < int, HdmiSourceInfo>::iterator  itr = _hdmiSrcAvailability.find(srcId);

   if (itr != _hdmiSrcAvailability.end())
   {
      activeState = itr->second.isSourceActive;
   }
   ETG_TRACE_USR4(("ExternalMediaClientHandler::getHdmiSourceActiveState  srcId %d    activeState   %d", srcId, activeState));
   return activeState;
}


void ExternalMediaClientHandler::setHdmiActiveState(int srcId, bool activeHdmiState)
{
   ETG_TRACE_USR4(("ExternalMediaClientHandler::setHdmiActiveState  srcId %d    activeHdmiState   %d", srcId, activeHdmiState));
   std::map <int, HdmiSourceInfo>::iterator  itr = _hdmiSrcAvailability.find(srcId);

   if (itr != _hdmiSrcAvailability.end())
   {
      itr->second.isSourceActive = activeHdmiState;
   }
}


void ExternalMediaClientHandler::resetMetadataForExternalUsbSources(int srcId)
{
   ETG_TRACE_USR4(("ExternalMediaClientHandler::resetMetadataForExternalUsbSources  srcId %d  ", srcId));
   IPluginClientAppl* l_iPluginClientAppl = PluginClientFactory::getPluginClient(srcId);
   if (l_iPluginClientAppl != NULL)
   {
      l_iPluginClientAppl->resetMetadata();
   }
}


void ExternalMediaClientHandler::onHmiStateUpdate(uint8 currentHmiState)
{
   int srcId = MediaSourceHandling::getInstance().getCurrentRegionSrcId();
   if ((currentHmiState == T_SPM_e32_HMI_STATE__SPM_U32_HMISTATE_SHOW_OFF) && (MediaSourceHandling::getInstance().IsMediaSourceRange(srcId)))
   {
      _isHmiOffWithActiveMediaSrc = true;
   }
   ETG_TRACE_USR1(("ExternalMediaClientHandler::onHmiStateUpdate  currentHmiState %d, _isHmiOffWithActiveMediaSrc = %d", currentHmiState, _isHmiOffWithActiveMediaSrc));
}


void ExternalMediaClientHandler::setHmiOffWithActiveMediaSrcState(bool state)
{
   ETG_TRACE_COMP(("ExternalMediaClientHandler::setHmiOffWithActiveMediaSrcState  state %d", state));
   _isHmiOffWithActiveMediaSrc = state;
}


}// namespace Core
}// namespace App
