/***************************************************************************
* Copyright(c) 2018-2020 Robert Bosch Car Multimedia GmbH
* This software is property of Robert Bosch GmbH.
***************************************************************************/
#include "CabinSystemStateHandler.h"
#include "ProjectBaseExchangingTextures.h"
#include "hmi_trace_if.h"
#include "hall_std_if.h"
#include <asf/core/Types.h>
#include <map>
#include <vector>
#include <iterator>
#include "spm_core_fi_typesConst.h"
#include "AppHmi_MasterBase/AudioInterface/AudioDefines.h"
#include "Core/SourceSwitch/MediaSourceHandling.h"
#include "Core/ApplicationSwitchClientHandler/ApplicationSwitchClientHandler.h"
#include "Core/MediaClientHandler/MediaClientHandler.h"
#include "ProjectBaseTypes.h"

#ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
#define ETG_DEFAULT_TRACE_CLASS         TR_CLASS_APPHMI_MEDIA_HALL
#define ETG_I_TRACE_CHANNEL             TR_TTFIS_APPHMI_MEDIA
#define ETG_I_TTFIS_CMD_PREFIX          "APPHMI_MEDIA_"
#define ETG_I_FILE_PREFIX               App::Core::CabinSystemStateHandler::
#include "trcGenProj/Header/CabinSystemStateHandler.cpp.trc.h"
#endif

using namespace ::bosch::cm::ai::hmi::hmiinfoservice::HmiInfoService;
using namespace ::spm_core_fi_types;

namespace App {
namespace Core {

CabinSystemStateHandler::CabinSystemStateHandler()
   : _systemInfoMap(NULL)
{
   _hmiInfoServiceProxyClient = (::HmiInfoServiceProxy::createProxy("hmiinfoservicePort", *this));
   if (_hmiInfoServiceProxyClient.get())
   {
      ETG_TRACE_USR4(("CabinSystemStateHandler::CabinSystemStateHandler StartupSync _hmiInfoProxyClient"));
      StartupSync::getInstance().registerPropertyRegistrationIF(this, _hmiInfoServiceProxyClient->getPortName());
   }
   initialiseCameraportSourceMap();
}


CabinSystemStateHandler::~CabinSystemStateHandler()
{
   ETG_TRACE_USR4(("CabinSystemStateHandler Destructor \n"));
}


void CabinSystemStateHandler::onAvailable(const ::boost::shared_ptr< asf::core::Proxy >& proxy, const ServiceStateChange& stateChange)
{
   ETG_TRACE_COMP(("CabinSystemStateHandler::onAvailable"));

   if (_hmiInfoServiceProxyClient && _hmiInfoServiceProxyClient == proxy)
   {
      _hmiInfoServiceProxyClient->sendCabinsSystemsInfoRegister(*this);
   }
}


void CabinSystemStateHandler::onUnavailable(const ::boost::shared_ptr< asf::core::Proxy >& proxy, const ServiceStateChange& stateChange)
{
   ETG_TRACE_COMP(("CabinSystemStateHandler::onUnavailable"));

   if (_hmiInfoServiceProxyClient && _hmiInfoServiceProxyClient == proxy)
   {
      _hmiInfoServiceProxyClient->sendCabinsSystemsInfoDeregisterAll();
   }
}


void CabinSystemStateHandler::registerProperties(const ::boost::shared_ptr< asf::core::Proxy >& proxy, const asf::core::ServiceStateChange& /*stateChange*/)
{
   ETG_TRACE_COMP(("CabinSystemStateHandler::registerProperties"));
}


void CabinSystemStateHandler::deregisterProperties(const ::boost::shared_ptr< asf::core::Proxy >& proxy, const asf::core::ServiceStateChange& /*stateChange*/)
{
   ETG_TRACE_COMP(("CabinSystemStateHandler::deregisterProperties"));
}


void CabinSystemStateHandler::setSystemInfoMapInstance(SystemInfoMap* _systemInfoMapInstance)
{
   if (_systemInfoMapInstance != NULL)
   {
      _systemInfoMap = _systemInfoMapInstance;
   }
   else
   {
      ETG_TRACE_USR4(("CabinSystemStateHandler::_systemInfoMapInstance is NULL"));
   }
}


void CabinSystemStateHandler::initialiseCameraportSourceMap()
{
   ETG_TRACE_USR4(("CabinSystemStateHandler::initialiseCameraportSourceMap is entered \n"));
   _externalSourceMap.insert(pair<int, int>(SRC_CAMERAPORT1_USB_VIDEO, SYS_CATEGORY_CMP1));
   _externalSourceMap.insert(pair<int, int>(SRC_CAMERAPORT1_USB_AUDIO, SYS_CATEGORY_CMP1));
   _externalSourceMap.insert(pair<int, int>(SRC_CAMERAPORT1_HDMI, SYS_CATEGORY_CMP1));
   _externalSourceMap.insert(pair<int, int>(SRC_CAMERAPORT2_HDMI, SYS_CATEGORY_CMP2));
   _externalSourceMap.insert(pair<int, int>(SRC_CAMERAPORT3_HDMI, SYS_CATEGORY_CMP3));
   _externalSourceMap.insert(pair<int, int>(SRC_CMR1_USB1_VIDEO, SYS_CATEGORY_CMR));
   _externalSourceMap.insert(pair<int, int>(SRC_CMR1_USB1_AUDIO, SYS_CATEGORY_CMR));
}


/************************************************************************
*FUNCTION		: onCabinsSystemsInfoError
*DESCRIPTION	: To update cabins systems information
*PARAMETER		: NA
*PARAMETER		: NA
*RETURNVALUE	: void
*SWFL			: [Task 1145740]
************************************************************************/


void CabinSystemStateHandler::onCabinsSystemsInfoError(const ::boost::shared_ptr< HmiInfoServiceProxy >& /*proxy*/,
      const ::boost::shared_ptr< CabinsSystemsInfoError >& error)
{
   ETG_TRACE_COMP(("CabinSystemStateHandler::onCabinsSystemsInfoError"));
}


/************************************************************************
*FUNCTION		: onCabinsSystemsInfoUpdate
*DESCRIPTION	: To update cabins systems information
*PARAMETER		: NA
*PARAMETER		: NA
*RETURNVALUE	: void
*SWFL			: [Task 1145740]
************************************************************************/

void CabinSystemStateHandler::onCabinsSystemsInfoUpdate(const ::boost::shared_ptr< HmiInfoServiceProxy >& /*proxy*/,
      const ::boost::shared_ptr< CabinsSystemsInfoUpdate >& update)
{
   ETG_TRACE_COMP(("CabinSystemStateHandler::onCabinsSystemsInfoUpdate"));
   uint8 prev_SysState;
   int current_ActFlag;
   if (update != NULL)
   {
      std::vector<SystemInfo>cabinSystemInfo = update->getCabinsSystemsInfo();
      if (!cabinSystemInfo.empty())
      {
         std::vector<SystemInfo>::const_iterator iter = cabinSystemInfo.begin();
         current_ActFlag = EN_ACKNOWLEDGEMENT_NEW;
         while (iter != cabinSystemInfo.end())
         {
            SystemCategory iSystemCategory = static_cast <SystemCategory>(iter->getSystemCategory());
            uint8 iSystemStatus = iter->getSystemStatus();
            uint8 iSystemPosition = iter->getSystemPosition();
            uint8 iSystemGroup = iter->getSystemGroup();
            ETG_TRACE_USR4(("CabinSystemStateHandler onCabinsSystemsInfoUpdate systemCategory:%d systemStatus:%d systemPosition:%d current_ActFlag:%d",
                            ETG_CENUM(SystemCategory, iSystemCategory), ETG_CENUM(enSystemStatus, iSystemStatus), iSystemPosition, current_ActFlag));

            if (iter->getSystemCategory() != SYSTEM_CATEGORY_NONE)
            {
               if (iter->getSystemCategory() == SYSTEM_CATEGORY_CMP)
               {
                  if (iSystemPosition == EN_POSITION_PRIMARY)
                  {
                     iSystemCategory  = SYS_CATEGORY_CMP1;
                  }
                  else if (iSystemPosition == EN_POSITION_SECONDARY)
                  {
                     iSystemCategory  = SYS_CATEGORY_CMP2;
                  }
                  else if (iSystemPosition == EN_POSITION_TERTIARY)
                  {
                     iSystemCategory  = SYS_CATEGORY_CMP3;
                  }
               }
               else if (iter->getSystemCategory() == SYSTEM_CATEGORY_CMG)
               {
                  if (iSystemGroup == SYSTEM_GROUP_CMG_A)
                  {
                     iSystemCategory = SYS_CATEGORY_CMGA;
                     prev_SysState = _systemInfoMap->getSystemStatus(SYS_CATEGORY_CMGA);
                     ETG_TRACE_USR4(("CabinSystemStateHandler onCabinsSystemsInfoUpdate systemCategory  %d", prev_SysState));
                     if ((prev_SysState == SYSTEM_STATUS_CONNECTED) && (current_ActFlag == EN_ACKNOWLEDGEMENT_ONGOING))
                     {
                        iSystemStatus = SYSTEM_STATUS_CONNECTED;
                     }
                  }
                  else if (iSystemGroup == SYSTEM_GROUP_CMG_A2)
                  {
                     iSystemCategory = SYS_CATEGORY_CMGA2;
                  }
                  current_ActFlag = EN_ACKNOWLEDGEMENT_ONGOING;
               }
               _systemInfoMap->setSystemStatus(iSystemCategory, iSystemStatus);
            }
            iter++;
         }
         onCabinSystemInfoUpdate();
         notifyCabinSystemInfoUpdateListeners();
      }
   }
}


bool CabinSystemStateHandler::isValidSubSystemDisconnectState(uint8 systemStatus)
{
   ETG_TRACE_USR4(("CabinSystemStateHandler::isValidSubSystemDisconnectState systemStatus:%d", ETG_CENUM(enSystemStatus, systemStatus)));

   if (systemStatus != SYSTEM_STATUS_CONNECTED && systemStatus != SYSTEM_STATUS_COCKPIT_DISCONNECTED &&
         systemStatus != SYSTEM_STATUS_NONE)
   {
      return true;
   }
   else
   {
      return false;
   }
}


void CabinSystemStateHandler::onCabinSystemInfoUpdate()
{
   //uint8 iActiveRegion = ApplicationSwitchClientHandler::poGetInstance()->getActiveRegion();
   int iActiveCabinSrcId = MediaSourceHandling::getInstance().getSinkSrcId(CABIN_A_SINK_ID);
   int iActiveGlass1SourceId = MediaSourceHandling::getInstance().getSinkSrcId(CABIN_A_MEDIAGLASS1_SINK_ID);
   int iActiveGlass2SourceId = MediaSourceHandling::getInstance().getSinkSrcId(CABIN_A_MEDIAGLASS2_SINK_ID);
   ETG_TRACE_USR4(("CabinSystemStateHandler::onCabinSystemInfoUpdate CabinSrcId : %d Glass1SrcId : %d  Glass2rcId : %d", iActiveCabinSrcId, iActiveGlass1SourceId, iActiveGlass2SourceId));
   if (iActiveCabinSrcId != SRC_INVALID)
   {
      if (isValidSubSystemDisconnectState(_systemInfoMap->getSystemStatus(SYS_CATEGORY_CMA)) == true)
      {
         ETG_TRACE_USR4(("CabinSystemStateHandler::onCabinSystemInfoUpdate CMA systemStatus:%d",
                         ETG_CENUM(enSystemStatus, _systemInfoMap->getSystemStatus(SYS_CATEGORY_CMA))));
         MediaSourceHandling::getInstance().vRequestSourceDeactivation(iActiveCabinSrcId, -1, REGION_CABIN_A);
      }
      else if (iActiveCabinSrcId != SRC_MEDIA_PLAYER && iActiveCabinSrcId != SRC_PHONE_BTAUDIO)
      {
         if ((isValidSubSystemDisconnectState(_systemInfoMap->getSystemStatus(SYS_CATEGORY_CMR)) == true)
               && (iActiveCabinSrcId == SRC_CMR1_USB1_VIDEO || iActiveCabinSrcId == SRC_CMR1_USB1_AUDIO))
         {
            ETG_TRACE_USR4(("CabinSystemStateHandler::onCabinSystemInfoUpdate CMR systemStatus:%d",
                            ETG_CENUM(enSystemStatus, _systemInfoMap->getSystemStatus(SYS_CATEGORY_CMR))));
            MediaSourceHandling::getInstance().vRequestSourceDeactivation(iActiveCabinSrcId, -1, REGION_CABIN_A);
         }
         else if ((isValidSubSystemDisconnectState(_systemInfoMap->getSystemStatus(SYS_CATEGORY_CMP1)) == true)
                  && (iActiveCabinSrcId != SRC_CMR1_USB1_VIDEO && iActiveCabinSrcId != SRC_CMR1_USB1_AUDIO))
         {
            ETG_TRACE_USR4(("CabinSystemStateHandler::onCabinSystemInfoUpdate CMP1 systemStatus:%d",
                            ETG_CENUM(enSystemStatus, _systemInfoMap->getSystemStatus(SYS_CATEGORY_CMP1))));
            MediaSourceHandling::getInstance().vRequestSourceDeactivation(iActiveCabinSrcId, -1, REGION_CABIN_A);
         }
         else if ((isValidSubSystemDisconnectState(_systemInfoMap->getSystemStatus(SYS_CATEGORY_CMP2)) == true)
                  && (_externalSourceMap[iActiveCabinSrcId] == SYS_CATEGORY_CMP2))
         {
            ETG_TRACE_USR4(("CabinSystemStateHandler::onCabinSystemInfoUpdate CMP2 systemStatus:%d",
                            ETG_CENUM(enSystemStatus, _systemInfoMap->getSystemStatus(SYS_CATEGORY_CMP2))));
            MediaSourceHandling::getInstance().vRequestSourceDeactivation(iActiveCabinSrcId, -1, REGION_CABIN_A);
         }
         else if ((isValidSubSystemDisconnectState(_systemInfoMap->getSystemStatus(SYS_CATEGORY_CMP3)) == true)
                  && (_externalSourceMap[iActiveCabinSrcId] == SYS_CATEGORY_CMP3))
         {
            ETG_TRACE_USR4(("CabinSystemStateHandler::onCabinSystemInfoUpdate CMP3 systemStatus:%d",
                            ETG_CENUM(enSystemStatus, _systemInfoMap->getSystemStatus(SYS_CATEGORY_CMP3))));
            MediaSourceHandling::getInstance().vRequestSourceDeactivation(iActiveCabinSrcId, -1, REGION_CABIN_A);
         }
         else if (isValidSubSystemDisconnectState(_systemInfoMap->getSystemStatus(SYS_CATEGORY_CMGA)) == true)
         {
            ETG_TRACE_USR4(("CabinSystemStateHandler::onCabinSystemInfoUpdate CMPGA systemStatus:%d",
                            ETG_CENUM(enSystemStatus, _systemInfoMap->getSystemStatus(SYS_CATEGORY_CMGA))));
            if (iActiveCabinSrcId ==  SRC_CAMERAPORT1_USB_VIDEO || iActiveCabinSrcId == SRC_CAMERAPORT1_HDMI ||
                  iActiveCabinSrcId == SRC_CAMERAPORT2_HDMI || iActiveCabinSrcId == SRC_CAMERAPORT3_HDMI ||
                  iActiveCabinSrcId == SRC_CMR1_USB1_VIDEO)
            {
               MediaSourceHandling::getInstance().vRequestSourceDeactivation(iActiveCabinSrcId, -1, REGION_CABIN_A);
            }
         }
      }
   }
   if ((iActiveGlass1SourceId == SRC_CAMERAPORT1_USB_VIDEO) || (iActiveGlass1SourceId == SRC_CMR1_USB1_VIDEO))
   {
      if ((MediaSourceHandling::getInstance().getMic1Status() == true)
            && (isValidSubSystemDisconnectState(_systemInfoMap->getSystemStatus(SYS_CATEGORY_CMA)) == true))
      {
         MediaSourceHandling::getInstance().vRequestGlassSourceDeactivation(iActiveGlass1SourceId, CABIN_A_MEDIAGLASS1_SINK_ID);
         ETG_TRACE_USR4(("CabinSystemStateHandler::onCabinSystemInfoUpdate Source Deactivation in Glass"));
      }
   }
   if ((iActiveGlass2SourceId == SRC_CAMERAPORT1_USB_VIDEO) || (iActiveGlass2SourceId == SRC_CMR1_USB1_VIDEO))
   {
      if ((MediaSourceHandling::getInstance().getMic1Status() == true)
            && (isValidSubSystemDisconnectState(_systemInfoMap->getSystemStatus(SYS_CATEGORY_CMA)) == true))
      {
         MediaSourceHandling::getInstance().vRequestGlassSourceDeactivation(iActiveGlass2SourceId, CABIN_A_MEDIAGLASS2_SINK_ID);
         ETG_TRACE_USR4(("CabinSystemStateHandler::onCabinSystemInfoUpdate Source Deactivation in Glass"));
      }
   }
}


void  CabinSystemStateHandler::addCabinSystemInfoUpdateObserver(ICabinSystemInfoUpdateObserver& imp)
{
   ETG_TRACE_USR1(("CabinSystemStateHandler::addCabinSystemInfoUpdateObserver is called"));

   std::vector< ICabinSystemInfoUpdateObserver* >::const_iterator itr = std::find(_cabinSystemInfoUpdateListener.begin(), _cabinSystemInfoUpdateListener.end(), (&imp));
   if (itr == _cabinSystemInfoUpdateListener.end())
   {
      _cabinSystemInfoUpdateListener.push_back((&imp));
   }
}


void  CabinSystemStateHandler::removeCabinSystemInfoUpdateObserver(ICabinSystemInfoUpdateObserver& imp)
{
   ETG_TRACE_USR1(("CabinSystemStateHandler::removeCabinSystemInfoUpdateObserver is called"));

   std::vector< ICabinSystemInfoUpdateObserver* >::iterator itr = std::find(_cabinSystemInfoUpdateListener.begin(), _cabinSystemInfoUpdateListener.end(), (&imp));
   if (itr != _cabinSystemInfoUpdateListener.end())
   {
      (void)_cabinSystemInfoUpdateListener.erase(itr);
   }
}


void CabinSystemStateHandler::notifyCabinSystemInfoUpdateListeners()
{
   ETG_TRACE_USR1(("CabinSystemStateHandler::notifyCabinSystemInfoUpdateListeners is called"));

   for (std::vector<ICabinSystemInfoUpdateObserver* >::const_iterator itr = _cabinSystemInfoUpdateListener.begin(); (itr != _cabinSystemInfoUpdateListener.end()); ++itr)
   {
      if (NULL != (*itr))
      {
         (*itr)->onCabinSystemUpdate();
      }
   }
}


}
}
