/**************************************************************************************
* @file         : CabinSourceRestoration.cpp
* @author       : ECG - vta5kor
* @addtogroup   : AppHmi_media
* @brief        :
* @copyright    : (c) 2018-2019 Robert Bosch Car Multimedia GmbH
*                 The reproduction, distribution and utilization of this file as
*                 well as the communication of its contents to others without express
*                 authorization is prohibited. Offenders will be held liable for the
*                 payment of damages. All rights reserved in the event of the grant
*                 of a patent, utility model or design.
**************************************************************************************/
#include "hall_std_if.h"
#include "CabinSourceRestoration.h"


#define UTFUTIL_S_IMPORT_INTERFACE_GENERIC
#include "utf_if.h"
#include "AppHmi_MasterBase/AudioInterface/AudioDefines.h"
#include "ProjectBaseTypes.h"
#include "Core/Utils/DbDownloadMap.h"
#include "../ApplicationSwitchClientHandler/ApplicationSwitchClientHandler.h"
#include "Core/SourceSwitch/MediaSourceHandling.h"
#include "Core/Utils/MediaUtils.h"
#include "Core/ExternalMediaDefines.h"
#include "DataModel/HMIModelComponent.h"
#ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
#define ETG_DEFAULT_TRACE_CLASS         TR_CLASS_APPHMI_MEDIA_HALL
#define ETG_I_TRACE_CHANNEL             TR_TTFIS_APPHMI_MEDIA
#define ETG_I_TTFIS_CMD_PREFIX          "APPHMI_MEDIA_"
#define ETG_I_FILE_PREFIX               App::Core::CabinSourceRestoration::

#include "trcGenProj/Header/CabinSourceRestoration.cpp.trc.h"
#endif


using namespace ::App::Core;
using namespace ::bosch::cm::ai::hmi::masteraudioservice::AudioSourceChange;

namespace App {
namespace Core {

/**
 * @Destructor
 */
CabinSourceRestoration::~CabinSourceRestoration()
{
   ETG_TRACE_USR1(("CabinSourceRestoration::CabinSourceRestoration destructor"));
   if (_externalMediaSourceHandler != NULL)
   {
      _externalMediaSourceHandler->removeRouterSourceAvailabilityInfoObservor((*this));
      _externalMediaSourceHandler->removeCamportUsbSourceAvailabilityObservor((*this));
      _externalMediaSourceHandler->removeHdmiSourceAvailabilityInfoObservor((*this));
   }
   if (_externalMediaListDBManager != NULL)
   {
      _externalMediaListDBManager->deregisterExternalDBDownloadObservor((*this));
   }
}


/**
* @Constructor
*/
CabinSourceRestoration::CabinSourceRestoration()
   : _externalMediaSourceHandler(NULL)
   , _playableContentMap(NULL)
   , _systemInfoMap(NULL)
   , _externalMediaListDBManager(NULL)
   , _externalMediaClientHandler(NULL)

{
   _mediaPlayerProxy = (::mplay_MediaPlayer_FI::Mplay_MediaPlayer_FIProxy::createProxy("mediaPlayerFiPort", *this));
   _audioSourceChangeProxy = (::AudioSourceChangeProxy::createProxy("audioSourceChangePort", *this));
   _hmiInfoServiceProxyClient = (::HmiInfoServiceProxy::createProxy("hmiinfoservicePort", *this));

   if (_mediaPlayerProxy.get())
   {
      StartupSync::getInstance().registerPropertyRegistrationIF(this, _mediaPlayerProxy->getPortName());
   }

   if (_audioSourceChangeProxy.get())
   {
      StartupSync::getInstance().registerPropertyRegistrationIF(this, _audioSourceChangeProxy->getPortName());
   }
   if (_hmiInfoServiceProxyClient.get())
   {
      StartupSync::getInstance().registerPropertyRegistrationIF(this, _hmiInfoServiceProxyClient->getPortName());
   }

   initializeMediaSourceMap();
   _sourceId = SRC_INVALID;
   _subSrcId = SUBSRCID_INVALID;
   _restorationStatus = CABIN_RESTORE_STATUS_NONE;
   _cabinRegion = REGION_CABIN_A;
   _isSourceConnected = false;
   _isSourceRestored = false;
   _sinkId = CABIN_A_SINK_ID;
}


void CabinSourceRestoration::setExternalMediaSourceHandlerInstance(ExternalMediaSourceHandler* _externalMediaSourceHandlerInstance)
{
   if (_externalMediaSourceHandlerInstance != NULL)
   {
      _externalMediaSourceHandler = _externalMediaSourceHandlerInstance;
      _externalMediaSourceHandler->addRouterSourceAvailabilityInfoObservor((*this));
      _externalMediaSourceHandler->addCamportUsbSourceAvailabilityObservor((*this));
      _externalMediaSourceHandler->addHdmiSourceAvailabilityInfoObservor((*this));
   }
   else
   {
      ETG_TRACE_USR1(("CabinSourceRestoration::_externalMediaSourceHandler is NULL"));
   }
}


void CabinSourceRestoration::setExternalMediaListDBManagerInstance(ExternalMediaListDBManager* _externalMediaListDBManagerInstance)
{
   ETG_TRACE_USR4(("CabinSourceRestoration::setExternalMediaListDBManagerInstance is entered"));

   if (_externalMediaListDBManagerInstance != NULL)
   {
      _externalMediaListDBManager = _externalMediaListDBManagerInstance;
      _externalMediaListDBManager->registerExternalDBDownloadObservor((*this));
   }
}


void CabinSourceRestoration::setExternalMediaClientHandlerInstance(ExternalMediaClientHandler* _externalMediaClientHandlerInstance)
{
   if (_externalMediaClientHandlerInstance != NULL)
   {
      _externalMediaClientHandler = _externalMediaClientHandlerInstance;
   }
   else
   {
      ETG_TRACE_USR1(("CabinSourceRestoration::_externalMediaClientHandler is NULL"));
   }
}


void CabinSourceRestoration::setSystemInfoMapInstance(SystemInfoMap* _systemInfoMapInstance)
{
   if (_systemInfoMapInstance != NULL)
   {
      _systemInfoMap = _systemInfoMapInstance;
   }
   else
   {
      ETG_TRACE_USR4(("CabinSourceRestoration::_systemInfoMapInstance is NULL"));
   }
}


void CabinSourceRestoration::setPlayableContentMapInstance(PlayableContentMap* _playableContentMapInstance)
{
   if (_playableContentMapInstance != NULL)
   {
      _playableContentMap = _playableContentMapInstance;
   }
   else
   {
      ETG_TRACE_USR1(("CabinSourceRestoration::_playableContentMapInstance is NULL"));
   }
}


/************************************************************************
*FUNCTION		: registerProperties
*DESCRIPTION	: Trigger property registration to mediaplayer,  called from MediaHall class
*PARAMETER		: proxy
*PARAMETER		: stateChange - state change service for corrosponding  proxy
*RETURNVALUE	: void
************************************************************************/

void CabinSourceRestoration::registerProperties(const ::boost::shared_ptr< asf::core::Proxy >& proxy, const asf::core::ServiceStateChange& /*stateChange*/)
{
   ETG_TRACE_COMP(("CabinSourceRestoration::registerProperties entered"));
}


/************************************************************************
*FUNCTION		: deregisterProperties
*DESCRIPTION	: Trigger property registration to mediaplayer,  called from MediaHall class
*PARAMETER		: proxy
*PARAMETER		: stateChange - state change service for corrosponding  proxy
*RETURNVALUE	: void
************************************************************************/

void CabinSourceRestoration::deregisterProperties(const ::boost::shared_ptr< asf::core::Proxy >& proxy, const asf::core::ServiceStateChange& /*stateChange*/)
{
   ETG_TRACE_COMP(("CabinSourceRestoration::deregisterProperties entered"));
}


/************************************************************************
*FUNCTION		: onUnavailable
*DESCRIPTION	: Method called when MediaPlayer service is not availble
*PARAMETER		: proxy
*PARAMETER		: stateChange - state change service for corrosponding  proxy
*RETURNVALUE	: void
************************************************************************/
void CabinSourceRestoration::onUnavailable(const boost::shared_ptr<asf::core::Proxy>& proxy, const asf::core::ServiceStateChange& /*stateChange*/)
{
   ETG_TRACE_COMP(("CabinSourceRestoration::onUnavailable entered"));

   if ((proxy == _mediaPlayerProxy) && (_mediaPlayerProxy.get()))
   {
      _mediaPlayerProxy->sendMediaPlayerDeviceConnectionsRelUpRegAll();
   }

   if (_audioSourceChangeProxy && _audioSourceChangeProxy == proxy)
   {
      _audioSourceChangeProxy->sendDeregisterAll();
   }
   if (_hmiInfoServiceProxyClient && _hmiInfoServiceProxyClient == proxy)
   {
      _hmiInfoServiceProxyClient->sendCabinsSourceRestorationInfoDeregisterAll();
      _hmiInfoServiceProxyClient->sendRegionsMapOutStatusInfoDeregisterAll();
      _hmiInfoServiceProxyClient->sendRestoreAVSourceDeregisterAll();
   }
}


void CabinSourceRestoration::onAvailable(const ::boost::shared_ptr< asf::core::Proxy >& proxy, const asf::core::ServiceStateChange& /*stateChange*/)
{
   ETG_TRACE_COMP(("CabinSourceRestoration::onAvailable entered"));

   //Add your code here
   // Register for all the properties from the service
   if ((proxy == _mediaPlayerProxy) && (_mediaPlayerProxy.get()))
   {
      _mediaPlayerProxy->sendMediaPlayerDeviceConnectionsUpReg(*this);
   }
   if (_audioSourceChangeProxy && _audioSourceChangeProxy == proxy)
   {
      _audioSourceChangeProxy->sendSourceAvailabilityChangedRegister(*this);
   }

   if (_hmiInfoServiceProxyClient && _hmiInfoServiceProxyClient == proxy)
   {
      _hmiInfoServiceProxyClient->sendCabinsSourceRestorationInfoRegister(*this);
      _hmiInfoServiceProxyClient->sendCabinsSourceRestorationInfoGet(*this);
      _hmiInfoServiceProxyClient->sendRegionsMapOutStatusInfoRegister(*this);
      _hmiInfoServiceProxyClient->sendRegionsMapOutStatusInfoGet(*this);
      _hmiInfoServiceProxyClient->sendRestoreAVSourceRegister(*this);
   }
}


void CabinSourceRestoration::initializeMediaSourceMap()
{
   ETG_TRACE_USR1(("ExternalMediaPlaybackHandler::initializePlaybackActionMap is entered \n"));
   _mediaSourceAvailabilityMap.insert(pair<int, bool>(SRC_CAMERAPORT1_USB_VIDEO, false));
   _mediaSourceAvailabilityMap.insert(pair<int, bool>(SRC_CAMERAPORT1_USB_AUDIO, false));
   _mediaSourceAvailabilityMap.insert(pair<int, bool>(SRC_CAMERAPORT1_HDMI, true));
   _mediaSourceAvailabilityMap.insert(pair<int, bool>(SRC_CAMERAPORT2_HDMI, true));
   _mediaSourceAvailabilityMap.insert(pair<int, bool>(SRC_CAMERAPORT3_HDMI, true));
   _mediaSourceAvailabilityMap.insert(pair<int, bool>(SRC_MEDIA_PLAYER, false));
   _mediaSourceAvailabilityMap.insert(pair<int, bool>(SRC_PHONE_BTAUDIO, false));
   _mediaSourceAvailabilityMap.insert(pair<int, bool>(SRC_CMR1_USB1_VIDEO, false));
   _mediaSourceAvailabilityMap.insert(pair<int, bool>(SRC_CMR1_USB1_AUDIO, false));
}


void CabinSourceRestoration::onCabinsSourceRestorationInfoError(const ::boost::shared_ptr< HmiInfoServiceProxy >& /*proxy*/,
      const ::boost::shared_ptr< CabinsSourceRestorationInfoError >& error)
{
   ETG_TRACE_COMP(("CabinSourceRestoration::onCabinsSourceRestorationInfoError"));
}


void CabinSourceRestoration::onCabinsSourceRestorationInfoUpdate(const ::boost::shared_ptr< HmiInfoServiceProxy >& /*proxy*/,
      const ::boost::shared_ptr< CabinsSourceRestorationInfoUpdate >& update)
{
   ETG_TRACE_COMP(("CabinSourceRestoration::onCabinsSourceRestorationInfoUpdate"));
   if (update != NULL)
   {
      std::vector<CabinSourceRestorationInfo>sourceRestorationInfo = update->getCabinsSourceRestorationInfo();
      if (!sourceRestorationInfo.empty())
      {
         std::vector<CabinSourceRestorationInfo>::const_iterator iter = sourceRestorationInfo.begin();
         while (iter != sourceRestorationInfo.end())
         {
            _cabinRegion = iter->getCabinId();
            std::vector<SourceInfoHolder> lastActiveSrcInfo = iter->getLastActiveSourcesInfo();

            _isSourceConnected = isSourceConnected(lastActiveSrcInfo);
            std::vector<SourceInfoHolder>::const_iterator itr = lastActiveSrcInfo.begin();
            while (itr != lastActiveSrcInfo.end())
            {
               _restorationStatus = itr->getStatus();
               _sinkId = itr->getSinkId();
               if (_sourceId != SRC_INVALID)
               {
                  restoreSourceinCabin(_sourceId, _cabinRegion, _sinkId, _restorationStatus, _isSourceConnected);
               }
               itr++;
            }
            iter++;
         }
      }
      else
      {
         ETG_TRACE_USR1(("CabinSourceRestoration::onCabinsSourceRestorationInfoUpdate sourceRestorationInfo is empty"));
      }
   }
   ETG_TRACE_USR4(("CabinSourceRestoration::onCabinsSourceRestorationInfoUpdate restoreSourceinCabin _sourceId  %d _cabinRegion %d _restorationStatus %d _isSourceConnected %d ",
                   _sourceId, _cabinRegion, _restorationStatus, _isSourceConnected));
}


void CabinSourceRestoration::restoreSourceinCabin(int srcID, uint32 cabinId, int16 sinkId, uint32 restorationStatus, bool sourceConnectionState)
{
   ETG_TRACE_USR4(("CabinSourceRestoration::restoreSourceinCabin _sourceId  %d cabinId %d restorationStatus %d sourceConnectionState %d",
                   srcID, cabinId, restorationStatus, sourceConnectionState));

   uint8 activeRegion = ApplicationSwitchClientHandler::poGetInstance()->getActiveRegion();
   int deviceType = FI_EN_E8DTY_UNKNOWN;
   EnumConst::entSourceType sourceType = EnumConst::entSourceType_DEFAULT ;
   ETG_TRACE_USR4(("CabinSourceRestoration::restoreSourceinCabin activeRegion :%d sinkId : %d ", activeRegion, sinkId));
   if (sinkId == CABIN_A_SINK_ID || sinkId == CABIN_B_SINK_ID)
   {
      if (restorationStatus == CABIN_RESTORE_STATUS_RESTORE)
      {
         if (sourceConnectionState == false)
         {
            _restorationStatus = CABIN_RESTORE_STATUS_NONE;
         }
         else if (sourceConnectionState == true)
         {
            if (srcID == SRC_CAMERAPORT1_USB_VIDEO || srcID == SRC_CAMERAPORT1_USB_AUDIO || srcID == SRC_CMR1_USB1_VIDEO || srcID == SRC_CMR1_USB1_AUDIO)
            {
               sourceType = MediaUtils::getInstance().getSourceType(srcID);
               deviceType = MediaUtils::getInstance().getDeviceType(sourceType);
               if (DbDownloadMap::getInstance().getDbDownloadState(sourceType) ==   EnumConst::entDownloadState_Default)
               {
                  _restorationStatus = CABIN_RESTORE_STATUS_EVALUATING;
                  _isSourceRestored =  true;
               }
               else if (DbDownloadMap::getInstance().getDbDownloadState(sourceType) !=   EnumConst::entDownloadState_Downloadsuccess)

               {
                  _restorationStatus = CABIN_RESTORE_STATUS_NONE;
               }
               else if ((_systemInfoMap->getSystemStatus(SYS_CATEGORY_CMGA) == SYSTEM_STATUS_UNAVAILABLE || _systemInfoMap->getSystemStatus(SYS_CATEGORY_CMGA) == SYSTEM_STATUS_DISCONNECTED)
                        && (srcID == SRC_CAMERAPORT1_USB_VIDEO || srcID == SRC_CMR1_USB1_VIDEO))
               {
                  _restorationStatus = CABIN_RESTORE_STATUS_NONE;
                  POST_MSG_NOTRACE((COURIER_MESSAGE_NEW(Popup_Pfo_MediaGlassUnavailableMessage_Open)()));
               }
               else
               {
                  if (_playableContentMap->getCRCStatus(sourceType) == EN_CRC_MATCH)

                  {
                     IExternalMediaSourceActivationHandler* iExternalMediaSourceActivationHandler = ExternalMediaSourceActivationFactory::getExternalActivationSourceHandlerObject(deviceType);

                     if (iExternalMediaSourceActivationHandler->getSourceIDFromActiveContent() != srcID)
                     {
                        iExternalMediaSourceActivationHandler->toggleSource();
                     }
                     if (srcID == SRC_CAMERAPORT1_USB_VIDEO || srcID == SRC_CMR1_USB1_VIDEO)
                     {
                        _externalMediaClientHandler->setHmiOffWithActiveMediaSrcState(true);
                     }

                     if (srcID == SRC_CAMERAPORT1_USB_AUDIO || srcID == SRC_CAMERAPORT1_USB_VIDEO) // Temporarily added as part of 1606014 - for disabling Router USB
                     {
                        iExternalMediaSourceActivationHandler->requestSourceActivation(cabinId);
                     }
                  }
                  else
                  {
                     _restorationStatus = CABIN_RESTORE_STATUS_NONE;
                  }
               }
            }
            else if (srcID == SRC_CAMERAPORT1_HDMI || srcID == SRC_CAMERAPORT2_HDMI || srcID == SRC_CAMERAPORT3_HDMI)
            {
               if (_systemInfoMap->getSystemStatus(SYS_CATEGORY_CMGA) == SYSTEM_STATUS_UNAVAILABLE || _systemInfoMap->getSystemStatus(SYS_CATEGORY_CMGA) == SYSTEM_STATUS_DISCONNECTED)
               {
                  _restorationStatus = CABIN_RESTORE_STATUS_NONE;
                  POST_MSG_NOTRACE((COURIER_MESSAGE_NEW(Popup_Pfo_MediaGlassUnavailableMessage_Open)()));
               }
               else
               {
                  MediaSourceHandling::getInstance().requestSourceActivation(srcID, _subSrcId, cabinId);
               }
            }
            else
            {
               MediaSourceHandling::getInstance().requestSourceActivation(srcID, _subSrcId, cabinId);
            }
         }
      }
      else if (restorationStatus == CABIN_RESTORE_STATUS_EVALUATING)
      {
         if ((_systemInfoMap->getSystemStatus(SYS_CATEGORY_CMGA) == SYSTEM_STATUS_UNAVAILABLE || _systemInfoMap->getSystemStatus(SYS_CATEGORY_CMGA) == SYSTEM_STATUS_DISCONNECTED) && (srcID == SRC_CAMERAPORT1_USB_VIDEO || srcID == SRC_CMR1_USB1_VIDEO))
         {
            _restorationStatus = CABIN_RESTORE_STATUS_NONE;
         }
         else
         {
            _restorationStatus = CABIN_RESTORE_STATUS_EVALUATING;
         }
      }
      else
      {
         ETG_TRACE_USR1(("CabinSourceRestoration::restoreSourceinCabin Restore Status None"));
      }
      updateMediaGadgetIndex(_restorationStatus);
   }
   ETG_TRACE_USR4(("CabinSourceRestoration::restoreSourceinCabin deviceType sourceType %d %d ", deviceType, ETG_CENUM(entSourceType, sourceType)));
}


bool CabinSourceRestoration::isSourceConnected(std::vector<SourceInfoHolder>SourcesInfo)
{
   ETG_TRACE_USR1(("CabinSourceRestoration::isSourceConnected is entered"));

   _sourceId = SRC_INVALID;
   bool sourceConnectionState = false;
   if (!SourcesInfo.empty())
   {
      std::vector<SourceInfoHolder>::const_iterator srcIter = SourcesInfo.begin();
      while (srcIter != SourcesInfo.end())
      {
         bool isMediaSource = MediaSourceHandling::getInstance().IsMediaSourceRange(srcIter->getSourceId());
         if (isMediaSource == true)
         {
            _sourceId = srcIter->getSourceId();
            if ((_sourceId == SRC_MEDIA_PLAYER) || (_sourceId == SRC_PHONE_BTAUDIO))
            {
               _subSrcId = srcIter->getSubSourceId();
               sourceConnectionState = isInternalSourceConnected(_subSrcId);
               break;
            }
            else if ((_sourceId == SRC_CAMERAPORT1_USB_VIDEO) || (_sourceId == SRC_CAMERAPORT1_USB_AUDIO)
                     || (_sourceId == SRC_CAMERAPORT1_HDMI) || (_sourceId == SRC_CAMERAPORT2_HDMI)
                     || (_sourceId == SRC_CAMERAPORT3_HDMI) || (_sourceId == SRC_CMR1_USB1_VIDEO) || (_sourceId == SRC_CMR1_USB1_AUDIO))
            {
               sourceConnectionState  = _mediaSourceAvailabilityMap[_sourceId];
               _subSrcId = EXTERNAL_SOURCE_SUBSRCID;
               break;
            }
         }
         srcIter++;
         ETG_TRACE_USR4(("CabinSourceRestoration::isSourceConnected inside loop _sourceId  %d connectedState  %d _subSrcId  %d", srcIter->getSourceId(), sourceConnectionState, srcIter->getSubSourceId()));
      }
   }
   ETG_TRACE_USR4(("CabinSourceRestoration::isSourceConnected connectedState %d", sourceConnectionState));
   return sourceConnectionState;
}


uint8 CabinSourceRestoration::getSystemCategory(int16 sinkId)
{
   uint8 systemCategory = SYS_CATEGORY_NONE;
   switch (sinkId)
   {
      case CABIN_A_SINK_ID:
      case CABIN_B_SINK_ID:
      {
         systemCategory = SYS_CATEGORY_CMA;
         break;
      }
      case CABIN_A_MEDIAGLASS1_SINK_ID:
      case CABIN_B_MEDIAGLASS1_SINK_ID:
      {
         systemCategory = SYS_CATEGORY_CMGA;
         break;
      }
      case CABIN_A_MEDIAGLASS2_SINK_ID:
      case CABIN_B_MEDIAGLASS2_SINK_ID:
      {
         systemCategory = SYS_CATEGORY_CMGA2;
         break;
      }
      default:
      {
         ETG_TRACE_USR1(("CabinSourceRestoration::getSystemCategory INVALID SINK ID"));
         break;
      }
   }
   ETG_TRACE_USR1(("CabinSourceRestoration::getSystemCategory:systemCategory %d", ETG_CENUM(SystemCategory, systemCategory)));
   return systemCategory;
}


bool CabinSourceRestoration::isInternalSourceConnected(uint8 subSrcId)
{
   ETG_TRACE_USR1(("CabinSourceRestoration::isInternalSourceConnected is entered"));

   bool connectionState = false;
   MPlay_fi_types::T_MPlayDeviceInfo::iterator itr = _oDeviceInfo.begin();
   for (; itr != _oDeviceInfo.end(); ++itr)
   {
      if (subSrcId == itr->getU8DeviceTag())
      {
         connectionState = itr->getBDeviceConnected();
      }
   }
   ETG_TRACE_USR4(("CabinSourceRestoration isInternalSourceConnected sourceConnectionState  %d", connectionState));
   return connectionState;
}


void CabinSourceRestoration::updateMediaGadgetIndex(uint32 restorationStatus)
{
   ETG_TRACE_USR1(("CabinSourceRestoration::updateMediaGadgetIndex is entered"));

   MediaGadgetIndex mediaGadgetIndex = (restorationStatus == CABIN_RESTORE_STATUS_EVALUATING) ? RESTORATION_TEXT_INDEX : NO_MEDIA_DEVICE_GADGET_INDEX;
   uint8 activeRegion = ApplicationSwitchClientHandler::poGetInstance()->getActiveRegion();
   ETG_TRACE_USR1(("CabinSourceRestoration::updateMediaGadgetIndex:restorationStatus %d, mediaGadgetIndex %d, activeRegion %d ,_sourceId %d", restorationStatus, ETG_CENUM(MediaGadgetIndex, mediaGadgetIndex), activeRegion, _sourceId));
   HMIModelComponent::getInstance().setCabinMediaGadgetIndex(mediaGadgetIndex);

   if (activeRegion == REGION_CABIN_A || activeRegion == REGION_CABIN_B)
   {
      HMIModelComponent::getInstance().onMediaGadgetIndexUpdate(mediaGadgetIndex);
   }
}


/************************************************************************
*FUNCTION		: onSourceAvailabilityChangedSignal
*DESCRIPTION	: This method returns the src type, tuner or media based on region
*PARAMETER		: int iRegion
*RETURNVALUE	: bool
************************************************************************/

void CabinSourceRestoration::onSourceAvailabilityChangedSignal(const ::boost::shared_ptr< MASTERAUDIOSERVICE_INTERFACE::AudioSourceChange::AudioSourceChangeProxy >& /*proxy*/,
      const ::boost::shared_ptr< MASTERAUDIOSERVICE_INTERFACE::AudioSourceChange::SourceAvailabilityChangedSignal >& signal)
{
   ETG_TRACE_COMP(("CabinSourceRestoration::onSourceAvailabilityChangedSignal SrcId: %d, SubSrcId: %d, Availbility: %d", signal->getSourceAvailability().getSrcId(), signal->getSourceAvailability().getSubSrcId(), signal->getSourceAvailability().getAvailability()));
   int sourceId = signal->getSourceAvailability().getSrcId();
   if (sourceId != SRC_CAMERAPORT1_HDMI && sourceId != SRC_CAMERAPORT2_HDMI && sourceId != SRC_CAMERAPORT3_HDMI)
   {
      if (_mediaSourceAvailabilityMap.count(sourceId))
      {
         _mediaSourceAvailabilityMap[sourceId] = (RES_AVAILABLE == signal->getSourceAvailability().getAvailability()) ? true : false;
      }
   }
}


void CabinSourceRestoration::onSourceAvailabilityChangedError(const ::boost::shared_ptr< MASTERAUDIOSERVICE_INTERFACE::AudioSourceChange::AudioSourceChangeProxy >& /*proxy*/,
      const ::boost::shared_ptr< MASTERAUDIOSERVICE_INTERFACE::AudioSourceChange::SourceAvailabilityChangedError >& error)
{
   ETG_TRACE_COMP(("CabinSourceRestoration::onSourceAvailabilityChangedError() \n"));
}


void CabinSourceRestoration::onActiveSourceListError(const ::boost::shared_ptr< MASTERAUDIOSERVICE_INTERFACE::AudioSourceChange::AudioSourceChangeProxy >& proxy,
      const ::boost::shared_ptr< MASTERAUDIOSERVICE_INTERFACE::AudioSourceChange::ActiveSourceListError >& error)
{
   ETG_TRACE_COMP(("ExternalMediaSourceHandler::onActiveSourceListError\n"));
}


void CabinSourceRestoration::onActiveSourceListUpdate(const ::boost::shared_ptr< MASTERAUDIOSERVICE_INTERFACE::AudioSourceChange::AudioSourceChangeProxy >& proxy,
      const ::boost::shared_ptr< MASTERAUDIOSERVICE_INTERFACE::AudioSourceChange::ActiveSourceListUpdate >& signal)
{
   ETG_TRACE_COMP(("CabinSourceRestoration::onActiveSourceListUpdate entered"));

   if (signal != NULL)
   {
      std::vector<sourceData>AudioSourceList = signal->getActiveSourceList();
      if (!AudioSourceList.empty())
      {
         std::vector<sourceData>::const_iterator iter = AudioSourceList.begin(); // stack top is list bottom
         ETG_TRACE_COMP(("CabinSourceRestoration::onActiveSourceListUpdate SrcID  %d Sink Id   %d   ConnectionState %d", iter->getSrcId(), iter->getSinkId(), iter->getConnectionState()));

         while (iter != AudioSourceList.end())
         {
            if ((iter->getSinkId() == CABIN_A_SINK_ID) || (iter->getSinkId() == CABIN_B_SINK_ID))
            {
               if ((MediaSourceHandling::getInstance().IsMediaSourceRange(iter->getSrcId())) || (MediaSourceHandling::getInstance().IsTunerSourceRange(iter->getSrcId())))
               {
                  _restorationStatus = CABIN_RESTORE_STATUS_NONE ;
               }
            }
            else if ((iter->getSinkId() == CABIN_A_MEDIAGLASS1_SINK_ID) || (iter->getSinkId() == CABIN_A_MEDIAGLASS2_SINK_ID))
            {
               if ((iter->getSrcId() == SRC_CAMERAPORT1_CAM1) || (iter->getSrcId() == SRC_CAMERAPORT1_CAM2) ||
                     (iter->getSrcId() == SRC_CAMERAPORT1_CAM3) || (iter->getSrcId() == SRC_CAMERAPORT1_CAM4))
               {
                  _restorationStatus = CABIN_RESTORE_STATUS_NONE ;
               }
            }

            iter++;
         }
      }
   }
}


void CabinSourceRestoration::onMediaPlayerDeviceConnectionsStatus(const ::boost::shared_ptr< ::mplay_MediaPlayer_FI::Mplay_MediaPlayer_FIProxy >& /*proxy*/,
      const boost::shared_ptr< ::mplay_MediaPlayer_FI::MediaPlayerDeviceConnectionsStatus >& status)
{
   ETG_TRACE_COMP(("CabinSourceRestoration::onMediaPlayerDeviceConnectionsStatus entered"));
   _oDeviceInfo = status->getODeviceInfo();
}


/* ***********************************************************
* Function 		: onMediaPlayerDeviceConnectionsError
* Description 	: called if DeviceConnection property is updated with error from media player
* Parameters 	: proxy, error
* Return value	: void
* ***********************************************************/
void CabinSourceRestoration::onMediaPlayerDeviceConnectionsError(const ::boost::shared_ptr< ::mplay_MediaPlayer_FI::Mplay_MediaPlayer_FIProxy >& /*proxy*/,
      const boost::shared_ptr< ::mplay_MediaPlayer_FI::MediaPlayerDeviceConnectionsError >& /*error*/)
{
   ETG_TRACE_USR4(("CabinSourceRestoration::onMediaPlayerDeviceConnectionsError received."));
}


void CabinSourceRestoration::onExternalDBDownloadUpdate(uint8 downloadState, EnumConst::entSourceType sourceType)
{
   ETG_TRACE_USR4(("CabinSourceRestoration::onExternalDBDownloadUpdate downloadState:%d sourceType:%d _isSourceConnected :%d", downloadState, sourceType, _isSourceConnected));
   if (_sourceId != SRC_INVALID)
   {
      if ((downloadState == EnumConst::entDownloadState_Downloadsuccess) &&
            (_restorationStatus == CABIN_RESTORE_STATUS_EVALUATING) &&
            (_isSourceConnected == true) && (_isSourceRestored == true))
      {
         restoreSourceinCabin(_sourceId, _cabinRegion, _sinkId, CABIN_RESTORE_STATUS_RESTORE, _isSourceConnected);
         _restorationStatus = CABIN_RESTORE_STATUS_NONE ;
         _isSourceRestored = false;
      }
   }
   else
   {
      _restorationStatus = CABIN_RESTORE_STATUS_NONE ;
      _isSourceRestored = false;
   }
}


void CabinSourceRestoration::onRouterSourceAvailabilityUpdate(int iRouterAvailability)
{
   ETG_TRACE_USR4(("CabinSourceRestoration::onRouterSourceAvailabilityUpdate iRouterAvailability  %d", iRouterAvailability));
   _mediaSourceAvailabilityMap[SRC_CMR1_USB1_VIDEO] = iRouterAvailability;
   _mediaSourceAvailabilityMap[SRC_CMR1_USB1_AUDIO] = iRouterAvailability;
}


void CabinSourceRestoration::onCamportUsbSourceAvailabilityUpdate(int iUSBCamportAvailability)
{
   ETG_TRACE_USR4(("CabinSourceRestoration::onCamportUsbSourceAvailabilityUpdate iUSBCamportAvailability  %d", iUSBCamportAvailability));
   _mediaSourceAvailabilityMap[SRC_CAMERAPORT1_USB_VIDEO] = iUSBCamportAvailability;
   _mediaSourceAvailabilityMap[SRC_CAMERAPORT1_USB_AUDIO] = iUSBCamportAvailability;
}


void CabinSourceRestoration::onRegionsMapOutStatusInfoError(const ::boost::shared_ptr< HmiInfoServiceProxy >& /*proxy*/,
      const ::boost::shared_ptr< RegionsMapOutStatusInfoError >& error)
{
   ETG_TRACE_COMP(("CabinSourceRestoration::onRegionsMapOutStatusInfoError"));
}


void CabinSourceRestoration::onRegionsMapOutStatusInfoUpdate(const ::boost::shared_ptr< HmiInfoServiceProxy >& /*proxy*/,
      const ::boost::shared_ptr< RegionsMapOutStatusInfoUpdate >& update)
{
   ETG_TRACE_COMP(("CabinSourceRestoration::onRegionsMapOutStatusInfoUpdate"));

   //RegionMapOutStatusInfo Iteration to know the RegionID
   ::std::vector< RegionMapOutStatusInfo> regionsMapOutStatusInfo = update->getRegionsMapOutStatusInfo();

   if (!regionsMapOutStatusInfo.empty())
   {
      ETG_TRACE_USR4(("CabinSourceRestoration:: onRegionsMapOutStatusInfoUpdate regionsMapOutStatusInfo IFF LOOP \n"));
      ::std::vector<RegionMapOutStatusInfo>::const_iterator regionMapoutStatusinfo_iter = regionsMapOutStatusInfo.begin();

      while (regionMapoutStatusinfo_iter != regionsMapOutStatusInfo.end())
      {
         int32  regionidstatus = regionMapoutStatusinfo_iter->getRegionId();
         ETG_TRACE_USR4(("CabinSourceRestoration:: onRegionsMapOutStatusInfoUpdate regionidstatus regionMapoutStatusinfo_iter =%d   \n", regionidstatus));

         //MapOutStatusInfo Iteration to get glass id and Connection status getRegionsMapOutStatusInfo
         ::std::vector< MapOutStatusInfo> glassMapOutStatusInfo = regionMapoutStatusinfo_iter->getMapOutStatusInfo();
         ::std::vector<MapOutStatusInfo>::const_iterator glassMapoutStatus_iter = glassMapOutStatusInfo.begin();

         while (glassMapoutStatus_iter != glassMapOutStatusInfo.end())
         {
            int32   glassId = glassMapoutStatus_iter->getGlassId();
            int32   activeGlassStatus = glassMapoutStatus_iter->getActiveStatus();
            ETG_TRACE_USR4(("CabinSourceRestoration:: onRegionsMapOutStatusInfoUpdate glassId activeglassstatus=%d %d\n", glassId, activeGlassStatus));
            if (((glassId == CABIN_A_MEDIAGLASS1_SINK_ID) || (glassId == CABIN_A_MEDIAGLASS2_SINK_ID)) && (activeGlassStatus == EN_GLASS_ACTIVE))
            {
               _restorationStatus = CABIN_RESTORE_STATUS_NONE;
            }
            glassMapoutStatus_iter++;
         }
         regionMapoutStatusinfo_iter++;
      }
   }
}


void CabinSourceRestoration::onHdmiSourceAvailabilityUpdate(int srcId, int hdmiavailable)
{
   ETG_TRACE_USR4(("CabinSourceRestoration::onHdmiSourceAvailabilityUpdate hdmiavailable  %d", hdmiavailable));
   std::map <int, bool>::iterator itr = _mediaSourceAvailabilityMap.find(srcId);
   if (itr != _mediaSourceAvailabilityMap.end())
   {
      itr -> second = hdmiavailable;
   }
}


void CabinSourceRestoration::onRestoreAVSourceError(const ::boost::shared_ptr<bosch::cm::ai::hmi::hmiinfoservice::HmiInfoService::HmiInfoServiceProxy >& /* proxy*/, const ::boost::shared_ptr<RestoreAVSourceError >& error)
{
   ETG_TRACE_USR4(("CabinSourceRestoration::onRestoreAVSourceError() \n"));
}


void CabinSourceRestoration::onRestoreAVSourceSignal(const ::boost::shared_ptr<bosch::cm::ai::hmi::hmiinfoservice::HmiInfoService::HmiInfoServiceProxy >& /*proxy*/, const ::boost::shared_ptr<RestoreAVSourceSignal >& signal)
{
   ETG_TRACE_USR4(("CabinSourceRestoration::onRestoreAVSourceSignal CabinId: %d, SrcId: %d", signal->getCabinId(), signal->getSourceId()));
   int srcId = signal->getSourceId();
   int cabinId = signal->getCabinId();
   bool sourceConnectionState = false;

   if (_externalMediaSourceHandler->isExternalMediaSource(srcId))
   {
      if (_mediaSourceAvailabilityMap.count(srcId))
      {
         sourceConnectionState  = _mediaSourceAvailabilityMap[srcId];
      }
   }

   ETG_TRACE_USR4(("CabinSourceRestoration::onRestoreAVSourceSignal sourceConnectionState: %d", sourceConnectionState));

   if ((cabinId == REGION_CABIN_A || cabinId == REGION_CABIN_B) && (sourceConnectionState == true))
   {
      requestExternalSourceActivation(srcId, cabinId);
   }
   else
   {
      ETG_TRACE_USR4(("CabinSourceRestoration::onRestoreAVSourceSignal() Source is not connected"));
   }
}


void CabinSourceRestoration::requestExternalSourceActivation(int srcId, int cabinId)
{
   EnumConst::entSourceType sourceType = EnumConst::entSourceType_DEFAULT ;
   int deviceType = FI_EN_E8DTY_UNKNOWN;

   ETG_TRACE_USR4(("CabinSourceRestoration::requestExternalSourceActivation srcId: %d", srcId));
   if (srcId == SRC_CAMERAPORT1_USB_VIDEO || srcId == SRC_CAMERAPORT1_USB_AUDIO || srcId == SRC_CMR1_USB1_VIDEO || srcId == SRC_CMR1_USB1_AUDIO)
   {
      sourceType = MediaUtils::getInstance().getSourceType(srcId);
      deviceType = MediaUtils::getInstance().getDeviceType(sourceType);
      if ((_systemInfoMap->getSystemStatus(SYS_CATEGORY_CMGA) == SYSTEM_STATUS_UNAVAILABLE || _systemInfoMap->getSystemStatus(SYS_CATEGORY_CMGA) == SYSTEM_STATUS_DISCONNECTED)
            && (srcId == SRC_CAMERAPORT1_USB_VIDEO || srcId == SRC_CMR1_USB1_VIDEO))
      {
         POST_MSG_NOTRACE((COURIER_MESSAGE_NEW(Popup_Pfo_MediaGlassUnavailableMessage_Open)()));
         ETG_TRACE_USR4(("CabinSourceRestoration::requestExternalSourceActivation Glass Unavailable"));
      }
      else if (DbDownloadMap::getInstance().getDbDownloadState(sourceType) == EnumConst::entDownloadState_Downloadsuccess)
      {
         if (_playableContentMap->getCRCStatus(sourceType) == EN_CRC_MATCH)
         {
            IExternalMediaSourceActivationHandler* iExternalMediaSourceActivationHandler = ExternalMediaSourceActivationFactory::getExternalActivationSourceHandlerObject(deviceType);
            if (iExternalMediaSourceActivationHandler->getSourceIDFromActiveContent() != srcId)
            {
               ETG_TRACE_USR4(("CabinSourceRestoration::requestExternalSourceActivation Toggle Source"));
               iExternalMediaSourceActivationHandler->toggleSource();
            }
            iExternalMediaSourceActivationHandler->requestSourceActivation(cabinId);
         }
      }
      else
      {
         ETG_TRACE_USR4(("CabinSourceRestoration::requestExternalSourceActivation Download Unsuccessful"));
      }
   }
   else if (srcId == SRC_CAMERAPORT1_HDMI || srcId == SRC_CAMERAPORT2_HDMI || srcId == SRC_CAMERAPORT3_HDMI)
   {
      if (_systemInfoMap->getSystemStatus(SYS_CATEGORY_CMGA) == SYSTEM_STATUS_UNAVAILABLE || _systemInfoMap->getSystemStatus(SYS_CATEGORY_CMGA) == SYSTEM_STATUS_DISCONNECTED)
      {
         POST_MSG_NOTRACE((COURIER_MESSAGE_NEW(Popup_Pfo_MediaGlassUnavailableMessage_Open)()));
         ETG_TRACE_USR4(("CabinSourceRestoration::requestExternalSourceActivation Glass Unavailable"));
      }
      else
      {
         MediaSourceHandling::getInstance().requestSourceActivation(srcId, EXTERNAL_SOURCE_SUBSRCID, cabinId);
      }
   }
   else
   {
      ETG_TRACE_USR4(("CabinSourceRestoration::onRestoreAVSourceSignal() Default Source"));
   }
}


}//App
}//Core
