/**
 * @file         : DataModelSourceTypes.h
 * @author       : INF4CV - AppHmi_Master Team
 * @addtogroup   : AppHmi_Master
 * @brief        : DataModelSourceTypes is to handle the DataModelSource types implementation
 * @copyright    : (c) 2017-2020 Robert Bosch Car Multimedia GmbH
 *                 The reproduction, distribution and utilization of this file as
 *                 well as the communication of its contents to others without express
 *                 authorization is prohibited. Offenders will be held liable for the
 *                 payment of damages. All rights reserved in the event of the grant
 *                 of a patent, utility model or design.
 */


#ifndef DATA_MODEL_SOURCE_TYPES_H
#define DATA_MODEL_SOURCE_TYPES_H


#include "AppHmi_MasterTypes.h"
#include <vector>
#include <string>
#include <asf/core/Types.h>
#include <CanderaWidget/String/String.h>
#include <Widgets/2D/List/Data/ListDataProviderDistributor.h>


/**
 *  Classes Forward Declaration
 */

namespace App {
namespace datamodel {


class SourceInfo;
class RegionSourcesInfo;
} //namespace datamodel
} //namespace App


/**
 *  Typedefs Declaration
 */

namespace App {
namespace datamodel {


typedef ::std::vector< SourceInfo > SourcesInfo;

typedef ::std::vector< RegionSourcesInfo > RegionsSourceInfo;
} //namespace datamodel
} //namespace App


/**
 *  Enums Declaration
 */

namespace App {
namespace datamodel {


enum enSourceKey
{
   SOURCE_KEY_NONE
   , SOURCE_KEY_TUNER_FM
   , SOURCE_KEY_TUNER_AM
   , SOURCE_KEY_TUNER_WX
   , SOURCE_KEY_TUNER_DAB
   , SOURCE_KEY_TUNER_SXM
   , SOURCE_KEY_MEDIA_BT
   , SOURCE_KEY_MEDIA_AUX
   , SOURCE_KEY_MEDIA_USB1
   , SOURCE_KEY_MEDIA_USB2
   , SOURCE_KEY_MEDIA_IPOD
   , SOURCE_KEY_MEDIA_CAMERAPORT1_USB
   , SOURCE_KEY_MEDIA_CAMERAPORT2_USB
   , SOURCE_KEY_MEDIA_CAMERAPORT1_HDMI
   , SOURCE_KEY_MEDIA_CAMERAPORT2_HDMI
   , SOURCE_KEY_MEDIA_CAMERAPORT3_HDMI
   , SOURCE_KEY_MEDIA_ROUTER_USB1_AUDIO
   , SOURCE_KEY_MEDIA_ROUTER_USB1_VIDEO
};


} //namespace datamodel
} //namespace App


/**
 *  Classes Declaration
 */

namespace App {
namespace datamodel {


class SourceInfo
{
   public:
      /**
       *  Member Functions Declaration
       */
      SourceInfo()
      {
         clear();
      }

      virtual ~SourceInfo()
      {
         clear();
      }

      SourceInfo(const SourceInfo& obj)
      {
         clear();
         *this = obj;
      }

      SourceInfo& operator= (const SourceInfo& obj)
      {
         if (this != (&obj))
         {
            clear();
            _deviceIdStatus        = obj.getSourceDeviceIdStatus();
            _srcKey                = obj.getSourceKey();
            _srcIndex              = obj.getSourceIndex();
            _deviceId              = obj.getSourceDeviceId();
            _contextId             = obj.getContextId();
            _srcName               = obj.getSourceName();
            _srcIconNormalBitmap   = obj.getSourceIconNormalBitmap();
            _srcIconActiveBitmap   = obj.getSourceIconActiveBitmap();
            _srcIconPressedBitmap  = obj.getSourceIconPressedBitmap();
         }
         return (*this);
      }

      inline void clear()
      {
         _deviceIdStatus        = false;
         _srcKey                = 0;
         _srcIndex              = -1;
         _deviceId              = -1;
         _contextId             = 0;
         _srcName               = "";
         _srcIconNormalBitmap   = "";
         _srcIconActiveBitmap   = "";
         _srcIconPressedBitmap  = "";
      }

      inline void setSourceDeviceIdStatus(const bool status)
      {
         _deviceIdStatus = status;
      }

      inline uint32 getSourceDeviceIdStatus() const
      {
         return _deviceIdStatus;
      }

      inline void setSourceKey(const uint32 key)
      {
         _srcKey = key;
      }

      inline uint32 getSourceKey() const
      {
         return _srcKey;
      }

      inline void setSourceIndex(const int32 index)
      {
         _srcIndex = index;
      }

      inline int32 getSourceIndex() const
      {
         return _srcIndex;
      }

      inline void setSourceDeviceId(const int32 id)
      {
         _deviceId = id;
      }

      inline int32 getSourceDeviceId() const
      {
         return _deviceId;
      }

      inline void setContextId(const uint32 id)
      {
         _contextId = id;
      }

      inline uint32 getContextId() const
      {
         return _contextId;
      }

      inline void setSourceName(const Candera::String& name)
      {
         _srcName = name;
      }

      inline const Candera::String& getSourceName() const
      {
         return _srcName;
      }

      inline Candera::String& getMutableSourceName()
      {
         return _srcName;
      }

      inline void setSourceIconNormalBitmap(const Candera::String& name)
      {
         _srcIconNormalBitmap = name;
      }

      inline const Candera::String& getSourceIconNormalBitmap() const
      {
         return _srcIconNormalBitmap;
      }

      inline Candera::String& getMutableSourceIconNormalBitmap()
      {
         return _srcIconNormalBitmap;
      }

      inline void setSourceIconActiveBitmap(const Candera::String& name)
      {
         _srcIconActiveBitmap = name;
      }

      inline const Candera::String& getSourceIconActiveBitmap() const
      {
         return _srcIconActiveBitmap;
      }

      inline Candera::String& getMutableSourceIconActiveBitmap()
      {
         return _srcIconActiveBitmap;
      }

      inline void setSourceIconPressedBitmap(const Candera::String& name)
      {
         _srcIconPressedBitmap = name;
      }

      inline const Candera::String& getSourceIconPressedBitmap() const
      {
         return _srcIconPressedBitmap;
      }

      inline Candera::String& getMutableSourceIconPressedBitmap()
      {
         return _srcIconPressedBitmap;
      }

   private:
      /**
       *  Member Variables Declaration
       */
      bool _deviceIdStatus;
      uint32 _srcKey;
      int32 _srcIndex;
      int32 _deviceId;
      uint32 _contextId;
      Candera::String _srcName;
      Candera::String _srcIconNormalBitmap;
      Candera::String _srcIconActiveBitmap;
      Candera::String _srcIconPressedBitmap;
};


class RegionSourcesInfo
{
   public:
      /**
       *  Member Functions Declaration
       */
      RegionSourcesInfo()
      {
         clear();
      }

      virtual ~RegionSourcesInfo()
      {
         clear();
      }

      RegionSourcesInfo(const RegionSourcesInfo& obj)
      {
         clear();
         *this = obj;
      }

      RegionSourcesInfo& operator= (const RegionSourcesInfo& obj)
      {
         if (this != (&obj))
         {
            clear();
            _hkCode           = obj.getHKCode();
            _listId           = obj.getListId();
            _sceneId          = obj.getSceneId();
            _regionId         = obj.getRegionId();
            _sceneStatus      = obj.getSceneStatus();
            _srcKeysInfo      = obj.getSourceKeysInfo();
            _listDataProvider = obj.getListDataProvider();
         }
         return (*this);
      }

      inline void clear()
      {
         _hkCode      = HARDKEYCODE_HK_KEY_CODE_INVALID;
         _listId      = 0;
         _sceneId     = 0;
         _regionId    = REGION_INVALID;
         _sceneStatus = SCENE_STATUS_NONE;
         _srcKeysInfo.clear();
         resetListDataProvider();
      }

      inline void setHKCode(const uint32 code)
      {
         _hkCode = code;
      }

      inline uint32 getHKCode() const
      {
         return _hkCode;
      }

      inline void setListId(const uint32 id)
      {
         _listId = id;
      }

      inline uint32 getListId() const
      {
         return _listId;
      }

      inline void setSceneId(const uint32 id)
      {
         _sceneId = id;
      }

      inline uint32 getSceneId() const
      {
         return _sceneId;
      }

      inline void setRegionId(const uint32 id)
      {
         _regionId = id;
      }

      inline uint32 getRegionId() const
      {
         return _regionId;
      }

      inline void setSceneStatus(const enSceneStatus status)
      {
         _sceneStatus = status;
      }

      inline enSceneStatus getSceneStatus() const
      {
         return _sceneStatus;
      }

      inline void setSourceKeysInfo(const ::std::vector< uint32 >& info)
      {
         _srcKeysInfo = info;
      }

      inline const ::std::vector< uint32 >& getSourceKeysInfo() const
      {
         return _srcKeysInfo;
      }

      inline ::std::vector< uint32 >& getMutableSourceKeysInfo()
      {
         return _srcKeysInfo;
      }

      void resetListDataProvider()
      {
         _listDataProvider.Release();
      }

      bool isListDataProviderLoaded() const
      {
         bool isValid = (!_listDataProvider.PointsToNull()) ? true : false;
         return isValid;
      }

      void setListDataProvider(const tSharedPtrDataProvider& info)
      {
         _listDataProvider = info;
      }

      tSharedPtrDataProvider getListDataProvider() const
      {
         return _listDataProvider;
      }

   private:
      /**
       *  Member Variables Declaration
       */
      uint32 _hkCode;
      uint32 _listId;
      uint32 _sceneId;
      uint32 _regionId;
      enSceneStatus _sceneStatus;
      ::std::vector< uint32 > _srcKeysInfo;
      tSharedPtrDataProvider _listDataProvider;
};


} //namespace datamodel
} //namespace App


#endif /* DATA_MODEL_SOURCE_TYPES_H */
