/**
 * @file         : DataModelSource.h
 * @author       : INF4CV - AppHmi_Master Team
 * @addtogroup   : AppHmi_Master
 * @brief        : DataModelSource is to handle the data model implementation of
 *                 feature source list
 * @copyright    : (C) 2016 Robert Bosch GmbH
 *                 (C) 2016 Robert Bosch Engineering and Business Solutions Limited
 *                 The reproduction, distribution and utilization of this file as
 *                 well as the communication of its contents to others without express
 *                 authorization is prohibited. Offenders will be held liable for the
 *                 payment of damages. All rights reserved in the event of the grant
 *                 of a patent, utility model or design.
 */


#ifndef DATA_MODEL_SOURCE_H
#define DATA_MODEL_SOURCE_H

#include "DataModelSourceTypes.h"
#include "AppHmi_MasterStateMachine.h"
#include <Common/DataModel/ListRegistry.h>
#include <Core/RegionHandling/RegionHandlingInterface.h>
#include <App/Core/AudioInterface/ResourceControllerPrjInterface.h>
#include <App/Core/MediaClientHandler/MediaClientHandlerInterface.h>
#include <App/Core/AudioInterface/ConnectionControllerPrjInterface.h>
#include <Core/HmiInfoService/HmiInfoServiceServerComponentHandlingInterface.h>

/**
 *  Defines Declaration
 */

#define NS_MASTER_AUDIO ::hmibase::apphmi_master::audio


/**
 *  Classes Forward Declaration
 */
namespace App {
namespace Core {


class VariantHandling;
class IContextSwitchHandler;
} //namespace Core
} //namespace App


/**
 *  Classes Declaration
 */

namespace App {
namespace datamodel {


class DataModelSource : public ListImplementation
   , public IConnectionInfoUpdate
   , public IUpdateSourceListListenerCB
   , public ::App::Core::IDeviceConnectionsCB
   , public ::App::Core::ICMAStatusListenerCB
   , public ICabinSourceConnectionStatusCB
{
   public:
      /**
       *  Member Function Declaration
       */
      DataModelSource();
      virtual ~DataModelSource();
      tSharedPtrDataProvider getListDataProvider(const ListDateProviderReqMsg& oMsg);
      void onUpdateSourceListChange(const uint32 srcIndex);
      void onUpdateConnectionInfo(std::map< int, NS_MASTER_AUDIO::Connection* > info);
      void onDeviceConnections(const ::std::vector< ::App::Core::DeviceConnectionInfo >& info);
      void onCMAStatusUpdate(const ::App::Core::SystemStatusInfo& info);
      void onCabinSourceStatusUpdate();

      inline void initialize()
      {
         initSourcesInfo();
         initRegionsSourceInfo();
         ListRegistry::s_getInstance().addListImplementation(LIST_ID_HMI_SRC, this);
         ListRegistry::s_getInstance().addListImplementation(LIST_ID_HMI_SRC_CABIN, this);
         if (NULL != _resourceController.get())
         {
            _resourceController->addSourceListUpdateListener((*this));
         }
         if (NULL != _connectionController.get())
         {
            _connectionController->vRegisterforConnectionInfoUpdate(this);
         }
         if (NULL != _mediaClientHandler)
         {
            _mediaClientHandler->registerDeviceConnectionsUpdate((*this));
         }
         if (NULL != _regionHandling)
         {
            _regionHandling->addCMAStatusListener((*this));
         }
         if (NULL != _hmiInfoServiceServerComponent)
         {
            _hmiInfoServiceServerComponent->addCabinSourceStatusListener((*this));
         }
      }

      inline void finalize()
      {
         ListRegistry::s_getInstance().removeListImplementation(LIST_ID_HMI_SRC);
         ListRegistry::s_getInstance().removeListImplementation(LIST_ID_HMI_SRC_CABIN);
         if (NULL != _resourceController.get())
         {
            _resourceController->removeSourceListUpdateListener((*this));
         }
         if (NULL != _connectionController.get())
         {
            _connectionController->vUnRegisterforConnectionInfoUpdate(this);
         }
         if (NULL != _mediaClientHandler)
         {
            _mediaClientHandler->deregisterDeviceConnectionsUpdate((*this));
         }
         if (NULL != _regionHandling)
         {
            _regionHandling->removeCMAStatusListener((*this));
         }
         if (NULL != _hmiInfoServiceServerComponent)
         {
            _hmiInfoServiceServerComponent->removeCabinSourceStatusListener((*this));
         }
      }

      inline void setVariantHandling(::App::Core::VariantHandling& imp)
      {
         _variantHandling = (&imp);
      }

      inline void setMediaClientHandler(::App::Core::IMediaClientHandler& imp)
      {
         _mediaClientHandler = (&imp);
      }

      inline void setContextSwitchHandler(::App::Core::IContextSwitchHandler& imp)
      {
         _contextSwitchHandler = (&imp);
      }

      inline void setResourceController(const ::boost::shared_ptr< IResourceControllerPrj >& imp)
      {
         _resourceController = imp;
      }

      inline void setConnectionController(const ::boost::shared_ptr< IConnectionControllerPrj >& imp)
      {
         _connectionController = imp;
      }

      inline void setRegionHandling(::App::Core::IRegionHandling& imp)
      {
         _regionHandling = (&imp);
      }

      inline void setHmiInfoServiceServerComponent(IHmiInfoServiceServerComponent& imp)
      {
         _hmiInfoServiceServerComponent = (&imp);
      }

      COURIER_MSG_MAP_BEGIN(TR_CLASS_APPHMI_MASTER_COURIER_PAYLOAD_MODEL_COMP)
      ON_COURIER_MESSAGE(HKStatusUpdateMsg)
      ON_COURIER_MESSAGE(SceneStatusInfoMsg)
      ON_COURIER_MESSAGE(ButtonListItemUpdMsg)
      COURIER_MSG_MAP_DELEGATE_DEF_BEGIN()
      COURIER_MSG_DELEGATE_TO_CLASS(ListImplementation)
      COURIER_MSG_MAP_DELEGATE_DEF_END()
      COURIER_MSG_MAP_DELEGATE_END()

   private:
      /**
       *  Member Function Declaration
       */
      DataModelSource(const DataModelSource& obj);
      DataModelSource& operator= (const DataModelSource& obj);
      SourcesInfo getSourcesInfo() const;
      RegionsSourceInfo getRegionsSourceInfo() const;
      const SourceInfo* getSourceInfo(const uint32 key) const;
      const RegionSourcesInfo* getRegionSourcesInfoForActiveSceneId() const;
      uint32 getActiveSourceKey(const uint32 regionId) const;
      bool onCourierMessage(const HKStatusUpdateMsg& oMsg);
      bool onCourierMessage(const SceneStatusInfoMsg& oMsg);
      bool onCourierMessage(const ButtonListItemUpdMsg& oMsg);
      bool updateSourceDeviceId(const ::std::vector< ::App::Core::DeviceConnectionInfo >& info, const ::std::string& type, const uint32 srcKey);
      void toggleUserSelectedKey(const RegionSourcesInfo& info);
      void activateSource(const RegionSourcesInfo& info, const uint32 srcKey);
      bool isSourceIdAvailable(const int16 sourceId) const;

      inline SourceInfo* getMutableSourceInfo(const uint32 key)
      {
         SourceInfo* imp = const_cast< SourceInfo* >(getSourceInfo(key));
         return imp;
      }

      inline RegionSourcesInfo* getMutableRegionSourcesInfoForActiveSceneId()
      {
         RegionSourcesInfo* imp = const_cast< RegionSourcesInfo* >(getRegionSourcesInfoForActiveSceneId());
         return imp;
      }

      inline void initSourcesInfo()
      {
         if (_sourcesInfo.empty())
         {
            _sourcesInfo = getSourcesInfo();
         }
      }

      inline void initRegionsSourceInfo()
      {
         if (_regionsSourceInfo.empty())
         {
            _regionsSourceInfo = getRegionsSourceInfo();
         }
      }
      /**
       *  Member Variables Declaration
       */
      uint32 _userSelectedKey;
      SourcesInfo _sourcesInfo;
      RegionsSourceInfo _regionsSourceInfo;
      ::App::Core::IRegionHandling* _regionHandling;
      ::App::Core::VariantHandling* _variantHandling;
      ::App::Core::IMediaClientHandler* _mediaClientHandler;
      ::App::Core::IContextSwitchHandler* _contextSwitchHandler;
      ::boost::shared_ptr< IResourceControllerPrj > _resourceController;
      ::boost::shared_ptr< IConnectionControllerPrj > _connectionController;
      IHmiInfoServiceServerComponent* _hmiInfoServiceServerComponent;
};


} //namespace DataModel
} //namespace App


#undef NS_MASTER_AUDIO


#endif /* DATA_MODEL_SOURCE_H */
