/**
 * @file         : DataModelSound.cpp
 * @author       : INF4CV - AppHmi_Master Team
 * @addtogroup   : AppHmi_Master
 * @brief        : DataModelSound is to handle model implementation of Sound features
 * @copyright    : (c) 2017-2020 Robert Bosch Car Multimedia GmbH
 *                 The reproduction, distribution and utilization of this file as
 *                 well as the communication of its contents to others without express
 *                 authorization is prohibited. Offenders will be held liable for the
 *                 payment of damages. All rights reserved in the event of the grant
 *                 of a patent, utility model or design.
 */


#include "hall_std_if.h"
#include "DataModelSound.h"
#include "SPM_CORE_FIProxy.h"
#include "spm_core_fi_typesConst.h"
#include "ApplicationSwitchConst.h"
#include "AudioControllerObjectManager.h"
#include <Core/Utility/MasterUtility.h>
#include <Core/AudioInterface/SoundControllerPrj.h>
#include <Core/RegionHandling/RegionHandlingTypes.h>
#include <Core/AudioInterface/ConnectionControllerPrj.h>
#include <AppHmi_MasterBase/AudioInterface/AudioDefines.h>


#ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
#define ETG_DEFAULT_TRACE_CLASS TR_CLASS_APPHMI_MASTER_DM
#define ETG_I_FILE_PREFIX App::DataModel::DataModelSound::
#include "trcGenProj/Header/DataModelSound.cpp.trc.h"
#endif

#define COCKPIT_VOLUME_STEP_INCREASE_VALUE  +1
#define COCKPIT_VOLUME_STEP_DECREASE_VALUE  -1
#define ENCODER_VOLUME_STEP_INCREASE_VALUE  +5
#define ENCODER_VOLUME_STEP_DECREASE_VALUE  -5

#define DEMUTE_REQUEST  1
#define MUTE_REQUEST  2
#define VOLUMEUP_REQUEST  3
#define VOLUMEDOWN_REQUEST  4


using namespace ::App::Core;
using namespace ::hmibase::apphmi_master::audio;
using namespace ::bosch::cm::ai::hmi::hmimasterservice::ApplicationSwitch;


namespace App {
namespace datamodel {


DataModelSound* DataModelSound::_theModelInstance = 0;


DataModelSound::DataModelSound() : _regionHandling(NULL)
   , _VehicleSrvClient(NULL)
{
   ETG_TRACE_USR4(("DataModelSound constructor"));
   _volumeReqCount.clear();
   _currentVolume = 0;
}


DataModelSound::~DataModelSound()
{
   _regionHandling          = NULL;
   _volumeReqCount.clear();
   _currentVolume = 0;
   _VehicleSrvClient = NULL;
}


void DataModelSound::deleteInstance()
{
   if (NULL != _theModelInstance)
   {
      delete _theModelInstance;
      _theModelInstance = NULL;
   }
}


DataModelSound& DataModelSound::getInstance()
{
   if (NULL == _theModelInstance)
   {
      _theModelInstance = new DataModelSound();
   }
   assert(_theModelInstance);
   return *_theModelInstance;
}


bool DataModelSound::onCourierMessage(const SetVolumeReqMsg& oMsg)
{
   ETG_TRACE_USR4(("SetVolumeReqMsg: Value: %d", oMsg.GetValue()));
   updateVolumeRequest(oMsg.GetValue(), oMsg.GetU32RegionID());
   return true;
}


bool DataModelSound::onCourierMessage(const SetMicVolumeReqMsg& oMsg)
{
   ETG_TRACE_USR4(("SetMicVolumeReqMsg: Value(): %d", oMsg.GetValue()));
   if (NULL != _regionHandling)
   {
      SupportedRegionsInfo tSupportedRegionsInfo = _regionHandling->getSupportedRegionsInfo();
      for (SupportedRegionsInfo::const_iterator itr = tSupportedRegionsInfo.begin(); (itr != tSupportedRegionsInfo.end()); ++itr)
      {
         if (_regionHandling->isCabinRegion((*itr).first))
         {
            updateVolumeRequest(oMsg.GetValue(), (*itr).first);
         }
      }
   }
   return true;
}


bool DataModelSound::onCourierMessage(const HKStatusUpdateMsg& oMsg)
{
   bool bretvalue = false;
   uint32 u32HkCode = oMsg.GetHkCode();
   ETG_TRACE_USR4(("HKStatusUpdateMsg: HkCode: %d, HkState: %d", u32HkCode, oMsg.GetHkState()));
   switch (u32HkCode)
   {
      case HARDKEYCODE_SWC_VOLUME_UP:
      case HARDKEYCODE_SWC_VOLUME_DOWN:
      {
         PerformSWCVolumeChange(u32HkCode);
         bretvalue = true;
         break;
      }
      case HARDKEYCODE_REMOTECONTROL_VOLUMEUP:
      case HARDKEYCODE_REMOTECONTROL_VOLUMEDOWN:
      {
         PerformVolumeChange(u32HkCode);
         bretvalue = true;
         break;
      }
      case HARDKEYCODE_SWC_MUTE:
      {
         performMuteDemuteRequest(Region__REG_COCKPIT, true);
         bretvalue = true;
         break;
      }
      case HARDKEYCODE_SWC_DEMUTE:
      {
         performMuteDemuteRequest(Region__REG_COCKPIT, false);
         bretvalue = true;
         break;
      }
      case HARDKEYCODE_REMOTECONTROL_MUTE:
      {
         performMuteRequest(_regionHandling->getRegionId());
         bretvalue = true;
         break;
      }
      default:
      {
         bretvalue = PerformEncoderStatusChange(u32HkCode);
         break;
      }
   }
   return bretvalue;
}


bool DataModelSound::PerformEncoderStatusChange(const uint32 u32HkCode)
{
   ETG_TRACE_USR4(("PerformEncoderStatusChange: HkCode: %d", u32HkCode));

   bool bretvalue = false;
   if ((ENCCODE_LEFT_ENCODER == u32HkCode) || (ENCCODE_RIGHT_ENCODER == u32HkCode))
   {
      performMuteRequest(getRegionIdForKey(u32HkCode));
      bretvalue = true;
   }
   else
   {
      bretvalue = false;
   }
   return bretvalue;
}


void DataModelSound::PerformSWCVolumeChange(const uint32 u32HkCode)
{
   ETG_TRACE_USR4(("PerformSWCVolumeChange: HkCode: %d", u32HkCode));
   uint8 currentVolReqCount = _volumeReqCount[Region__REG_COCKPIT];
   if (u32HkCode == HARDKEYCODE_SWC_VOLUME_UP)
   {
      updateVolumeRequest(1, Region__REG_COCKPIT);
      currentVolReqCount += 1;
      _volumeReqCount[Region__REG_COCKPIT] = currentVolReqCount;
   }
   else if (u32HkCode == HARDKEYCODE_SWC_VOLUME_DOWN)
   {
      updateVolumeRequest(-1, Region__REG_COCKPIT);
      currentVolReqCount += 1;
      _volumeReqCount[Region__REG_COCKPIT] = currentVolReqCount;
   }
   else
   {}
}


void DataModelSound::PerformVolumeChange(const uint32 u32HkCode)
{
   ETG_TRACE_USR4(("PerformVolumeChange: HkCode: %d", u32HkCode));
   if (NULL != _regionHandling)
   {
      uint32 regionId = _regionHandling->getRegionId();
      uint8 currentVolReqCount = _volumeReqCount[regionId];
      if (u32HkCode == HARDKEYCODE_REMOTECONTROL_VOLUMEUP)
      {
         updateVolumeRequest(1, regionId);
         currentVolReqCount += 1;
         _volumeReqCount[regionId] = currentVolReqCount;
      }
      else if (u32HkCode == HARDKEYCODE_REMOTECONTROL_VOLUMEDOWN)
      {
         updateVolumeRequest(-1, regionId);
         currentVolReqCount += 1;
         _volumeReqCount[regionId] = currentVolReqCount;
      }
      else
      {}
   }
}


bool DataModelSound::onCourierMessage(const Courier::StartupMsg& /*oMsg*/)
{
   updateDataBindningVariablesAtStartup();
   updateMuteStatusAtStartup();
   return false;
}


void DataModelSound::updateDataBindningVariablesAtStartup()
{
   (*_volumeInfoData).mVolumeBarRed = true;
   (*_volumeInfoData).mVolumeBarGreen = false;
   (*_volumeInfoData).mCabinAVolumeValue = 0; // range 1-40
   (*_volumeInfoData).mCabinAVolumeValueString = "";
   (*_volumeInfoData).mCabinAControlPanel_Icon_Status = true;
   _volumeInfoData.SendUpdate(true);
}


bool DataModelSound::onCourierMessage(const SetSystemMuteReqMsg& oMsg)
{
   SoundControllerPrj* imp = getSoundController();
   if (NULL != imp)
   {
      if (oMsg.GetRegionId() == REGION_COCKPIT)
      {
         imp->updateSetSystemMuteReq(oMsg.GetMuteStatus(), oMsg.GetRegionId());
      }
      else
      {
         if ((NULL != _regionHandling) && (_regionHandling->isRegionSupported(oMsg.GetRegionId())))
         {
            if (oMsg.GetMuteStatus() == true)
            {
               imp->updateSetSystemMuteReq(oMsg.GetMuteStatus(), oMsg.GetRegionId());
            }
            else
            {
               uint8 tSystemStatus = _regionHandling->getStatus(SystemEnquiryInfo(oMsg.GetRegionId(), SYSTEM_CATEGORY_CMA, STATUS_CATEGORY_SYSTEM));
               if ((tSystemStatus == SYSTEM_STATUS_CONNECTED) || (tSystemStatus == SYSTEM_STATUS_HOURLOGIC_DISCONNECTED))
               {
                  imp->updateSetSystemMuteReq(oMsg.GetMuteStatus(), oMsg.GetRegionId());
               }
            }
         }
      }
   }
   return true;
}


bool DataModelSound::onCourierMessage(const EncoderStatusUpdateMsg& oMsg)
{
   ETG_TRACE_USR4(("DataModelSound::EncoderStatusUpdateMsg Steps: %d, Code: %d", oMsg.GetEncoderSteps(), oMsg.GetU32EncoderKey()));
#ifdef VARIANT_S_FTR_ENABLE_ANTI_THEFT
   if (IN_tu8AntitheftStatus != HMI_SM_C_HU_UNLOCKED)
   {
      ETG_TRACE_USR4(("DataModelSound: AntiTheft Enabled, HU is not Unlocked "));
      return true;
   }
#endif
   uint32 regionId = getRegionIdForKey(oMsg.GetU32EncoderKey());
   uint8 currentVolReqCount = _volumeReqCount[regionId];
   int steps = oMsg.GetEncoderSteps();
   if (steps < ENCODER_VOLUME_STEP_DECREASE_VALUE) //-5
   {
      updateVolumeRequest(ENCODER_VOLUME_STEP_DECREASE_VALUE, regionId);
      currentVolReqCount += 1;
      _volumeReqCount[regionId] = currentVolReqCount;
   }
   else if ((steps <= ENCODER_VOLUME_STEP_INCREASE_VALUE) && (steps >= ENCODER_VOLUME_STEP_DECREASE_VALUE))
   {
      updateVolumeRequest(steps, regionId);
      currentVolReqCount += 1;
      _volumeReqCount[regionId] = currentVolReqCount;
   }
   else if (steps > ENCODER_VOLUME_STEP_INCREASE_VALUE) //5
   {
      updateVolumeRequest(ENCODER_VOLUME_STEP_INCREASE_VALUE, regionId);
      currentVolReqCount += 1;
      _volumeReqCount[regionId] = currentVolReqCount;
   }
   else
   {}
   return true;
}


bool DataModelSound::onCourierMessage(const SetHmiStartupMuteRequest& oMsg)
{
   SoundControllerPrj* imp = getSoundController();
   if (NULL != imp)
   {
      imp->setHmiStartupMute(oMsg.GetMuteStatus());
   }
   return true;
}


bool DataModelSound::onCourierBindingItemChange_VolumeInfoItem(const Courier::Request& request)
{
   ETG_TRACE_USR4(("AppHmi_Master DataModel : onCourierBindingItemChange_VolumeInfoItem "));
   if (NULL != _regionHandling)
   {
      switch (request.ItemKey())
      {
         case ItemKey::VolumeInfo::VolumeValueItem:
         case ItemKey::VolumeInfo::CabinAVolumeValueItem:
         {
            const int* temp = request.GetItemValue().GetValue<int>();
            if (NULL != temp)
            {
               int currentValue = (request.ItemKey() == ItemKey::VolumeInfo::VolumeValueItem) ? (*_volumeInfoData).mVolumeValue : (*_volumeInfoData).mCabinAVolumeValue;
               if ((*temp) != currentValue)
               {
                  int value = ((*temp) > currentValue) ? COCKPIT_VOLUME_STEP_INCREASE_VALUE : COCKPIT_VOLUME_STEP_DECREASE_VALUE;
                  updateVolumeRequest(value, _regionHandling->getRegionId());
               }
            }
            break;
         }
         default:
            break;
      }
   }
   return true;
}


#ifdef VARIANT_S_FTR_ENABLE_MIC_ACTIVATION
bool DataModelSound::onCourierBindingItemChange_MIC1VolumeInfoItem(const Courier::Request& request)
{
   ETG_TRACE_USR4(("AppHmi_Master DataModel : onCourierBindingItemChange_MIC1VolumeInfoItem"));
   if (NULL != _regionHandling)
   {
      if (request.ItemKey() == ItemKey::MIC1VolumeInfo::Mic1ProgressBarVolumeValueItem)
      {
         const int* temp = request.GetItemValue().GetValue<int>();
         if (NULL != temp)
         {
            int currentValue = (*m_MICVolData).mMic1ProgressBarVolumeValue;
            if ((*temp) != currentValue)
            {
               int value = ((*temp) > currentValue) ? COCKPIT_VOLUME_STEP_INCREASE_VALUE : COCKPIT_VOLUME_STEP_DECREASE_VALUE;
               SupportedRegionsInfo tSupportedRegionsInfo = _regionHandling->getSupportedRegionsInfo();
               for (SupportedRegionsInfo::const_iterator itr = tSupportedRegionsInfo.begin(); (itr != tSupportedRegionsInfo.end()); ++itr)
               {
                  if (_regionHandling->isCabinRegion((*itr).first))
                  {
                     updateVolumeRequest(value, (*itr).first);
                  }
               }
            }
         }
      }
   }
   return true;
}


#endif


void DataModelSound::onMuteStateUpdate(const uint32 sinkId, const int state)
{
   ETG_TRACE_USR4(("DataModelSound: onMuteStateUpdate: SinkId = %d : state = %d", sinkId, state));
   if (NULL != _regionHandling)
   {
      switch (_regionHandling->getRegionIdForAudioSink(sinkId))
      {
         case REGION_COCKPIT:
         {
            updateSystemMuteInfo_Cockpit(state);
            break;
         }
         case REGION_CABIN_A:
         {
            updateSystemMuteInfo_CabinA(state);
            break;
         }
         case REGION_CABIN_B:
         {
            updateSystemMuteInfo_CabinB(state);
            break;
         }
         default :
            break;
      }
   }
#ifdef VARIANT_S_FTR_ENABLE_MIC_ACTIVATION
   updateMicMuteInfo(sinkId, state);
#endif
}


void DataModelSound::updateSystemMuteInfo_Cockpit(int state)
{
   ETG_TRACE_USR4(("DataModelSound updateSystemMuteInfo_Cockpit: state:%d", state));
   if (state == STATE_UNMUTED)
   {
      (*_SystemMuteInfoData).mMute_UnMuteVal_Cockpit = 0;
      _SystemMuteInfoData.MarkItemModified(::ItemKey::SystemMuteInfo::Mute_UnMuteVal_CockpitItem);
      _SystemMuteInfoData.SendUpdate(false);
   }
   else if (state == STATE_MUTED)
   {
      (*_SystemMuteInfoData).mMute_UnMuteVal_Cockpit = 1;
      _SystemMuteInfoData.MarkItemModified(::ItemKey::SystemMuteInfo::Mute_UnMuteVal_CockpitItem);
      _SystemMuteInfoData.SendUpdate(false);
   }
   else
   {}
}


#ifdef VARIANT_S_FTR_ENABLE_MIC_ACTIVATION
void DataModelSound::updateMicMuteInfo(const uint32 sinkId, int state)
{
   if (NULL != _connectionControllerPrj.get())
   {
      ETG_TRACE_USR4(("DataModelSound::updateMicMuteInfo: sinkId=%d, state:%d, getCurrentActiveSource=%d", sinkId, state, _connectionControllerPrj->getCurrentActiveAnnouncementSource(sinkId)));
      if (_connectionControllerPrj->getCurrentActiveAnnouncementSource(sinkId) == SRC_MIC1_ANNOUNCEMENT)
      {
         if ((sinkId == CABIN_A_SINK_ID) || (sinkId == CABIN_B_SINK_ID))
         {
            bool temp = (state == STATE_MUTED) ? true : false;
            if ((*m_MICVolData).mMic1MuteStatus != temp)
            {
               (*m_MICVolData).mMic1MuteStatus = temp;
               m_MICVolData.MarkItemModified(::ItemKey::MIC1VolumeInfo::Mic1MuteStatusItem);
               m_MICVolData.SendUpdate(false);
            }
         }
      }
   }
}


#endif

void DataModelSound::updateSystemMuteInfo_CabinA(int state)
{
   ETG_TRACE_USR4(("DataModelSound updateSystemMuteInfo_CabinA: state:%d", state));
   if (state == STATE_UNMUTED)
   {
      (*_SystemMuteInfoData).mMute_UnMuteVal_CabinA = 0;
      _SystemMuteInfoData.MarkItemModified(::ItemKey::SystemMuteInfo::Mute_UnMuteVal_CabinAItem);
      _SystemMuteInfoData.SendUpdate(false);
   }
   else if (state == STATE_MUTED)
   {
      (*_SystemMuteInfoData).mMute_UnMuteVal_CabinA = 1;
      _SystemMuteInfoData.MarkItemModified(::ItemKey::SystemMuteInfo::Mute_UnMuteVal_CabinAItem);
      _SystemMuteInfoData.SendUpdate(false);
   }
   else
   {}
}


void DataModelSound::updateSystemMuteInfo_CabinB(int state)
{
   ETG_TRACE_USR4(("DataModelSound updateSystemMuteInfo_CabinB: state:%d", state));
   if (state == STATE_UNMUTED)
   {
      (*_SystemMuteInfoData).mMute_UnMuteVal_CabinB = 0;
      _SystemMuteInfoData.MarkItemModified(::ItemKey::SystemMuteInfo::Mute_UnMuteVal_CabinBItem);
      _SystemMuteInfoData.SendUpdate(false);
   }
   else if (state == STATE_MUTED)
   {
      (*_SystemMuteInfoData).mMute_UnMuteVal_CabinB = 1;
      _SystemMuteInfoData.MarkItemModified(::ItemKey::SystemMuteInfo::Mute_UnMuteVal_CabinBItem);
      _SystemMuteInfoData.SendUpdate(false);
   }
   else
   {}
}


void DataModelSound::onVolumeChangeUpdate(const uint32 sinkId, const int volume)
{
   ETG_TRACE_USR4(("DataModelSound: onVolumeChangeUpdate: SinkId = %d : Volume = %d", sinkId, volume));
   if (NULL != _regionHandling)
   {
      uint32 regionId = _regionHandling->getRegionIdForAudioSink(sinkId);
      switch (regionId)
      {
         case REGION_COCKPIT:
            updateVolumeInfo(volume);
            break;
         case REGION_CABIN_A:
            updateCabinAVolumeInfo(volume);
            break;
         default :
            break;
      }
      uint8 currentVolReqCount = _volumeReqCount[regionId];
      if (currentVolReqCount > 0)
      {
         POST_MSG((COURIER_MESSAGE_NEW(VolumePopupReqMsg)(regionId)));
         currentVolReqCount -= 1;
         _volumeReqCount[regionId] = currentVolReqCount;
      }
   }
#ifdef VARIANT_S_FTR_ENABLE_MIC_ACTIVATION
   updateMicVolumeInfo(sinkId, volume);
#endif
}


void DataModelSound::onCMAStatusUpdate(const SystemStatusInfo& info)
{
   ETG_TRACE_USR4(("DataModelSound onCMAStatusUpdate: Status: %d, SystemId: %d", info.getStatus(), info.getSystemId()));
   const RegionsIdInfo& tRegionsIdInfo = info.getRegionsIdInfo();
   for (RegionsIdInfo::const_iterator itr = tRegionsIdInfo.begin(); (itr != tRegionsIdInfo.end()); ++itr)
   {
      updateSystemMuteForCMAStatus((*itr), info.getStatus(), info.getSystemId());
      updateVolumeAndMuteForCMAStatus((*itr), info.getStatus(), info.getSystemId());
   }
}


bool DataModelSound::isCabinVolumeAndMuteApplicable(const uint8 status)
{
   bool muteStatus = false;
   switch (status)
   {
      case SYSTEM_STATUS_CONNECTED:
      case SYSTEM_STATUS_HOURLOGIC_DISCONNECTED:
      case SYSTEM_STATUS_PNM_DISCONNECTED:
      {
         muteStatus = true;
         break;
      }
      default:
      {
         break;
      }
   }

   return muteStatus;
}


void DataModelSound::updateVolumeAndMuteForCMAStatus(const uint32 regionId, const uint8 status, const uint8 /* systemId */)
{
   SoundControllerPrj* imp = getSoundController();
   if (NULL != _regionHandling)
   {
      switch (regionId)
      {
         case Enum_REGION_CABIN_A:
         {
            if (!isCabinVolumeAndMuteApplicable(status))
            {
               if ((*_volumeInfoData).mCabinAControlPanel_Icon_Status == true)
               {
                  (*_volumeInfoData).mCabinAControlPanel_Icon_Status = false;
                  _volumeInfoData.MarkItemModified(ItemKey::VolumeInfo::CabinAControlPanel_Icon_StatusItem);
                  _volumeInfoData.SendUpdate(false);
               }
               uint8 avdecc_CMAStatus = _regionHandling->getStatus(SystemEnquiryInfo(regionId, SYSTEM_CATEGORY_CMA, STATUS_CATEGORY_AVDECC));
               if ((avdecc_CMAStatus != SYSTEM_STATUS_CONNECTED) && (((*_volumeInfoData).mCabinAVolumeValue != 0) || ((*_volumeInfoData).mCabinAVolumeValueString != "")))
               {
                  (*_volumeInfoData).mCabinAVolumeValue = 0; // range 1-40
                  (*_volumeInfoData).mCabinAVolumeValueString = "";
                  _volumeInfoData.SendUpdate(true);
               }
            }
            else
            {
               if ((*_volumeInfoData).mCabinAControlPanel_Icon_Status == false)
               {
                  (*_volumeInfoData).mCabinAControlPanel_Icon_Status = true;
                  _volumeInfoData.MarkItemModified(ItemKey::VolumeInfo::CabinAControlPanel_Icon_StatusItem);
                  _volumeInfoData.SendUpdate(false);
               }
            }
            break;
         }
         default:
         {
            break;
         }
      }
   }
}


void DataModelSound::updateSystemMuteForCMAStatus(const uint32 regionId, const uint8 status, const uint8 /* systemId */)
{
   SoundControllerPrj* imp = getSoundController();
   if (NULL != imp)
   {
      imp->updateSetSystemMuteReq(!isCabinVolumeAndMuteApplicable(status), regionId);
   }
}


void DataModelSound::updateVolumeInfo(int val)
{
   ETG_TRACE_USR4(("DataModelSound updateVolumeInfo: val:%d", val));
   _currentVolume = val;
   ::std::string volText = App::Core::MasterUtility::u8ConversionToString(val);
   (*_volumeInfoData).mVolumeValue = val; // range 1-40
   (*_volumeInfoData).mVolumeValueString = Candera::String(volText.c_str(), volText.length());
   _volumeInfoData.MarkItemModified(ItemKey::VolumeInfo::VolumeValueItem);
   _volumeInfoData.MarkItemModified(ItemKey::VolumeInfo::VolumeValueStringItem);
   _volumeInfoData.SendUpdate(false);
}


void DataModelSound::updateCabinAVolumeInfo(int val)
{
   ETG_TRACE_USR4(("DataModelSound updateCabinAVolumeInfo: val:%d", val));
   ::std::string volText = App::Core::MasterUtility::u8ConversionToString(val);
   (*_volumeInfoData).mCabinAVolumeValue = val; // range 1-40
   (*_volumeInfoData).mCabinAVolumeValueString = Candera::String(volText.c_str(), volText.length());
   _volumeInfoData.MarkItemModified(ItemKey::VolumeInfo::CabinAVolumeValueItem);
   _volumeInfoData.MarkItemModified(ItemKey::VolumeInfo::CabinAVolumeValueStringItem);
   _volumeInfoData.SendUpdate(false);
}


#ifdef VARIANT_S_FTR_ENABLE_MIC_ACTIVATION
void DataModelSound::updateMicVolumeInfo(const uint32 sinkId, int val)
{
   if (NULL != _connectionControllerPrj.get())
   {
      ETG_TRACE_USR4(("DataModelSound::updateMicVolumeInfo: sinkId=%d, Volume:%d, getCurrentActiveSource=%d", sinkId, val, _connectionControllerPrj->getCurrentActiveAnnouncementSource(sinkId)));
      ::std::string micVolText = App::Core::MasterUtility::u8ConversionToString(val);
      if (_connectionControllerPrj->getCurrentActiveAnnouncementSource(sinkId) == SRC_MIC1_ANNOUNCEMENT)
      {
         if ((sinkId == CABIN_A_SINK_ID) || (sinkId == CABIN_B_SINK_ID))
         {
            if ((*m_MICVolData).mMic1ProgressBarVolumeValue != val)
            {
               ETG_TRACE_USR4(("DataModelSound::updateMicVolumeInfo::updatingBindignValues:%d", val));
               (*m_MICVolData).mMic1ProgressBarVolumeValue = val; // range 1-40
               (*m_MICVolData).mMic1VolumeValue = Candera::String(micVolText.c_str(), micVolText.length());
               m_MICVolData.MarkItemModified(ItemKey::MIC1VolumeInfo::Mic1ProgressBarVolumeValueItem);
               m_MICVolData.MarkItemModified(ItemKey::MIC1VolumeInfo::Mic1VolumeValueItem);
               m_MICVolData.SendUpdate(false);
            }
         }
      }
   }
}


#endif


void DataModelSound::updateVolumeRequest(int VolumeChange, uint32 regionId)
{
   ETG_TRACE_USR1(("DataModelSound::updateVolumeRequest: <-: VolumeChange= %d regionId=%d", VolumeChange, regionId));
   SoundControllerPrj* imp = getSoundController();
   if ((NULL != imp) && (NULL != _regionHandling) && ((regionId == REGION_COCKPIT) || (isValidToProcessCabinRequest(regionId))))
   {
      uint32 sinkId = _regionHandling->getAudioSinkIdForRegion(regionId);
      if (regionId == REGION_COCKPIT)
      {
         ConnectionControllerPrj* imp = AudioControllerObjectManager::getInstance().getConnectionController<ConnectionControllerPrj>().get();
         if ((NULL != imp) && (imp->isPrivateSpeakerConnected()))
         {
            sinkId = PRIVATE_SPEAKER_SINK_ID;
         }
      }
      ETG_TRACE_USR1(("DataModelSound::updateVolumeRequest: ->: VolumeChange= %d sinkid=%d", VolumeChange, sinkId));
      imp->processLeftEncoder(VolumeChange, sinkId);
   }
}


bool DataModelSound::onCourierMessage(const ToggleMuteStateReqMsg& msg)
{
   ETG_TRACE_USR4(("DataModelSound: ToggleMuteStateReqMsg"));
   performMuteRequest(msg.GetU32RegionID());
   return true;
}


void DataModelSound::performMuteRequest(uint32 regionID)
{
   ETG_TRACE_USR1(("DataModelSound: performMuteRequest: <-: regionID: %d", regionID));
   SoundControllerPrj* imp = getSoundController();
   if ((NULL != imp) && (NULL != _regionHandling) && ((regionID == REGION_COCKPIT) || (isValidToProcessCabinRequest(regionID))))
   {
      uint32 sinkId  = _regionHandling->getAudioSinkIdForRegion(regionID);
      bool muteState = (!imp->requestUserMuteStatus(sinkId)) ? true : false;
      ETG_TRACE_USR1(("DataModelSound: performMuteRequest: ->: muteState=%d, sinkId=%d", muteState, sinkId));
      imp->requestUserMuteState(muteState, sinkId);
   }
}


void DataModelSound::performMuteDemuteRequest(uint32 regionID, bool bMuteState)
{
   ETG_TRACE_USR1(("DataModelSound: performMuteDemuteRequest: regionID: %d", regionID));
   SoundControllerPrj* imp = getSoundController();
   if ((NULL != imp) && (NULL != _regionHandling) && ((regionID == REGION_COCKPIT) || (isValidToProcessCabinRequest(regionID))))
   {
      uint32 sinkId  = _regionHandling->getAudioSinkIdForRegion(regionID);
      ETG_TRACE_USR1(("DataModelSound: performMuteDemuteRequest: bMuteState=%d, sinkId=%d", bMuteState, sinkId));
      imp->requestUserMuteState(bMuteState, sinkId);
   }
}


SoundControllerPrj* DataModelSound::getSoundController() const
{
   SoundControllerPrj* imp = dynamic_cast< SoundControllerPrj* >(_soundController.get());
   return imp;
}


uint8 DataModelSound::getRegionIdForKey(const uint8 code) const
{
   ETG_TRACE_USR4(("DataModelSound: getRegionIdForKey code: %d", code));
   uint32 regionID = 0;
   if (NULL != _regionHandling)
   {
      switch (code)
      {
         case ENCCODE_LEFT_ENCODER:
         {
            regionID = Region__REG_COCKPIT;
            break;
         }
         case ENCCODE_RIGHT_ENCODER:
         {
            regionID =  _regionHandling->getLastActiveCabinRegionId();
            break;
         }
         default:
         {
            break;
         }
      }
   }

   ETG_TRACE_USR3(("DataModelSound: getRegionIdForKey, regionID: %d", regionID));
   return regionID;
}


bool DataModelSound::isValidToProcessCabinRequest(const uint32 regionId) const
{
   bool isValid = false;
   if ((NULL != _regionHandling) && (regionId != REGION_COCKPIT) && (_regionHandling->isRegionSupported(regionId)))
   {
      uint8 tSystemStatus = _regionHandling->getStatus(SystemEnquiryInfo(regionId, SYSTEM_CATEGORY_CMA, STATUS_CATEGORY_SYSTEM));
      switch (tSystemStatus)
      {
         case SYSTEM_STATUS_CONNECTED:
         case SYSTEM_STATUS_HOURLOGIC_DISCONNECTED:
         case SYSTEM_STATUS_PNM_DISCONNECTED:
         {
            isValid = true;
         }
         default:
            break;
      }
   }
   return isValid;
}


void DataModelSound::updateMuteStatusAtStartup()
{
   ETG_TRACE_USR4(("DataModelSound: updateMuteStatusAtStartup"));
   SoundControllerPrj* imp = getSoundController();
   if (imp != NULL)
   {
      int cockpitStatus = imp->getSoundPropertySink(MSP_USER_MUTE_STATUS, COCKPIT_SINK_ID);
      int cabinAStatus = imp->getSoundPropertySink(MSP_USER_MUTE_STATUS, CABIN_A_SINK_ID);
      int cabinBStatus = imp->getSoundPropertySink(MSP_USER_MUTE_STATUS, CABIN_B_SINK_ID);
      ETG_TRACE_USR4(("DataModelSound: updateMuteOnPropertiesChange  cockpitStatus = %d , cabinAStatus = %d , cabinBStatus =%d", cockpitStatus, cabinAStatus, cabinBStatus));
      updateCockpitMuteStateAtStartup(cockpitStatus);
      updateCabinAMuteStateAtStartup(cabinAStatus);
      updateCabinBMuteStateAtStartup(cabinBStatus);
   }
}


void DataModelSound::updateCockpitMuteStateAtStartup(int status)
{
   if ((status != -1) && ((*_SystemMuteInfoData).mMute_UnMuteVal_Cockpit == 255))
   {
      updateSystemMuteInfo_Cockpit(status == 1 ? 1 : 2);
   }
}


void DataModelSound::updateCabinAMuteStateAtStartup(int status)
{
   if (status != -1)
   {
      updateSystemMuteInfo_CabinA(status == 1 ? 1 : 2);
   }
}


void DataModelSound::updateCabinBMuteStateAtStartup(int status)
{
   if (status != -1 && ((*_SystemMuteInfoData).mMute_UnMuteVal_CabinB == 255))
   {
      updateSystemMuteInfo_CabinB(status == 1 ? 1 : 2);
   }
}


void DataModelSound::onSoundPropertyUpdate(const int16 nSinkId)
{
   ETG_TRACE_USR4(("DataModelSound: onSoundPropertyUpdate nSinkId = %d,", nSinkId));
   SoundControllerPrj* imp = getSoundController();
   if (imp != NULL)
   {
      int muteStatus = imp->getSoundPropertySink(MSP_USER_MUTE_STATUS, nSinkId);
      ETG_TRACE_USR4(("DataModelSound: onSoundPropertyUpdate nSinkId = %d,", muteStatus));
      switch (nSinkId)
      {
         case COCKPIT_SINK_ID:
            updateCockpitMuteStateAtStartup(muteStatus);
            break;

         case CABIN_A_SINK_ID:
            updateCabinAMuteStateAtStartup(muteStatus);
            break;

         case CABIN_B_SINK_ID:
            updateCabinBMuteStateAtStartup(muteStatus);
            break;

         default:
            break;
      }
   }
}


void DataModelSound::onVolumeCommandValueUpdate(uint32 regionId, const uint8 volumeCmndValue)
{
   ETG_TRACE_USR4(("DataModelSound: onVolumeCommandValueUpdate volumeCmndValue = %d regionId = %d", volumeCmndValue, regionId));
   uint8 currentVolReqCount = _volumeReqCount[regionId];
   switch (volumeCmndValue)
   {
      case DEMUTE_REQUEST :
      case MUTE_REQUEST :
      {
         performMuteDemuteRequest(regionId, (volumeCmndValue == MUTE_REQUEST ? true : false));
         break;
      }
      case VOLUMEUP_REQUEST:
      {
         updateVolumeRequest(1, regionId);
         currentVolReqCount += 1;
         _volumeReqCount[regionId] = currentVolReqCount;
         break;
      }
      case VOLUMEDOWN_REQUEST:
      {
         updateVolumeRequest(-1, regionId);
         currentVolReqCount += 1;
         _volumeReqCount[regionId] = currentVolReqCount;
         break;
      }
      default:
         break;
   }
}


} //namespace DataModel
} //namespace App
#undef DEMUTE_REQUEST
#undef MUTE_REQUEST
#undef VOLUMEUP_REQUEST
#undef VOLUMEDOWN_REQUEST
