/* ***************************************************************************************
* FILE:          DataModelPower.cpp
* SW-COMPONENT:  APPHMI_MASTER
*  DESCRIPTION:  DataModelPower.cpp is part of HMI-Master application data model
*    COPYRIGHT:  (c) 2020-2020 Robert Bosch Car Multimedia GmbH
*
* The reproduction, distribution and utilization of this file as well as the
* communication of its contents to others without express authorization is
* prohibited. Offenders will be held liable for the payment of damages.
* All rights reserved in the event of the grant of a patent, utility model or design.
*
*************************************************************************************** */

#include "hall_std_if.h"
#include "DataModelPower.h"
#include "ProjectBaseTypes.h"
#include "ApplicationSwitchConst.h"
#include "App/Core/RegionHandling/RegionHandling.h"
#include "App/Core/VariantHandling/VariantHandling.h"
//#include "App/Core/ApplicationSwitchServerComponent/ApplicationSwitchServerComponent.h"
#include "DataModelSound.h"
#include "DataModel.h"
#include "App/Core/RVCTestMode/RearViewCamera.h"
#include "spm_core_fi_typesConst.h"
#include "App/Core/HmiState/HmiStateHandlerInterface.h"

#ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
#define ETG_DEFAULT_TRACE_CLASS TR_CLASS_APPHMI_MASTER_DM
#define ETG_I_FILE_PREFIX App::DataModel::DataModelPower::
#include "trcGenProj/Header/DataModelPower.cpp.trc.h"
#endif

using namespace ::App::Core;
using namespace ::App::datamodel;
namespace App {
namespace datamodel {


DataModelPower* DataModelPower::_theModelInstance = 0;


DataModelPower::DataModelPower():
   _variantHandling(NULL)
   , _regionHandling(NULL)
   , _dataModelSound(NULL)
   , _dataModel(NULL)
   , _hmiStateHandler(NULL)
   , _rearViewCamera(NULL)
   , _powerKeyState(HKPOWERBUTTON_NONE)
   , _vehicleSrvClient(NULL)
{
   ETG_TRACE_USR4(("DataModelPower constructor"));
}


DataModelPower::~DataModelPower()
{
   ETG_TRACE_USR4(("DataModelPower Destructor"));
   _variantHandling      = NULL;
   _regionHandling       = NULL;
   _dataModelSound       = NULL;
   _dataModel            = NULL;
   _hmiStateHandler      = NULL;
   _rearViewCamera       = NULL;
   _powerKeyState        = HKPOWERBUTTON_NONE;
   _vehicleSrvClient     = NULL;
}


void DataModelPower::deleteInstance()
{
   if (NULL != _theModelInstance)
   {
      delete _theModelInstance;
      _theModelInstance = NULL;
   }
}


DataModelPower& DataModelPower::getInstance()
{
   if (NULL == _theModelInstance)
   {
      _theModelInstance = new DataModelPower();
   }
   assert(_theModelInstance);
   return *_theModelInstance;
}


bool DataModelPower::onCourierMessage(const HKStatusUpdateMsg& oMsg)
{
   ETG_TRACE_USR4(("DataModelPower::HKStatusUpdateMsg: HkCode: %d", oMsg.GetHkCode()));
   bool retvalue = false;
   uint8 hkState = oMsg.GetHkState();
   if (oMsg.GetHkCode() == HARDKEYCODE_HK_POWER)
   {
      retvalue = true;
      ETG_TRACE_USR4(("DataModelPower::HKStatusUpdateMsg: hkStatepower: %d", hkState));
      if (NULL != _hmiStateHandler)
      {
         switch (_hmiStateHandler->getHmiState())
         {
            case spm_core_fi_types::T_SPM_e32_HMI_STATE__SPM_U32_HMISTATE_SHOW_HMI:
            case spm_core_fi_types::T_SPM_e32_HMI_STATE__SPM_U32_HMISTATE_SHOW_OFF_DIAG:
            case spm_core_fi_types::T_SPM_e32_HMI_STATE__SPM_U32_HMISTATE_SHOW_HMI_HOUR_MODE:
            case spm_core_fi_types::T_SPM_e32_HMI_STATE__SPM_U32_HMISTATE_SHOW_ON_PARTIAL_USER:
            {
               updateHardKeyState(hkState);
               handleHMIStateON(hkState);
               break;
            }
            case spm_core_fi_types::T_SPM_e32_HMI_STATE__SPM_U32_HMISTATE_SHOW_STANDBY:
            case spm_core_fi_types::T_SPM_e32_HMI_STATE__SPM_U32_HMISTATE_SHOW_GOODBYE:
            case spm_core_fi_types::T_SPM_e32_HMI_STATE__SPM_U32_HMISTATE_SHOW_OFF:
            {
               updateHardKeyState(hkState);
               handleHMIStateOff(_hmiStateHandler->getHmiState(), hkState);
               break;
            }
            case spm_core_fi_types::T_SPM_e32_HMI_STATE__SPM_U32_HMISTATE_REMOTE_PROGRAMMING:
            {
               updateHardKeyState(hkState);
               handleHMIStateRemoteProgramming(hkState);
               break;
            }
            default:
            {
               break;
            }
         }
      }
   }
   else if ((oMsg.GetHkCode() == HARDKEYCODE_HK_POWER_LONGPRESS) && (hkState == hmibase::HARDKEYSTATE_UP))
   {
      retvalue = true;
      ETG_TRACE_USR4(("DataModelPower::HKStatusUpdateMsg: hkStatepower: %d", hkState));
      performSPMStateChange();
   }
   else
   {}
   return retvalue;
}


void DataModelPower::updateHardKeyState(uint8 hkState)
{
   ETG_TRACE_USR4(("DataModelPower::updateHardKeyState hkState: %d", hkState));
   switch (hkState)
   {
      case hmibase::HARDKEYSTATE_UP:
      {
         _powerKeyState = HKPOWERBUTTON_MUTE;
         break;
      }
      case hmibase::HARDKEYSTATE_LONG1:
      {
         if (NULL != _variantHandling)
         {
            if (VEHICLE_TYPE_SMART == _variantHandling->getVariantId())
            {
               _powerKeyState = HKPOWERBUTTON_USEROFF;
            }
            else if (VEHICLE_TYPE_COACH == _variantHandling->getVariantId())
            {
               if (NULL != _dataModel)
               {
                  if (!_dataModel->isMasterApplicationForeground())
                  {
                     _powerKeyState = HKPOWERBUTTON_CONTEXTSWITCH;
                  }
                  else
                  {
                     _powerKeyState = HKPOWERBUTTON_REGIONSWITCH;
                  }
               }
            }
            else
            {
               _powerKeyState = HKPOWERBUTTON_NONE;
            }
         }
         break;
      }

      case hmibase::HARDKEYSTATE_LONG2:
      {
         if (NULL != _variantHandling)
         {
            if (VEHICLE_TYPE_SMART != _variantHandling->getVariantId())
            {
               _powerKeyState = HKPOWERBUTTON_STANDBY;
            }
         }
         break;
      }

      case hmibase::HARDKEYSTATE_LONG3:
      {
         if (NULL != _variantHandling)
         {
            if (VEHICLE_TYPE_SMART != _variantHandling->getVariantId())
            {
               _powerKeyState = HKPOWERBUTTON_REBOOT;
            }
         }
         break;
      }

      case hmibase::HARDKEYSTATE_LONGUP:
      {
         break;
      }

      default:
      {
         _powerKeyState = HKPOWERBUTTON_NONE;
         break;
      }
   }
}


void DataModelPower::handleHMIStateON(uint8 hkState)
{
   ETG_TRACE_USR4(("DataModelPower::handleHMIStateON _powerKeyState %d", _powerKeyState));
   uint32 regionID = REGION_INVALID;
   if (NULL != _regionHandling)
   {
      regionID = _regionHandling->getRegionId();
   }
   switch (_powerKeyState)
   {
      case HKPOWERBUTTON_MUTE:
      {
         if ((NULL != _dataModelSound) && ((hkState == hmibase::HARDKEYSTATE_UP)))
         {
            _dataModelSound->performMuteRequest(regionID);
         }
         break;
      }
      case HKPOWERBUTTON_USEROFF:
      {
         updateGpioOnOffState();
         break;
      }

      case HKPOWERBUTTON_CONTEXTSWITCH:
      {
         if ((hkState == hmibase::HARDKEYSTATE_LONGUP) && (NULL != _dataModel))
         {
            _dataModel->performHomeScreenContextSwitch(regionID);
         }
         break;
      }
      case HKPOWERBUTTON_REGIONSWITCH:
      {
         if ((hkState == hmibase::HARDKEYSTATE_LONGUP) && (NULL != _dataModel))
         {
            _dataModel->performHomeScreenRegionSwitch();
         }
         break;
      }

      case HKPOWERBUTTON_STANDBY:
      {
         if (hkState == hmibase::HARDKEYSTATE_LONGUP)
         {
            toggleSPMState();
         }
         break;
      }

      case HKPOWERBUTTON_REBOOT:
      {
         if (NULL != _hmiStateHandler)
         {
            _hmiStateHandler->sendRestartRequest();
         }
         break;
      }

      default:
      {
         break;
      }
   }
}


void DataModelPower::handleHMIStateOff(uint8 hmistate, uint8 hkState) const
{
   ETG_TRACE_USR4(("DataModelPower::handleHMIStateOff: PowerKeyState: %d, HKState: %d", _powerKeyState, hkState));
   if (NULL != _variantHandling)
   {
      switch (_variantHandling->getVariantId())
      {
         case VEHICLE_TYPE_SMART:
         {
            if (hkState == hmibase::HARDKEYSTATE_UP)
            {
               toggleSPMState();
            }
            break;
         }
         default:
         {
            if (hmistate == spm_core_fi_types::T_SPM_e32_HMI_STATE__SPM_U32_HMISTATE_SHOW_GOODBYE)
            {
               if ((_powerKeyState != HKPOWERBUTTON_REBOOT) && ((hkState == hmibase::HARDKEYSTATE_UP) || (hkState == hmibase::HARDKEYSTATE_LONGUP)))
               {
                  toggleSPMState();
               }
               else if ((_powerKeyState == HKPOWERBUTTON_REBOOT))
               {
                  if (NULL != _hmiStateHandler)
                  {
                     _hmiStateHandler->sendRestartRequest();
                  }
               }
               else
               {}
            }
            else
            {
               if ((hkState == hmibase::HARDKEYSTATE_UP) || (hkState == hmibase::HARDKEYSTATE_LONGUP))
               {
                  toggleSPMState();
               }
            }
            break;
         }
      }
   }
}


void DataModelPower::handleHMIStateRemoteProgramming(uint8 hkState) const
{
   ETG_TRACE_USR4(("DataModelPower::handleHMIStateRemoteProgramming: PowerKeyState: %d", _powerKeyState));
   if ((_powerKeyState == HKPOWERBUTTON_REBOOT) && (NULL != _hmiStateHandler))
   {
      _hmiStateHandler->sendRestartRequest();
   }
}


void DataModelPower::toggleSPMState() const
{
   ETG_TRACE_USR4(("DataModelPower::toggleSPMState"));
   if (NULL != _variantHandling)
   {
      switch (_variantHandling->getVariantId())
      {
         case VEHICLE_TYPE_SMART:
         {
            updateGpioOnOffState();
            break;
         }
         default:
         {
            performSPMStateChange();
            break;
         }
      }
   }
}


void DataModelPower::updateGpioOnOffState() const
{
   ETG_TRACE_USR4(("DataModelPower::updateGpioOnOffState"));
   if (NULL != _rearViewCamera)
   {
      _rearViewCamera->onToggleGpioPinOnOffState();
   }
}


void DataModelPower::performSPMStateChange() const
{
   ETG_TRACE_USR4(("DataModelPower::performSPMStateChange"));
   if (NULL != _hmiStateHandler)
   {
      _hmiStateHandler->sendHKeyPressedRequest();
   }
}


void DataModelPower::onRadioCommandValueUpdate(const uint8 radioCmndValue)
{
   switch (radioCmndValue)
   {
      case RADIOPWR_ON:
      {
         if ((NULL != _hmiStateHandler) && (!_hmiStateHandler->isPowerStateON()))
         {
            performSPMStateChange();
         }
         break;
      }
      case RADIOPWR_OFF:
      {
         if ((NULL != _hmiStateHandler) && (_hmiStateHandler->isPowerStateON()))
         {
            performSPMStateChange();
         }
         break;
      }

      default:
         break;
   }
}


} // namespace
} // namespace
