/**
 * @file         : DataModelErrorPopup.cpp
 * @author       : INF4CV - AppHmi_Master Team
 * @addtogroup   : AppHmi_Master
 * @brief        : DataModelErrorPopup is to handle the data model implementation of
 *                 Error popups.
 * @copyright    : (c) 2020-2020 Robert Bosch Car Multimedia GmbH
 *                 The reproduction, distribution and utilization of this file as
 *                 well as the communication of its contents to others without express
 *                 authorization is prohibited. Offenders will be held liable for the
 *                 payment of damages. All rights reserved in the event of the grant
 *                 of a patent, utility model or design.
 */

#include "hmi_trace_if.h"
#include "DataModelErrorPopup.h"
#include "AppHmi_MasterMessages.h"
#include "spm_core_fi_typesConst.h"
#include <Core/HmiState/HmiStateHandlerHelper.h>
#include <Core/RegionHandling/RegionHandlingTypes.h>
#include <Core/CabinConnectionHandler/CabinConnectionUtility.h>
#include <Core/HmiInfoService/HmiInfoServiceServerComponentHandlingInterface.h>


using namespace ::App::Core;


#ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
#define ETG_DEFAULT_TRACE_CLASS TR_CLASS_APPHMI_MASTER_DM
#define ETG_I_FILE_PREFIX App::DataModel::DataModelErrorPopup::
#include "trcGenProj/Header/DataModelErrorPopup.cpp.trc.h"
#endif


namespace App {
namespace datamodel {


DataModelErrorPopup::DataModelErrorPopup(): _regionHandling(NULL)
   , _hmiStateHandler(NULL)
   , _cabinConnectionUtility(NULL)
   , _hmiInfoServiceServerComponent(NULL)
{}


DataModelErrorPopup::~DataModelErrorPopup()
{
   _regionHandling = NULL;
   _hmiStateHandler = NULL;
   _cabinConnectionUtility = NULL;
   _hmiInfoServiceServerComponent = NULL;
}


void DataModelErrorPopup::onHmiStateUpdate(const uint8 hmiState)
{
   ETG_TRACE_USR4(("DataModelErrorPopup: onHmiStateUpdate: hmiState: %d", hmiState));

   if (hmiState == ::spm_core_fi_types::T_SPM_e32_HMI_STATE__SPM_U32_HMISTATE_SHOW_ON_PARTIAL_USER)
   {
      Courier::Message* msg = COURIER_MESSAGE_NEW(SystemFailureUpdate)(REGION_COCKPIT, SYSTEM_CATEGORY_CMC, STATUS_CATEGORY_AVDECC, SYSTEM_STATUS_DISCONNECTED, SYSTEM_GROUP_CATEGORY_NONE);
      if (msg != 0)
      {
         (void)msg->Post();
      }
   }
   else
   {
      Courier::Message* msg = COURIER_MESSAGE_NEW(SystemFailureUpdate)(REGION_COCKPIT, SYSTEM_CATEGORY_CMC, STATUS_CATEGORY_AVDECC, SYSTEM_STATUS_CONNECTED, SYSTEM_GROUP_CATEGORY_NONE);
      if (msg != 0)
      {
         (void)msg->Post();
      }
   }

   if (isSPMHMIStateInfotainmentOFF(hmiState))
   {
      evalAndUpdateHourLogicStatus();
      evalAndUpdateVoltageStatus();
   }
}


void DataModelErrorPopup::onHourLogicExpireStatusUpdate(const bool hourLogicExpireStatus)
{
   ETG_TRACE_USR4(("DataModelErrorPopup: onHourLogicExpireStatusUpdate: hourLogicExpireStatus: %d", hourLogicExpireStatus));
   bool isUpdate = true;
   if (hourLogicExpireStatus)
   {
      isUpdate = ((_hmiStateHandler != NULL) && (isSPMHMIStateInfotainmentOFF(_hmiStateHandler->getHmiState()))) ? true : false;
   }
   if (isUpdate)
   {
      Courier::Message* msg = COURIER_MESSAGE_NEW(SystemFailureUpdate)(REGION_COCKPIT, SYSTEM_CATEGORY_CMC, STATUS_CATEGORY_HOURLOGIC, hourLogicExpireStatus, SYSTEM_GROUP_CATEGORY_NONE);
      if (msg != 0)
      {
         (void)msg->Post();
      }
   }
}


void DataModelErrorPopup::onCMAStatusUpdate(const SystemStatusInfo& info)
{
   ETG_TRACE_USR4(("DataModelErrorPopup: onCMAStatusUpdate: Status: %d, SystemId: %d", info.getStatus(), info.getSystemId()));
   const RegionsIdInfo& tRegionsIdInfo = info.getRegionsIdInfo();
   for (RegionsIdInfo::const_iterator itr = tRegionsIdInfo.begin(); (itr != tRegionsIdInfo.end()); ++itr)
   {
      if (info.getStatus() == SYSTEM_STATUS_CONNECTED)
      {
         Courier::Message* msg = COURIER_MESSAGE_NEW(SystemFailureUpdate)(static_cast<enRegionID>((*itr)), SYSTEM_CATEGORY_CMA, STATUS_CATEGORY_SYSTEM, SYSTEM_STATUS_CONNECTED, SYSTEM_GROUP_CATEGORY_NONE);
         if (msg != 0)
         {
            (void)msg->Post();
         }
      }
      else if (info.getStatus() != SYSTEM_STATUS_NONE)
      {
         Courier::Message* msg = COURIER_MESSAGE_NEW(SystemFailureUpdate)(static_cast<enRegionID>((*itr)), SYSTEM_CATEGORY_CMA, STATUS_CATEGORY_SYSTEM, SYSTEM_STATUS_DISCONNECTED, SYSTEM_GROUP_CATEGORY_NONE);
         if (msg != 0)
         {
            (void)msg->Post();
         }
      }
      else
      {}
   }
}


void DataModelErrorPopup::onAVDECCCMAStatusUpdate(const SystemStatusInfo& info)
{
   ETG_TRACE_USR4(("DataModelErrorPopup: onAVDECCCMAStatusUpdate: Status: %d, SystemId: %d", info.getStatus(), info.getSystemId()));
   if ((info.getStatus() == SYSTEM_STATUS_DISCONNECTED) || (info.getStatus() == SYSTEM_STATUS_CONNECTED))
   {
      const RegionsIdInfo& tRegionsIdInfo = info.getRegionsIdInfo();
      for (RegionsIdInfo::const_iterator itr = tRegionsIdInfo.begin(); (itr != tRegionsIdInfo.end()); ++itr)
      {
         Courier::Message* msg = COURIER_MESSAGE_NEW(SystemFailureUpdate)(static_cast<enRegionID>((*itr)), SYSTEM_CATEGORY_CMA, STATUS_CATEGORY_AVDECC, info.getStatus(), SYSTEM_GROUP_CATEGORY_NONE);
         if (msg != 0)
         {
            (void)msg->Post();
         }
      }
   }
}


void DataModelErrorPopup::onAVDECCCMPStatusUpdate(const SystemStatusInfo& info)
{
   ETG_TRACE_USR4(("DataModelErrorPopup: onAVDECCCMPStatusUpdate: Status: %d, SystemId: %d", info.getStatus(), info.getSystemId()));
   if (info.getSystemPosition() == 1)
   {
      if ((info.getStatus() == SYSTEM_STATUS_DISCONNECTED) || (info.getStatus() == SYSTEM_STATUS_CONNECTED))
      {
         Courier::Message* msg = COURIER_MESSAGE_NEW(SystemFailureUpdate)(static_cast<enRegionID>(REGION_INVALID), SYSTEM_CATEGORY_CMP, STATUS_CATEGORY_AVDECC, info.getStatus(), SYSTEM_GROUP_CATEGORY_NONE);
         if (msg != 0)
         {
            (void)msg->Post();
         }
      }
   }
}


void DataModelErrorPopup::onAVDECCCMGStatusUpdate(const SystemStatusInfo& info)
{
   ETG_TRACE_USR4(("DataModelErrorPopup: onAVDECCCMGStatusUpdate: Status: %d, SystemId: %d", info.getStatus(), info.getSystemId()));
   const RegionsIdInfo& tRegionsIdInfo = info.getRegionsIdInfo();
   if (NULL != _regionHandling)
   {
      for (RegionsIdInfo::const_iterator itr = tRegionsIdInfo.begin(); (itr != tRegionsIdInfo.end()); ++itr)
      {
         SystemEnquiryInfo tSystemEnquiryInfo(static_cast<enRegionID>((*itr)), SYSTEM_CATEGORY_CMG, STATUS_CATEGORY_AVDECC);
         if ((!isGlassA2Configured(*itr)) || (getVideoMixIndex(*itr) == CABIN_VIDEO_MIX_INDEX_NONE))
         {
            if (info.getStatus() == SYSTEM_STATUS_DISCONNECTED)
            {
               Courier::Message* msg = COURIER_MESSAGE_NEW(SystemFailureUpdate)(static_cast<enRegionID>((*itr)), SYSTEM_CATEGORY_CMG, STATUS_CATEGORY_AVDECC, SYSTEM_STATUS_DISCONNECTED, SYSTEM_GROUP_CATEGORY_CMG_CABIN);
               if (msg != 0)
               {
                  (void)msg->Post();
               }
            }
            else if (info.getStatus() == SYSTEM_STATUS_CONNECTED)
            {
               if (_regionHandling->isAllStatusSame(tSystemEnquiryInfo, SYSTEM_STATUS_CONNECTED))
               {
                  Courier::Message* msg = COURIER_MESSAGE_NEW(SystemFailureUpdate)(static_cast<enRegionID>((*itr)), SYSTEM_CATEGORY_CMG, STATUS_CATEGORY_AVDECC, SYSTEM_STATUS_CONNECTED, SYSTEM_GROUP_CATEGORY_CMG_CABIN);
                  if (msg != 0)
                  {
                     (void)msg->Post();
                  }
               }
            }
            else
            {}
         }
         else
         {
            int16 sinkId = getSinkIdForSystemId(info.getSystemId());
            if (info.getStatus() == SYSTEM_STATUS_DISCONNECTED)
            {
               Courier::Message* msg = COURIER_MESSAGE_NEW(SystemFailureUpdate)(static_cast<enRegionID>((*itr)), SYSTEM_CATEGORY_CMG, STATUS_CATEGORY_AVDECC, SYSTEM_STATUS_DISCONNECTED, (_regionHandling->getSystemGroupCategoryForSinkId(sinkId)));
               if (msg != 0)
               {
                  (void)msg->Post();
               }
            }
            else if (info.getStatus() == SYSTEM_STATUS_CONNECTED)
            {
               if (_regionHandling->isAllStatusSame(tSystemEnquiryInfo, SYSTEM_STATUS_CONNECTED, sinkId))
               {
                  Courier::Message* msg = COURIER_MESSAGE_NEW(SystemFailureUpdate)(static_cast<enRegionID>((*itr)), SYSTEM_CATEGORY_CMG, STATUS_CATEGORY_AVDECC, SYSTEM_STATUS_CONNECTED, (_regionHandling->getSystemGroupCategoryForSinkId(sinkId)));
                  if (msg != 0)
                  {
                     (void)msg->Post();
                  }
               }
            }
            else
            {}
         }
      }
   }
}


void DataModelErrorPopup::onAVDECCCMRStatusUpdate(const SystemStatusInfo& info)
{
   ETG_TRACE_USR4(("DataModelErrorPopup: onAVDECCCMRStatusUpdate: Status: %d, SystemId: %d", info.getStatus(), info.getSystemId()));
   if ((info.getStatus() == SYSTEM_STATUS_DISCONNECTED) || (info.getStatus() == SYSTEM_STATUS_CONNECTED))
   {
      Courier::Message* msg = COURIER_MESSAGE_NEW(SystemFailureUpdate)(static_cast<enRegionID>(REGION_INVALID), SYSTEM_CATEGORY_CMR, STATUS_CATEGORY_AVDECC, info.getStatus(), SYSTEM_GROUP_CATEGORY_NONE);
      if (msg != 0)
      {
         (void)msg->Post();
      }
   }
}


void DataModelErrorPopup::onNANOMsgCMAStatusUpdate(const SystemStatusInfo& info)
{
   ETG_TRACE_USR4(("DataModelErrorPopup: onNANOMsgCMAStatusUpdate: Status: %d, SystemId: %d", info.getStatus(), info.getSystemId()));
   if ((info.getStatus() == SYSTEM_STATUS_DISCONNECTED) || (info.getStatus() == SYSTEM_STATUS_CONNECTED))
   {
      const RegionsIdInfo& tRegionsIdInfo = info.getRegionsIdInfo();
      for (RegionsIdInfo::const_iterator itr = tRegionsIdInfo.begin(); (itr != tRegionsIdInfo.end()); ++itr)
      {
         Courier::Message* msg = COURIER_MESSAGE_NEW(SystemFailureUpdate)(static_cast<enRegionID>((*itr)), SYSTEM_CATEGORY_CMA, STATUS_CATEGORY_NANOMSG, info.getStatus(), SYSTEM_GROUP_CATEGORY_NONE);
         if (msg != 0)
         {
            (void)msg->Post();
         }
      }
   }
}


void DataModelErrorPopup::onNANOMsgCMPStatusUpdate(const SystemStatusInfo& info)
{
   ETG_TRACE_USR4(("DataModelErrorPopup: onNANOMsgCMPStatusUpdate: Status: %d, SystemId: %d", info.getStatus(), info.getSystemId()));
   if (info.getSystemPosition() == 1)
   {
      if ((info.getStatus() == SYSTEM_STATUS_DISCONNECTED) || (info.getStatus() == SYSTEM_STATUS_CONNECTED))
      {
         Courier::Message* msg = COURIER_MESSAGE_NEW(SystemFailureUpdate)(static_cast<enRegionID>(REGION_INVALID), SYSTEM_CATEGORY_CMP, STATUS_CATEGORY_NANOMSG, info.getStatus(), SYSTEM_GROUP_CATEGORY_NONE);
         if (msg != 0)
         {
            (void)msg->Post();
         }
      }
   }
}


void DataModelErrorPopup::onNANOMsgCMGStatusUpdate(const SystemStatusInfo& info)
{
   ETG_TRACE_USR4(("DataModelErrorPopup: onNANOMsgCMGStatusUpdate: Status: %d, SystemId: %d", info.getStatus(), info.getSystemId()));
   const RegionsIdInfo& tRegionsIdInfo = info.getRegionsIdInfo();
   if (NULL != _regionHandling)
   {
      for (RegionsIdInfo::const_iterator itr = tRegionsIdInfo.begin(); (itr != tRegionsIdInfo.end()); ++itr)
      {
         SystemEnquiryInfo tSystemEnquiryInfo(static_cast<enRegionID>((*itr)), SYSTEM_CATEGORY_CMG, STATUS_CATEGORY_NANOMSG);
         if ((!isGlassA2Configured(*itr)) || (getVideoMixIndex(*itr) == CABIN_VIDEO_MIX_INDEX_NONE))
         {
            if (info.getStatus() == SYSTEM_STATUS_DISCONNECTED)
            {
               Courier::Message* msg = COURIER_MESSAGE_NEW(SystemFailureUpdate)(static_cast<enRegionID>((*itr)), SYSTEM_CATEGORY_CMG, STATUS_CATEGORY_NANOMSG, SYSTEM_STATUS_DISCONNECTED, SYSTEM_GROUP_CATEGORY_CMG_CABIN);
               if (msg != 0)
               {
                  (void)msg->Post();
               }
            }
            else if (info.getStatus() == SYSTEM_STATUS_CONNECTED)
            {
               if (_regionHandling->isAllStatusSame(tSystemEnquiryInfo, SYSTEM_STATUS_CONNECTED))
               {
                  Courier::Message* msg = COURIER_MESSAGE_NEW(SystemFailureUpdate)(static_cast<enRegionID>((*itr)), SYSTEM_CATEGORY_CMG, STATUS_CATEGORY_NANOMSG, SYSTEM_STATUS_CONNECTED, SYSTEM_GROUP_CATEGORY_CMG_CABIN);
                  if (msg != 0)
                  {
                     (void)msg->Post();
                  }
               }
            }
            else
            {}
         }
         else
         {
            int16 sinkId = getSinkIdForSystemId(info.getSystemId());
            if (info.getStatus() == SYSTEM_STATUS_DISCONNECTED)
            {
               Courier::Message* msg = COURIER_MESSAGE_NEW(SystemFailureUpdate)(static_cast<enRegionID>((*itr)), SYSTEM_CATEGORY_CMG, STATUS_CATEGORY_NANOMSG, SYSTEM_STATUS_DISCONNECTED, (_regionHandling->getSystemGroupCategoryForSinkId(sinkId)));
               if (msg != 0)
               {
                  (void)msg->Post();
               }
            }
            else if (info.getStatus() == SYSTEM_STATUS_CONNECTED)
            {
               if (_regionHandling->isAllStatusSame(tSystemEnquiryInfo, SYSTEM_STATUS_CONNECTED, sinkId))
               {
                  Courier::Message* msg = COURIER_MESSAGE_NEW(SystemFailureUpdate)(static_cast<enRegionID>((*itr)), SYSTEM_CATEGORY_CMG, STATUS_CATEGORY_NANOMSG, SYSTEM_STATUS_CONNECTED, (_regionHandling->getSystemGroupCategoryForSinkId(sinkId)));
                  if (msg != 0)
                  {
                     (void)msg->Post();
                  }
               }
            }
            else
            {}
         }
      }
   }
}


void DataModelErrorPopup::onNANOMsgCMRStatusUpdate(const SystemStatusInfo& info)
{
   ETG_TRACE_USR4(("DataModelErrorPopup: onNANOMsgCMRStatusUpdate: Status: %d, SystemId: %d", info.getStatus(), info.getSystemId()));
   if ((info.getStatus() == SYSTEM_STATUS_DISCONNECTED) || (info.getStatus() == SYSTEM_STATUS_CONNECTED))
   {
      Courier::Message* msg = COURIER_MESSAGE_NEW(SystemFailureUpdate)(static_cast<enRegionID>(REGION_INVALID), SYSTEM_CATEGORY_CMR, STATUS_CATEGORY_NANOMSG, info.getStatus(), SYSTEM_GROUP_CATEGORY_NONE);
      if (msg != 0)
      {
         (void)msg->Post();
      }
   }
}


void DataModelErrorPopup::onVoltageCMCStatusUpdate(const SystemStatusInfo& info)
{
   ETG_TRACE_USR4(("DataModelErrorPopup: onVoltageCMCStatusUpdate: Status: %d, SystemId: %d", info.getStatus(), info.getSystemId()));
   const RegionsIdInfo& tRegionsIdInfo = info.getRegionsIdInfo();
   for (RegionsIdInfo::const_iterator itr = tRegionsIdInfo.begin(); (itr != tRegionsIdInfo.end()); ++itr)
   {
      Courier::Message* msg = COURIER_MESSAGE_NEW(SystemFailureUpdate)(static_cast<enRegionID>((*itr)), SYSTEM_CATEGORY_CMC, STATUS_CATEGORY_VOLTAGE, info.getStatus(), SYSTEM_GROUP_CATEGORY_NONE);
      if (msg != 0)
      {
         (void)msg->Post();
      }
   }
}


void DataModelErrorPopup::onVoltageCMAStatusUpdate(const SystemStatusInfo& info)
{
   ETG_TRACE_USR4(("DataModelErrorPopup: onVoltageCMAStatusUpdate: Status: %d, SystemId: %d", info.getStatus(), info.getSystemId()));
   uint8 voltageStatus = SYSTEM_VOLTAGE_STATUS_NOERROR;
   bool isUpdate = true;
   switch (info.getStatus())
   {
      case SYSTEM_VOLTAGE_STATUS_OVER_WARNING_ERROR:
      case SYSTEM_VOLTAGE_STATUS_UNDER_WARNING_ERROR:
      {
         voltageStatus = info.getStatus();
         isUpdate = ((_hmiStateHandler != NULL) && (isSPMHMIStateInfotainmentOFF(_hmiStateHandler->getHmiState()))) ? true : false;
         break;
      }
      default:
         break;
   }

   if (isUpdate)
   {
      const RegionsIdInfo& tRegionsIdInfo = info.getRegionsIdInfo();
      for (RegionsIdInfo::const_iterator itr = tRegionsIdInfo.begin(); (itr != tRegionsIdInfo.end()); ++itr)
      {
         Courier::Message* msg = COURIER_MESSAGE_NEW(SystemFailureUpdate)(static_cast<enRegionID>((*itr)), SYSTEM_CATEGORY_CMA, STATUS_CATEGORY_VOLTAGE, voltageStatus, SYSTEM_GROUP_CATEGORY_NONE);
         if (msg != 0)
         {
            (void)msg->Post();
         }
      }
   }
}


void DataModelErrorPopup::onVoltageCMPStatusUpdate(const SystemStatusInfo& info)
{
   ETG_TRACE_USR4(("DataModelErrorPopup: onVoltageCMPStatusUpdate: Status: %d, SystemId: %d", info.getStatus(), info.getSystemId()));
   if (info.getSystemPosition() == 1)
   {
      Courier::Message* msg = COURIER_MESSAGE_NEW(SystemFailureUpdate)(static_cast<enRegionID>(REGION_INVALID), SYSTEM_CATEGORY_CMP, STATUS_CATEGORY_VOLTAGE, info.getStatus(), SYSTEM_GROUP_CATEGORY_NONE);
      if (msg != 0)
      {
         (void)msg->Post();
      }
   }
}


void DataModelErrorPopup::onVoltageCMGStatusUpdate(const SystemStatusInfo& info)
{
   ETG_TRACE_USR4(("DataModelErrorPopup: onVoltageCMGStatusUpdate: Status: %d, SystemId: %d", info.getStatus(), info.getSystemId()));
   const RegionsIdInfo& tRegionsIdInfo = info.getRegionsIdInfo();
   if (NULL != _regionHandling)
   {
      for (RegionsIdInfo::const_iterator itr = tRegionsIdInfo.begin(); (itr != tRegionsIdInfo.end()); ++itr)
      {
         bool triggerRequest    = true;
         bool isValidForNoError = false;
         uint8 voltageStatus    = SYSTEM_VOLTAGE_STATUS_NOERROR;
         uint8 groupCategory    = SYSTEM_GROUP_CATEGORY_NONE;
         SystemEnquiryInfo tSystemEnquiryInfo(static_cast<enRegionID>((*itr)), SYSTEM_CATEGORY_CMG, STATUS_CATEGORY_VOLTAGE);
         if ((!isGlassA2Configured(*itr)) || (getVideoMixIndex(*itr) == CABIN_VIDEO_MIX_INDEX_NONE))
         {
            groupCategory     = SYSTEM_GROUP_CATEGORY_CMG_CABIN;
            isValidForNoError = (!_regionHandling->isAnyStatusSame(tSystemEnquiryInfo, SYSTEM_VOLTAGE_STATUS_OVER_WARNING_ERROR)) ? true : false;
            isValidForNoError = ((isValidForNoError) && (!_regionHandling->isAnyStatusSame(tSystemEnquiryInfo, SYSTEM_VOLTAGE_STATUS_UNDER_WARNING_ERROR))) ? true : false;
         }
         else
         {
            int16 sinkId      = getSinkIdForSystemId(info.getSystemId());
            groupCategory     = (_regionHandling->getSystemGroupCategoryForSinkId(sinkId));
            isValidForNoError = (!_regionHandling->isAnyStatusSame(tSystemEnquiryInfo, SYSTEM_VOLTAGE_STATUS_OVER_WARNING_ERROR, sinkId)) ? true : false;
            isValidForNoError = ((isValidForNoError) && (!_regionHandling->isAnyStatusSame(tSystemEnquiryInfo, SYSTEM_VOLTAGE_STATUS_UNDER_WARNING_ERROR, sinkId))) ? true : false;
         }
         if (!isValidForNoError)
         {
            voltageStatus = info.getStatus();
            if (!((info.getStatus() == SYSTEM_VOLTAGE_STATUS_OVER_WARNING_ERROR) || (info.getStatus() == SYSTEM_VOLTAGE_STATUS_UNDER_WARNING_ERROR)))
            {
               triggerRequest = false;
            }
         }
         if (triggerRequest)
         {
            Courier::Message* msg = COURIER_MESSAGE_NEW(SystemFailureUpdate)(static_cast<enRegionID>((*itr)), SYSTEM_CATEGORY_CMG, STATUS_CATEGORY_VOLTAGE, voltageStatus, groupCategory);
            if (msg != 0)
            {
               (void)msg->Post();
            }
         }
      }
   }
}


void DataModelErrorPopup::onVoltageCMRStatusUpdate(const SystemStatusInfo& info)
{
   ETG_TRACE_USR4(("DataModelErrorPopup: onVoltageCMRStatusUpdate: Status: %d, SystemId: %d", info.getStatus(), info.getSystemId()));
   Courier::Message* msg = COURIER_MESSAGE_NEW(SystemFailureUpdate)(static_cast<enRegionID>(REGION_INVALID), SYSTEM_CATEGORY_CMR, STATUS_CATEGORY_VOLTAGE, info.getStatus(), SYSTEM_GROUP_CATEGORY_NONE);
   if (msg != 0)
   {
      (void)msg->Post();
   }
}


void DataModelErrorPopup::onThermalCMCStatusUpdate(const SystemStatusInfo& info)
{
   const RegionsIdInfo& tRegionsIdInfo = info.getRegionsIdInfo();
   for (RegionsIdInfo::const_iterator itr = tRegionsIdInfo.begin(); (itr != tRegionsIdInfo.end()); ++itr)
   {
      Courier::Message* msg = COURIER_MESSAGE_NEW(SystemFailureUpdate)(static_cast<enRegionID>((*itr)), SYSTEM_CATEGORY_CMC, STATUS_CATEGORY_THERMAL, info.getStatus(), SYSTEM_GROUP_CATEGORY_NONE);
      if (msg != 0)
      {
         (void)msg->Post();
      }
   }
}


void DataModelErrorPopup::onThermalCMAStatusUpdate(const SystemStatusInfo& info)
{
   ETG_TRACE_USR4(("DataModelErrorPopup: onThermalCMAStatusUpdate: Status: %d, SystemId: %d", info.getStatus(), info.getSystemId()));

   uint8 thermalStatus = SYSTEM_THERMAL_STATUS_NOERROR;
   const RegionsIdInfo& tRegionsIdInfo = info.getRegionsIdInfo();
   if (info.getStatus() == SYSTEM_THERMAL_STATUS_CRITICAL_HIGH_ERROR)
   {
      thermalStatus = SYSTEM_THERMAL_STATUS_CRITICAL_HIGH_ERROR;
   }

   for (RegionsIdInfo::const_iterator itr = tRegionsIdInfo.begin(); (itr != tRegionsIdInfo.end()); ++itr)
   {
      Courier::Message* msg = COURIER_MESSAGE_NEW(SystemFailureUpdate)(static_cast<enRegionID>((*itr)), SYSTEM_CATEGORY_CMA, STATUS_CATEGORY_THERMAL, thermalStatus, SYSTEM_GROUP_CATEGORY_NONE);
      if (msg != 0)
      {
         (void)msg->Post();
      }
   }
}


void DataModelErrorPopup::onThermalCMPStatusUpdate(const SystemStatusInfo& info)
{
   ETG_TRACE_USR4(("DataModelErrorPopup: onThermalCMPStatusUpdate: Status: %d, SystemId: %d", info.getStatus(), info.getSystemId()));
   if (info.getSystemPosition() == 1)
   {
      Courier::Message* msg = COURIER_MESSAGE_NEW(SystemFailureUpdate)(static_cast<enRegionID>(REGION_INVALID), SYSTEM_CATEGORY_CMP, STATUS_CATEGORY_THERMAL, info.getStatus(), SYSTEM_GROUP_CATEGORY_NONE);
      if (msg != 0)
      {
         (void)msg->Post();
      }
   }
}


void DataModelErrorPopup::onThermalCMGStatusUpdate(const SystemStatusInfo& info)
{
   ETG_TRACE_USR4(("DataModelErrorPopup: onThermalCMGStatusUpdate: Status: %d, SystemId: %d", info.getStatus(), info.getSystemId()));
   const RegionsIdInfo& tRegionsIdInfo = info.getRegionsIdInfo();
   if (NULL != _regionHandling)
   {
      for (RegionsIdInfo::const_iterator itr = tRegionsIdInfo.begin(); (itr != tRegionsIdInfo.end()); ++itr)
      {
         bool triggerRequest = true;
         bool isValidForNoError = false;
         uint8 thermalStatus = SYSTEM_THERMAL_STATUS_NOERROR;
         uint8 groupCategory = SYSTEM_GROUP_CATEGORY_NONE;
         SystemEnquiryInfo tSystemEnquiryInfo(static_cast<enRegionID>((*itr)), SYSTEM_CATEGORY_CMG, STATUS_CATEGORY_THERMAL);
         if ((!isGlassA2Configured(*itr)) || (getVideoMixIndex(*itr) == CABIN_VIDEO_MIX_INDEX_NONE))
         {
            groupCategory = SYSTEM_GROUP_CATEGORY_CMG_CABIN;
            isValidForNoError = (!_regionHandling->isAnyStatusSame(tSystemEnquiryInfo, SYSTEM_THERMAL_STATUS_CRITICAL_HIGH_ERROR)) ? true : false;
         }
         else
         {
            int16 sinkId = getSinkIdForSystemId(info.getSystemId());
            groupCategory = (_regionHandling->getSystemGroupCategoryForSinkId(sinkId));
            isValidForNoError = (!_regionHandling->isAnyStatusSame(tSystemEnquiryInfo, SYSTEM_THERMAL_STATUS_CRITICAL_HIGH_ERROR, sinkId)) ? true : false;
         }
         if (!isValidForNoError)
         {
            thermalStatus = info.getStatus();
            if (!(info.getStatus() == SYSTEM_THERMAL_STATUS_CRITICAL_HIGH_ERROR))
            {
               triggerRequest = false;
            }
         }
         if (triggerRequest)
         {
            Courier::Message* msg = COURIER_MESSAGE_NEW(SystemFailureUpdate)(static_cast<enRegionID>((*itr)), SYSTEM_CATEGORY_CMG, STATUS_CATEGORY_THERMAL, thermalStatus, groupCategory);
            if (msg != 0)
            {
               (void)msg->Post();
            }
         }
      }
   }
}


void DataModelErrorPopup::onThermalCMRStatusUpdate(const SystemStatusInfo& info)
{
   ETG_TRACE_USR4(("DataModelErrorPopup: onThermalCMRStatusUpdate: Status: %d, SystemId: %d", info.getStatus(), info.getSystemId()));
   Courier::Message* msg = COURIER_MESSAGE_NEW(SystemFailureUpdate)(static_cast<enRegionID>(REGION_INVALID), SYSTEM_CATEGORY_CMR, STATUS_CATEGORY_THERMAL, info.getStatus(), SYSTEM_GROUP_CATEGORY_NONE);
   if (msg != 0)
   {
      (void)msg->Post();
   }
}


void DataModelErrorPopup::evalAndUpdateHourLogicStatus()
{
   if (_hmiStateHandler != NULL)
   {
      bool status = _hmiStateHandler->getHourLogicExpireStatus();
      if (status)
      {
         Courier::Message* msg = COURIER_MESSAGE_NEW(SystemFailureUpdate)(REGION_COCKPIT, SYSTEM_CATEGORY_CMC, STATUS_CATEGORY_HOURLOGIC, status, SYSTEM_GROUP_CATEGORY_NONE);
         if (msg != 0)
         {
            (void)msg->Post();
         }
      }
   }
}


void DataModelErrorPopup::evalAndUpdateVoltageStatus()
{
   if (NULL != _regionHandling)
   {
      SupportedRegionsInfo tSupportedRegionsInfo = _regionHandling->getSupportedRegionsInfo();
      for (SupportedRegionsInfo::const_iterator itr = tSupportedRegionsInfo.begin(); (itr != tSupportedRegionsInfo.end()); ++itr)
      {
         uint8 cmaVoltageStatus = _regionHandling->getStatus(SystemEnquiryInfo(static_cast<enRegionID>((*itr).first), SYSTEM_CATEGORY_CMA, STATUS_CATEGORY_VOLTAGE));
         switch (cmaVoltageStatus)
         {
            case SYSTEM_VOLTAGE_STATUS_OVER_WARNING_ERROR:
            case SYSTEM_VOLTAGE_STATUS_UNDER_WARNING_ERROR:
            {
               Courier::Message* msg = COURIER_MESSAGE_NEW(SystemFailureUpdate)(static_cast<enRegionID>((*itr).first), SYSTEM_CATEGORY_CMA, STATUS_CATEGORY_VOLTAGE, cmaVoltageStatus, SYSTEM_GROUP_CATEGORY_NONE);
               if (msg != 0)
               {
                  (void)msg->Post();
               }
               break;
            }
            default:
               break;
         }
      }
   }
}


bool DataModelErrorPopup::isGlassA2Configured(const uint32 regionId)
{
   bool isGlassA2Configured = false;
   if (NULL != _cabinConnectionUtility)
   {
      SinkIdsInfo tSinkIdsInfo = _cabinConnectionUtility->getGlassSinksInfo(regionId, SYSTEM_GROUP_CATEGORY_CMG_CABIN_MIX);
      if (!tSinkIdsInfo.empty())
      {
         isGlassA2Configured = true;
      }
   }
   return isGlassA2Configured;
}


uint8 DataModelErrorPopup::getVideoMixIndex(const uint32 regionId) const
{
   uint8 videoMixIndex = CABIN_VIDEO_MIX_INDEX_NONE;
   if (NULL != _hmiInfoServiceServerComponent)
   {
      videoMixIndex = _hmiInfoServiceServerComponent->getCabinVideoMixIndex(regionId);
   }
   return videoMixIndex;
}


int16 DataModelErrorPopup::getSinkIdForSystemId(const uint8 systemId) const
{
   uint8 sinkId = 0;
   if (NULL != _regionHandling)
   {
      SinkIdsInfo tSinkIdsInfo;
      if ((_regionHandling->fetchSinkIdsInfoForSystemId(tSinkIdsInfo, systemId)) && (!tSinkIdsInfo.empty()))
      {
         /*INFO: Force reading because CMG is mounted to a region / sink */
         sinkId = tSinkIdsInfo[0];
      }
   }
   return sinkId;
}


} //namespace datamodel
} //namespace App
