/**
 * @file         : DataModelControlPanel.h
 * @author       : INF4CV - AppHmi_Master Team
 * @addtogroup   : AppHmi_Master
 * @brief        : DataModelControlPanel is to handle the data model implementation of
 *                 control panel feature.
 * @copyright    : (c) 2020-2020 Robert Bosch Car Multimedia GmbH
 *                 The reproduction, distribution and utilization of this file as
 *                 well as the communication of its contents to others without express
 *                 authorization is prohibited. Offenders will be held liable for the
 *                 payment of damages. All rights reserved in the event of the grant
 *                 of a patent, utility model or design.
 */


#ifndef DATA_MODEL_CONTROL_PANEL_H
#define DATA_MODEL_CONTROL_PANEL_H

#include "AppHmi_MasterStateMachine.h"
#include "hmi_trace_if.h"
#include <CgiExtensions/DataBindingItem.hpp>
#include <Core/RegionHandling/RegionHandlingInterface.h>
#include <Core/CabinConnectionHandler/CabinConnectionHandlerInterface.h>
#include <Core/ContextSwitchHandler/ContextSwitchHandlerInterface.h>
#include <Core/ApplicationSwitchServerComponent/ApplicationSwitchServerComponentInterface.h>

/**
 *  Classes Declaration
 */

namespace App {
namespace datamodel {


/**
 * DataModelControlPanel is used to
 *  - Handle the data model implementation of control panel feature.
 */
class DataModelControlPanel : public ::App::Core::IActiveContextListenerCB
   , public ::App::Core::IRegionListenerCB
#ifdef VARIANT_S_FTR_ENABLE_MIC_ACTIVATION
   , public ::App::Core::ICMAStatusListenerCB
   , public ::App::Core::IMIC1ConnectionStatus
#endif
{
   public:
      /**
       *  Member Functions Declaration
       */
      DataModelControlPanel();
      virtual ~DataModelControlPanel();
      void finalize();
      void initialize();
      void onRegionUpdate(const uint32 regionId);
      void onActiveContextUpdate(const ::App::Core::ApplicationContextInfo& info);
#ifdef VARIANT_S_FTR_ENABLE_MIC_ACTIVATION
      void onCMAStatusUpdate(const ::App::Core::SystemStatusInfo& info);
      virtual void currentMIC1ConnectionStatus(const bool status);
      void onUpdateMic1Status(const bool micStatus, const uint8 keyType);
#endif
      void onApplicationSwitchComplete(const uint32 appId);
      bool getCommonButtonVisibility(const uint32 contextId);

      inline void setRegionHandling(::App::Core::IRegionHandling& imp)
      {
         _regionHandling = (&imp);
      }

      inline void setContextSwitchHandler(::App::Core::IContextSwitchHandler& imp)
      {
         _contextSwitchHandler = (&imp);
      }

      inline void setCabinConnectionHandler(::App::Core::ICabinConnectionHandler& imp)
      {
         _cabinConnectionHandler = (&imp);
      }

      inline void setApplicationSwitchServerComponent(::App::Core::IApplicationSwitchServerComponent& imp)
      {
         _applicationSwitchServerComponent = (&imp);
      }

      COURIER_MSG_MAP_BEGIN(TR_CLASS_APPHMI_MASTER_COURIER_PAYLOAD_MODEL_COMP)
      ON_COURIER_MESSAGE(SubSurfaceUpdMsg)
      ON_COURIER_MESSAGE(GuiStartupFinishedUpdMsg)
      ON_COURIER_MESSAGE(CommonButtonVisibleStatusUpdMsg)
      ON_COURIER_MESSAGE(ExecuteCommonButtonActivationRequestMsg)
      ON_COURIER_MESSAGE(ControlPanelButtonPressUpdate)
      COURIER_MSG_MAP_END()

   private:
      /**
       *  Member Functions Declaration
       */
      DataModelControlPanel(const DataModelControlPanel& obj);
      DataModelControlPanel& operator= (const DataModelControlPanel& obj);
      bool onCourierMessage(const SubSurfaceUpdMsg& oMsg);
      bool onCourierMessage(const GuiStartupFinishedUpdMsg& oMsg);
      bool onCourierMessage(const CommonButtonVisibleStatusUpdMsg& oMsg);
      bool onCourierMessage(const ExecuteCommonButtonActivationRequestMsg& oMsg);
      bool onCourierMessage(const ControlPanelButtonPressUpdate& msg);
      void initActivationLockedContextsInfo();
      void evalCommonButtonActivationRequest(const bool status);
      void executeCommonButtonActivationRequest(const bool status);
#ifdef VARIANT_S_FTR_ENABLE_MIC_ACTIVATION
      bool isMicIconEnable(const uint8 status);
      void updateMicIcon(const uint32 regionId, const uint8 status);
#endif

      inline bool isSwitchOngoing() const
      {
         bool isValid = (_displayReqStatus != -1) ? true : false;
         return isValid;
      }

      inline bool isDisplayStatusSameAsCurrent(const bool status)
      {
         bool isValid = ((_displayReqStatus == -1) && (_displayStatus == status)) ? true : false;
         return isValid;
      }

      inline bool isDisplayStatusSameAsRequest(const bool status)
      {
         bool isValid = ((_displayReqStatus != -1) && (_displayReqStatus == status)) ? true : false;
         return isValid;
      }

      inline void clear()
      {
         _displayStatus                    = false;
         _queuedReqStatus                  = -1;
         _displayReqStatus                 = -1;
         _regionHandling                   = NULL;
         _contextSwitchHandler             = NULL;
         _cabinConnectionHandler           = NULL;
         _applicationSwitchServerComponent = NULL;
         _activationLockedContextsInfo.clear();
      }
      /**
       *  Member Variables Declaration
       */
      bool _displayStatus;
      int _queuedReqStatus;
      int _displayReqStatus;
      ::std::vector< uint32 > _activationLockedContextsInfo;
      ::App::Core::IRegionHandling* _regionHandling;
      ::App::Core::IContextSwitchHandler* _contextSwitchHandler;
      ::App::Core::ICabinConnectionHandler* _cabinConnectionHandler;
      ::App::Core::IApplicationSwitchServerComponent* _applicationSwitchServerComponent;
      DataBindingItem<ControlPanelInfoDataBindingSource> _controlPanelInfoDB;
      DataBindingItem<Mic1KeyInfoDataBindingSource> _mic1KeyInfoDB;
};


} //namespace datamodel
} //namespace App


#endif /* DATA_MODEL_CONTROL_PANEL_H */
