/**************************************************************************************
* @file         : MasterUtility.cpp
* @author       :
* @addtogroup   : Master HALL
* @brief        : Class contains necessary business logic required to handle utility functions.
* @copyright    : (c) 2017-2020 Robert Bosch Car Multimedia GmbH
*                 The reproduction, distribution and utilization of this file as
*                 well as the communication of its contents to others without express
*                 authorization is prohibited. Offenders will be held liable for the
*                 payment of damages. All rights reserved in the event of the grant
*                 of a patent, utility model or design.
**************************************************************************************/

#include "AppUtils/HmiMacros.h"
#include <stdio.h>
#include<sstream>
#include "MasterUtility.h"
#define DP_S_IMPORT_INTERFACE_KDS_FI
#include "dp_kds_if.h"
#define FM_FREQ_CONVERT       1000
#define FM_FREQ_FRACTION      100
#define ARRAY_SIZE            20
#define START_WINDOW_FMLIST   3
#define THIRD_POS_FMLIST      2
#define LAST_WINDOW_FMLIST    6
#include "AppHmi_MasterStateMachine.h"
#ifdef DP_DATAPOOL_ID
#define DP_S_IMPORT_INTERFACE_FI
#include "dp_generic_if.h"
#endif


namespace App {
namespace Core {
/**
 * Description     : Destructor of class MasterUtility
 */
MasterUtility::~MasterUtility()
{
}


/**
 * Description     : Constructor of class MasterUtility
 */
MasterUtility::MasterUtility()
{
   //ETG_I_REGISTER_FILE();
}


uint32 MasterUtility::_AMHighFrequency = 0;
uint32 MasterUtility::_AMLowFrequency = 0;
uint32 MasterUtility::_FMHighFrequency = 0;
uint32 MasterUtility::_FMLowFrequency = 0;
uint8 MasterUtility::_AMStepSize = 0;
uint8 MasterUtility::_FMStepSize = 0;

void MasterUtility::setFMFreqMinMax(const uint32& FMLowFrequency, const uint32& FMHighFrequency)
{
   _FMLowFrequency = FMLowFrequency;
   _FMHighFrequency = FMHighFrequency;
}


void MasterUtility::setAMFreqMinMax(const uint32& AMLowFrequency, const uint32& AMHighFrequency)
{
   _AMLowFrequency = AMLowFrequency;
   _AMHighFrequency = AMHighFrequency;
}


/**
 * Description     : Function implements to convert the given frequency into required format for the display.
 *
 * @param[in]      : frequency : Frequency selected by the user.
 * @return         : std::string
 */
std::string MasterUtility::formatFrequency(const uint32& frequency)
{
   char freqString[ARRAY_SIZE] = "\0";

   if ((frequency >= _FMLowFrequency) && (frequency <= _FMHighFrequency))
   {
      uint8 frequencyFraction = 100;
      int mhz = frequency / FM_FREQ_CONVERT;
      int fraction = 0;
      // ETG_TRACE_USR3(("MasterUtility : check before _FMStepSize in Master utility  %d", _FMStepSize));
      //if (_FMStepSize == 50)
      if (1)
      {
         frequencyFraction = 10;     // Required to show two digits after decimal if step size is 50
         fraction = (frequency % FM_FREQ_CONVERT) / frequencyFraction;
         //snprintf(freqString, sizeof freqString, "%d.%02d MHz", mhz, fraction);
         snprintf(freqString, sizeof freqString, "%d.%02d", mhz, fraction);
      }
      else
      {
         fraction = (frequency % FM_FREQ_CONVERT) / frequencyFraction;
         // snprintf(freqString, sizeof freqString, "%d.%-2dMHz", mhz, fraction);
         snprintf(freqString, sizeof freqString, "%d.%-2d", mhz, fraction);
      }
   }
   else if ((frequency >= _AMLowFrequency) && (frequency <= _AMHighFrequency))
   {
      snprintf(freqString, sizeof freqString, "%d kHz", frequency);
   }
   return freqString;
}


/**
 * Description     : Function implements to convert the given frequency into required format for the display for testmode classes.
 *
 * @param[in]      : frequency : Frequency selected by the user.
 * @param[in]      : band : the band for whcih the frequency is selected.
 * @return         : std::string
 */


std::string MasterUtility::formatTestmodeFrequency(const uint32& frequency, std::string band)
{
   char freqString[ARRAY_SIZE] = "\0";

   uint8 frequencyFraction = 100;
   int mhz = frequency / FM_FREQ_CONVERT;
   int fraction = 0;

   if (band == "FM")
   {
      if (_FMStepSize == 50)
      {
         frequencyFraction = 10;     // Required to show two digits after decimal if step size is 50
         fraction = (frequency % FM_FREQ_CONVERT) / frequencyFraction;
         snprintf(freqString, sizeof freqString, "%d.%02d", mhz, fraction);
      }
      else
      {
         fraction = (frequency % FM_FREQ_CONVERT) / frequencyFraction;
         snprintf(freqString, sizeof freqString, "%d.%-2d", mhz, fraction);
      }
   }
   else if (band == "AM")
   {
      snprintf(freqString, sizeof freqString, "%d", frequency);
   }
   else if (band == "DAB")
   {
      fraction = (frequency % FM_FREQ_CONVERT);
      snprintf(freqString, sizeof freqString, "%d.%03d", mhz, fraction);
   }

   return freqString;
}


/**
 * Description     : Function implements to convert the given EU frequency into required format for the display.
 *
 * @param[in]      : frequency : Frequency selected by the user.
 * @return         : std::string
 */
std::string MasterUtility::formatGadgetFrequency(const uint32& frequency)
{
   char freqString[ARRAY_SIZE] = "\0";

   if ((frequency >= _FMLowFrequency) && (frequency <= _FMHighFrequency))
   {
      uint8 frequencyFraction = 100;
      int mhz = frequency / FM_FREQ_CONVERT;
      int fraction = 0;
      if (_FMStepSize == 50)
      {
         frequencyFraction = 10;     // Required to show two digits after decimal if step size is 50
         fraction = (frequency % FM_FREQ_CONVERT) / frequencyFraction;
         snprintf(freqString, sizeof freqString, "%d.%02d", mhz, fraction);
      }
      else
      {
         fraction = (frequency % FM_FREQ_CONVERT) / frequencyFraction;
         snprintf(freqString, sizeof freqString, "%d.%d", mhz, fraction);
      }
   }
   else if ((frequency >= _AMLowFrequency) && (frequency <= _AMHighFrequency))
   {
      snprintf(freqString, sizeof freqString, "%d", frequency);
   }
   return freqString;
}


/**
 * Description     : Function implements to convert the given frequency into required format for the display.
 *
 * @param[in]      : frequency : Frequency selected by the user.
 * @return         : std::string
 */
std::string MasterUtility::formatFreqwithBand(const uint32& frequency)
{
   char freqString[ARRAY_SIZE] = "\0";
   if ((frequency >= _FMLowFrequency) && (frequency <= _FMHighFrequency))
   {
      uint8 frequencyFraction = 100;
      int mhz = frequency / FM_FREQ_CONVERT;
      int fraction = 0;
      if (_FMStepSize == 50)
      {
         frequencyFraction = 10;     // Required to show two digits after decimal if step size is 50
         fraction = (frequency % FM_FREQ_CONVERT) / frequencyFraction;
         snprintf(freqString, sizeof freqString, "%d.%02d FM", mhz, fraction);
      }
      else
      {
         fraction = (frequency % FM_FREQ_CONVERT) / frequencyFraction;
         snprintf(freqString, sizeof freqString, "%d.%-2d FM", mhz, fraction);
      }
   }
   else if ((frequency >= _AMLowFrequency) && (frequency <= _AMHighFrequency))
   {
      snprintf(freqString, sizeof freqString, "%d AM", frequency);
   }
   return freqString;
}


/**
 * Description     : Function implements to convert uint8 value to string.
 *
 * @param[in]      : u8Value : uint8 value.
 * @return         : std::string
*/
std::string MasterUtility::u8ConversionToString(uint8 u8Value)
{
   char stringToConv[ARRAY_SIZE] = "\0";

   snprintf(stringToConv, sizeof stringToConv, "%d", u8Value);

   return stringToConv;
}


/**
 * Description     : Function implements to convert uint16 value to string.
 *
 * @param[in]      : u16Value : uint16 value.
 * @return         : std::string
 */
std::string MasterUtility::u16ConversionToString(uint16 u16Value)
{
   char stringToConv[ARRAY_SIZE] = "\0";

   snprintf(stringToConv, sizeof stringToConv, "%d", u16Value);

   return stringToConv;
}


/**
 * Description     : Function implements to convert uint32 value to string.
 *
 * @param[in]      : u16Value : uint16 value.
 * @return         : std::string
 */
std::string MasterUtility::u32ConversionToString(uint32 u32Value)
{
   char stringToConv[ARRAY_SIZE] = "\0";

   snprintf(stringToConv, sizeof stringToConv, "%d", u32Value);

   return stringToConv;
}


/**
 * Description     : Function implements to convert uint16 value to Hex value string.
 *
 * @param[in]      : u16Value : uint16 value.
 * @return         : std::string
 */
std::string MasterUtility::u16ConversionToHexString(uint16 u16Value)
{
   char stringToConv[ARRAY_SIZE] = "\0";

   snprintf(stringToConv, sizeof stringToConv, "%x", u16Value);

   return stringToConv;
}


/**
 * Description     : Function implements to concatenate two values.
 *
 * @param[in]      : u8Value : uint8 value.
 * @param[in]      : u8Value : uint8 value.
 * @return         : std::string
*/
std::string MasterUtility::quadConcatenate(const uint8& u8Val1, const uint8& u8Val2)
{
   char sQuadConcat[ARRAY_SIZE] = "\0";

   snprintf(sQuadConcat, sizeof sQuadConcat, "%d/%d", u8Val1, u8Val2);

   return sQuadConcat;
}


void MasterUtility::setAMFMStepSizeFromKDS(const uint8& KDSAMStepSize, const uint8& KDSFMStepSize)
{
   _AMStepSize = KDSAMStepSize;
   _FMStepSize = KDSFMStepSize;
}


/**
 * Description     : Function implements to format FM direct frequency.
 *
 * @param[in]      : float : currentFreq value.
 * @return         : uint32
*/
uint32 MasterUtility::formatFMDirectFreq(const float& currentFreq)
{
   uint32 fmfreq = 0;
   if ((currentFreq >= _FMLowFrequency) && (currentFreq <= _FMHighFrequency))
   {
      fmfreq = (uint32)(currentFreq / FM_FREQ_FRACTION);
      fmfreq = fmfreq * FM_FREQ_FRACTION;
   }
   return fmfreq;
}


/**
 * Description     : Function implements to format AM direct frequency.
 *
 * @param[in]      : float : currentFreq value.
 * @return         : uint32
*/
uint32 MasterUtility::formatAMDirectFreq(const float& currentFreq)
{
   uint32 frequency = (uint32)currentFreq;
   if ((currentFreq >= _AMLowFrequency) && (currentFreq <= _AMHighFrequency))
   {
      uint32 mod = 0;
      uint32 val = 0;
      if (_AMStepSize != 0)
      {
         mod = frequency % _AMStepSize;
         val = frequency / _AMStepSize;
      }
      frequency = (uint32)val * _AMStepSize;

      if (mod > 0)
      {
         frequency = frequency + _AMStepSize;
      }
   }
   return frequency;
}


/**
 * Description     : Function implements to calculate FM list Start Index.
 *
 * @param[in]      : listSize : FM list size.
 * @param[in]      : selectedElemID : Tuned station position in FM list.
 * @return         : uint32
 */
uint32 MasterUtility::getListStartIndex(const uint8& listSize, const uint32& selectedElemID)
{
   uint8 lastWindow = listSize - START_WINDOW_FMLIST;
   uint8 startIndex = 0;
   if (listSize > LAST_WINDOW_FMLIST)
   {
      if (selectedElemID < START_WINDOW_FMLIST)
      {
         startIndex = 0;
      }
      else if ((selectedElemID >= START_WINDOW_FMLIST) && (selectedElemID < lastWindow))
      {
         startIndex = selectedElemID - THIRD_POS_FMLIST;
      }
      else
      {
         startIndex = listSize - LAST_WINDOW_FMLIST;
      }
   }
   return startIndex;
}


std::string MasterUtility::getSecureString(const Candera::String& string)
{
   static std::string secureString = "";
   SECURE_FEATSTD_STRING_ACCESS_BEGIN(string); // locks the access of GetCString within this scope.
   secureString = string.GetCString();
   SECURE_FEATSTD_STRING_ACCESS_END()
   return secureString;
}


/**
 * Description     : Function to convert Carrier Noise value to valid string format
 *
 * @param[in]      : value : Carrier Noise value
 * @return         : std::string
 */
std::string MasterUtility::formatDTVCNdata(const uint16& value)
{
   std::ostringstream s;
   s << value;
   std::string str = s.str();
   str = str + " dB";
   return str;
}


/**
 * Description     : Function to convert PER value to valid string format
 *
 * @param[in]      : value : PER value
 * @param[in]      : layer : A or B or C
 * @return         : std::string
 */
std::string MasterUtility::formatDTVPERdata(const uint16& value, char layer)
{
   std::ostringstream s;
   s << value;
   std::string str = s.str();
   str = str + " %" + " (" + layer + ")";
   return str;
}


int MasterUtility::getVariantSelection_KDSValue()
{
   tU8 variantSelectionvalue;
   dp_tclKdsAIVIVariantCoding  m_dp_AIVIVariantcoding;
   m_dp_AIVIVariantcoding.u8GetVariantSelection(variantSelectionvalue);
   //variantSelectionvalue = 1;
   return variantSelectionvalue;
}


unsigned int MasterUtility::getKDSVehicleType()
{
   uint8 u8VehicleType[2] = {0};
   uint16 u16VehicleType = 0;
   if (DP_S32_NO_ERR == DP_s32GetConfigItem("VehicleInformation", "VehicleType", &u8VehicleType[0], 2))
   {
      //etg_trace_usr4(("navidatapoolconfig::getkdsvehicletype, first byte : %x, second byte : %x", u8vehicletype[0], u8vehicletype[1]));

      //motorola-highlow-order needs to be reversed
      u16VehicleType = static_cast<uint16>((u8VehicleType[0] << 8) | u8VehicleType[1]);
      //etg_trace_comp(("masterutility::getkdsvehicletype, vehicle type : %d", u16VehicleType));
   }
   //u16vehicletype = 0;//testing
   //etg_trace_usr4(("masterutility::getkdsvehicletype, vehicle type : %d", u16VehicleType));
   return u16VehicleType;
}


#ifdef VARIANT_S_FTR_ENABLE_ANTI_THEFT
unsigned int MasterUtility::getDpAntiTheft()
{
   uint8 u8AntiTheft = 0;
   dp_tclKdsSystemConfiguration1  m_dp_tclKdsSystemConfiguration1;
   m_dp_tclKdsSystemConfiguration1.u8GetAntiTheft(u8AntiTheft);
   //variantSelectionvalue = 1;
   return 	u8AntiTheft;
}


#endif
} // namespace Core
} // namespace App
