/**
 * @file         : SystemMonitorHandler.h
 * @author       : INF4CV - AppHmi_Master Team
 * @addtogroup   : AppHmi_Master
 * @brief        : SystemMonitorHandler is to monitor the the CAN signal information
 * @copyright    : (c) 2020-2020 Robert Bosch Car Multimedia GmbH
 *                 The reproduction, distribution and utilization of this file as
 *                 well as the communication of its contents to others without express
 *                 authorization is prohibited. Offenders will be held liable for the
 *                 payment of damages. All rights reserved in the event of the grant
 *                 of a patent, utility model or design.
 */


#ifndef SYSTEM_MONITOR_HANDLER_H
#define SYSTEM_MONITOR_HANDLER_H


#include "SystemMonitorHandlerInterfaces.h"
#include <App/Core/AudioInterface/SoundControllerPrjInterface.h>
#include <App/Core/AudioInterface/ConnectionControllerPrjInterface.h>


/**
 *  Defines Declaration
 */

#define NS_MASTER_AUDIO ::hmibase::apphmi_master::audio


/**
 *  Classes Declaration
 */

namespace App {
namespace Core {


class SystemMonitorHandler : public ISystemMonitorHandler
   , public IMuteStateListenerCB
   , public IConnectionInfoUpdate
   , public IVolumeChangeListenerCB
{
   public:
      /**
       *  Member Functions Declaration
       */
      SystemMonitorHandler();
      virtual ~SystemMonitorHandler();
      void onMuteStateUpdate(const uint32 sinkId, const int state);
      void onVolumeChangeUpdate(const uint32 sinkId, const int volume);
      void onUpdateConnectionInfo(std::map< int, NS_MASTER_AUDIO::Connection* > connectionStatusInfo);

      inline void initialize()
      {
         if (NULL != _soundController.get())
         {
            _soundController->addMuteStateListener((*this));
            _soundController->addVolumeChangeListener((*this));
         }
         if (NULL != _connectionController.get())
         {
            _connectionController->vRegisterforConnectionInfoUpdate(this);
         }
      }

      inline void finalize()
      {
         if (NULL != _soundController.get())
         {
            _soundController->removeMuteStateListener((*this));
            _soundController->removeVolumeChangeListener((*this));
         }
         if (NULL != _connectionController.get())
         {
            _connectionController->vUnRegisterforConnectionInfoUpdate(this);
         }
      }

      inline void setRegionHandling(::App::Core::IRegionHandling& imp)
      {
         _regionHandling = (&imp);
      }

      inline void setVehicleSrvClient(::App::Core::IVehicleSrvClient& imp)
      {
         _vehicleServiceClient = (&imp);
      }

      inline void setMediaClientHandler(::App::Core::IMediaClientHandler& imp)
      {
         _mediaClientHandler = (&imp);
      }

      inline void setSoundController(const ::boost::shared_ptr< ISoundControllerPrj >& imp)
      {
         _soundController = imp;
      }

      inline void setResourceController(const ::boost::shared_ptr< IResourceControllerPrj >& imp)
      {
         _resourceController = imp;
      }

      inline void setConnectionController(const ::boost::shared_ptr< IConnectionControllerPrj >& imp)
      {
         _connectionController = imp;
      }

   private:
      /**
       *  Member Functions Declaration
       */
      bool isValidSource(const uint32 sourceId, const uint32 deviceId);
      bool fetchActiveSourceCMDIndexForRegion(const std::map< int, NS_MASTER_AUDIO::Connection* >& info, const uint32 regionId, uint32& index);
      uint8 getCMDMuteState(const int state) const;
      uint8 getCMDAudioSourceIndex(const uint32 sourceId, const uint32 deviceId) const;
      SystemMonitorHandler(const SystemMonitorHandler& obj);
      SystemMonitorHandler& operator= (const SystemMonitorHandler& obj);
      ::std::string getDeviceTypeforDeviceId(const uint32 sourceId, const uint32 deviceId) const;
      void initializeAudioSourceToCMDIndexInfo();

      inline uint32 getCurrentCMDSourceIndexForRegion(const uint32 regionId) const
      {
         uint32 tIndex = 0;
         ::std::map< uint32, uint32 >::const_iterator itr = _currentCMDSourceIndexInfo.find(regionId);
         if (itr != _currentCMDSourceIndexInfo.end())
         {
            tIndex = itr->second;
         }
         return tIndex;
      }
      /**
       *  Member Variables Declaration
       */
      uint32 _currentCMDSourceIndex;
      IRegionHandling* _regionHandling;
      IVehicleSrvClient* _vehicleServiceClient;
      ::App::Core::IMediaClientHandler* _mediaClientHandler;
      ::std::map< uint32, uint32 > _currentCMDSourceIndexInfo;
      ::boost::shared_ptr< ISoundControllerPrj > _soundController;
      ::boost::shared_ptr< IResourceControllerPrj > _resourceController;
      ::boost::shared_ptr< IConnectionControllerPrj > _connectionController;
      /* INFO:
       * Key:
       *  -> Param1: Source Index
       *  -> Param2: Device type
       * Value:
       *  -> CMD index
       */
      ::std::map< ::std::pair< uint32, ::std::string >, uint32 > _audioSourceCMDIndexInfo;
};


} //namespace Core
} //namespace App


/**
 *  undef
 */

#undef NS_MASTER_AUDIO


#endif /* SYSTEM_MONITOR_HANDLER_H */
