/**
 * @file         : SyncDisplayHandler.cpp
 * @author       : INF4CV - AppHmi_Master Team
 * @addtogroup   : AppHmi_Master
 * @brief        : SyncDisplayHandler is to handle HALL implementation for
 *                 Syncdisplay_main_fi service
 * @copyright    : (c) 2017-2020 Robert Bosch Car Multimedia GmbH
 *                 The reproduction, distribution and utilization of this file as
 *                 well as the communication of its contents to others without express
 *                 authorization is prohibited. Offenders will be held liable for the
 *                 payment of damages. All rights reserved in the event of the grant
 *                 of a patent, utility model or design.
 */


#include "hall_std_if.h"
#include "SyncDisplayHandler.h"
#include "spm_core_fi_typesConst.h"
#include "AppHmi_MasterStateMachine.h"
#include "syncdisplay_main_fi_typesConst.h"
#include <Core/VariantHandling/VariantHandling.h>


using namespace ::spm_core_fi_types;
using namespace ::syncdisplay_main_fi;
using namespace ::syncdisplay_main_fi_types;


#ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
#define ETG_DEFAULT_TRACE_CLASS           TR_CLASS_APPHMI_MASTER_HALL
#define ETG_I_TRACE_CHANNEL               TR_TTFIS_APPHMI_MASTER
#define ETG_I_TTFIS_CMD_PREFIX            "APPHMI_MASTER_"
#define ETG_I_FILE_PREFIX                 App::Core::SyncDisplayHandler::
#include "trcGenProj/Header/SyncDisplayHandler.cpp.trc.h"
#endif


namespace App {
namespace Core {


SyncDisplayHandler::SyncDisplayHandler() : _mic1Status(false)
   , _isAnimationPlayed(false)
   , _isFirstAppRequested(false)
   , _guiInitializationStatus(false)
   , _isLastHMIStateDiagnosis(false)
   , _regIdAnimationStatusUpdate(0)
   , _regIdEarlyVideoLayerUpdate(0)
   , _variantHandling(NULL)
   , _hmiStateHandler(NULL)
   , _syncDisplayProxy(Syncdisplay_main_fiProxy::createProxy("syncdisplayFiPort", *this))
{
   ETG_TRACE_USR4(("SyncDisplayHandler: CTOR"));
   if (NULL != _syncDisplayProxy.get())
   {
      StartupSync::getInstance().registerPropertyRegistrationIF(this, _syncDisplayProxy->getPortName());
   }
}


SyncDisplayHandler::~SyncDisplayHandler()
{
   ETG_TRACE_USR4(("SyncDisplayHandler: DTOR"));
   _mic1Status                 = false;
   _isAnimationPlayed          = false;
   _isFirstAppRequested        = false;
   _guiInitializationStatus    = false;
   _isLastHMIStateDiagnosis    = false;
   _regIdAnimationStatusUpdate = 0;
   _regIdEarlyVideoLayerUpdate = 0;
   _variantHandling            = NULL;
   _hmiStateHandler            = NULL;
   _hmiFirstAppListeners.clear();
}


void SyncDisplayHandler::onAvailable(const ::boost::shared_ptr< asf::core::Proxy >& proxy, const asf::core::ServiceStateChange& stateChange)
{
   StartupSync::getInstance().onAvailable(proxy, stateChange);
   if ((NULL != _syncDisplayProxy.get()) && (proxy == _syncDisplayProxy))
   {
      _regIdAnimationStatusUpdate = _syncDisplayProxy->sendAnimationStatusUpReg(*this);
      _regIdEarlyVideoLayerUpdate = _syncDisplayProxy->sendEarlyVideoLayerStatusUpReg(*this);
   }
}


void SyncDisplayHandler::onUnavailable(const ::boost::shared_ptr< asf::core::Proxy >& proxy, const asf::core::ServiceStateChange& stateChange)
{
   StartupSync::getInstance().onUnavailable(proxy, stateChange);
   if ((NULL != _syncDisplayProxy.get()) && (proxy == _syncDisplayProxy))
   {
      _syncDisplayProxy->sendAnimationStatusRelUpRegAll();
      _syncDisplayProxy->sendEarlyVideoLayerStatusRelUpRegAll();
      _regIdAnimationStatusUpdate = 0;
      _regIdEarlyVideoLayerUpdate = 0;
   }
}


void SyncDisplayHandler::registerProperties(const boost::shared_ptr< asf::core::Proxy>& /* proxy */, const asf::core::ServiceStateChange& /* stateChange */)
{}


void SyncDisplayHandler::deregisterProperties(const boost::shared_ptr< asf::core::Proxy>& /* proxy */, const asf::core::ServiceStateChange& /* stateChange */)
{}


void SyncDisplayHandler::onAnimationStatusError(const ::boost::shared_ptr< Syncdisplay_main_fiProxy >& proxy, const boost::shared_ptr< AnimationStatusError >& error)
{
   if ((NULL != _syncDisplayProxy.get()) && (proxy == _syncDisplayProxy) && (NULL != error.get()) && (error->getAct() == _regIdAnimationStatusUpdate))
   {
      ETG_TRACE_ERR(("SyncDisplayHandler: onAnimationStatusError"));
   }
}


void SyncDisplayHandler::onAnimationStatusStatus(const ::boost::shared_ptr< Syncdisplay_main_fiProxy >& proxy, const boost::shared_ptr< AnimationStatusStatus >& status)
{
   if ((NULL != _syncDisplayProxy.get()) && (proxy == _syncDisplayProxy) && (NULL != status.get()) && (status->getAct() == _regIdAnimationStatusUpdate))
   {
      ETG_TRACE_USR4(("SyncDisplayHandler: onAnimationStatusStatus: <-: Status: %d", ETG_CENUM(T_e8_Animation_Status, status->getT_e8_Animation_Status())));
      sm_animation_status = status->getT_e8_Animation_Status();
      bool isAnimationStatusValid = (status->getT_e8_Animation_Status() == T_e8_Animation_Status__ANIMATION_ERROR) ? true : false;
      isAnimationStatusValid = ((isAnimationStatusValid) || ((status->getT_e8_Animation_Status() == T_e8_Animation_Status__ANIMATION_FINISHED))) ? true : false;
      isAnimationStatusValid = ((isAnimationStatusValid) && (!_isAnimationPlayed)) ? true : false;
      if (isAnimationStatusValid)
      {
         _isAnimationPlayed = true;
         if (_guiInitializationStatus)
         {
            sendShowFirstHMIAppRequest();
         }
      }
   }
}


void SyncDisplayHandler::onEarlyVideoLayerStatusError(const ::boost::shared_ptr< Syncdisplay_main_fiProxy >& proxy, const ::boost::shared_ptr< EarlyVideoLayerStatusError >& error)
{
   if ((NULL != _syncDisplayProxy.get()) && (proxy == _syncDisplayProxy) && (NULL != error.get()) && (error->getAct() == _regIdEarlyVideoLayerUpdate))
   {
      ETG_TRACE_ERR(("SyncDisplayHandler: onEarlyVideoLayerStatusError"));
   }
}


void SyncDisplayHandler::onEarlyVideoLayerStatusStatus(const ::boost::shared_ptr< Syncdisplay_main_fiProxy >& proxy, const ::boost::shared_ptr< EarlyVideoLayerStatusStatus >& status)
{
   if ((NULL != _syncDisplayProxy.get()) && (proxy == _syncDisplayProxy) && (NULL != status.get()) && (status->getAct() == _regIdEarlyVideoLayerUpdate))
   {
      ETG_TRACE_USR4(("SyncDisplayHandler: onEarlyVideoLayerStatusStatus: <-: Status: %d", ETG_CENUM(T_e8_EarlyVideoLayer_Status, status->getEEarlyVideoLayerStatus())));
      bool isEarlyVideoLayerStatusValid = (status->getEEarlyVideoLayerStatus() == T_e8_EarlyVideoLayer_Status__VIDEOLAYER_IDLE) ? true : false;
      isEarlyVideoLayerStatusValid = ((isEarlyVideoLayerStatusValid) && (getCurrentAnimationState() == T_e8_Animation_Status__ANIMATION_RUNNING)) ? true : false;
      isEarlyVideoLayerStatusValid = ((isEarlyVideoLayerStatusValid) && (!_isAnimationPlayed)) ? true : false;
      if (isEarlyVideoLayerStatusValid)
      {
         _isAnimationPlayed = true;
         if (_guiInitializationStatus)
         {
            sendShowFirstHMIAppRequest();
         }
      }
   }
}


void SyncDisplayHandler::onHmiStateUpdate(const uint8 hmiState)
{
   _isLastHMIStateDiagnosis = false;
   switch (hmiState)
   {
      case T_SPM_e32_HMI_STATE__SPM_U32_HMISTATE_SHOW_HMI:
      case T_SPM_e32_HMI_STATE__SPM_U32_HMISTATE_SHOW_ON_PARTIAL_USER:
      {
         if ((NULL != _hmiStateHandler) && (_hmiStateHandler->getLastHmiState() == T_SPM_e32_HMI_STATE__SPM_U32_HMISTATE_SHOW_OFF_DIAG))
         {
            _isLastHMIStateDiagnosis = true;
            if (_guiInitializationStatus)
            {
               sendShowFirstHMIAppRequest();
            }
         }
         break;
      }
      case T_SPM_e32_HMI_STATE__SPM_U32_HMISTATE_SHOW_OFF_PHONE:
      {
         if (_guiInitializationStatus)
         {
            sendShowTempHMIAppRequest();
         }
         break;
      }
      case T_SPM_e32_HMI_STATE__SPM_U32_HMISTATE_SHOW_HMI_HOUR_MODE:
      case T_SPM_e32_HMI_STATE__SPM_U32_HMISTATE_REMOTE_PROGRAMMING:
      {
         if (_guiInitializationStatus)
         {
            sendShowFirstHMIAppRequest();
         }
         break;
      }
      default:
      {
         break;
      }
   }
}


#ifdef VARIANT_S_FTR_ENABLE_MIC_ACTIVATION
void SyncDisplayHandler::onMIC1StatusUpdate(const bool status)
{
   if (_mic1Status != status)
   {
      _mic1Status = status;
      if ((status) && (_guiInitializationStatus))
      {
         sendShowFirstHMIAppRequest();
      }
   }
}


#endif


void SyncDisplayHandler::onGUIInitializationStatus(const bool status)
{
#ifndef GEN3X86
   if (_guiInitializationStatus != status)
   {
      _guiInitializationStatus = status;
      if (status)
      {
         bool isValidToShowFirstHMI = (_isAnimationPlayed) ? true : false;
         isValidToShowFirstHMI      = ((isValidToShowFirstHMI) || (_mic1Status)) ? true : false;
         isValidToShowFirstHMI      = ((isValidToShowFirstHMI) || (isHMIStateHourLogic())) ? true : false;
         isValidToShowFirstHMI      = ((isValidToShowFirstHMI) || (_isLastHMIStateDiagnosis)) ? true : false;
         isValidToShowFirstHMI      = ((isValidToShowFirstHMI) || (isHMIStateRemoteProgramming())) ? true : false;
         if (isValidToShowFirstHMI)
         {
            sendShowFirstHMIAppRequest();
         }
         else
         {
            bool isValidToShowTempHMI = (isHMIStateOnTel()) ? true : false;
            if (isValidToShowTempHMI)
            {
               sendShowTempHMIAppRequest();
            }
         }
      }
   }
#else
   ETG_TRACE_USR4(("SyncDisplayHandler: onGUIInitializationStatus => for LSIM do an unconditional call of sendShowFirstHMIAppRequest to skip startup animation"));
   sendShowFirstHMIAppRequest();
#endif
}


void SyncDisplayHandler::sendFirstApplicationActiveUpdate()
{
   static bool isFirstAppActivated = false;
   bool isValid                    = (isFirstAppActivated == false) ? true : false;
   isValid                         = ((isValid) && (NULL != _variantHandling)) ? true : false;
   isValid                         = ((isValid) && (NULL != _hmiStateHandler)) ? true : false;
   isValid                         = ((isValid) && (NULL != _syncDisplayProxy.get())) ? true : false;
   isValid                         = ((isValid) && (_syncDisplayProxy->isAvailable())) ? true : false;
   isValid                         = ((isValid) && (_isFirstAppRequested)) ? true : false;
   if (isValid)
   {
      isFirstAppActivated = true;
      ETG_TRACE_USR1(("SyncDisplayHandler: sendFirstApplicationActiveUpdate: ->: Timestamp %u ms", hmibase::utils::Ticker::getTickCountMsec()));
      _syncDisplayProxy->sendSetFirstScreenReadyStart(*this, T_tU8_ScreenType__HMI_FIRST_SCREEN);
      /* INFO: First pic drawn is updated from SM (=> Disclaimer state) for SMART variant */
      if (_variantHandling->getVariantId() != VEHICLE_TYPE_SMART)
      {
         _hmiStateHandler->sendFirstPicDrawnUpdate();
      }
      HMI_STARTUP_INVESTIGATION_POST(hmibase::utils::trace::StartupInvestigation::MP_DISABLE_STARTUP_LOGO)
   }
}


void SyncDisplayHandler::sendShowTempHMIAppRequest()
{
   ETG_TRACE_USR4(("SyncDisplayHandler: sendShowTempHMIAppRequest: -> _isFirstAppRequested=%d", _isFirstAppRequested));
   if (!_isFirstAppRequested)
   {
      ::Courier::Message* msg = COURIER_MESSAGE_NEW(ShowTempHMIAppRequestMsg)();
      if (NULL != msg)
      {
         (void)msg->Post();
      }
   }
}


void SyncDisplayHandler::sendShowFirstHMIAppRequest()
{
   ETG_TRACE_USR4(("SyncDisplayHandler: sendShowFirstHMIAppRequest: -> isFirstAppRequested=%d", _isFirstAppRequested));
   if (!_isFirstAppRequested)
   {
      _isFirstAppRequested = true;
      ::Courier::Message* msg = COURIER_MESSAGE_NEW(ShowFirstHMIAppRequestMsg)();
      if (NULL != msg)
      {
         (void)msg->Post();
      }
      for (::std::vector< IHmiFirstAppListenerCB* >::iterator itr = _hmiFirstAppListeners.begin(); (itr != _hmiFirstAppListeners.end()); ++itr)
      {
         if (NULL != (*itr))
         {
            (*itr)->onHmiFirstAppRequest();
         }
      }
   }
}


uint8 SyncDisplayHandler::getCurrentAnimationState() const
{
   uint8 state = 0;
   if ((NULL != _syncDisplayProxy.get()) && (_syncDisplayProxy->hasAnimationStatus()))
   {
      state = _syncDisplayProxy->getAnimationStatus().getT_e8_Animation_Status();
   }
   return state;
}


bool SyncDisplayHandler::isHMIStateOnTel()
{
   bool isValid = false;
   if ((NULL != _hmiStateHandler) && (_hmiStateHandler->getHmiState() == T_SPM_e32_HMI_STATE__SPM_U32_HMISTATE_SHOW_OFF_PHONE))
   {
      isValid = true;
   }
   return isValid;
}


bool SyncDisplayHandler::isHMIStateHourLogic()
{
   bool isValid = false;
   if ((NULL != _hmiStateHandler) && (_hmiStateHandler->getHmiState() == T_SPM_e32_HMI_STATE__SPM_U32_HMISTATE_SHOW_HMI_HOUR_MODE))
   {
      isValid = true;
   }
   return isValid;
}


bool SyncDisplayHandler::isHMIStateRemoteProgramming()
{
   bool isValid = false;
   if ((NULL != _hmiStateHandler) && (_hmiStateHandler->getHmiState() == T_SPM_e32_HMI_STATE__SPM_U32_HMISTATE_REMOTE_PROGRAMMING))
   {
      isValid = true;
   }
   return isValid;
}


} //namespace Core
} //namespace App
