/**
 * @file         : RegionHandlingTypes.h
 * @author       : INF4CV - AppHmi_Master Team
 * @addtogroup   : AppHmi_Master
 * @brief        : RegionHandlingTypes is to handle the RegionHandling
 *                 feature types implementation
 * @copyright    : (c) 2019-2020 Robert Bosch Car Multimedia GmbH
 *                 The reproduction, distribution and utilization of this file as
 *                 well as the communication of its contents to others without express
 *                 authorization is prohibited. Offenders will be held liable for the
 *                 payment of damages. All rights reserved in the event of the grant
 *                 of a patent, utility model or design.
 */


#ifndef REGION_HANDLING_TYPES_H
#define REGION_HANDLING_TYPES_H


#include "AppHmi_MasterTypes.h"
#include "RegionHandlingConstants.h"


/**
 *  Classes Declaration
 */

namespace App {
namespace Core {


class SinkStaticInfo
{
   public:
      /**
       *  Member Functions Declaration
       */
      SinkStaticInfo()
      {
         clear();
      }

      virtual ~SinkStaticInfo()
      {
         clear();
      }

      SinkStaticInfo(const SinkStaticInfo& obj)
      {
         clear();
         *this = obj;
      }

      SinkStaticInfo& operator = (const SinkStaticInfo& obj)
      {
         clear();
         if (this != (&obj))
         {
            _sinkId        = obj._sinkId;
            _group         = obj._group;
            _category      = obj._category;
            _groupCategory = obj._groupCategory;
            _regionId      = obj._regionId;
            _sinkName      = obj._sinkName;
         }
         return (*this);
      }

      void clear()
      {
         _sinkId        = -1;
         _group         = SYSTEM_GROUP_NONE;
         _category      = SYSTEM_CATEGORY_NONE;
         _groupCategory = SYSTEM_GROUP_CATEGORY_NONE;
         _regionId      = REGION_INVALID;
         _sinkName      = "";
      }

      inline void setSinkId(const int16 sinkId)
      {
         _sinkId = sinkId;
      }

      inline int16 getSinkId() const
      {
         return _sinkId;
      }

      inline void setSystemGroup(const uint8 group)
      {
         _group = group;
      }

      inline uint8 getSystemGroup() const
      {
         return _group;
      }

      inline void setSystemCategory(const uint8 category)
      {
         _category = category;
      }

      inline uint8 getSystemCategory() const
      {
         return _category;
      }

      inline void setSystemGroupCategory(const uint8 category)
      {
         _groupCategory = category;
      }

      inline uint8 getSystemGroupCategory() const
      {
         return _groupCategory;
      }

      inline void setRegionId(const uint32 regionId)
      {
         _regionId = regionId;
      }

      inline uint32 getRegionId() const
      {
         return _regionId;
      }

      inline void setSinkName(const ::std::string& name)
      {
         _sinkName = name;
      }

      inline const ::std::string& getSinkName() const
      {
         return _sinkName;
      }

      inline ::std::string& getMutableSinkName()
      {
         return _sinkName;
      }

   private:
      /**
       *  Member Variables Declaration
       */
      int16 _sinkId;
      uint8 _group;
      uint8 _category;
      uint8 _groupCategory;
      uint32 _regionId;
      ::std::string _sinkName;
};


class SystemRegionInfo
{
   public:
      /**
       *  Member Functions Declaration
       */
      SystemRegionInfo()
      {
         clear();
      }

      virtual ~SystemRegionInfo()
      {
         clear();
      }

      SystemRegionInfo(const uint32 regionId)
      {
         clear();
         setRegionId(regionId);
      }

      SystemRegionInfo(const SystemRegionInfo& obj)
      {
         clear();
         *this = obj;
      }

      SystemRegionInfo& operator = (const SystemRegionInfo& obj)
      {
         clear();
         if (this != (&obj))
         {
            _group         = obj._group;
            _regionId      = obj._regionId;
            _sinkIdsInfo   = obj._sinkIdsInfo;
            _sourceIdsInfo = obj._sourceIdsInfo;
         }
         return (*this);
      }

      void clear()
      {
         _group    = SYSTEM_GROUP_NONE;
         _regionId = REGION_INVALID;
         _sinkIdsInfo.clear();
         _sourceIdsInfo.clear();
      }

      inline void setSystemGroup(const uint8 group)
      {
         _group = group;
      }

      inline uint8 getSystemGroup() const
      {
         return _group;
      }

      inline void setRegionId(const uint32 regionId)
      {
         _regionId = regionId;
      }

      inline uint32 getRegionId() const
      {
         return _regionId;
      }

      void setSinkIdsInfo(const SinkIdsInfo& info)
      {
         _sinkIdsInfo = info;
      }

      const SinkIdsInfo& getSinkIdsInfo() const
      {
         return _sinkIdsInfo;
      }

      SinkIdsInfo& getMutableSinkIdsInfo()
      {
         return _sinkIdsInfo;
      }

      void setSourceIdsInfo(const SourceIdsInfo& info)
      {
         _sourceIdsInfo = info;
      }

      const SourceIdsInfo& getSourceIdsInfo() const
      {
         return _sourceIdsInfo;
      }

      SourceIdsInfo& getMutableSourceIdsInfo()
      {
         return _sourceIdsInfo;
      }

      bool isSinkIdAvailable(const int16 sinkId) const
      {
         bool isValid = false;
         if (::std::find(_sinkIdsInfo.begin(), _sinkIdsInfo.end(), sinkId) != _sinkIdsInfo.end())
         {
            isValid = true;
         }
         return isValid;
      }

      bool isSourceIdAvailable(const int16 sourceId) const
      {
         bool isValid = false;
         if (::std::find(_sourceIdsInfo.begin(), _sourceIdsInfo.end(), sourceId) != _sourceIdsInfo.end())
         {
            isValid = true;
         }
         return isValid;
      }

      void addToSinkIdsInfo(const int16 sinkId)
      {
         if (::std::find(_sinkIdsInfo.begin(), _sinkIdsInfo.end(), sinkId) == _sinkIdsInfo.end())
         {
            _sinkIdsInfo.push_back(sinkId);
         }
      }

      void removeFromSinkIdsInfo(const int16 sinkId)
      {
         SinkIdsInfo::iterator itr = ::std::find(_sinkIdsInfo.begin(), _sinkIdsInfo.end(), sinkId);
         if (itr != _sinkIdsInfo.end())
         {
            (void)_sinkIdsInfo.erase(itr);
         }
      }

      void addToSourceIdsInfo(const int16 sourceId)
      {
         if (::std::find(_sourceIdsInfo.begin(), _sourceIdsInfo.end(), sourceId) == _sourceIdsInfo.end())
         {
            _sourceIdsInfo.push_back(sourceId);
         }
      }

      void removeFromSourceIdsInfo(const int16 sourceId)
      {
         SourceIdsInfo::iterator itr = ::std::find(_sourceIdsInfo.begin(), _sourceIdsInfo.end(), sourceId);
         if (itr != _sourceIdsInfo.end())
         {
            (void)_sourceIdsInfo.erase(itr);
         }
      }

   private:
      /**
       *  Member Variables Declaration
       */
      uint8 _group;
      uint32 _regionId;
      SinkIdsInfo _sinkIdsInfo;
      SourceIdsInfo _sourceIdsInfo; /* NOTE: Region specific sources are not declared as requirement */
};


class SystemRegionsInfo
{
   public:
      /**
       *  Member Functions Declaration
       */
      SystemRegionsInfo()
      {
         clear();
      }

      virtual ~SystemRegionsInfo()
      {
         clear();
      }

      SystemRegionsInfo(const SystemRegionsInfo& obj)
      {
         clear();
         *this = obj;
      }

      SystemRegionsInfo& operator = (const SystemRegionsInfo& obj)
      {
         clear();
         if (this != (&obj))
         {
            _systemRegionsInfo = obj._systemRegionsInfo;
         }
         return (*this);
      }

      void clear()
      {
         _systemRegionsInfo.clear();
      }

      void setSystemRegionsInfo(const SystemRegionsInfoType& info)
      {
         _systemRegionsInfo = info;
      }

      const SystemRegionsInfoType& getSystemRegionsInfo() const
      {
         return _systemRegionsInfo;
      }

      SystemRegionsInfoType& getMutableSystemRegionsInfo()
      {
         return _systemRegionsInfo;
      }

      RegionsIdInfo getSystemRegionsIdInfo() const
      {
         RegionsIdInfo tInfo;
         for (SystemRegionsInfoType::const_iterator itr = _systemRegionsInfo.begin(); (itr != _systemRegionsInfo.end()); ++itr)
         {
            if (::std::find(tInfo.begin(), tInfo.end(), (*itr).getRegionId()) == tInfo.end())
            {
               tInfo.push_back((*itr).getRegionId());
            }
         }
         return tInfo;
      }

      const SystemRegionInfo* getSystemRegionInfo(const uint32 regionId) const
      {
         const SystemRegionInfo* imp = NULL;
         for (SystemRegionsInfoType::const_iterator itr = _systemRegionsInfo.begin(); (itr != _systemRegionsInfo.end()); ++itr)
         {
            if ((*itr).getRegionId() == regionId)
            {
               imp = (&(*itr));
            }
         }
         return imp;
      }

      SystemRegionInfo* getMutableSystemRegionInfo(const uint32 regionId)
      {
         SystemRegionInfo* imp = const_cast< SystemRegionInfo* >(getSystemRegionInfo(regionId));
         return imp;
      }

      const SystemRegionInfo* getSystemRegionInfoForSinkId(const int16 sinkId) const
      {
         const SystemRegionInfo* imp = NULL;
         for (SystemRegionsInfoType::const_iterator itr = _systemRegionsInfo.begin(); (itr != _systemRegionsInfo.end()); ++itr)
         {
            if ((*itr).isSinkIdAvailable(sinkId) == true)
            {
               imp = (&(*itr));
            }
         }
         return imp;
      }

      SystemRegionInfo* getMutableSystemRegionInfoForSinkId(const int16 sinkId)
      {
         SystemRegionInfo* imp = const_cast< SystemRegionInfo* >(getSystemRegionInfoForSinkId(sinkId));
         return imp;
      }

      const SystemRegionInfo* getSystemRegionInfoForSourceId(const int16 sourceId) const
      {
         const SystemRegionInfo* imp = NULL;
         for (SystemRegionsInfoType::const_iterator itr = _systemRegionsInfo.begin(); (itr != _systemRegionsInfo.end()); ++itr)
         {
            if ((*itr).isSourceIdAvailable(sourceId) == true)
            {
               imp = (&(*itr));
            }
         }
         return imp;
      }

      SystemRegionInfo* getMutableSystemRegionInfoForSourceId(const int16 sourceId)
      {
         SystemRegionInfo* imp = const_cast< SystemRegionInfo* >(getSystemRegionInfoForSourceId(sourceId));
         return imp;
      }

      /*
       * NOTE:
       *    -> The region id "REGION_INVALID" denotes the group of first region
       */
      uint8 getSystemGroup(const uint32 regionId = REGION_INVALID) const
      {
         uint8 tGroup   = SYSTEM_GROUP_NONE;
         for (SystemRegionsInfoType::const_iterator itr = _systemRegionsInfo.begin(); (itr != _systemRegionsInfo.end()) ; ++itr)
         {
            if ((regionId == REGION_INVALID) || (regionId == (*itr).getRegionId()))
            {
               tGroup    = (*itr).getSystemGroup();
               break;
            }
         }
         return tGroup;
      }

      /*
       * NOTE:
       *    -> REGION_INVALID denotes analyse all the regions of the system
       */
      bool isSystemGroupSupported(const uint8 systemGroup, const uint32 regionId = REGION_INVALID) const
      {
         bool status = false;
         bool terminate = false;
         for (SystemRegionsInfoType::const_iterator itr = _systemRegionsInfo.begin(); ((itr != _systemRegionsInfo.end()) && (!terminate)) ; ++itr)
         {
            if ((regionId == REGION_INVALID) || (regionId == (*itr).getRegionId()))
            {
               if ((*itr).getSystemGroup() == systemGroup)
               {
                  status = true;
                  break;
               }
               terminate = (regionId != REGION_INVALID) ? true : false;
            }
         }
         return status;
      }

      bool fetchRegionIdForSinkId(const int16 sinkId, uint32& regionId) const
      {
         bool isFetched = false;
         for (SystemRegionsInfoType::const_iterator itr = _systemRegionsInfo.begin(); (itr != _systemRegionsInfo.end()); ++itr)
         {
            const SinkIdsInfo& tSinkIdsInfo = (*itr).getSinkIdsInfo();
            if (::std::find(tSinkIdsInfo.begin(), tSinkIdsInfo.end(), sinkId) != tSinkIdsInfo.end())
            {
               isFetched = true;
               regionId  = (*itr).getRegionId();
               break;
            }
         }
         return isFetched;
      }

      bool fetchSinkIdsInfo(SinkIdsInfo& info, const uint32 regionId = REGION_INVALID) const
      {
         bool isFetched = false;
         bool terminate = false;
         for (SystemRegionsInfoType::const_iterator itr = _systemRegionsInfo.begin(); ((itr != _systemRegionsInfo.end()) && (!terminate)); ++itr)
         {
            if ((regionId == REGION_INVALID) || (regionId == (*itr).getRegionId()))
            {
               terminate                       = (regionId != REGION_INVALID) ? true : false;
               const SinkIdsInfo& tSinkIdsInfo = (*itr).getSinkIdsInfo();
               for (SinkIdsInfo::const_iterator tItr = tSinkIdsInfo.begin(); (tItr != tSinkIdsInfo.end()); ++tItr)
               {
                  if (::std::find(info.begin(), info.end(), (*tItr)) == info.end())
                  {
                     isFetched = true;
                     info.push_back((*tItr));
                  }
               }
            }
         }
         return isFetched;
      }

      bool fetchSourceIdsInfo(SourceIdsInfo& info, const uint32 regionId = REGION_INVALID) const
      {
         bool isFetched = false;
         bool terminate = false;
         for (SystemRegionsInfoType::const_iterator itr = _systemRegionsInfo.begin(); ((itr != _systemRegionsInfo.end()) && (!terminate)); ++itr)
         {
            if ((regionId == REGION_INVALID) || (regionId == (*itr).getRegionId()))
            {
               terminate                         = (regionId != REGION_INVALID) ? true : false;
               const SinkIdsInfo& tSourceIdsInfo = (*itr).getSourceIdsInfo();
               for (SourceIdsInfo::const_iterator tItr = tSourceIdsInfo.begin(); (tItr != tSourceIdsInfo.end()); ++tItr)
               {
                  if (::std::find(info.begin(), info.end(), (*tItr)) == info.end())
                  {
                     isFetched = true;
                     info.push_back((*tItr));
                  }
               }
            }
         }
         return isFetched;
      }

      bool fetchSystemsGroupInfo(SystemsGroupInfo& info, const uint32 regionId = REGION_INVALID) const
      {
         bool isFetched = false;
         bool terminate = false;
         for (SystemRegionsInfoType::const_iterator itr = _systemRegionsInfo.begin(); ((itr != _systemRegionsInfo.end()) && (!terminate)) ; ++itr)
         {
            if ((regionId == REGION_INVALID) || (regionId == (*itr).getRegionId()))
            {
               info.push_back((*itr).getSystemGroup());
               isFetched = true;
               terminate = (regionId != REGION_INVALID) ? true : false;
            }
         }
         return isFetched;
      }

      bool fetchSinkIdsInfoForGroup(SinkIdsInfo& info, const uint8 systemGroup, const uint32 regionId = REGION_INVALID) const
      {
         bool isFetched = false;
         bool terminate = false;
         for (SystemRegionsInfoType::const_iterator itr = _systemRegionsInfo.begin(); ((itr != _systemRegionsInfo.end()) && (!terminate)); ++itr)
         {
            if (((*itr).getSystemGroup() == systemGroup) && ((regionId == REGION_INVALID) || (regionId == (*itr).getRegionId())))
            {
               terminate                       = (regionId != REGION_INVALID) ? true : false;
               const SinkIdsInfo& tSinkIdsInfo = (*itr).getSinkIdsInfo();
               for (SinkIdsInfo::const_iterator tItr = tSinkIdsInfo.begin(); (tItr != tSinkIdsInfo.end()); ++tItr)
               {
                  if (::std::find(info.begin(), info.end(), (*tItr)) == info.end())
                  {
                     isFetched = true;
                     info.push_back((*tItr));
                  }
               }
            }
         }
         return isFetched;
      }

      void addToSystemRegionsInfo(const SystemRegionInfo& info)
      {
         if (NULL == getSystemRegionInfo(info.getRegionId()))
         {
            _systemRegionsInfo.push_back(info);
         }
      }

      void removeFromSystemRegionsInfo(const uint32 regionId)
      {
         for (SystemRegionsInfoType::iterator itr = _systemRegionsInfo.begin(); (itr != _systemRegionsInfo.end()); ++itr)
         {
            if ((*itr).getRegionId() == regionId)
            {
               _systemRegionsInfo.erase(itr);
               break;
            }
         }
      }

      void addToSinkIdsInfo(const int16 sinkId, const uint32 regionId)
      {
         for (SystemRegionsInfoType::iterator itr = _systemRegionsInfo.begin(); (itr != _systemRegionsInfo.end()); ++itr)
         {
            if ((*itr).getRegionId() == regionId)
            {
               (*itr).addToSinkIdsInfo(sinkId);
               break;
            }
         }
      }

      void removeFromSinkIdsInfo(const int16 sinkId, const uint32 regionId = REGION_INVALID)
      {
         for (SystemRegionsInfoType::iterator itr = _systemRegionsInfo.begin(); (itr != _systemRegionsInfo.end()); ++itr)
         {
            if ((regionId == REGION_INVALID) || ((*itr).getRegionId() == regionId))
            {
               (*itr).removeFromSinkIdsInfo(sinkId);
               if (regionId != REGION_INVALID)
               {
                  break;
               }
            }
         }
      }

      void addToSourceIdsInfo(const int16 sourceId, const uint32 regionId)
      {
         for (SystemRegionsInfoType::iterator itr = _systemRegionsInfo.begin(); (itr != _systemRegionsInfo.end()); ++itr)
         {
            if ((*itr).getRegionId() == regionId)
            {
               (*itr).addToSourceIdsInfo(sourceId);
               break;
            }
         }
      }

      void removeFromSourceIdsInfo(const int16 sourceId, const uint32 regionId = REGION_INVALID)
      {
         for (SystemRegionsInfoType::iterator itr = _systemRegionsInfo.begin(); (itr != _systemRegionsInfo.end()); ++itr)
         {
            if ((regionId == REGION_INVALID) || ((*itr).getRegionId() == regionId))
            {
               (*itr).removeFromSourceIdsInfo(sourceId);
               if (regionId != REGION_INVALID)
               {
                  break;
               }
            }
         }
      }

      bool loadSystemRegionInfo(const SinksStaticInfo& info, const uint8 systemCategory, const ::std::string& sinkName)
      {
         bool isLoaded = false;
         for (SinksStaticInfo::const_iterator itr = info.begin(); (itr != info.end()); ++itr)
         {
            if (((*itr).getSystemCategory() == systemCategory) && ((*itr).getSinkName() == sinkName))
            {
               addToSystemRegionsInfo(SystemRegionInfo((*itr).getRegionId()));
               isLoaded              = true;
               SystemRegionInfo* imp = getMutableSystemRegionInfo((*itr).getRegionId());
               if (NULL != imp) //for safety
               {
                  imp->addToSinkIdsInfo((*itr).getSinkId());
                  imp->setSystemGroup((*itr).getSystemGroup());
               }
               break;
            }
         }
         return isLoaded;
      }

   private:
      /**
       *  Member Variables Declaration
       */
      SystemRegionsInfoType _systemRegionsInfo;
};


class SystemInfo
{
   public:
      /**
       *  Member Functions Declaration
       */
      SystemInfo()
      {
         clear();
      }

      virtual ~SystemInfo()
      {
         clear();
      }

      SystemInfo(const uint8 systemId)
      {
         clear();
         setSystemId(systemId);
      }

      SystemInfo(const SystemInfo& obj)
      {
         clear();
         *this = obj;
      }

      SystemInfo& operator = (const SystemInfo& obj)
      {
         clear();
         if (this != (&obj))
         {
            _category          = obj._category;
            _systemId          = obj._systemId;
            _position          = obj._position;
            _avdeccStatus      = obj._avdeccStatus;
            _systemStatus      = obj._systemStatus;
            _lastSystemStatus  = obj._lastSystemStatus;
            _nanoMsgStatus     = obj._nanoMsgStatus;
            _voltageStatus     = obj._voltageStatus;
            _thermalStatus     = obj._thermalStatus;
            _powerModeStatus   = obj._powerModeStatus;
            _entityId          = obj._entityId;
            _systemIP          = obj._systemIP;
            _sourceIdsInfo     = obj._sourceIdsInfo;
            _systemRegionsInfo = obj._systemRegionsInfo;
         }
         return (*this);
      }

      void clear()
      {
         _category         = SYSTEM_CATEGORY_NONE;
         _systemId         = 0;
         _position         = 0;
         _systemStatus     = SYSTEM_STATUS_NONE;
         _lastSystemStatus = SYSTEM_STATUS_NONE;
         _avdeccStatus     = SYSTEM_STATUS_NONE;
         _nanoMsgStatus    = SYSTEM_STATUS_NONE;
         _voltageStatus    = SYSTEM_STATUS_NONE;
         _thermalStatus    = SYSTEM_STATUS_NONE;
         _powerModeStatus  = SYSTEM_STATUS_NONE;
         _entityId         = 0;
         _systemIP         = "";
         _sourceIdsInfo.clear();
         _systemRegionsInfo.clear();
      }

      inline void setSystemCategory(const uint8 category)
      {
         _category = category;
      }

      inline uint8 getSystemCategory() const
      {
         return _category;
      }

      inline void setSystemId(const uint8 systemId)
      {
         _systemId = systemId;
      }

      inline uint8 getSystemId() const
      {
         return _systemId;
      }

      inline void setSystemPosition(const uint8 position)
      {
         _position = position;
      }

      inline uint8 getSystemPosition() const
      {
         return _position;
      }

      inline void setSystemStatus(const uint8 status)
      {
         _systemStatus = status;
      }

      inline uint8 getSystemStatus() const
      {
         return _systemStatus;
      }

      inline void setLastSystemStatus(const uint8 status)
      {
         _lastSystemStatus = status;
      }

      inline uint8 getLastSystemStatus() const
      {
         return _lastSystemStatus;
      }

      inline void setAVDECCStatus(const uint8 status)
      {
         _avdeccStatus = status;
      }

      inline uint8 getAVDECCStatus() const
      {
         return _avdeccStatus;
      }

      inline void setNANOMsgStatus(const uint8 status)
      {
         _nanoMsgStatus = status;
      }

      inline uint8 getNANOMsgStatus() const
      {
         return _nanoMsgStatus;
      }

      inline void setVoltageStatus(const uint8 status)
      {
         _voltageStatus = status;
      }

      inline uint8 getVoltageStatus() const
      {
         return _voltageStatus;
      }

      inline void setThermalStatus(const uint8 status)
      {
         _thermalStatus = status;
      }

      inline uint8 getThermalStatus() const
      {
         return _thermalStatus;
      }

      inline void setPowerModeStatus(const uint8 status)
      {
         _powerModeStatus = status;
      }

      inline uint8 getPowerModeStatus() const
      {
         return _powerModeStatus;
      }

      inline void setEntityId(const uint64 entityId)
      {
         _entityId = entityId;
      }

      inline uint64 getEntityId() const
      {
         return _entityId;
      }

      inline void setSystemIP(const ::std::string& ip)
      {
         _systemIP = ip;
      }

      inline ::std::string getSystemIP() const
      {
         return _systemIP;
      }

      void setSourceIdsInfo(const SourceIdsInfo& info)
      {
         _sourceIdsInfo = info;
      }

      const SourceIdsInfo& getSourceIdsInfo() const
      {
         return _sourceIdsInfo;
      }

      SourceIdsInfo& getMutableSourceIdsInfo()
      {
         return _sourceIdsInfo;
      }

      void setSystemRegionsInfo(const SystemRegionsInfo& info)
      {
         _systemRegionsInfo = info;
      }

      const SystemRegionsInfo& getSystemRegionsInfo() const
      {
         return _systemRegionsInfo;
      }

      SystemRegionsInfo& getMutableSystemRegionsInfo()
      {
         return _systemRegionsInfo;
      }

      uint8 getStatus(const uint8 statusCategory) const
      {
         uint8 status = SYSTEM_STATUS_NONE;
         switch (statusCategory)
         {
            case STATUS_CATEGORY_SYSTEM:
            {
               status = _systemStatus;
               break;
            }
            case STATUS_CATEGORY_AVDECC:
            {
               status = _avdeccStatus;
               break;
            }
            case STATUS_CATEGORY_NANOMSG:
            {
               status = _nanoMsgStatus;
               break;
            }
            case STATUS_CATEGORY_VOLTAGE:
            {
               status = _voltageStatus;
               break;
            }
            case STATUS_CATEGORY_THERMAL:
            {
               status = _thermalStatus;
               break;
            }
            case STATUS_CATEGORY_POWERMODE:
            {
               status = _powerModeStatus;
               break;
            }
            default:
               break;
         }
         return status;
      }

      uint8 getLastStatus(const uint8 statusCategory) const
      {
         uint8 status = SYSTEM_STATUS_NONE;
         switch (statusCategory)
         {
            case STATUS_CATEGORY_SYSTEM:
            {
               status = _lastSystemStatus;
               break;
            }
            default:
               break;
         }
         return status;
      }

      bool resetAllStatus()
      {
         bool isModified = (setStatus(STATUS_CATEGORY_SYSTEM, SYSTEM_STATUS_NONE)) ? true : false;
         isModified      = ((setStatus(STATUS_CATEGORY_AVDECC, SYSTEM_STATUS_NONE)) || (isModified)) ? true : false;
         isModified      = ((setStatus(STATUS_CATEGORY_NANOMSG, SYSTEM_STATUS_NONE)) || (isModified)) ? true : false;
         isModified      = ((setStatus(STATUS_CATEGORY_VOLTAGE, SYSTEM_STATUS_NONE)) || (isModified)) ? true : false;
         isModified      = ((setStatus(STATUS_CATEGORY_THERMAL, SYSTEM_STATUS_NONE)) || (isModified)) ? true : false;
         isModified      = ((setStatus(STATUS_CATEGORY_POWERMODE, SYSTEM_STATUS_NONE)) || (isModified)) ? true : false;
         return isModified;
      }

      bool isCabinSystem() const
      {
         bool isValid = true;
         switch (_category)
         {
            case SYSTEM_CATEGORY_CMC:
            case SYSTEM_CATEGORY_NONE:
            {
               isValid = false;
               break;
            }
            default:
               break;
         }
         return isValid;
      }

      bool isCockpitSystem() const
      {
         bool isValid = false;
         switch (_category)
         {
            case SYSTEM_CATEGORY_CMC:
            {
               isValid = true;
               break;
            }
            default:
               break;
         }
         return isValid;
      }

      bool isSystemIdentified() const
      {
         bool isValid = (_avdeccStatus != SYSTEM_STATUS_NONE) ? true : false;
         isValid      = ((isValid) && (_nanoMsgStatus != SYSTEM_STATUS_NONE)) ? true : false;
         isValid      = ((isValid) && (_powerModeStatus != SYSTEM_STATUS_NONE)) ? true : false;
         return isValid;
      }

      bool isSinkIdAvailable(const int16 sinkId) const
      {
         bool isValid = false;
         if (NULL != _systemRegionsInfo.getSystemRegionInfoForSinkId(sinkId))
         {
            isValid = true;
         }
         return isValid;
      }

      bool isRegionSupported(const uint32 regionId) const
      {
         bool isValid = false;
         if (NULL != _systemRegionsInfo.getSystemRegionInfo(regionId))
         {
            isValid = true;
         }
         return isValid;
      }

      bool isSourceIdAvailable(const int16 sourceId) const
      {
         bool isValid = true;
         if (::std::find(_sourceIdsInfo.begin(), _sourceIdsInfo.end(), sourceId) == _sourceIdsInfo.end())
         {
            isValid = (NULL != _systemRegionsInfo.getSystemRegionInfoForSourceId(sourceId)) ? true : false;
         }
         return isValid;
      }

      bool setStatus(const uint8 statusCategory, const uint8 status)
      {
         bool isUpdated = false;
         switch (statusCategory)
         {
            case STATUS_CATEGORY_SYSTEM:
            {
               if (_systemStatus != status)
               {
                  isUpdated         = true;
                  _lastSystemStatus = (_systemStatus != SYSTEM_GROUP_NONE) ? _systemStatus : _lastSystemStatus;
                  _systemStatus     = status;
               }
               break;
            }
            case STATUS_CATEGORY_AVDECC:
            {
               if (_avdeccStatus != status)
               {
                  isUpdated     = true;
                  _avdeccStatus = status;
               }
               break;
            }
            case STATUS_CATEGORY_NANOMSG:
            {
               if (_nanoMsgStatus != status)
               {
                  isUpdated     = true;
                  _nanoMsgStatus = status;
               }
               break;
            }
            case STATUS_CATEGORY_VOLTAGE:
            {
               if (_voltageStatus != status)
               {
                  isUpdated      = true;
                  _voltageStatus = status;
               }
               break;
            }
            case STATUS_CATEGORY_THERMAL:
            {
               if (_thermalStatus != status)
               {
                  isUpdated      = true;
                  _thermalStatus = status;
               }
               break;
            }
            case STATUS_CATEGORY_POWERMODE:
            {
               if (_powerModeStatus != status)
               {
                  isUpdated      = true;
                  _powerModeStatus = status;
               }
               break;
            }
            default:
               break;
         }
         return isUpdated;
      }

      bool fetchSourceIdsInfo(SourceIdsInfo& info, const uint32 regionId = REGION_INVALID) const
      {
         bool isFetched = false;
         (void)_systemRegionsInfo.fetchSourceIdsInfo(info, regionId);
         for (SourceIdsInfo::const_iterator tItr = _sourceIdsInfo.begin(); (tItr != _sourceIdsInfo.end()); ++tItr)
         {
            if (::std::find(info.begin(), info.end(), (*tItr)) == info.end())
            {
               isFetched = true;
               info.push_back((*tItr));
            }
         }
         return isFetched;
      }

      void addToSourceIdsInfo(const int16 sourceId, const uint32 regionId = REGION_INVALID)
      {
         if (regionId == REGION_INVALID)
         {
            if (::std::find(_sourceIdsInfo.begin(), _sourceIdsInfo.end(), sourceId) == _sourceIdsInfo.end())
            {
               _sourceIdsInfo.push_back(sourceId);
            }
         }
         else
         {
            _systemRegionsInfo.addToSourceIdsInfo(sourceId, regionId);
         }
      }

      void removeFromSourceIdsInfo(const int16 sourceId, const uint32 regionId = REGION_INVALID)
      {
         _systemRegionsInfo.removeFromSourceIdsInfo(sourceId, regionId);
         SourceIdsInfo::iterator itr = ::std::find(_sourceIdsInfo.begin(), _sourceIdsInfo.end(), sourceId);
         if (itr != _sourceIdsInfo.end())
         {
            (void)_sourceIdsInfo.erase(itr);
         }
      }

   private:
      /**
       *  Member Variables Declaration
       */
      uint8 _category;
      uint8 _systemId;
      uint8 _position;
      uint8 _avdeccStatus;
      uint8 _systemStatus;
      uint8 _lastSystemStatus;
      uint8 _nanoMsgStatus;
      uint8 _voltageStatus;
      uint8 _thermalStatus;
      uint8 _powerModeStatus;
      uint64 _entityId;
      ::std::string _systemIP;
      SourceIdsInfo _sourceIdsInfo;
      SystemRegionsInfo _systemRegionsInfo;
};


/*
 * INFO:
 *   -> Tree model of SystemsInfo
 *        SystemsInfo
 *        |-> SystemInfo => (0 .. n)
 *            |-> system IP
 *            |-> system id
 *            |-> entity id
 *            |-> system category
 *            |-> system position
 *            |-> status (AVDECC, SYSTEM, Power Mode ...)
 *            |-> sources => (0 .. n)
 *            |-> Regions
 *                |-> region id
 *                |-> system group
 *                |-> sinks => (0 .. n)
 *                |-> sources => (0 .. n) => For future use
 */
class SystemsInfo
{
   public:
      /**
       *  Member Functions Declaration
       */
      SystemsInfo()
      {
         clear();
      }

      virtual ~SystemsInfo()
      {
         clear();
      }

      SystemsInfo(const SystemsInfo& obj)
      {
         clear();
         *this = obj;
      }

      SystemsInfo& operator = (const SystemsInfo& obj)
      {
         clear();
         if (this != (&obj))
         {
            _systemsInfo = obj._systemsInfo;
         }
         return (*this);
      }

      void clear()
      {
         _systemsInfo.clear();
      }

      void setSystemsInfo(const SystemsInfoType& info)
      {
         _systemsInfo = info;
      }

      const SystemsInfoType& getSystemsInfo() const
      {
         return _systemsInfo;
      }

      SystemsInfoType& getMutableSystemsInfo()
      {
         return _systemsInfo;
      }

      SupportedRegionsInfo getSupportedRegionsInfo() const
      {
         SupportedRegionsInfo regionsInfo;
         ::std::vector< uint32 > insertedRegionsInfo;
         for (SystemsInfoType::const_iterator itr = _systemsInfo.begin(); (itr != _systemsInfo.end()); ++itr)
         {
            uint8 systemCategory = (*itr).getSystemCategory();
            if ((systemCategory == SYSTEM_CATEGORY_CMC) || (systemCategory == SYSTEM_CATEGORY_CMA))
            {
               const SystemRegionsInfoType& tInfo = (*itr).getSystemRegionsInfo().getSystemRegionsInfo();
               for (SystemRegionsInfoType::const_iterator sItr = tInfo.begin(); (sItr != tInfo.end()); ++sItr)
               {
                  if (::std::find(insertedRegionsInfo.begin(), insertedRegionsInfo.end(), (*sItr).getRegionId()) == insertedRegionsInfo.end())
                  {
                     insertedRegionsInfo.push_back((*sItr).getRegionId());
                     regionsInfo.push_back(::std::make_pair((*sItr).getRegionId(), (*sItr).getSinkIdsInfo()));
                  }
               }
            }
         }
         return regionsInfo;
      }

      const SystemInfo* getSystemInfoForSystemId(const uint8 systemId) const
      {
         const SystemInfo* imp = NULL;
         for (SystemsInfoType::const_iterator itr = _systemsInfo.begin(); (itr != _systemsInfo.end()); ++itr)
         {
            if ((*itr).getSystemId() == systemId)
            {
               imp = (&(*itr));
            }
         }
         return imp;
      }

      void evalAndUpdateSystemPosition(const uint8 category)
      {
         std::vector < uint8 > systemId;
         SystemsIdInfo tInfo;
         if (fetchSystemsIdInfo(tInfo, category, REGION_INVALID))
         {
            for (SystemsInfoType::iterator itr = _systemsInfo.begin(); (itr != _systemsInfo.end()); ++itr)
            {
               if (((*itr).getSystemCategory() == category) && ((*itr).getSystemPosition() == 255))
               {
                  if (tInfo.size() == 1)
                  {
                     (*itr).setSystemPosition(1);
                     break;
                  }
                  else if (tInfo.size() > 1)
                  {
                     systemId.push_back((*itr).getSystemId());
                  }
                  else
                  {}
               }
            }
         }
         if (!systemId.empty())
         {
            for (std::vector<uint8>::iterator itr = systemId.begin(); (itr != systemId.end()); ++itr)
            {
               removeFromSystemsInfo((*itr));
            }
         }
      }

      SystemInfo* getMutableSystemInfoForSystemId(const uint8 systemId)
      {
         SystemInfo* imp = const_cast< SystemInfo* >(getSystemInfoForSystemId(systemId));
         return imp;
      }

      SystemsInfoType getCabinsSystemsInfo() const
      {
         SystemsInfoType tSystemsInfo;
         for (SystemsInfoType::const_iterator itr = _systemsInfo.begin(); (itr != _systemsInfo.end()); ++itr)
         {
            if ((*itr).isCabinSystem())
            {
               tSystemsInfo.push_back((*itr));
            }
         }
         return tSystemsInfo;
      }

      /*
       * NOTE:
       *    -> The system id "0xFF" denotes the status of first system in
       *       particular category of the region
       */
      ::std::string getSystemIPAddressForSystemId(const SystemEnquiryInfo& info, const uint8 systemId = 0xFF) const
      {
         bool terminate    = false;
         ::std::string tIP = "";
         for (SystemsInfoType::const_iterator itr = _systemsInfo.begin(); ((itr != _systemsInfo.end()) && (!terminate)); ++itr)
         {
            if ((systemId != 0xFF) || (((*itr).getSystemCategory() == info.get<0x01>()) && ((*itr).isRegionSupported(info.get<0x00>()))))
            {
               if ((systemId == 0xFF) || ((*itr).getSystemId() == systemId))
               {
                  tIP       = (*itr).getSystemIP();
                  terminate = true;
                  break;
               }
            }
         }
         return tIP;
      }

      uint8 getSystemCategoryForSinkId(const int16 sinkId) const
      {
         uint8 systemCategory = SYSTEM_CATEGORY_NONE;
         for (SystemsInfoType::const_iterator itr = _systemsInfo.begin(); (itr != _systemsInfo.end()); ++itr)
         {
            if ((*itr).isSinkIdAvailable(sinkId))
            {
               systemCategory = (*itr).getSystemCategory();
               break;
            }
         }
         return systemCategory;
      }

      uint8 getSystemCategoryForSourceId(const int16 sourceId) const
      {
         uint8 systemCategory = SYSTEM_CATEGORY_NONE;
         for (SystemsInfoType::const_iterator itr = _systemsInfo.begin(); (itr != _systemsInfo.end()); ++itr)
         {
            if ((*itr).isSourceIdAvailable(sourceId))
            {
               systemCategory = (*itr).getSystemCategory();
               break;
            }
         }
         return systemCategory;
      }

      /*
       * NOTE:
       *    -> The system id "0xFF" denotes the status of first system in
       *       particular category of the region
       */
      uint8 getStatusForSystemId(const SystemEnquiryInfo& info, const uint8 systemId = 0xFF) const
      {
         uint8 status   = SYSTEM_STATUS_NONE;
         bool terminate = false;
         for (SystemsInfoType::const_iterator itr = _systemsInfo.begin(); ((itr != _systemsInfo.end()) && (!terminate)); ++itr)
         {
            if ((systemId != 0xFF) || (((*itr).getSystemCategory() == info.get<0x01>()) && ((*itr).isRegionSupported(info.get<0x00>()))))
            {
               if ((systemId == 0xFF) || ((*itr).getSystemId() == systemId))
               {
                  status    = (*itr).getStatus(info.get<0x02>());
                  terminate = true;
                  break;
               }
            }
         }
         return status;
      }

      /*
       * NOTE:
       *    -> The system id "0xFF" denotes the status of first system in
       *       particular category of the region
       */
      uint8 getLastStatusForSystemId(const SystemEnquiryInfo& info, const uint8 systemId = 0xFF) const
      {
         uint8 status   = SYSTEM_STATUS_NONE;
         bool terminate = false;
         for (SystemsInfoType::const_iterator itr = _systemsInfo.begin(); ((itr != _systemsInfo.end()) && (!terminate)); ++itr)
         {
            if ((systemId != 0xFF) || (((*itr).getSystemCategory() == info.get<0x01>()) && ((*itr).isRegionSupported(info.get<0x00>()))))
            {
               if ((systemId == 0xFF) || ((*itr).getSystemId() == systemId))
               {
                  status    = (*itr).getLastStatus(info.get<0x02>());
                  terminate = true;
                  break;
               }
            }
         }
         return status;
      }

      bool isAllCabinSystemsIdentified() const
      {
         bool isValid = true;
         for (SystemsInfoType::const_iterator itr = _systemsInfo.begin(); (itr != _systemsInfo.end()); ++itr)
         {
            if (((*itr).isCabinSystem()) && (!(*itr).isSystemIdentified()))
            {
               isValid = false;
               break;
            }
         }
         return isValid;
      }

      bool isSystemIdentified(const uint8 systemId) const
      {
         bool isValid = false;
         const SystemInfo* imp = getSystemInfoForSystemId(systemId);
         if ((NULL != imp) && (imp->isSystemIdentified()))
         {
            isValid = true;
         }
         return isValid;
      }

      bool fetchRegionIdForSinkId(const int16 sinkId, uint32& regionId) const
      {
         bool isFetched = false;
         for (SystemsInfoType::const_iterator itr = _systemsInfo.begin(); (itr != _systemsInfo.end()); ++itr)
         {
            if ((*itr).getSystemRegionsInfo().fetchRegionIdForSinkId(sinkId, regionId))
            {
               isFetched = true;
               break;
            }
         }
         return isFetched;
      }

      bool fetchSystemIdForEntityId(const uint64 entityId, uint8& systemId) const
      {
         bool isFetched = false;
         for (SystemsInfoType::const_iterator itr = _systemsInfo.begin(); (itr != _systemsInfo.end()); ++itr)
         {
            if ((*itr).getEntityId() == entityId)
            {
               systemId  = (*itr).getSystemId();
               isFetched = true;
               break;
            }
         }
         return isFetched;
      }

      bool fetchSinkIdsInfoForSystemId(SinkIdsInfo& info, const uint8 systemId) const
      {
         bool isFetched = false;
         for (SystemsInfoType::const_iterator itr = _systemsInfo.begin(); (itr != _systemsInfo.end()); ++itr)
         {
            if ((*itr).getSystemId() == systemId)
            {
               isFetched = (((*itr).getSystemRegionsInfo().fetchSinkIdsInfo(info)) || (isFetched));
               break;
            }
         }
         return isFetched;
      }

      bool isAllStatusSame(const SystemEnquiryInfo& info, const uint8 status, const int16 sinkId) const
      {
         bool isValid = false;
         for (SystemsInfoType::const_iterator itr = _systemsInfo.begin(); (itr != _systemsInfo.end()); ++itr)
         {
            if ((*itr).getSystemCategory() == info.get<0x01>())
            {
               if ((info.get<0>() == REGION_INVALID) || ((*itr).isRegionSupported(info.get<0>())))
               {
                  if ((sinkId == -1) || ((*itr).isSinkIdAvailable(sinkId)))
                  {
                     isValid = true;
                     if ((*itr).getStatus(info.get<0x02>()) != status)
                     {
                        isValid = false;
                        break;
                     }
                  }
               }
            }
         }
         return isValid;
      }

      bool isAnyStatusSame(const SystemEnquiryInfo& info, const uint8 status, const int16 sinkId) const
      {
         bool isValid = false;
         for (SystemsInfoType::const_iterator itr = _systemsInfo.begin(); (itr != _systemsInfo.end()); ++itr)
         {
            if ((*itr).getSystemCategory() == info.get<0x01>())
            {
               if ((info.get<0>() == REGION_INVALID) || ((*itr).isRegionSupported(info.get<0>())))
               {
                  if ((sinkId == -1) || ((*itr).isSinkIdAvailable(sinkId)))
                  {
                     if ((*itr).getStatus(info.get<0x02>()) == status)
                     {
                        isValid = true;
                        break;
                     }
                  }
               }
            }
         }
         return isValid;
      }

      bool fetchSinkIdsInfo(SinkIdsInfo& info, const uint8 systemCategory, const uint32 regionId) const
      {
         bool isFetched = false;
         for (SystemsInfoType::const_iterator itr = _systemsInfo.begin(); (itr != _systemsInfo.end()); ++itr)
         {
            if ((systemCategory == SYSTEM_CATEGORY_NONE) || ((*itr).getSystemCategory() == systemCategory))
            {
               isFetched = (((*itr).getSystemRegionsInfo().fetchSinkIdsInfo(info, regionId)) || (isFetched));
            }
         }
         return isFetched;
      }

      bool isSinkIdSupported(const int16 sinkId, const uint8 systemCategory, const uint32 regionId) const
      {
         SystemsIdInfo tInfo;
         bool isValid = false;
         if (fetchSystemsIdInfoForSinkId(tInfo, sinkId, systemCategory, regionId))
         {
            isValid = true;
         }
         return isValid;
      }

      bool fetchSourceIdsInfo(SourceIdsInfo& info, const uint8 systemCategory, const uint32 regionId) const
      {
         bool isFetched = false;
         for (SystemsInfoType::const_iterator itr = _systemsInfo.begin(); (itr != _systemsInfo.end()); ++itr)
         {
            if ((systemCategory == SYSTEM_CATEGORY_NONE) || ((*itr).getSystemCategory() == systemCategory))
            {
               isFetched = (((*itr).fetchSourceIdsInfo(info, regionId)) || (isFetched));
            }
         }
         return isFetched;
      }

      bool fetchSystemsIdInfo(SystemsIdInfo& info, const uint8 systemCategory, const uint32 regionId) const
      {
         bool isFetched = false;
         for (SystemsInfoType::const_iterator itr = _systemsInfo.begin(); (itr != _systemsInfo.end()); ++itr)
         {
            if ((systemCategory == SYSTEM_CATEGORY_NONE) || ((*itr).getSystemCategory() == systemCategory))
            {
               if ((regionId == REGION_INVALID) || ((*itr).isRegionSupported(regionId)))
               {
                  isFetched = true;
                  info.push_back((*itr).getSystemId());
               }
            }
         }
         return isFetched;
      }

      bool isSourceIdSupported(const int16 sourceId, const uint8 systemCategory, const uint32 regionId) const
      {
         SystemsIdInfo tInfo;
         bool isValid = false;
         if (fetchSystemsIdInfoForSourceId(tInfo, sourceId, systemCategory, regionId))
         {
            isValid = true;
         }
         return isValid;
      }

      bool fetchSystemsGroupInfo(SystemsGroupInfo& info, const uint8 systemCategory, const uint32 regionId) const
      {
         bool isFetched = false;
         for (SystemsInfoType::const_iterator itr = _systemsInfo.begin(); (itr != _systemsInfo.end()); ++itr)
         {
            if ((systemCategory == SYSTEM_CATEGORY_NONE) || ((*itr).getSystemCategory() == systemCategory))
            {
               isFetched = (((*itr).getSystemRegionsInfo().fetchSystemsGroupInfo(info, regionId)) || (isFetched));
            }
         }
         return isFetched;
      }

      bool fetchSinkIdsInfoForGroup(SinkIdsInfo& info, const uint8 systemGroup, const uint8 systemCategory, const uint32 regionId) const
      {
         bool isFetched = false;
         for (SystemsInfoType::const_iterator itr = _systemsInfo.begin(); (itr != _systemsInfo.end()); ++itr)
         {
            if ((systemCategory == SYSTEM_CATEGORY_NONE) || ((*itr).getSystemCategory() == systemCategory))
            {
               isFetched = (((*itr).getSystemRegionsInfo().fetchSinkIdsInfoForGroup(info, systemGroup, regionId)) || (isFetched));
            }
         }
         return isFetched;
      }

      bool fetchSystemsIdInfoForSinkId(SystemsIdInfo& info, const int16 sinkId, const uint8 systemCategory, const uint32 regionId) const
      {
         bool isFetched = false;
         for (SystemsInfoType::const_iterator itr = _systemsInfo.begin(); (itr != _systemsInfo.end()); ++itr)
         {
            if ((systemCategory == SYSTEM_CATEGORY_NONE) || ((*itr).getSystemCategory() == systemCategory))
            {
               if ((regionId == REGION_INVALID) || ((*itr).isRegionSupported(regionId)))
               {
                  if ((*itr).isSinkIdAvailable(sinkId))
                  {
                     isFetched = true;
                     info.push_back((*itr).getSystemId());
                  }
               }
            }
         }
         return isFetched;
      }

      bool fetchSystemsIdInfoForGroup(SystemsIdInfo& info, const uint8 systemGroup, const uint8 systemCategory, const uint32 regionId) const
      {
         bool isFetched = false;
         for (SystemsInfoType::const_iterator itr = _systemsInfo.begin(); (itr != _systemsInfo.end()); ++itr)
         {
            if ((systemCategory == SYSTEM_CATEGORY_NONE) || ((*itr).getSystemCategory() == systemCategory))
            {
               if ((*itr).getSystemRegionsInfo().isSystemGroupSupported(systemGroup, regionId))
               {
                  isFetched = true;
                  info.push_back((*itr).getSystemId());
               }
            }
         }
         return isFetched;
      }

      bool fetchSystemsIdInfoForSourceId(SystemsIdInfo& info, const int16 sourceId, const uint8 systemCategory, const uint32 regionId) const
      {
         bool isFetched = false;
         for (SystemsInfoType::const_iterator itr = _systemsInfo.begin(); (itr != _systemsInfo.end()); ++itr)
         {
            if ((systemCategory == SYSTEM_CATEGORY_NONE) || ((*itr).getSystemCategory() == systemCategory))
            {
               if ((regionId == REGION_INVALID) || ((*itr).isRegionSupported(regionId)))
               {
                  if ((*itr).isSourceIdAvailable(sourceId))
                  {
                     isFetched = true;
                     info.push_back((*itr).getSystemId());
                  }
               }
            }
         }
         return isFetched;
      }

      void addToSystemsInfo(const SystemInfo& info)
      {
         if (NULL == getSystemInfoForSystemId(info.getSystemId()))
         {
            _systemsInfo.push_back(info);
         }
      }

      void removeFromSystemsInfo(const uint8 systemId)
      {
         for (SystemsInfoType::iterator itr = _systemsInfo.begin(); (itr != _systemsInfo.end()); ++itr)
         {
            if ((*itr).getSystemId() == systemId)
            {
               _systemsInfo.erase(itr);
               break;
            }
         }
      }

   private:
      /**
       *  Member Variables Declaration
       */
      SystemsInfoType _systemsInfo;
};


class SystemStatusInfo
{
   public:
      /**
       *  Member Functions Declaration
       */
      SystemStatusInfo()
      {
         clear();
      }

      virtual ~SystemStatusInfo()
      {
         clear();
      }

      SystemStatusInfo(const SystemStatusInfo& obj)
      {
         clear();
         *this = obj;
      }

      SystemStatusInfo& operator = (const SystemStatusInfo& obj)
      {
         clear();
         if (this != (&obj))
         {
            _group         = obj._group;
            _status        = obj._status;
            _systemId      = obj._systemId;
            _position      = obj._position;
            _regionsIdInfo = obj._regionsIdInfo;
         }
         return (*this);
      }

      void clear()
      {
         _group    = SYSTEM_GROUP_NONE;
         _status   = SYSTEM_STATUS_NONE;
         _systemId = 0;
         _position = 0;
         _regionsIdInfo.clear();
      }

      inline void setSystemGroup(const uint8 group)
      {
         _group = group;
      }

      inline uint8 getSystemGroup() const
      {
         return _group;
      }

      inline void setStatus(const uint8 status)
      {
         _status = status;
      }

      inline uint8 getStatus() const
      {
         return _status;
      }

      inline void setSystemId(const uint8 systemId)
      {
         _systemId = systemId;
      }

      inline uint8 getSystemId() const
      {
         return _systemId;
      }

      inline void setSystemPosition(const uint8 position)
      {
         _position = position;
      }

      inline uint8 getSystemPosition() const
      {
         return _position;
      }

      inline void setRegionsIdInfo(const RegionsIdInfo& info)
      {
         _regionsIdInfo = info;
      }

      inline const RegionsIdInfo& getRegionsIdInfo() const
      {
         return _regionsIdInfo;
      }

      inline RegionsIdInfo& getMutableRegionsIdInfo()
      {
         return _regionsIdInfo;
      }

   private:
      /**
       *  Member Variables Declaration
       */
      uint8 _group;
      uint8 _status;
      uint8 _systemId;
      uint8 _position;
      RegionsIdInfo _regionsIdInfo;
};


} //namespace Core
} //namespace App


#endif /* REGION_HANDLING_TYPES_H */
