/**
 * @file         : RegionHandlingInterface.h
 * @author       : INF4CV - AppHmi_Master Team
 * @addtogroup   : AppHmi_Master
 * @brief        : RegionHandlingInterface is to handle the RegionHandling
 *                 feature interfaces implementation
 * @copyright    : (c) 2019-2020 Robert Bosch Car Multimedia GmbH
 *                 The reproduction, distribution and utilization of this file as
 *                 well as the communication of its contents to others without express
 *                 authorization is prohibited. Offenders will be held liable for the
 *                 payment of damages. All rights reserved in the event of the grant
 *                 of a patent, utility model or design.
 */


#ifndef REGION_HANDLING_INTERFACE_H
#define REGION_HANDLING_INTERFACE_H


#include "ProjectPluginMsgs.h"
#include "RegionHandlingConstants.h"


/**
 *  Classes Forward Declaration
 */

class IHmiInfoServiceServerComponent;

namespace App {
namespace Core {


class SystemStatusInfo;
class IHmiStateHandler;
class IRegionListenerCB;
class ICMAStatusListenerCB;
class ICMPStatusListenerCB;
class ICMGStatusListenerCB;
class ICMRStatusListenerCB;
class IAVDECCCMAStatusListenerCB;
class IAVDECCCMPStatusListenerCB;
class IAVDECCCMGStatusListenerCB;
class IAVDECCCMRStatusListenerCB;
class INANOMsgCMAStatusListenerCB;
class INANOMsgCMPStatusListenerCB;
class INANOMsgCMGStatusListenerCB;
class INANOMsgCMRStatusListenerCB;
class IVoltageCMCStatusListenerCB;
class IVoltageCMAStatusListenerCB;
class IVoltageCMPStatusListenerCB;
class IVoltageCMGStatusListenerCB;
class IVoltageCMRStatusListenerCB;
class IThermalCMCStatusListenerCB;
class IThermalCMAStatusListenerCB;
class IThermalCMPStatusListenerCB;
class IThermalCMGStatusListenerCB;
class IThermalCMRStatusListenerCB;
class IPowerModeCMAStatusListenerCB;
class IPowerModeCMPStatusListenerCB;
class IPowerModeCMGStatusListenerCB;
class IPowerModeCMRStatusListenerCB;
class IApplicationSwitchServerComponent;
} //namespace Core
} //namespace App


/**
 *  Classes Declaration
 */

namespace App {
namespace Core {


class IRegionHandling
{
   public:
      /**
       *  Member Functions Declaration
       */
      IRegionHandling() {}
      virtual ~IRegionHandling() {}
      virtual SupportedRegionsInfo getSupportedRegionsInfo() const = 0;
      /*
       * NOTE:
       *    -> The system id "0xFF" denotes the status of first system in
       *       particular category of the region
       */
      virtual ::std::string getSystemIPAddress(const SystemEnquiryInfo& info, const uint8 systemId = 0xFF) const = 0;
      virtual int16 getAudioSinkId() const = 0;
      virtual int16 getAudioSinkIdForRegion(const uint32 regionId) const = 0;
      virtual uint8 getSystemGroupForSinkId(const int16 sinkId) const = 0;
      virtual uint8 getSystemCategoryForSinkId(const int16 sinkId) const = 0;
      virtual uint8 getSystemCategoryForSourceId(const int16 sourceId) const = 0;
      virtual uint8 getSystemGroupCategoryForSinkId(const int16 sinkId) const = 0;
      virtual bool isCMGConfiguredForGroupCategory(const uint32 regionId, const uint8 groupCategory) = 0;
      /*
       * INFO:
       *    -> The value "0xFF" of variable "systemId" denotes the status of first
       *       system in particular category of the region
       */
      virtual uint8 getStatus(const SystemEnquiryInfo& info, const uint8 systemId = 0xFF) const = 0;
      /*
       * INFO:
       *    -> The value "0xFF" of variable "systemId" denotes the status of first
       *       system in particular category of the region
       */
      virtual uint8 getLastStatus(const SystemEnquiryInfo& info, const uint8 systemId = 0xFF) const = 0;
      virtual uint32 getRegionId() const = 0;
      virtual uint32 getLastActiveCabinRegionId() const = 0;
      virtual uint32 getRegionIdForAudioSink(const uint32 sinkId) const = 0;
      virtual bool isCabinRegion(const uint32 regionId) const = 0;
      virtual bool isRegionSupported(const uint32 regionId) const = 0;
      virtual bool fetchRegionIdForSinkId(const int16 sinkId, uint32& regionId) const = 0;
      virtual bool isSinkGroupCategorySame(const int16 sinkId, const uint8 category) const = 0;
      /*
       * INFO:
       *    -> REGION_INVALID denotes analyse all the regions of the system
       *    -> Sink id "-1" denotes all the system
       */
      virtual bool isAllStatusSame(const SystemEnquiryInfo& info, const uint8 status, const int16 sinkId = -1) const = 0;
      /*
       * INFO:
       *    -> REGION_INVALID denotes analyse all the regions of the system
       *    -> Sink id "-1" denotes all the system
       */
      virtual bool isAnyStatusSame(const SystemEnquiryInfo& info, const uint8 status, const int16 sinkId = -1) const = 0;
      /*
       * INFO:
       *    -> SYSTEM_CATEGORY_NONE denotes analyse all the systems
       *    -> REGION_INVALID denotes analyse all the regions of the system
       */
      virtual bool fetchSinkIdsInfo(SinkIdsInfo& info, const uint8 systemCategory, const uint32 regionId) const = 0;
      /*
       * INFO:
       *    -> SYSTEM_CATEGORY_NONE denotes analyse all the systems
       *    -> REGION_INVALID denotes analyse all the regions of the system
       */
      virtual bool isSinkIdSupported(const int16 sinkId, const uint8 systemCategory, const uint32 regionId) const = 0;
      /*
       * INFO:
       *    -> SYSTEM_CATEGORY_NONE denotes analyse all the systems
       *    -> REGION_INVALID denotes analyse all the regions of the system
       */
      virtual bool fetchSourceIdsInfo(SourceIdsInfo& info, const uint8 systemCategory, const uint32 regionId) const = 0;
      /*
       * INFO:
       *    -> SYSTEM_CATEGORY_NONE denotes analyse all the systems
       *    -> REGION_INVALID denotes analyse all the regions of the system
       */
      virtual bool fetchSystemsIdInfo(SystemsIdInfo& info, const uint8 systemCategory, const uint32 regionId) const = 0;
      /*
       * INFO:
       *    -> SYSTEM_CATEGORY_NONE denotes analyse all the systems
       *    -> REGION_INVALID denotes analyse all the regions of the system
       */
      virtual bool isSourceIdSupported(const int16 sourceId, const uint8 systemCategory, const uint32 regionId) const = 0;
      /*
        * INFO:
        *    -> SYSTEM_CATEGORY_NONE denotes analyse all the systems
        *    -> REGION_INVALID denotes analyse all the regions of the system
        */
      virtual bool isSystemGroupSupported(const uint8 systemGroup, const uint8 systemCategory, const uint32 regionId) const = 0;
      /*
       * INFO:
       *    -> SYSTEM_CATEGORY_NONE denotes analyse all the systems
       *    -> REGION_INVALID denotes analyse all the regions of the system
       */
      virtual bool fetchSystemsGroupInfo(SystemsGroupInfo& info, const uint8 systemCategory, const uint32 regionId) const = 0;
      /*
       * INFO:
       *    -> SYSTEM_CATEGORY_NONE denotes analyse all the systems
       *    -> REGION_INVALID denotes analyse all the regions of the system
       */
      virtual bool fetchSinkIdsInfoForGroup(SinkIdsInfo& info, const uint8 systemGroup, const uint8 systemCategory, const uint32 regionId) const = 0;
      /*
       * INFO:
       *    -> SYSTEM_CATEGORY_NONE denotes analyse all the systems
       *    -> REGION_INVALID denotes analyse all the regions of the system
       */
      virtual bool fetchSystemsIdInfoForGroup(SystemsIdInfo& info, const uint8 systemGroup, const uint8 systemCategory, const uint32 regionId) const = 0;
      /*
       * INFO:
       *    -> SYSTEM_CATEGORY_NONE denotes analyse all the systems
       *    -> REGION_INVALID denotes analyse all the regions of the system
       */
      virtual bool fetchSystemsIdInfoForSinkId(SystemsIdInfo& info, const int16 sinkId, const uint8 systemCategory, const uint32 regionId) const = 0;
      /*
       * INFO:
       *    -> SYSTEM_CATEGORY_NONE denotes analyse all the systems
       *    -> REGION_INVALID denotes analyse all the regions of the system
       */
      virtual bool fetchSystemsIdInfoForSourceId(SystemsIdInfo& info, const int16 sourceId, const uint8 systemCategory, const uint32 regionId) const = 0;
      /*
       * INFO:
       *    -> SYSTEM_GROUP_CATEGORY_NONE denotes analyse all groupCategory
       *    -> SYSTEM_CATEGORY_NONE denotes analyse all the systems
       *    -> REGION_INVALID denotes analyse all the regions of the system
       */
      virtual bool fetchSinkIdsInfoForGroupCategory(SinkIdsInfo& info, const uint8 groupCategory, const uint8 systemCategory, const uint32 regionId) const = 0;
      virtual bool fetchSinkIdsInfoForSystemId(SinkIdsInfo& info, const uint8 systemId) const = 0;
      /*
       * INFO:
       *    -> SYSTEM_GROUP_CATEGORY_NONE denotes analyse all groupCategory
       *    -> SYSTEM_CATEGORY_NONE denotes analyse all the systems
       *    -> REGION_INVALID denotes analyse all the regions of the system
       */
      virtual bool fetchSystemsIdInfoForGroupCategory(SystemsIdInfo& info, const uint8 groupCategory, const uint8 systemCategory, const uint32 regionId) const = 0;
      virtual void finalize() = 0;
      virtual void initialize() = 0;
      virtual void addRegionListener(IRegionListenerCB& imp) = 0;
      virtual void addCMAStatusListener(ICMAStatusListenerCB& imp) = 0;
      virtual void addCMPStatusListener(ICMPStatusListenerCB& imp) = 0;
      virtual void addCMGStatusListener(ICMGStatusListenerCB& imp) = 0;
      virtual void addCMRStatusListener(ICMRStatusListenerCB& imp) = 0;
      virtual void addAVDECCCMAStatusListener(IAVDECCCMAStatusListenerCB& imp) = 0;
      virtual void addAVDECCCMPStatusListener(IAVDECCCMPStatusListenerCB& imp) = 0;
      virtual void addAVDECCCMGStatusListener(IAVDECCCMGStatusListenerCB& imp) = 0;
      virtual void addAVDECCCMRStatusListener(IAVDECCCMRStatusListenerCB& imp) = 0;
      virtual void addNANOMsgCMAStatusListener(INANOMsgCMAStatusListenerCB& imp) = 0;
      virtual void addNANOMsgCMPStatusListener(INANOMsgCMPStatusListenerCB& imp) = 0;
      virtual void addNANOMsgCMGStatusListener(INANOMsgCMGStatusListenerCB& imp) = 0;
      virtual void addNANOMsgCMRStatusListener(INANOMsgCMRStatusListenerCB& imp) = 0;
      virtual void addVoltageCMCStatusListener(IVoltageCMCStatusListenerCB& imp) = 0;
      virtual void addVoltageCMAStatusListener(IVoltageCMAStatusListenerCB& imp) = 0;
      virtual void addVoltageCMPStatusListener(IVoltageCMPStatusListenerCB& imp) = 0;
      virtual void addVoltageCMGStatusListener(IVoltageCMGStatusListenerCB& imp) = 0;
      virtual void addVoltageCMRStatusListener(IVoltageCMRStatusListenerCB& imp) = 0;
      virtual void addThermalCMCStatusListener(IThermalCMCStatusListenerCB& imp) = 0;
      virtual void addThermalCMAStatusListener(IThermalCMAStatusListenerCB& imp) = 0;
      virtual void addThermalCMPStatusListener(IThermalCMPStatusListenerCB& imp) = 0;
      virtual void addThermalCMGStatusListener(IThermalCMGStatusListenerCB& imp) = 0;
      virtual void addThermalCMRStatusListener(IThermalCMRStatusListenerCB& imp) = 0;
      virtual void addPowerModeCMAStatusListener(IPowerModeCMAStatusListenerCB& imp) = 0;
      virtual void addPowerModeCMPStatusListener(IPowerModeCMPStatusListenerCB& imp) = 0;
      virtual void addPowerModeCMGStatusListener(IPowerModeCMGStatusListenerCB& imp) = 0;
      virtual void addPowerModeCMRStatusListener(IPowerModeCMRStatusListenerCB& imp) = 0;
      virtual void removeRegionListener(IRegionListenerCB& imp) = 0;
      virtual void removeCMAStatusListener(ICMAStatusListenerCB& imp) = 0;
      virtual void removeCMPStatusListener(ICMPStatusListenerCB& imp) = 0;
      virtual void removeCMGStatusListener(ICMGStatusListenerCB& imp) = 0;
      virtual void removeCMRStatusListener(ICMRStatusListenerCB& imp) = 0;
      virtual void removeAVDECCCMAStatusListener(IAVDECCCMAStatusListenerCB& imp) = 0;
      virtual void removeAVDECCCMPStatusListener(IAVDECCCMPStatusListenerCB& imp) = 0;
      virtual void removeAVDECCCMGStatusListener(IAVDECCCMGStatusListenerCB& imp) = 0;
      virtual void removeAVDECCCMRStatusListener(IAVDECCCMRStatusListenerCB& imp) = 0;
      virtual void removeNANOMsgCMAStatusListener(INANOMsgCMAStatusListenerCB& imp) = 0;
      virtual void removeNANOMsgCMPStatusListener(INANOMsgCMPStatusListenerCB& imp) = 0;
      virtual void removeNANOMsgCMGStatusListener(INANOMsgCMGStatusListenerCB& imp) = 0;
      virtual void removeNANOMsgCMRStatusListener(INANOMsgCMRStatusListenerCB& imp) = 0;
      virtual void removeVoltageCMCStatusListener(IVoltageCMCStatusListenerCB& imp) = 0;
      virtual void removeVoltageCMAStatusListener(IVoltageCMAStatusListenerCB& imp) = 0;
      virtual void removeVoltageCMPStatusListener(IVoltageCMPStatusListenerCB& imp) = 0;
      virtual void removeVoltageCMGStatusListener(IVoltageCMGStatusListenerCB& imp) = 0;
      virtual void removeVoltageCMRStatusListener(IVoltageCMRStatusListenerCB& imp) = 0;
      virtual void removeThermalCMCStatusListener(IThermalCMCStatusListenerCB& imp) = 0;
      virtual void removeThermalCMAStatusListener(IThermalCMAStatusListenerCB& imp) = 0;
      virtual void removeThermalCMPStatusListener(IThermalCMPStatusListenerCB& imp) = 0;
      virtual void removeThermalCMGStatusListener(IThermalCMGStatusListenerCB& imp) = 0;
      virtual void removeThermalCMRStatusListener(IThermalCMRStatusListenerCB& imp) = 0;
      virtual void removePowerModeCMAStatusListener(IPowerModeCMAStatusListenerCB& imp) = 0;
      virtual void removePowerModeCMPStatusListener(IPowerModeCMPStatusListenerCB& imp) = 0;
      virtual void removePowerModeCMGStatusListener(IPowerModeCMGStatusListenerCB& imp) = 0;
      virtual void removePowerModeCMRStatusListener(IPowerModeCMRStatusListenerCB& imp) = 0;
      virtual void setRegionId(const uint32 regionId) = 0;
      virtual void setHMIStateHandler(IHmiStateHandler& imp) = 0;
      virtual void setHmiInfoServiceServerComponent(IHmiInfoServiceServerComponent& imp) = 0;
      virtual void setApplicationSwitchServerComponent(IApplicationSwitchServerComponent& imp) = 0;
      virtual void onPluginEntityDetailsUpdate(const ::boost::shared_ptr< EventListDataUtility >& info) = 0;
      virtual void dumpSystemsInfo() = 0;
};


class IRegionListenerCB
{
   public:
      /**
       *  Member Functions Declaration
       */
      IRegionListenerCB() {}
      virtual ~IRegionListenerCB() {}
      virtual void onRegionUpdate(const uint32 regionId) = 0;
};


class ICMAStatusListenerCB
{
   public:
      /**
       *  Member Functions Declaration
       */
      ICMAStatusListenerCB() {}
      virtual ~ICMAStatusListenerCB() {}
      virtual void onCMAStatusUpdate(const SystemStatusInfo& info) = 0;
};


class ICMPStatusListenerCB
{
   public:
      /**
       *  Member Functions Declaration
       */
      ICMPStatusListenerCB() {}
      virtual ~ICMPStatusListenerCB() {}
      virtual void onCMPStatusUpdate(const SystemStatusInfo& info) = 0;
};


class ICMGStatusListenerCB
{
   public:
      /**
       *  Member Functions Declaration
       */
      ICMGStatusListenerCB() {}
      virtual ~ICMGStatusListenerCB() {}
      virtual void onCMGStatusUpdate(const SystemStatusInfo& info) = 0;
};


class ICMRStatusListenerCB
{
   public:
      /**
       *  Member Functions Declaration
       */
      ICMRStatusListenerCB() {}
      virtual ~ICMRStatusListenerCB() {}
      virtual void onCMRStatusUpdate(const SystemStatusInfo& info) = 0;
};


class IAVDECCCMAStatusListenerCB
{
   public:
      /**
       *  Member Functions Declaration
       */
      IAVDECCCMAStatusListenerCB() {}
      virtual ~IAVDECCCMAStatusListenerCB() {}
      virtual void onAVDECCCMAStatusUpdate(const SystemStatusInfo& info) = 0;
};


class IAVDECCCMPStatusListenerCB
{
   public:
      /**
       *  Member Functions Declaration
       */
      IAVDECCCMPStatusListenerCB() {}
      virtual ~IAVDECCCMPStatusListenerCB() {}
      virtual void onAVDECCCMPStatusUpdate(const SystemStatusInfo& info) = 0;
};


class IAVDECCCMGStatusListenerCB
{
   public:
      /**
       *  Member Functions Declaration
       */
      IAVDECCCMGStatusListenerCB() {}
      virtual ~IAVDECCCMGStatusListenerCB() {}
      virtual void onAVDECCCMGStatusUpdate(const SystemStatusInfo& info) = 0;
};


class IAVDECCCMRStatusListenerCB
{
   public:
      /**
       *  Member Functions Declaration
       */
      IAVDECCCMRStatusListenerCB() {}
      virtual ~IAVDECCCMRStatusListenerCB() {}
      virtual void onAVDECCCMRStatusUpdate(const SystemStatusInfo& info) = 0;
};


class INANOMsgCMAStatusListenerCB
{
   public:
      /**
       *  Member Functions Declaration
       */
      INANOMsgCMAStatusListenerCB() {}
      virtual ~INANOMsgCMAStatusListenerCB() {}
      virtual void onNANOMsgCMAStatusUpdate(const SystemStatusInfo& info) = 0;
};


class INANOMsgCMPStatusListenerCB
{
   public:
      /**
       *  Member Functions Declaration
       */
      INANOMsgCMPStatusListenerCB() {}
      virtual ~INANOMsgCMPStatusListenerCB() {}
      virtual void onNANOMsgCMPStatusUpdate(const SystemStatusInfo& info) = 0;
};


class INANOMsgCMGStatusListenerCB
{
   public:
      /**
       *  Member Functions Declaration
       */
      INANOMsgCMGStatusListenerCB() {}
      virtual ~INANOMsgCMGStatusListenerCB() {}
      virtual void onNANOMsgCMGStatusUpdate(const SystemStatusInfo& info) = 0;
};


class INANOMsgCMRStatusListenerCB
{
   public:
      /**
       *  Member Functions Declaration
       */
      INANOMsgCMRStatusListenerCB() {}
      virtual ~INANOMsgCMRStatusListenerCB() {}
      virtual void onNANOMsgCMRStatusUpdate(const SystemStatusInfo& info) = 0;
};


class IVoltageCMCStatusListenerCB
{
   public:
      /**
       *  Member Functions Declaration
       */
      IVoltageCMCStatusListenerCB() {}
      virtual ~IVoltageCMCStatusListenerCB() {}
      virtual void onVoltageCMCStatusUpdate(const SystemStatusInfo& info) = 0;
};


class IVoltageCMAStatusListenerCB
{
   public:
      /**
       *  Member Functions Declaration
       */
      IVoltageCMAStatusListenerCB() {}
      virtual ~IVoltageCMAStatusListenerCB() {}
      virtual void onVoltageCMAStatusUpdate(const SystemStatusInfo& info) = 0;
};


class IVoltageCMPStatusListenerCB
{
   public:
      /**
       *  Member Functions Declaration
       */
      IVoltageCMPStatusListenerCB() {}
      virtual ~IVoltageCMPStatusListenerCB() {}
      virtual void onVoltageCMPStatusUpdate(const SystemStatusInfo& info) = 0;
};


class IVoltageCMGStatusListenerCB
{
   public:
      /**
       *  Member Functions Declaration
       */
      IVoltageCMGStatusListenerCB() {}
      virtual ~IVoltageCMGStatusListenerCB() {}
      virtual void onVoltageCMGStatusUpdate(const SystemStatusInfo& info) = 0;
};


class IVoltageCMRStatusListenerCB
{
   public:
      /**
       *  Member Functions Declaration
       */
      IVoltageCMRStatusListenerCB() {}
      virtual ~IVoltageCMRStatusListenerCB() {}
      virtual void onVoltageCMRStatusUpdate(const SystemStatusInfo& info) = 0;
};


class IThermalCMCStatusListenerCB
{
   public:
      /**
       *  Member Functions Declaration
       */
      IThermalCMCStatusListenerCB() {}
      virtual ~IThermalCMCStatusListenerCB() {}
      virtual void onThermalCMCStatusUpdate(const SystemStatusInfo& info) = 0;
};


class IThermalCMAStatusListenerCB
{
   public:
      /**
       *  Member Functions Declaration
       */
      IThermalCMAStatusListenerCB() {}
      virtual ~IThermalCMAStatusListenerCB() {}
      virtual void onThermalCMAStatusUpdate(const SystemStatusInfo& info) = 0;
};


class IThermalCMPStatusListenerCB
{
   public:
      /**
       *  Member Functions Declaration
       */
      IThermalCMPStatusListenerCB() {}
      virtual ~IThermalCMPStatusListenerCB() {}
      virtual void onThermalCMPStatusUpdate(const SystemStatusInfo& info) = 0;
};


class IThermalCMGStatusListenerCB
{
   public:
      /**
       *  Member Functions Declaration
       */
      IThermalCMGStatusListenerCB() {}
      virtual ~IThermalCMGStatusListenerCB() {}
      virtual void onThermalCMGStatusUpdate(const SystemStatusInfo& info) = 0;
};


class IThermalCMRStatusListenerCB
{
   public:
      /**
       *  Member Functions Declaration
       */
      IThermalCMRStatusListenerCB() {}
      virtual ~IThermalCMRStatusListenerCB() {}
      virtual void onThermalCMRStatusUpdate(const SystemStatusInfo& info) = 0;
};


class IPowerModeCMAStatusListenerCB
{
   public:
      /**
       *  Member Functions Declaration
       */
      IPowerModeCMAStatusListenerCB() {}
      virtual ~IPowerModeCMAStatusListenerCB() {}
      virtual void onPowerModeCMAStatusUpdate(const SystemStatusInfo& info) = 0;
};


class IPowerModeCMPStatusListenerCB
{
   public:
      /**
       *  Member Functions Declaration
       */
      IPowerModeCMPStatusListenerCB() {}
      virtual ~IPowerModeCMPStatusListenerCB() {}
      virtual void onPowerModeCMPStatusUpdate(const SystemStatusInfo& info) = 0;
};


class IPowerModeCMGStatusListenerCB
{
   public:
      /**
       *  Member Functions Declaration
       */
      IPowerModeCMGStatusListenerCB() {}
      virtual ~IPowerModeCMGStatusListenerCB() {}
      virtual void onPowerModeCMGStatusUpdate(const SystemStatusInfo& info) = 0;
};


class IPowerModeCMRStatusListenerCB
{
   public:
      /**
       *  Member Functions Declaration
       */
      IPowerModeCMRStatusListenerCB() {}
      virtual ~IPowerModeCMRStatusListenerCB() {}
      virtual void onPowerModeCMRStatusUpdate(const SystemStatusInfo& info) = 0;
};


} //namespace Core
} //namespace App


#endif /* REGION_HANDLING_INTERFACE_H */
