/**
 * @file         : RegionHandling.cpp
 * @author       : INF4CV - AppHmi_Master Team
 * @addtogroup   : AppHmi_Master
 * @brief        : RegionHandling is to handle the region information
 * @copyright    : (c) 2018-2020 Robert Bosch Car Multimedia GmbH
 *                 The reproduction, distribution and utilization of this file as
 *                 well as the communication of its contents to others without express
 *                 authorization is prohibited. Offenders will be held liable for the
 *                 payment of damages. All rights reserved in the event of the grant
 *                 of a patent, utility model or design.
 */

#include "hall_std_if.h"
#include "RegionHandling.h"
#include "tinyxml.h"
#include "ApplicationSwitchConst.h"
#include <sstream>
#include <stdlib.h>
#include <Core/HmiState/HmiStateHandlerHelper.h>
#include <AppHmi_MasterBase/AudioInterface/Source.h>
#include <AppHmi_MasterBase/AudioInterface/AudioDefines.h>
#ifdef VARIANT_S_FTR_ENABLE_PLUGINS
#include <Common/PluginClientHandler/PluginClientHandler.h>
#endif
#include <Core/HmiInfoService/HmiInfoServiceServerComponentHandlingInterface.h>
#include <Core/ApplicationSwitchServerComponent/ApplicationSwitchServerComponentInterface.h>


using namespace ::hmibase::apphmi_master::audio;
using namespace ::bosch::cm::ai::hmi::hmimasterservice::ApplicationSwitch;


#ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
#define ETG_DEFAULT_TRACE_CLASS TR_CLASS_APPHMI_MASTER_MAIN
#include "trcGenProj/Header/RegionHandling.cpp.trc.h"
#endif


/**
 *  Defines Declaration
 */

#define REGION_HANDLING_LISTENER_DATA_DEFN_BEGIN()

#define ADD_LISTENER_DATA_DEFN(_dataName_, _dataType_)                                                 \
                                                                                                       \
void RegionHandling::clear##_dataName_##ListenersInfo()                                                \
{                                                                                                      \
   if (!_##_dataName_##Listeners.empty())                                                              \
   {                                                                                                   \
      _##_dataName_##Listeners.clear();                                                                \
   }                                                                                                   \
}                                                                                                      \
                                                                                                       \
void RegionHandling::send##_dataName_##Update(const _dataType_& info)                                  \
{                                                                                                      \
   ::std::vector< I##_dataName_##ListenerCB* >::const_iterator itr = _##_dataName_##Listeners.begin(); \
   for (; (itr != _##_dataName_##Listeners.end()); ++itr)                                              \
   {                                                                                                   \
      if (NULL != (*itr))                                                                              \
      {                                                                                                \
         (*itr)->on##_dataName_##Update(info);                                                         \
      }                                                                                                \
   }                                                                                                   \
}                                                                                                      \
                                                                                                       \
void RegionHandling::add##_dataName_##Listener(I##_dataName_##ListenerCB& imp)                         \
{                                                                                                      \
   addListener<I##_dataName_##ListenerCB>(_##_dataName_##Listeners, imp);                              \
}                                                                                                      \
                                                                                                       \
void RegionHandling::remove##_dataName_##Listener(I##_dataName_##ListenerCB& imp)                      \
{                                                                                                      \
   removeListener<I##_dataName_##ListenerCB>(_##_dataName_##Listeners, imp);                           \
}


#define REGION_HANDLING_LISTENER_DATA_DEFN_END()

#ifdef WIN32 /* Windows env */

#define TRIGGER_MUTEX_LOCK(_var_)

#define TRIGGER_MUTEX_UNLOCK(_var_)

#else /* Linux / Other OS env */

#define TRIGGER_MUTEX_LOCK(_var_)    pthread_mutex_lock(&_var_);

#define TRIGGER_MUTEX_UNLOCK(_var_)  pthread_mutex_unlock(&_var_);

#endif


/**
 *  Global Variables Declaration
 */

namespace App {
namespace Core {


IRegionHandling* _regionHandling = NULL;
} //namespace Core
} //namespace App


/**
 *  Static Variables Initialization
 */

namespace App {
namespace Core {


#ifndef WIN32 /* Linux / Other OS env except Windows */
pthread_mutex_t RegionHandling::_safeLock = PTHREAD_MUTEX_INITIALIZER;
#endif
} //namespace Core
} //namespace App


/**
 *  SM "C" Functions Definition
 */

uint32 fnGetRegionId()
{
   uint32 tRegionId = 0;
   if (NULL != ::App::Core::_regionHandling)
   {
      tRegionId = ::App::Core::_regionHandling->getRegionId();
   }
   return tRegionId;
}


uint32 fnGetLastActiveCabinRegionId()
{
   uint32 tRegionId = 0;
   if (NULL != ::App::Core::_regionHandling)
   {
      tRegionId = ::App::Core::_regionHandling->getLastActiveCabinRegionId();
   }
   return tRegionId;
}


uint32 fnGetAudioSinkId()
{
   uint32 tSinkId = 0;
   if (NULL != ::App::Core::_regionHandling)
   {
      tSinkId = ::App::Core::_regionHandling->getAudioSinkId();
   }
   return tSinkId;
}


uint32 fnGetRegionIdForAudioSink(const uint32 sinkId)
{
   uint32 tRegionId = 0;
   if (NULL != ::App::Core::_regionHandling)
   {
      tRegionId = ::App::Core::_regionHandling->getRegionIdForAudioSink(sinkId);
   }
   return tRegionId;
}


uint32 fnGetAudioSinkIdForRegion(const uint32 regionId)
{
   uint32 tSinkId = 0;
   if (NULL != ::App::Core::_regionHandling)
   {
      tSinkId = ::App::Core::_regionHandling->getAudioSinkIdForRegion(regionId);
   }
   return tSinkId;
}


bool fnIsRegionSupported(const uint32 regionId)
{
   bool isValid = false;
   if (NULL != ::App::Core::_regionHandling)
   {
      isValid = ::App::Core::_regionHandling->isRegionSupported(regionId);
   }
   return isValid;
}


/**
 *  Memeber Functions Definition
 */

namespace App {
namespace Core {


RegionHandling::RegionHandling() : _regionId(REGION_COCKPIT)
   , _lastActiveCabinRegionId(REGION_CABIN_A)
   , _hmiStateHandler(NULL)
   , _hmiInfoServiceServerComponent(NULL)
   , _applicationSwitchServerComponent(NULL)
   , _systemIdentificationTimer(NULL)
{
   ETG_TRACE_USR1(("RegionHandling: CTOR"));
   _regionHandling = this;
   initializeSystemsInfo();
   initializeSupportedRegionsInfo();
   initializeListenerUpdateFunctionsInfo();
}


RegionHandling::~RegionHandling()
{
   ETG_TRACE_USR1(("RegionHandling: DTOR"));
   stopSystemIdentificationTimer(true);
   clearRegionListenersInfo();
   clearCMAStatusListenersInfo();
   clearCMPStatusListenersInfo();
   clearCMGStatusListenersInfo();
   clearCMRStatusListenersInfo();
   clearAVDECCCMAStatusListenersInfo();
   clearAVDECCCMPStatusListenersInfo();
   clearAVDECCCMGStatusListenersInfo();
   clearAVDECCCMRStatusListenersInfo();
   clearNANOMsgCMAStatusListenersInfo();
   clearNANOMsgCMPStatusListenersInfo();
   clearNANOMsgCMGStatusListenersInfo();
   clearNANOMsgCMRStatusListenersInfo();
   clearVoltageCMAStatusListenersInfo();
   clearVoltageCMPStatusListenersInfo();
   clearVoltageCMGStatusListenersInfo();
   clearVoltageCMRStatusListenersInfo();
   clearThermalCMAStatusListenersInfo();
   clearThermalCMPStatusListenersInfo();
   clearThermalCMGStatusListenersInfo();
   clearThermalCMRStatusListenersInfo();
   clearPowerModeCMAStatusListenersInfo();
   clearPowerModeCMPStatusListenersInfo();
   clearPowerModeCMGStatusListenersInfo();
   clearPowerModeCMRStatusListenersInfo();
   _regionId                         = REGION_COCKPIT;
   _lastActiveCabinRegionId          = REGION_CABIN_A;
   _regionHandling                   = NULL;
   _hmiStateHandler                  = NULL;
   _hmiInfoServiceServerComponent    = NULL;
   _applicationSwitchServerComponent = NULL;
   _systemsInfo.clear();
   _sinksStaticInfo.clear();
   _supportedRegionsInfo.clear();
   _listenerUpdateFunctionsInfo.clear();
}


void RegionHandling::initializeSystemsInfo()
{
   initializeCockpitSystemInfo();
#ifdef VARIANT_S_FTR_ENABLE_COACH_AUDIO
   initializeCabinSystemsInfo();
#endif
   dumpSystemsInfo();
}


void RegionHandling::initializeCabinSystemsInfo()
{
   TiXmlDocument tDoc;
   try
   {
      if (tDoc.LoadFile("/usr/lib/avdecc/plugins/AVRoutingAdapter.xml"))
      {
         TiXmlElement* tAdapterGroupElement = tDoc.FirstChildElement("avdecc_routing_adapter");
         if (NULL != tAdapterGroupElement)
         {
            initializeCabinSinksStaticInfo();
            initializeCabinCMASystemsInfo(tAdapterGroupElement);
            initializeCabinCMGSystemsInfo(tAdapterGroupElement);
            initializeCabinCMPSystemsInfo(tAdapterGroupElement);
            initializeCabinCMRSystemsInfo(tAdapterGroupElement);
         }
      }
      else
      {
         ETG_TRACE_ERR(("RegionHandling: initializeCabinSystemsInfo: Failed to read file AVRoutingAdapter.xml"));
         ETG_TRACE_ERRMEM(("!!!!!! APPHMI_MASTER : RegionHandling: initializeCabinSystemsInfo: Failed to read file AVRoutingAdapter.xml: INFO:Restart not required !!!!!!"));
      }
   }
   catch (...)
   {
      ETG_TRACE_ERR(("RegionHandling: initializeCabinSystemsInfo: Failed to read file AVRoutingAdapter.xml"));
   }
}


void RegionHandling::initializeCockpitSystemInfo()
{
   SystemInfo tSystemInfo;
   SystemRegionInfo tSystemRegionInfo;
   SystemRegionsInfo tSystemRegionsInfo;
   tSystemRegionInfo.setRegionId(REGION_COCKPIT);
   tSystemRegionInfo.addToSinkIdsInfo(COCKPIT_SINK_ID);
   tSystemRegionInfo.addToSinkIdsInfo(PRIVATE_SPEAKER_SINK_ID);
   tSystemRegionsInfo.addToSystemRegionsInfo(tSystemRegionInfo);
   tSystemInfo.setSystemCategory(SYSTEM_CATEGORY_CMC);
   tSystemInfo.setSystemRegionsInfo(tSystemRegionsInfo);
   tSystemInfo.setSystemStatus(SYSTEM_STATUS_CONNECTED);
   _systemsInfo.addToSystemsInfo(tSystemInfo);
}


void RegionHandling::initializeSupportedRegionsInfo()
{
   if (_supportedRegionsInfo.empty())
   {
      SupportedRegionsInfo tInfo = _systemsInfo.getSupportedRegionsInfo();
      TRIGGER_MUTEX_LOCK(_safeLock)
      _supportedRegionsInfo = tInfo;
      TRIGGER_MUTEX_UNLOCK(_safeLock)
   }
}


void RegionHandling::initializeCabinSinksStaticInfo()
{
   _sinksStaticInfo.clear();
   SinkStaticInfo tSinkStaticInfo;
   tSinkStaticInfo.setSinkName("AMP_A");
   tSinkStaticInfo.setSinkId(CABIN_A_SINK_ID);
   tSinkStaticInfo.setRegionId(REGION_CABIN_A);
   tSinkStaticInfo.setSystemCategory(SYSTEM_CATEGORY_CMA);
   _sinksStaticInfo.push_back(tSinkStaticInfo);
   tSinkStaticInfo.clear();
   tSinkStaticInfo.setSinkName("Glass_A1");
   tSinkStaticInfo.setRegionId(REGION_CABIN_A);
   tSinkStaticInfo.setSystemGroup(SYSTEM_GROUP_CMG_A);
   tSinkStaticInfo.setSinkId(CABIN_A_MEDIAGLASS1_SINK_ID);
   tSinkStaticInfo.setSystemCategory(SYSTEM_CATEGORY_CMG);
   tSinkStaticInfo.setSystemGroupCategory(SYSTEM_GROUP_CATEGORY_CMG_CABIN);
   _sinksStaticInfo.push_back(tSinkStaticInfo);
   tSinkStaticInfo.clear();
   tSinkStaticInfo.setSinkName("Glass_A2");
   tSinkStaticInfo.setRegionId(REGION_CABIN_A);
   tSinkStaticInfo.setSystemGroup(SYSTEM_GROUP_CMG_A2);
   tSinkStaticInfo.setSinkId(CABIN_A_MEDIAGLASS2_SINK_ID);
   tSinkStaticInfo.setSystemCategory(SYSTEM_CATEGORY_CMG);
   tSinkStaticInfo.setSystemGroupCategory(SYSTEM_GROUP_CATEGORY_CMG_CABIN_MIX);
   _sinksStaticInfo.push_back(tSinkStaticInfo);
   /* INFO: Applicable for SOP2
   tSinkStaticInfo.clear();
   tSinkStaticInfo.setSinkName("AMP_B");
   tSinkStaticInfo.setSinkId(CABIN_B_SINK_ID);
   tSinkStaticInfo.setRegionId(REGION_CABIN_B);
   tSinkStaticInfo.setSystemCategory(SYSTEM_CATEGORY_CMA);
   _sinksStaticInfo.push_back(tSinkStaticInfo);
   tSinkStaticInfo.clear();
   tSinkStaticInfo.setSinkName("Glass_B1");
   tSinkStaticInfo.setRegionId(REGION_CABIN_B);
   tSinkStaticInfo.setSystemGroup(SYSTEM_GROUP_CMG_A);
   tSinkStaticInfo.setSinkId(CABIN_B_MEDIAGLASS1_SINK_ID);
   tSinkStaticInfo.setSystemCategory(SYSTEM_CATEGORY_CMG);
   tSinkStaticInfo.setSystemGroupCategory(SYSTEM_GROUP_CATEGORY_CMG_CABIN);
   _sinksStaticInfo.push_back(tSinkStaticInfo);
   tSinkStaticInfo.clear();
   tSinkStaticInfo.setSinkName("Glass_B2");
   tSinkStaticInfo.setRegionId(REGION_CABIN_B);
   tSinkStaticInfo.setSystemGroup(SYSTEM_GROUP_CMG_A2);
   tSinkStaticInfo.setSinkId(CABIN_B_MEDIAGLASS2_SINK_ID);
   tSinkStaticInfo.setSystemCategory(SYSTEM_CATEGORY_CMG);
   tSinkStaticInfo.setSystemGroupCategory(SYSTEM_GROUP_CATEGORY_CMG_CABIN_MIX);
   _sinksStaticInfo.push_back(tSinkStaticInfo); */
}


void RegionHandling::initializeListenerUpdateFunctionsInfo()
{
   typedef ::std::pair< uint8, uint8 > SystemStatusTypeInfo;
   typedef ::std::pair< SystemStatusTypeInfo, void(RegionHandling::*)(const SystemStatusInfo&) > ListenerUpdateFunctionInfo;
   if (_listenerUpdateFunctionsInfo.empty())
   {
      _listenerUpdateFunctionsInfo.insert(ListenerUpdateFunctionInfo(SystemStatusTypeInfo(SYSTEM_CATEGORY_CMA, STATUS_CATEGORY_SYSTEM),    &RegionHandling::sendCMAStatusUpdate));
      _listenerUpdateFunctionsInfo.insert(ListenerUpdateFunctionInfo(SystemStatusTypeInfo(SYSTEM_CATEGORY_CMP, STATUS_CATEGORY_SYSTEM),    &RegionHandling::sendCMPStatusUpdate));
      _listenerUpdateFunctionsInfo.insert(ListenerUpdateFunctionInfo(SystemStatusTypeInfo(SYSTEM_CATEGORY_CMG, STATUS_CATEGORY_SYSTEM),    &RegionHandling::sendCMGStatusUpdate));
      _listenerUpdateFunctionsInfo.insert(ListenerUpdateFunctionInfo(SystemStatusTypeInfo(SYSTEM_CATEGORY_CMR, STATUS_CATEGORY_SYSTEM),    &RegionHandling::sendCMRStatusUpdate));
      _listenerUpdateFunctionsInfo.insert(ListenerUpdateFunctionInfo(SystemStatusTypeInfo(SYSTEM_CATEGORY_CMA, STATUS_CATEGORY_AVDECC),    &RegionHandling::sendAVDECCCMAStatusUpdate));
      _listenerUpdateFunctionsInfo.insert(ListenerUpdateFunctionInfo(SystemStatusTypeInfo(SYSTEM_CATEGORY_CMP, STATUS_CATEGORY_AVDECC),    &RegionHandling::sendAVDECCCMPStatusUpdate));
      _listenerUpdateFunctionsInfo.insert(ListenerUpdateFunctionInfo(SystemStatusTypeInfo(SYSTEM_CATEGORY_CMG, STATUS_CATEGORY_AVDECC),    &RegionHandling::sendAVDECCCMGStatusUpdate));
      _listenerUpdateFunctionsInfo.insert(ListenerUpdateFunctionInfo(SystemStatusTypeInfo(SYSTEM_CATEGORY_CMR, STATUS_CATEGORY_AVDECC),    &RegionHandling::sendAVDECCCMRStatusUpdate));
      _listenerUpdateFunctionsInfo.insert(ListenerUpdateFunctionInfo(SystemStatusTypeInfo(SYSTEM_CATEGORY_CMA, STATUS_CATEGORY_NANOMSG),   &RegionHandling::sendNANOMsgCMAStatusUpdate));
      _listenerUpdateFunctionsInfo.insert(ListenerUpdateFunctionInfo(SystemStatusTypeInfo(SYSTEM_CATEGORY_CMP, STATUS_CATEGORY_NANOMSG),   &RegionHandling::sendNANOMsgCMPStatusUpdate));
      _listenerUpdateFunctionsInfo.insert(ListenerUpdateFunctionInfo(SystemStatusTypeInfo(SYSTEM_CATEGORY_CMG, STATUS_CATEGORY_NANOMSG),   &RegionHandling::sendNANOMsgCMGStatusUpdate));
      _listenerUpdateFunctionsInfo.insert(ListenerUpdateFunctionInfo(SystemStatusTypeInfo(SYSTEM_CATEGORY_CMR, STATUS_CATEGORY_NANOMSG),   &RegionHandling::sendNANOMsgCMRStatusUpdate));
      _listenerUpdateFunctionsInfo.insert(ListenerUpdateFunctionInfo(SystemStatusTypeInfo(SYSTEM_CATEGORY_CMA, STATUS_CATEGORY_VOLTAGE),   &RegionHandling::sendVoltageCMAStatusUpdate));
      _listenerUpdateFunctionsInfo.insert(ListenerUpdateFunctionInfo(SystemStatusTypeInfo(SYSTEM_CATEGORY_CMP, STATUS_CATEGORY_VOLTAGE),   &RegionHandling::sendVoltageCMPStatusUpdate));
      _listenerUpdateFunctionsInfo.insert(ListenerUpdateFunctionInfo(SystemStatusTypeInfo(SYSTEM_CATEGORY_CMG, STATUS_CATEGORY_VOLTAGE),   &RegionHandling::sendVoltageCMGStatusUpdate));
      _listenerUpdateFunctionsInfo.insert(ListenerUpdateFunctionInfo(SystemStatusTypeInfo(SYSTEM_CATEGORY_CMR, STATUS_CATEGORY_VOLTAGE),   &RegionHandling::sendVoltageCMRStatusUpdate));
      _listenerUpdateFunctionsInfo.insert(ListenerUpdateFunctionInfo(SystemStatusTypeInfo(SYSTEM_CATEGORY_CMA, STATUS_CATEGORY_THERMAL),   &RegionHandling::sendThermalCMAStatusUpdate));
      _listenerUpdateFunctionsInfo.insert(ListenerUpdateFunctionInfo(SystemStatusTypeInfo(SYSTEM_CATEGORY_CMP, STATUS_CATEGORY_THERMAL),   &RegionHandling::sendThermalCMPStatusUpdate));
      _listenerUpdateFunctionsInfo.insert(ListenerUpdateFunctionInfo(SystemStatusTypeInfo(SYSTEM_CATEGORY_CMG, STATUS_CATEGORY_THERMAL),   &RegionHandling::sendThermalCMGStatusUpdate));
      _listenerUpdateFunctionsInfo.insert(ListenerUpdateFunctionInfo(SystemStatusTypeInfo(SYSTEM_CATEGORY_CMR, STATUS_CATEGORY_THERMAL),   &RegionHandling::sendThermalCMRStatusUpdate));
      _listenerUpdateFunctionsInfo.insert(ListenerUpdateFunctionInfo(SystemStatusTypeInfo(SYSTEM_CATEGORY_CMA, STATUS_CATEGORY_POWERMODE), &RegionHandling::sendPowerModeCMAStatusUpdate));
      _listenerUpdateFunctionsInfo.insert(ListenerUpdateFunctionInfo(SystemStatusTypeInfo(SYSTEM_CATEGORY_CMP, STATUS_CATEGORY_POWERMODE), &RegionHandling::sendPowerModeCMPStatusUpdate));
      _listenerUpdateFunctionsInfo.insert(ListenerUpdateFunctionInfo(SystemStatusTypeInfo(SYSTEM_CATEGORY_CMG, STATUS_CATEGORY_POWERMODE), &RegionHandling::sendPowerModeCMGStatusUpdate));
      _listenerUpdateFunctionsInfo.insert(ListenerUpdateFunctionInfo(SystemStatusTypeInfo(SYSTEM_CATEGORY_CMR, STATUS_CATEGORY_POWERMODE), &RegionHandling::sendPowerModeCMRStatusUpdate));
   }
}


void RegionHandling::initializeCabinCMASystemsInfo(TiXmlElement* element)
{
   TiXmlElement* tEntityGroupElement = (NULL != element) ? element->FirstChildElement("avdecc_entity_cfg") : NULL;
   if (NULL != tEntityGroupElement)
   {
      for (TiXmlElement* tEntityElement = tEntityGroupElement->FirstChildElement("tENTITY"); (NULL != tEntityElement); tEntityElement = tEntityElement->NextSiblingElement("tENTITY"))
      {
         if (::std::strcmp(getAttribute(tEntityElement, "name", "CMA: tENTITY").c_str(), "CMA") == 0)
         {
            SystemRegionsInfo tSystemRegionsInfo;
            ::std::string tSystemIP(getAttribute(tEntityElement, "IP", "CMA: tENTITY"));
            for (TiXmlElement* tEntitySinkElement = tEntityElement->FirstChildElement("tSINK"); (NULL != tEntitySinkElement); tEntitySinkElement = tEntitySinkElement->NextSiblingElement("tSINK"))
            {
               ::std::string tSinkName = getGroupCFGSinkName(element, getAttribute(tEntitySinkElement, "link", "CMA: tSINK", false), "CMA");
               if (!tSinkName.empty())
               {
                  if (!tSystemRegionsInfo.loadSystemRegionInfo(_sinksStaticInfo, SYSTEM_CATEGORY_CMA, tSinkName))
                  {
                     ETG_TRACE_ERR(("RegionHandling: initializeCabinCMASystemsInfo: Unknown sink configured: SinkName: %s", tSinkName.c_str()));
                  }
               }
            }
            if (!tSystemRegionsInfo.getSystemRegionsInfo().empty())
            {
               uint8 tSystemId = getSystemId(tSystemIP);
               _systemsInfo.addToSystemsInfo(SystemInfo(tSystemId));
               SystemInfo* imp = _systemsInfo.getMutableSystemInfoForSystemId(tSystemId);
               if (NULL != imp)
               {
                  imp->setSystemIP(tSystemIP);
                  imp->setSystemCategory(SYSTEM_CATEGORY_CMA);
                  imp->setSystemRegionsInfo(tSystemRegionsInfo);
                  imp->setEntityId(getEntityId(tEntityElement, "CMA"));
                  imp->setSystemPosition(getSystemPosition(tEntityElement, "CMA"));
                  (void)fetchEntitySourceIdsInfo(imp->getMutableSourceIdsInfo(), tEntityElement, "CMA");
               }
            }
            else
            {
               ETG_TRACE_ERR(("RegionHandling: initializeCabinCMASystemsInfo: No region recognized in entity: IP: %s", tSystemIP.c_str()));
            }
         }
      }
   }
}


void RegionHandling::initializeCabinCMPSystemsInfo(TiXmlElement* element)
{
   TiXmlElement* tEntityGroupElement = (NULL != element) ? element->FirstChildElement("avdecc_entity_cfg") : NULL;
   if (NULL != tEntityGroupElement)
   {
      for (TiXmlElement* tEntityElement = tEntityGroupElement->FirstChildElement("tENTITY"); (NULL != tEntityElement); tEntityElement = tEntityElement->NextSiblingElement("tENTITY"))
      {
         if (::std::strcmp(getAttribute(tEntityElement, "name", "CMP: tENTITY").c_str(), "CMP") == 0)
         {
            ::std::string tSystemIP = getAttribute(tEntityElement, "IP", "CMP: tENTITY");
            uint8 tSystemId         = getSystemId(tSystemIP);
            _systemsInfo.addToSystemsInfo(SystemInfo(tSystemId));
            SystemInfo* imp = _systemsInfo.getMutableSystemInfoForSystemId(tSystemId);
            if (NULL != imp)
            {
               imp->setSystemIP(tSystemIP);
               imp->setSystemCategory(SYSTEM_CATEGORY_CMP);
               imp->setEntityId(getEntityId(tEntityElement, "CMP"));
               imp->setSystemPosition(getSystemPosition(tEntityElement, "CMP"));
               (void)fetchEntitySourceIdsInfo(imp->getMutableSourceIdsInfo(), tEntityElement, "CMP");
            }
         }
      }
   }
   _systemsInfo.evalAndUpdateSystemPosition(SYSTEM_CATEGORY_CMP);
}


void RegionHandling::initializeCabinCMGSystemsInfo(TiXmlElement* element)
{
   TiXmlElement* tEntityGroupElement = (NULL != element) ? element->FirstChildElement("avdecc_entity_cfg") : NULL;
   if (NULL != tEntityGroupElement)
   {
      for (TiXmlElement* tEntityElement = tEntityGroupElement->FirstChildElement("tENTITY"); (NULL != tEntityElement); tEntityElement = tEntityElement->NextSiblingElement("tENTITY"))
      {
         if (::std::strcmp(getAttribute(tEntityElement, "name", "CMG: tENTITY").c_str(), "CMG") == 0)
         {
            SystemRegionsInfo tSystemRegionsInfo;
            ::std::string tSystemIP(getAttribute(tEntityElement, "IP", "CMG: tENTITY"));
            for (TiXmlElement* tEntitySinkElement = tEntityElement->FirstChildElement("tSINK"); (NULL != tEntitySinkElement); tEntitySinkElement = tEntitySinkElement->NextSiblingElement("tSINK"))
            {
               ::std::string tSinkName = getGroupCFGSinkName(element, getAttribute(tEntitySinkElement, "link", "CMG: tSINK", false), "CMG");
               if (!tSinkName.empty())
               {
                  if (!tSystemRegionsInfo.loadSystemRegionInfo(_sinksStaticInfo, SYSTEM_CATEGORY_CMG, tSinkName))
                  {
                     ETG_TRACE_ERR(("RegionHandling: initializeCabinCMGSystemsInfo: Unknown sink configured: SinkName: %s", tSinkName.c_str()));
                  }
               }
            }
            if (!tSystemRegionsInfo.getSystemRegionsInfo().empty())
            {
               uint8 tSystemId = getSystemId(tSystemIP);
               _systemsInfo.addToSystemsInfo(SystemInfo(tSystemId));
               SystemInfo* imp = _systemsInfo.getMutableSystemInfoForSystemId(tSystemId);
               if (NULL != imp)
               {
                  imp->setSystemIP(tSystemIP);
                  imp->setSystemCategory(SYSTEM_CATEGORY_CMG);
                  imp->setSystemRegionsInfo(tSystemRegionsInfo);
                  imp->setEntityId(getEntityId(tEntityElement, "CMG"));
                  imp->setSystemPosition(getSystemPosition(tEntityElement, "CMG"));
               }
            }
            else
            {
               ETG_TRACE_ERR(("RegionHandling: initializeCabinCMGSystemsInfo: No region recognized in entity: IP: %s", tSystemIP.c_str()));
            }
         }
      }
   }
}


void RegionHandling::initializeCabinCMRSystemsInfo(TiXmlElement* element)
{
   TiXmlElement* tEntityGroupElement = (NULL != element) ? element->FirstChildElement("avdecc_entity_cfg") : NULL;
   if (NULL != tEntityGroupElement)
   {
      for (TiXmlElement* tEntityElement = tEntityGroupElement->FirstChildElement("tENTITY"); (NULL != tEntityElement); tEntityElement = tEntityElement->NextSiblingElement("tENTITY"))
      {
         if (::std::strcmp(getAttribute(tEntityElement, "name", "CMR: tENTITY").c_str(), "CMR") == 0)
         {
            ::std::string tSystemIP = getAttribute(tEntityElement, "IP", "CMR: tENTITY");
            uint8 tSystemId         = getSystemId(tSystemIP);
            _systemsInfo.addToSystemsInfo(SystemInfo(tSystemId));
            SystemInfo* imp = _systemsInfo.getMutableSystemInfoForSystemId(tSystemId);
            if (NULL != imp)
            {
               imp->setSystemIP(tSystemIP);
               imp->setSystemCategory(SYSTEM_CATEGORY_CMR);
               imp->setEntityId(getEntityId(tEntityElement, "CMR"));
               imp->setSystemPosition(getSystemPosition(tEntityElement, "CMR"));
               (void)fetchEntitySourceIdsInfo(imp->getMutableSourceIdsInfo(), tEntityElement, "CMR");
            }
         }
      }
   }
}


void RegionHandling::setRegionId(const uint32 regionId)
{
   ETG_TRACE_USR1(("RegionHandling: setRegionId: RegionId: %d", regionId));
   uint32 tRegionId = getRegionId();
   if (tRegionId != regionId)
   {
      TRIGGER_MUTEX_LOCK(_safeLock);
      _regionId = regionId;
      TRIGGER_MUTEX_UNLOCK(_safeLock);
      storeLastActiveCabinRegionId(regionId);
      if (NULL != _applicationSwitchServerComponent)
      {
         _applicationSwitchServerComponent->sendRegionChangePropertyUpdate(regionId);
      }
      sendRegionUpdate(regionId);
   }
}


void RegionHandling::onHmiStateUpdate(const uint8 hmiState)
{
   SystemsInfoType tModifiedSystems;
   uint8 tSystemStatus  = getSystemStatusForSPMState(hmiState, getPNMGrp10Status());
   if (tSystemStatus == SYSTEM_STATUS_NONE) //INFO: Indirectly addresses the SHOW_HMI
   {
      if (!isSPMHMIStateAnimation(getLastHmiState()) && !isSPMHMIStateShowHMI(getLastHmiState())) //INFO: If last state is show/animation then no need to refresh the identification
      {
         SystemsInfoType& tSystemsInfo = _systemsInfo.getMutableSystemsInfo();
         for (SystemsInfoType::iterator itr = tSystemsInfo.begin(); (itr != tSystemsInfo.end()); ++itr)
         {
            if (((*itr).isCabinSystem()) && ((*itr).resetAllStatus()))
            {
               tModifiedSystems.push_back((*itr));
            }
         }
         startSystemIdentificationTimer();
         if (!tModifiedSystems.empty())
         {
            dumpSystemsInfo();
            sendCabinsSystemsInfoUpdate(tModifiedSystems);
         }
         if (NULL != _hmiStateHandler)
         {
            _hmiStateHandler->sendSubsystemerrorStatusGetRequest();
         }
#ifdef VARIANT_S_FTR_ENABLE_PLUGINS
         if (NULL != PluginClientHandler::getInstance())
         {
            PluginClientHandler::getInstance()->sendEntityDetailsUpdateGetRequest();
         }
#endif
      }
   }
   else
   {
      stopSystemIdentificationTimer(false);
      evalAndUpdateCabinSystemStatus(hmiState);
   }
}


void RegionHandling::onPNMGrp10StatusUpdate(const bool /* status */)
{
   stopSystemIdentificationTimer(false);
   evalAndUpdateCabinSystemStatus(getHmiState());
}


void RegionHandling::onSubSystemsErrorStatus(const SubSystemsErrorStatusInfo& info)
{
   if (!info.empty())
   {
      SystemsInfoType tModifiedSystems;
      for (SubSystemsErrorStatusInfo::const_iterator itr = info.begin(); (itr != info.end()); ++itr)
      {
         SystemInfo* imp = _systemsInfo.getMutableSystemInfoForSystemId(itr->first);
         if ((NULL != imp) && (imp->isCabinSystem()))
         {
            if (imp->setStatus(STATUS_CATEGORY_NANOMSG, convert2BitSystemConnectionStatusFromLCM(getBitValue(itr->second.second, 0, 0x02))))
            {
               tModifiedSystems.push_back((*imp));
               sendStatusUpdate((*imp), STATUS_CATEGORY_NANOMSG);
            }
            if (imp->setStatus(STATUS_CATEGORY_POWERMODE, convert3BitSystemPowerModeStatusFromLCM(getBitValue(itr->second.second, 0x0C, 0x03))))
            {
               sendStatusUpdate((*imp), STATUS_CATEGORY_POWERMODE);
            }
            if (imp->setStatus(STATUS_CATEGORY_VOLTAGE, convert3BitSystemVoltageErrorStatusFromLCM(getBitValue(itr->second.second, 0x06, 0x03))))
            {
               sendStatusUpdate((*imp), STATUS_CATEGORY_VOLTAGE);
            }
            if (imp->setStatus(STATUS_CATEGORY_THERMAL, convert3BitSystemThermalErrorStatusFromLCM(getBitValue(itr->second.second, 0x09, 0x03))))
            {
               sendStatusUpdate((*imp), STATUS_CATEGORY_THERMAL);
            }
         }
      }
      if (_systemsInfo.isAllCabinSystemsIdentified())
      {
         stopSystemIdentificationTimer(false);
      }
      sendCabinsSystemsInfoUpdate(tModifiedSystems);
      evalAndUpdateCabinSystemStatus(getHmiState());
   }
}


void RegionHandling::onPluginEntityDetailsUpdate(const ::boost::shared_ptr< EventListDataUtility >& info)
{
   if (NULL != info.get())
   {
      const ::std::vector< const EventDataUtility* > dataUtilities = info->getData();
      if (!dataUtilities.empty())
      {
         SystemsInfoType tModifiedSystems;
         for (::std::vector< const EventDataUtility* >::const_iterator itr = dataUtilities.begin(); (itr != dataUtilities.end()); ++itr)
         {
            if (NULL != (*itr))
            {
               const ::std::vector< EventDataItem* >& dataItems = (*itr)->getData();
               uint8 tSystemId      = 0;
               uint8 tAVDECCStatus  = SYSTEM_STATUS_NONE;
               uint64 tEntityId     = 0;
               if ((dataItems.size() > 0) && (NULL != dataItems[0]))
               {
                  tEntityId = dataItems[0]->getData()._value._uint64Value;
               }
               if ((dataItems.size() > 0x02) && (NULL != dataItems[0x02]))
               {
                  tAVDECCStatus = (dataItems[0x02]->getData()._value._boolValue) ? SYSTEM_STATUS_CONNECTED : SYSTEM_STATUS_DISCONNECTED;
               }
               if ((_systemsInfo.fetchSystemIdForEntityId(tEntityId, tSystemId)) && (tAVDECCStatus != SYSTEM_STATUS_NONE))
               {
                  ETG_TRACE_USR1(("RegionHandling: onPluginEntityDetailsUpdate: <-: SystemId: %d : Status: %d", \
                                  tSystemId, ETG_CENUM(enSystemStatus, tAVDECCStatus)));
                  SystemInfo* imp = _systemsInfo.getMutableSystemInfoForSystemId(tSystemId);
                  if ((NULL != imp) && (imp->isCabinSystem()) && (imp->setStatus(STATUS_CATEGORY_AVDECC, tAVDECCStatus)))
                  {
                     tModifiedSystems.push_back((*imp));
                     sendStatusUpdate((*imp), STATUS_CATEGORY_AVDECC);
                  }
               }
               else
               {
                  ETG_TRACE_ERR(("RegionHandling: onPluginEntityDetailsUpdate: Unknow system or status: EntityId: %lx", tEntityId));
               }
            }
         }
         if (_systemsInfo.isAllCabinSystemsIdentified())
         {
            stopSystemIdentificationTimer(false);
         }
         sendCabinsSystemsInfoUpdate(tModifiedSystems);
         evalAndUpdateCabinSystemStatus(getHmiState());
      }
   }
}


void RegionHandling::onExpired(asf::core::Timer& timer, boost::shared_ptr<asf::core::TimerPayload> data)
{
   if ((NULL != _systemIdentificationTimer) && ((&timer) == (_systemIdentificationTimer)) && (data->getReason() == asf::core::TimerPayload_Reason__Completed))
   {
      ETG_TRACE_ERR(("RegionHandling: onExpired: SystemIdentificationTimer"));
      SystemsInfoType tModifiedSystems;
      SystemsInfoType& tSystemsInfo = _systemsInfo.getMutableSystemsInfo();
      for (SystemsInfoType::iterator itr = tSystemsInfo.begin(); (itr != tSystemsInfo.end()); ++itr)
      {
         bool isValidToUpdate = ((*itr).isCabinSystem()) ? true : false;
         isValidToUpdate      = ((isValidToUpdate) && (!(*itr).isSystemIdentified())) ? true : false;
         isValidToUpdate      = ((isValidToUpdate) && ((*itr).getStatus(STATUS_CATEGORY_SYSTEM) == SYSTEM_STATUS_NONE)) ? true : false;
         isValidToUpdate      = ((isValidToUpdate) && ((*itr).setStatus(STATUS_CATEGORY_SYSTEM, SYSTEM_STATUS_UNAVAILABLE))) ? true : false;
         if (isValidToUpdate)
         {
            tModifiedSystems.push_back((*itr));
         }
      }
      dumpSystemsInfo();
      stopSystemIdentificationTimer(false);
      sendCabinsSystemsInfoUpdate(tModifiedSystems);
      sendStatusUpdate(tModifiedSystems, STATUS_CATEGORY_SYSTEM);
   }
}


void RegionHandling::dumpSystemsInfo()
{
   ETG_TRACE_USR1(("\n"
                   "//----------------------------------------------------------------------------------------------------------//""\n"
                   "                                                 SYSTEMS INFO ""\n"
                   "//----------------------------------------------------------------------------------------------------------//""\n"));
   ETG_TRACE_USR1(("\n"
                   "   ACTIVE REGION: %d""\n"
                   "  ------------------------------------------------------------------------------------------------""\n", \
                   ETG_CENUM(enRegionID, getRegionId())));
   ETG_TRACE_USR1(("\n"
                   "   ACTIVE CABIN REGION (LAST): %d""\n"
                   "  ------------------------------------------------------------------------------------------------""\n", \
                   ETG_CENUM(enRegionID, getLastActiveCabinRegionId())));
   ETG_TRACE_USR1(("\n"
                   "   SYSTEMS INFO:""\n"));
   const SystemsInfoType& tSystemsInfo = _systemsInfo.getSystemsInfo();
   for (SystemsInfoType::const_iterator itr = tSystemsInfo.begin(); (itr != tSystemsInfo.end()); ++itr)
   {
      ETG_TRACE_USR1(("\n"
                      "   IP                : %s""\n", (*itr).getSystemIP().c_str()));
      ETG_TRACE_USR1(("\n"
                      "   ENTITY ID         : %lx""\n", (*itr).getEntityId()));
      ETG_TRACE_USR1(("\n"
                      "   CATEGORY          : %d""\n"
                      "   SYSTEM ID         : %d""\n"
                      "   SYSTEM POS        : %d""\n"
                      "   SYSTEM STATUS     : %d""\n"
                      "   AVDECC STATUS     : %d""\n"
                      "   NANOMSG STATUS    : %d""\n"
                      "   VOLTAGE STATUS    : %d""\n"
                      "   THERMAL STATUS    : %d""\n"
                      "   POWER MODE STATUS : %d""\n",                                        \
                      ETG_CENUM(enSystemCategory, (*itr).getSystemCategory()),                \
                      (*itr).getSystemId(), (*itr).getSystemPosition(),                       \
                      ETG_CENUM(enSystemStatus, (*itr).getSystemStatus()),                    \
                      ETG_CENUM(enSystemStatus, (*itr).getAVDECCStatus()),                    \
                      ETG_CENUM(enSystemStatus, (*itr).getNANOMsgStatus()),                   \
                      ETG_CENUM(enSystemVoltageStatus, (*itr).getVoltageStatus()),            \
                      ETG_CENUM(enSystemThermalStatus, (*itr).getThermalStatus()),            \
                      ETG_CENUM(enSystemStatus, (*itr).getPowerModeStatus())));
      if (!(*itr).getSourceIdsInfo().empty())
      {
         ETG_TRACE_USR1(("\n"
                         "   SYSTEM SOURCES INFO:""\n"));
         for (SourceIdsInfo::const_iterator tItr = (*itr).getSourceIdsInfo().begin(); (tItr != (*itr).getSourceIdsInfo().end()); ++tItr)
         {
            ETG_TRACE_USR1(("\n"
                            "      SOURCE ID: %d""\n", (*tItr)));
         }
      }
      const SystemRegionsInfoType& tSystemRegionsInfo = (*itr).getSystemRegionsInfo().getSystemRegionsInfo();
      if (!tSystemRegionsInfo.empty())
      {
         ETG_TRACE_USR1(("\n"
                         "   SYSTEM REGIONS INFO:""\n"));
         for (SystemRegionsInfoType::const_iterator sItr = tSystemRegionsInfo.begin(); (sItr != tSystemRegionsInfo.end()); ++sItr)
         {
            ETG_TRACE_USR1(("\n"
                            "   REGION ID: %d""\n", ETG_CENUM(enRegionId, (*sItr).getRegionId())));
            ETG_TRACE_USR1(("\n"
                            "   GROUP ID : %d""\n", ETG_CENUM(enSystemGroup, (*sItr).getSystemGroup())));
            if (!(*sItr).getSinkIdsInfo().empty())
            {
               ETG_TRACE_USR1(("\n"
                               "   REGION SINKS INFO:""\n"));
               for (SinkIdsInfo::const_iterator tItr = (*sItr).getSinkIdsInfo().begin(); (tItr != (*sItr).getSinkIdsInfo().end()); ++tItr)
               {
                  ETG_TRACE_USR1(("\n"
                                  "      SINK ID: %d""\n", (*tItr)));
               }
            }
            if (!(*sItr).getSourceIdsInfo().empty())
            {
               ETG_TRACE_USR1(("\n"
                               "   REGION SOURCES INFO:""\n"));
               for (SourceIdsInfo::const_iterator tItr = (*sItr).getSourceIdsInfo().begin(); (tItr != (*sItr).getSourceIdsInfo().end()); ++tItr)
               {
                  ETG_TRACE_USR1(("\n"
                                  "      SOURCE ID: %d""\n", (*tItr)));
               }
            }
         }
      }
      ETG_TRACE_USR1(("\n""  ------------------------------------------------------------------------------------------------""\n"));
   }
}


void RegionHandling::startSystemIdentificationTimer()
{
   if (NULL == _systemIdentificationTimer)
   {
      _systemIdentificationTimer = new ::asf::core::Timer(*this, 70000);
   }
   if (NULL != _systemIdentificationTimer)
   {
      ETG_TRACE_USR4(("RegionHandling: startSystemIdentificationTimer"));
      _systemIdentificationTimer->start();
   }
}


void RegionHandling::stopSystemIdentificationTimer(const bool flush)
{
   if (NULL != _systemIdentificationTimer)
   {
      ETG_TRACE_USR4(("RegionHandling: stopSystemIdentificationTimer: Flush: %d", flush));
      _systemIdentificationTimer->stop();
      if (flush)
      {
         delete _systemIdentificationTimer;
         _systemIdentificationTimer = NULL;
      }
   }
}


void RegionHandling::storeLastActiveCabinRegionId(const uint32 regionId)
{
   ETG_TRACE_USR1(("RegionHandling: storeLastActiveCabinRegionId: RegionId: %d", regionId));
   uint32 lastActiveCabinRegionId = getLastActiveCabinRegionId();
   if ((lastActiveCabinRegionId != regionId) && (Region__REG_COCKPIT != regionId))
   {
      TRIGGER_MUTEX_LOCK(_safeLock);
      _lastActiveCabinRegionId = regionId;
      TRIGGER_MUTEX_UNLOCK(_safeLock);
   }
}


void RegionHandling::sendCabinsSystemsInfoUpdate(const SystemsInfoType& info)
{
   if ((NULL != _hmiInfoServiceServerComponent) && (!info.empty()))
   {
      _hmiInfoServiceServerComponent->sendCabinsSystemsInfoUpdate(info);
   }
}


void RegionHandling::sendStatusUpdate(const SystemInfo& info, const uint8 statusCategory)
{
   SystemStatusInfo tSystemStatusInfo;
   const SystemRegionsInfo& tRegionsInfo = info.getSystemRegionsInfo();
   tSystemStatusInfo.setSystemId(info.getSystemId());
   tSystemStatusInfo.setStatus(info.getStatus(statusCategory));
   tSystemStatusInfo.setSystemPosition(info.getSystemPosition());
   tSystemStatusInfo.setSystemGroup(tRegionsInfo.getSystemGroup());
   tSystemStatusInfo.setRegionsIdInfo(tRegionsInfo.getSystemRegionsIdInfo());
   ::std::pair< uint8, uint8 > temp(info.getSystemCategory(), statusCategory);
   ListenerUpdateFunctionsInfo::iterator sItr = _listenerUpdateFunctionsInfo.find(temp);
   if (sItr != _listenerUpdateFunctionsInfo.end())
   {
      ((this)->*(sItr->second))(tSystemStatusInfo);
   }
}


void RegionHandling::sendStatusUpdate(const SystemsInfoType& info, const uint8 statusCategory)
{
   for (SystemsInfoType::const_iterator itr = info.begin(); (itr != info.end()); ++itr)
   {
      sendStatusUpdate((*itr), statusCategory);
   }
}


void RegionHandling::evalAndUpdateCabinSystemStatus(const uint8 hmiState, const uint8 systemId)
{
   SystemsInfoType tModifiedSystems;
   bool terminate                = false;
   uint8 tCMCSystemStatus        = getSystemStatusForSPMState(hmiState, getPNMGrp10Status());
   SystemsInfoType& tSystemsInfo = _systemsInfo.getMutableSystemsInfo();
   for (SystemsInfoType::iterator itr = tSystemsInfo.begin(); ((itr != tSystemsInfo.end()) && (!terminate)); ++itr)
   {
      if ((*itr).isCabinSystem())
      {
         if ((systemId == 0xFF) || ((*itr).getSystemId() == systemId))
         {
            terminate = (systemId != 0xFF) ? true : false;
            if ((*itr).isSystemIdentified())
            {
               uint8 tSystemStatus = SYSTEM_STATUS_NONE;
               if (tCMCSystemStatus != SYSTEM_STATUS_NONE)
               {
                  tSystemStatus = tCMCSystemStatus;
                  /*
                   * INFO: The interface "hmiState" update is happening after the "SubsystemerrorStatus"
                   *       and the same is leading to status mismatch in cockpit mode(Ex: PNM replaced
                   *       with cockpit mode). Hence added handling to retain the last status if no
                   *       valid error reason for cokpit mode
                   */
                  if (tSystemStatus == SYSTEM_STATUS_COCKPIT_DISCONNECTED)
                  {
                     bool isAVDECCInvalid  = ((*itr).getAVDECCStatus() != SYSTEM_STATUS_CONNECTED) ? true : false;
                     bool isNANOMsgInvalid = ((*itr).getNANOMsgStatus() != SYSTEM_STATUS_CONNECTED) ? true : false;
                     if ((!getPNMGrp10Status()) && (!isAVDECCInvalid) && (!isNANOMsgInvalid))
                     {
                        tSystemStatus = (*itr).getSystemStatus();
                     }
                  }
               }
               else
               {
                  bool isPowerModeInValid = ((*itr).getPowerModeStatus() != SYSTEM_STATUS_CONNECTED) ? true : false;
                  isPowerModeInValid      = ((isPowerModeInValid) && ((*itr).getPowerModeStatus() != SYSTEM_STATUS_DIAGNOSIS_DISCONNECTED)) ? true : false;
                  if (isPowerModeInValid)
                  {
                     tSystemStatus = (*itr).getPowerModeStatus();
                  }
                  else
                  {
                     tSystemStatus         = SYSTEM_STATUS_CONNECTED;
                     bool isAVDECCInvalid  = ((*itr).getAVDECCStatus() != SYSTEM_STATUS_CONNECTED) ? true : false;
                     bool isNANOMsgInvalid = ((*itr).getNANOMsgStatus() != SYSTEM_STATUS_CONNECTED) ? true : false;
                     if ((isAVDECCInvalid) || (isNANOMsgInvalid))
                     {
                        tSystemStatus = SYSTEM_STATUS_DISCONNECTED;
                     }
                  }
               }
               if ((*itr).setStatus(STATUS_CATEGORY_SYSTEM, tSystemStatus))
               {
                  tModifiedSystems.push_back((*itr));
               }
            }
         }
      }
   }
   if (!tModifiedSystems.empty())
   {
      dumpSystemsInfo();
      sendCabinsSystemsInfoUpdate(tModifiedSystems);
      sendStatusUpdate(tModifiedSystems, STATUS_CATEGORY_SYSTEM);
   }
}


SupportedRegionsInfo RegionHandling::getSupportedRegionsInfo() const
{
   SupportedRegionsInfo tInfo;
   TRIGGER_MUTEX_LOCK(_safeLock);
   tInfo = _supportedRegionsInfo;
   TRIGGER_MUTEX_UNLOCK(_safeLock);
   return tInfo;
}


::std::string RegionHandling::getGroupCFGSinkName(TiXmlElement* element, ::std::string linkName, ::std::string category)
{
   ::std::string tSinkName = "";
   if (!linkName.empty())
   {
      TiXmlElement* tGroupElement = element->FirstChildElement("avdecc_group_cfg");
      if (NULL != tGroupElement)
      {
         for (TiXmlElement* tGroupSinkElement = tGroupElement->FirstChildElement("tSINK"); (NULL != tGroupSinkElement); tGroupSinkElement = tGroupSinkElement->NextSiblingElement("tSINK"))
         {
            if (linkName == getAttribute(tGroupSinkElement, "name", (category + ": tSINK")))
            {
               tSinkName = getAttribute(tGroupSinkElement, "sinkNam", (category + ": tSINK"));
               break;
            }
         }
      }
   }
   return tSinkName;
}


::std::string RegionHandling::getAttribute(TiXmlElement* element, const char* attribute, ::std::string category, const bool errmem)
{
   ::std::string attributeValue = "";
   if (NULL != element)
   {
      if (element->Attribute(attribute) != NULL)
      {
         attributeValue = element->Attribute(attribute);
      }
      else
      {
         ETG_TRACE_ERR(("RegionHandling: getAttribute: Attribute: %100s is NULL for Category: %100s", attribute, category.c_str()));
         if (errmem)
         {
            ETG_TRACE_ERRMEM(("!!!!!! APPHMI_MASTER : RegionHandling: getAttribute: Attribute: %100s is NULL for Category: %100s :  INFO:Restart not required !!!!!!",
                              attribute, category.c_str()));
         }
      }
   }
   return attributeValue;
}


int16 RegionHandling::getAudioSinkIdForRegion(const uint32 regionId) const
{
   int16 tSinkId = SINK_INVALID;
   TRIGGER_MUTEX_LOCK(_safeLock);
   SupportedRegionsInfo tSupportedRegionsInfo = _supportedRegionsInfo;
   TRIGGER_MUTEX_UNLOCK(_safeLock);
   for (SupportedRegionsInfo::const_iterator itr = tSupportedRegionsInfo.begin(); (itr != tSupportedRegionsInfo.end()); ++itr)
   {
      if ((*itr).first == regionId)
      {
         if (!(*itr).second.empty())
         {
            tSinkId = (*itr).second[0];
         }
         break;
      }
   }
   return tSinkId;
}


uint8 RegionHandling::getSystemGroupForSinkId(const int16 sinkId) const
{
   uint8 group = SYSTEM_GROUP_NONE;
   for (SinksStaticInfo::const_iterator itr = _sinksStaticInfo.begin(); (itr != _sinksStaticInfo.end()); ++itr)
   {
      if ((*itr).getSinkId() == sinkId)
      {
         group = (*itr).getSystemGroup();
         break;
      }
   }
   return group;
}


uint8 RegionHandling::getSystemGroupCategoryForSinkId(const int16 sinkId) const
{
   uint8 systemGroupCategory = SYSTEM_GROUP_CATEGORY_NONE;
   for (SinksStaticInfo::const_iterator itr = _sinksStaticInfo.begin(); (itr != _sinksStaticInfo.end()); ++itr)
   {
      if ((*itr).getSinkId() == sinkId)
      {
         systemGroupCategory = (*itr).getSystemGroupCategory();
         break;
      }
   }
   return systemGroupCategory;
}


uint8 RegionHandling::getSystemId(const ::std::string ipAddress) const
{
   uint8 tSystemId = 0;
   if (!ipAddress.empty())
   {
      ::std::size_t end   = ipAddress.length();
      ::std::size_t begin = ipAddress.find_last_of(".");
      if (begin != ::std::string::npos)
      {
         begin             = begin + 0x01;
         ::std::string tIP = ipAddress.substr(begin, end);
         if (!tIP.empty())
         {
            uint32 temp = 0;
            ::std::istringstream tStream(tIP);
            tStream >> temp;
            tSystemId = static_cast<uint8>(temp);
         }
      }
   }
   return tSystemId;
}


uint8 RegionHandling::getSystemPosition(TiXmlElement* entityElement, const ::std::string category)
{
   uint8 tPosition    = 255;
   ::std::string tStr = getAttribute(entityElement, "pos", (category + ": tENTITY"), false);
   if (!tStr.empty())
   {
      uint32 temp = 0;
      ::std::istringstream tStream(tStr);
      tStream >> temp;
      tPosition = static_cast<uint8>(temp);
   }
   return tPosition;
}


uint32 RegionHandling::getRegionId() const
{
   uint32 tRegionId = Region__MAX_REG;
   TRIGGER_MUTEX_LOCK(_safeLock);
   tRegionId = _regionId;
   TRIGGER_MUTEX_UNLOCK(_safeLock);
   return tRegionId;
}


uint32 RegionHandling::getLastActiveCabinRegionId() const
{
   uint32 tlastActiveCabinRegionId = Region__MAX_REG;
   TRIGGER_MUTEX_LOCK(_safeLock);
   tlastActiveCabinRegionId = _lastActiveCabinRegionId;
   TRIGGER_MUTEX_UNLOCK(_safeLock);
   return tlastActiveCabinRegionId;
}


uint32 RegionHandling::getRegionIdForAudioSink(const uint32 sinkId) const
{
   uint32 regionId = REGION_INVALID;
   TRIGGER_MUTEX_LOCK(_safeLock);
   SupportedRegionsInfo tSupportedRegionsInfo = _supportedRegionsInfo;
   TRIGGER_MUTEX_UNLOCK(_safeLock);
   for (SupportedRegionsInfo::const_iterator itr = tSupportedRegionsInfo.begin(); (itr != tSupportedRegionsInfo.end()); ++itr)
   {
      const SinkIdsInfo& tSinkIdsInfo = (*itr).second;
      if (::std::find(tSinkIdsInfo.begin(), tSinkIdsInfo.end(), sinkId) != tSinkIdsInfo.end())
      {
         regionId = (*itr).first;
         break;
      }
   }
   return regionId;
}


uint64 RegionHandling::getEntityId(TiXmlElement* entityElement, const ::std::string category)
{
   uint64 tEntityId = 0;
   ::std::string tStr = getAttribute(entityElement, "eId", (category + ": tENTITY"), true);
   if (!tStr.empty())
   {
      char* tEnd;
      tEntityId = ::std::strtoull(tStr.c_str(), &tEnd, 16);
   }
   return tEntityId;
}


bool RegionHandling::isRegionSupported(const uint32 regionId) const
{
   bool isValid = false;
   TRIGGER_MUTEX_LOCK(_safeLock);
   SupportedRegionsInfo tSupportedRegionsInfo = _supportedRegionsInfo;
   TRIGGER_MUTEX_UNLOCK(_safeLock);
   for (SupportedRegionsInfo::const_iterator itr = tSupportedRegionsInfo.begin(); (itr != tSupportedRegionsInfo.end()); ++itr)
   {
      if ((*itr).first == regionId)
      {
         isValid = true;
         break;
      }
   }
   return isValid;
}


bool RegionHandling::isSinkGroupCategorySame(const int16 sinkId, const uint8 category) const
{
   bool isValid = false;
   for (SinksStaticInfo::const_iterator itr = _sinksStaticInfo.begin(); (itr != _sinksStaticInfo.end()); ++itr)
   {
      if ((*itr).getSinkId() == sinkId)
      {
         isValid = ((*itr).getSystemGroupCategory() == category) ? true : false;
         break;
      }
   }
   return isValid;
}


bool RegionHandling::fetchEntitySourceIdsInfo(SourceIdsInfo& info, TiXmlElement* entityElement, const ::std::string category)
{
   bool isFetched = false;
   if (NULL != entityElement)
   {
      for (TiXmlElement* tSourceElement = entityElement->FirstChildElement("tSOURCE"); (NULL != tSourceElement); tSourceElement = tSourceElement->NextSiblingElement("tSOURCE"))
      {
         ::std::string tSourceName = getAttribute(tSourceElement, "srcNam", (category + ": tSOURCE"), false);
         int16 tSourceId           = SrcMap::getSrcMapInstance()->getSourceId(tSourceName);
         if (tSourceId != SRC_INVALID)
         {
            if (::std::find(info.begin(), info.end(), tSourceId) == info.end())
            {
               isFetched = true;
               info.push_back(tSourceId);
            }
         }
         else
         {
            ETG_TRACE_USR4(("RegionHandling: fetchEntitySourceIdsInfo: Unknown source configured: SourceName: %s", tSourceName.c_str()));
         }
      }
   }
   return isFetched;
}


bool RegionHandling::isCMGConfiguredForGroupCategory(const uint32 regionId, const uint8 groupCategory)
{
   bool status = false;
   SinkIdsInfo tInfo;
   if ((fetchSinkIdsInfoForGroupCategory(tInfo, groupCategory, SYSTEM_CATEGORY_CMG, regionId)) && (!tInfo.empty()))
   {
      status = true;
   }
   return status;
}


bool RegionHandling::fetchSinkIdsInfoForGroupCategory(SinkIdsInfo& info, const uint8 groupCategory, const uint8 systemCategory, const uint32 regionId) const
{
   bool isFetched = false;
   isFetched = fetchSinkIdsInfo(info, systemCategory, regionId);
   if ((groupCategory != SYSTEM_GROUP_CATEGORY_NONE) && (isFetched))
   {
      for (SinkIdsInfo::iterator itr = info.begin(); itr != info.end();)
      {
         if (!isSinkGroupCategorySame((*itr), groupCategory))
         {
            itr = info.erase(itr);
         }
         else
         {
            ++itr;
         }
      }
      isFetched = (info.empty()) ? false : true;
   }
   return isFetched;
}


bool RegionHandling::fetchSystemsIdInfoForGroupCategory(SystemsIdInfo& info, const uint8 groupCategory, const uint8 systemCategory, const uint32 regionId) const
{
   bool isFetched = false;
   for (SinksStaticInfo::const_iterator itr = _sinksStaticInfo.begin(); (itr != _sinksStaticInfo.end()); ++itr)
   {
      if ((systemCategory == SYSTEM_CATEGORY_NONE) || ((*itr).getSystemCategory() == systemCategory))
      {
         if ((regionId == REGION_INVALID) || ((*itr).getRegionId() == regionId))
         {
            if ((groupCategory == SYSTEM_GROUP_CATEGORY_NONE) || ((*itr).getSystemGroupCategory() == groupCategory))
            {
               isFetched = fetchSystemsIdInfoForSinkId(info, (*itr).getSinkId(), systemCategory, regionId) || isFetched;
            }
         }
      }
   }
   return isFetched;
}


REGION_HANDLING_LISTENER_DATA_DEFN_BEGIN()
ADD_LISTENER_DATA_DEFN(Region, uint32)
ADD_LISTENER_DATA_DEFN(CMAStatus, SystemStatusInfo)
ADD_LISTENER_DATA_DEFN(CMPStatus, SystemStatusInfo)
ADD_LISTENER_DATA_DEFN(CMGStatus, SystemStatusInfo)
ADD_LISTENER_DATA_DEFN(CMRStatus, SystemStatusInfo)
ADD_LISTENER_DATA_DEFN(AVDECCCMAStatus, SystemStatusInfo)
ADD_LISTENER_DATA_DEFN(AVDECCCMPStatus, SystemStatusInfo)
ADD_LISTENER_DATA_DEFN(AVDECCCMGStatus, SystemStatusInfo)
ADD_LISTENER_DATA_DEFN(AVDECCCMRStatus, SystemStatusInfo)
ADD_LISTENER_DATA_DEFN(NANOMsgCMAStatus, SystemStatusInfo)
ADD_LISTENER_DATA_DEFN(NANOMsgCMPStatus, SystemStatusInfo)
ADD_LISTENER_DATA_DEFN(NANOMsgCMGStatus, SystemStatusInfo)
ADD_LISTENER_DATA_DEFN(NANOMsgCMRStatus, SystemStatusInfo)
ADD_LISTENER_DATA_DEFN(VoltageCMCStatus, SystemStatusInfo)
ADD_LISTENER_DATA_DEFN(VoltageCMAStatus, SystemStatusInfo)
ADD_LISTENER_DATA_DEFN(VoltageCMPStatus, SystemStatusInfo)
ADD_LISTENER_DATA_DEFN(VoltageCMGStatus, SystemStatusInfo)
ADD_LISTENER_DATA_DEFN(VoltageCMRStatus, SystemStatusInfo)
ADD_LISTENER_DATA_DEFN(ThermalCMCStatus, SystemStatusInfo)
ADD_LISTENER_DATA_DEFN(ThermalCMAStatus, SystemStatusInfo)
ADD_LISTENER_DATA_DEFN(ThermalCMPStatus, SystemStatusInfo)
ADD_LISTENER_DATA_DEFN(ThermalCMGStatus, SystemStatusInfo)
ADD_LISTENER_DATA_DEFN(ThermalCMRStatus, SystemStatusInfo)
ADD_LISTENER_DATA_DEFN(PowerModeCMAStatus, SystemStatusInfo)
ADD_LISTENER_DATA_DEFN(PowerModeCMPStatus, SystemStatusInfo)
ADD_LISTENER_DATA_DEFN(PowerModeCMGStatus, SystemStatusInfo)
ADD_LISTENER_DATA_DEFN(PowerModeCMRStatus, SystemStatusInfo)
REGION_HANDLING_LISTENER_DATA_DEFN_END()
} //namespace Core
} //namespace App


/**
 *  undef
 */

#undef REGION_HANDLING_LISTENER_DATA_DEFN_BEGIN
#undef ADD_LISTENER_DATA_DEFN
#undef REGION_HANDLING_LISTENER_DATA_DEFN_END
#undef TRIGGER_MUTEX_LOCK
#undef TRIGGER_MUTEX_UNLOCK
