/**
 * @file         : MediaClientHandler.cpp
 * @author       : INF4CV - AppHmi_Master Team
 * @addtogroup   : AppHmi_Master
 * @brief        : MediaClientHandler is to handle the service client implementation of
 *                 MediaPlayer services
 * @copyright    : (C) 2016 Robert Bosch GmbH
 *                 (C) 2016 Robert Bosch Engineering and Business Solutions Limited
 *                 The reproduction, distribution and utilization of this file as
 *                 well as the communication of its contents to others without express
 *                 authorization is prohibited. Offenders will be held liable for the
 *                 payment of damages. All rights reserved in the event of the grant
 *                 of a patent, utility model or design.
 */


#include "hall_std_if.h"
#include "MediaClientHandler.h"
#include "mplay_MediaPlayer_FI.h"
#include "MediaClientHandlerTypes.h"


using namespace ::mplay_MediaPlayer_FI;


#ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
#define ETG_DEFAULT_TRACE_CLASS TR_CLASS_APPHMI_MASTER_HALL
#define ETG_I_FILE_PREFIX App::Core::MediaClientHandler::
#include "trcGenProj/Header/MediaClientHandler.cpp.trc.h"
#endif


namespace App {
namespace Core {


MediaClientHandler::MediaClientHandler() : _guiInitialized(false)
   , _regIdDeviceConnectionsUpdate(0)
   , _mediaPlayerProxy(Mplay_MediaPlayer_FIProxy::createProxy("mediaPlayerFiPort", *this))
{
   ETG_TRACE_USR4(("MediaClientHandler: CTOR"));
   _deviceConnectionsCBInfo.clear();
   if (NULL != _mediaPlayerProxy.get())
   {
      StartupSync::getInstance().registerPropertyRegistrationIF(this, _mediaPlayerProxy->getPortName());
   }
}


MediaClientHandler::~MediaClientHandler()
{
   ETG_TRACE_USR4(("MediaClientHandler: DTOR"));
   _guiInitialized               = false;
   _regIdDeviceConnectionsUpdate = 0;
   _deviceConnectionsCBInfo.clear();
}


void MediaClientHandler::registerDeviceConnectionsUpdate(IDeviceConnectionsCB& imp)
{
   ::std::vector< IDeviceConnectionsCB* >::const_iterator itr = ::std::find(_deviceConnectionsCBInfo.begin(), _deviceConnectionsCBInfo.end(), (&imp));
   if (itr == _deviceConnectionsCBInfo.end())
   {
      _deviceConnectionsCBInfo.push_back((&imp));
      if (_regIdDeviceConnectionsUpdate == 0)
      {
         registerDeviceConnectionsProperty();
      }
      else
      {
         ::std::vector< DeviceConnectionInfo > tInfo;
         if (fetchDeviceConnectionsInfo(tInfo))
         {
            if (NULL != (&imp))
            {
               (&imp)->onDeviceConnections(tInfo);
            }
         }
      }
   }
}


void MediaClientHandler::deregisterDeviceConnectionsUpdate(IDeviceConnectionsCB& imp)
{
   ::std::vector< IDeviceConnectionsCB* >::iterator itr = ::std::find(_deviceConnectionsCBInfo.begin(), _deviceConnectionsCBInfo.end(), (&imp));
   if (itr != _deviceConnectionsCBInfo.end())
   {
      (void)_deviceConnectionsCBInfo.erase(itr);
      if (_deviceConnectionsCBInfo.empty())
      {
         deregisterDeviceConnectionsProperty();
      }
   }
}


void MediaClientHandler::onMediaPlayerDeviceConnectionsError(const ::boost::shared_ptr< ::mplay_MediaPlayer_FI::Mplay_MediaPlayer_FIProxy >& proxy, const boost::shared_ptr< ::mplay_MediaPlayer_FI::MediaPlayerDeviceConnectionsError >& error)
{
   if ((NULL != _mediaPlayerProxy.get()) && (proxy == _mediaPlayerProxy) && (error->getAct() == _regIdDeviceConnectionsUpdate))
   {
      ETG_TRACE_ERR(("MediaClientHandler: onMediaPlayerDeviceConnectionsError"));
   }
}


void MediaClientHandler::onMediaPlayerDeviceConnectionsStatus(const ::boost::shared_ptr< ::mplay_MediaPlayer_FI::Mplay_MediaPlayer_FIProxy >& proxy, const boost::shared_ptr< ::mplay_MediaPlayer_FI::MediaPlayerDeviceConnectionsStatus >& status)
{
   if ((NULL != _mediaPlayerProxy.get()) && (proxy == _mediaPlayerProxy) && (status->getAct() == _regIdDeviceConnectionsUpdate))
   {
      ::std::vector< DeviceConnectionInfo > tInfo = getDeviceConnectionsInfo((*(status.get())));
      for (::std::vector< IDeviceConnectionsCB* >::iterator itr = _deviceConnectionsCBInfo.begin(); (itr != _deviceConnectionsCBInfo.end()); ++itr)
      {
         if (NULL != (*itr))
         {
            (*itr)->onDeviceConnections(tInfo);
         }
      }
   }
}


bool MediaClientHandler::fetchDeviceConnectionsInfo(::std::vector< DeviceConnectionInfo >& info)
{
   bool isFetched = false;
   if ((NULL != _mediaPlayerProxy.get()) && (_mediaPlayerProxy->hasMediaPlayerDeviceConnections()))
   {
      info      = getDeviceConnectionsInfo(_mediaPlayerProxy->getMediaPlayerDeviceConnections());
      isFetched = true;
   }
   return isFetched;
}


::std::vector< DeviceConnectionInfo > MediaClientHandler::getDeviceConnectionsInfo(const MediaPlayerDeviceConnectionsStatus& info) const
{
   ::std::vector< DeviceConnectionInfo > tInfo;
   const ::MPlay_fi_types::T_MPlayDeviceInfo& tDeviceInfo = info.getODeviceInfo();
   for (::MPlay_fi_types::T_MPlayDeviceInfo::const_iterator itr = tDeviceInfo.begin(); (itr != tDeviceInfo.end()); ++itr)
   {
      if ((*itr).getBDeviceConnected())
      {
         DeviceConnectionInfo tDeviceConnectionInfo;
         tDeviceConnectionInfo.setDeviceId((*itr).getU8DeviceTag());
         tDeviceConnectionInfo.setDeviceName((*itr).getSDeviceName());
         switch ((*itr).getE8DeviceType())
         {
            case ::MPlay_fi_types::T_e8_MPlayDeviceType__e8DTY_USB:
            case ::MPlay_fi_types::T_e8_MPlayDeviceType__e8DTY_MTP:
            {
               tDeviceConnectionInfo.setDeviceType("USB1");
               ::std::size_t tPos = (*itr).getSDeviceSysPath().find_last_of(".");
               if (tPos != ::std::string::npos)
               {
                  ::std::string tPath = (*itr).getSDeviceSysPath();
                  int32 tNumber       = atoi(tPath.substr(tPos + 1).c_str());
                  if ((tNumber > 0) && (!((tNumber % 0x02) == 0)))
                  {
                     tDeviceConnectionInfo.setDeviceType("USB2");
                  }
               }
               break;
            }
            case ::MPlay_fi_types::T_e8_MPlayDeviceType__e8DTY_IPOD:
            case ::MPlay_fi_types::T_e8_MPlayDeviceType__e8DTY_IPHONE:
            {
               tDeviceConnectionInfo.setDeviceType("IPOD");
               break;
            }
            default:
               break;
         }
         if (!tDeviceConnectionInfo.getDeviceType().empty())
         {
            bool isAdded = false;
            for (::std::vector< DeviceConnectionInfo >::const_iterator itr = tInfo.begin(); (itr != tInfo.end()); ++itr)
            {
               if (tDeviceConnectionInfo.getDeviceType() == (*itr).getDeviceType())
               {
                  isAdded = true;
                  break;
               }
            }
            if (!isAdded)
            {
               tInfo.push_back(tDeviceConnectionInfo);
            }
         }
      }
   }
   return tInfo;
}


}// namespace Core
}// namespace App
