/**************************************************************************************
 * @file         : KeyRouterFactory.cpp
 * @author       : INF4CV - AppHmi_Master Team
 * @addtogroup   : AppHmi_Master
 * @brief        : KeyRouterFactory is to provide factory impl for KeyRouter
 *                 (Factory & SingleTon Design Pattern)
 * @copyright    : (c) 2019-2019 Robert Bosch Car Multimedia GmbH
 *                 The reproduction, distribution and utilization of this file as
 *                 well as the communication of its contents to others without express
 *                 authorization is prohibited. Offenders will be held liable for the
 *                 payment of damages. All rights reserved in the event of the grant
 *                 of a patent, utility model or design.
 **************************************************************************************/


#include <hmibase/hall_std_if.h>
#include "KeyRouterFactory.h"
#include "KeyRouter.h"
#include "IKeyRouterStubInterface.h"
#include "IKeyRouterUserInterface.h"
#include <App/Core/HmiAppCtrl/IHmiAppCtrlAppInfoInterface.h>


#ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
#define ETG_DEFAULT_TRACE_CLASS           TR_CLASS_APPHMI_MASTER_MAIN
#define ETG_I_TRACE_CHANNEL               TR_TTFIS_APPHMI_MASTER
#define ETG_I_TTFIS_CMD_PREFIX            "HMI_MASTER_"
#define ETG_I_FILE_PREFIX                 App::Core::KeyRouterFactory::
#include "trcGenProj/Header/KeyRouterFactory.cpp.trc.h"
#endif // VARIANT_S_FTR_ENABLE_TRC_GEN


using namespace ::keyRouter;
using namespace ::hmi::apphmi_master;


namespace keyRouterFactory {


KeyRouterFactory* KeyRouterFactory::_keyRouterFactory = NULL;


KeyRouterFactory::KeyRouterFactory()
{ }


KeyRouterFactory::~KeyRouterFactory()
{
   clearKeyRouters();
}


void KeyRouterFactory::createInstance()
{
   if (NULL == _keyRouterFactory)
   {
      _keyRouterFactory = new KeyRouterFactory();
   }
}


KeyRouterFactory* KeyRouterFactory::deleteInstance()
{
   if (NULL != _keyRouterFactory)
   {
      delete _keyRouterFactory;
      _keyRouterFactory = NULL;
   }
   return _keyRouterFactory;
}


KeyRouterFactory* KeyRouterFactory::getInstance()
{
   createInstance();
   return _keyRouterFactory;
}


void KeyRouterFactory::createKeyRouters(const uint32 routersCount)
{
   for (uint32 id = 0; (id < routersCount); ++id)
   {
      _keyRouters.insert(::std::make_pair< uint32, IKeyRouterInterface* >(id, new KeyRouter()));
      IKeyRouterInterface* imp = getKeyRouterOfId(id);
      if (NULL != imp)
      {
         imp->setDisplayId(id);
         imp->setKeyRouterId(id);
      }
      else
      {
         ETG_TRACE_ERR(("KeyRouterFactory: Invalid Key Router Id : %d", id));
      }
   }
}


void KeyRouterFactory::createKeyRouters(const ::std::vector< uint32 >& routersInfo)
{
   uint32 id = 0xFFFF;
   ::std::vector< uint32 >::const_iterator routersInfoItr = routersInfo.begin();
   for (; (routersInfoItr != routersInfo.end()); ++routersInfoItr)
   {
      id = ::std::distance(routersInfo.begin(), routersInfoItr);
      _keyRouters.insert(::std::make_pair< uint32, IKeyRouterInterface* >(id, new KeyRouter()));
      IKeyRouterInterface* imp = getKeyRouterOfId(id);
      if (NULL != imp)
      {
         imp->setDisplayId((*routersInfoItr));
         imp->setKeyRouterId(id);
      }
      else
      {
         ETG_TRACE_ERR(("KeyRouterFactory: Invalid Key Router Id : %d", (*routersInfoItr)));
      }
   }
}


void KeyRouterFactory::clearKeyRouters()
{
   if (!_keyRouters.empty()) //for safety
   {
      ::std::map< uint32, IKeyRouterInterface* >::iterator itr = _keyRouters.begin();
      for (; (itr != _keyRouters.end()); ++itr)
      {
         if (NULL != itr->second)
         {
            delete(itr->second);
            itr->second = NULL;
         }
      }
      _keyRouters.clear();
   }
}


void KeyRouterFactory::setKeyRouterStubImpl(IKeyRouterStubInterface* imp)
{
   ::std::map< uint32, IKeyRouterInterface* >::iterator itr = _keyRouters.begin();
   for (; (itr != _keyRouters.end()); ++itr)
   {
      if (NULL != itr->second) //for safety
      {
         itr->second->setKeyRouterStubImpl(imp);
      }
   }
}


void KeyRouterFactory::setHmiAppCtrlAppInfoImpl(IHmiAppCtrlAppInfoInterface* imp)
{
   ::std::map< uint32, IKeyRouterInterface* >::iterator itr = _keyRouters.begin();
   for (; (itr != _keyRouters.end()); ++itr)
   {
      if (NULL != itr->second) //for safety
      {
         itr->second->setHmiAppCtrlAppInfoImpl(imp);
      }
   }
}


void KeyRouterFactory::setKeyRouterUserInterfaceImpl(::keyRouter::IKeyRouterUserInterface* imp)
{
   ::std::map< uint32, IKeyRouterInterface* >::iterator itr = _keyRouters.begin();
   for (; (itr != _keyRouters.end()); ++itr)
   {
      if (NULL != itr->second) //for safety
      {
         itr->second->setKeyRouterUserInterfaceImpl(imp);
      }
   }
}


::std::vector< ::keyRouter::IKeyRouterInterface* > KeyRouterFactory::getAllKeyRoutersInfo() const
{
   ::std::vector< IKeyRouterInterface* > keyRoutersInfo;
   ::std::map< uint32, IKeyRouterInterface* >::const_iterator itr = _keyRouters.begin();
   for (; (itr != _keyRouters.end()); ++itr)
   {
      keyRoutersInfo.push_back(itr->second);
   }
   return keyRoutersInfo;
}


IKeyRouterInterface* KeyRouterFactory::getKeyRouterOfId(const uint32 id) const
{
   IKeyRouterInterface* imp = NULL;
   ::std::map< uint32, IKeyRouterInterface* >::const_iterator itr = _keyRouters.find(id);
   if (itr != _keyRouters.end())
   {
      imp = itr->second;
   }
   return imp;
}


IKeyRouterInterface* KeyRouterFactory::getKeyRouterOfDisplayId(const uint32 id) const
{
   IKeyRouterInterface* imp = NULL;
   ::std::map< uint32, IKeyRouterInterface* >::const_iterator itr = _keyRouters.begin();
   for (; (itr != _keyRouters.end()); ++itr)
   {
      if ((NULL != itr->second) && (itr->second->getDisplayId() == id))
      {
         imp = itr->second;
         break;
      }
   }
   return imp;
}


} //namespace keyRouterFactory
