/**************************************************************************************
 * @file         : KeyRouter.h
 * @author       : INF4CV - AppHmi_Master Team
 * @addtogroup   : AppHmi_Master
 * @brief        : KeyRouter is to handle key forwarding impl
 *                 (HardKey & Encoder)
 * @copyright    : (c) 2019-2019 Robert Bosch Car Multimedia GmbH
 *                 The reproduction, distribution and utilization of this file as
 *                 well as the communication of its contents to others without express
 *                 authorization is prohibited. Offenders will be held liable for the
 *                 payment of damages. All rights reserved in the event of the grant
 *                 of a patent, utility model or design.
 **************************************************************************************/


#ifndef KEY_ROUTER_H
#define KEY_ROUTER_H


#include "IKeyRouterInterface.h"
#include <vector>
#include <boost/shared_ptr.hpp>


/**
 *  Classes Declaration
 */

namespace keyRouter {


/**
 * KeyRouter is used to
 *  - Provide handler control for key forwarding main implementations
 *  - Info: Handler control is based on display configurations
 */
class KeyRouter : public IKeyRouterInterface
{
   public:
      /**
       *  Member Function Declaration
       */
      KeyRouter();
      virtual ~KeyRouter();
      uint32 getDisplayId() const;
      void setDisplayId(const uint32 id);
      void setKeyRouterId(const uint32 id);
      void setKeyRouterStubImpl(IKeyRouterStubInterface* imp);
      void setKeyRouterUserInterfaceImpl(IKeyRouterUserInterface* imp);
      void setHmiAppCtrlAppInfoImpl(::hmi::apphmi_master::IHmiAppCtrlAppInfoInterface* imp);
      void onHardKeyForwardingReq(const ::keyRouterTypes::KeyInfo< uint32 >& request);
      void onEncoderForwardingReq(const ::keyRouterTypes::KeyInfo< int32 >& request);

   private:
      /**
       * Typedef declaration
       */
      typedef ::std::pair< ::boost::shared_ptr< ::keyRouterTypes::IKeyInfo >,  ::std::vector< int > > LastKeyInfoType;
      /**
       *  Member Function Declaration
       */
      void initLastKeyInfo();
      void clearLastKeyInfo();
      void printErrorInfo(const ::std::string func = "", const ::std::string info = "") const;
      ::std::vector< int > getLastKeyProcessedAppInfo(const enKeyType type) const;
      template< typename CodeType, typename exCodeType >
      ::keyRouterTypes::KeyInfo< exCodeType > convertKeyInfo(const ::keyRouterTypes::KeyInfo< CodeType >& info) const;
      template< typename CodeType >
      ::keyRouterTypes::KeyInfo< CodeType >* getLastProcessedKeyInfo(const enKeyType type) const;
      template< typename CodeType >
      bool isKeyValid(const enKeyType type, const ::keyRouterTypes::KeyInfo< CodeType >& info) const;
      template< typename CodeType >
      bool isValidReceiver(const enKeyType type, const ::keyRouterTypes::KeyInfo< CodeType >& info, const int pid) const;
      template< typename CodeType >
      bool isKeyProcessedForReceiver(const enKeyType type, const ::keyRouterTypes::KeyInfo< CodeType >& info, const int pid) const;
      template< typename CodeType >
      bool isKeySecondDataSameAsLast(const enKeyType type, const ::keyRouterTypes::KeyInfo< CodeType >& current, const ::keyRouterTypes::KeyInfo< CodeType >& last) const;
      template< typename CodeType>
      int getRegisteredApp(const enKeyType type, const ::keyRouterTypes::KeyInfo< CodeType >& info) const;
      template< typename CodeType >
      int getNextReceiverPidForKey(const enKeyType type, const ::keyRouterTypes::KeyInfo< CodeType >& info);
      template< typename CodeType >
      void onKeyForward(const enKeyType type, const ::keyRouterTypes::KeyInfo< CodeType >& info);
      template< typename CodeType >
      void sendForwardingSig(const enKeyType type, const ::keyRouterTypes::KeyInfo< CodeType >& info);
      template< typename CodeType >
      void sendInfoToBaseApp(const enKeyType type, const ::keyRouterTypes::KeyInfo< CodeType >& info);
      template< typename CodeType >
      void storeLastProcessedKeyInfo(const enKeyType type, const ::keyRouterTypes::KeyInfo< CodeType >& info);
      template< typename CodeType >
      void storeLastKeyProcessedAppInfo(const bool clearLastInfoDirectly, const enKeyType type, const ::keyRouterTypes::KeyInfo< CodeType >& info);
      template< typename CodeType >
      void printKeyInfo(const ::std::string function, const enKeyType keyType, const ::keyRouterTypes::KeyInfo< CodeType >& info);
      template< typename CodeType >
      void printKeySpecificInfo(const ::std::string function, const enKeyType keyType, const ::keyRouterTypes::KeyInfo< CodeType >& info);
      bool isAppPid(const int pid) const;
      int getPidOfApp(const int appId) const;
      ::std::string getAppNameOfPid(const int pid) const;
      int getNextVisibleSurfacePid(const ::std::vector< int >& excludePidList, int pid);
      int getNextVisibleSurfacePidFromVisibleSurfaceList(const ::std::vector< int >& visibleSurfaceList, int pid) const;
      void removePidFromVisibleSurfaceList(const ::std::vector< int >& excludePidList,  ::std::vector< int >& visibleSurfaceList);
      void clearLastKeyProcessedAppInfo(const enKeyType type);
      void addToLastKeyProcessedApp(const enKeyType type, const int pid);
      void removeFromLastKeyProcessedApp(const enKeyType type, const int pid);
      /**
       *  Member Variables Declaration
       */
      uint32 _displayId;
      uint32 _keyRouterId;
      IKeyRouterStubInterface* _keyRouterStub;
      IKeyRouterUserInterface* _keyRouterUserInterface;
      ::hmi::apphmi_master::IHmiAppCtrlAppInfoInterface* _appInfoImpl;
      ::std::map< enKeyType, LastKeyInfoType > _lastKeyInfo;
};


} // namespace keyRouter


#endif /* KEY_ROUTER_H */
