/**************************************************************************************
 * @file         : KeyRouter.cpp
 * @author       : INF4CV - AppHmi_Master Team
 * @addtogroup   : AppHmi_Master
 * @brief        : KeyRouter is to handle key forwarding impl
 *                 (HardKey & Encoder)
 * @copyright    : (c) 2019-2020 Robert Bosch Car Multimedia GmbH
 *                 The reproduction, distribution and utilization of this file as
 *                 well as the communication of its contents to others without express
 *                 authorization is prohibited. Offenders will be held liable for the
 *                 payment of damages. All rights reserved in the event of the grant
 *                 of a patent, utility model or design.
 **************************************************************************************/


#include <hmibase/hall_std_if.h>
#include "KeyRouter.h"
#include "IKeyRouterStubInterface.h"
#include "IKeyRouterUserInterface.h"
#include <App/Core/IlmClient/IlmClient.h>
#include <App/Core/HmiAppCtrl/IHmiAppCtrlAppInfoInterface.h>


#ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
#define ETG_DEFAULT_TRACE_CLASS           TR_CLASS_APPHMI_MASTER_MAIN
#define ETG_I_TRACE_CHANNEL               TR_TTFIS_APPHMI_MASTER
#define ETG_I_TTFIS_CMD_PREFIX            "HMI_MASTER_"
#define ETG_I_FILE_PREFIX                 App::Core::KeyRouter::
#include "trcGenProj/Header/KeyRouter.cpp.trc.h"
#endif // VARIANT_S_FTR_ENABLE_TRC_GEN


#define VALID_PTR(_ptr_, _func_)                     \
      if(NULL == _ptr_) {                            \
         printErrorInfo(_func_, "Invalid Instance"); \
         return;                                     \
      }


#define VALID_PTR_WITH_RETURN(_ptr_, _func_, _rs_)   \
      if(NULL == _ptr_) {                            \
         printErrorInfo(_func_, "Invalid Instance"); \
         return _rs_;                                \
      }

using namespace ::hmibase;
using namespace ::hmi::apphmi_master;
using namespace ::keyRouterTypes;


namespace keyRouter {


KeyRouter::KeyRouter() : _displayId(0), _keyRouterId(0)
   , _keyRouterStub(NULL), _appInfoImpl(NULL), _keyRouterUserInterface(NULL)
{
   initLastKeyInfo();
}


KeyRouter::~KeyRouter()
{
   _displayId              = 0;
   _keyRouterId            = 0;
   _appInfoImpl            = NULL;
   _keyRouterStub          = NULL;
   _keyRouterUserInterface = NULL;
   clearLastKeyInfo();
}


void KeyRouter::setDisplayId(const uint32 id)
{
   _displayId = id;
}


uint32 KeyRouter::getDisplayId() const
{
   return _displayId;
}


void KeyRouter::setKeyRouterId(const uint32 id)
{
   _keyRouterId = id;
}


void KeyRouter::setKeyRouterStubImpl(IKeyRouterStubInterface* imp)
{
   _keyRouterStub = imp;
}


void KeyRouter::setKeyRouterUserInterfaceImpl(IKeyRouterUserInterface* imp)
{
   _keyRouterUserInterface = imp;
}


void KeyRouter::setHmiAppCtrlAppInfoImpl(IHmiAppCtrlAppInfoInterface* imp)
{
   _appInfoImpl = imp;
}


void KeyRouter::printErrorInfo(const ::std::string func, const ::std::string info) const
{
   ::std::string displayInfo = func + info;
   ETG_TRACE_ERR(("KEYROUTER: ERROR: %200s: KEYROUTERID : KeyRouter_%d : DISPLAYID : Display_%d", displayInfo.c_str(), _keyRouterId, _displayId));
}


void KeyRouter::initLastKeyInfo()
{
   LastKeyInfoType obj;
   obj.second.clear();
   obj.first = ::boost::shared_ptr< IKeyInfo >(new KeyInfo< uint32 >(0, _displayId, HARDKEYCODE_HK_KEY_CODE_INVALID, HARDKEYSTATE_UNKNOWN, 0, 0, 0, 0));
   _lastKeyInfo.insert(::std::pair< enKeyType, LastKeyInfoType >(HARDKEY_TYPE, obj));
   obj.first = ::boost::shared_ptr< IKeyInfo >(new KeyInfo< int32 >(0, _displayId, ENCCODE_ENCODER_CODE_INVALID, 0, 0, 0, 0, 0));
   _lastKeyInfo.insert(::std::pair< enKeyType, LastKeyInfoType >(ENCODER_TYPE, obj));
}


void KeyRouter::clearLastKeyInfo()
{
   ::std::map< enKeyType, LastKeyInfoType >::iterator itr = _lastKeyInfo.begin();
   for (; (itr != _lastKeyInfo.end()); ++itr)
   {
      itr->second.first.reset();
      itr->second.second.clear();
   }
   _lastKeyInfo.clear();
}


void KeyRouter::onHardKeyForwardingReq(const KeyInfo< uint32 >& request)
{
   printKeyInfo< uint32 >("onHardKeyForwardingReq: <-", HARDKEY_TYPE, request);
   onKeyForward< uint32 >(HARDKEY_TYPE, request);
}


void KeyRouter::onEncoderForwardingReq(const KeyInfo< int32 >& request)
{
   printKeyInfo< int32 >("onEncoderForwardingReq: <-", ENCODER_TYPE, request);
   onKeyForward< int32 >(ENCODER_TYPE, request);
}


template< typename CodeType >
void KeyRouter::onKeyForward(const enKeyType type, const KeyInfo< CodeType >& info)
{
   KeyInfo< CodeType > obj = info;
   if (isKeyValid< CodeType >(type, info))
   {
      int receiverPid = getNextReceiverPidForKey< CodeType >(type, info);
      if (receiverPid != 0)
      {
         obj.setPidReceiver(receiverPid);
         obj.setUserData(obj.getUserData() + 1);
         sendForwardingSig< CodeType >(type, obj);
         storeLastKeyProcessedAppInfo< CodeType >(((info.getUserData() == 0) ? true : false), type, obj);
         storeLastProcessedKeyInfo< CodeType >(type, obj);
      }
      else
      {
         receiverPid = getPidOfApp(APPID_APPHMI_MASTER);
         if (isValidReceiver< CodeType >(type, info, receiverPid))
         {
            obj.setPidReceiver(receiverPid);
            obj.setUserData(obj.getUserData() + 1);
            sendInfoToBaseApp< CodeType >(type, obj);
            storeLastKeyProcessedAppInfo< CodeType >(((info.getUserData() == 0) ? true : false), type, obj);
            storeLastProcessedKeyInfo< CodeType >(type, obj);
         }
         else
         {
            ETG_TRACE_USR4(("KEYROUTER: onKeyForward: KEYROUTERID : KeyRouter_%d : DISPLAYID : Display_%d", _keyRouterId, _displayId));
            printKeySpecificInfo< CodeType >("onKeyForward: No Further Receiver", type, info);
         }
      }
   }
   else
   {
      ETG_TRACE_USR4(("KEYROUTER: onKeyForward: KEYROUTERID : KeyRouter_%d : DISPLAYID : Display_%d", _keyRouterId, _displayId));
      printKeySpecificInfo< CodeType >("onKeyForward: Invalid key", type, info);
   }
}


template< typename CodeType >
void KeyRouter::sendForwardingSig(const enKeyType type, const KeyInfo< CodeType >& info)
{
   VALID_PTR(_keyRouterStub, "sendForwardingSig: IKeyRouterStubInterface*: ");
   switch (type)
   {
      case HARDKEY_TYPE:
      {
         printKeyInfo< CodeType >("sendForwardingSig: ->", type, info);
         KeyInfo< uint32 > hardKeyInfo = convertKeyInfo< CodeType, uint32 >(info);
         _keyRouterStub->sendHardKeyInfoForwardingSig(hardKeyInfo);
      }
      break;
      case ENCODER_TYPE:
      {
         printKeyInfo< CodeType >("sendForwardingSig: ->", type, info);
         KeyInfo< int32 > encoderInfo = convertKeyInfo< CodeType, int32 >(info);
         _keyRouterStub->sendEncoderInfoForwardingSig(encoderInfo);
      }
      break;
      default:
         break;
   }
}


template< typename CodeType >
void KeyRouter::sendInfoToBaseApp(const enKeyType type, const KeyInfo< CodeType >& info)
{
   VALID_PTR(_keyRouterStub, "sendInfoToBaseApp: IKeyRouterStubInterface*: ");
   switch (type)
   {
      case HARDKEY_TYPE:
      {
         printKeyInfo< CodeType >("sendInfoToBaseApp: ->", type, info);
         KeyInfo< uint32 > hardKeyInfo = convertKeyInfo< CodeType, uint32 >(info);
         _keyRouterStub->sendHardKeyInfoToBaseApp(hardKeyInfo);
      }
      break;
      case ENCODER_TYPE:
      {
         printKeyInfo< CodeType >("sendInfoToBaseApp: ->", type, info);
         KeyInfo< int32 > encoderInfo = convertKeyInfo< CodeType, int32 >(info);
         _keyRouterStub->sendEncoderInfoToBaseApp(encoderInfo);
      }
      break;
      default:
         break;
   }
}


template< typename CodeType >
void KeyRouter::storeLastKeyProcessedAppInfo(const bool clearLastInfoDirectly, const enKeyType type, const KeyInfo< CodeType >& info)
{
   KeyInfo< CodeType >* lastKeyInfo = getLastProcessedKeyInfo< CodeType >(type);
   VALID_PTR(lastKeyInfo, "storeLastKeyProcessedAppInfo: KeyInfo*: ");
   if ((clearLastInfoDirectly == true) || (info.getCode() != lastKeyInfo->getCode()) || (!isKeySecondDataSameAsLast< CodeType >(type, info, *lastKeyInfo)))
   {
      clearLastKeyProcessedAppInfo(type);
   }
   //addToLastKeyProcessedApp(type, info.getPidSender());
   addToLastKeyProcessedApp(type, info.getPidReceiver());
}


template< typename CodeType >
void KeyRouter::storeLastProcessedKeyInfo(const enKeyType type, const KeyInfo< CodeType >& info)
{
   ::std::map< enKeyType, LastKeyInfoType >::iterator itr = _lastKeyInfo.find(type);
   if (itr != _lastKeyInfo.end()) //for safety
   {
      itr->second.first = ::boost::shared_ptr< IKeyInfo >(new KeyInfo< CodeType >(info));
   }
}


void KeyRouter::addToLastKeyProcessedApp(const enKeyType type, const int pid)
{
   removeFromLastKeyProcessedApp(type, pid);
   ::std::map< enKeyType, LastKeyInfoType >::iterator itr = _lastKeyInfo.find(type);
   if (itr != _lastKeyInfo.end()) //for safety
   {
      itr->second.second.push_back(pid);
   }
}


void KeyRouter::removeFromLastKeyProcessedApp(const enKeyType type, const int pid)
{
   ::std::map< enKeyType, LastKeyInfoType >::iterator itr = _lastKeyInfo.find(type);
   if (itr != _lastKeyInfo.end()) //for safety
   {
      ::std::vector< int >::iterator lastKeyProcessedAppItr = ::std::find(itr->second.second.begin(), itr->second.second.end(), pid);
      if (lastKeyProcessedAppItr != itr->second.second.end())
      {
         itr->second.second.erase(lastKeyProcessedAppItr);
      }
   }
}


void KeyRouter::clearLastKeyProcessedAppInfo(const enKeyType type)
{
   ::std::map< enKeyType, LastKeyInfoType >::iterator itr = _lastKeyInfo.find(type);
   if (itr != _lastKeyInfo.end()) //for safety
   {
      itr->second.second.clear();
   }
}


void KeyRouter::removePidFromVisibleSurfaceList(const ::std::vector< int >& excludePidList,  ::std::vector< int >& visibleSurfaceList)
{
   for (::std::vector< int >::const_iterator itr = excludePidList.begin(); (itr != excludePidList.end()); ++itr)
   {
      ::std::vector< int >::iterator visibleSurfaceItr = ::std::find(visibleSurfaceList.begin(), visibleSurfaceList.end(), (*itr));
      if (visibleSurfaceItr != visibleSurfaceList.end())
      {
         (void)visibleSurfaceList.erase(visibleSurfaceItr);
      }
   }
}


bool KeyRouter::isAppPid(const int pid) const
{
   bool isValid = false;
   VALID_PTR_WITH_RETURN(_appInfoImpl, "isAppPid: IHmiAppCtrlAppInfoInterface*: ", isValid);
   if (_appInfoImpl->isAppPid(pid))
   {
      isValid = true;
   }
   return isValid;
}


template< typename CodeType >
bool KeyRouter::isKeyValid(const enKeyType type, const KeyInfo< CodeType >& info) const
{
   //Note:
   // -> This validation check will may blog some inputs routing because of corresponding inputs new entry.
   //        (i.e. If input is forwarded to the respective app and meantime new input is received for same guy
   //              then the first input may not be forwarded to further users. It will concentrate only on new input [Especially for encoder]).
   // -> Currently no solution is for the above issue and Its considered as known limitation.
   bool isValid = true;
   KeyInfo< CodeType >* lastKeyInfo = getLastProcessedKeyInfo< CodeType >(type);
   VALID_PTR_WITH_RETURN(lastKeyInfo, "isKeyValid: KeyInfo*: ", isValid);
   switch (type)
   {
      case HARDKEY_TYPE:
      {
         if ((info.getState() != HARDKEYSTATE_UNKNOWN) && ((info.getUserData() > lastKeyInfo->getUserData()) || \
               (((info.getCode() != lastKeyInfo->getCode()) || (info.getState() != lastKeyInfo->getState())) && (lastKeyInfo->getUserData() != 0) && (lastKeyInfo->getUserData() <= info.getUserData()))))
         {
            isValid = false;
         }
      }
      break;
      case ENCODER_TYPE:
      {
         //Info: Direction: 1 - positive 0 - negative
         bool newKeyDirection  = (info.getValue() > 0) ? 1 : 0;
         bool lastKeyDirection = (lastKeyInfo->getValue() > 0) ? 1 : 0;
         if ((info.getUserData() > lastKeyInfo->getUserData()) || ((info.getCode() != lastKeyInfo->getCode()) && \
               (lastKeyDirection != newKeyDirection) && (lastKeyInfo->getUserData() != 0) && (lastKeyInfo->getUserData() <= info.getUserData())))
         {
            isValid = false;
         }
      }
      break;
      default:
         isValid = false;
         break;
   }
   return isValid;
}


template< typename CodeType >
bool KeyRouter::isValidReceiver(const enKeyType type, const KeyInfo< CodeType >& info, const int pid) const
{
   bool isValid = false;
   if ((pid) && (pid != static_cast<int>(info.getPidSender())) && (isAppPid(pid)) && (!(isKeyProcessedForReceiver< CodeType >(type, info, pid))))
   {
      isValid = true;
   }
   return isValid;
}


template< typename CodeType >
bool KeyRouter::isKeyProcessedForReceiver(const enKeyType type, const KeyInfo< CodeType >& info, const int pid) const
{
   bool isProcessed = false;
   KeyInfo< CodeType >* lastKeyInfo  = getLastProcessedKeyInfo< CodeType >(type);
   VALID_PTR_WITH_RETURN(lastKeyInfo, "isKeyProcessedForReceiver: KeyInfo*: ", isProcessed);
   if ((info.getCode() == lastKeyInfo->getCode()) && (isKeySecondDataSameAsLast< CodeType >(type, info, *lastKeyInfo)) && (info.getUserData() == lastKeyInfo->getUserData()))
   {
      ::std::vector< int > lastKeyProcessedAppInfo = getLastKeyProcessedAppInfo(type);
      ::std::vector< int >::const_iterator itr     = ::std::find(lastKeyProcessedAppInfo.begin(), lastKeyProcessedAppInfo.end(), pid);
      if (itr != lastKeyProcessedAppInfo.end())
      {
         isProcessed = true;
      }
   }
   return isProcessed;
}


template< typename CodeType >
bool KeyRouter::isKeySecondDataSameAsLast(const enKeyType type, const KeyInfo< CodeType >& current, const KeyInfo< CodeType >& last) const
{
   bool isSame = false;
   switch (type)
   {
      case HARDKEY_TYPE:
         isSame = (current.getState() == last.getState()) ? true : false;
         break;
      case ENCODER_TYPE:
         isSame = (current.getValue() == last.getValue()) ? true : false;
         break;
      default:
         break;
   }
   return isSame;
}


::std::string KeyRouter::getAppNameOfPid(const int pid) const
{
   ::std::string retApp = "";
   VALID_PTR_WITH_RETURN(_appInfoImpl, "getAppNameOfPid: IHmiAppCtrlAppInfoInterface*: ", retApp);
   if (pid != 0)
   {
      retApp = _appInfoImpl->getApplicationNameForPid(pid);
   }
   return retApp;
}


::std::vector< int > KeyRouter::getLastKeyProcessedAppInfo(const enKeyType type) const
{
   ::std::vector< int > info;
   ::std::map< enKeyType, LastKeyInfoType >::const_iterator itr = _lastKeyInfo.find(type);
   if (itr != _lastKeyInfo.end()) //for safety
   {
      info = (itr->second.second);
   }
   return info;
}


int KeyRouter::getPidOfApp(const int appId) const
{
   int retPid = 0;
   VALID_PTR_WITH_RETURN(_appInfoImpl, "getPidOfApp: IHmiAppCtrlAppInfoInterface*: ", retPid);
   retPid = _appInfoImpl->getPidOfApp(appId);
   return retPid;
}


int KeyRouter::getNextVisibleSurfacePid(const ::std::vector< int >& excludePidList, int pid)
{
   int retPid = 0;
   //Note: Multi-Component access (Btw HALL & GUI) on IlmClient
   //TODO: Thread safety (mutex lock) may required in IlmClient if GUI & HALL component msg queue are separated -sve2cob
   IlmClient* imp = IlmClient::getInstance();
   VALID_PTR_WITH_RETURN(imp, "getNextVisibleSurfacePid: IlmClient*: ", retPid);
   std::vector< int > surfacePidVector;
   imp->getVisibleSurfaceList(surfacePidVector, _displayId, VISIBLE_INPUT_KEY, CREATOR_PID);
   removePidFromVisibleSurfaceList(excludePidList, surfacePidVector);
   retPid = getNextVisibleSurfacePidFromVisibleSurfaceList(surfacePidVector, pid);
   return retPid;
}


int KeyRouter::getNextVisibleSurfacePidFromVisibleSurfaceList(const ::std::vector< int >& visibleSurfaceList, int pid) const
{
   int retPid = 0;
   std::vector< int >::const_iterator itr = ::std::find(visibleSurfaceList.begin(), visibleSurfaceList.end(), pid);
   if (itr != visibleSurfaceList.end() && itr != visibleSurfaceList.begin())
   {
      retPid = *(itr - 1);
   }
   else if (!visibleSurfaceList.empty() && itr == visibleSurfaceList.end())
   {
      retPid = *(visibleSurfaceList.end() - 1);
   }
   else
   {
      retPid = 0;
   }
   return retPid;
}


template< typename CodeType >
int KeyRouter::getNextReceiverPidForKey(const enKeyType type, const KeyInfo< CodeType >& info)
{
   int receiverPid = 0;
   int registerAppPid = getRegisteredApp< CodeType >(type, info);
   if (isValidReceiver< CodeType >(type, info, registerAppPid))
   {
      receiverPid = registerAppPid;
   }
   else
   {
      ::std::vector< int > excludePidList;
      excludePidList.push_back(registerAppPid);
      int visibleSurfacePid = info.getPidSender();
      while (visibleSurfacePid != 0)
      {
         visibleSurfacePid = getNextVisibleSurfacePid(excludePidList, visibleSurfacePid);
         if (isValidReceiver< CodeType >(type, info, visibleSurfacePid))
         {
            receiverPid = visibleSurfacePid;
            break;
         }
      }
   }
   return receiverPid;
}


template< typename CodeType>
int KeyRouter::getRegisteredApp(const enKeyType type, const KeyInfo< CodeType >& info)  const
{
   int registerAppPid = 0;
   VALID_PTR_WITH_RETURN(_keyRouterUserInterface, "getRegisteredApp: IKeyRouterUserInterface*: ", registerAppPid);

   switch (type)
   {
      case HARDKEY_TYPE:
      {
         KeyInfo< uint32 > hardKeyInfo = convertKeyInfo< CodeType, uint32 >(info);
         registerAppPid = getPidOfApp(_keyRouterUserInterface->getHardKeyRegisteredApp(hardKeyInfo));
      }
      break;
      case ENCODER_TYPE:
      {
         KeyInfo< int32 > encoderKeyInfo = convertKeyInfo< CodeType, int32 >(info);
         registerAppPid = getPidOfApp(_keyRouterUserInterface->getEncoderRegisteredApp(encoderKeyInfo));
      }
      break;
      default:
         break;
   }
   return registerAppPid;
}


template< typename CodeType >
KeyInfo< CodeType >* KeyRouter::getLastProcessedKeyInfo(const enKeyType type) const
{
   KeyInfo< CodeType >* imp = NULL;
   ::std::map< enKeyType, LastKeyInfoType >::const_iterator itr = _lastKeyInfo.find(type);
   if (itr != _lastKeyInfo.end()) //for safety
   {
      imp = dynamic_cast< KeyInfo< CodeType >* >(itr->second.first.get());
   }
   return imp;
}


template< typename CodeType, typename exCodeType >
KeyInfo< exCodeType > KeyRouter::convertKeyInfo(const KeyInfo< CodeType >& info) const
{
   KeyInfo< exCodeType > keyInfo;
   keyInfo.setActId(info.getActId());
   keyInfo.setDisplayId(info.getDisplayId());
   keyInfo.setCode(info.getCode());
   keyInfo.setState(info.getState());
   keyInfo.setValue(info.getValue());
   keyInfo.setUserData(info.getUserData());
   keyInfo.setPidSender(info.getPidSender());
   keyInfo.setPidReceiver(info.getPidReceiver());
   return keyInfo;
}


template< typename CodeType >
void KeyRouter::printKeyInfo(const ::std::string function, const enKeyType keyType, const KeyInfo< CodeType >& info)
{
   ETG_TRACE_USR4(("KEYROUTER: %100s: KEYROUTERID    :  KeyRouter_%d : DISPLAYID : Display_%d", function.c_str(), _keyRouterId, _displayId));
   ETG_TRACE_USR4(("KEYROUTER: %100s: Type           :  %d", function.c_str(), ETG_CENUM(enKeyType, keyType)));
   printKeySpecificInfo< CodeType >(function, keyType, info);
   ETG_TRACE_USR4(("KEYROUTER: %100s: UserData       :  %d", function.c_str(), info.getUserData()));
   ETG_TRACE_USR4(("KEYROUTER: %100s: PidSender      :  %d", function.c_str(), info.getPidSender()));
   ETG_TRACE_USR4(("KEYROUTER: %100s: SenderApp      :  %s", function.c_str(), getAppNameOfPid(info.getPidSender()).c_str()));
   ETG_TRACE_USR4(("KEYROUTER: %100s: PidReceiver    :  %d", function.c_str(), info.getPidReceiver()));
   ETG_TRACE_USR4(("KEYROUTER: %100s: ReceiverApp    :  %s", function.c_str(), getAppNameOfPid(info.getPidReceiver()).c_str()));
}


template< typename CodeType >
void KeyRouter::printKeySpecificInfo(const ::std::string function, const enKeyType keyType, const KeyInfo< CodeType >& info)
{
   switch (keyType)
   {
      case HARDKEY_TYPE:
      {
         ETG_TRACE_USR4(("KEYROUTER: %100s: Code        : %d", function.c_str(), ETG_CENUM(HardKeyCodeEnum, info.getCode())));
         ETG_TRACE_USR4(("KEYROUTER: %100s: State       : %d", function.c_str(), ETG_CENUM(HardKeyStateEnum, info.getState())));
      }
      break;
      case ENCODER_TYPE:
      {
         ETG_TRACE_USR4(("KEYROUTER: %100s: Code        : %d", function.c_str(), ETG_CENUM(EncCodeEnum, info.getCode())));
         ETG_TRACE_USR4(("KEYROUTER: %100s: Value       : %d", function.c_str(), info.getValue()));
      }
      break;
      default:
         break;
   }
}


} //namespace keyRouter


#undef VALID_PTR
#undef VALID_PTR_WITH_RETURN
