/**
 * @file         : HmiStateHandler.cpp
 * @author       : INF4CV - AppHmi_Master Team
 * @addtogroup   : AppHmi_Master
 * @brief        : HmiStateHandler is to handle HALL implementation for
 *                 SPM_CORE_FI service
 * @copyright    : (c) 2017-2020 Robert Bosch Car Multimedia GmbH
 *                 The reproduction, distribution and utilization of this file as
 *                 well as the communication of its contents to others without express
 *                 authorization is prohibited. Offenders will be held liable for the
 *                 payment of damages. All rights reserved in the event of the grant
 *                 of a patent, utility model or design.
 */


#include "hall_std_if.h"
#include "HmiStateHandler.h"
#include "HmiStateHandlerHelper.h"
#include "AppHmi_MasterMessages.h"
#include "lcmdbusclient_main_fi.h"
#include "spm_core_fi_typesConst.h"


using namespace ::SPM_CORE_FI;
using namespace ::spm_core_fi_types;
using namespace ::lcmdbusclient_main_fi;


#ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
#define ETG_DEFAULT_TRACE_CLASS           TR_CLASS_APPHMI_MASTER_HALL
#define ETG_I_TRACE_CHANNEL               TR_TTFIS_APPHMI_MASTER
#define ETG_I_TTFIS_CMD_PREFIX            "APPHMI_MASTER_"
#define ETG_I_FILE_PREFIX                 App::Core::HmiStateHandler::
#include "trcGenProj/Header/HmiStateHandler.cpp.trc.h"
#endif


namespace App {
namespace Core {


HmiStateHandler::HmiStateHandler() : _pnmGrp10Status(false)
   , _hourLogicExpireStatus(false)
   , _lastHMIState(0)
   , _currentHMIState(0)
   , _regIdHmiStateUpdate(0)
   , _regIdSubsystemerrorGetUpdate(0)
   , _regIdSubsystemerrorStatusUpdate(0)
   , _spmProxy(SPM_CORE_FIProxy::createProxy("spmFiPort", *this))
   , _lcmDBusClientProxy(Lcmdbusclient_main_fiProxy::createProxy("lcmdbusclient_main_fiPort", (*this)))
{
   ETG_TRACE_USR1(("HmiStateHandler: CTOR"));
}


HmiStateHandler::~HmiStateHandler()
{
   ETG_TRACE_USR1(("HmiStateHandler: DTOR"));
   _pnmGrp10Status                  = false;
   _lastHMIState                    = 0;
   _currentHMIState                 = 0;
   _regIdHmiStateUpdate             = 0;
   _regIdSubsystemerrorGetUpdate    = 0;
   _regIdSubsystemerrorStatusUpdate = 0;
   _hmiStateListeners.clear();
   _pnmGrp10StatusListeners.clear();
   _hourLogicExpireStatusListeners.clear();
   _subSytemsErrorStatusListeners.clear();
}


void HmiStateHandler::onAvailable(const ::boost::shared_ptr< asf::core::Proxy >& proxy, const asf::core::ServiceStateChange& /* stateChange */)
{
   if (isMySPMProxy(proxy))
   {
      _regIdHmiStateUpdate = _spmProxy->sendHmiStateUpReg(*this);
   }
   if (isMyLCMDBusClientProxy(proxy))
   {
      _regIdSubsystemerrorStatusUpdate = _lcmDBusClientProxy->sendSubsystemerrorStatusUpReg((*this));
   }
}


void HmiStateHandler::onUnavailable(const ::boost::shared_ptr< asf::core::Proxy >& proxy, const asf::core::ServiceStateChange& /* stateChange */)
{
   if (isMySPMProxy(proxy))
   {
      _spmProxy->sendHmiStateRelUpReg(_regIdHmiStateUpdate);
      _regIdHmiStateUpdate = 0;
   }
   if (isMyLCMDBusClientProxy(proxy))
   {
      _lcmDBusClientProxy->sendSubsystemerrorStatusRelUpReg(_regIdSubsystemerrorStatusUpdate);
      _regIdSubsystemerrorGetUpdate    = 0;
      _regIdSubsystemerrorStatusUpdate = 0;
   }
}


void HmiStateHandler::onHmiStateError(const ::boost::shared_ptr< SPM_CORE_FIProxy >& proxy, const boost::shared_ptr< HmiStateError >& error)
{
   if (isMySPMProxy(proxy) && (NULL != error.get()) && (error->getAct() == _regIdHmiStateUpdate))
   {
      ETG_TRACE_ERR(("HmiStateHandler: onHmiStateError"));
   }
}


void HmiStateHandler::onHmiStateStatus(const ::boost::shared_ptr< SPM_CORE_FIProxy >& proxy, const boost::shared_ptr< HmiStateStatus >& status)
{
   ETG_TRACE_USR1(("HmiStateHandler: onHmiStateStatus: <-: State: %d", ETG_CENUM(T_SPM_e32_HMI_STATE, status->getHmiState())));
   if ((NULL != _spmProxy.get()) && (proxy == _spmProxy) && (NULL != status.get()) && (status->getAct() == _regIdHmiStateUpdate))
   {
      onHmiStateStatus(status->getHmiState());
   }
}


void HmiStateHandler::onHmiStateStatus(const uint8 hmiState)
{
   ETG_TRACE_USR1(("HmiStateHandler: onHmiStateChange: <-: State: %d", ETG_CENUM(T_SPM_e32_HMI_STATE, hmiState)));
   if (_currentHMIState != hmiState)
   {
      _lastHMIState    = _currentHMIState;
      _currentHMIState = hmiState;
      updateHMIStateChangeToSM(hmiState);
      for (::std::vector< IHmiStateListenerCB* >::iterator itr = _hmiStateListeners.begin(); (itr != _hmiStateListeners.end()); ++itr)
      {
         if (NULL != (*itr))
         {
            (*itr)->onHmiStateUpdate(hmiState);
         }
      }
   }
}


void HmiStateHandler::updateHMIStateChangeToSM(const uint8 hmiState)
{
   enHmiState hmiStateValue = HMI_STATE_ON;
   switch (hmiState)
   {
      case T_SPM_e32_HMI_STATE__SPM_U32_HMISTATE_SHOW_OFF:
      {
         hmiStateValue = HMI_STATE_OFF;
         break;
      }
      case T_SPM_e32_HMI_STATE__SPM_U32_HMISTATE_SHOW_STANDBY:
      {
         hmiStateValue = HMI_STATE_STANDBY;
         break;
      }
      case T_SPM_e32_HMI_STATE__SPM_U32_HMISTATE_INACTIVE:
      {
         hmiStateValue = HMI_STATE_INACTIVE;
         break;
      }
      case T_SPM_e32_HMI_STATE__SPM_U32_HMISTATE_SHOW_INFOTAINMENT_OFF:
      {
         hmiStateValue = HMI_STATE_INFOTAINMENT_OFF;
         break;
      }
      case T_SPM_e32_HMI_STATE__SPM_U32_HMISTATE_SHOW_GOODBYE:
      {
         hmiStateValue = HMI_STATE_GOODBYE;
         break;
      }
      default:
      {
         break;
      }
   }
   ::Courier::Message* msg = COURIER_MESSAGE_NEW(updateHMIStateChange)(hmiStateValue);
   if (NULL != msg)
   {
      (void)msg->Post();
   }
}


void HmiStateHandler::onSubsystemerrorStatusError(const ::boost::shared_ptr< Lcmdbusclient_main_fiProxy >& proxy, const ::boost::shared_ptr< SubsystemerrorStatusError >& error)
{
   if (isMyLCMDBusClientProxy(proxy) && (NULL != error.get()) && (error->getAct() == _regIdSubsystemerrorStatusUpdate))
   {
      ETG_TRACE_ERR(("HmiStateHandler: onSubsystemerrorStatusError"));
   }
}


void HmiStateHandler::onSubsystemerrorStatusStatus(const ::boost::shared_ptr< Lcmdbusclient_main_fiProxy >& proxy, const ::boost::shared_ptr< SubsystemerrorStatusStatus >& status)
{
   if (isMyLCMDBusClientProxy(proxy) && (NULL != status.get()) && ((status->getAct() == _regIdSubsystemerrorStatusUpdate) || (status->getAct() == _regIdSubsystemerrorGetUpdate)))
   {
      SubSystemsErrorStatusInfo tInfo;
      const ::lcmdbusclient_main_fi_types::TheSubsystemList& tTheSubsystemListInfo = status->getSubsystemerrorconstate();
      for (::lcmdbusclient_main_fi_types::TheSubsystemList::const_iterator itr = tTheSubsystemListInfo.begin(); (itr != tTheSubsystemListInfo.end()); ++itr)
      {
         uint8 category   = convertSystemCategoryFromLCM((*itr).getDevId());
         uint8 systemId   = (*itr).getDevIP();
         uint16 errorInfo = (*itr).getDevError();
         ETG_TRACE_USR1(("HmiStateHandler: onSubsystemerrorStatusStatus: <-: Category: %d, systemID: %d, ErrorInfo: %d", \
                         ETG_CENUM(enSystemCategory, category), systemId, errorInfo));
         ::std::pair< uint8, uint16 > temp(category, errorInfo);
         tInfo.insert(::std::pair< uint8, ::std::pair< uint8, uint16 > >(systemId, temp));
      }
      handleSubSystemsErrorStatus(tInfo);
   }
}


void HmiStateHandler::setPNMGrp10Status(const bool status)
{
   if (_pnmGrp10Status != status)
   {
      _pnmGrp10Status = status;
      for (::std::vector< IPNMGrp10StatusCB* >::const_iterator itr = _pnmGrp10StatusListeners.begin(); (itr != _pnmGrp10StatusListeners.end()); ++itr)
      {
         if (NULL != (*itr))
         {
            (*itr)->onPNMGrp10StatusUpdate(status);
         }
      }
   }
}


void HmiStateHandler::setHourLogicExpireStatus(const bool status)
{
   if (_hourLogicExpireStatus != status)
   {
      _hourLogicExpireStatus = status;
      for (::std::vector< IHourLogicExpireStatusCB* >::const_iterator itr = _hourLogicExpireStatusListeners.begin(); (itr != _hourLogicExpireStatusListeners.end()); ++itr)
      {
         if (NULL != (*itr))
         {
            (*itr)->onHourLogicExpireStatusUpdate(status);
         }
      }
   }
}


void HmiStateHandler::sendRestartRequest()
{
   if (NULL != _spmProxy.get())
   {
      ETG_TRACE_USR1(("HmiStateHandler: sendRestartRequest"));
      ETG_TRACE_FATAL(("!!!!!! APPHMI_MASTER: HKPower: Restart system !!!!!!"));
      ETG_TRACE_ERRMEM(("!!!!!! APPHMI_MASTER: HKPower: Restart system !!!!!!"));
      ::std::string subSystemRestartScript = "/var/opt/bosch/persistent/avdecc/cisinstallation/doipcmcrestart.py";
      struct stat buffer;
      if ((stat(subSystemRestartScript.c_str(), &buffer) == 0) && (S_ISREG(buffer.st_mode) && (buffer.st_size > 0)))
      {
         ::std::string cmd = "python ";
         cmd.append(subSystemRestartScript);
         if (system(cmd.c_str()) != 0)
         {
            ETG_TRACE_ERR(("HmiStateHandler: sendRestartRequest: Sub system restart command failed"));
         }
         else
         {
            ETG_TRACE_USR1(("HmiStateHandler: sendRestartRequest: Sub system restart command success"));
         }
      }
      else
      {
         ETG_TRACE_USR1(("HmiStateHandler: sendRestartRequest: Sub system restart script is not available"));
      }
      _spmProxy->sendRestartSystemStart(*this, T_SPM_e32_RESTART_TYPE__SPM_U32_RESTART_USER_RESET);
   }
}


void HmiStateHandler::sendHKeyPressedRequest()
{
   if (NULL != _spmProxy.get())
   {
      ETG_TRACE_USR4(("HmiStateHandler: sendHKeyPressedRequest"));
      //INFO: Reset the trigger first before firing it as per the SPM guidelines
      _spmProxy->sendOnOffTriggerStart(*this, T_SPM_e32_SubStateType__SPM_U32_SUBSTATE_HKEY_PRESSED, false);
      _spmProxy->sendOnOffTriggerStart(*this, T_SPM_e32_SubStateType__SPM_U32_SUBSTATE_HKEY_PRESSED, true);
   }
}


void HmiStateHandler::sendFirstPicDrawnUpdate()
{
   if (NULL != _spmProxy.get())
   {
      ETG_TRACE_USR4(("HmiStateHandler: sendFirstPicDrawnUpdate"));
      _spmProxy->sendOnOffTriggerStart(*this, T_SPM_e32_SubStateType__SPM_U32_SUBSTATE_FIRST_PIC_DRAWN, true);
   }
}


void HmiStateHandler::sendUserOnRequest(const bool status)
{
   if (NULL != _spmProxy.get())
   {
      ETG_TRACE_USR4(("HmiStateHandler: sendUserOnRequest: ->: Status: %d", status));
      _spmProxy->sendOnOffTriggerStart(*this, T_SPM_e32_SubStateType__SPM_U32_SUBSTATE_USER_ON_REQ, status);
   }
}


void HmiStateHandler::sendSubsystemerrorStatusGetRequest()
{
   if (NULL != _lcmDBusClientProxy.get())
   {
      _regIdSubsystemerrorGetUpdate = _lcmDBusClientProxy->sendSubsystemerrorStatusGet(*this);
   }
}


void HmiStateHandler::sendDiagStatus(const bool diagstatus)
{
   if (NULL != _spmProxy.get())
   {
      _spmProxy->sendOnOffTriggerStart(*this, T_SPM_e32_SubStateType__SPM_U32_SUBSTATE_DIAG_REQUEST, diagstatus);
   }
}


void HmiStateHandler ::sendStandByRequest(const bool status)
{
   if (NULL != _spmProxy.get())
   {
      ETG_TRACE_USR4(("HmiStateHandler: sendStandByRequest: ->: Status: %d", status));
      _spmProxy->sendOnOffTriggerStart(*this, T_SPM_e32_SubStateType__SPM_U32_SUBSTATE_STANDBY_REQ, status);
   }
}


void HmiStateHandler::handleSubSystemsErrorStatus(const SubSystemsErrorStatusInfo& info)
{
   for (SubSystemsErrorStatusInfo::const_iterator itr = info.begin(); (itr != info.end()); ++itr)
   {
      if (itr->second.first == SYSTEM_CATEGORY_CMC)
      {
         uint16 tErrorInfo     = itr->second.second;
         bool tPNMErrorStatus  = (getBitValue(tErrorInfo, 0x0C, 0x03) == 0x02) ? true : false;
         bool tHourLogicStatus = (getBitValue(tErrorInfo, 0x0F, 0x01) == 0) ? false : true;
         setPNMGrp10Status(tPNMErrorStatus);
         setHourLogicExpireStatus(tHourLogicStatus);
         break;
      }
   }
   for (::std::vector< ISubSytemsErrorStatusCB* >::iterator itr = _subSytemsErrorStatusListeners.begin(); (itr != _subSytemsErrorStatusListeners.end()); ++itr)
   {
      if (NULL != (*itr))
      {
         (*itr)->onSubSystemsErrorStatus(info);
      }
   }
}


bool HmiStateHandler::isPowerStateON() const
{
   return (isSPMHMIStateON(_currentHMIState));
}


}// namespace Core
}// namespace App
