/****************************************************************************
* Copyright (c) 2019-2020 Robert Bosch Car Multimedia GmbH
* duplication and disclosure to third parties is prohibited.
*
* FILE                  : HmiInfoServiceServerComponent.h
* COMPONENT Name        : AppHmi_Master
* DESCRIPTION           : DBus client for Application switch intrerface.
* AUTHOR                : Pushpa Talawar
* Date                  : 23.02.2018
* Revision History      : 0.1
* Date 23.08.2019       : Initial version
****************************************************************************/

/*****************************************************************
| includes
|----------------------------------------------------------------*/
#include "HmiInfoServiceServerComponent.h"
#include "AppHmi_MasterMessages.h"
#include "hmi_trace_if.h"
#include <string>
#include <Core/LanguageHandler.h>
#include <Core/RegionHandling/RegionHandlingTypes.h>
#include <Core/CabinConnectionHandler/CabinRestorationHandlerTypes.h>


/*****************************************************************
| defines and macros (scope: global)
|----------------------------------------------------------------*/
#define PORTNAME "hmiInfoServiceServerPort"
#ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
#define ETG_DEFAULT_TRACE_CLASS TR_CLASS_APPHMI_MASTER_MAIN
#include "trcGenProj/Header/HmiInfoServiceServerComponent.cpp.trc.h"
#endif


using namespace ::bosch::cm::ai::hmi::hmiinfoservice::HmiInfoService;
typedef ::std::vector< CabinSourceConnectionStatusInfo > CabinSourceConnectionStatus;

HmiInfoServiceServerComponent* HmiInfoServiceServerComponent::m_poSelfHmiInfoServiceServer = NULL;


HmiInfoServiceServerComponent::HmiInfoServiceServerComponent()
   : HmiInfoServiceStub(PORTNAME)
   , _cabinVideoMixInfoImpl(NULL)
{
   ETG_TRACE_USR4(("HmiInfoServiceServerComponent constructor"));
   ::languageHandler::LanguageHandler* imp = ::languageHandler::LanguageHandler::getInstance();
   if (NULL != imp)
   {
      ::languageHandler::IServiceClient* imp1 = imp->getServiceClient();
      if (NULL != imp1)
      {
         imp1->registerLanguageStatusPropertyUpdate((*this));
      }
      const ::languageHandler::ILanguageDpHandler* imp2 = imp->getLanguageDpHandler();
      if (NULL != imp2)
      {
         ETG_TRACE_USR4(("HmiInfoServiceServerComponent: constructor: languageId: %d", imp2->getLanguageId()));
         setLanguageStatusInfo(imp2->getLanguageId());
      }
   }
   sendRegionsMapOutStatusInfo();
}


HmiInfoServiceServerComponent::~HmiInfoServiceServerComponent()
{
   ETG_TRACE_USR4(("HmiInfoServiceServerComponent destructor"));
   if (NULL != m_poSelfHmiInfoServiceServer)
   {
      m_poSelfHmiInfoServiceServer = NULL;
   }
   ::languageHandler::LanguageHandler* imp = ::languageHandler::LanguageHandler::getInstance();
   if (NULL != imp)
   {
      ::languageHandler::IServiceClient* imp1 = imp->getServiceClient();
      if (NULL != imp1)
      {
         imp1->deregisterLanguageStatusPropertyUpdate((*this));
      }
   }
}


HmiInfoServiceServerComponent* HmiInfoServiceServerComponent::poGetInstance()
{
   /* No more than one instance should be created */
   if (NULL == m_poSelfHmiInfoServiceServer)
   {
      m_poSelfHmiInfoServiceServer =  new HmiInfoServiceServerComponent();
   }
   return m_poSelfHmiInfoServiceServer;
}


void HmiInfoServiceServerComponent::onLanguageStatusUpdate(const uint32 languageId)
{
   ETG_TRACE_USR4(("HmiInfoServiceServerComponent: onLanguageStatusUpdate: languageId: %d", languageId));
   setLanguageStatusInfo(languageId);
}


void HmiInfoServiceServerComponent::onGetRVCSignalStatusRequest(const ::boost::shared_ptr< GetRVCSignalStatusRequest >& request)
{
   ETG_TRACE_USR4(("HmiInfoServiceServerComponent onGetRVCSignalStatusRequest"));
}


void HmiInfoServiceServerComponent::onToggleGadgetAudioSourceRequest(const ::boost::shared_ptr< ToggleGadgetAudioSourceRequest >& request)
{
   sendToggleGadgetAudioSourceRequestSignal(request->getRegionId(), request->getSrcApplicationId(), request->getDestApplicationId());
}


void HmiInfoServiceServerComponent::onSpiConnectionStatusInfoSet(const ::boost::shared_ptr< SpiConnectionStatusInfoSet >& spiConnectionStatusInfo)
{
   ETG_TRACE_USR4(("HmiInfoServiceServerComponent onSpiConnectionStatusInfoSet "));

   HmiInfoServiceStub::onSpiConnectionStatusInfoSet(spiConnectionStatusInfo);

   ::std::string tdeviceName = spiConnectionStatusInfo->getSpiConnectionStatusInfo().getDeviceName();
   uint8 tdeviceType = spiConnectionStatusInfo->getSpiConnectionStatusInfo().getDeviceType();
   uint8 tdeviceStatus = spiConnectionStatusInfo->getSpiConnectionStatusInfo().getStatus();

   updateListeners(tdeviceName, tdeviceType, tdeviceStatus);
}


void HmiInfoServiceServerComponent::onGetCurrentForegroundApplicationIdRequest(const ::boost::shared_ptr< GetCurrentForegroundApplicationIdRequest >& request)
{
   ETG_TRACE_USR4(("HmiInfoServiceServerComponent onGetCurrentForegroundApplicationIdRequest"));
}


void HmiInfoServiceServerComponent::onSetCabinSourceConnectionStatusRequest(const ::boost::shared_ptr< SetCabinSourceConnectionStatusRequest >& request)
{
   ETG_TRACE_USR4(("HmiInfoServiceServerComponent onSetCabinSourceConnectionStatusRequest"));
   bool modified = false;
   CabinSourceConnectionStatus cabinSourceConnectionStatusUpdate = request->getCabinSourcesConnectionStatusInfo();
   CabinSourceConnectionStatus cabinSourceConnectionStatusContainer = HmiInfoServiceStub::getCabinSourcesConnectionStatus();
   for (CabinSourceConnectionStatus::iterator updateItr = cabinSourceConnectionStatusUpdate.begin(); updateItr != cabinSourceConnectionStatusUpdate.end(); updateItr++)
   {
      CabinSourceConnectionStatus::iterator containerItr = cabinSourceConnectionStatusContainer.begin();
      for (; containerItr != cabinSourceConnectionStatusContainer.end(); containerItr++)
      {
         if ((*updateItr).getSourceId() == (*containerItr).getSourceId())
         {
            bool isContentChanged = ((*updateItr).getStatus() != (*containerItr).getStatus()) ? true : false;
            isContentChanged      = ((isContentChanged) || ((*updateItr).getUserData1() != (*containerItr).getUserData1())) ? true : false;
            isContentChanged      = ((isContentChanged) || ((*updateItr).getUserData2() != (*containerItr).getUserData2())) ? true : false;
            if (isContentChanged)
            {
               (*containerItr).setStatus((*updateItr).getStatus());
               (*containerItr).setUserData1((*updateItr).getUserData1());
               (*containerItr).setUserData2((*updateItr).getUserData2());
               modified = true;
               ETG_TRACE_USR4(("HmiInfoServiceServerComponent onSetCabinSourceConnectionStatusRequest modified srcID = %d status = %d", (*containerItr).getSourceId(), (*containerItr).getStatus()));
            }
            break;
         }
      }
      if (containerItr == cabinSourceConnectionStatusContainer.end())
      {
         cabinSourceConnectionStatusContainer.push_back(CabinSourceConnectionStatusInfo((*updateItr).getSourceId(), (*updateItr).getStatus(), (*updateItr).getUserData1(), (*updateItr).getUserData2()));
         modified = true;
         ETG_TRACE_USR4(("HmiInfoServiceServerComponent onSetCabinSourceConnectionStatusRequest added srcID = %d status = %d", (*updateItr).getSourceId(), (*updateItr).getStatus()));
      }
   }
   if (modified)
   {
      HmiInfoServiceStub::setCabinSourcesConnectionStatus(cabinSourceConnectionStatusContainer);
      updateCabinSouceStatusListeners();
   }
}


void HmiInfoServiceServerComponent::onRemoveCabinSourceConnectionRequest(const ::boost::shared_ptr< RemoveCabinSourceConnectionRequest >& request)
{
   ETG_TRACE_USR4(("HmiInfoServiceServerComponent onRemoveCabinSourceConnectionStatusRequest"));
   bool modified = false;
   std::vector< uint32 > removeSourceIdsInfo = request->getSourcesIdInfo();
   CabinSourceConnectionStatus cabinSourceConnectionStatusContainer = HmiInfoServiceStub::getCabinSourcesConnectionStatus();
   for (std::vector< uint32 >::iterator removeItr = removeSourceIdsInfo.begin(); removeItr != removeSourceIdsInfo.end(); removeItr++)
   {
      for (CabinSourceConnectionStatus::iterator containerItr = cabinSourceConnectionStatusContainer.begin(); containerItr != cabinSourceConnectionStatusContainer.end(); containerItr++)
      {
         if ((*removeItr) == (*containerItr).getSourceId())
         {
            ETG_TRACE_USR4(("HmiInfoServiceServerComponent onRemoveCabinSourceConnectionStatusRequest erased source Id: %d", (*removeItr)));
            cabinSourceConnectionStatusContainer.erase(containerItr);
            modified = true;
            break;
         }
      }
   }
   if (modified)
   {
      HmiInfoServiceStub::setCabinSourcesConnectionStatus(cabinSourceConnectionStatusContainer);
      updateCabinSouceStatusListeners();
   }
}


void HmiInfoServiceServerComponent::sendRegionsMapOutStatusInfo()
{
   ETG_TRACE_USR4(("HmiInfoServiceServerComponent sendRegionsMapOutStatusInfo "));
   sendRegionsMapOutStatusInfoUpdate();
}


void HmiInfoServiceServerComponent::sendRVCSignalUpdate(bool status)
{
   ETG_TRACE_USR4(("HmiInfoServiceServerComponent sendRVCSignalUpdate status = %d", status));
   setRVCSignalStatus(status);
   sendRVCSignalStatusUpdate();
}


void HmiInfoServiceServerComponent::sendForegroundApplicationUpdate(int appID)
{
   ETG_TRACE_USR4(("HmiInfoServiceServerComponent sendForegroundApplicationUpdate appID = %d ", appID));
   setCurrentForegroundApplicationId(appID);
   sendCurrentForegroundApplicationIdUpdate();
}


void HmiInfoServiceServerComponent::sendCabinsSystemsInfoUpdate(const ::App::Core::SystemsInfoType& info)
{
   ETG_TRACE_USR4(("HmiInfoServiceServerComponent: sendCabinsSystemsInfoUpdate"));
   ::std::vector< SystemInfo > tSystemsInfo = HmiInfoServiceStub::getCabinsSystemsInfo();
   for (::App::Core::SystemsInfoType::const_iterator itr = info.begin(); (itr != info.end()); ++itr)
   {
      SystemInfo* imp = NULL;
      SystemInfo tSystemInfo;
      const ::App::Core::SystemRegionsInfo& tRegionsInfo = (*itr).getSystemRegionsInfo();
      tSystemInfo.setSystemId((*itr).getSystemId());
      tSystemInfo.setEntityId((*itr).getEntityId());
      tSystemInfo.setSystemStatus((*itr).getSystemStatus());
      tSystemInfo.setAvdeccStatus((*itr).getAVDECCStatus());
      tSystemInfo.setNanoMsgStatus((*itr).getNANOMsgStatus());
      tSystemInfo.setSystemPosition((*itr).getSystemPosition());
      tSystemInfo.setSystemCategory((*itr).getSystemCategory());
      tSystemInfo.setSystemGroup(tRegionsInfo.getSystemGroup());
      tSystemInfo.setCabinsInfo(tRegionsInfo.getSystemRegionsIdInfo());
      for (::std::vector< SystemInfo >::iterator tItr = tSystemsInfo.begin(); (tItr != tSystemsInfo.end()); ++tItr)
      {
         if ((*tItr).getSystemId() == (*itr).getSystemId())
         {
            imp = (&(*tItr));
            break;
         }
      }
      if (NULL == imp)
      {
         tSystemsInfo.push_back(tSystemInfo);
      }
      else
      {
         (*imp) = tSystemInfo;
      }
   }
   HmiInfoServiceStub::setCabinsSystemsInfo(tSystemsInfo);
}


void HmiInfoServiceServerComponent::sendCabinsSourceRestorationInfo(const ::App::Core::RestoreRegionsInfo& info)
{
   ETG_TRACE_USR4(("HmiInfoServiceServerComponent: sendCabinsSourceRestorationInfo"));
   ::std::vector< CabinSourceRestorationInfo > tCabinsSourceRestorationInfo = getCabinsSourceRestorationInfo();
   if (info.empty())
   {
      tCabinsSourceRestorationInfo.clear();
   }
   else
   {
      for (::App::Core::RestoreRegionsInfo::const_iterator itr = info.begin(); (itr != info.end()); ++itr)
      {
         CabinSourceRestorationInfo tCabinSourceRestorationInfo;
         tCabinSourceRestorationInfo.setCabinId((*itr).getRegionId());
         ::std::vector< SourceInfoHolder > tSourcesInfo;
         const ::App::Core::ActivitiesRestoreSinksInfo& tActivitiesRestoreSinksInfo = (*itr).getActivitiesRestoreSinksInfo();
         for (::App::Core::ActivitiesRestoreSinksInfo::const_iterator sitr = tActivitiesRestoreSinksInfo.begin(); (sitr != tActivitiesRestoreSinksInfo.end()); ++sitr)
         {
            if ((*sitr).getActivityType() != CABIN_ACTIVITY_MAPOUT)
            {
               for (::App::Core::RestoreSinksInfoGroup::const_iterator groupItr = (*sitr).getRestoreSinksInfoGroup().begin(); (groupItr != (*sitr).getRestoreSinksInfoGroup().end()); ++groupItr)
               {
                  for (::App::Core::RestoreSinksInfo::const_iterator tItr = (*groupItr).second.begin(); (tItr != (*groupItr).second.end()); ++tItr)
                  {
                     SourceInfoHolder tSourceInfoHolder;
                     tSourceInfoHolder.setSinkId((*tItr).getSinkId());
                     tSourceInfoHolder.setSourceId((*tItr).getSourceId());
                     tSourceInfoHolder.setSourceState((*tItr).getState());
                     tSourceInfoHolder.setSubSourceId((*tItr).getSubSourceId());
                     tSourceInfoHolder.setStatus((*tItr).getRestoreStatus());
                     tSourcesInfo.push_back(tSourceInfoHolder);
                  }
               }
            }
         }
         tCabinSourceRestorationInfo.setLastActiveSourcesInfo(tSourcesInfo);

         CabinSourceRestorationInfo* imp = NULL;
         for (::std::vector< CabinSourceRestorationInfo >::iterator tItr = tCabinsSourceRestorationInfo.begin(); (tItr != tCabinsSourceRestorationInfo.end()); ++tItr)
         {
            if ((*tItr).getCabinId() == (*itr).getRegionId())
            {
               imp = (&(*tItr));
               break;
            }
         }
         if (NULL == imp)
         {
            tCabinsSourceRestorationInfo.push_back(tCabinSourceRestorationInfo);
         }
         else
         {
            (*imp) = tCabinSourceRestorationInfo;
         }
      }
   }
   HmiInfoServiceStub::setCabinsSourceRestorationInfo(tCabinsSourceRestorationInfo);
}


void HmiInfoServiceServerComponent::addOrUpdateRegionsMapOutStatusInfo(const uint32 regionID, const uint32 glassID, const bool activeStatus)
{
   ETG_TRACE_USR4(("HmiInfoServiceServerComponent addOrUpdateRegionsMapOutStatusInfo :<- regionID = %d , glassID = %d, activeStatus = %d", regionID, glassID, activeStatus));
   ::std::vector< RegionMapOutStatusInfo >& regionMapOutStatusInfo = HmiInfoServiceStub::getRegionsMapOutStatusInfoMutable();
   ::std::vector< RegionMapOutStatusInfo >::iterator itr = regionMapOutStatusInfo.begin();
   for (; itr != regionMapOutStatusInfo.end(); ++itr)
   {
      if (regionID == itr->getRegionId())
      {
         break;
      }
   }
   if (itr == regionMapOutStatusInfo.end())
   {
      regionMapOutStatusInfo.push_back(RegionMapOutStatusInfo(regionID, std::vector <MapOutStatusInfo> ()));
      itr = regionMapOutStatusInfo.end() - 1;
   }

   std::vector< MapOutStatusInfo >::iterator itr1 = itr->getMapOutStatusInfoMutable().begin();
   for (; itr1 != itr->getMapOutStatusInfoMutable().end(); ++itr1)
   {
      if (glassID == itr1->getGlassId())
      {
         break;
      }
   }
   if (itr1 == itr->getMapOutStatusInfoMutable().end())
   {
      itr->getMapOutStatusInfoMutable().push_back(MapOutStatusInfo(glassID, activeStatus));
   }
   else
   {
      (*itr1).setActiveStatus(activeStatus);
   }
}


void HmiInfoServiceServerComponent::updateListeners(const std::string deviceName, const uint8 deviceType, const uint8 deviceStatus)
{
   for (::std::vector< ISpiDeviceInfoListenerCB* >::const_iterator itr = _spiDeviceInfoListener.begin(); (itr != _spiDeviceInfoListener.end()); ++itr)
   {
      if (NULL != (*itr))
      {
         (*itr)->onDeviceInfoUpdate(deviceName, deviceType, deviceStatus);
      }
   }
}


void HmiInfoServiceServerComponent::updateCabinSouceStatusListeners()
{
   for (::std::vector< ICabinSourceConnectionStatusCB* >::const_iterator itr = _cabinSouceStatusListener.begin(); (itr != _cabinSouceStatusListener.end()); ++itr)
   {
      if (NULL != (*itr))
      {
         (*itr)->onCabinSourceStatusUpdate();
      }
   }
}


bool HmiInfoServiceServerComponent::getCabinSourceStatus(const int16 srcID)
{
   bool status = false;
   const CabinSourceConnectionStatus& cabinSourceConnectionStatusContainer = HmiInfoServiceStub::getCabinSourcesConnectionStatus();
   for (CabinSourceConnectionStatus::const_iterator itr = cabinSourceConnectionStatusContainer.begin(); itr != cabinSourceConnectionStatusContainer.end(); itr++)
   {
      if ((*itr).getSourceId() == srcID)
      {
         status = (*itr).getStatus();
         break;
      }
   }
   return status;
}


void HmiInfoServiceServerComponent::onSetCabinVideoMixIndexRequest(const ::boost::shared_ptr< SetCabinVideoMixIndexRequest >& request)
{
   CabinVideoMixInfo cabinVideoMixInfo = request->getCabinVideoMixInfo();
   ETG_TRACE_USR4(("HmiInfoServiceServerComponent onSetCabinVideoMixIndexRequest : regionID = %d , videoMixIndex = %d reason = %d", cabinVideoMixInfo.getCabinId(), cabinVideoMixInfo.getVideoMixIndex(), cabinVideoMixInfo.getReason()));
   if (_cabinVideoMixInfoImpl != NULL)
   {
      _cabinVideoMixInfoImpl->onSetCabinVideoMixInfo(cabinVideoMixInfo);
   }
}


void HmiInfoServiceServerComponent::setCabinVideoMixInfo(const CabinsVideoMixInfo& cabinVideoMixInfoUpdate)
{
   bool modified = false;
   CabinsVideoMixInfo cabinsVideoMixInfo = HmiInfoServiceStub::getCabinsVideoMixInfo();
   for (CabinsVideoMixInfo::const_iterator updateItr = cabinVideoMixInfoUpdate.begin(); updateItr != cabinVideoMixInfoUpdate.end(); updateItr++)
   {
      uint32 cabinId = (*updateItr).getCabinId();
      uint8 videoMixIndex = (*updateItr).getVideoMixIndex();
      uint8 reason = (*updateItr).getReason();
      ::std::vector< CabinVideoMixInfo >::iterator itr = cabinsVideoMixInfo.begin();
      for (; itr != cabinsVideoMixInfo.end(); itr++)
      {
         if ((*itr).getCabinId() == cabinId)
         {
            (*itr).setVideoMixIndex(videoMixIndex);
            (*itr).setReason(reason);
            modified = true;
            break;
         }
      }
      if (itr == cabinsVideoMixInfo.end())
      {
         CabinVideoMixInfo newCabinVideoMixInfo(cabinId, reason, videoMixIndex);
         cabinsVideoMixInfo.push_back(newCabinVideoMixInfo);
         modified = true;
      }
   }
   if (modified)
   {
      HmiInfoServiceStub::setCabinsVideoMixInfo(cabinsVideoMixInfo);
      updateVideoMixIndexListeners(cabinsVideoMixInfo);
   }
}


uint8 HmiInfoServiceServerComponent::getCabinVideoMixIndex(const uint32 regionId) const
{
   uint8 videoMixIndex = CABIN_VIDEO_MIX_INDEX_NONE;
   const CabinsVideoMixInfo& cabinsVideoMixInfo = HmiInfoServiceStub::getCabinsVideoMixInfo();
   for (::std::vector< CabinVideoMixInfo >::const_iterator itr = cabinsVideoMixInfo.begin(); (itr != cabinsVideoMixInfo.end()); itr++)
   {
      if ((*itr).getCabinId() == regionId)
      {
         videoMixIndex = (*itr).getVideoMixIndex();
         break;
      }
   }
   return videoMixIndex;
}


const CabinVideoMixInfo* HmiInfoServiceServerComponent::getVideoMixInfoForRegion(const uint32 regionId) const
{
   const CabinVideoMixInfo* imp = NULL;
   const CabinsVideoMixInfo& cabinsVideoMixInfo = HmiInfoServiceStub::getCabinsVideoMixInfo();
   for (::std::vector< CabinVideoMixInfo >::const_iterator itr = cabinsVideoMixInfo.begin(); (itr != cabinsVideoMixInfo.end()); itr++)
   {
      if ((*itr).getCabinId() == regionId)
      {
         imp = &(*itr);
         break;
      }
   }
   return imp;
}


void HmiInfoServiceServerComponent::updateVideoMixIndexListeners(CabinsVideoMixInfo& cabinsVideoMixInfo)
{
   for (::std::vector< ICabinVideoMixIndexListener* >::const_iterator itr = _cabinVideoMixIndexListener.begin(); (itr != _cabinVideoMixIndexListener.end()); ++itr)
   {
      if (NULL != (*itr))
      {
         for (::std::vector< CabinVideoMixInfo >::iterator titr = cabinsVideoMixInfo.begin(); (titr != cabinsVideoMixInfo.end()); titr++)
         {
            (*itr)->onVideoMixIndexUpdate((*titr).getCabinId(), (*titr).getVideoMixIndex(), (*titr).getReason());
         }
      }
   }
}


void HmiInfoServiceServerComponent::sendAVRestorationSignal(const uint32 cabinId, const int16 sourceId)
{
   ETG_TRACE_USR4(("HmiInfoServiceServerComponent sendAVRestorationSignal cabinId = %d sourceId = %d", cabinId, sourceId));
   HmiInfoServiceStub::sendRestoreAVSourceSignal(cabinId, sourceId);
}
