/**
 * @file        : diagnosisClientHandler.cpp
 * @author      : Nikhil
 * @addtogroup  : AppHmi_Master
 * @brief       :
 * @copyright   : (c) 2018-2019 Robert Bosch Car Multimedia GmbH
 *                The reproduction, distribution and utilization of this file as
 *                well as the communication of its contents to others without express
 *                authorization is prohibited. Offenders will be held liable for the
 *                payment of damages. All rights reserved in the event of the grant
 *                of a patent, utility model or design.
 */
/****************************************************************************************/

#include "hall_std_if.h"

#define OSAL_S_IMPORT_INTERFACE_GENERIC
#include "osal_if.h"

#include <string>
#include <stdio.h>
#include<sstream>

#include "../../../../../../../../ai_osal_linux/components/system/system_types.h"
#include "Core/Diagnosis/diagnosisClientHandler.h"
#include "AppHmi_MasterMessages.h"
#include "AppBase/ScreenBrokerClient/ScreenBrokerClient.h"
#include "midw_diagnostics_fi_typesConst.h"
#ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
#define ETG_DEFAULT_TRACE_CLASS TR_CLASS_APPHMI_MASTER_DM
#include "trcGenProj/Header/diagnosisClientHandler.cpp.trc.h"
#endif

#define KDSENTRY_VID_ADDRESS  0x0D72 //in HEX3442
#define WIFI_HOTSPOT_NAME_CONFIG_KEY_SIZE 4
using namespace hmibase;
using namespace std;
using namespace ::asf::core;
using namespace Midw_Diagnostics_FI;
using namespace midw_diagnostics_fi_types;

namespace App {
namespace Core {

diagnosisClientHandler* diagnosisClientHandler::_diagnosisClientHandler = NULL;

diagnosisClientHandler::diagnosisClientHandler()
   : _diagnosticProxy(Midw_Diagnostics_FI::Midw_Diagnostics_FIProxy::createProxy("diagnosticsFiPort", *this))
   , _dataModelInstance(NULL)
{
   ETG_TRACE_USR4(("diagnosisClientHandler:diagnosisClientHandler Class is created"));
   if (_diagnosticProxy.get())
   {
      StartupSync::getInstance().registerPropertyRegistrationIF(this, _diagnosticProxy->getPortName());
   }

   m_vid_value = 0;
#ifdef VARIANT_S_FTR_ENABLE_ANTI_THEFT
   longint_VehicleId = 0;
#endif
}


/**
 * @Destructor
 */
diagnosisClientHandler::~diagnosisClientHandler()
{
   ETG_TRACE_USR4(("diagnosisClientHandler:~diagnosisClientHandler Destructor - Entry"));
   if (_diagnosisClientHandler != NULL)
   {
      delete _diagnosisClientHandler;
   }
   _dataModelInstance = NULL;
   ETG_TRACE_USR4(("diagnosisClientHandler:~diagnosisClientHandler Destructor - Exit"));
}


/**
 * To invoke the registerProperties function of the dependent classes.
 * @param[in] <proxy> proxies to middleware server
 * @param  <stateChange> --todo: need to add details
 */
void diagnosisClientHandler::registerProperties(const ::boost::shared_ptr< asf::core::Proxy >& proxy,
      const asf::core::ServiceStateChange& stateChange)
{
   ETG_TRACE_USR4(("RegisterProperties ---- diagnosisClientHandler"));

   if (proxy == _diagnosticProxy)
   {
      ETG_TRACE_USR4(("diagnosisClientHandler: registerProperties called With State:%d, Action: UpReg", stateChange.getCurrentState()));

      _diagnosticProxy->sendRemoteControlUpReg(*this);
   }
}


/**
 * send relupregall property for device manager connection notification
 * @param[in] <proxy> diagnostic fi proxy object
 * @param[in] <stateChange> state information
 */
void diagnosisClientHandler::deregisterProperties(const ::boost::shared_ptr< asf::core::Proxy >& proxy,
      const asf::core::ServiceStateChange& stateChange)
{
   ETG_TRACE_USR4(("DeregisterProperties ---- diagnosisClientHandler"));

   if (proxy == _diagnosticProxy)
   {
      ETG_TRACE_USR4(("diagnosisClientHandler: deregisterProperties called With State:%d", stateChange.getCurrentState()));
   }
}


/*****************************************************************************
* @brief   onUnAvailable
*****************************************************************************/
void diagnosisClientHandler::onUnavailable(const boost::shared_ptr<asf::core::Proxy>& proxy, const asf::core::ServiceStateChange& /*stateChange*/)
{
   if ((proxy == _diagnosticProxy) && (_diagnosticProxy.get()))
   {
      _diagnosticProxy->sendRemoteControlRelUpRegAll();
      _diagnosticProxy->sendConfigUpdateViaUSBRelUpRegAll();
#ifdef VARIANT_S_FTR_ENABLE_ANTI_THEFT
      _diagnosticProxy->sendKdsEntryRelUpRegAll();
      _diagnosticProxy->sendAntiTheftUnlockHMIRelUpRegAll();
      _diagnosticProxy->sendAntiTheftUnlockHURelUpRegAll();
      _diagnosticProxy->sendAntiTheftStatusRelUpRegAll();
#endif
   }
}


/*****************************************************************************
* @brief   onAvailable
*****************************************************************************/
void diagnosisClientHandler::onAvailable(const ::boost::shared_ptr< asf::core::Proxy >& proxy, const asf::core::ServiceStateChange& /*stateChange*/)
{
   ETG_TRACE_USR3(("diagnosisClientHandler::onAvailable()"));

   if ((proxy == _diagnosticProxy) && (_diagnosticProxy.get()))
   {
      _diagnosticProxy->sendRemoteControlUpReg(*this);
      _diagnosticProxy->sendConfigUpdateViaUSBUpReg(*this);
#ifdef VARIANT_S_FTR_ENABLE_ANTI_THEFT
      _diagnosticProxy->sendKdsEntryUpReg(*this, KDSENTRY_VID_ADDRESS);
      _diagnosticProxy->sendAntiTheftUnlockHMIUpReg(*this);
      _diagnosticProxy->sendAntiTheftUnlockHUUpReg(*this);
      _diagnosticProxy->sendAntiTheftStatusUpReg(*this);
#endif
   }
}


/**
 * receive Remote control status property message for device manager connection notification
 * @param[in] <proxy> diagnostic fi proxy object
 * @param[in] <status> status object in case no error
 */

/************************************************
Function 	: onRemoteControlStatus
Parameters 	: status
Description :  receive Remote control status property message for device manager connection notification

SYSFL/SWFL : Implemented SWFL 2549
************************************************/

void diagnosisClientHandler::onRemoteControlStatus(const ::boost::shared_ptr< Midw_Diagnostics_FI::Midw_Diagnostics_FIProxy >& proxy, const boost::shared_ptr< Midw_Diagnostics_FI::RemoteControlStatus >& status)
{
   ETG_TRACE_USR4(("diagnosisClientHandler: onRemoteControlStatus --entry"));

   if (proxy == _diagnosticProxy)
   {
      ETG_TRACE_USR4(("diagnosisClientHandler:: _diagnosticProxy is valid"));

      ETG_TRACE_USR4(("diagnosisClientHandler:: status->getCmd : %d", status->getCmd()));
      ETG_TRACE_USR4(("diagnosisClientHandler:: status->getSrc : %d", status->getSrc()));

      if ((status->getCmd() == midw_diagnostics_fi_types::T_e8_RCtrlCommand__RCTRL_CMD_ACTIVATE) && \
            (status->getSrc() == ((midw_diagnostics_fi_types::T_e8_RCtrlSource__RCTRL_SRC_HMI) || (midw_diagnostics_fi_types::T_e8_RCtrlSource__RCTRL_SRC_TEF))))
      {
         ETG_TRACE_USR4(("diagnosisClientHandler:: RemoteControl - ON"));

         //For SPM handling for change of state
         //HmiStateHandler::getInstance()->sendDiagStatus(true);

         //send +ve response to remote control to activate cmd
         _diagnosticProxy->sendRemoteControlSet(*this, status->getSrc() , status->getCmd() , midw_diagnostics_fi_types::T_e8_RCtrlSignal__RCTRL_SIG_ALL , true); // T_e8_RCtrlSignal=T_e8_RCtrlSignal__RCTRL_SIG_ALL
         ETG_TRACE_USR4(("diagnosisClientHandler:: Send Positive Response to diagnosis"));

         POST_MSG((COURIER_MESSAGE_NEW(MmiStateStatusUpdMsg)(MMISTATE_ON_FULL_STANDBY)));
      }

      else if ((status->getCmd() == midw_diagnostics_fi_types::T_e8_RCtrlCommand__RCTRL_CMD_DEACTIVATE) && \
               (status->getSrc() == ((midw_diagnostics_fi_types::T_e8_RCtrlSource__RCTRL_SRC_HMI) || (midw_diagnostics_fi_types::T_e8_RCtrlSource__RCTRL_SRC_TEF))))
      {
         ETG_TRACE_USR4(("diagnosisClientHandler:: Remote Control - OFF"));

         //For SPM handling for change of state
         //HmiStateHandler::getInstance()->sendDiagStatus(false);

         //send +ve response to remote control to deactivate cmd
         _diagnosticProxy->sendRemoteControlSet(*this, status->getSrc() , status->getCmd() , midw_diagnostics_fi_types::T_e8_RCtrlSignal__RCTRL_SIG_ALL , true);
         // T_e8_RCtrlSignal=T_e8_RCtrlSignal__RCTRL_SIG_ALL
         ETG_TRACE_USR4(("diagnosisClientHandler:: Send Positive Response to diagnosis"));

         POST_MSG((COURIER_MESSAGE_NEW(MmiStateStatusUpdMsg)(MMISTATE_ON_FULL)));
      }
   }
   ETG_TRACE_USR4(("diagnosisClientHandler:: onRemoteControlStatus --exit"));
}


/************************************************
Function 	: onRemoteControlError
Parameters 	:
Description : receive remote Control error message for diagnostic notification

SYSFL/SWFL : Implemented SWFL 2549
************************************************/
/**
 * receive remote Control error message for diagnostic notification
 * @param[in] <proxy> diagnostic fi proxy object
 * @param[in] <error> error object in case of error
 */
void diagnosisClientHandler::onRemoteControlError(const ::boost::shared_ptr< Midw_Diagnostics_FI::Midw_Diagnostics_FIProxy >& /*proxy*/, const boost::shared_ptr< Midw_Diagnostics_FI::SetDisplayError >& /*error*/)
{
   ETG_TRACE_USR4(("diagnosisClientHandler: onRemoteControlError"));
}


void diagnosisClientHandler::onConfigUpdateViaUSBError(const ::boost::shared_ptr< Midw_Diagnostics_FIProxy >& proxy, const ::boost::shared_ptr< ConfigUpdateViaUSBError >& error)
{
   ETG_TRACE_USR4(("diagnosisClientHandler::onConfigUpdateViaUSBError"));
}


void diagnosisClientHandler::onConfigUpdateViaUSBStatus(const ::boost::shared_ptr< Midw_Diagnostics_FIProxy >& proxy, const ::boost::shared_ptr< ConfigUpdateViaUSBStatus >& status)
{
   ETG_TRACE_USR4(("diagnosisClientHandler:: onConfigUpdateViaUSBStatus status->getConfigUpdateStatus : %d", status->getConfigUpdateStatus()));
   uint8 kdsStatus = getKDSUpdateStatus();
   if ((NULL != _dataModelInstance) && (NULL != _diagnosticProxy.get()) && (proxy == _diagnosticProxy) && (kdsStatus != status->getConfigUpdateStatus()))
   {
      _dataModelInstance->updateKDSValueToModel(status->getConfigUpdateStatus());
   }
}


uint8 diagnosisClientHandler::getKDSUpdateStatus()
{
   uint8 status = 255;
   if ((NULL != _diagnosticProxy.get()) && (_diagnosticProxy->hasConfigUpdateViaUSB()))
   {
      status = _diagnosticProxy->getConfigUpdateViaUSB().getConfigUpdateStatus();
   }
   return status;
}


#ifdef VARIANT_S_FTR_ENABLE_ANTI_THEFT

void diagnosisClientHandler::onAntiTheftUnlockHMIError(const ::boost::shared_ptr< Midw_Diagnostics_FIProxy >& proxy, const ::boost::shared_ptr< AntiTheftUnlockHMIError >& error)
{
   ETG_TRACE_USR4(("diagnosisClientHandler: onAntiTheftUnlockHMIError"));
}


void diagnosisClientHandler::onAntiTheftUnlockHMIStatus(const ::boost::shared_ptr< Midw_Diagnostics_FIProxy >& proxy, const ::boost::shared_ptr< AntiTheftUnlockHMIStatus >& status)
{
   ETG_TRACE_USR4(("diagnosisClientHandler: onAntiTheftUnlockHMIStatus"));
   POST_MSG((COURIER_MESSAGE_NEW(UpdateVINStatus)()));
}


void diagnosisClientHandler::onAntiTheftUnlockHUError(const ::boost::shared_ptr< Midw_Diagnostics_FIProxy >& proxy, const ::boost::shared_ptr< AntiTheftUnlockHUError >& error)
{
   ETG_TRACE_USR4(("diagnosisClientHandler: onAntiTheftUnlockHUError"));
}


void diagnosisClientHandler::onAntiTheftUnlockHUStatus(const ::boost::shared_ptr< Midw_Diagnostics_FIProxy >& proxy, const ::boost::shared_ptr< AntiTheftUnlockHUStatus >& status)
{
   ETG_TRACE_USR4(("diagnosisClientHandler: onAntiTheftUnlockHUStatus"));
   POST_MSG((COURIER_MESSAGE_NEW(UpdateVINStatus)()));
   //uint32 m_vid_value = _diagnosticProxy->getKdsEntry();
}


void diagnosisClientHandler::onKdsEntryError(const ::boost::shared_ptr< Midw_Diagnostics_FIProxy >& proxy, const ::boost::shared_ptr< KdsEntryError >& error)
{
   ETG_TRACE_USR4(("diagnosisClientHandler: onKdsEntryError"));
}


void diagnosisClientHandler::onKdsEntryStatus(const ::boost::shared_ptr< Midw_Diagnostics_FIProxy >& proxy, const ::boost::shared_ptr< KdsEntryStatus >& status)
{
   ETG_TRACE_USR4(("diagnosisClientHandler: onKdsEntryStatus"));

   //ETG_TRACE_USR4(("diagnosisClientHandler:: status->getKdsEntry : %d", status->getKdsEntry()));

   uint32 m_vid_value = status->getKdsEntry();

   ETG_TRACE_USR4((" KDS entry =  %x", m_vid_value));
}


void diagnosisClientHandler::onAntiTheftStatusError(const ::boost::shared_ptr< Midw_Diagnostics_FIProxy >& proxy, const ::boost::shared_ptr< AntiTheftStatusError >& error)
{
   ETG_TRACE_USR4(("diagnosisClientHandler: onAntiTheftStatusError"));
}


void diagnosisClientHandler::onAntiTheftStatusStatus(const ::boost::shared_ptr< Midw_Diagnostics_FIProxy >& proxy, const ::boost::shared_ptr< AntiTheftStatusStatus >& status)
{
   ETG_TRACE_USR4(("diagnosisClientHandler: onAntiTheftStatusStatus"));
   uint8 u8AntiTheftStatus = App::datamodel::DataModel::getInstance().u8GetAntiTheftStatus();
   _diagnosticProxy->sendAntiTheftStatusSet(*this, T_AntiTheftStatus(u8AntiTheftStatus));
}


long int diagnosisClientHandler::u32GetVINValue()
{
   ETG_TRACE_USR4((" KDS entry =  %x", m_vid_value));
   std::string kdsValue = "";
   /*  if (bReadData((KDSENTRY_VID_ADDRESS), 4, kdsValue))
   {
      ETG_TRACE_USR4((" KDS entry =  return true %s",kdsValue.c_str()));
   }
   else
   {
      ETG_TRACE_USR4((" KDS entry =  return false"));
   } */
   //ETG_TRACE_USR4((" KDS entry =  %x", m_vid_value));
   std::string sWiFiName;

   unsigned char u8DeviceName[WIFI_HOTSPOT_NAME_CONFIG_KEY_SIZE + 1];
   memset(u8DeviceName, 0, sizeof(u8DeviceName));

   if (bReadData(KDSENTRY_VID_ADDRESS, WIFI_HOTSPOT_NAME_CONFIG_KEY_SIZE, u8DeviceName))//to be changes
   {
      sWiFiName = ((char*)u8DeviceName);
      kdsValue = ((char*)u8DeviceName);

      ETG_TRACE_USR2(("Default WiFi Name without invalid UTF-8 characters: %s", u8DeviceName));
      ETG_TRACE_USR2(("Default WiFi Name without invalid UTF-8 characters: %s", sWiFiName.c_str()));
      ETG_TRACE_USR2(("Default WiFi Name without invalid UTF-8 characters: %s", kdsValue.c_str()));
   }
   return longint_VehicleId;
}


//bool diagnosisClientHandler::bReadData(uint16 u16KdsKey, uint16 u16DataLen, std::string& kdsData)
bool diagnosisClientHandler::bReadData(uint16 u16KdsKey, uint16 u16DataLen, uint8* pu8DataBuffer)
{
   ETG_TRACE_USR4(("FC_Bluetooth_KdsConfig::bReadData() entered: u16KdsKey = 0x%x, u16DataLen = 0x%x \n",
                   u16KdsKey, u16DataLen));
   bool bReadSuccess = FALSE;
   NORMAL_M_ASSERT(OSAL_NULL != pu8DataBuffer);
   if (OSAL_NULL != pu8DataBuffer)
   {
      ETG_TRACE_USR4(("FC_Blu"));
      //! Open KDS file
      OSAL_tIODescriptor tIoKdsHandle = OSAL_IOOpen(OSAL_C_STRING_DEVICE_KDS, OSAL_EN_READONLY);
      ETG_TRACE_USR4(("FC_Blu"));
      if (OSAL_ERROR != tIoKdsHandle)
      {
         ETG_TRACE_USR4(("FC_Blu"));
         tsKDSEntry rKDSEntryData;
         rKDSEntryData.u16Entry = u16KdsKey;
         rKDSEntryData.u16EntryLength = u16DataLen;
         rKDSEntryData.u16EntryFlags = M_KDS_ENTRY_FLAG_NONE;
         rKDSEntryData.au8EntryData[0] = 0;

         //! Read required data from file
         tS32 s32OsalReadErr = OSAL_s32IORead(tIoKdsHandle, (tPS8) &rKDSEntryData, (sizeof(rKDSEntryData)));

         if (OSAL_ERROR != s32OsalReadErr)
         {
            (tVoid) OSAL_pvMemoryCopy((tVoid*)(pu8DataBuffer), rKDSEntryData.au8EntryData, u16DataLen);

            bReadSuccess = TRUE;
         }
         else
         {
            ETG_TRACE_ERR(("Kds::bReadData: Error reading KDS file: %d \n", OSAL_u32ErrorCode()));
         }

         int intbuf[u16DataLen];

         longint_VehicleId = 0;
         if (u16DataLen > 0)
         {
            /* for (unsigned int i = u16DataLen-1; i >= 0; i--)
            {
            	intbuf[i] = rKDSEntryData.au8EntryData[i];
            	longint_VehicleId= (longint_VehicleId<<8)+intbuf[i];
            } */
            for (unsigned int i = 0; i < u16DataLen; i++)
            {
               //intbuf[i] = rKDSEntryData.au8EntryData[u16DataLen-i];
               longint_VehicleId = (longint_VehicleId << 8) + rKDSEntryData.au8EntryData[i];//chnaged the order of value received
            }
            ETG_TRACE_USR4(("KDSReader::readVINIDFromKDS - KDSEntry data : %x" , longint_VehicleId));
         }
         ETG_TRACE_USR4(("KDSReader::readVINIDFromKDS - KDSEntry data : %x" , longint_VehicleId));
         //! Close KDS file
         (tVoid) OSAL_s32IOClose(tIoKdsHandle);
      }//if (OSAL_ERROR != tIoKdsHandle)
      else
      {
         ETG_TRACE_ERR(("Kds::bReadData: Error opening KDS file: %d \n", OSAL_u32ErrorCode()));
      }
   }//if (OSAL_NULL != pu8DataBuffer)
   return bReadSuccess;
}


bool diagnosisClientHandler::performASCIIHexToStringConversion(uint8 asciiValue[], uint8 byteSize, std::string& resultString)
{
   bool isValid = false;
   if (byteSize > 0)
   {
      for (unsigned int i = 0; i < byteSize; i++)
      {
         resultString.push_back(asciiValue[i]);
      }
      isValid = true;
   }
   return isValid;
}


#endif
}


}
