/**
 * @file         : ContextSwitchHandlerTypes.h
 * @author       : INF4CV - AppHmi_Master Team
 * @addtogroup   : AppHmi_Master
 * @brief        : ContextSwitchHandlerTypes is to handle the ContextSwitchHandler
 *                 feature types implementation.
 * @copyright    : (c) 2020-2020 Robert Bosch Car Multimedia GmbH
 *                 The reproduction, distribution and utilization of this file as
 *                 well as the communication of its contents to others without express
 *                 authorization is prohibited. Offenders will be held liable for the
 *                 payment of damages. All rights reserved in the event of the grant
 *                 of a patent, utility model or design.
 */


#ifndef CONTEXT_SWITCH_HANDLER_TYPES_H
#define CONTEXT_SWITCH_HANDLER_TYPES_H


#include "ApplicationSwitchConst.h"
#include "ContextSwitchHandlerConstants.h"


/**
 *  Defines Declaration
 */

#define NS_APPLICATION_SWITCH  ::bosch::cm::ai::hmi::hmimasterservice::ApplicationSwitch


/**
 *  Classes Declaration
 */

namespace App {
namespace Core {


/**
 * RequestBase is used to
 *  - Handle base class implementation for
 *      ContextSwitchHandler requests.
 */
class RequestBase
{
   public:
      RequestBase() {}
      virtual ~RequestBase() {}
};


/**
 * ApplicationContextInfo is used to
 *  - Hold the application context information
 */
class ApplicationContextInfo : public RequestBase
{
   public:
      /**
       *  Member Functions Declaration
       */
      ApplicationContextInfo()
      {
         clear();
      }

      virtual ~ApplicationContextInfo()
      {
         clear();
      }

      ApplicationContextInfo(const ApplicationContextInfo& obj)
      {
         clear();
         *this = obj;
      }

      inline ApplicationContextInfo& operator = (const ApplicationContextInfo& obj)
      {
         if (this != &obj)
         {
            clear();
            _priority      = obj._priority;
            _regionId      = obj._regionId;
            _contextId     = obj._contextId;
            _applicationId = obj._applicationId;
         }
         return *this;
      }

      inline bool operator == (const ApplicationContextInfo& obj)
      {
         bool isValid = (_priority == obj._priority) ? true : false;
         isValid      = ((isValid) && (_regionId == obj._regionId)) ? true : false;
         isValid      = ((isValid) && (_contextId == obj._contextId)) ? true : false;
         isValid      = ((isValid) && (_applicationId == obj._applicationId)) ? true : false;
         return isValid;
      }

      inline void clear()
      {
         _priority      = 0xFF;
         _regionId      = NS_APPLICATION_SWITCH::Region__MAX_REG;
         _contextId     = 0;
         _applicationId = 0;
      }

      inline void setPriority(const uint8 priority)
      {
         _priority = priority;
      }

      inline uint8 getPriority() const
      {
         return _priority;
      }

      inline void setRegionId(const uint32 id)
      {
         _regionId = id;
      }

      inline uint32 getRegionId() const
      {
         return _regionId;
      }

      inline void setContextId(const uint32 id)
      {
         _contextId = id;
      }

      inline uint32 getContextId() const
      {
         return _contextId;
      }

      inline void setApplicationId(const uint32 id)
      {
         _applicationId = id;
      }

      inline uint32 getApplicationId() const
      {
         return _applicationId;
      }

   private:
      /**
       *  Member Variables Declaration
       */
      uint8 _priority;
      uint32 _regionId;
      uint32 _contextId;
      uint32 _applicationId;
};


/**
 * ContextSwitchRequestInfo is used to
 *  - Hold the context switch request information
 */
class ContextSwitchRequestInfo : public RequestBase
{
   public:
      /**
       *  Member Functions Declaration
       */
      ContextSwitchRequestInfo()
      {
         clear();
      }

      virtual ~ContextSwitchRequestInfo()
      {
         clear();
      }

      ContextSwitchRequestInfo(const ContextSwitchRequestInfo& obj)
      {
         clear();
         *this = obj;
      }

      inline ContextSwitchRequestInfo& operator = (const ContextSwitchRequestInfo& obj)
      {
         if (this != &obj)
         {
            clear();
            _regionId            = obj._regionId;
            _sourceContextId     = obj._sourceContextId;
            _targetContextId     = obj._targetContextId;
            _sourceApplicationId = obj._sourceApplicationId;
            _targetApplicationId = obj._targetApplicationId;
         }
         return *this;
      }

      inline void clear()
      {
         _regionId            = NS_APPLICATION_SWITCH::Region__MAX_REG;
         _sourceContextId     = 0;
         _targetContextId     = 0;
         _sourceApplicationId = 0;
         _targetApplicationId = 0;
      }

      inline void setRegionId(const uint32 id)
      {
         _regionId = id;
      }

      inline uint32 getRegionId() const
      {
         return _regionId;
      }

      inline void setSourceContextId(const uint32 id)
      {
         _sourceContextId = id;
      }

      inline uint32 getSourceContextId() const
      {
         return _sourceContextId;
      }

      inline void setTargetContextId(const uint32 id)
      {
         _targetContextId = id;
      }

      inline uint32 getTargetContextId() const
      {
         return _targetContextId;
      }

      inline void setSourceApplicationId(const uint32 id)
      {
         _sourceApplicationId = id;
      }

      inline uint32 getSourceApplicationId() const
      {
         return _sourceApplicationId;
      }

      inline void setTargetApplicationId(const uint32 id)
      {
         _targetApplicationId = id;
      }

      inline uint32 getTargetApplicationId() const
      {
         return _targetApplicationId;
      }

   private:
      /**
       *  Member Variables Declaration
       */
      uint32 _regionId;
      uint32 _sourceContextId;
      uint32 _targetContextId;
      uint32 _sourceApplicationId;
      uint32 _targetApplicationId;
};


/**
 * ContextSwitchInfo is used to
 *  - Hold the context switch information
 */
class ContextSwitchInfo
{
   public:
      /**
       *  Member Functions Declaration
       */
      ContextSwitchInfo()
      {
         clear();
      }

      virtual ~ContextSwitchInfo()
      {
         clear();
      }

      ContextSwitchInfo(const ContextSwitchInfo& obj)
      {
         clear();
         *this = obj;
      }

      inline ContextSwitchInfo& operator = (const ContextSwitchInfo& obj)
      {
         if (this != &obj)
         {
            clear();
            _state        = obj._state;
            _contextsInfo = obj._contextsInfo;
         }
         return *this;
      }

      inline void clear()
      {
         _state = CONTEXT_SWITCH_STATE_IDLE;
         _contextsInfo.clear();
      }

      inline void setSwitchState(const enContextSwitchState state)
      {
         _state = state;
      }

      inline uint32 getSwitchState() const
      {
         return _state;
      }

      inline bool isSwitchOngoing() const
      {
         bool isValid = (_state == CONTEXT_SWITCH_STATE_PROCESS) ? true : false;
         return isValid;
      }

      inline void setApplicationContextInfo(const ApplicationContextInfo& info)
      {
         _contextsInfo = info;
      }

      inline const ApplicationContextInfo& getApplicationContextInfo() const
      {
         return _contextsInfo;
      }

      inline ApplicationContextInfo& getMutableApplicationContextInfo()
      {
         return _contextsInfo;
      }

   private:
      /**
       *  Member Variables Declaration
       */
      enContextSwitchState _state;
      ApplicationContextInfo _contextsInfo;
};


} //namespace Core
} //namespace App


/**
 *  Undef
 */

#undef NS_APPLICATION_SWITCH


#endif /* CONTEXT_SWITCH_HANDLER_TYPES_H */
