/**
 * @file         : ContextSwitchHandler.h
 * @author       : INF4CV - AppHmi_Master Team
 * @addtogroup   : AppHmi_Master
 * @brief        : ContextSwitchHandler is to handle the context based
 *                 application switch.
 * @copyright    : (c) 2020-2020 Robert Bosch Car Multimedia GmbH
 *                 The reproduction, distribution and utilization of this file as
 *                 well as the communication of its contents to others without express
 *                 authorization is prohibited. Offenders will be held liable for the
 *                 payment of damages. All rights reserved in the event of the grant
 *                 of a patent, utility model or design.
 */


#ifndef CONTEXT_SWITCH_HANDLER_H
#define CONTEXT_SWITCH_HANDLER_H


#include "ProjectBaseTypes.h"
#include "ContextSwitchHandlerTypes.h"
#include "ContextSwitchHandlerInterface.h"
#include <asf/core/Timer.h>


/**
 *  Classes Declaration
 */

namespace App {
namespace Core {


/**
 * ContextSwitchHandler is used to
 *  - Handle the context based application switch
 */
class ContextSwitchHandler : public IContextSwitchHandler
   , public ::asf::core::TimerCallbackIF
{
   public:
      /**
       *  Member Functions Declaration
       */
      ContextSwitchHandler();
      virtual ~ContextSwitchHandler();
      bool fetchActiveContextInfo(ApplicationContextInfo& info) const;
      bool fetchActiveContextInfo(const uint32 appId, ApplicationContextInfo& info) const;
      bool fetchDefaultContextInfoForRegion(const uint32 regionId, ApplicationContextInfo& info) const;
      void addActiveContextListener(IActiveContextListenerCB& imp);
      void removeActiveContextListener(IActiveContextListenerCB& imp);
      void dumpSwitchStackInfo();
      void onApplicationSwitchComplete(const uint32 appId);
      void onStoreContext(const ApplicationContextInfo& info);
      void onContextSwitchRequest(const ContextSwitchRequestInfo& info);
      void onSwitchLastApplication(const bool lastApplicationDebounceTimerStatus = false);
      void onExecuteContextSwitchRequest(const ContextSwitchRequestInfo& info, const uint32 code);
      void onClearLastApplication();
      void onExpired(::asf::core::Timer& timer, ::boost::shared_ptr< ::asf::core::TimerPayload > data);

      inline void setRegionHandling(IRegionHandling& imp)
      {
         _regionHandling = (&imp);
      }

      inline void setHMIStateHandler(IHmiStateHandler& imp)
      {
         _hmiStateHandler = (&imp);
      }

      inline void setRootContextHandler(IRootContextHandler& imp)
      {
         _rootContextHandler = (&imp);
      }

      inline void setContextPriorityHandler(IContextPriorityHandler& imp)
      {
         _contextPriorityHandler = (&imp);
      }

      inline void setDefaultApplicationsContextInfo(const ApplicationsContextInfo& info)
      {
         _defaultContextInfo = info;
      }

      inline void setApplicationSurfaceInfoHandler(IApplicationSurfaceInfoHandler& imp)
      {
         _applicationSurfaceInfoHandler = (&imp);
      }

      inline void setApplicationSwitchServerComponent(IApplicationSwitchServerComponent& imp)
      {
         _applicationSwitchServerComponent = (&imp);
      }

      inline uint32 getActiveApplicationId() const
      {
         return _activeApplicationId;
      }

   private:
      /**
       *  Member Typedefs Declaration
       */
      typedef bool (ContextSwitchHandler::*validateCondFunction)(const ContextSwitchRequestInfo& info) const;
      typedef void (ContextSwitchHandler::*switchRequestPerformFunction)(const ContextSwitchRequestInfo& info);
      typedef ::std::pair< validateCondFunction, switchRequestPerformFunction > SwitchRequestFunctionInfo;
      typedef ::std::vector< SwitchRequestFunctionInfo > SwitchRequestFunctionsInfo;
      /**
       *  Member Functions Declaration
       */
      ContextSwitchHandler(const ContextSwitchHandler& obj);
      ContextSwitchHandler& operator= (const ContextSwitchHandler& obj);
      uint8 getContextPriority(const uint32 contextId) const;
      bool isSwitchExists(const uint32 id) const;
      bool isValidRegion(const uint32 regionId) const;
      bool isAppExistsInApplicationsContextInfo(const uint32 id) const;
      bool isUpdateRequest(const ContextSwitchRequestInfo& info) const;
      bool isForwardRequest(const ContextSwitchRequestInfo& info) const;
      bool isBackwardRequest(const ContextSwitchRequestInfo& info) const;
      bool isCompleteRequest(const ContextSwitchRequestInfo& info) const;
      bool isBackgroundCompleteRequest(const ContextSwitchRequestInfo& info) const;
      bool updateToApplicationsContextInfo(const ApplicationContextInfo& info);
      bool performQueuedStoreContextRequest(const RequestBase* imp, bool& terminate);
      bool performQueuedContextSwitchRequest(const RequestBase* imp, bool& terminate);
      void clearQueuedRequestsInfo();
      void initSwitchRequestFunctionsInfo();
      void removeFromSwitchStackInfo(const uint32 id);
      void insertToSwitchStackInfo(const ApplicationContextInfo& info);
      void addToQueuedRequestsInfo(const ApplicationContextInfo& info);
      void addToQueuedRequestsInfo(const ContextSwitchRequestInfo& info);
      void performStackRequest();
      void performConsistencySwitch();
      void performDefaultApplicationRequest();
      void performUpdateRequest(const ContextSwitchRequestInfo& info);
      void performSwitchRequest(const ContextSwitchRequestInfo& info);
      void performCompleteRequest(const ContextSwitchRequestInfo& info);
      void performBackwardRequest(const ContextSwitchRequestInfo& info);
      void performBackwardAppRequest(const ContextSwitchRequestInfo& info);
      void sendRegionInfo(const ApplicationContextInfo& info);
      void sendSwitchAppRequest(const ApplicationContextInfo& info);
      void sendActiveContextUpdate(const ApplicationContextInfo& info);
      void sendEvalContextSwitchRequest(const ContextSwitchRequestInfo& info);
      void processQueuedRequests();
      void switchLastApplication();
      void startSafetyTimer(const uint32 period = 0xBB8);
      void startLastAppcationDebounceTimer(const uint32 period = 0x3E8);
      void evalAndClearContextsInfo(const ApplicationContextInfo& info);
      void executeContextSwitchRequest(const ContextSwitchRequestInfo& info);
      void executeBackgroundContextSwitchRequest(const ContextSwitchRequestInfo& info);
      bool isRequestForActiveContext(const uint32 contextId) const;

      inline ApplicationContextInfo getApplicationContextInfo(const ContextSwitchRequestInfo& info) const
      {
         ApplicationContextInfo tInfo;
         tInfo.setRegionId(info.getRegionId());
         tInfo.setContextId(info.getTargetContextId());
         tInfo.setApplicationId(info.getTargetApplicationId());
         tInfo.setPriority(getContextPriority(info.getTargetContextId()));
         return tInfo;
      }

      inline bool isStackRequestValid() const
      {
         bool isValid = (!_stackInfo.empty()) ? true : false;
         return isValid;
      }

      inline bool isBackgroundRequestValid() const
      {
         bool isValid = (NULL != _backgroundRequestInfo) ? true : false;
         return isValid;
      }

      inline bool isLowerPriority(const uint8 priority) const
      {
         bool isValid = false;
         ApplicationContextInfo tInfo;
         if ((fetchActiveContextInfo(tInfo)) && (priority > tInfo.getPriority()))
         {
            isValid = true;
         }
         return isValid;
      }

      inline bool isActiveApplicationSame(const uint32 id) const
      {
         bool isValid = ((_activeApplicationId != 0) && (_activeApplicationId == id)) ? true : false;
         return isValid;
      }

      inline bool isRequestForActiveApplication(const uint32 appId) const
      {
         bool isValid = ((!_stackInfo.empty()) && (_stackInfo.back() == appId)) ? true : false;
         return isValid;
      }

      inline bool isBackwardAppRequest(const ContextSwitchRequestInfo& info) const
      {
         ApplicationContextInfo tInfo;
         bool isValid = ((info.getTargetApplicationId() != 0) && (info.getTargetContextId() == 0)) ? true : false;
         isValid      = ((isValid) && (fetchActiveContextInfo(info.getTargetApplicationId(), tInfo))) ? true : false;
         return isValid;
      }

      inline void clear()
      {
         _activeApplicationId              = 0;
         _rootContextRegionId              = REGION_COCKPIT;
         _regionHandling                   = NULL;
         _hmiStateHandler                  = NULL;
         _rootContextHandler               = NULL;
         _contextPriorityHandler           = NULL;
         _applicationSurfaceInfoHandler    = NULL;
         _applicationSwitchServerComponent = NULL;
         _stackInfo.clear();
         _switchInfo.clear();
         _contextInfo.clear();
         _defaultContextInfo.clear();
         _contextListenersInfo.clear();
         _switchRequestFunctionsInfo.clear();
         stopSafetyTimer(true);
         stopLastAppcationDebounceTimer(true);
         clearQueuedRequestsInfo();
         clearBackgroundRequestInfo();
      }

      inline void clearBackgroundRequestInfo()
      {
         if (isBackgroundRequestValid())
         {
            delete _backgroundRequestInfo;
            _backgroundRequestInfo = NULL;
         }
      }

      inline void stopSafetyTimer(const bool flush = false)
      {
         if (NULL != _safetyTimer)
         {
            _safetyTimer->stop();
            if (flush)
            {
               delete _safetyTimer;
               _safetyTimer = NULL;
            }
         }
      }

      inline void loadSwitch(const ApplicationContextInfo& info)
      {
         _switchInfo.clear();
         _switchInfo.setSwitchState(CONTEXT_SWITCH_STATE_PROCESS);
         _switchInfo.setApplicationContextInfo(info);
      }

      inline void stopLastAppcationDebounceTimer(const bool flush = false)
      {
         if (NULL != _lastApplicationDebounceTimer)
         {
            _lastApplicationDebounceTimer->stop();
            if (flush)
            {
               delete _lastApplicationDebounceTimer;
               _lastApplicationDebounceTimer = NULL;
            }
         }
      }

      inline void updateBackgroundRequestInfo(const ContextSwitchRequestInfo& info)
      {
         clearBackgroundRequestInfo();
         _backgroundRequestInfo = new ContextSwitchRequestInfo(info);
      }

      inline void performBackgroundCompleteRequest(const ContextSwitchRequestInfo& /* info */)
      {
         clearBackgroundRequestInfo();
      }
      /**
       *  Member Variables Declaration
       */
      uint32 _rootContextRegionId;
      uint32 _activeApplicationId;
      SwitchStackInfo _stackInfo;
      ContextSwitchInfo _switchInfo;
      IRegionHandling* _regionHandling;
      IHmiStateHandler* _hmiStateHandler;
      ApplicationsContextInfo _contextInfo;
      QueuedRequestsInfo _queuedRequestsInfo;
      IRootContextHandler* _rootContextHandler;
      ApplicationsContextInfo _defaultContextInfo;
      ActiveContextListenersInfo _contextListenersInfo;
      IContextPriorityHandler* _contextPriorityHandler;
      ContextSwitchRequestInfo* _backgroundRequestInfo;
      SwitchRequestFunctionsInfo _switchRequestFunctionsInfo;
      IApplicationSurfaceInfoHandler* _applicationSurfaceInfoHandler;
      IApplicationSwitchServerComponent* _applicationSwitchServerComponent;
      ::asf::core::Timer* _safetyTimer;
      ::asf::core::Timer* _lastApplicationDebounceTimer;
};


} //namespace Core
} //namespace App


#endif /* CONTEXT_SWITCH_HANDLER_H */
