/**
 * @file         : ContextSwitchHandler.cpp
 * @author       : INF4CV - AppHmi_Master Team
 * @addtogroup   : AppHmi_Master
 * @brief        : ContextSwitchHandler is to handle the context based
 *                 application switch.
 * @copyright    : (c) 2020-2020 Robert Bosch Car Multimedia GmbH
 *                 The reproduction, distribution and utilization of this file as
 *                 well as the communication of its contents to others without express
 *                 authorization is prohibited. Offenders will be held liable for the
 *                 payment of damages. All rights reserved in the event of the grant
 *                 of a patent, utility model or design.
 */


#include "ContextSwitchHandler.h"
#include "hmi_trace_if.h"
#include "AppHmi_MasterMessages.h"
#include "ApplicationSwitchConst.h"
#include <Core/RegionHandling/RegionHandlingInterface.h>
#include <Core/RootContextHandler/RootContextHandlerInterface.h>
#include <Core/ContextPriorityHandler/ContextPriorityHandlerInterface.h>
#include <Core/ApplicationSurfaceInfoHandler/ApplicationSurfaceInfoHandlerInterface.h>
#include <Core/ApplicationSwitchServerComponent/ApplicationSwitchServerComponentInterface.h>
#include <Core/HmiState/HmiStateHandlerInterface.h>

using namespace ::asf::core;
using namespace ::bosch::cm::ai::hmi::hmimasterservice::ApplicationSwitch;


#ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
#define ETG_DEFAULT_TRACE_CLASS TR_CLASS_APPHMI_MASTER_MAIN
#include "trcGenProj/Header/ContextSwitchHandler.cpp.trc.h"
#endif


namespace App {
namespace Core {


ContextSwitchHandler::ContextSwitchHandler() : _backgroundRequestInfo(NULL)
   , _safetyTimer(NULL)
   , _lastApplicationDebounceTimer(NULL)
{
   clear();
   initSwitchRequestFunctionsInfo();
}


ContextSwitchHandler::~ContextSwitchHandler()
{
   clear();
}


ContextSwitchHandler::ContextSwitchHandler(const ContextSwitchHandler& obj) : _backgroundRequestInfo(NULL)
   , _safetyTimer(NULL)
   , _lastApplicationDebounceTimer(NULL)
{
   clear();
   *this = obj;
}


ContextSwitchHandler& ContextSwitchHandler::operator= (const ContextSwitchHandler& obj)
{
   if (this != (&obj))
   {
      clear();
      _regionHandling                   = obj._regionHandling;
      _hmiStateHandler                  = obj._hmiStateHandler;
      _rootContextHandler               = obj._rootContextHandler;
      _contextListenersInfo             = obj._contextListenersInfo;
      _contextPriorityHandler           = obj._contextPriorityHandler;
      _switchRequestFunctionsInfo       = obj._switchRequestFunctionsInfo;
      _applicationSurfaceInfoHandler    = obj._applicationSurfaceInfoHandler;
      _applicationSwitchServerComponent = obj._applicationSwitchServerComponent;
   }
   return (*this);
}


bool ContextSwitchHandler::updateToApplicationsContextInfo(const ApplicationContextInfo& info)
{
   bool isUpdated = false;
   if (!isAppExistsInApplicationsContextInfo(info.getApplicationId()))
   {
      isUpdated = true;
      _contextInfo.push_back(info);
   }
   else
   {
      for (ApplicationsContextInfo::iterator itr = _contextInfo.begin(); (itr != _contextInfo.end()); ++itr)
      {
         if (((*itr).getApplicationId() == info.getApplicationId()) && (!((*itr) == info)))
         {
            (*itr)    = info;
            isUpdated = true;
            break;
         }
      }
   }
   return isUpdated;
}


void ContextSwitchHandler::addActiveContextListener(IActiveContextListenerCB& imp)
{
   ActiveContextListenersInfo::const_iterator itr = ::std::find(_contextListenersInfo.begin(), _contextListenersInfo.end(), (&imp));
   if (itr == _contextListenersInfo.end())
   {
      _contextListenersInfo.push_back((&imp));
   }
}


void ContextSwitchHandler::removeActiveContextListener(IActiveContextListenerCB& imp)
{
   ActiveContextListenersInfo::iterator itr = ::std::find(_contextListenersInfo.begin(), _contextListenersInfo.end(), (&imp));
   if (itr != _contextListenersInfo.end())
   {
      _contextListenersInfo.erase(itr);
   }
}


void ContextSwitchHandler::clearQueuedRequestsInfo()
{
   QueuedRequestsInfo::iterator itr = _queuedRequestsInfo.begin();
   while (itr != _queuedRequestsInfo.end())
   {
      if (NULL != (*itr))
      {
         delete (*itr);
         (*itr) = NULL;
      }
      itr = _queuedRequestsInfo.erase(itr);
   }
}


void ContextSwitchHandler::initSwitchRequestFunctionsInfo()
{
   if (_switchRequestFunctionsInfo.empty())
   {
      _switchRequestFunctionsInfo.push_back(SwitchRequestFunctionInfo((&ContextSwitchHandler::isBackgroundCompleteRequest), (&ContextSwitchHandler::performBackgroundCompleteRequest)));
      _switchRequestFunctionsInfo.push_back(SwitchRequestFunctionInfo((&ContextSwitchHandler::isUpdateRequest), (&ContextSwitchHandler::performUpdateRequest)));
      _switchRequestFunctionsInfo.push_back(SwitchRequestFunctionInfo((&ContextSwitchHandler::isForwardRequest), (&ContextSwitchHandler::sendEvalContextSwitchRequest)));
      _switchRequestFunctionsInfo.push_back(SwitchRequestFunctionInfo((&ContextSwitchHandler::isBackwardRequest), (&ContextSwitchHandler::sendEvalContextSwitchRequest)));
      _switchRequestFunctionsInfo.push_back(SwitchRequestFunctionInfo((&ContextSwitchHandler::isCompleteRequest), (&ContextSwitchHandler::performCompleteRequest)));
   }
}


void ContextSwitchHandler::removeFromSwitchStackInfo(const uint32 id)
{
   SwitchStackInfo::iterator itr = ::std::find(_stackInfo.begin(), _stackInfo.end(), id);
   if (itr != _stackInfo.end())
   {
      (void)_stackInfo.erase(itr);
   }
}


void ContextSwitchHandler::insertToSwitchStackInfo(const ApplicationContextInfo& info)
{
   for (SwitchStackInfo::iterator itr = _stackInfo.begin(); (itr != _stackInfo.end()); ++itr)
   {
      ApplicationContextInfo tInfo;
      if ((fetchActiveContextInfo((*itr), tInfo)) && (info.getPriority() > tInfo.getPriority()))
      {
         _stackInfo.insert(itr, info.getApplicationId());
         break;
      }
   }
}


void ContextSwitchHandler::addToQueuedRequestsInfo(const ApplicationContextInfo& info)
{
   _queuedRequestsInfo.push_back(new ApplicationContextInfo(info));
}


void ContextSwitchHandler::addToQueuedRequestsInfo(const ContextSwitchRequestInfo& info)
{
   _queuedRequestsInfo.push_back(new ContextSwitchRequestInfo(info));
}


void ContextSwitchHandler::onStoreContext(const ApplicationContextInfo& info)
{
   ETG_TRACE_USR4(("ContextSwitchHandler: onStoreContext: <-: RegionId: %d", ETG_CENUM(Region, info.getRegionId())));
   ETG_TRACE_USR4(("ContextSwitchHandler: onStoreContext: <-: ContextId: %d", ETG_CENUM(enActivityIDs, info.getContextId())));
   ETG_TRACE_USR4(("ContextSwitchHandler: onStoreContext: <-: ApplicationId: %d", ETG_CENUM(enApplicationId, info.getApplicationId())));
   if (isValidRegion(info.getRegionId()))
   {
      ApplicationContextInfo tInfo(info);
      tInfo.setPriority(getContextPriority(info.getContextId()));
      if (!_switchInfo.isSwitchOngoing())
      {
         bool isUpdated = updateToApplicationsContextInfo(tInfo);
         if ((isUpdated) && (isRequestForActiveApplication(tInfo.getApplicationId())))
         {
            sendRegionInfo(tInfo);
            sendActiveContextUpdate(tInfo);
            evalAndClearContextsInfo(tInfo);
         }
      }
      else
      {
         ETG_TRACE_USR1(("ContextSwitchHandler: onStoreContext: ^: Switch is ongoing so added into queue"));
         addToQueuedRequestsInfo(tInfo);
      }
   }
   else
   {
      ETG_TRACE_ERR(("ContextSwitchHandler: onStoreContext: ^: Request is received for not supported region"));
   }
}


void ContextSwitchHandler::onContextSwitchRequest(const ContextSwitchRequestInfo& info)
{
   ETG_TRACE_USR1(("ContextSwitchHandler: onContextSwitchRequest: <-: RegionId: %d", ETG_CENUM(Region, info.getRegionId())));
   ETG_TRACE_USR1(("ContextSwitchHandler: onContextSwitchRequest: <-: SourceContextId: %d", ETG_CENUM(enActivityIDs, info.getSourceContextId())));
   ETG_TRACE_USR1(("ContextSwitchHandler: onContextSwitchRequest: <-: TargetContextId: %d", ETG_CENUM(enActivityIDs, info.getTargetContextId())));
   ETG_TRACE_USR1(("ContextSwitchHandler: onContextSwitchRequest: <-: SourceApplicationId: %d", ETG_CENUM(enApplicationId, info.getSourceApplicationId())));
   ETG_TRACE_USR1(("ContextSwitchHandler: onContextSwitchRequest: <-: TargetApplicationId: %d", ETG_CENUM(enApplicationId, info.getTargetApplicationId())));
   if (isValidRegion(info.getRegionId()))
   {
      if (!_switchInfo.isSwitchOngoing())
      {
         bool isPerformed = false;
         for (SwitchRequestFunctionsInfo::const_iterator itr = _switchRequestFunctionsInfo.begin(); (itr != _switchRequestFunctionsInfo.end()); ++itr)
         {
            bool isValid = (NULL != (*itr).first) ? true : false;
            isValid      = ((isValid) && (NULL != (*itr).second)) ? true : false;
            isValid      = ((isValid) && ((this->*((*itr).first))(info))) ? true : false;
            if (isValid)
            {
               (this->*((*itr).second))(info);
               isPerformed = true;
               break;
            }
         }
         if (!isPerformed)
         {
            ETG_TRACE_ERR(("ContextSwitchHandler: onContextSwitchRequest: ^: Invalid request"));
         }
      }
      else
      {
         ETG_TRACE_USR1(("ContextSwitchHandler: onContextSwitchRequest: ^: Switch is ongoing so added into queue"));
         addToQueuedRequestsInfo(info);
      }
   }
   else
   {
      ETG_TRACE_ERR(("ContextSwitchHandler: onContextSwitchRequest: ^: Request is received for not supported region"));
   }
}


void ContextSwitchHandler::onApplicationSwitchComplete(const uint32 appId)
{
   ETG_TRACE_USR4(("ContextSwitchHandler: onApplicationSwitchComplete: <-: AppId: %d", ETG_CENUM(enApplicationId, appId)));
   _activeApplicationId = appId;
   if ((NULL != _hmiStateHandler) && (_hmiStateHandler->isPowerStateON()))
   {
      if (_switchInfo.isSwitchOngoing())
      {
         const ApplicationContextInfo& tInfo = _switchInfo.getApplicationContextInfo();
         if (_activeApplicationId == tInfo.getApplicationId())
         {
            ETG_TRACE_USR1(("ContextSwitchHandler: onApplicationSwitchComplete: ^: Switch is completed: RegionId: %d", ETG_CENUM(Region, tInfo.getRegionId())));
            ETG_TRACE_USR1(("ContextSwitchHandler: onApplicationSwitchComplete: ^: Switch is completed: Priority: %d", tInfo.getPriority()));
            ETG_TRACE_USR1(("ContextSwitchHandler: onApplicationSwitchComplete: ^: Switch is completed: ContextId: %d", ETG_CENUM(enActivityIDs, tInfo.getContextId())));
            ETG_TRACE_USR1(("ContextSwitchHandler: onApplicationSwitchComplete: ^: Switch is completed: ApplicationId: %d", ETG_CENUM(enApplicationId, tInfo.getApplicationId())));
            _switchInfo.clear();
            stopSafetyTimer();
            dumpSwitchStackInfo();
            if (!_queuedRequestsInfo.empty())
            {
               ETG_TRACE_USR1(("ContextSwitchHandler: onApplicationSwitchComplete: ->: Process queued requests"));
               processQueuedRequests();
            }
         }
      }
      else
      {
         ApplicationContextInfo contextInfo;
         if ((fetchActiveContextInfo(contextInfo)) && (_activeApplicationId != contextInfo.getApplicationId()))
         {
            ETG_TRACE_USR1(("ContextSwitchHandler: onApplicationSwitchComplete: ^: Reactivate active application ApplicationId: %d", ETG_CENUM(enApplicationId, contextInfo.getApplicationId())));
            SwitchApplicationMsg* msg = COURIER_MESSAGE_NEW(SwitchApplicationMsg)(contextInfo.getApplicationId());
            if (NULL != msg)
            {
               msg->Post();
            }
         }
      }
   }
   else
   {
      if (APPID_APPHMI_MASTER != _activeApplicationId)
      {
         ETG_TRACE_USR1(("ContextSwitchHandler: onApplicationSwitchComplete: ^: Reactivate master application ApplicationId: %d", ETG_CENUM(enApplicationId, APPID_APPHMI_MASTER)));
         ActivateApplicationReqMsg* msg = COURIER_MESSAGE_NEW(ActivateApplicationReqMsg)(::hmibase::SURFACESTATE_VISIBLE, SURFACEID_MAIN_SURFACE_MASTER, APPID_APPHMI_MASTER);
         if (NULL != msg)
         {
            msg->Post();
         }
      }
   }
}


void ContextSwitchHandler::onSwitchLastApplication(const bool lastApplicationDebounceTimerStatus)
{
   ETG_TRACE_USR1(("ContextSwitchHandler: onSwitchLastApplication: <-: LastApplicationDebounceTimerStatus: %d", lastApplicationDebounceTimerStatus));
   if (lastApplicationDebounceTimerStatus)
   {
      startLastAppcationDebounceTimer();
   }
   else
   {
      stopLastAppcationDebounceTimer();
      switchLastApplication();
   }
}


void ContextSwitchHandler::onExecuteContextSwitchRequest(const ContextSwitchRequestInfo& info, const uint32 code)
{
   ETG_TRACE_USR1(("ContextSwitchHandler: onExecuteContextSwitchRequest: <-: RegionId: %d", ETG_CENUM(Region, info.getRegionId())));
   ETG_TRACE_USR1(("ContextSwitchHandler: onExecuteContextSwitchRequest: <-: SourceContextId: %d", ETG_CENUM(enActivityIDs, info.getSourceContextId())));
   ETG_TRACE_USR1(("ContextSwitchHandler: onExecuteContextSwitchRequest: <-: TargetContextId: %d", ETG_CENUM(enActivityIDs, info.getTargetContextId())));
   ETG_TRACE_USR1(("ContextSwitchHandler: onExecuteContextSwitchRequest: <-: SourceApplicationId: %d", ETG_CENUM(enApplicationId, info.getSourceApplicationId())));
   ETG_TRACE_USR1(("ContextSwitchHandler: onExecuteContextSwitchRequest: <-: TargetApplicationId: %d", ETG_CENUM(enApplicationId, info.getTargetApplicationId())));
   ETG_TRACE_USR1(("ContextSwitchHandler: onExecuteContextSwitchRequest: <-: RequestExecuteCode: %d", ETG_CENUM(enContextSwitchRequestExecuteCode, code)));
   switch (code)
   {
      case CONTEXT_SWITCH_REQUEST_EXECUTE_CODE_NONE:
      {
         executeContextSwitchRequest(info);
         break;
      }
      case CONTEXT_SWITCH_REQUEST_EXECUTE_CODE_BACKGROUND:
      {
         executeBackgroundContextSwitchRequest(info);
         break;
      }
      default:
         break;
   }
   if ((!_switchInfo.isSwitchOngoing()) && (!_queuedRequestsInfo.empty()))
   {
      ETG_TRACE_USR1(("ContextSwitchHandler: onExecuteContextSwitchRequest: ->: Process queued requests"));
      processQueuedRequests();
   }
}


void ContextSwitchHandler::onExpired(Timer& timer, ::boost::shared_ptr< TimerPayload > data)
{
   if ((NULL != _safetyTimer) && (_safetyTimer == (&timer)))
   {
      const ApplicationContextInfo& tInfo = _switchInfo.getApplicationContextInfo();
      ETG_TRACE_ERR(("ContextSwitchHandler: onExpired: <-: RegionId: %d", ETG_CENUM(Region, tInfo.getRegionId())));
      ETG_TRACE_ERR(("ContextSwitchHandler: onExpired: <-: Priority: %d", tInfo.getPriority()));
      ETG_TRACE_ERR(("ContextSwitchHandler: onExpired: <-: ContextId: %d", ETG_CENUM(enActivityIDs, tInfo.getContextId())));
      ETG_TRACE_ERR(("ContextSwitchHandler: onExpired: <-: ApplicationId: %d", ETG_CENUM(enApplicationId, tInfo.getApplicationId())));
      _switchInfo.clear();
      _stackInfo.pop_back();
      stopSafetyTimer();
      dumpSwitchStackInfo();
      if (!_queuedRequestsInfo.empty())
      {
         ETG_TRACE_USR1(("ContextSwitchHandler: onExpired: ->: Process queued requests"));
         processQueuedRequests();
      }
      if (!_switchInfo.isSwitchOngoing())
      {
         ETG_TRACE_USR1(("ContextSwitchHandler: onExpired: ->: Perform default application request"));
         performDefaultApplicationRequest();
      }
   }
   if ((NULL != _lastApplicationDebounceTimer) && (_lastApplicationDebounceTimer == (&timer)))
   {
      ETG_TRACE_USR4(("ContextSwitchHandler: onExpired: <-: LastApplicationDebounceTimer"));
      switchLastApplication();
   }
}


bool ContextSwitchHandler::performQueuedStoreContextRequest(const RequestBase* imp, bool& terminate)
{
   bool isPerformed                   = false;
   const ApplicationContextInfo* tImp = dynamic_cast< const ApplicationContextInfo* >(imp);
   if (NULL != tImp)
   {
      ApplicationContextInfo tInfo((*tImp));
      onStoreContext(tInfo);
      terminate   = false;
      isPerformed = true;
   }
   return isPerformed;
}


bool ContextSwitchHandler::performQueuedContextSwitchRequest(const RequestBase* imp, bool& terminate)
{
   bool isPerformed                     = false;
   const ContextSwitchRequestInfo* tImp = dynamic_cast< const ContextSwitchRequestInfo* >(imp);
   if (NULL != tImp)
   {
      ContextSwitchRequestInfo tInfo((*tImp));
      onContextSwitchRequest(tInfo);
      terminate   = (_switchInfo.isSwitchOngoing()) ? true : false;
      isPerformed = true;
   }
   return isPerformed;
}


void ContextSwitchHandler::performStackRequest()
{
   ApplicationContextInfo tInfo;
   if (fetchActiveContextInfo(tInfo))
   {
      ContextSwitchRequestInfo tRequestInfo;
      tRequestInfo.setRegionId(tInfo.getRegionId());
      tRequestInfo.setTargetContextId(tInfo.getContextId());
      tRequestInfo.setTargetApplicationId(tInfo.getApplicationId());
      performSwitchRequest(tRequestInfo);
   }
}


void ContextSwitchHandler::performConsistencySwitch()
{
   if (!_switchInfo.isSwitchOngoing())
   {
      if (isStackRequestValid())
      {
         performStackRequest();
      }
      else
      {
         performDefaultApplicationRequest();
      }
   }
}


void ContextSwitchHandler::performDefaultApplicationRequest()
{
   bool isLoaded   = false;
   ContextSwitchRequestInfo tRequestInfo;
   for (ApplicationsContextInfo::const_iterator itr = _defaultContextInfo.begin(); (itr != _defaultContextInfo.end()); ++itr)
   {
      if (itr == _defaultContextInfo.begin())
      {
         isLoaded = true;
         tRequestInfo.setRegionId((*itr).getRegionId());
         tRequestInfo.setTargetContextId((*itr).getContextId());
         tRequestInfo.setTargetApplicationId((*itr).getApplicationId());
      }
      else if (_rootContextRegionId == (*itr).getRegionId())
      {
         isLoaded = true;
         tRequestInfo.setRegionId((*itr).getRegionId());
         tRequestInfo.setTargetContextId((*itr).getContextId());
         tRequestInfo.setTargetApplicationId((*itr).getApplicationId());
         break;
      }
      else
      {}
   }
   if (isLoaded)
   {
      performSwitchRequest(tRequestInfo);
   }
   else
   {
      ETG_TRACE_ERR(("ContextSwitchHandler: performDefaultApplicationRequest: ^: No default application configured"));
   }
}


bool ContextSwitchHandler::fetchDefaultContextInfoForRegion(const uint32 regionId, ApplicationContextInfo& info) const
{
   bool isFetched = false;
   for (ApplicationsContextInfo::const_iterator itr = _defaultContextInfo.begin(); (itr != _defaultContextInfo.end()); ++itr)
   {
      if ((*itr).getRegionId() == regionId)
      {
         info      = (*itr);
         isFetched = true;
         break;
      }
   }
   return isFetched;
}


void ContextSwitchHandler::performUpdateRequest(const ContextSwitchRequestInfo& info)
{
   ApplicationContextInfo tInfo(getApplicationContextInfo(info));
   bool isUpdated = updateToApplicationsContextInfo(tInfo);
   if ((isUpdated) && (isRequestForActiveApplication(info.getTargetApplicationId())))
   {
      sendRegionInfo(tInfo);
      sendSwitchAppRequest(tInfo);
      sendActiveContextUpdate(tInfo);
      evalAndClearContextsInfo(tInfo);
   }
}


void ContextSwitchHandler::performSwitchRequest(const ContextSwitchRequestInfo& info)
{
   ApplicationContextInfo tInfo(getApplicationContextInfo(info));
   removeFromSwitchStackInfo(tInfo.getApplicationId());
   (void)updateToApplicationsContextInfo(tInfo);
   if (isLowerPriority(tInfo.getPriority()))
   {
      ETG_TRACE_USR4(("ContextSwitchHandler: performSwitchRequest: ^: Low priority context hence inserting in stack"));
      insertToSwitchStackInfo(tInfo);
      dumpSwitchStackInfo();
   }
   else
   {
      _stackInfo.push_back(tInfo.getApplicationId());
      sendRegionInfo(tInfo);
      sendSwitchAppRequest(tInfo);
      sendActiveContextUpdate(tInfo);
      evalAndClearContextsInfo(tInfo);
      if (!isActiveApplicationSame(tInfo.getApplicationId()))
      {
         loadSwitch(tInfo);
         startSafetyTimer();
      }
      else
      {
         dumpSwitchStackInfo();
      }
   }
}


void ContextSwitchHandler::performCompleteRequest(const ContextSwitchRequestInfo& info)
{
   removeFromSwitchStackInfo(info.getSourceApplicationId());
   performUpdateRequest(info);
}


void ContextSwitchHandler::performBackwardRequest(const ContextSwitchRequestInfo& info)
{
   _stackInfo.pop_back();
   if (isBackwardAppRequest(info))
   {
      performBackwardAppRequest(info);
   }
   else if (isBackgroundRequestValid())
   {
      ContextSwitchRequestInfo tInfo((*_backgroundRequestInfo));
      clearBackgroundRequestInfo();
      performSwitchRequest(tInfo);
   }
   else
   {}
   performConsistencySwitch();
}


void ContextSwitchHandler::performBackwardAppRequest(const ContextSwitchRequestInfo& info)
{
   bool isRequestForBackground = (isBackgroundRequestValid()) ? true : false;
   isRequestForBackground      = ((isRequestForBackground) && (_backgroundRequestInfo->getTargetApplicationId() == info.getTargetApplicationId())) ? true : false;
   if (isRequestForBackground)
   {
      ContextSwitchRequestInfo tInfo((*_backgroundRequestInfo));
      clearBackgroundRequestInfo();
      performSwitchRequest(tInfo);
   }
   else
   {
      ApplicationContextInfo tInfo;
      if (fetchActiveContextInfo(info.getTargetApplicationId(), tInfo))
      {
         ContextSwitchRequestInfo tRequestInfo;
         tRequestInfo.setRegionId(tInfo.getRegionId());
         tRequestInfo.setTargetContextId(tInfo.getContextId());
         tRequestInfo.setTargetApplicationId(tInfo.getApplicationId());
         performSwitchRequest(tRequestInfo);
      }
   }
}


void ContextSwitchHandler::sendRegionInfo(const ApplicationContextInfo& info)
{
   ETG_TRACE_USR4(("ContextSwitchHandler: sendRegionInfo: ->: RegionId: %d", ETG_CENUM(Region, info.getRegionId())));
   ETG_TRACE_USR4(("ContextSwitchHandler: sendRegionInfo: ->: Priority: %d", info.getPriority()));
   ETG_TRACE_USR4(("ContextSwitchHandler: sendRegionInfo: ->: ContextId: %d", ETG_CENUM(enActivityIDs, info.getContextId())));
   ETG_TRACE_USR4(("ContextSwitchHandler: sendRegionInfo: ->: ApplicationId: %d", ETG_CENUM(enApplicationId, info.getApplicationId())));
   if (NULL != _regionHandling)
   {
      _regionHandling->setRegionId(info.getRegionId());
   }
}


void ContextSwitchHandler::sendSwitchAppRequest(const ApplicationContextInfo& info)
{
   ETG_TRACE_USR1(("ContextSwitchHandler: sendSwitchAppRequest: ->: RegionId: %d", ETG_CENUM(Region, info.getRegionId())));
   ETG_TRACE_USR1(("ContextSwitchHandler: sendSwitchAppRequest: ->: Priority: %d", info.getPriority()));
   ETG_TRACE_USR1(("ContextSwitchHandler: sendSwitchAppRequest: ->: ContextId: %d", ETG_CENUM(enActivityIDs, info.getContextId())));
   ETG_TRACE_USR1(("ContextSwitchHandler: sendSwitchAppRequest: ->: ApplicationId: %d", ETG_CENUM(enApplicationId, info.getApplicationId())));
   SwitchApplicationMsg* msg = COURIER_MESSAGE_NEW(SwitchApplicationMsg)(info.getApplicationId());
   if (NULL != msg)
   {
      msg->Post();
   }
}


void ContextSwitchHandler::sendActiveContextUpdate(const ApplicationContextInfo& info)
{
   ETG_TRACE_USR4(("ContextSwitchHandler: sendActiveContextUpdate: ->: RegionId: %d", ETG_CENUM(Region, info.getRegionId())));
   ETG_TRACE_USR4(("ContextSwitchHandler: sendActiveContextUpdate: ->: Priority: %d", info.getPriority()));
   ETG_TRACE_USR4(("ContextSwitchHandler: sendActiveContextUpdate: ->: ContextId: %d", ETG_CENUM(enActivityIDs, info.getContextId())));
   ETG_TRACE_USR4(("ContextSwitchHandler: sendActiveContextUpdate: ->: ApplicationId: %d", ETG_CENUM(enApplicationId, info.getApplicationId())));
   if (NULL != _applicationSwitchServerComponent)
   {
      _applicationSwitchServerComponent->sendActivateContextSignal(info.getRegionId(), info.getApplicationId(), info.getContextId());
   }
   for (ActiveContextListenersInfo::const_iterator itr = _contextListenersInfo.begin(); (itr != _contextListenersInfo.end()); ++itr)
   {
      if (NULL != (*itr))
      {
         (*itr)->onActiveContextUpdate(info);
      }
   }
}


void ContextSwitchHandler::sendEvalContextSwitchRequest(const ContextSwitchRequestInfo& info)
{
   ETG_TRACE_USR1(("ContextSwitchHandler: sendEvalContextSwitchRequest: ->: RegionId: %d", ETG_CENUM(Region, info.getRegionId())));
   ETG_TRACE_USR1(("ContextSwitchHandler: sendEvalContextSwitchRequest: ->: SourceContextId: %d", ETG_CENUM(enActivityIDs, info.getSourceContextId())));
   ETG_TRACE_USR1(("ContextSwitchHandler: sendEvalContextSwitchRequest: ->: TargetContextId: %d", ETG_CENUM(enActivityIDs, info.getTargetContextId())));
   ETG_TRACE_USR1(("ContextSwitchHandler: sendEvalContextSwitchRequest: ->: SourceApplicationId: %d", ETG_CENUM(enApplicationId, info.getSourceApplicationId())));
   ETG_TRACE_USR1(("ContextSwitchHandler: sendEvalContextSwitchRequest: ->: TargetApplicationId: %d", ETG_CENUM(enApplicationId, info.getTargetApplicationId())));
   EvalContextSwitchRequestMsg* msg = COURIER_MESSAGE_NEW(EvalContextSwitchRequestMsg)(0, 0, 0, 0, 0);
   if (NULL != msg)
   {
      msg->SetRegionId(info.getRegionId());
      msg->SetSourceAppId(info.getSourceApplicationId());
      msg->SetTargetAppId(info.getTargetApplicationId());
      msg->SetSourceActivityId(info.getSourceContextId());
      msg->SetTargetActivityId(info.getTargetContextId());
      msg->Post();
   }
}


bool ContextSwitchHandler::fetchActiveContextInfo(ApplicationContextInfo& info) const
{
   bool isFetched = false;
   if (!_stackInfo.empty())
   {
      isFetched = fetchActiveContextInfo(_stackInfo.back(), info);
   }
   return isFetched;
}


bool ContextSwitchHandler::fetchActiveContextInfo(const uint32 appId, ApplicationContextInfo& info) const
{
   bool isFetched = false;
   for (ApplicationsContextInfo::const_iterator itr = _contextInfo.begin(); (itr != _contextInfo.end()); ++itr)
   {
      if ((*itr).getApplicationId() == appId)
      {
         info      = (*itr);
         isFetched = true;
         break;
      }
   }
   return isFetched;
}


void ContextSwitchHandler::dumpSwitchStackInfo()
{
   ETG_TRACE_USR1(("\n"
                   "//----------------------------------------------------------------------------------------------------------//""\n"
                   "                                                 STACK INFO ""\n"
                   "//----------------------------------------------------------------------------------------------------------//""\n"));
   ETG_TRACE_USR1(("\n"
                   "   ACTIVE APP ID : %d""\n"
                   "  ------------------------------------------------------------------------------------------------""\n", \
                   ETG_CENUM(enApplicationId, _activeApplicationId)));
   for (SwitchStackInfo::const_reverse_iterator itr = _stackInfo.rbegin(); (itr != _stackInfo.rend()); ++itr)
   {
      ApplicationContextInfo tInfo;
      (void)fetchActiveContextInfo((*itr), tInfo);
      ETG_TRACE_USR1(("\n"
                      "   APP ID     : %d""\n"
                      "   CONTEXT ID : %d""\n"
                      "   PRIORITY   : %d""\n"
                      "  ------------------------------------------------------------------------------------------------""\n", \
                      ETG_CENUM(enApplicationId, (*itr)), ETG_CENUM(enActivityIDs, tInfo.getContextId()), tInfo.getPriority()));
   }
   if (NULL != _backgroundRequestInfo)
   {
      ETG_TRACE_USR1(("\n"
                      "   BACKGROUND REQUEST :""\n"
                      "//----------------------------------------------------------------------------------------------------------//""\n"));
      ETG_TRACE_USR1(("\n"
                      "   APP ID     : %d""\n"
                      "   CONTEXT ID : %d""\n"
                      "  ------------------------------------------------------------------------------------------------""\n", \
                      ETG_CENUM(enApplicationId, _backgroundRequestInfo->getTargetApplicationId()), ETG_CENUM(enActivityIDs, _backgroundRequestInfo->getTargetContextId())));
   }
}


void ContextSwitchHandler::processQueuedRequests()
{
   QueuedRequestsInfo::iterator itr = _queuedRequestsInfo.begin();
   while (itr != _queuedRequestsInfo.end())
   {
      bool terminate   = false;
      bool isPerformed = (performQueuedStoreContextRequest((*itr), terminate)) ? true : false;
      isPerformed      = ((!isPerformed) && (performQueuedContextSwitchRequest((*itr), terminate))) ? true : false;
      delete (*itr);
      (*itr) = NULL;
      itr    = _queuedRequestsInfo.erase(itr);
      if (terminate)
      {
         break;
      }
   }
}


void ContextSwitchHandler::switchLastApplication()
{
   if (!(_switchInfo.isSwitchOngoing()))
   {
      if (isBackgroundRequestValid())
      {
         ContextSwitchRequestInfo tInfo((*_backgroundRequestInfo));
         clearBackgroundRequestInfo();
         performSwitchRequest(tInfo);
      }
      performConsistencySwitch();
   }
   else
   {
      ETG_TRACE_USR1(("ContextSwitchHandler: switchLastApplication: ^: Switch is ongoing"));
   }
}


void ContextSwitchHandler::startSafetyTimer(const uint32 period)
{
   if (NULL == _safetyTimer)
   {
      _safetyTimer = new ::asf::core::Timer((*this), period);
   }
   if (NULL != _safetyTimer)
   {
      (void)_safetyTimer->start();
   }
}


void ContextSwitchHandler::startLastAppcationDebounceTimer(const uint32 period)
{
   if (NULL == _lastApplicationDebounceTimer)
   {
      _lastApplicationDebounceTimer = new ::asf::core::Timer((*this), period);
   }
   if (NULL != _lastApplicationDebounceTimer)
   {
      (void)_lastApplicationDebounceTimer->start();
   }
}


void ContextSwitchHandler::evalAndClearContextsInfo(const ApplicationContextInfo& info)
{
   bool isValid = (!_stackInfo.empty()) ? true : false;
   isValid      = ((isValid) && (NULL != _rootContextHandler)) ? true : false;
   isValid      = ((isValid) && (_rootContextHandler->isContextValidToClearStack(info.getContextId()))) ? true : false;
   if (isValid)
   {
      _rootContextRegionId = info.getRegionId();
      _stackInfo.clear();
      _stackInfo.push_back(info.getApplicationId());
   }
}


void ContextSwitchHandler::executeContextSwitchRequest(const ContextSwitchRequestInfo& info)
{
   if (!_switchInfo.isSwitchOngoing())
   {
      stopLastAppcationDebounceTimer();
      if (isForwardRequest(info))
      {
         performSwitchRequest(info);
      }
      else if (isBackwardRequest(info))
      {
         performBackwardRequest(info);
      }
      else
      {
         ETG_TRACE_ERR(("ContextSwitchHandler: executeContextSwitchRequest: ^: Invalid request"));
      }
   }
   else
   {
      addToQueuedRequestsInfo(info);
   }
}


void ContextSwitchHandler::executeBackgroundContextSwitchRequest(const ContextSwitchRequestInfo& info)
{
   if (isForwardRequest(info))
   {
      updateBackgroundRequestInfo(info);
   }
   else if (isBackwardRequest(info))
   {
      _stackInfo.pop_back();
      ApplicationContextInfo tInfo;
      if (fetchActiveContextInfo(tInfo))
      {
         sendRegionInfo(tInfo);
         sendActiveContextUpdate(tInfo);
         evalAndClearContextsInfo(tInfo);
      }
      dumpSwitchStackInfo();
   }
   else
   {
      ETG_TRACE_ERR(("ContextSwitchHandler: executeBackgroundContextSwitchRequest: ^: Invalid request"));
   }
}


void ContextSwitchHandler::onClearLastApplication()
{
   _stackInfo.clear();
   clearBackgroundRequestInfo();
}


uint8 ContextSwitchHandler::getContextPriority(const uint32 contextId) const
{
   uint8 tPriority = 0xFF;
   if (NULL != _contextPriorityHandler)
   {
      tPriority = _contextPriorityHandler->getContextPriority(contextId);
   }
   return tPriority;
}


bool ContextSwitchHandler::isSwitchExists(const uint32 id) const
{
   bool isValid = false;
   SwitchStackInfo::const_iterator itr = ::std::find(_stackInfo.begin(), _stackInfo.end(), id);
   if (itr != _stackInfo.end())
   {
      isValid = true;
   }
   return isValid;
}


bool ContextSwitchHandler::isValidRegion(const uint32 regionId) const
{
   bool isValid = false;
   if ((NULL != _regionHandling) && (_regionHandling->isRegionSupported(regionId)))
   {
      isValid = true;
   }
   return isValid;
}


bool ContextSwitchHandler::isAppExistsInApplicationsContextInfo(const uint32 appId) const
{
   bool isValid = false;
   for (ApplicationsContextInfo::const_iterator itr = _contextInfo.begin(); (itr != _contextInfo.end()); ++itr)
   {
      if ((*itr).getApplicationId() == appId)
      {
         isValid = true;
         break;
      }
   }
   return isValid;
}


bool ContextSwitchHandler::isUpdateRequest(const ContextSwitchRequestInfo& info) const
{
   bool isValid = (info.getSourceApplicationId() == info.getTargetApplicationId()) ? true : false;
   return isValid;
}


bool ContextSwitchHandler::isForwardRequest(const ContextSwitchRequestInfo& info) const
{
   bool isValid = (info.getTargetApplicationId() != 0) ? true : false;
   isValid      = ((isValid) && (info.getTargetContextId() != 0)) ? true : false;
   return isValid;
}


bool ContextSwitchHandler::isBackwardRequest(const ContextSwitchRequestInfo& info) const
{
   bool isValid = (info.getSourceApplicationId() != 0) ? true : false;
   isValid      = ((isValid) && (isRequestForActiveApplication(info.getSourceApplicationId()))) ? true : false;
   isValid      = ((isValid) && ((info.getSourceContextId() == 0) || isRequestForActiveContext(info.getSourceContextId()))) ? true : false;
   isValid      = ((isValid) && ((info.getTargetApplicationId() == 0) || (info.getTargetContextId() == 0))) ? true : false;
   return isValid;
}


bool ContextSwitchHandler::isCompleteRequest(const ContextSwitchRequestInfo& info) const
{
   bool isValid = (info.getSourceApplicationId() != 0) ? true : false;
   isValid      = ((isValid) && (isSwitchExists(info.getSourceApplicationId()))) ? true : false;
   isValid      = ((isValid) && (!isRequestForActiveApplication(info.getSourceApplicationId()))) ? true : false;
   return isValid;
}


bool ContextSwitchHandler::isBackgroundCompleteRequest(const ContextSwitchRequestInfo& info) const
{
   bool isValid = (NULL != _backgroundRequestInfo) ? true : false;
   isValid      = ((isValid) && (info.getSourceApplicationId() != 0)) ? true : false;
   isValid      = ((isValid) && (_backgroundRequestInfo->getTargetApplicationId() == info.getSourceApplicationId())) ? true : false;
   isValid      = ((isValid) && ((info.getSourceContextId() == 0) || (info.getSourceContextId() == _backgroundRequestInfo->getTargetContextId()))) ? true : false;
   isValid      = ((isValid) && ((info.getTargetApplicationId() == 0) || (info.getTargetContextId() == 0))) ? true : false;
   return isValid;
}


bool ContextSwitchHandler::isRequestForActiveContext(const uint32 contextId) const
{
   bool isValid = false;
   ApplicationContextInfo tInfo;
   if (fetchActiveContextInfo(tInfo))
   {
      isValid = ((tInfo).getContextId() == contextId) ? true : false;
   }
   return isValid;
}


} //namespace Core
} //namespace App
