/**
 * @file         : VideoMixConnectionController.cpp
 * @author       : INF4CV - mve8cob
 * @addtogroup   : AppHmi_Master
 * @brief        : Controls connection/disconnection of cabin sources to
 *                 glass A2 sink
 * @copyright    : (C) 2016 Robert Bosch GmbH
 *                 (C) 2016 Robert Bosch Engineering and Business Solutions Limited
 *                 The reproduction, distribution and utilization of this file as
 *                 well as the communication of its contents to others without express
 *                 authorization is prohibited. Offenders will be held liable for the
 *                 payment of damages. All rights reserved in the event of the grant
 *                 of a patent, utility model or design.
 */

#include "VideoMixController.h"
#include "hmi_trace_if.h"
#include "CabinSrcInfo.h"
#include "CabinConnectionHandlerTypes.h"
#include <Core/CabinConnectionHandler/CabinConnectionUtility.h>


using namespace hmibase::apphmi_master::audio;


#ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
#define ETG_DEFAULT_TRACE_CLASS TR_CLASS_APPHMI_MASTER_MAIN
#include "trcGenProj/Header/VideoMixController.cpp.trc.h"
#endif


namespace App {
namespace Core {


VideoMixController::VideoMixController() : _cabinConnectionUtility(NULL)
{
   ETG_TRACE_USR4(("VideoMixController: CTOR"));
}


VideoMixController::~VideoMixController()
{
   ETG_TRACE_USR4(("VideoMixController: DTOR"));
   _cabinConnectionUtility = NULL;
}


bool VideoMixController::onSourceConnect(CabinConnectionHandlerInfo& info)
{
   const stSourceInfo& tSourceInfo(info.getSourceInfo());
   bool retVal           = false;
   bool isValidToProcess = (NULL != _cabinConnectionUtility) ? true : false;
   isValidToProcess      = ((isValidToProcess) && (!_cabinConnectionUtility->isConnectionAvailable(info))) ? true : false;
   isValidToProcess      = ((isValidToProcess) && (info.isActiveSinksInfoValid())) ? true : false;
   isValidToProcess      = ((isValidToProcess) && (isGlassSinkSupported(tSourceInfo.sinkId))) ? true : false;
   isValidToProcess      = ((isValidToProcess) && (_cabinConnectionUtility->isGlassSystemsValid(tSourceInfo.sinkId))) ? true : false;
   if (isValidToProcess)
   {
      retVal = true;
      ETG_TRACE_USR4(("VideoMixController: onSourceConnect: SourceId: %d, DeviceId: %d, SinkId: %d", tSourceInfo.srcId, tSourceInfo.deviceId, tSourceInfo.sinkId));
      SinkConnectionMap::const_iterator sItr = info.getActiveSinksInfo()->find(tSourceInfo.sinkId);
      if ((sItr != info.getActiveSinksInfo()->end()) && (_cabinConnectionUtility->isValidStateToDisconnect(sItr->second.connectionState)))
      {
         info.addToConnectionRequestsQueue(stSourceInfo(sItr->second.srcId, sItr->second.deviceId, STATE_DISCONNECT, sItr->second.sinkId));
      }
      info.addToConnectionRequestsQueue(stSourceInfo(tSourceInfo.srcId, tSourceInfo.deviceId, STATE_UNKNOWN, tSourceInfo.sinkId));
   }
   return retVal;
}


bool VideoMixController::onSourceDisconnect(CabinConnectionHandlerInfo& info)
{
   bool retVal = false;
   const stSourceInfo& tSourceInfo = info.getSourceInfo();
   if ((isGlassSinkSupported(tSourceInfo.sinkId)) && (CabinSrcInfo::isVideoSource(tSourceInfo.srcId)) && (NULL != _cabinConnectionUtility))
   {
      if (_cabinConnectionUtility->isConnectionAvailable(info))
      {
         ETG_TRACE_USR4(("VideoMixController: onSourceDisconnect: SourceId: %d, DeviceId: %d, SinkId: %d", tSourceInfo.srcId, tSourceInfo.deviceId, tSourceInfo.sinkId));
         retVal = true;
         info.addToConnectionRequestsQueue(stSourceInfo(tSourceInfo.srcId, tSourceInfo.deviceId, STATE_DISCONNECT, tSourceInfo.sinkId));
      }
      else
      {
         ETG_TRACE_USR4(("VideoMixController: onSourceDisconnect: No Active Connection for SourceId: %d, DeviceId: %d, SinkId: %d", tSourceInfo.srcId, tSourceInfo.deviceId, tSourceInfo.sinkId));
         _cabinConnectionUtility->sendSourceDisconnect(tSourceInfo.srcId, tSourceInfo.deviceId, tSourceInfo.sinkId);
      }
   }
   return retVal;
}


void VideoMixController::onNewConnectionUpdate(CabinConnectionHandlerInfo& info)
{
   const stSourceInfo& tSourceInfo = info.getSourceInfo();
   if (isGlassSinkSupported(tSourceInfo.sinkId))
   {
      ETG_TRACE_USR4(("VideoMixController: onNewConnectionUpdate: SourceId: %d, DeviceId: %d, SinkId: %d", tSourceInfo.srcId, tSourceInfo.deviceId, tSourceInfo.sinkId));
#ifdef VARIANT_S_FTR_ENABLE_MAP_STREAMING
      if (CabinSrcInfo::isVideoSource(tSourceInfo.srcId))
      {
         RegionMapOutInfo* imp = info.getMutableRegionMapOutInfo();
         if (NULL != imp)
         {
            const SinkMapOutInfo* tInfo = imp->getSinkMapOutInfo(tSourceInfo.sinkId);
            if (NULL != tInfo)
            {
               imp->removeFromSinksMapOutInfo(tSourceInfo.sinkId);
               info.addToModifiedMapSinks(tSourceInfo.sinkId);
            }
         }
      }
#endif /* VARIANT_S_FTR_ENABLE_MAP_STREAMING */
   }
}


#ifdef VARIANT_S_FTR_ENABLE_MAP_STREAMING
void VideoMixController::onMapOutActivation(CabinConnectionHandlerInfo& info)
{
   RegionMapOutInfo* imp = info.getMutableRegionMapOutInfo();
   if ((NULL != imp) && (info.isActiveSinksInfoValid()))
   {
      ETG_TRACE_USR4(("VideoMixController: onMapOutActivation: RegionId: %d", imp->getRegionId()));
      SinkIdsInfo tGlassSinkInfo = getSupportedGlassSinksInfo(imp->getRegionId());
      for (SinkIdsInfo::const_iterator itr = tGlassSinkInfo.begin(); (itr != tGlassSinkInfo.end()); ++itr)
      {
         if (imp->isSinkStateSame((*itr), STATE_CONNECTING))
         {
            imp->updateSinkState(STATE_CONNECTED, (*itr));
            ETG_TRACE_USR4(("VideoMixController: onMapOutActivation: Modified Sink: %d", (*itr)));
            info.addToModifiedMapSinks((*itr));
            //NOTE: Safe handling for the source request which is in progress
            RequestsQueueInfo* tInfo = info.getMutableConnectionRequestsQueueInfo();
            if (NULL != tInfo)
            {
               RequestsQueueInfo temp = (*tInfo);
               for (RequestsQueueInfo::iterator tItr = temp.begin(); (tItr != temp.end()); ++tItr)
               {
                  if ((*tItr).sinkId == (*itr))
                  {
                     if ((*tItr).connectionState != STATE_CONNECTING)
                     {
                        info.removeFromConnectionRequestsQueue((*tItr));
                     }
                     else
                     {
                        //TODO: Add rejection handling in FN: onNewConnectionUpdate() if map out is overrrided by camera -sve2cob
                        info.addToConnectionRequestsQueue(stSourceInfo((*tItr).srcId, (*tItr).deviceId, STATE_DISCONNECT, (*tItr).sinkId));
                     }
                  }
               }
            }
            SinkConnectionMap::const_iterator sItr = info.getActiveSinksInfo()->find((*itr));
            if ((sItr != info.getActiveSinksInfo()->end()) && (NULL != _cabinConnectionUtility) && (_cabinConnectionUtility->isValidStateToDisconnect(sItr->second.connectionState)))
            {
               info.addToConnectionRequestsQueue(stSourceInfo(sItr->second.srcId, sItr->second.deviceId, STATE_DISCONNECT, sItr->second.sinkId));
            }
         }
      }
   }
}


void VideoMixController::onMapOutDeactivation(CabinConnectionHandlerInfo& info)
{
   RegionMapOutInfo* imp = info.getMutableRegionMapOutInfo();
   if (NULL != imp)
   {
      ETG_TRACE_USR4(("VideoMixController: onMapOutDeactivation: RegionId: %d", imp->getRegionId()));
      SinkIdsInfo tGlassSinkInfo = getSupportedGlassSinksInfo(imp->getRegionId());
      for (SinkIdsInfo::const_iterator itr = tGlassSinkInfo.begin(); (itr != tGlassSinkInfo.end()); ++itr)
      {
         if (imp->isSinkStateSame((*itr), STATE_DISCONNECTING))
         {
            imp->removeFromSinksMapOutInfo((*itr));
            ETG_TRACE_USR4(("VideoMixController: onMapOutDeactivation: Removed Sink: %d", (*itr)));
            info.addToModifiedMapSinks((*itr));
         }
      }
   }
}


#endif


bool VideoMixController::isGlassSinkSupported(const int16 sinkId) const
{
   bool isValid = false;
   if (NULL != _cabinConnectionUtility)
   {
      isValid = _cabinConnectionUtility->isGlassSinkGroupCategorySame(sinkId, SYSTEM_GROUP_CATEGORY_CMG_CABIN_MIX);
   }
   ETG_TRACE_USR4(("VideoMixController: isGlassSinkSupported: isValid: %d", isValid));
   return isValid;
}


uint32 VideoMixController::getRegionIdForSinkId(const int16 sinkId) const
{
   uint32 regionId = REGION_INVALID;
   if (NULL != _cabinConnectionUtility)
   {
      regionId = _cabinConnectionUtility->getRegionIdForSinkId(sinkId);
   }
   return regionId;
}


SinkIdsInfo VideoMixController::getSupportedGlassSinksInfo(const uint32 regionId) const
{
   SinkIdsInfo tInfo;
   if (NULL != _cabinConnectionUtility)
   {
      tInfo = _cabinConnectionUtility->getGlassSinksInfo(regionId, SYSTEM_GROUP_CATEGORY_CMG_CABIN_MIX);
   }
   return tInfo;
}


} //namespace Core
} //namespace App
