/**
 * @file         : MonoGlassController.cpp
 * @author       : INF4CV - pgi5cob
 * @addtogroup   : AppHmi_Master
 * @brief        : Controls connection/disconnection of cabin sources to glass sink
 * @copyright    : (C) 2016 Robert Bosch GmbH
 *                 (C) 2016 Robert Bosch Engineering and Business Solutions Limited
 *                 The reproduction, distribution and utilization of this file as
 *                 well as the communication of its contents to others without express
 *                 authorization is prohibited. Offenders will be held liable for the
 *                 payment of damages. All rights reserved in the event of the grant
 *                 of a patent, utility model or design.
 */

#include "MonoGlassController.h"
#include "hmi_trace_if.h"
#include "CabinSrcInfo.h"
#include "CabinConnectionHandlerTypes.h"
#include "CabinRestorationHandlerInterface.h"
#include <datapool/CabinActivityInfoDPIF.h>
#include <Core/RegionHandling/RegionHandlingTypes.h>
#include <Core/RegionHandling/RegionHandlingInterface.h>
#include <Core/CabinConnectionHandler/CabinConnectionUtility.h>
#include <Core/HmiInfoService/HmiInfoServiceServerComponentHandlingInterface.h>


using namespace hmibase::apphmi_master::audio;


#ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
#define ETG_DEFAULT_TRACE_CLASS TR_CLASS_APPHMI_MASTER_MAIN
#include "trcGenProj/Header/MonoGlassController.cpp.trc.h"
#endif


namespace App {
namespace Core {


MonoGlassController::MonoGlassController() : _cabinConnectionUtility(NULL)
   , _hmiInfoServiceServerComponent(NULL)
{
   ETG_TRACE_USR4(("MonoGlassController: CTOR"));
}


MonoGlassController::~MonoGlassController()
{
   ETG_TRACE_USR4(("MonoGlassController: DTOR"));
   _monoModeInfo.clear();
   _cabinConnectionUtility        = NULL;
   _hmiInfoServiceServerComponent = NULL;
}


bool MonoGlassController::onSourceConnect(CabinConnectionHandlerInfo& info)
{
   bool retVal                     = false;
   const stSourceInfo& tSourceInfo = info.getSourceInfo();
   if ((NULL != _cabinConnectionUtility) && (!_cabinConnectionUtility->isConnectionAvailable(info)))
   {
      ETG_TRACE_USR4(("MonoGlassController: onSourceConnect: SourceId: %d, DeviceId: %d, SinkId: %d", tSourceInfo.srcId, tSourceInfo.deviceId, tSourceInfo.sinkId));
      switch (CabinSrcInfo::getSrcType(tSourceInfo.srcId))
      {
         case CabinSrcInfo::COCKPIT_SOURCE:
         case CabinSrcInfo::CABIN_AUDIO_SOURCE:
         {
            removeSinkFromAVRestoreInfo(tSourceInfo.sinkId);
            retVal = onAmplifierSourceRequest(info);
            break;
         }
         case CabinSrcInfo::CABIN_VIDEO_SOURCE:
         {
            retVal = onCameraSourceRequest(info);
            break;
         }
         case CabinSrcInfo::CABIN_AV_SOURCE:
         {
            removeSinkFromAVRestoreInfo(tSourceInfo.sinkId);
            retVal = onAVSourceRequest(info);
            break;
         }
         default:
         {
            info.addToConnectionRequestsQueue(tSourceInfo);
            retVal = true;
            break;
         }
      }
   }
   else
   {
      ETG_TRACE_USR4(("MonoGlassController: onSourceConnect: Source already connected: SourceId: %d, DeviceId: %d, SinkId: %d", tSourceInfo.srcId, tSourceInfo.deviceId, tSourceInfo.sinkId));
      //INFO: If amp has AV source then connect to glass as well
      retVal = onAvPausedConnectGlass(info);
   }
   return retVal;
}


bool MonoGlassController::onAmplifierSourceRequest(CabinConnectionHandlerInfo& info)
{
   bool retVal                     = false;
   const stSourceInfo& tSourceInfo = info.getSourceInfo();
   if ((NULL != _cabinConnectionUtility) && (_cabinConnectionUtility->isAmpSink(tSourceInfo.sinkId)) && (info.isActiveSinksInfoValid()))
   {
      ETG_TRACE_USR4(("MonoGlassController: onAmplifierSourceRequest: SourceId: %d, DeviceId: %d, SinkId: %d", tSourceInfo.srcId, tSourceInfo.deviceId, tSourceInfo.sinkId));
      retVal = true;
      if (tSourceInfo.srcId != SRC_MIC1_ANNOUNCEMENT)
      {
         uint32 regionId            = getRegionIdForSinkId(tSourceInfo.sinkId);
         SinkIdsInfo tGlassSinkInfo = getSupportedGlassSinksInfo(regionId);
         for (SinkIdsInfo::const_iterator itr = tGlassSinkInfo.begin(); (itr != tGlassSinkInfo.end()); ++itr)
         {
            //INFO: Video disconnection is sent for glasses which are connected AV source
            SinkConnectionMap::const_iterator sItr = info.getActiveSinksInfo()->find((*itr));
            if ((sItr != info.getActiveSinksInfo()->end()) && (CabinSrcInfo::isAVSource(sItr->second.srcId)) && (_cabinConnectionUtility->isValidStateToDisconnect(sItr->second.connectionState)))
            {
               info.addToConnectionRequestsQueue(stSourceInfo(sItr->second.srcId, sItr->second.deviceId, STATE_DISCONNECT, sItr->second.sinkId));
            }
         }
      }
      info.addToConnectionRequestsQueue(tSourceInfo);
   }
   return retVal;
}


bool MonoGlassController::onCameraSourceRequest(CabinConnectionHandlerInfo& info)
{
   bool retVal                     = false;
   const stSourceInfo& tSourceInfo = info.getSourceInfo();
   if ((isGlassSinkSupported(tSourceInfo.sinkId)) && (info.isActiveSinksInfoValid()))
   {
      ETG_TRACE_USR4(("MonoGlassController: onCameraSourceRequest: SourceId: %d, DeviceId: %d, SinkId: %d", tSourceInfo.srcId, tSourceInfo.deviceId, tSourceInfo.sinkId));
      retVal          = true;
      uint32 regionId = getRegionIdForSinkId(tSourceInfo.sinkId);
      /*
       * INFO: While switching from AV or analogue camera source, glasses disconnection should be sent
       *       before the connection request (Reason: Same pin in CMP)
       * NOTE: Amp and glasses are disconnected in general to avoid flicker and above mentioned restriction
       */
      SinkConnectionMap::const_iterator ampItr = info.getActiveSinksInfo()->find(getAmpSinkIdForRegionId(regionId));
      if ((ampItr != info.getActiveSinksInfo()->end()) && (CabinSrcInfo::isAVSource(ampItr->second.srcId)))
      {
         if ((NULL != _cabinConnectionUtility) && (_cabinConnectionUtility->isValidStateToDisconnect(ampItr->second.connectionState)))
         {
            info.addToConnectionRequestsQueue(stSourceInfo(ampItr->second.srcId, ampItr->second.deviceId, STATE_DISCONNECT, ampItr->second.sinkId));
         }
         //INFO: Remember AV source if it is disconnected due to restriction
         stSourceInfo rememberSrcInfo(ampItr->second.srcId, ampItr->second.deviceId, STATE_UNKNOWN, ampItr->second.sinkId);
         _AVRestoreInfo.insert(std::pair< int16, stSourceInfo >(ampItr->second.sinkId, rememberSrcInfo));
      }
      SinkIdsInfo tGlassSinkInfo = getSupportedGlassSinksInfo(regionId, true);
      for (SinkIdsInfo::const_iterator itr = tGlassSinkInfo.begin(); (itr != tGlassSinkInfo.end()); ++itr)
      {
         SinkConnectionMap::const_iterator sItr = info.getActiveSinksInfo()->find((*itr));
         if ((sItr != info.getActiveSinksInfo()->end()) && (NULL != _cabinConnectionUtility) && (_cabinConnectionUtility->isValidStateToDisconnect(sItr->second.connectionState)))
         {
            info.addToConnectionRequestsQueue(stSourceInfo(sItr->second.srcId, sItr->second.deviceId, STATE_DISCONNECT, sItr->second.sinkId));
         }
      }
      for (SinkIdsInfo::const_iterator itr = tGlassSinkInfo.begin(); (itr != tGlassSinkInfo.end()); ++itr)
      {
         info.addToConnectionRequestsQueue(stSourceInfo(tSourceInfo.srcId, tSourceInfo.deviceId, STATE_UNKNOWN, (*itr)));
      }
   }
   return retVal;
}


bool MonoGlassController::onAVSourceRequest(CabinConnectionHandlerInfo& info)
{
   bool retVal                     = false;
   const stSourceInfo& tSourceInfo = info.getSourceInfo();
   if ((_cabinConnectionUtility != NULL) && (_cabinConnectionUtility->isAmpSink(tSourceInfo.sinkId)) && (info.isActiveSinksInfoValid()))
   {
      ETG_TRACE_USR4(("MonoGlassController: onAVSourceRequest: SourceId: %d, DeviceId: %d, SinkId: %d", tSourceInfo.srcId, tSourceInfo.deviceId, tSourceInfo.sinkId));
      retVal          = true;
      uint32 regionId = getRegionIdForSinkId(tSourceInfo.sinkId);
      /*
       * INFO: While switching from AV or analogue camera source, glasses disconnection should be sent
       *       before the connection request (Reason: Same pin in CMP)
       * NOTE: Amp and glasses are disconnected in general to avoid flicker and above mentioned restriction
       */
      SinkIdsInfo tGlassSinkInfo = getSupportedGlassSinksInfo(regionId);
      for (SinkIdsInfo::const_iterator itr = tGlassSinkInfo.begin(); (itr != tGlassSinkInfo.end()); ++itr)
      {
         SinkConnectionMap::const_iterator sItr = info.getActiveSinksInfo()->find((*itr));
         if ((sItr != info.getActiveSinksInfo()->end()) && (_cabinConnectionUtility->isValidStateToDisconnect(sItr->second.connectionState)))
         {
            info.addToConnectionRequestsQueue(stSourceInfo(sItr->second.srcId, sItr->second.deviceId, STATE_DISCONNECT, sItr->second.sinkId));
         }
      }
      info.addToConnectionRequestsQueue(tSourceInfo);
   }
   return retVal;
}


bool MonoGlassController::onAvPausedConnectGlass(CabinConnectionHandlerInfo& info)
{
   bool retVal                     = false;
   const stSourceInfo& tSourceInfo = info.getSourceInfo();
   if ((NULL != _cabinConnectionUtility) && (_cabinConnectionUtility->isAmpSink(tSourceInfo.sinkId)) && (CabinSrcInfo::isAVSource(tSourceInfo.srcId)) && (info.isActiveSinksInfoValid()))
   {
      ETG_TRACE_USR4(("MonoGlassController: onAvPausedConnectGlass: SourceId: %d, DeviceId: %d, SinkId: %d", tSourceInfo.srcId, tSourceInfo.deviceId, tSourceInfo.sinkId));
      uint32 regionId = getRegionIdForSinkId(tSourceInfo.sinkId);
      SinkIdsInfo tGlassSinkInfo = getSupportedGlassSinksInfo(regionId, true);
      for (SinkIdsInfo::const_iterator itr = tGlassSinkInfo.begin(); (itr != tGlassSinkInfo.end()); ++itr)
      {
         SinkConnectionMap::const_iterator sItr = info.getActiveSinksInfo()->find((*itr));
         bool isValidToDisconnect = ((sItr != info.getActiveSinksInfo()->end()) && (!CabinSrcInfo::isAVSource(sItr->second.srcId)));
         isValidToDisconnect      = (isValidToDisconnect && (_cabinConnectionUtility->isValidStateToDisconnect(sItr->second.connectionState)));
         if (isValidToDisconnect)
         {
            retVal = true;
            info.addToConnectionRequestsQueue(stSourceInfo(sItr->second.srcId, sItr->second.deviceId, STATE_DISCONNECT, sItr->second.sinkId));
         }
      }
      for (SinkIdsInfo::const_iterator itr = tGlassSinkInfo.begin(); (itr != tGlassSinkInfo.end()); ++itr)
      {
         SinkConnectionMap::const_iterator sItr = info.getActiveSinksInfo()->find((*itr));
         if (sItr != info.getActiveSinksInfo()->end())
         {
            if (!CabinSrcInfo::isAVSource(sItr->second.srcId))
            {
               retVal = true;
               info.addToConnectionRequestsQueue(stSourceInfo(tSourceInfo.srcId, tSourceInfo.deviceId, STATE_UNKNOWN, (*itr)));
            }
         }
         else
         {
            retVal = true;
            info.addToConnectionRequestsQueue(stSourceInfo(tSourceInfo.srcId, tSourceInfo.deviceId, STATE_UNKNOWN, (*itr)));
         }
      }
   }
   return retVal;
}


bool MonoGlassController::onSourceDisconnect(CabinConnectionHandlerInfo& info)
{
   bool retVal                     = false;
   const stSourceInfo& tSourceInfo = info.getSourceInfo();
   if (NULL != _cabinConnectionUtility)
   {
      if (_cabinConnectionUtility->isConnectionAvailable(info))
      {
         ETG_TRACE_USR4(("MonoGlassController: onSourceDisconnect: SourceId: %d, DeviceId: %d, SinkId: %d", tSourceInfo.srcId, tSourceInfo.deviceId, tSourceInfo.sinkId));
         switch (CabinSrcInfo::getSrcType(tSourceInfo.srcId))
         {
            case CabinSrcInfo::CABIN_VIDEO_SOURCE:
            {
               retVal = onVideoSourceDisconnect(info);
               break;
            }
            default:
            {
               info.addToConnectionRequestsQueue(tSourceInfo);
               retVal = true;
               break;
            }
         }
      }
      else
      {
         ETG_TRACE_USR4(("MonoGlassController: onSourceDisconnect: No Active Connection for SourceId: %d, DeviceId: %d, SinkId: %d", tSourceInfo.srcId, tSourceInfo.deviceId, tSourceInfo.sinkId));
         _cabinConnectionUtility->sendSourceDisconnect(tSourceInfo.srcId, tSourceInfo.deviceId, tSourceInfo.sinkId);
      }
   }
   return retVal;
}


bool MonoGlassController::onVideoSourceDisconnect(CabinConnectionHandlerInfo& info)
{
   bool retVal                     = false;
   const stSourceInfo& tSourceInfo = info.getSourceInfo();
   ETG_TRACE_USR4(("MonoGlassController: onVideoSourceDisconnect: SourceId: %d, DeviceId: %d, SinkId: %d", tSourceInfo.srcId, tSourceInfo.deviceId, tSourceInfo.sinkId));
   if (isGlassSinkSupported(tSourceInfo.sinkId))
   {
      uint32 regionId = getRegionIdForSinkId(tSourceInfo.sinkId);
      if (info.isActiveSinksInfoValid())
      {
         SinkIdsInfo tGlassSinkInfo = getSupportedGlassSinksInfo(regionId);
         for (SinkIdsInfo::const_iterator itr = tGlassSinkInfo.begin(); (itr != tGlassSinkInfo.end()); ++itr)
         {
            SinkConnectionMap::const_iterator sItr = info.getActiveSinksInfo()->find((*itr));
            bool isValidToDisconnect = ((sItr != info.getActiveSinksInfo()->end()) && (tSourceInfo.srcId == sItr->second.srcId));
            isValidToDisconnect      = (isValidToDisconnect && (NULL != _cabinConnectionUtility) && (_cabinConnectionUtility->isValidStateToDisconnect(sItr->second.connectionState)));
            if (isValidToDisconnect)
            {
               retVal = true;
               info.addToConnectionRequestsQueue(stSourceInfo(tSourceInfo.srcId, tSourceInfo.deviceId, STATE_DISCONNECT, (*itr)));
            }
         }
      }
      //INFO: On deactivating camera if AV Source needs to be restored back
      if (retVal)
      {
#ifdef VARIANT_S_FTR_ENABLE_MAP_STREAMING
         const RegionMapOutInfo* imp = info.getRegionMapOutInfo();
         if (((NULL != imp) && (!imp->isSinkStateSame(tSourceInfo.sinkId, STATE_CONNECTED))) || (NULL == imp))
#endif
         {
            restoreAVSource(regionId);
         }
      }
   }
   return retVal;
}


void MonoGlassController::onNewConnectionUpdate(CabinConnectionHandlerInfo& info)
{
   const stSourceInfo& tSourceInfo = info.getSourceInfo();
   if (NULL != _cabinConnectionUtility)
   {
      if (_cabinConnectionUtility->isAmpSink(tSourceInfo.sinkId))
      {
         onNewConnectionUpdateForAmp(info);
      }
      else
      {
         onNewConnectionUpdateForGlass(info);
      }
   }
}


void MonoGlassController::onNewConnectionUpdateForAmp(CabinConnectionHandlerInfo& info)
{
   const stSourceInfo& tSourceInfo = info.getSourceInfo();
   ETG_TRACE_USR4(("MonoGlassController: onNewConnectionUpdateForAmp: SourceId: %d, DeviceId: %d, SinkId: %d", tSourceInfo.srcId, tSourceInfo.deviceId, tSourceInfo.sinkId));
   //INFO: Handles glasses connection for AV source
   CabinSrcInfo::SrcType newSrcType = CabinSrcInfo::getSrcType(tSourceInfo.srcId);
   if (CabinSrcInfo::CABIN_AV_SOURCE == newSrcType)
   {
      uint32 regionId            = getRegionIdForSinkId(tSourceInfo.sinkId);
      SinkIdsInfo tGlassSinkInfo = getSupportedGlassSinksInfo(regionId, true);
      CabinConnectionHandlerInfo tInfo = info;
      for (SinkIdsInfo::const_iterator itr = tGlassSinkInfo.begin(); (itr != tGlassSinkInfo.end()); ++itr)
      {
         tInfo.setSourceInfo(stSourceInfo(tSourceInfo.srcId, tSourceInfo.deviceId, STATE_UNKNOWN, (*itr)));
         if ((NULL != _cabinConnectionUtility) && (!_cabinConnectionUtility->isConnectionAvailable(tInfo)))
         {
            info.addToConnectionRequestsQueue(stSourceInfo(tSourceInfo.srcId, tSourceInfo.deviceId, STATE_UNKNOWN, (*itr)));
         }
      }
   }
}


void MonoGlassController::onNewConnectionUpdateForGlass(CabinConnectionHandlerInfo& info)
{
   const stSourceInfo& tSourceInfo = info.getSourceInfo();
   if (isGlassSinkSupported(tSourceInfo.sinkId))
   {
      ETG_TRACE_USR4(("MonoGlassController: onNewConnectionUpdateForGlass: SourceId: %d, DeviceId: %d, SinkId: %d", tSourceInfo.srcId, tSourceInfo.deviceId, tSourceInfo.sinkId));
#ifdef VARIANT_S_FTR_ENABLE_MAP_STREAMING
      CabinSrcInfo::SrcType newSrcType = CabinSrcInfo::getSrcType(tSourceInfo.srcId);
      if ((CabinSrcInfo::CABIN_AV_SOURCE == newSrcType) || (CabinSrcInfo::CABIN_VIDEO_SOURCE == newSrcType))
      {
         RegionMapOutInfo* imp = info.getMutableRegionMapOutInfo();
         if (NULL != imp)
         {
            const SinkMapOutInfo* tInfo = imp->getSinkMapOutInfo(tSourceInfo.sinkId);
            if (NULL != tInfo)
            {
               imp->removeFromSinksMapOutInfo(tSourceInfo.sinkId);
               info.addToModifiedMapSinks(tSourceInfo.sinkId);
            }
         }
      }
#endif /* VARIANT_S_FTR_ENABLE_MAP_STREAMING */
   }
}


void MonoGlassController::onConnectionRemoved(CabinConnectionHandlerInfo& info)
{
   const stSourceInfo& tSourceInfo = info.getSourceInfo();
   ETG_TRACE_USR4(("MonoGlassController: onConnectionRemoved: SourceId: %d, DeviceId: %d, SinkId: %d", tSourceInfo.srcId, tSourceInfo.deviceId, tSourceInfo.sinkId));
   //INFO: Disconnect glass if AV source connection to amplifier was removed and there are no more queued requests
   bool isValidToDisconnect = (info.isActiveSinksInfoValid()) ? true : false;
   isValidToDisconnect      = ((isValidToDisconnect) && (NULL != _cabinConnectionUtility)) ? true : false;
   isValidToDisconnect      = ((isValidToDisconnect) && (_cabinConnectionUtility->isAmpSink(tSourceInfo.sinkId))) ? true : false;
   isValidToDisconnect      = ((isValidToDisconnect) && (info.isConnectionRequestsQueueInfoValid())) ? true : false;
   isValidToDisconnect      = ((isValidToDisconnect) && (info.getConnectionRequestsQueueInfo()->empty())) ? true : false;
   isValidToDisconnect      = ((isValidToDisconnect) && (CabinSrcInfo::CABIN_AV_SOURCE == CabinSrcInfo::getSrcType(tSourceInfo.srcId))) ? true : false;
   if (isValidToDisconnect)
   {
      uint32 regionId            = getRegionIdForSinkId(tSourceInfo.sinkId);
      SinkIdsInfo tGlassSinkInfo = getSupportedGlassSinksInfo(regionId);
      for (SinkIdsInfo::const_iterator itr = tGlassSinkInfo.begin(); (itr != tGlassSinkInfo.end()); ++itr)
      {
         SinkConnectionMap::const_iterator sItr = info.getActiveSinksInfo()->find((*itr));
         bool isValidToDisconnect = ((sItr != info.getActiveSinksInfo()->end()) && (CabinSrcInfo::isAVSource(sItr->second.srcId)));
         isValidToDisconnect      = ((isValidToDisconnect) && (_cabinConnectionUtility->isValidStateToDisconnect(sItr->second.connectionState)));
         if (isValidToDisconnect)
         {
            info.addToConnectionRequestsQueue(stSourceInfo(sItr->second.srcId, sItr->second.deviceId, STATE_DISCONNECT, (*itr)));
         }
      }
   }
}


#ifdef VARIANT_S_FTR_ENABLE_MAP_STREAMING
void MonoGlassController::onMapOutActivation(CabinConnectionHandlerInfo& info)
{
   RegionMapOutInfo* imp = info.getMutableRegionMapOutInfo();
   if ((NULL != imp) && (info.isActiveSinksInfoValid()))
   {
      ETG_TRACE_USR4(("MonoGlassController: onMapOutActivation: RegionId: %d", imp->getRegionId()));
      bool isAmpUpdated          = false;
      SinkIdsInfo tGlassSinkInfo = getSupportedGlassSinksInfo(imp->getRegionId());
      for (SinkIdsInfo::const_iterator itr = tGlassSinkInfo.begin(); (itr != tGlassSinkInfo.end()); ++itr)
      {
         if (imp->isSinkStateSame((*itr), STATE_CONNECTING))
         {
            imp->updateSinkState(STATE_CONNECTED, (*itr));
            ETG_TRACE_USR4(("MonoGlassController: onMapOutActivation: Modified Sink: %d", (*itr)));
            info.addToModifiedMapSinks((*itr));
            if (!isAmpUpdated)
            {
               isAmpUpdated = true;
               SinkConnectionMap::const_iterator ampItr = info.getActiveSinksInfo()->find(getAmpSinkIdForRegionId(imp->getRegionId()));
               if ((ampItr != info.getActiveSinksInfo()->end()) && (CabinSrcInfo::isAVSource(ampItr->second.srcId)))
               {
                  if ((NULL != _cabinConnectionUtility) && (_cabinConnectionUtility->isValidStateToDisconnect(ampItr->second.connectionState)))
                  {
                     info.addToConnectionRequestsQueue(stSourceInfo(ampItr->second.srcId, ampItr->second.deviceId, STATE_DISCONNECT, ampItr->second.sinkId));
                  }
                  //INFO: Remember AV source if it is disconnected due to restriction
                  stSourceInfo rememberSrcInfo(ampItr->second.srcId, ampItr->second.deviceId, STATE_UNKNOWN, ampItr->second.sinkId);
                  _AVRestoreInfo.insert(std::pair< int16, stSourceInfo >(ampItr->second.sinkId, rememberSrcInfo));
               }
            }
            SinkConnectionMap::const_iterator sItr = info.getActiveSinksInfo()->find((*itr));
            if ((sItr != info.getActiveSinksInfo()->end()) && (NULL != _cabinConnectionUtility) && (_cabinConnectionUtility->isValidStateToDisconnect(sItr->second.connectionState)))
            {
               info.addToConnectionRequestsQueue(stSourceInfo(sItr->second.srcId, sItr->second.deviceId, STATE_DISCONNECT, sItr->second.sinkId));
            }
         }
      }
   }
}


void MonoGlassController::onMapOutDeactivation(CabinConnectionHandlerInfo& info)
{
   RegionMapOutInfo* imp = info.getMutableRegionMapOutInfo();
   if (NULL != imp)
   {
      ETG_TRACE_USR4(("MonoGlassController: onMapOutDeactivation: RegionId: %d", imp->getRegionId()));
      SinkIdsInfo tGlassSinkInfo = getSupportedGlassSinksInfo(imp->getRegionId());
      for (SinkIdsInfo::iterator itr = tGlassSinkInfo.begin(); (itr != tGlassSinkInfo.end());)
      {
         if (imp->isSinkStateSame((*itr), STATE_DISCONNECTING))
         {
            ETG_TRACE_USR4(("MonoGlassController: onMapOutDeactivation: Removed Sink: %d", (*itr)));
            imp->removeFromSinksMapOutInfo((*itr));
            info.addToModifiedMapSinks((*itr));
            itr = tGlassSinkInfo.erase(itr);
         }
         else
         {
            ++itr;
         }
      }
      if (tGlassSinkInfo.empty())
      {
         restoreAVSource(imp->getRegionId());
      }
   }
}


#endif /* VARIANT_S_FTR_ENABLE_MAP_STREAMING */


void MonoGlassController::onCMAStatusUpdate(const SystemStatusInfo& info)
{
   if ((info.getStatus() != SYSTEM_STATUS_CONNECTED) && (NULL != _cabinConnectionUtility))
   {
      SinkIdsInfo tInfo = _cabinConnectionUtility->getSinkIdsInfo(info.getRegionsIdInfo());
      for (SinkIdsInfo::const_iterator itr = tInfo.begin(); (itr != tInfo.end()); ++itr)
      {
         removeSinkFromAVRestoreInfo((*itr));
      }
   }
}


bool MonoGlassController::fetchAVRestoreInfo(const int16 sinkId, stSourceInfo& info)
{
   bool retVal = false;
   SinkConnectionMap::iterator itr = _AVRestoreInfo.find(sinkId);
   if (itr != _AVRestoreInfo.end())
   {
      info   = itr->second;
      retVal = true;
   }
   return retVal;
}


void MonoGlassController::removeSinkFromAVRestoreInfo(const int16 sinkId)
{
   if (!_AVRestoreInfo.empty())
   {
      if (sinkId == -1)
      {
         _AVRestoreInfo.clear();
      }
      else
      {
         SinkConnectionMap::iterator itr = _AVRestoreInfo.find(sinkId);
         if (itr != _AVRestoreInfo.end())
         {
            _AVRestoreInfo.erase(itr);
            ETG_TRACE_USR4(("MonoGlassController: removeSinkFromAVRestoreInfo: SinkId: %d", sinkId));
         }
      }
   }
}


void MonoGlassController::restoreAVSource(const uint32 regionId)
{
   if (_hmiInfoServiceServerComponent != NULL)
   {
      int16 sinkId = getAmpSinkIdForRegionId(regionId);
      if (_AVRestoreInfo.count(sinkId) > 0)
      {
         stSourceInfo tInfo = _AVRestoreInfo[sinkId];
         ETG_TRACE_USR4(("MonoGlassController: restoreAVSource: SourceId: %d, SinkId: %d ", tInfo.srcId, tInfo.sinkId));
         _hmiInfoServiceServerComponent->sendAVRestorationSignal(regionId, tInfo.srcId);
         removeSinkFromAVRestoreInfo(sinkId);
      }
   }
}


bool MonoGlassController::isGlassSinkSupported(const int16 sinkId) const
{
   bool isValid = false;
   if (NULL != _cabinConnectionUtility)
   {
      uint8 tCategory = (isMonoModeActive(getRegionIdForSinkId(sinkId))) ? SYSTEM_GROUP_CATEGORY_NONE : SYSTEM_GROUP_CATEGORY_CMG_CABIN;
      isValid         = _cabinConnectionUtility->isGlassSinkGroupCategorySame(sinkId, tCategory);
   }
   ETG_TRACE_USR4(("MonoGlassController: isGlassSinkSupported: isValid: %d", isValid));
   return isValid;
}


uint32 MonoGlassController::getRegionIdForSinkId(const int16 sinkId) const
{
   uint32 regionId = REGION_INVALID;
   if (NULL != _cabinConnectionUtility)
   {
      regionId = _cabinConnectionUtility->getRegionIdForSinkId(sinkId);
   }
   return regionId;
}


int16 MonoGlassController::getAmpSinkIdForRegionId(const uint32 regionId) const
{
   int16 sinkId = SINK_INVALID;
   if (NULL != _cabinConnectionUtility)
   {
      sinkId = _cabinConnectionUtility->getAmpSinkIdForRegionId(regionId);
   }
   return sinkId;
}


SinkIdsInfo MonoGlassController::getSupportedGlassSinksInfo(const uint32 regionId, const bool checkSystems) const
{
   SinkIdsInfo tInfo;
   if (NULL != _cabinConnectionUtility)
   {
      uint8 tCategory = (isMonoModeActive(regionId)) ? SYSTEM_GROUP_CATEGORY_NONE : SYSTEM_GROUP_CATEGORY_CMG_CABIN;
      tInfo           = _cabinConnectionUtility->getGlassSinksInfo(regionId, tCategory);
      if (checkSystems)
      {
         for (SinkIdsInfo::iterator itr = tInfo.begin(); itr != tInfo.end();)
         {
            if (!_cabinConnectionUtility->isGlassSystemsValid((*itr)))
            {
               itr = tInfo.erase(itr);
            }
            else
            {
               ++itr;
            }
         }
      }
   }
   return tInfo;
}


} //namespace Core
} //namespace App
