/****************************************************************************
* Copyright (c) 2021-2022 Robert Bosch Car Multimedia GmbH
* duplication and disclosure to third parties is prohibited.
*
* FILE                : CabinsActivityInfoDpHandler.hpp
* COMPONENT Name      : AppHmi_Master
* DESCRIPTION         : Handles datapool implementation for CabinActivityInfo.
* AUTHOR              : Dubba Venkata Sairam
* Date                : 01.09.2021
* Revision History    : 0.1
* Date 11.02.2019     : Initial version
****************************************************************************/

#include "CabinsActivityInfoDpHandler.h"
#include "hmi_trace_if.h"
#include <datapool/CabinActivityInfoDPIF.h>


#ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
#define ETG_DEFAULT_TRACE_CLASS TR_CLASS_APPHMI_MASTER_MAIN
#include "trcGenProj/Header/CabinsActivityInfoDpHandler.cpp.trc.h"
#endif


namespace App {
namespace Core {


CabinsActivityInfoDpHandler::CabinsActivityInfoDpHandler()
{
   ETG_TRACE_USR1(("CabinsActivityInfoDpHandler"));
   clear();
   dumpInfoFromDP();
}


CabinsActivityInfoDpHandler::~CabinsActivityInfoDpHandler()
{
   ETG_TRACE_USR1(("~CabinsActivityInfoDpHandler"));
   clear();
}


void CabinsActivityInfoDpHandler::clear()
{
   ETG_TRACE_USR1(("CabinsActivityInfoDpHandler::clear"));
   _updater = NULL;
   _tempInfo.clear();
   _dupDpInfo.clear();
}


void CabinsActivityInfoDpHandler::prepareDPUpdate(ICabinActivityInfoUpdater& imp)
{
   ETG_TRACE_USR1(("CabinsActivityInfoDpHandler::prepareDPUpdate"));
   if (NULL == _updater)
   {
      _tempInfo.clear();
      _updater = (&imp);
   }
   else
   {
      ETG_TRACE_ERR(("CabinsActivityInfoDpHandler::prepareDPUpdate: Update is ongoing"));
   }
}


void CabinsActivityInfoDpHandler::finalizeDPUpdate(ICabinActivityInfoUpdater& imp)
{
   ETG_TRACE_USR1(("CabinsActivityInfoDpHandler::finalizeDPUpdate"));
   if ((NULL != _updater) && ((&imp) == _updater))
   {
      if (isContentChanged())
      {
         _dupDpInfo.clear();
         _dpInfo.vClearList();
         dumpInfoToDP();
      }
      _updater = NULL;
      _tempInfo.clear();
   }
}


void CabinsActivityInfoDpHandler::updateDPData(ICabinActivityInfoUpdater& imp, const CabinActivityInfoDPHolder& newCabinsActivityInfo)
{
   ETG_TRACE_USR1(("CabinsActivityInfoDpHandler::updateDPData"));
   if ((NULL != _updater) && ((&imp) == _updater))
   {
      CabinsActivitiesInfo::iterator tempInfoItr = _tempInfo.begin();
      for (; (tempInfoItr != _tempInfo.end()); ++tempInfoItr)
      {
         if (tempInfoItr->getSinkId() == newCabinsActivityInfo.getSinkId())
         {
            ETG_TRACE_USR1(("CabinsActivityInfoDpHandler::updateDPData: Existed SinkId: %d", tempInfoItr->getSinkId()));
            break;
         }
      }
      if (tempInfoItr == _tempInfo.end())
      {
         ETG_TRACE_USR1(("CabinsActivityInfoDpHandler::updateDPData: SinkId: %d", newCabinsActivityInfo.getSinkId()));
         _tempInfo.push_back(newCabinsActivityInfo);
      }
   }
}


bool CabinsActivityInfoDpHandler::isContentChanged()
{
   bool update = false;
   for (CabinsActivitiesInfo::iterator tempInfoItr = _tempInfo.begin(); ((tempInfoItr != _tempInfo.end()) && (update == false)); ++tempInfoItr)
   {
      update = true;
      for (CabinsActivitiesInfo::iterator dupInfoItr = _dupDpInfo.begin(); (dupInfoItr != _dupDpInfo.end()); ++dupInfoItr)
      {
         bool isValid = (tempInfoItr->getSinkId() == dupInfoItr->getSinkId()) ? true : false;
         isValid = ((isValid) && (tempInfoItr->getCabinId() == dupInfoItr->getCabinId())) ? true : false;
         isValid = ((isValid) && (tempInfoItr->getActivityType() == dupInfoItr->getActivityType())) ? true : false;
         isValid = ((isValid) && (tempInfoItr->getSourceId() == dupInfoItr->getSourceId())) ? true : false;
         isValid = ((isValid) && (tempInfoItr->getState() == dupInfoItr->getState())) ? true : false;
         isValid = ((isValid) && (tempInfoItr->getLastSourceId() == dupInfoItr->getLastSourceId())) ? true : false;
         isValid = ((isValid) && (tempInfoItr->getLastState() == dupInfoItr->getLastState())) ? true : false;
         isValid = ((isValid) && (tempInfoItr->getSubSourceId() == dupInfoItr->getSubSourceId())) ? true : false;
         isValid = ((isValid) && (tempInfoItr->getLastSubSourceId() == dupInfoItr->getLastSubSourceId())) ? true : false;
         if (isValid)
         {
            update = false;
            break;
         }
      }
   }
   if (!update)
   {
      for (CabinsActivitiesInfo::iterator dupInfoItr = _dupDpInfo.begin(); ((dupInfoItr != _dupDpInfo.end()) && (update == false)); ++dupInfoItr)
      {
         bool isSinkAvailable = false;
         for (CabinsActivitiesInfo::iterator tempInfoItr = _tempInfo.begin(); (tempInfoItr != _tempInfo.end()); ++tempInfoItr)
         {
            if ((*dupInfoItr).getSinkId() == (*tempInfoItr).getSinkId())
            {
               isSinkAvailable = true;
               break;
            }
         }
         update = (!isSinkAvailable) ? true : false;
      }
   }
   return update;
}


void CabinsActivityInfoDpHandler::dumpInfoToDP()
{
   if (!_tempInfo.empty())
   {
      _dpInfo.vClearList();
      for (CabinsActivitiesInfo::iterator tempInfoItr = _tempInfo.begin(); (tempInfoItr != _tempInfo.end()); ++tempInfoItr)
      {
         ETG_TRACE_USR1(("CabinsActivityInfoDpHandler::dumpInfoToDP: CabinId: %d, SinkId: %d, ActivityType: %d, SourceId: %d, LastSourceId: %d", \
                         tempInfoItr->getCabinId(), tempInfoItr->getSinkId(), tempInfoItr->getActivityType(), tempInfoItr->getSourceId(), tempInfoItr->getLastSourceId()));
         _dupDpInfo.push_back(*tempInfoItr);
         _dpInfo.vPushBack(*tempInfoItr);
      }
   }
}


void CabinsActivityInfoDpHandler::dumpInfoFromDP()
{
   _dupDpInfo.clear();
   for (uint32 index = 0; (index < _dpInfo.u32GetCount()); ++index)
   {
      CabinActivityInfoDPHolder cabinInfo;
      if (_dpInfo.s32GetElem(index, cabinInfo) != DP_S32_ERR_NO_ELEMENT)
      {
         ETG_TRACE_USR1(("CabinsActivityInfoDpHandler::dumpInfoFromDP: CabinId: %d, SinkId: %d, ActivityType: %d, SourceId: %d, LastSourceId: %d", \
                         cabinInfo.getCabinId(), cabinInfo.getSinkId(), cabinInfo.getActivityType(), cabinInfo.getSourceId(), cabinInfo.getLastSourceId()));
         _dupDpInfo.push_back(cabinInfo);
      }
   }
}


CabinActivitiesInfo CabinsActivityInfoDpHandler::getCabinActivitiesInfo(const uint32 cabinId)
{
   CabinActivitiesInfo tCabinActivitiesInfo;
   for (CabinsActivitiesInfo::iterator dupInfoItr = _dupDpInfo.begin(); (dupInfoItr != _dupDpInfo.end()); ++dupInfoItr)
   {
      if (cabinId == dupInfoItr->getCabinId())
      {
         tCabinActivitiesInfo.push_back(*dupInfoItr);
      }
   }
   return tCabinActivitiesInfo;
}


} //namespace Core
} //namespace App
