/****************************************************************************
* Copyright (c) 2019-2019 Robert Bosch Car Multimedia GmbH
* duplication and disclosure to third parties is prohibited.
*
* FILE                : CabinSrcInfo.cpp
* COMPONENT Name      : AppHmi_Master
* DESCRIPTION         : Provides information about the sources supported in cabin regions
* AUTHOR              : Preethi Alagappan
* Date                : 11.02.2019
* Revision History    : 0.1
* Date 11.02.2019     : Initial version
****************************************************************************/

#include "hall_std_if.h"
#include "CabinSrcInfo.h"
#include "PluginMain/PluginDataType.h"


namespace App {
namespace Core {


std::map<uint32_t, uint8_t> pluginSourceIndexInfo;


const int CabinSrcInfo::nCameraPortSources[MAX_CAMERAPORTS][MAX_SRCS_IN_CAMERAPORT] =
{
   {
      SRC_CAMERAPORT1_CAM1, SRC_CAMERAPORT1_CAM2, SRC_CAMERAPORT1_CAM3, SRC_CAMERAPORT1_CAM4,
      SRC_CAMERAPORT1_IPCAM, SRC_CAMERAPORT1_SPLITSCREEN, SRC_CAMERAPORT1_USB_VIDEO, SRC_CAMERAPORT1_HDMI
   },

   {
      SRC_CAMERAPORT2_CAM1, SRC_CAMERAPORT2_CAM2, SRC_CAMERAPORT2_CAM3, SRC_CAMERAPORT2_CAM4,
      SRC_CAMERAPORT2_IPCAM, SRC_CAMERAPORT2_SPLITSCREEN, SRC_CAMERAPORT2_USB_VIDEO, SRC_CAMERAPORT2_HDMI
   },

   {
      SRC_CAMERAPORT3_CAM1, SRC_CAMERAPORT3_CAM2, SRC_CAMERAPORT3_CAM3, SRC_CAMERAPORT3_CAM4,
      SRC_CAMERAPORT3_IPCAM, SRC_CAMERAPORT3_SPLITSCREEN, SRC_CAMERAPORT3_USB_VIDEO, SRC_CAMERAPORT3_HDMI
   },

   {
      SRC_CAMERAPORT4_CAM1, SRC_CAMERAPORT4_CAM2, SRC_CAMERAPORT4_CAM3, SRC_CAMERAPORT4_CAM4,
      SRC_CAMERAPORT4_IPCAM, SRC_CAMERAPORT4_SPLITSCREEN, SRC_CAMERAPORT4_USB_VIDEO, SRC_CAMERAPORT4_HDMI
   }
};


/************************************************************************
*NAME        : getSrcType
*SWFLs       : SWFL-2851, SWFL-2856
************************************************************************/
CabinSrcInfo::SrcType CabinSrcInfo::getSrcType(int nSrcId)
{
   CabinSrcInfo::SrcType enCabinSrcType = SRC_TYPE_UNKNOWN;
   switch (nSrcId)
   {
      case SRC_TUNER_FM:
      case SRC_TUNER_AM:
      case SRC_DAB:
      case SRC_TUNER_XM:
      case SRC_TUNER_WX:
      case SRC_MEDIA_PLAYER:
      case SRC_PHONE_BTAUDIO:
      case SRC_MIC1_ANNOUNCEMENT:
      {
         enCabinSrcType = COCKPIT_SOURCE;
      }
      break;

      case SRC_CAMERAPORT1_USB_VIDEO:
      case SRC_CAMERAPORT1_HDMI:
      case SRC_CAMERAPORT2_USB_VIDEO:
      case SRC_CAMERAPORT2_HDMI:
      case SRC_CAMERAPORT3_USB_VIDEO:
      case SRC_CAMERAPORT3_HDMI:
      case SRC_CAMERAPORT4_USB_VIDEO:
      case SRC_CAMERAPORT4_HDMI:
      case SRC_CMR1_USB1_VIDEO:
      case SRC_CMR1_USB2_VIDEO:
      {
         enCabinSrcType = CABIN_AV_SOURCE;
      }
      break;

      case SRC_CAMERAPORT1_CAM1:
      case SRC_CAMERAPORT1_CAM2:
      case SRC_CAMERAPORT1_CAM3:
      case SRC_CAMERAPORT1_CAM4:
      case SRC_CAMERAPORT1_IPCAM:
      case SRC_CAMERAPORT1_SPLITSCREEN:
      case SRC_CAMERAPORT2_CAM1:
      case SRC_CAMERAPORT2_CAM2:
      case SRC_CAMERAPORT2_CAM3:
      case SRC_CAMERAPORT2_CAM4:
      case SRC_CAMERAPORT2_IPCAM:
      case SRC_CAMERAPORT2_SPLITSCREEN:
      case SRC_CAMERAPORT3_CAM1:
      case SRC_CAMERAPORT3_CAM2:
      case SRC_CAMERAPORT3_CAM3:
      case SRC_CAMERAPORT3_CAM4:
      case SRC_CAMERAPORT3_IPCAM:
      case SRC_CAMERAPORT3_SPLITSCREEN:
      case SRC_CAMERAPORT4_CAM1:
      case SRC_CAMERAPORT4_CAM2:
      case SRC_CAMERAPORT4_CAM3:
      case SRC_CAMERAPORT4_CAM4:
      case SRC_CAMERAPORT4_IPCAM:
      case SRC_CAMERAPORT4_SPLITSCREEN:
      {
         enCabinSrcType = CABIN_VIDEO_SOURCE;
      }
      break;

      case SRC_MIC2_ANNOUNCEMENT:
      case SRC_MIC3_ANNOUNCEMENT:
      {
         enCabinSrcType = CABIN_MIX_SOURCE;
      }
      break;

      case SRC_CAMERAPORT1_USB_AUDIO:
      case SRC_CMR1_USB1_AUDIO:
      case SRC_CMR1_USB2_AUDIO:
      {
         enCabinSrcType = CABIN_AUDIO_SOURCE;
      }
      break;

      default:
         break;
   }
   return enCabinSrcType;
}


void initialisePluginSourceIndexMap()
{
   if (pluginSourceIndexInfo.empty())
   {
      pluginSourceIndexInfo.insert(std::make_pair<uint32_t, uint8_t>(SRC_CAMERAPORT1_USB_VIDEO, CMP1_USB1));
      pluginSourceIndexInfo.insert(std::make_pair<uint32_t, uint8_t>(SRC_CAMERAPORT2_USB_VIDEO, CMP2_USB1));
      pluginSourceIndexInfo.insert(std::make_pair<uint32_t, uint8_t>(SRC_CAMERAPORT3_USB_VIDEO, CMP3_USB1));
      pluginSourceIndexInfo.insert(std::make_pair<uint32_t, uint8_t>(SRC_CAMERAPORT4_USB_VIDEO, CMP4_USB1));
      pluginSourceIndexInfo.insert(std::make_pair<uint32_t, uint8_t>(SRC_CAMERAPORT1_HDMI, CMP1_HDMI));
      pluginSourceIndexInfo.insert(std::make_pair<uint32_t, uint8_t>(SRC_CAMERAPORT2_HDMI, CMP2_HDMI));
      pluginSourceIndexInfo.insert(std::make_pair<uint32_t, uint8_t>(SRC_CAMERAPORT3_HDMI, CMP3_HDMI));
      pluginSourceIndexInfo.insert(std::make_pair<uint32_t, uint8_t>(SRC_CAMERAPORT4_HDMI, CMP4_HDMI));
      pluginSourceIndexInfo.insert(std::make_pair<uint32_t, uint8_t>(SRC_CAMERAPORT1_CAM1, CMP1_CAMERA1));
      pluginSourceIndexInfo.insert(std::make_pair<uint32_t, uint8_t>(SRC_CAMERAPORT1_CAM2, CMP1_CAMERA2));
      pluginSourceIndexInfo.insert(std::make_pair<uint32_t, uint8_t>(SRC_CAMERAPORT1_CAM3, CMP1_CAMERA3));
      pluginSourceIndexInfo.insert(std::make_pair<uint32_t, uint8_t>(SRC_CAMERAPORT1_CAM4, CMP1_CAMERA4));
      pluginSourceIndexInfo.insert(std::make_pair<uint32_t, uint8_t>(SRC_CAMERAPORT2_CAM1, CMP2_CAMERA1));
      pluginSourceIndexInfo.insert(std::make_pair<uint32_t, uint8_t>(SRC_CAMERAPORT2_CAM2, CMP2_CAMERA2));
      pluginSourceIndexInfo.insert(std::make_pair<uint32_t, uint8_t>(SRC_CAMERAPORT2_CAM3, CMP2_CAMERA3));
      pluginSourceIndexInfo.insert(std::make_pair<uint32_t, uint8_t>(SRC_CAMERAPORT2_CAM4, CMP2_CAMERA4));
      pluginSourceIndexInfo.insert(std::make_pair<uint32_t, uint8_t>(SRC_CAMERAPORT3_CAM1, CMP3_CAMERA1));
      pluginSourceIndexInfo.insert(std::make_pair<uint32_t, uint8_t>(SRC_CAMERAPORT3_CAM2, CMP3_CAMERA2));
      pluginSourceIndexInfo.insert(std::make_pair<uint32_t, uint8_t>(SRC_CAMERAPORT3_CAM3, CMP3_CAMERA3));
      pluginSourceIndexInfo.insert(std::make_pair<uint32_t, uint8_t>(SRC_CAMERAPORT3_CAM4, CMP3_CAMERA4));
      pluginSourceIndexInfo.insert(std::make_pair<uint32_t, uint8_t>(SRC_CAMERAPORT4_CAM1, CMP4_CAMERA1));
      pluginSourceIndexInfo.insert(std::make_pair<uint32_t, uint8_t>(SRC_CAMERAPORT4_CAM2, CMP4_CAMERA2));
      pluginSourceIndexInfo.insert(std::make_pair<uint32_t, uint8_t>(SRC_CAMERAPORT4_CAM3, CMP4_CAMERA3));
      pluginSourceIndexInfo.insert(std::make_pair<uint32_t, uint8_t>(SRC_CAMERAPORT4_CAM4, CMP4_CAMERA4));
   }
}


uint8_t CabinSrcInfo::getPluginSourceIndex(uint32_t nSrcId)
{
   initialisePluginSourceIndexMap();

   std::map<uint32_t, uint8_t>::iterator it = pluginSourceIndexInfo.find(nSrcId) ;

   uint8_t PluginSourceIndex = MAXSOURCE;

   if (it != pluginSourceIndexInfo.end())
   {
      PluginSourceIndex = it->second ;
   }

   return PluginSourceIndex;
}


/************************************************************************
*NAME        : isCockpitSource
*SWFLs       : SWFL-2697
************************************************************************/
bool CabinSrcInfo::isCockpitSource(int nSrcId)
{
   bool isCockpitSource = false;
   CabinSrcInfo::SrcType srcType = getSrcType(nSrcId);
   if (COCKPIT_SOURCE == srcType)
   {
      isCockpitSource = true;
   }
   return isCockpitSource;
}


/************************************************************************
*NAME        : isAVSource
*SWFLs       : SWFL-2851, SWFL-2856
************************************************************************/
bool CabinSrcInfo::isAVSource(int nSrcId)
{
   bool isAVSource = false;
   CabinSrcInfo::SrcType srcType = getSrcType(nSrcId);
   if (CABIN_AV_SOURCE == srcType)
   {
      isAVSource = true;
   }
   return isAVSource;
}


/************************************************************************
*NAME        : isVideoSource
*SWFLs       : SWFL-2851, SWFL-2856
************************************************************************/
bool CabinSrcInfo::isVideoSource(int nSrcId)
{
   bool isVideoSource = false;
   CabinSrcInfo::SrcType srcType = getSrcType(nSrcId);
   if (CABIN_VIDEO_SOURCE == srcType)
   {
      isVideoSource = true;
   }
   return isVideoSource;
}


bool CabinSrcInfo::isCabinAudioSource(int nSrcId)
{
   bool isAudioSource = false;
   CabinSrcInfo::SrcType srcType = getSrcType(nSrcId);
   if (CABIN_AUDIO_SOURCE == srcType)
   {
      isAudioSource = true;
   }
   return isAudioSource;
}


bool CabinSrcInfo::isCamportHDMI(int nSrcId)
{
   bool isValid = false;
   switch (nSrcId)
   {
      case SRC_CAMERAPORT1_HDMI:
      case SRC_CAMERAPORT2_HDMI:
      case SRC_CAMERAPORT3_HDMI:
      case SRC_CAMERAPORT4_HDMI:
         isValid = true;
         break;
      default:
         break;
   }
   return isValid;
}


bool CabinSrcInfo::isCamportUSB(int nSrcId)
{
   bool isValid = false;
   switch (nSrcId)
   {
      case SRC_CAMERAPORT1_USB_VIDEO:
      case SRC_CAMERAPORT2_USB_VIDEO:
      case SRC_CAMERAPORT3_USB_VIDEO:
      case SRC_CAMERAPORT4_USB_VIDEO:
         isValid = true;
         break;
      default:
         break;
   }
   return isValid;
}


/************************************************************************
*NAME        : isHUTunerSource
*SWFLs       : SWFL-2697
************************************************************************/
bool CabinSrcInfo::isHUTunerSource(int nSrcId)
{
   int tunerSrcMinValue = GROUP_TUNER * SRC_GROUP_SIZE;
   int tunerSrcMaxValue = tunerSrcMinValue + SRC_GROUP_SIZE - 1;

   bool isHUTunerSource = ((nSrcId >= tunerSrcMinValue) && (nSrcId <= tunerSrcMaxValue)) ? true : false;
   return isHUTunerSource;
}


/************************************************************************
*NAME        : isIPCameraSource
*SWFLs       :
************************************************************************/
bool CabinSrcInfo::isIPCameraSource(int nSrcId)
{
   bool isIPCameraSource = false;
   switch (nSrcId)
   {
      case SRC_CAMERAPORT1_IPCAM:
      case SRC_CAMERAPORT2_IPCAM:
      case SRC_CAMERAPORT3_IPCAM:
      case SRC_CAMERAPORT4_IPCAM:
      {
         isIPCameraSource = true;
         break;
      }

      default:
         break;
   }
   return isIPCameraSource;
}


/************************************************************************
*NAME        : isCMRVideoSource
*SWFLs       :
************************************************************************/
bool CabinSrcInfo::isCMRVideoSource(int nSrcId)
{
   bool isCMRVideoSource = false;
   switch (nSrcId)
   {
      case SRC_CMR1_USB1_VIDEO:
      case SRC_CMR1_USB2_VIDEO:
      {
         isCMRVideoSource = true;
         break;
      }

      default:
         break;
   }
   return isCMRVideoSource;
}


/************************************************************************
*NAME        : isCMRAudioSource
*SWFLs       :
************************************************************************/
bool CabinSrcInfo::isCMRAudioSource(int nSrcId)
{
   bool isCMRAudioSource = false;
   switch (nSrcId)
   {
      case SRC_CMR1_USB1_AUDIO:
      case SRC_CMR1_USB2_AUDIO:
      {
         isCMRAudioSource = true;
         break;
      }

      default:
         break;
   }
   return isCMRAudioSource;
}


/************************************************************************
*NAME        : getCameraPortID
*SWFLs       :
************************************************************************/
CabinSrcInfo::CameraPortID CabinSrcInfo::getCameraPortID(int nCameraPortSrcId)
{
   CabinSrcInfo::CameraPortID cameraPortId = MAX_CAMERAPORTS;
   for (int index = CAMERAPORT1; index < MAX_CAMERAPORTS; index++)
   {
      for (int srcIndex = 0; srcIndex < MAX_SRCS_IN_CAMERAPORT; srcIndex++)
      {
         if (nCameraPortSrcId == nCameraPortSources[index][srcIndex])
         {
            cameraPortId = static_cast<CabinSrcInfo::CameraPortID>(index);
            return cameraPortId;
         }
      }
   }
   return cameraPortId;
}


bool CabinSrcInfo::isMIXSource(int nSrcId)
{
   bool isMixSource = false;
   CabinSrcInfo::SrcType srcType = getSrcType(nSrcId);
   if (CABIN_MIX_SOURCE == srcType)
   {
      isMixSource = true;
   }
   return isMixSource;
}


int CabinSrcInfo::getMixSrcIndex(int nSrcId)
{
   int MicIndex = NO_MixSource;
   switch (nSrcId)
   {
      case SRC_MIC2_ANNOUNCEMENT:
      {
         MicIndex = MIC2_MixSource;
      }
      break;
      case SRC_MIC3_ANNOUNCEMENT:
      {
         MicIndex = MIC3_MixSource;
      }
      break;

      default:
         break;
   }
   return MicIndex;
}


enCabinActiviyType CabinSrcInfo::getActivityTypeForSource(int nSrcId)
{
   enCabinActiviyType activitytype = CABIN_ACTIVITY_NONE;
   CabinSrcInfo::SrcType srcType = getSrcType(nSrcId);
   switch (srcType)
   {
      case CABIN_AV_SOURCE:
         activitytype = CABIN_ACTIVITY_AV;
         break;
      case CABIN_VIDEO_SOURCE:
         activitytype = CABIN_ACTIVITY_VIDEO;
         break;
      case CABIN_AUDIO_SOURCE:
         activitytype = CABIN_ACTIVITY_AUDIO;
         break;
      case COCKPIT_SOURCE:
         activitytype = CABIN_ACTIVITY_COCKPIT_SOURCE;
         break;
      default:
         break;
   }
   return activitytype;
}


} //namespace Core
} //namespace App
