/**
 * @file         : CabinRestorationHandler.cpp
 * @author       : INF4CV - AppHmi_Master Team
 * @addtogroup   : AppHmi_Master
 * @brief        : CabinRestorationHandler is to handle the cabin activity restoration
                   for mono glass group
 * @copyright    : (c) 2018-2020 Robert Bosch Car Multimedia GmbH
 *                 The reproduction, distribution and utilization of this file as
 *                 well as the communication of its contents to others without express
 *                 authorization is prohibited. Offenders will be held liable for the
 *                 payment of damages. All rights reserved in the event of the grant
 *                 of a patent, utility model or design.
 */


#include "hall_std_if.h"
#include "CabinRestorationHandler.h"
#include "CabinRestorationHandlerTypes.h"
#include "CabinVideoMixInfoDpHandlerInterface.h"
#include <datapool/CabinActivityInfoDPIF.h>
#include <datapool/CabinVideoMixInfoDPIF.h>
#include <Core/HmiState/HmiStateHandlerHelper.h>
#include <Core/RegionHandling/RegionHandlingTypes.h>
#include <bosch/cm/ai/hmi/hmiinfoservice/HmiInfoService.h>
#include <Core/HmiInfoService/HmiInfoServiceServerComponentHandlingInterface.h>


using namespace ::bosch::cm::ai::hmi::hmiinfoservice::HmiInfoService;

#ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
#define ETG_DEFAULT_TRACE_CLASS TR_CLASS_APPHMI_MASTER_MAIN
#include "trcGenProj/Header/CabinRestorationHandler.cpp.trc.h"
#endif


/**
 *  Memeber Functions Definition
 */

namespace App {
namespace Core {


CabinRestorationHandler::CabinRestorationHandler() : _regionHandling(NULL)
   , _hmiStateHandler(NULL)
   , _syncDisplayHandler(NULL)
   , _cabinConnectionHandler(NULL)
   , _cabinVideoMixInfoDpHandler(NULL)
   , _cabinsActivityInfoDpHandler(NULL)
   , _hmiInfoServiceServerComponent(NULL)
{
   ETG_TRACE_USR1(("CabinRestorationHandler: CTOR"));
   _restoreRegionsInfo.clear();
   _activitiesSinkLoaderFunctionsInfo.clear();
   initActivitiesSinkLoaderFunctionsInfo();
   initActivitiesValidatorFunctionsInfo();
}


CabinRestorationHandler::~CabinRestorationHandler()
{
   ETG_TRACE_USR1(("CabinRestorationHandler: DTOR"));
   _regionHandling                = NULL;
   _hmiStateHandler               = NULL;
   _syncDisplayHandler            = NULL;
   _cabinConnectionHandler        = NULL;
   _cabinVideoMixInfoDpHandler    = NULL;
   _cabinsActivityInfoDpHandler   = NULL;
   _hmiInfoServiceServerComponent = NULL;
   _restoreRegionsInfo.clear();
   _activitiesSinkLoaderFunctionsInfo.clear();
}


void CabinRestorationHandler::initialize()
{
   if (NULL != _regionHandling)
   {
      _regionHandling->addCMAStatusListener((*this));
      _regionHandling->addCMPStatusListener((*this));
      _regionHandling->addCMGStatusListener((*this));
      _regionHandling->addCMRStatusListener((*this));
   }
   if (NULL != _hmiStateHandler)
   {
      _hmiStateHandler->addHmiStateListener((this));
   }
   if (NULL != _syncDisplayHandler)
   {
      _syncDisplayHandler->addHmiFirstAppListener((*this));
   }
   if (NULL != _cabinConnectionHandler)
   {
#ifdef VARIANT_S_FTR_ENABLE_MIC_ACTIVATION
      _cabinConnectionHandler->addMic1StatusListener((this));
#endif /* VARIANT_S_FTR_ENABLE_MIC_ACTIVATION */
   }
}


void CabinRestorationHandler::finalize()
{
   if (NULL != _regionHandling)
   {
      _regionHandling->removeCMAStatusListener((*this));
      _regionHandling->removeCMPStatusListener((*this));
      _regionHandling->removeCMGStatusListener((*this));
      _regionHandling->removeCMRStatusListener((*this));
   }
   if (NULL != _hmiStateHandler)
   {
      _hmiStateHandler->removeHmiStateListener((this));
   }
   if (NULL != _syncDisplayHandler)
   {
      _syncDisplayHandler->removeHmiFirstAppListener((*this));
   }
   if (NULL != _cabinConnectionHandler)
   {
#ifdef VARIANT_S_FTR_ENABLE_MIC_ACTIVATION
      _cabinConnectionHandler->removeMic1StatusListener((this));
#endif /* VARIANT_S_FTR_ENABLE_MIC_ACTIVATION */
   }
}


void CabinRestorationHandler::initRestoreRegionsInfo()
{
   if ((NULL != _regionHandling) && (NULL != _cabinsActivityInfoDpHandler))
   {
      _restoreRegionsInfo.clear(); //For safety
      SupportedRegionsInfo tSupportedRegionsInfo = _regionHandling->getSupportedRegionsInfo();
      for (SupportedRegionsInfo::iterator itr = tSupportedRegionsInfo.begin(); (itr != tSupportedRegionsInfo.end()); ++itr)
      {
         if (_regionHandling->isCabinRegion((*itr).first))
         {
            CabinActivitiesInfo tCabinActivitiesInfo      = _cabinsActivityInfoDpHandler->getCabinActivitiesInfo((*itr).first);
            std::vector <uint8> activitiesInfo            = getActivitiesInfo(((*itr).first), tCabinActivitiesInfo);
            if (!activitiesInfo.empty())
            {
               RestoreRegionInfo tRestoreRegionInfo;
               tRestoreRegionInfo.setRegionId((*itr).first);
               for (std::vector< uint8 >::iterator itr1 = activitiesInfo.begin(); itr1 != activitiesInfo.end(); ++itr1)
               {
                  ActivitiesSinkLoaderFunctionsInfo::iterator sItr = _activitiesSinkLoaderFunctionsInfo.find((*itr1));
                  if (sItr != _activitiesSinkLoaderFunctionsInfo.end())
                  {
                     ETG_TRACE_USR1(("CabinRestorationHandler: initRestoreRegionsInfo: RegionId: %d, ActivityType: %d", (*itr).first, (*itr1)));
                     tRestoreRegionInfo.addActivity((*itr1));
                     for (SinkLoaderFunctionsInfo::iterator tItr = sItr->second.begin(); (tItr != sItr->second.end()); ++tItr)
                     {
                        ((this)->*((*tItr)))(tCabinActivitiesInfo, tRestoreRegionInfo, (*itr1));
                     }
                  }
               }
               _restoreRegionsInfo.push_back(tRestoreRegionInfo);
            }
         }
      }
      updateCabinVideoMixInfoProperty();
      sendCabinsRestorationInfo();
   }
}


void CabinRestorationHandler::initActivitiesSinkLoaderFunctionsInfo()
{
   if (_activitiesSinkLoaderFunctionsInfo.empty())
   {
      SinkLoaderFunctionsInfo tSinkLoaderFunctionsInfo;
      tSinkLoaderFunctionsInfo.push_back(&CabinRestorationHandler::loadAmpSinkToRestoreRegionInfo);
      _activitiesSinkLoaderFunctionsInfo.insert(::std::pair< uint8, SinkLoaderFunctionsInfo >(CABIN_ACTIVITY_COCKPIT_SOURCE, tSinkLoaderFunctionsInfo));
      tSinkLoaderFunctionsInfo.push_back(&CabinRestorationHandler::loadAmpSourceToRestoreRegionInfo);
      _activitiesSinkLoaderFunctionsInfo.insert(::std::pair< uint8, SinkLoaderFunctionsInfo >(CABIN_ACTIVITY_AUDIO, tSinkLoaderFunctionsInfo));
      tSinkLoaderFunctionsInfo.push_back(&CabinRestorationHandler::loadGlassSinkToRestoreRegionInfo);
      _activitiesSinkLoaderFunctionsInfo.insert(::std::pair< uint8, SinkLoaderFunctionsInfo >(CABIN_ACTIVITY_AV, tSinkLoaderFunctionsInfo));
      tSinkLoaderFunctionsInfo.clear();
      tSinkLoaderFunctionsInfo.push_back(&CabinRestorationHandler::loadGlassSinkToRestoreRegionInfo);
      tSinkLoaderFunctionsInfo.push_back(&CabinRestorationHandler::loadAmpSystemToRestoreRegionInfoGlassSink);
      _activitiesSinkLoaderFunctionsInfo.insert(::std::pair< uint8, SinkLoaderFunctionsInfo >(CABIN_ACTIVITY_MAPOUT, tSinkLoaderFunctionsInfo));
      tSinkLoaderFunctionsInfo.clear();
      tSinkLoaderFunctionsInfo.push_back(&CabinRestorationHandler::loadGlassSinkToRestoreRegionInfo);
      tSinkLoaderFunctionsInfo.push_back(&CabinRestorationHandler::loadGlassSourceToRestoreRegionInfo);
      tSinkLoaderFunctionsInfo.push_back(&CabinRestorationHandler::loadAmpSystemToRestoreRegionInfoGlassSink);
      _activitiesSinkLoaderFunctionsInfo.insert(::std::pair< uint8, SinkLoaderFunctionsInfo >(CABIN_ACTIVITY_VIDEO, tSinkLoaderFunctionsInfo));
      tSinkLoaderFunctionsInfo.clear();
   }
}


void CabinRestorationHandler::initActivitiesValidatorFunctionsInfo()
{
   if (_activitiesValidatorFunctionsInfo.empty())
   {
      ValidatorFunctionsInfo tValidatorFunctionsInfo;
      /* ::std::pair of ValidatorFunction and the data expected i.e. validToRestore */
      tValidatorFunctionsInfo.push_back(::std::pair < ValidatorFunction, bool >(&CabinRestorationHandler::isLastSystemStatusCockpitModeDisconnected, false));
      _activitiesValidatorFunctionsInfo.insert(::std::pair< uint8, ValidatorFunctionsInfo >(CABIN_ACTIVITY_COCKPIT_SOURCE, tValidatorFunctionsInfo));
      _activitiesValidatorFunctionsInfo.insert(::std::pair< uint8, ValidatorFunctionsInfo >(CABIN_ACTIVITY_AUDIO, tValidatorFunctionsInfo));
      /* ::std::pair of ValidatorFunction and the data expected i.e. validToRestore */
      tValidatorFunctionsInfo.push_back(::std::pair < ValidatorFunction, bool >(&CabinRestorationHandler::isLastSystemStatusPNMDisconnected, false));
      _activitiesValidatorFunctionsInfo.insert(::std::pair< uint8, ValidatorFunctionsInfo >(CABIN_ACTIVITY_AV, tValidatorFunctionsInfo));
      _activitiesValidatorFunctionsInfo.insert(::std::pair< uint8, ValidatorFunctionsInfo >(CABIN_ACTIVITY_VIDEO, tValidatorFunctionsInfo));
      _activitiesValidatorFunctionsInfo.insert(::std::pair< uint8, ValidatorFunctionsInfo >(CABIN_ACTIVITY_MAPOUT, tValidatorFunctionsInfo));
      tValidatorFunctionsInfo.clear();
   }
}


void CabinRestorationHandler::onHmiFirstAppRequest()
{
   evalAndUpdateRestoreStatus();
}


void CabinRestorationHandler::onHmiStateUpdate(const uint8 currentHmiState)
{
   if (isSPMHMIStateShowHMI(currentHmiState) && (NULL != _hmiStateHandler) && (!isSPMHMIStateShowHMI(_hmiStateHandler->getLastHmiState())))
   {
      initRestoreRegionsInfo();
#ifdef VARIANT_S_FTR_ENABLE_MIC_ACTIVATION
      evalAndUpdateWaitStatusForMic1();
#endif /* VARIANT_S_FTR_ENABLE_MIC_ACTIVATION */
      evalAndUpdateRestoreStatus();
   }
   else
   {
      _restoreRegionsInfo.clear();
      sendCabinsRestorationInfo();
   }
}


void CabinRestorationHandler::updateCabinVideoMixInfoProperty()
{
   if ((NULL != _hmiInfoServiceServerComponent) && (NULL != _cabinVideoMixInfoDpHandler) && (NULL != _cabinConnectionHandler))
   {
      const CabinsVideoMixInfoHolder& cabinsVideoMixInfo = _cabinVideoMixInfoDpHandler->getCabinsVideoMixInfo();
      for (CabinsVideoMixInfoHolder::const_iterator itr = cabinsVideoMixInfo.begin(); (itr != cabinsVideoMixInfo.end()); ++itr)
      {
         for (RestoreRegionsInfo::const_iterator regItr = _restoreRegionsInfo.begin(); (regItr != _restoreRegionsInfo.end()); ++regItr)
         {
            if ((*regItr).getRegionId() == (*itr).getCabinId())
            {
               for (ActivitiesRestoreSinksInfo::const_iterator actItr = (*regItr).getActivitiesRestoreSinksInfo().begin(); (actItr != (*regItr).getActivitiesRestoreSinksInfo().end()); ++actItr)
               {
                  if ((*actItr).getActivityType() == getActivityTypeForVideoMixIndex((*itr).getVideoMixIndex()))
                  {
                     const RestoreSinksInfo* info = (*actItr).getRestoreSinksInfoForGroup(SINKGROUP_VIDEOMIX);
                     if ((info != NULL) && (!(info->empty())))
                     {
                        ETG_TRACE_USR1(("CabinRestorationHandler: updateCabinVideoMixInfoProperty: CabinId: %d, VideoMixIndex: %d", (*itr).getCabinId(), (*itr).getVideoMixIndex()));
                        _cabinConnectionHandler->onSetCabinVideoMixInfo(((*itr).getCabinId()), ((*itr).getVideoMixIndex()), CABIN_VIDEO_MIX_CHANGE_REASON_RESTORE);
                     }
                     break;
                  }
               }
               break;
            }
         }
      }
   }
}


uint8 CabinRestorationHandler::getActivityTypeForVideoMixIndex(const uint8 videoMixIndex)
{
   uint8 activityType = CABIN_ACTIVITY_NONE;
   switch (videoMixIndex)
   {
      case CABIN_VIDEO_MIX_INDEX_MAP_OUT:
      {
         activityType = CABIN_ACTIVITY_MAPOUT;
         break;
      }
      case CABIN_VIDEO_MIX_INDEX_IP_CAMERA_1:
      case CABIN_VIDEO_MIX_INDEX_IP_CAMERA_2:
      case CABIN_VIDEO_MIX_INDEX_IP_CAMERA_3:
      {
         activityType = CABIN_ACTIVITY_VIDEO;
         break;
      }
      default:
         break;
   }
   return activityType;
}


void CabinRestorationHandler::onCMAStatusUpdate(const SystemStatusInfo& info)
{
   bool systemStatus = info.getStatus() == SYSTEM_STATUS_CONNECTED ? true : false;
   evalAndUpdateSystemStatus(info, systemStatus);
}


void CabinRestorationHandler::onCMPStatusUpdate(const SystemStatusInfo& info)
{
   bool systemStatus = info.getStatus() == SYSTEM_STATUS_CONNECTED ? true : false;
   evalAndUpdateSystemStatus(info, systemStatus);
}


void CabinRestorationHandler::onCMGStatusUpdate(const SystemStatusInfo& info)
{
   bool systemStatus = (info.getStatus() == SYSTEM_STATUS_CONNECTED) ? true : false;
   if (info.getStatus() == SYSTEM_STATUS_UNAVAILABLE)
   {
      if (NULL != _regionHandling)
      {
         const RegionsIdInfo& tRegionsIdInfo = info.getRegionsIdInfo();
         for (RegionsIdInfo::const_iterator itr = tRegionsIdInfo.begin(); (itr != tRegionsIdInfo.end()); ++itr)
         {
            SystemEnquiryInfo tSystemEnquiryInfo(static_cast<enRegionID>((*itr)), SYSTEM_CATEGORY_CMG, STATUS_CATEGORY_SYSTEM);
            bool isVideoMixGlassConfigured = _regionHandling->isCMGConfiguredForGroupCategory((*itr), SYSTEM_GROUP_CATEGORY_CMG_CABIN_MIX);
            bool isVideoMixIndexConfigured = ((NULL != _hmiInfoServiceServerComponent) && (_hmiInfoServiceServerComponent->getCabinVideoMixIndex((*itr)) != CABIN_VIDEO_MIX_INDEX_NONE)) ? true : false;
            if ((!isVideoMixGlassConfigured) || (!isVideoMixIndexConfigured))
            {
               if (!_regionHandling->isAllStatusSame(tSystemEnquiryInfo, SYSTEM_STATUS_UNAVAILABLE))
               {
                  evalAndUpdateSystemStatus(info, true, (*itr));
               }
               else
               {
                  evalAndUpdateSystemStatus(info, false, (*itr));
               }
            }
            else
            {
               SinkIdsInfo tInfo;
               if ((_regionHandling->fetchSinkIdsInfoForGroup(tInfo, info.getSystemGroup(), SYSTEM_CATEGORY_CMG, (*itr))) && (!tInfo.empty()))
               {
                  if (!_regionHandling->isAllStatusSame(tSystemEnquiryInfo, SYSTEM_STATUS_UNAVAILABLE, tInfo.front()))
                  {
                     evalAndUpdateSystemStatus(info, true, (*itr));
                  }
                  else
                  {
                     evalAndUpdateSystemStatus(info, false, (*itr));
                  }
               }
            }
         }
      }
   }
   else
   {
      evalAndUpdateSystemStatus(info, systemStatus);
   }
}


void CabinRestorationHandler::onCMRStatusUpdate(const SystemStatusInfo& info)
{
   bool systemStatus = info.getStatus() == SYSTEM_STATUS_CONNECTED ? true : false;
   evalAndUpdateSystemStatus(info, systemStatus);
}


void CabinRestorationHandler::sendCabinsRestorationInfo()
{
   if (NULL != _hmiInfoServiceServerComponent)
   {
      _hmiInfoServiceServerComponent->sendCabinsSourceRestorationInfo(_restoreRegionsInfo);
   }
   if (NULL != _cabinConnectionHandler)
   {
      _cabinConnectionHandler->onCabinsActivityRestorationInfo(_restoreRegionsInfo);
   }
}


void CabinRestorationHandler::evalAndUpdateRestoreStatus()
{
   if ((NULL != _syncDisplayHandler) && (_syncDisplayHandler->getFirstAppRequestedStatus()) && (evalAndUpdateSinkRestoreStatus(CABIN_RESTORE_STATUS_RESTORE)))
   {
      sendCabinsRestorationInfo();
   }
}


#ifdef VARIANT_S_FTR_ENABLE_MIC_ACTIVATION
void CabinRestorationHandler::evalAndUpdateWaitStatusForMic1()
{
   bool isModified  = false;
   bool tMic1Status = (NULL != _cabinConnectionHandler) ? _cabinConnectionHandler->getMic1Status() : false;
   if (tMic1Status)
   {
      isModified = evalAndUpdateSinkRestoreStatus(CABIN_RESTORE_STATUS_PAUSE_MIC1);
   }
   else
   {
      isModified = evalAndUpdateSinkRestoreStatus(CABIN_RESTORE_STATUS_EVALUATING);
   }
   if (isModified)
   {
      sendCabinsRestorationInfo();
      evalAndUpdateRestoreStatus();
   }
}


#endif /* VARIANT_S_FTR_ENABLE_MIC_ACTIVATION */

bool CabinRestorationHandler::evalAndUpdateSinkRestoreStatus(const uint8 status, const int16 sinkId, const uint8 group)
{
   bool isModified = false;
   bool terminate  = false;
   for (RestoreRegionsInfo::iterator regItr = _restoreRegionsInfo.begin(); ((regItr != _restoreRegionsInfo.end()) && (!terminate)); ++regItr)
   {
      ActivitiesRestoreSinksInfo* actRestoreSinksInfo = &((*regItr).getMutableActivitiesRestoreSinksInfo());
      if (actRestoreSinksInfo != NULL)
      {
         for (ActivitiesRestoreSinksInfo::iterator actItr = actRestoreSinksInfo->begin(); ((actItr != actRestoreSinksInfo->end()) && (!terminate)); ++actItr)
         {
            for (RestoreSinksInfoGroup::iterator itr = (*actItr).getMutableRestoreSinksInfoGroup().begin(); ((itr != (*actItr).getMutableRestoreSinksInfoGroup().end()) && (!terminate)); ++itr)
            {
               if (((*itr).first == group) || (group == SINKGROUP_NONE))
               {
                  for (RestoreSinksInfo::iterator titr = (*itr).second.begin(); ((titr != (*itr).second.end()) && (!terminate)); ++titr)
                  {
                     if (((*titr).getSinkId() == sinkId) || (sinkId == -1))
                     {
                        switch (status)
                        {
                           case CABIN_RESTORE_STATUS_RESTORE:
                           {
                              if (((*titr).getRestoreStatus() == CABIN_RESTORE_STATUS_EVALUATING) && ((*actItr).isAllSystemsReadyForGroup((*itr).first)))
                              {
                                 isModified = true;
                                 (*titr).setRestoreStatus(CABIN_RESTORE_STATUS_RESTORE);
                              }
                              break;
                           }
                           case CABIN_RESTORE_STATUS_PAUSE_MIC1:
                           {
                              if ((*titr).getRestoreStatus() == CABIN_RESTORE_STATUS_EVALUATING)
                              {
                                 isModified = true;
                                 (*titr).setRestoreStatus(CABIN_RESTORE_STATUS_PAUSE_MIC1);
                              }
                              break;
                           }
                           case CABIN_RESTORE_STATUS_EVALUATING:
                           {
                              if ((*titr).getRestoreStatus() != CABIN_RESTORE_STATUS_RESTORE)
                              {
                                 isModified = true;
                                 (*titr).setRestoreStatus(CABIN_RESTORE_STATUS_EVALUATING);
                              }
                              break;
                           }
                           default:
                              break;
                        }
                        ETG_TRACE_USR1(("CabinRestorationHandler: evalAndUpdateSinkRestoreStatus: SinkId: %d, RestoreStatus: %d", (*titr).getSinkId(), (*titr).getRestoreStatus()));
                        terminate = (sinkId != -1) ? true : false;
                     }
                  }
                  terminate = ((group != SINKGROUP_NONE) || terminate) ? true : false;
               }
            }
         }
      }
   }
   return isModified;
}


void CabinRestorationHandler::loadAmpSinkToRestoreRegionInfo(const CabinActivitiesInfo& activitiesInfo, RestoreRegionInfo& info, uint8 activityType)
{
   if (NULL != _regionHandling)
   {
      for (CabinActivitiesInfo::const_iterator itr = activitiesInfo.begin(); (itr != activitiesInfo.end()); ++itr)
      {
         if ((_regionHandling->getSystemCategoryForSinkId((*itr).getSinkId()) == SYSTEM_CATEGORY_CMA) && ((*itr).getActivityType() == activityType))
         {
            SystemsIdInfo tSystemsIdInfo;
            RestoreSinkInfo tRestoreSinkInfo;
            tRestoreSinkInfo.setState((*itr).getState());
            tRestoreSinkInfo.setSinkId((*itr).getSinkId());
            tRestoreSinkInfo.setSourceId((*itr).getSourceId());
            tRestoreSinkInfo.setLastState((*itr).getLastState());
            tRestoreSinkInfo.setSubSourceId((*itr).getSubSourceId());
            tRestoreSinkInfo.setLastSourceId((*itr).getLastSourceId());
            tRestoreSinkInfo.setLastSubSourceId((*itr).getLastSubSourceId());
            tRestoreSinkInfo.setRestoreStatus(CABIN_RESTORE_STATUS_EVALUATING);
            if (_regionHandling->fetchSystemsIdInfoForSinkId(tSystemsIdInfo, (*itr).getSinkId(), SYSTEM_CATEGORY_CMA, info.getRegionId()))
            {
               SystemEnquiryInfo temp(info.getRegionId(), SYSTEM_CATEGORY_CMA, STATUS_CATEGORY_SYSTEM);
               for (SystemsIdInfo::const_iterator tItr = tSystemsIdInfo.begin(); (tItr != tSystemsIdInfo.end()); ++tItr)
               {
                  bool tStatus = (_regionHandling->getStatus(temp, (*tItr)) == SYSTEM_STATUS_CONNECTED) ? true : false;
                  tRestoreSinkInfo.addToSystemsStatusInfo((*tItr));
                  tRestoreSinkInfo.setSystemStatus((*tItr), tStatus);
               }
            }
            info.addToActivity(activityType, tRestoreSinkInfo, SINKGROUP_CABIN);
            const RestoreSinkInfo* tImp = info.getRestoreSinkInfo((*itr).getSinkId(), SINKGROUP_CABIN, activityType);
            if (NULL != tImp)
            {
               ETG_TRACE_USR1(("CabinRestorationHandler: loadAmpSinkToRestoreRegionInfo: RegionId: %d, SinkId: %d, SourceId: %d, SubSourceId: %d", \
                               info.getRegionId(), tImp->getSinkId(), tImp->getSourceId(), tImp->getSubSourceId()));
               const SystemsStatusInfo& tSystemsStatusInfo = tImp->getSystemsStatusInfo();
               for (SystemsStatusInfo::const_iterator tItr = tSystemsStatusInfo.begin(); (tItr != tSystemsStatusInfo.end()); ++tItr)
               {
                  ETG_TRACE_USR1(("CabinRestorationHandler: loadAmpSinkToRestoreRegionInfo: RegionId: %d, SinkId: %d, SystemId: %d", \
                                  info.getRegionId(), tImp->getSinkId(), (*tItr).first));
               }
            }
            break;
         }
      }
   }
}


void CabinRestorationHandler::loadGlassSinkToRestoreRegionInfo(const CabinActivitiesInfo& activitiesInfo, RestoreRegionInfo& info, uint8 activityType)
{
   if (NULL != _regionHandling)
   {
      for (CabinActivitiesInfo::const_iterator itr = activitiesInfo.begin(); (itr != activitiesInfo.end()); ++itr)
      {
         if ((_regionHandling->getSystemCategoryForSinkId((*itr).getSinkId()) == SYSTEM_CATEGORY_CMG) && ((*itr).getActivityType() == activityType))
         {
            SystemsIdInfo tSystemsIdInfo;
            RestoreSinkInfo tRestoreSinkInfo;
            tRestoreSinkInfo.setState((*itr).getState());
            tRestoreSinkInfo.setSinkId((*itr).getSinkId());
            tRestoreSinkInfo.setSourceId((*itr).getSourceId());
            tRestoreSinkInfo.setLastState((*itr).getLastState());
            tRestoreSinkInfo.setSubSourceId((*itr).getSubSourceId());
            tRestoreSinkInfo.setLastSourceId((*itr).getLastSourceId());
            tRestoreSinkInfo.setLastSubSourceId((*itr).getLastSubSourceId());
            tRestoreSinkInfo.setRestoreStatus(CABIN_RESTORE_STATUS_EVALUATING);
            if (_regionHandling->fetchSystemsIdInfoForSinkId(tSystemsIdInfo, (*itr).getSinkId(), SYSTEM_CATEGORY_CMG, info.getRegionId()))
            {
               SystemEnquiryInfo temp(info.getRegionId(), SYSTEM_CATEGORY_CMG, STATUS_CATEGORY_SYSTEM);
               for (SystemsIdInfo::const_iterator tItr = tSystemsIdInfo.begin(); (tItr != tSystemsIdInfo.end()); ++tItr)
               {
                  bool tStatus = (_regionHandling->getStatus(temp, (*tItr)) == SYSTEM_STATUS_CONNECTED) ? true : false;
                  tRestoreSinkInfo.addToSystemsStatusInfo((*tItr));
                  tRestoreSinkInfo.setSystemStatus((*tItr), tStatus);
               }
            }
            uint8 group = SINKGROUP_CABIN;
            if (_regionHandling->getSystemGroupCategoryForSinkId((*itr).getSinkId()) == SYSTEM_GROUP_CATEGORY_CMG_CABIN_MIX)
            {
               if (NULL != _cabinVideoMixInfoDpHandler)
               {
                  group = (_cabinVideoMixInfoDpHandler->getVideoMixIndexForCabin(info.getRegionId()) == CABIN_VIDEO_MIX_INDEX_NONE) ? SINKGROUP_CABIN : SINKGROUP_VIDEOMIX;
               }
            }
            info.addToActivity(activityType, tRestoreSinkInfo, group);
            const RestoreSinkInfo* tImp = info.getRestoreSinkInfo((*itr).getSinkId(), group, activityType);
            if (NULL != tImp)
            {
               ETG_TRACE_USR1(("CabinRestorationHandler: loadGlassSinkToRestoreRegionInfo: RegionId: %d, SinkId: %d,  Group:%d, SourceId: %d, LastSourceId: %d", \
                               info.getRegionId(), tImp->getSinkId(), group, tImp->getSourceId(), tImp->getLastSourceId()));
               const SystemsStatusInfo& tSystemsStatusInfo = tImp->getSystemsStatusInfo();
               for (SystemsStatusInfo::const_iterator tItr = tSystemsStatusInfo.begin(); (tItr != tSystemsStatusInfo.end()); ++tItr)
               {
                  ETG_TRACE_USR1(("CabinRestorationHandler: loadGlassSinkToRestoreRegionInfo: RegionId: %d, SinkId: %d, SystemId: %d", \
                                  info.getRegionId(), tImp->getSinkId(), (*tItr).first));
               }
            }
         }
      }
   }
}


void CabinRestorationHandler::loadAmpSourceToRestoreRegionInfo(const CabinActivitiesInfo& activitiesInfo, RestoreRegionInfo& info, uint8 activityType)
{
   if (NULL != _regionHandling)
   {
      for (CabinActivitiesInfo::const_iterator itr = activitiesInfo.begin(); (itr != activitiesInfo.end()); ++itr)
      {
         if ((_regionHandling->getSystemCategoryForSinkId((*itr).getSinkId()) == SYSTEM_CATEGORY_CMA)  && ((*itr).getActivityType() == activityType))
         {
            RestoreSinkInfo* tRestoreSinkInfo = info.getMutableRestoreSinkInfo((*itr).getSinkId(), SINKGROUP_NONE, activityType);
            if (NULL != tRestoreSinkInfo)
            {
               SystemsIdInfo tSystemsIdInfo;
               if (_regionHandling->fetchSystemsIdInfoForSourceId(tSystemsIdInfo, (*itr).getSourceId(), SYSTEM_CATEGORY_NONE, REGION_INVALID))
               {
                  SystemEnquiryInfo temp(info.getRegionId(), SYSTEM_CATEGORY_CMA, STATUS_CATEGORY_SYSTEM);
                  for (SystemsIdInfo::const_iterator tItr = tSystemsIdInfo.begin(); (tItr != tSystemsIdInfo.end()); ++tItr)
                  {
                     bool tStatus = (_regionHandling->getStatus(temp, (*tItr)) == SYSTEM_STATUS_CONNECTED) ? true : false;
                     tRestoreSinkInfo->addToSystemsStatusInfo((*tItr));
                     tRestoreSinkInfo->setSystemStatus((*tItr), tStatus);
                     ETG_TRACE_USR1(("CabinRestorationHandler: loadAmpSourceToRestoreRegionInfo: RegionId: %d, SinkId: %d, SystemId: %d", \
                                     info.getRegionId(), tRestoreSinkInfo->getSinkId(), (*tItr)));
                  }
               }
            }
            else
            {
               ETG_TRACE_ERR(("CabinRestorationHandler: loadAmpSourceToRestoreRegionInfo: Amp sink is not loaded: RegionId : %d, SinkId: %d", info.getRegionId(), (*itr).getSinkId()));
            }
            break;
         }
      }
   }
}


void CabinRestorationHandler::loadGlassSourceToRestoreRegionInfo(const CabinActivitiesInfo& activitiesInfo, RestoreRegionInfo& info, uint8 activityType)
{
   if (NULL != _regionHandling)
   {
      for (CabinActivitiesInfo::const_iterator itr = activitiesInfo.begin(); (itr != activitiesInfo.end()); ++itr)
      {
         if ((_regionHandling->getSystemCategoryForSinkId((*itr).getSinkId()) == SYSTEM_CATEGORY_CMG)  && ((*itr).getActivityType() == activityType))
         {
            RestoreSinkInfo* tRestoreSinkInfo = info.getMutableRestoreSinkInfo((*itr).getSinkId(), SINKGROUP_NONE, activityType);
            if (NULL != tRestoreSinkInfo)
            {
               SystemsIdInfo tSystemsIdInfo;
               if (_regionHandling->fetchSystemsIdInfoForSourceId(tSystemsIdInfo, (*itr).getSourceId(), SYSTEM_CATEGORY_NONE, REGION_INVALID))
               {
                  SystemEnquiryInfo temp(info.getRegionId(), SYSTEM_CATEGORY_CMG, STATUS_CATEGORY_SYSTEM);
                  for (SystemsIdInfo::const_iterator tItr = tSystemsIdInfo.begin(); (tItr != tSystemsIdInfo.end()); ++tItr)
                  {
                     bool tStatus = (_regionHandling->getStatus(temp, (*tItr)) == SYSTEM_STATUS_CONNECTED) ? true : false;
                     tRestoreSinkInfo->addToSystemsStatusInfo((*tItr));
                     tRestoreSinkInfo->setSystemStatus((*tItr), tStatus);
                     ETG_TRACE_USR1(("CabinRestorationHandler: loadGlassSourceToRestoreRegionInfo: RegionId: %d, SinkId: %d, SystemId: %d", \

                                     info.getRegionId(), tRestoreSinkInfo->getSinkId(), (*tItr)));
                  }
               }
            }
            else
            {
               ETG_TRACE_ERR(("CabinRestorationHandler: loadGlassSourceToRestoreRegionInfo: Glass sink is not loaded: RegionId : %d, SinkId: %d", info.getRegionId(), (*itr).getSinkId()));
            }
         }
      }
   }
}


void CabinRestorationHandler::loadAmpSystemToRestoreRegionInfoGlassSink(const CabinActivitiesInfo& activitiesInfo, RestoreRegionInfo& info, uint8 activityType)
{
   if (NULL != _regionHandling)
   {
      for (CabinActivitiesInfo::const_iterator itr = activitiesInfo.begin(); (itr != activitiesInfo.end()); ++itr)
      {
         if ((_regionHandling->getSystemCategoryForSinkId((*itr).getSinkId()) == SYSTEM_CATEGORY_CMG)  && ((*itr).getActivityType() == activityType))
         {
            RestoreSinkInfo* tRestoreSinkInfo = info.getMutableRestoreSinkInfo((*itr).getSinkId(), SINKGROUP_NONE, activityType);
            if (NULL != tRestoreSinkInfo)
            {
               SystemsIdInfo tSystemsIdInfo;
               if (_regionHandling->fetchSystemsIdInfo(tSystemsIdInfo, SYSTEM_CATEGORY_CMA, info.getRegionId()))
               {
                  SystemEnquiryInfo temp(info.getRegionId(), SYSTEM_CATEGORY_CMA, STATUS_CATEGORY_SYSTEM);
                  for (SystemsIdInfo::const_iterator tItr = tSystemsIdInfo.begin(); (tItr != tSystemsIdInfo.end()); ++tItr)
                  {
                     bool tStatus = (_regionHandling->getStatus(temp, (*tItr)) == SYSTEM_STATUS_CONNECTED) ? true : false;
                     tRestoreSinkInfo->addToSystemsStatusInfo((*tItr));
                     tRestoreSinkInfo->setSystemStatus((*tItr), tStatus);
                     ETG_TRACE_USR1(("CabinRestorationHandler: loadAmpSystemToGlassSink: RegionId: %d, SystemId: %d", \
                                     info.getRegionId(), (*tItr)));
                  }
               }
            }
            else
            {
               ETG_TRACE_ERR(("CabinRestorationHandler: loadGlassSourceToRestoreRegionInfo: Glass sink is not loaded: RegionId : %d", info.getRegionId()));
            }
         }
      }
   }
}


bool CabinRestorationHandler::isLastSystemStatusPNMDisconnected(const uint8 regionId)
{
   bool tStatus = false;
   if (NULL != _regionHandling)
   {
      SystemEnquiryInfo temp(regionId, SYSTEM_CATEGORY_CMA, STATUS_CATEGORY_SYSTEM);
      tStatus = (_regionHandling->getLastStatus(temp) == SYSTEM_STATUS_PNM_DISCONNECTED) ? true : false;
   }
   return tStatus;
}


bool CabinRestorationHandler::isLastSystemStatusCockpitModeDisconnected(const uint8 regionId)
{
   bool tStatus = false;
   if (NULL != _regionHandling)
   {
      SystemEnquiryInfo temp(regionId, SYSTEM_CATEGORY_CMA, STATUS_CATEGORY_SYSTEM);
      tStatus = (_regionHandling->getLastStatus(temp) == SYSTEM_STATUS_COCKPIT_DISCONNECTED) ? true : false;
   }
   return tStatus;
}


void CabinRestorationHandler::evalAndUpdateSystemStatus(const SystemStatusInfo& info, bool systemStatus, uint32 regionId)
{
   bool isModified = false;
   for (RestoreRegionsInfo::iterator itr = _restoreRegionsInfo.begin(); (itr != _restoreRegionsInfo.end()); ++itr)
   {
      if ((regionId == REGION_INVALID) || (regionId == (*itr).getRegionId()))
      {
         isModified = (((*itr).setSystemStatus(info.getSystemId(), systemStatus)) || (isModified)) ? true : false;
         if (regionId != REGION_INVALID)
         {
            break;
         }
      }
   }
   if (isModified)
   {
      evalAndUpdateRestoreStatus();
   }
}


std::vector< uint8 > CabinRestorationHandler::getActivitiesInfo(const uint8 regionId, const CabinActivitiesInfo& info)
{
   std::vector< uint8 > activitiesInfo;
   if (NULL != _regionHandling)
   {
      for (CabinActivitiesInfo::const_iterator itr = info.begin(); (itr != info.end()); ++itr)
      {
         if ((_regionHandling->getSystemCategoryForSinkId((*itr).getSinkId()) == SYSTEM_CATEGORY_CMA) &&
               (isActivityValidToRestore(regionId, (*itr).getActivityType())))
         {
            activitiesInfo.push_back((*itr).getActivityType());
            break;
         }
      }
      for (CabinActivitiesInfo::const_iterator itr = info.begin(); (itr != info.end()); ++itr)
      {
         if ((_regionHandling->getSystemCategoryForSinkId((*itr).getSinkId()) == SYSTEM_CATEGORY_CMG) &&
               (isActivityValidToRestore(regionId, (*itr).getActivityType())))
         {
            std::vector< uint8 >::iterator titr = std::find(activitiesInfo.begin(), activitiesInfo.end(), (*itr).getActivityType());
            if (titr == activitiesInfo.end())
            {
               activitiesInfo.push_back((*itr).getActivityType());
            }
         }
      }
   }
   for (std::vector< uint8 >::const_iterator itr = activitiesInfo.begin(); (itr != activitiesInfo.end()); ++itr)
   {
      ETG_TRACE_USR1(("CabinRestorationHandler: getActivitiesInfo: ActivityType: %d", (*itr)));
   }
   return activitiesInfo;
}


bool CabinRestorationHandler::isActivityValidToRestore(const uint8 regionId, const uint8 activityType)
{
   bool isValidToRestore = true;

   ActivitiesValidatorFunctionsInfo::iterator titr = _activitiesValidatorFunctionsInfo.find(activityType);
   if (titr != _activitiesValidatorFunctionsInfo.end())
   {
      for (ValidatorFunctionsInfo::iterator sItr = titr->second.begin(); (sItr != (titr->second.end())); ++sItr)
      {
         isValidToRestore = ((((this)->*((*sItr).first))(regionId)) == ((*sItr).second)) ? true : false;
         if (!isValidToRestore)
         {
            break;
         }
      }
   }
   ETG_TRACE_USR1(("CabinRestorationHandler: isActivityValidToRestore: RegionId: %d, ActivityType: %d, isValidToRestore: %d", regionId, activityType, isValidToRestore));
   return isValidToRestore;
}


void CabinRestorationHandler::onUserInteraction(const uint8 regionId, const uint32 sinkId)
{
   bool isModified = false;
   for (RestoreRegionsInfo::iterator itr = _restoreRegionsInfo.begin(); (itr != _restoreRegionsInfo.end()); ++itr)
   {
      if ((*itr).getRegionId() == regionId)
      {
         if ((*itr).abortRestoreForSink(sinkId))
         {
            isModified = true;
            ETG_TRACE_USR1(("CabinRestorationHandler: onUserInteractionCabin: Erased sinkId: %d", sinkId));
         }
         break;
      }
   }
   if (isModified)
   {
      sendCabinsRestorationInfo();
   }
}


void CabinRestorationHandler::onUserInteraction(const uint8 regionId, const SinkIdsInfo sinkIds)
{
   bool isModified = false;
   for (RestoreRegionsInfo::iterator itr = _restoreRegionsInfo.begin(); (itr != _restoreRegionsInfo.end()); ++itr)
   {
      if ((*itr).getRegionId() == regionId)
      {
         for (SinkIdsInfo::const_iterator tItr = sinkIds.begin(); (tItr != sinkIds.end()); ++tItr)
         {
            if ((*itr).abortRestoreForSink((*tItr)))
            {
               isModified = true;
               ETG_TRACE_USR1(("CabinRestorationHandler: onUserInteractionCabin: Erased sinkId: %d isModified = %d", (*tItr), isModified));
            }
         }
         break;
      }
   }
   if (isModified)
   {
      sendCabinsRestorationInfo();
   }
}


uint8 CabinRestorationHandler::getSinkRestoreStatus(const uint8 sinkId, const uint8 regionId, const uint8 group)
{
   uint8 restoreStatus = CABIN_RESTORE_STATUS_NONE;
   for (RestoreRegionsInfo::iterator itr = _restoreRegionsInfo.begin(); (itr != _restoreRegionsInfo.end()); ++itr)
   {
      if ((*itr).getRegionId() == regionId)
      {
         restoreStatus = (*itr).getSinkRestoreStatus(sinkId, group);
         break;
      }
   }
   ETG_TRACE_USR1(("CabinRestorationHandler: getSinkRestoreStatus: SinkId: %d RestoreStatus: %d", sinkId, restoreStatus));
   return restoreStatus;
}


} //namespace Core
} //namespace App
