/**
 * @file         : CabinConnectionHandler.h
 * @author       : INF4CV - pgi5cob
 * @addtogroup   : AppHmi_Master
 * @brief        : Handles connection/disconnection of cabin sources to glass / sink
 * @copyright    : (C) 2016 Robert Bosch GmbH
 *                 (C) 2016 Robert Bosch Engineering and Business Solutions Limited
 *                 The reproduction, distribution and utilization of this file as
 *                 well as the communication of its contents to others without express
 *                 authorization is prohibited. Offenders will be held liable for the
 *                 payment of damages. All rights reserved in the event of the grant
 *                 of a patent, utility model or design.
 */


#ifndef CABINCONNECTIONHANDLER_H
#define CABINCONNECTIONHANDLER_H

#include "CabinSrcInfo.h"
#include "hmi_trace_if.h"
#include "CabinConnectionHandlerTypes.h"
#include "CabinConnectionHandlerInterface.h"
#include "CabinsActivityInfoDpHandlerInterface.h"
#include <asf/core/Timer.h>
#include <Core/RegionHandling/RegionHandlingInterface.h>
#include <Core/AudioInterface/ResourceControllerPrjInterface.h>
#ifdef VARIANT_S_FTR_ENABLE_MIC_ACTIVATION
#include <App/Core/HmiState/HmiStateHandlerInterface.h>
#include <App/Core/NanoMsgClientHandler/NanoMsgClientHandlerInterface.h>
#endif


#define HMI_INFOSERVICE_INTERFACE ::bosch::cm::ai::hmi::hmiinfoservice::HmiInfoService


/**
 *  Classes Forward Declaration
 */

class IHmiInfoServiceServerComponent;

namespace App {
namespace Core {


class SystemStatusInfo;
class IHmiStateHandler;
class IMonoGlassController;
class IMIC1ConnectionStatus;
class IMixSrcConnectionStatus;
class IMapOutStatusListenerCB;
class ICabinRestorationHandler;
class ICabinConnectionController;
class ICabinVideoMixInfoDpHandler;
} //namespace Core
} //namespace App


/**
 *  Classes Declaration
 */

namespace App {
namespace Core {


class CabinConnectionHandler : public ICabinConnectionHandler
   , public ICMAStatusListenerCB
   , public ICabinActivityInfoUpdater
   , public ::asf::core::TimerCallbackIF
#ifdef VARIANT_S_FTR_ENABLE_MIC_ACTIVATION
   , public IHmiStateListenerCB
   , public ICMGStatusListenerCB
   , public IUpdateSinkListenerCB
   , public INanoMSGMic2StatusUpdCB
   , public INanoMSGMic3StatusUpdCB
   , public IUpdateSourceListListenerCB
#endif /* VARIANT_S_FTR_ENABLE_MIC_ACTIVATION */
{
   public:
      /**
       *  Member Functions Declaration
       */
      CabinConnectionHandler();
      virtual ~CabinConnectionHandler();
      bool onSourceRequest(const int16 srcId, const int16 devId, const int16 sinkId);
      void finalize();
      void initialize();
      void onCMAStatusUpdate(const SystemStatusInfo& info);
      void onCabinsActivityRestorationInfo(const RestoreRegionsInfo& info);
      void onSourceDisconnect(const int16 srcId, const int16 devId, const int16 sinkId);
      void onExpired(::asf::core::Timer& timer, ::boost::shared_ptr< ::asf::core::TimerPayload > data);
      void onSetCabinVideoMixInfo(const HMI_INFOSERVICE_INTERFACE::CabinVideoMixInfo cabinVideoMixInfoUpdate);
      void updateCabinConnections(const SinkConnectionMap& newSinkInfos, const SinkConnectionMap& suspendedSinkInfos);
      void updateCabinConnectionsForMixSource(const SinkConnectionMap& newSinkInfos, const SinkConnectionMap& suspendedSinkInfos);
      void onSetCabinVideoMixInfo(const uint32 regionId, const uint8 videoMixIndex, const uint8 reason = CABIN_VIDEO_MIX_CHANGE_REASON_USER);
#ifdef VARIANT_S_FTR_ENABLE_MAP_STREAMING
      uint8 getCabinMapOutStatus(const uint32 regionId) const;
      void onToggleMapOutStatus(const uint32 regionId);
      void onCMGStatusUpdate(const SystemStatusInfo& info);
      void onMapOutStatusUpdMsg(const uint32 regionId, const bool status);
#endif /* VARIANT_S_FTR_ENABLE_MAP_STREAMING */
#ifdef VARIANT_S_FTR_ENABLE_MIC_ACTIVATION
      bool getMic1Status();
      void onUpdateMic1Status(const bool status);
      void onUpdateMic2Status(const bool status);
      void onUpdateMic3Status(const bool status);
      void onUpdateSinkChange(const uint32 sinkIndex);
      void onHmiStateUpdate(const uint8 currentHmiState);
      void onUpdateSourceListChange(const uint32 srcIndex);
#endif /* VARIANT_S_FTR_ENABLE_MIC_ACTIVATION */

      inline void setRegionHandling(IRegionHandling& imp)
      {
         _regionHandling = (&imp);
      }

      inline void setHMIStateHandler(IHmiStateHandler& imp)
      {
         _hmiStateHandler = (&imp);
      }

      inline void setCabinRestorationHandler(ICabinRestorationHandler& imp)
      {
         _cabinRestorationHandler = (&imp);
      }

      inline void setCabinConnectionUtility(CabinConnectionUtility& imp)
      {
         _cabinConnectionUtility = (&imp);
      }

      inline void setCabinVideoMixInfoDpHandler(ICabinVideoMixInfoDpHandler& imp)
      {
         _cabinsVideoMixInfoDpHandler = (&imp);
      }

      inline void setCabinsActivityInfoDpHandler(ICabinsActivityInfoDpHandler& imp)
      {
         _cabinsActivityInfoDpHandler = (&imp);
      }

      inline void setHmiInfoServiceServerComponent(IHmiInfoServiceServerComponent& imp)
      {
         _hmiInfoServiceServerComponent = (&imp);
      }

      inline void setResourceController(const ::boost::shared_ptr< IResourceControllerPrj >& imp)
      {
         _resourceController = imp;
      }

#ifdef VARIANT_S_FTR_ENABLE_MIC_ACTIVATION
      inline void setNanoMsgClientHandler(INanoMsgClientHandler& imp)
      {
         _nanoMsgClientHandler = (&imp);
      }

      void addMic1StatusListener(IMIC1ConnectionStatus* imp)
      {
         addListener<IMIC1ConnectionStatus>(_mic1ConnectionStatusListener, imp);
      }

      void removeMic1StatusListener(IMIC1ConnectionStatus* imp)
      {
         removeListener<IMIC1ConnectionStatus>(_mic1ConnectionStatusListener, imp);
      }

      void addMixSrcStatusListener(IMixSrcConnectionStatus* imp)
      {
         addListener<IMixSrcConnectionStatus>(_mixSourceConnectionStatusListener, imp);
      }

      void removeMixSourceStatusListener(IMixSrcConnectionStatus* imp)
      {
         removeListener<IMixSrcConnectionStatus>(_mixSourceConnectionStatusListener, imp);
      }
#endif /* VARIANT_S_FTR_ENABLE_MIC_ACTIVATION */

#ifdef VARIANT_S_FTR_ENABLE_MAP_STREAMING
      inline void addMapOutStateListener(IMapOutStatusListenerCB& imp)
      {
         addListener<IMapOutStatusListenerCB>(_mapOutStatusListeners, (&imp));
      }

      inline void removeMapOutStateListener(IMapOutStatusListenerCB& imp)
      {
         removeListener<IMapOutStatusListenerCB>(_mapOutStatusListeners, (&imp));
      }
#endif /* VARIANT_S_FTR_ENABLE_MAP_STREAMING */

   private:
      /**
       *  Member Functions Declaration
       */
      uint32 getRegionIdForSinkId(const int16 sinkId) const;
      bool isMixSourceActive(const int16 srcId);
      void updateActiveConnectionDP();
      void updateMapoutConnectionDP();
      void evalAndResetRequestQueues();
      void updateCabinsActivityInfoDp();
      void executeSourceRequestsQueue();
      void evalAndUpdateMonoModeStatus();
      void executeConnectionRequestsQueue();
      void evalAndUpdateVideoMixController();
      void executeMixSourceConnectionRequestsQueue();
      void startConnectionReqTimer();
      void startMixConnectionReqTimer();
      void stopConnectionReqTimer(const bool flush);
      void stopMixConnectionReqTimer(const bool flush);
      void onConnectionRemoved(const HB_AUDIO::stSourceInfo& info);
      void onNewConnectionUpdate(const HB_AUDIO::stSourceInfo& info);
      void clearRequestQueuesOnCMAError(const SystemStatusInfo& info);
      void updateCabinVideoMixInfoProperty(const CabinsVideoMixInfo& info);
      void onNewConnectionForMixSource(const HB_AUDIO::stSourceInfo& info);
      bool isMixSourceConnectionAvailable(const HB_AUDIO::stSourceInfo info);
      void updateCabinVideoMixInfoForCMAStatus(const SystemStatusInfo& info);
      void removeSinkIdfromSourceQueue(const int16 sinkId);
      void removeSinkIdfromConnectionQueue(const int16 sinkId);
      void clearMixSourceRequestQueuesOnCMAError(const SystemStatusInfo& info);
      void onConnectionRemovedForMixSource(const HB_AUDIO::stSourceInfo& info);
      void checkAndUpdateNewMixSourceConnections(const SinkConnectionMap& info);
      void updateVideoMixPropertyOnRestoreAbort(const RestoreRegionsInfo& info);
      void handleSuspendedMixSources(const SinkConnectionMap& suspendedSinkInfos);
      void checkAndUpdateActiveMixSourceConnections(const SinkConnectionMap& info);
      void printSinkMapData(const SinkConnectionMap& info, const uint8 sourceType);
      void sendPluginMsgForMIXSource(const int16 mixSourceIndex, const int16 sinkId);
      CabinConnectionHandlerInfo getCabinConnectionHandlerInfo(const HB_AUDIO::stSourceInfo& info);
      void updateMixSourceConnectionStatus(const int16 sourceId, const bool status, const int16 sinkId);
      void evalAndSyncMixGlassesConnections(const uint32 regionId, const uint8 reason, const CabinsVideoMixInfo& lastInfo);

      template <typename controllerType>
      controllerType* getControllerInstance()
      {
         controllerType* imp = NULL;
         for (CabinConnectionControllersInfo::iterator itr = _cabinConnectionControllers.begin(); (itr != _cabinConnectionControllers.end()); ++itr)
         {
            controllerType* temp = dynamic_cast< controllerType* >((*itr));
            if (NULL != temp)
            {
               imp = temp;
               break;
            }
         }
         return imp;
      }

      template <typename controllerType>
      void removeControllerInstance()
      {
         for (CabinConnectionControllersInfo::iterator itr = _cabinConnectionControllers.begin(); (itr != _cabinConnectionControllers.end()); ++itr)
         {
            controllerType* temp = dynamic_cast< controllerType* >((*itr));
            if (NULL != temp)
            {
               delete (*itr);
               (*itr) = NULL;
               (void)_cabinConnectionControllers.erase(itr);
               break;
            }
         }
      }

      template <typename CBType>
      inline void addListener(::std::vector< CBType* >& info, CBType* imp)
      {
         if (::std::find(info.begin(), info.end(), imp) == info.end())
         {
            info.push_back(imp);
         }
      }

      template< typename CBType >
      inline void removeListener(::std::vector< CBType* >& info, CBType* imp)
      {
         typename ::std::vector< CBType* >::iterator itr = ::std::find(info.begin(), info.end(), imp);
         if (itr != info.end())
         {
            info.erase(itr);
         }
      }

#ifdef VARIANT_S_FTR_ENABLE_MIC_ACTIVATION
      bool isValidToActivateMIC1(const uint32 regionId);
      bool isValidToActivateMICMixSource(const uint32 regionId, const uint32 sourceId);
      void updateMIC1ConnectionStatus(const bool status);
      void updateMicSourceRestoration(const RegionsIdInfo& info);
      void updateMicSourceDisconnection(const SystemStatusInfo& info);
      void handleMic1Status(const bool status, const RegionsIdInfo& info);
      void handleMic2Status(const bool status, const RegionsIdInfo& info);
      void handleMic3Status(const bool status, const RegionsIdInfo& info);
      void setSysMuteStatusForMic1(const bool status, const uint32 regionId);
      void sendPluginMsgForMICBeep(const int16 sinkId, const uint8 beepOutput);
#endif /* VARIANT_S_FTR_ENABLE_MIC_ACTIVATION */
#ifdef VARIANT_S_FTR_ENABLE_MAP_STREAMING
      void onMapOutActivation(RegionMapOutInfo* imp);
      void onMapOutDeactivation(RegionMapOutInfo* imp);
      void restoreMapOut(const RestoreRegionsInfo& info);
      void clearMapOutInfoOnCMAError(const SystemStatusInfo& info);
      void activateMapOut(const uint32 regionId, const SinkIdsInfo& info);
      void deactivateMapOut(const uint32 regionId, const SinkIdsInfo& info);
      void sendMapOutRequestToNavi(const uint32 regionId, const bool status);
      void sendMapOutUpdate(const uint32 regionId, const SinksMapOutInfo& info);

      inline const RegionMapOutInfo* getRegionMapOutInfo(const uint32 regionId) const
      {
         const RegionMapOutInfo* imp = NULL;
         for (RegionsMapOutInfo::const_iterator itr = _regionsMapOutInfo.begin(); (itr != _regionsMapOutInfo.end()); ++itr)
         {
            if ((*itr).getRegionId() == regionId)
            {
               imp = (&(*itr));
               break;
            }
         }
         return imp;
      }

      inline RegionMapOutInfo* getMutableRegionMapOutInfo(const uint32 regionId)
      {
         RegionMapOutInfo* imp = const_cast< RegionMapOutInfo* >(getRegionMapOutInfo(regionId));
         return imp;
      }

      inline void addToRegionsMapOutInfo(const uint32 regionId)
      {
         if (NULL == getRegionMapOutInfo(regionId))
         {
            _regionsMapOutInfo.push_back(RegionMapOutInfo(regionId));
         }
      }

      inline void removeFromRegionsMapOutInfo(const uint32 regionId)
      {
         for (RegionsMapOutInfo::iterator itr = _regionsMapOutInfo.begin(); (itr != _regionsMapOutInfo.end()); ++itr)
         {
            if ((*itr).getRegionId() == regionId)
            {
               (void)_regionsMapOutInfo.erase(itr);
               break;
            }
         }
      }
#endif /* VARIANT_S_FTR_ENABLE_MAP_STREAMING */
      /**
       *  Member Variables Declaration
       */
      bool _mic1Status;
      RequestsQueueInfo _srcReqQueue;
      RequestsQueueInfo _connReqQueue;
      IRegionHandling* _regionHandling;
      ::asf::core::Timer* _connReqTimer;
      IHmiStateHandler* _hmiStateHandler;
      ::asf::core::Timer* _mixConnReqTimer;
      RequestsQueueInfo _mixSrcConnReqQueue;
      SinkConnectionMap _activeConnectionsInfo;
      CabinConnectionUtility* _cabinConnectionUtility;
      SinkConnectionMap _mixSourceActiveConnectionsInfo;
      ICabinRestorationHandler* _cabinRestorationHandler;
      ICabinVideoMixInfoDpHandler* _cabinsVideoMixInfoDpHandler;
      ICabinsActivityInfoDpHandler* _cabinsActivityInfoDpHandler;
      CabinConnectionControllersInfo _cabinConnectionControllers;
      IHmiInfoServiceServerComponent* _hmiInfoServiceServerComponent;
      ::boost::shared_ptr< IResourceControllerPrj > _resourceController;
#ifdef VARIANT_S_FTR_ENABLE_MIC_ACTIVATION
      INanoMsgClientHandler* _nanoMsgClientHandler;
      ::std::vector< IMIC1ConnectionStatus* > _mic1ConnectionStatusListener;
      ::std::vector< IMixSrcConnectionStatus* > _mixSourceConnectionStatusListener;
#endif /* VARIANT_S_FTR_ENABLE_MIC_ACTIVATION */
#ifdef VARIANT_S_FTR_ENABLE_MAP_STREAMING
      RegionsMapOutInfo _regionsMapOutInfo;
      ::std::vector< IMapOutStatusListenerCB* > _mapOutStatusListeners;
#endif /* VARIANT_S_FTR_ENABLE_MAP_STREAMING */
};


} //namespace Core
} //namespace App


#undef HMI_INFOSERVICE_INTERFACE

#endif
