/**
 * @file         : CabinConnectionHandler.h
 * @author       : INF4CV - pgi5cob
 * @addtogroup   : AppHmi_Master
 * @brief        : Handles connection/disconnection of cabin sources to glass / sink
 * @copyright    : (C) 2016 Robert Bosch GmbH
 *                 (C) 2016 Robert Bosch Engineering and Business Solutions Limited
 *                 The reproduction, distribution and utilization of this file as
 *                 well as the communication of its contents to others without express
 *                 authorization is prohibited. Offenders will be held liable for the
 *                 payment of damages. All rights reserved in the event of the grant
 *                 of a patent, utility model or design.
 */

#include "CabinConnectionHandler.h"
#include "VideoMixController.h"
#include "MonoGlassController.h"
#include "AppHmi_MasterMessages.h"
#include "CabinConnectionUtility.h"
#include "CabinRestorationHandlerTypes.h"
#include "AudioControllerObjectManager.h"
#include "CabinRestorationHandlerInterface.h"
#include "CabinConnectionControllerInterface.h"
#include "CabinVideoMixInfoDpHandlerInterface.h"
#include <App/DataModel/DataModel.h>
#include <Core/Utility/MasterUtility.h>
#include <datapool/CabinActivityInfoDPIF.h>
#include <datapool/CabinVideoMixInfoDPIF.h>
#include <PluginConstants/PluginConstants.h>
#include <Core/HmiState/HmiStateHandlerHelper.h>
#include <Core/HmiState/HmiStateHandlerInterface.h>
#include <Core/AudioInterface/SoundControllerPrj.h>
#include <Core/RegionHandling/RegionHandlingTypes.h>
#include <AppHmi_MasterBase/AudioInterface/Source.h>
#include <Core/AudioInterface/ConnectionControllerPrj.h>
#include <Core/AudioInterface/ConnectionControllerPrj.h>
#include <bosch/cm/ai/hmi/hmiinfoservice/HmiInfoService.h>
#include <AppHmi_MasterBase/AudioInterface/AudioControllerObjectManager.h>
#include <Core/HmiInfoService/HmiInfoServiceServerComponentHandlingInterface.h>
#ifdef VARIANT_S_FTR_ENABLE_MIC_ACTIVATION
#include <App/Core/NanoMsgClientHandler/NanoMsgClientHandler.h>
#endif /* VARIANT_S_FTR_ENABLE_MIC_ACTIVATION */


using namespace ::hmibase::apphmi_master::audio;
using namespace ::bosch::cm::ai::hmi::hmiinfoservice::HmiInfoService;


#ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
#define ETG_DEFAULT_TRACE_CLASS TR_CLASS_APPHMI_MASTER_MAIN
#include "trcGenProj/Header/CabinConnectionHandler.cpp.trc.h"
#endif


namespace App {
namespace Core {


CabinConnectionHandler::CabinConnectionHandler() : _mic1Status(false)
   , _regionHandling(NULL)
   , _connReqTimer(NULL)
   , _hmiStateHandler(NULL)
   , _mixConnReqTimer(NULL)
   , _cabinConnectionUtility(NULL)
   , _cabinRestorationHandler(NULL)
   , _cabinsVideoMixInfoDpHandler(NULL)
   , _cabinsActivityInfoDpHandler(NULL)
   , _hmiInfoServiceServerComponent(NULL)
#ifdef VARIANT_S_FTR_ENABLE_MIC_ACTIVATION
   , _nanoMsgClientHandler(NULL)
#endif /* VARIANT_S_FTR_ENABLE_MIC_ACTIVATION */
{
   ETG_TRACE_USR4(("CabinConnectionHandler: CTOR"));
   _cabinConnectionControllers.push_back(new MonoGlassController());
#ifdef VARIANT_S_FTR_ENABLE_MAP_STREAMING
   _regionsMapOutInfo.clear();
#endif /* VARIANT_S_FTR_ENABLE_MAP_STREAMING */
}


CabinConnectionHandler::~CabinConnectionHandler()
{
   ETG_TRACE_USR4(("CabinConnectionHandler: DTOR"));
   _mic1Status                       = false;
   _regionHandling                   = NULL;
   _hmiStateHandler                  = NULL;
   _cabinRestorationHandler          = NULL;
   _cabinConnectionUtility           = NULL;
   _cabinsVideoMixInfoDpHandler      = NULL;
   _cabinsActivityInfoDpHandler      = NULL;
   _hmiInfoServiceServerComponent    = NULL;
   _srcReqQueue.clear();
   _connReqQueue.clear();
   _mixSrcConnReqQueue.clear();
   _activeConnectionsInfo.clear();
   _mixSourceActiveConnectionsInfo.clear();
   for (CabinConnectionControllersInfo::iterator itr = _cabinConnectionControllers.begin(); (itr != _cabinConnectionControllers.end()); ++itr)
   {
      if (NULL != (*itr))
      {
         delete (*itr);
         (*itr) = NULL;
      }
   }
   stopConnectionReqTimer(true);
   stopMixConnectionReqTimer(true);
   _cabinConnectionControllers.clear();
#ifdef VARIANT_S_FTR_ENABLE_MIC_ACTIVATION
   _nanoMsgClientHandler = NULL;
   _mic1ConnectionStatusListener.clear();
   _mixSourceConnectionStatusListener.clear();
#endif
#ifdef VARIANT_S_FTR_ENABLE_MAP_STREAMING
   _mapOutStatusListeners.clear();
#endif
}


void CabinConnectionHandler::initialize()
{
   if (NULL != _regionHandling)
   {
      _regionHandling->addCMAStatusListener((*this));
#ifdef VARIANT_S_FTR_ENABLE_MAP_STREAMING
      if (fnGetVariantType() ==  VARIANT_TYPE_NAVI)
      {
         _regionHandling->addCMGStatusListener((*this));
      }
#endif /* VARIANT_S_FTR_ENABLE_MAP_STREAMING */
   }
   for (CabinConnectionControllersInfo::const_iterator itr = _cabinConnectionControllers.begin(); (itr != _cabinConnectionControllers.end()); ++itr)
   {
      if (NULL != (*itr))
      {
         (*itr)->setCabinConnectionUtility(*_cabinConnectionUtility);
         (*itr)->setHmiInfoServiceServerComponent(*_hmiInfoServiceServerComponent);
      }
   }
#ifdef VARIANT_S_FTR_ENABLE_MIC_ACTIVATION
   if (NULL != _hmiStateHandler)
   {
      _hmiStateHandler->addHmiStateListener((this));
   }
   if (NULL != _resourceController.get())
   {
      _resourceController->addSourceListUpdateListener((*this));
      _resourceController->addSinkUpdateListener((*this));
   }
   if (NULL != _nanoMsgClientHandler)
   {
      _nanoMsgClientHandler->addMic2StatusListener((*this));
      _nanoMsgClientHandler->addMic3StatusListener((*this));
   }
#endif /* VARIANT_S_FTR_ENABLE_MIC_ACTIVATION */
}


void CabinConnectionHandler::finalize()
{
   if (NULL != _regionHandling)
   {
      _regionHandling->removeCMAStatusListener((*this));
#ifdef VARIANT_S_FTR_ENABLE_MAP_STREAMING
      if (fnGetVariantType() ==  VARIANT_TYPE_NAVI)
      {
         _regionHandling->removeCMGStatusListener((*this));
      }
#endif /* VARIANT_S_FTR_ENABLE_MAP_STREAMING */
   }
#ifdef VARIANT_S_FTR_ENABLE_MIC_ACTIVATION
   if (NULL != _hmiStateHandler)
   {
      _hmiStateHandler->removeHmiStateListener((this));
   }
   if (NULL != _resourceController.get())
   {
      _resourceController->removeSourceListUpdateListener((*this));
      _resourceController->removeSinkUpdateListener((*this));
   }
   if (NULL != _nanoMsgClientHandler)
   {
      _nanoMsgClientHandler->removeMic2StatusListener((*this));
      _nanoMsgClientHandler->removeMic3StatusListener((*this));
   }
#endif /* VARIANT_S_FTR_ENABLE_MIC_ACTIVATION */
}


void CabinConnectionHandler::onCabinsActivityRestorationInfo(const RestoreRegionsInfo& info)
{
#ifdef VARIANT_S_FTR_ENABLE_MAP_STREAMING
   restoreMapOut(info);
#endif /* VARIANT_S_FTR_ENABLE_MAP_STREAMING */
   updateVideoMixPropertyOnRestoreAbort(info);
}


void CabinConnectionHandler::updateVideoMixPropertyOnRestoreAbort(const RestoreRegionsInfo& info)
{
   if ((NULL != _hmiInfoServiceServerComponent) && (NULL != _regionHandling) && (NULL != _cabinConnectionUtility))
   {
      const CabinsVideoMixInfo& tInfo = _hmiInfoServiceServerComponent->getCabinsVideoMixInfo();
      CabinsVideoMixInfo cabinsVideoMixInfoUpdate;
      for (CabinsVideoMixInfo::const_iterator itr = tInfo.begin(); (itr != tInfo.end()); ++itr)
      {
         if ((*itr).getReason() == CABIN_VIDEO_MIX_CHANGE_REASON_RESTORE)
         {
            bool restoreAbort = true;
            SinkIdsInfo sinksInfo = _cabinConnectionUtility->getGlassSinksInfo((*itr).getCabinId(), SYSTEM_GROUP_CATEGORY_CMG_CABIN_MIX);
            if (!sinksInfo.empty())
            {
               for (RestoreRegionsInfo::const_iterator tItr = info.begin(); (tItr != info.end()); ++tItr)
               {
                  if ((*tItr).getRegionId() == (*itr).getCabinId())
                  {
                     restoreAbort = ((*tItr).getRestoreSinkInfo(sinksInfo[0], SINKGROUP_VIDEOMIX) == NULL) ? true : false; // INFO: Force checking because all CABIN_MIX sink has same group
                     break;
                  }
               }
               if (restoreAbort)
               {
                  uint8 reason = (_regionHandling->getStatus(SystemEnquiryInfo((*itr).getCabinId(), SYSTEM_CATEGORY_CMA, STATUS_CATEGORY_SYSTEM)) != SYSTEM_STATUS_CONNECTED) ? CABIN_VIDEO_MIX_CHANGE_REASON_CMA_ERROR : CABIN_VIDEO_MIX_CHANGE_REASON_NONE;
                  CabinVideoMixInfo cabinVideoMixInfo;
                  cabinVideoMixInfo.setCabinId((*itr).getCabinId());
                  cabinVideoMixInfo.setVideoMixIndex(CABIN_VIDEO_MIX_INDEX_NONE);
                  cabinVideoMixInfo.setReason(reason);
                  cabinsVideoMixInfoUpdate.push_back(cabinVideoMixInfo);
               }
            }
         }
      }
      if (!cabinsVideoMixInfoUpdate.empty())
      {
         updateCabinVideoMixInfoProperty(cabinsVideoMixInfoUpdate);
      }
   }
}


bool CabinConnectionHandler::onSourceRequest(const int16 srcId, const int16 devId, const int16 sinkId)
{
   ETG_TRACE_USR4(("CabinConnectionHandler: onSourceRequest: SourceId: %d, DeviceId: %d, SinkId: %d", srcId, devId, sinkId));
   bool retVal = false;
   evalAndResetRequestQueues();
   if (_connReqQueue.empty())
   {
      CabinConnectionHandlerInfo tInfo = getCabinConnectionHandlerInfo(stSourceInfo(srcId, devId, STATE_UNKNOWN, sinkId));
      for (CabinConnectionControllersInfo::const_iterator itr = _cabinConnectionControllers.begin(); (itr != _cabinConnectionControllers.end()); ++itr)
      {
         if ((NULL != (*itr)) && ((*itr)->onSourceConnect(tInfo)))
         {
            retVal = true;
            break;
         }
      }
      executeConnectionRequestsQueue();
   }
   else
   {
      ETG_TRACE_USR4(("CabinConnectionHandler: onSourceRequest: Request is on going so queued the new request"));
      removeSinkIdfromSourceQueue(sinkId);
      _srcReqQueue.push_back(stSourceInfo(srcId, devId, STATE_UNKNOWN, sinkId));
   }
   return retVal;
}


void CabinConnectionHandler::onSourceDisconnect(const int16 srcId, const int16 devId, const int16 sinkId)
{
   ETG_TRACE_USR4(("CabinConnectionController: onSourceDisconnect: SourceId: %d, DeviceId: %d, SinkId: %d", srcId, devId, sinkId));
   evalAndResetRequestQueues();
   if (_connReqQueue.empty())
   {
      CabinConnectionHandlerInfo tInfo = getCabinConnectionHandlerInfo(stSourceInfo(srcId, devId, STATE_DISCONNECT, sinkId));
      bool retVal = false;
      for (CabinConnectionControllersInfo::const_iterator itr = _cabinConnectionControllers.begin(); (itr != _cabinConnectionControllers.end()); ++itr)
      {
         if ((NULL != (*itr)) && ((*itr)->onSourceDisconnect(tInfo)))
         {
            retVal = true;
            break;
         }
      }
      executeConnectionRequestsQueue();
   }
   else
   {
      ETG_TRACE_USR4(("CabinConnectionHandler: onSourceDisconnect: Request is on going queued the new request"));
      removeSinkIdfromSourceQueue(sinkId);
      _srcReqQueue.push_back(stSourceInfo(srcId, devId, STATE_DISCONNECT, sinkId));
   }
}


void CabinConnectionHandler::updateCabinConnections(const SinkConnectionMap& newSinkInfos, const SinkConnectionMap& suspendedSinkInfos)
{
   ETG_TRACE_USR4(("CabinConnectionHandler: updateCabinConnections"));
   bool isCabinDataChanged                     = false;
   SinkConnectionMap::const_iterator newInfosIter    = newSinkInfos.begin();
   SinkConnectionMap::iterator activeInfosIter = _activeConnectionsInfo.begin();
   /*
    * INFO:
    *   => Checks for change in SrcInfos by comparing the newSinkInfos values with last updated values in _activeConnectionsInfo
    *   => Since map maintains data in sorted order, sink ids are compared to see if entries are changed/added/removed
    */
   for (; (newInfosIter != newSinkInfos.end()) && (activeInfosIter != _activeConnectionsInfo.end());)
   {
      if (newInfosIter->first == activeInfosIter->first)  //INFO: If sink ids are same, check if there is a change in SrcInfos
      {
         if ((newInfosIter->second.srcId != activeInfosIter->second.srcId)
               || (newInfosIter->second.deviceId != activeInfosIter->second.deviceId))
         {
            onConnectionRemoved(activeInfosIter->second);
            onNewConnectionUpdate(newInfosIter->second);
            isCabinDataChanged = true;
         }
         activeInfosIter->second = newInfosIter->second;
         newInfosIter++;
         activeInfosIter++;
      }
      else if (newInfosIter->first < activeInfosIter->first) //INFO: New sink added
      {
         _activeConnectionsInfo.insert(std::pair<uint16, stSourceInfo>(newInfosIter->first, newInfosIter->second));
         onNewConnectionUpdate(newInfosIter->second);
         isCabinDataChanged = true;
         newInfosIter++;
      }
      else  // INFO: Existing sink removed
      {
         SinkConnectionMap::iterator removeIter = activeInfosIter;
         activeInfosIter++;
         stSourceInfo removedSrcInfo = removeIter->second;
         _activeConnectionsInfo.erase(removeIter);
         onConnectionRemoved(removedSrcInfo);
         isCabinDataChanged = true;
      }
   }
   //INFO: New sinks added
   while (newInfosIter != newSinkInfos.end())
   {
      _activeConnectionsInfo.insert(std::pair<uint16, stSourceInfo>(newInfosIter->first, newInfosIter->second));
      onNewConnectionUpdate(newInfosIter->second);
      isCabinDataChanged = true;
      newInfosIter++;
   }
   //INFO: Existing sinks removed
   while (activeInfosIter != _activeConnectionsInfo.end())
   {
      SinkConnectionMap::iterator removeIter = activeInfosIter;
      activeInfosIter++;
      stSourceInfo removedSrcInfo = removeIter->second;
      _activeConnectionsInfo.erase(removeIter);
      onConnectionRemoved(removedSrcInfo);
      isCabinDataChanged = true;
   }
   if ((!suspendedSinkInfos.empty()) && (!_connReqQueue.empty()))
   {
      stSourceInfo tInfo = _connReqQueue.front();
      for (SinkConnectionMap::const_iterator itr = suspendedSinkInfos.begin(); itr != suspendedSinkInfos.end(); itr ++)
      {
         bool isSuspended  = (tInfo.sinkId == itr->second.sinkId) ? true : false;
         isSuspended       = ((isSuspended) && (tInfo.srcId == itr->second.srcId)) ? true : false;
         isSuspended       = ((isSuspended) && (tInfo.deviceId == itr->second.deviceId)) ? true : false;
         isSuspended       = ((isSuspended) && (tInfo.connectionState == STATE_CONNECTING)) ? true : false;
         if (isSuspended)
         {
            (void)_connReqQueue.erase(_connReqQueue.begin());
            isCabinDataChanged = true;
            break;
         }
      }
   }
   if (isCabinDataChanged)
   {
      updateCabinsActivityInfoDp();
      executeConnectionRequestsQueue();
      executeSourceRequestsQueue();
   }
   printSinkMapData(newSinkInfos, SOURCE_MAIN);
}


void CabinConnectionHandler::onNewConnectionUpdate(const stSourceInfo& info)
{
   ETG_TRACE_USR4(("CabinConnectionHandler: onNewConnectionUpdate: SourceId: %d, DeviceId: %d, SinkId: %d", info.srcId, info.deviceId, info.sinkId));
   ETG_TRACE_USR4(("CabinConnectionHandler: onNewConnectionUpdate: ConnectionQueue size: %d", _connReqQueue.size()));
   if (!_connReqQueue.empty())
   {
      const stSourceInfo& tInfo = _connReqQueue.front();
      bool isValidToRemove      = (STATE_CONNECTING == tInfo.connectionState) ? true : false;
      isValidToRemove           = ((isValidToRemove) && (info.srcId == tInfo.srcId)) ? true : false;
      isValidToRemove           = ((isValidToRemove) && (info.sinkId == tInfo.sinkId)) ? true : false;
      isValidToRemove           = ((isValidToRemove) && (info.deviceId == tInfo.deviceId)) ? true : false;
      if (isValidToRemove)
      {
         _connReqQueue.erase(_connReqQueue.begin());
         stopConnectionReqTimer(false);
      }
   }
   uint32 regionId = getRegionIdForSinkId(info.sinkId);
#ifdef VARIANT_S_FTR_ENABLE_MIC_ACTIVATION
   if (info.srcId == SRC_MIC1_ANNOUNCEMENT)
   {
      if (false == isValidToActivateMIC1(regionId))
      {
         _connReqQueue.push_back(stSourceInfo(info.srcId, info.deviceId, STATE_DISCONNECT, info.sinkId));
      }
      else
      {
         sendPluginMsgForMICBeep(info.sinkId, DEFAULT_BEEPOUTPUTVALUE);
         if (info.sinkId == CABIN_A_SINK_ID)
         {
            updateMIC1ConnectionStatus(true);
         }
      }
   }
#endif /* VARIANT_S_FTR_ENABLE_MIC_ACTIVATION */
   CabinConnectionHandlerInfo tCabinConnectionHandlerInfo;
   tCabinConnectionHandlerInfo.setSourceInfo(info);
   tCabinConnectionHandlerInfo.setSourceRequestsQueueInfo(_srcReqQueue);
   tCabinConnectionHandlerInfo.setActiveSinksInfo(_activeConnectionsInfo);
   tCabinConnectionHandlerInfo.setConnectionRequestsQueueInfo(_connReqQueue);
#ifdef VARIANT_S_FTR_ENABLE_MAP_STREAMING
   RegionMapOutInfo* imp  = getMutableRegionMapOutInfo(regionId);
   if (NULL != imp)
   {
      tCabinConnectionHandlerInfo.setRegionMapOutInfo((*imp));
   }
#endif /* VARIANT_S_FTR_ENABLE_MAP_STREAMING */
   for (CabinConnectionControllersInfo::const_iterator itr = _cabinConnectionControllers.begin(); (itr != _cabinConnectionControllers.end()); ++itr)
   {
      if (NULL != (*itr))
      {
         (*itr)->onNewConnectionUpdate(tCabinConnectionHandlerInfo);
      }
   }
#ifdef VARIANT_S_FTR_ENABLE_MAP_STREAMING
   if ((tCabinConnectionHandlerInfo.isModifiedSinksInfoValid()) && (NULL != _cabinConnectionUtility) && (NULL != imp))
   {
      _cabinConnectionUtility->updateSinksMapOutStatusAndStream(regionId, tCabinConnectionHandlerInfo.getModifiedSinksInfo(), false);
      //INFO: Map out streaming is disabled if map out is not active in any of the sinks in the region
      if ((!imp->isSinksMapOutInfoValid()) && (imp->getState() != STATE_DISCONNECTING))
      {
         imp->setState(STATE_DISCONNECTING);
         _cabinConnectionUtility->sendMapOutRequestToNavi(regionId, false);
      }
      sendMapOutUpdate(regionId, imp->getSinksMapOutInfo());
   }
#endif /* VARIANT_S_FTR_ENABLE_MAP_STREAMING */
}


void CabinConnectionHandler::onConnectionRemoved(const stSourceInfo& info)
{
   ETG_TRACE_USR4(("CabinConnectionHandler: onConnectionRemoved: SourceId: %d, DeviceId: %d, SinkId: %d", info.srcId, info.deviceId, info.sinkId));
   if (!_connReqQueue.empty())
   {
      const stSourceInfo& tInfo = _connReqQueue.front();
      bool isValidToRemove      = (STATE_DISCONNECTING == tInfo.connectionState) ? true : false;
      isValidToRemove           = ((isValidToRemove) && (info.srcId == tInfo.srcId)) ? true : false;
      isValidToRemove           = ((isValidToRemove) && (info.sinkId == tInfo.sinkId)) ? true : false;
      isValidToRemove           = ((isValidToRemove) && (info.deviceId == tInfo.deviceId)) ? true : false;
      if (isValidToRemove)
      {
         _connReqQueue.erase(_connReqQueue.begin());
      }
   }
   CabinConnectionHandlerInfo tCabinConnectionHandlerInfo;
   tCabinConnectionHandlerInfo.setSourceInfo(info);
   tCabinConnectionHandlerInfo.setSourceRequestsQueueInfo(_srcReqQueue);
   tCabinConnectionHandlerInfo.setActiveSinksInfo(_activeConnectionsInfo);
   tCabinConnectionHandlerInfo.setConnectionRequestsQueueInfo(_connReqQueue);
#ifdef VARIANT_S_FTR_ENABLE_MAP_STREAMING
   RegionMapOutInfo* imp  = getMutableRegionMapOutInfo(getRegionIdForSinkId(info.sinkId));
   if (NULL != imp)
   {
      tCabinConnectionHandlerInfo.setRegionMapOutInfo((*imp));
   }
#endif /* VARIANT_S_FTR_ENABLE_MAP_STREAMING */
   for (CabinConnectionControllersInfo::const_iterator itr = _cabinConnectionControllers.begin(); (itr != _cabinConnectionControllers.end()); ++itr)
   {
      if ((*itr) != NULL)
      {
         (*itr)->onConnectionRemoved(tCabinConnectionHandlerInfo);
      }
   }
}


void CabinConnectionHandler::executeConnectionRequestsQueue()
{
   if (NULL != _cabinConnectionUtility)
   {
      for (RequestsQueueInfo::iterator itr = _connReqQueue.begin(); (itr != _connReqQueue.end());)
      {
         if ((*itr).connectionState == STATE_UNKNOWN)
         {
#ifdef VARIANT_S_FTR_ENABLE_MIC_ACTIVATION
            if (((*itr).srcId == SRC_MIC1_ANNOUNCEMENT) && (NULL != _regionHandling))
            {
               setSysMuteStatusForMic1(true, _regionHandling->getRegionIdForAudioSink((*itr).sinkId));
            }
#endif
            ETG_TRACE_USR4(("CabinConnectionHandler: executeConnectionRequestsQueue: Connect: SrcId: %d, SinkId: %d", (*itr).srcId, (*itr).sinkId));
            bool status = _cabinConnectionUtility->sendSourceRequest((*itr).srcId, (*itr).deviceId, (*itr).sinkId);
            if (status)
            {
               (*itr).connectionState = STATE_CONNECTING;
               startConnectionReqTimer();
               break;
            }
            else
            {
               itr = _connReqQueue.erase(itr);
            }
         }
         else if ((*itr).connectionState == STATE_DISCONNECT)
         {
#ifdef VARIANT_S_FTR_ENABLE_MIC_ACTIVATION
            if (((*itr).srcId == SRC_MIC1_ANNOUNCEMENT) && (NULL != _regionHandling))
            {
               setSysMuteStatusForMic1(false, _regionHandling->getRegionIdForAudioSink((*itr).sinkId));
            }
#endif
            ETG_TRACE_USR4(("CabinConnectionHandler: executeConnectionRequestsQueue: Disconnect: SrcId: %d, SinkId: %d", (*itr).srcId, (*itr).sinkId));
            _cabinConnectionUtility->sendSourceDisconnect((*itr).srcId, (*itr).deviceId, (*itr).sinkId);
            if (_cabinConnectionUtility->isConnectionAvailable(getCabinConnectionHandlerInfo((*itr))))
            {
               (*itr).connectionState = STATE_DISCONNECTING;
               break;
            }
            else
            {
               itr = _connReqQueue.erase(itr);
            }
         }
         else
         {
            break;
         }
      }
   }
}


void CabinConnectionHandler::executeSourceRequestsQueue()
{
   ETG_TRACE_USR4(("CabinConnectionHandler: executeSourceRequestsQueue: ConnectionQueue size: %d SourceQueue size: %d", _connReqQueue.size(), _srcReqQueue.size()));
   if ((!_srcReqQueue.empty()) && (_connReqQueue.empty()))
   {
      ETG_TRACE_USR4(("CabinConnectionHandler: executeSourceRequestsQueue: Connection request queue is empty so execute source request queue"));
      for (RequestsQueueInfo::iterator itr = _srcReqQueue.begin(); (itr != _srcReqQueue.end());)
      {
         stSourceInfo tInfo((*itr));
         itr = _srcReqQueue.erase(itr);
         if (tInfo.connectionState == STATE_UNKNOWN)
         {
            onSourceRequest(tInfo.srcId, tInfo.deviceId, tInfo.sinkId);
         }
         else if ((*itr).connectionState == STATE_DISCONNECT)
         {
            onSourceDisconnect(tInfo.srcId, tInfo.deviceId, tInfo.sinkId);
         }
         else
         {}
         if (!_connReqQueue.empty())
         {
            break;
         }
      }
   }
}


void CabinConnectionHandler::clearRequestQueuesOnCMAError(const SystemStatusInfo& info)
{
   if ((info.getStatus() != SYSTEM_STATUS_CONNECTED) && (NULL != _cabinConnectionUtility))
   {
      SinkIdsInfo tInfo = _cabinConnectionUtility->getSinkIdsInfo(info.getRegionsIdInfo());
      for (SinkIdsInfo::const_iterator itr = tInfo.begin(); (itr != tInfo.end()); ++itr)
      {
         ETG_TRACE_USR4(("CabinConnectionHandler: clearRequestQueuesOnCMAError: SinkId: %d", (*itr)));
         if ((!_connReqQueue.empty()) && (_connReqQueue.front().sinkId == (*itr)) && (_connReqQueue.front().connectionState == STATE_CONNECTING))
         {
            stopConnectionReqTimer(false);
         }
         removeSinkIdfromSourceQueue((*itr));
         removeSinkIdfromConnectionQueue((*itr));
      }
   }
}


void CabinConnectionHandler::printSinkMapData(const SinkConnectionMap& sinkMap, uint8 sourceType)
{
   ETG_TRACE_USR4(("CabinConnectionHandler: printSinkMapData: --------------- SOURCE SINK INFOS START: sourceType: %d -----------------", sourceType));
   for (SinkConnectionMap::const_iterator iter = sinkMap.begin(); iter != sinkMap.end(); iter++)
   {
      ETG_TRACE_USR4(("CabinConnectionHandler: printSinkMapData: SinkID: %d, SrcID: %d, DevID: %d, ConnState: %d",
                      iter->first, iter->second.srcId, iter->second.deviceId, iter->second.connectionState));
   }
   ETG_TRACE_USR4(("CabinConnectionHandler: printSinkMapData: --------------- SOURCE SINK INFOS END: sourceType: %d -----------------", sourceType));
}


void CabinConnectionHandler::updateCabinConnectionsForMixSource(const SinkConnectionMap& newMixedSourceSinksInfo, const SinkConnectionMap& suspendedSinkInfos)
{
   ETG_TRACE_USR4(("CabinConnectionHandler: updateCabinConnectionsForMixSource"));
   checkAndUpdateNewMixSourceConnections(newMixedSourceSinksInfo);
   checkAndUpdateActiveMixSourceConnections(newMixedSourceSinksInfo);
   handleSuspendedMixSources(suspendedSinkInfos);
   executeMixSourceConnectionRequestsQueue();
   printSinkMapData(newMixedSourceSinksInfo, SOURCE_MIX);
}


void CabinConnectionHandler::handleSuspendedMixSources(const SinkConnectionMap& suspendedSinkInfos)
{
   if ((!suspendedSinkInfos.empty()) && (!_mixSrcConnReqQueue.empty()))
   {
      stSourceInfo tInfo = _mixSrcConnReqQueue.front();
      for (SinkConnectionMap::const_iterator itr = suspendedSinkInfos.begin(); itr != suspendedSinkInfos.end(); itr ++)
      {
         bool isSuspended   = (tInfo.sinkId == itr->second.sinkId) ? true : false;
         isSuspended        = ((isSuspended) && (tInfo.srcId == itr->second.srcId)) ? true : false;
         isSuspended        = ((isSuspended) && (tInfo.deviceId == itr->second.deviceId)) ? true : false;
         isSuspended        = ((isSuspended) && (tInfo.connectionState == STATE_CONNECTING)) ? true : false;
         if (isSuspended)
         {
            (void)_mixSrcConnReqQueue.erase(_mixSrcConnReqQueue.begin());
            break;
         }
      }
   }
}


void CabinConnectionHandler::executeMixSourceConnectionRequestsQueue()
{
   if (_cabinConnectionUtility != NULL)
   {
      for (RequestsQueueInfo::iterator itr = _mixSrcConnReqQueue.begin(); (itr != _mixSrcConnReqQueue.end());)
      {
         if ((*itr).connectionState == STATE_UNKNOWN)
         {
            ETG_TRACE_USR4(("CabinConnectionController: executeMixSourceConnectionRequestsQueue: Connect: SrcId: %d, SinkId: %d", (*itr).srcId, (*itr).sinkId));
            bool status = _cabinConnectionUtility->sendSourceRequest((*itr).srcId, (*itr).deviceId, (*itr).sinkId);
            if (status)
            {
               (*itr).connectionState = STATE_CONNECTING;
               startMixConnectionReqTimer();
               break;
            }
            else
            {
               itr = _mixSrcConnReqQueue.erase(itr);
            }
         }
         else if ((*itr).connectionState == STATE_DISCONNECT)
         {
            ETG_TRACE_USR4(("CabinConnectionController: executeMixSourceConnectionRequestsQueue: Disconnect: SrcId: %d, SinkId: %d", (*itr).srcId, (*itr).sinkId));
            _cabinConnectionUtility->sendSourceDisconnect((*itr).srcId, (*itr).deviceId, (*itr).sinkId);
            if (isMixSourceActive((*itr).srcId))
            {
               (*itr).connectionState = STATE_DISCONNECTING;
               break;
            }
            else
            {
               itr = _mixSrcConnReqQueue.erase(itr);
            }
         }
         else
         {
            break;
         }
      }
   }
}


void CabinConnectionHandler::sendPluginMsgForMIXSource(const int16 mixSourceIndex, const int16 sinkId)
{
   ETG_TRACE_USR4(("CabinConnectionHandler: sendPluginMsgForMIXSource: MixSourceIndex: %d, SinkId: %d", mixSourceIndex, sinkId));
   boost::shared_ptr<EventDataUtility> pluginData(EventDataUtility::newEventDataUtility());
   if (NULL != pluginData.get())
   {
      pluginData->addEventData(static_cast<uint8>(sinkId));
      pluginData->addEventData(static_cast<uint8>(mixSourceIndex));
      POST_MSG_NOTRACE((COURIER_MESSAGE_NEW(PluginCommandReqMsg)(PLUGIN_NAME_MASTER, CTRLBLOCK_NAME_MIX_SOURCE_CHANGE, pluginData)));
   }
}


void CabinConnectionHandler::onNewConnectionForMixSource(const stSourceInfo& info)
{
   ETG_TRACE_USR4(("CabinConnectionHandler: onNewConnectionForMixSource: SourceId: %d, SinkId: %d", info.srcId, info.sinkId));
   if (!_mixSrcConnReqQueue.empty())
   {
      const stSourceInfo& tInfo = _mixSrcConnReqQueue.front();
      bool isValidToRemove      = (STATE_CONNECTING == tInfo.connectionState) ? true : false;
      isValidToRemove           = ((isValidToRemove) && (info.srcId == tInfo.srcId)) ? true : false;
      isValidToRemove           = ((isValidToRemove) && (info.sinkId == tInfo.sinkId)) ? true : false;
      isValidToRemove           = ((isValidToRemove) && (info.deviceId == tInfo.deviceId)) ? true : false;
      if (isValidToRemove)
      {
         _mixSrcConnReqQueue.erase(_mixSrcConnReqQueue.begin());
         stopMixConnectionReqTimer(false);
      }
   }
   if (false == isValidToActivateMICMixSource(getRegionIdForSinkId(info.sinkId), info.srcId))
   {
      _mixSrcConnReqQueue.push_back(stSourceInfo(info.srcId, info.deviceId, STATE_DISCONNECT, info.sinkId));
   }
   else
   {
      int mixSourceIndex = CabinSrcInfo::getMixSrcIndex(info.srcId);
      updateMixSourceConnectionStatus(info.srcId, true, info.sinkId);
      sendPluginMsgForMIXSource(mixSourceIndex, info.sinkId);
#ifdef VARIANT_S_FTR_ENABLE_MIC_ACTIVATION
      sendPluginMsgForMICBeep(info.sinkId, DEFAULT_BEEPOUTPUTVALUE);
#endif
      POST_MSG_NOTRACE((COURIER_MESSAGE_NEW(MixSourceIndexConnectionStatusReqMsg)(mixSourceIndex, true)));
   }
}


void CabinConnectionHandler::onConnectionRemovedForMixSource(const stSourceInfo& info)
{
   ETG_TRACE_USR4(("CabinConnectionHandler: onConnectionRemovedForMixSource: SourceId: %d, SinkId: %d", info.srcId, info.sinkId));
   if (!_mixSrcConnReqQueue.empty())
   {
      const stSourceInfo& tInfo = _mixSrcConnReqQueue.front();
      bool isValidToRemove      = (STATE_DISCONNECTING == tInfo.connectionState) ? true : false;
      isValidToRemove           = ((isValidToRemove) && (info.srcId == tInfo.srcId)) ? true : false;
      isValidToRemove           = ((isValidToRemove) && (info.sinkId == tInfo.sinkId)) ? true : false;
      isValidToRemove           = ((isValidToRemove) && (info.deviceId == tInfo.deviceId)) ? true : false;
      if (isValidToRemove)
      {
         _mixSrcConnReqQueue.erase(_mixSrcConnReqQueue.begin());
      }
   }
   updateMixSourceConnectionStatus(info.srcId, false, info.sinkId);
   sendPluginMsgForMIXSource(CabinSrcInfo::NO_MixSource, info.sinkId);
}


void CabinConnectionHandler::checkAndUpdateNewMixSourceConnections(const SinkConnectionMap& newMixedSourceSinksInfo)
{
   for (SinkConnectionMap::const_iterator newMixSrcInfosIter = newMixedSourceSinksInfo.begin(); newMixSrcInfosIter != newMixedSourceSinksInfo.end(); newMixSrcInfosIter++)
   {
      SinkConnectionMap::iterator activeMixSrcInfosIter = _mixSourceActiveConnectionsInfo.find(newMixSrcInfosIter->second.sinkId);
      if (activeMixSrcInfosIter == _mixSourceActiveConnectionsInfo.end())
      {
         _mixSourceActiveConnectionsInfo.insert(std::pair<uint16, stSourceInfo>(newMixSrcInfosIter->first, newMixSrcInfosIter->second));
         onNewConnectionForMixSource(newMixSrcInfosIter->second);
      }
   }
}


void CabinConnectionHandler::clearMixSourceRequestQueuesOnCMAError(const SystemStatusInfo& info)
{
   if (info.getStatus() != SYSTEM_STATUS_CONNECTED)
   {
      SinkIdsInfo tInfo;
      if (NULL != _regionHandling)
      {
         for (RegionsIdInfo::const_iterator itr = info.getRegionsIdInfo().begin(); (itr != info.getRegionsIdInfo().end()); ++itr)
         {
            (void)_regionHandling->fetchSinkIdsInfo(tInfo, SYSTEM_CATEGORY_NONE, (*itr));
         }
      }
      for (SinkIdsInfo::const_iterator itr = tInfo.begin(); (itr != tInfo.end()); ++itr)
      {
         ETG_TRACE_USR4(("CabinConnectionHandler: clearMixSourceRequestQueuesOnCMAError: SinkId: %d", (*itr)));
         if ((!_mixSrcConnReqQueue.empty()) && (_mixSrcConnReqQueue.front().sinkId == (*itr)) && (_mixSrcConnReqQueue.front().connectionState == STATE_CONNECTING))
         {
            stopMixConnectionReqTimer(false);
         }

         for (RequestsQueueInfo::iterator tItr = _mixSrcConnReqQueue.begin(); (tItr != _mixSrcConnReqQueue.end());)
         {
            if ((*tItr).sinkId == (*itr))
            {
               tItr = _mixSrcConnReqQueue.erase(tItr);
            }
            else
            {
               ++tItr;
            }
         }
      }
   }
}


void CabinConnectionHandler::checkAndUpdateActiveMixSourceConnections(const SinkConnectionMap& newMixedSourceSinksInfo)
{
   for (SinkConnectionMap::iterator activeMixSrcInfosIter = _mixSourceActiveConnectionsInfo.begin(); (activeMixSrcInfosIter != _mixSourceActiveConnectionsInfo.end());)
   {
      SinkConnectionMap::const_iterator newMixSrcInfosIter = newMixedSourceSinksInfo.find(activeMixSrcInfosIter->second.sinkId);
      //INFO: If sink ids are same, check if there is a change in SrcInfos
      if (newMixSrcInfosIter != newMixedSourceSinksInfo.end())
      {
         if ((newMixSrcInfosIter->second.srcId != activeMixSrcInfosIter->second.srcId)
               || (newMixSrcInfosIter->second.deviceId != activeMixSrcInfosIter->second.deviceId)
               || (newMixSrcInfosIter->second.connectionState != activeMixSrcInfosIter->second.connectionState))
         {
            _mixSourceActiveConnectionsInfo[newMixSrcInfosIter->first] = newMixSrcInfosIter->second;
            if (STATE_CONNECTED == newMixSrcInfosIter->second.connectionState)
            {
               onNewConnectionForMixSource(newMixSrcInfosIter->second);
            }
         }
         ++activeMixSrcInfosIter;
      }
      else
      {
         SinkConnectionMap::iterator removeIter = activeMixSrcInfosIter;
         stSourceInfo removedSrcInfo = removeIter->second;
         ++activeMixSrcInfosIter;
         _mixSourceActiveConnectionsInfo.erase(removeIter);
         onConnectionRemovedForMixSource(removedSrcInfo);
      }
   }
}


void CabinConnectionHandler::updateMixSourceConnectionStatus(const int16 sourceId, const bool status, const int16 sinkId)
{
   ETG_TRACE_USR4(("CabinConnectionHandler: updateMixSourceConnectionStatus: SourceId: %d, Status: %d", sourceId, status));
   int mixSourceIndex = CabinSrcInfo::getMixSrcIndex(sourceId);
   for (std::vector<IMixSrcConnectionStatus*>::const_iterator itr = _mixSourceConnectionStatusListener.begin(); (itr != _mixSourceConnectionStatusListener.end()); ++itr)
   {
      if (NULL != (*itr))
      {
         (*itr)->currentMixSourceConnectionStatus(sourceId, status, sinkId, mixSourceIndex);
      }
   }
}


bool CabinConnectionHandler::isMixSourceActive(const int16 srcId)
{
   bool isActive = false;
   if (!_mixSourceActiveConnectionsInfo.empty())
   {
      for (SinkConnectionMap::iterator activeMixSrcInfosIter = _mixSourceActiveConnectionsInfo.begin(); (activeMixSrcInfosIter != _mixSourceActiveConnectionsInfo.end()); ++activeMixSrcInfosIter)
      {
         if ((activeMixSrcInfosIter->second.srcId == srcId) && (activeMixSrcInfosIter->second.connectionState == STATE_CONNECTED))
         {
            isActive = true;
            break;
         }
      }
   }
   ETG_TRACE_USR4(("CabinConnectionHandler: isMixSourceActive: SrcId: %d, IsActive: %d", srcId, isActive));
   return isActive;
}


bool CabinConnectionHandler::isMixSourceConnectionAvailable(const stSourceInfo info)
{
   bool isValid = false;
   if (!_mixSourceActiveConnectionsInfo.empty())
   {
      SinkConnectionMap::iterator itr = _mixSourceActiveConnectionsInfo.find(info.sinkId);
      if (itr != _mixSourceActiveConnectionsInfo.end())
      {
         isValid = (itr->second.srcId == info.srcId) ? true : false;
         isValid = ((isValid) && (itr->second.deviceId == info.deviceId)) ? true : false;
         isValid = ((isValid) && (itr->second.connectionState == STATE_CONNECTED)) ? true : false;
      }
   }
   ETG_TRACE_USR4(("CabinConnectionHandler: isMixSourceConnectionAvailable: isValid: %d", isValid));
   return isValid;
}


#ifdef VARIANT_S_FTR_ENABLE_MIC_ACTIVATION


void CabinConnectionHandler::onUpdateMic1Status(const bool status)
{
   _mic1Status = status;
   handleMic1Status(status, RegionsIdInfo());
}


void CabinConnectionHandler::onUpdateMic2Status(const bool status)
{
   handleMic2Status(status, RegionsIdInfo());
}


void CabinConnectionHandler::onUpdateMic3Status(const bool status)
{
   handleMic3Status(status, RegionsIdInfo());
}


void CabinConnectionHandler::onUpdateSinkChange(const uint32 sinkIndex)
{
   ETG_TRACE_USR4(("CabinConnectionHandler: onUpdateSinkChange: SinkIndex: %d", sinkIndex));
   switch (sinkIndex)
   {
      case CABIN_A_SINK_ID:
      case CABIN_B_SINK_ID:
      {
         updateMicSourceRestoration(RegionsIdInfo());
         break;
      }
      default:
         break;
   }
}


void CabinConnectionHandler::onUpdateSourceListChange(const uint32 srcIndex)
{
   ETG_TRACE_USR4(("CabinConnectionHandler: onUpdateSourceListChange: SrcIndex: %d", srcIndex));
   if ((srcIndex == SRC_MIC1_ANNOUNCEMENT) || (CabinSrcInfo::isMIXSource(srcIndex)))
   {
      updateMicSourceRestoration(RegionsIdInfo());
   }
}


void CabinConnectionHandler::updateMIC1ConnectionStatus(const bool status)
{
   ETG_TRACE_USR4(("CabinConnectionHandler: updateMIC1ConnectionStatus: Status: %d ", status));
   if (NULL != _nanoMsgClientHandler)
   {
      _nanoMsgClientHandler->setMic1AnnouncementRelay(static_cast<uint8>(status));
   }
   for (std::vector<IMIC1ConnectionStatus*>::const_iterator itr = _mic1ConnectionStatusListener.begin(); (itr != _mic1ConnectionStatusListener.end()); ++itr)
   {
      if (NULL != (*itr))
      {
         (*itr)->currentMIC1ConnectionStatus(status);
      }
   }
}


void CabinConnectionHandler::setSysMuteStatusForMic1(const bool status, const uint32 regionId)
{
   SoundControllerPrj* imp = AudioControllerObjectManager::getInstance().getSoundController<SoundControllerPrj>().get();
   if ((NULL != _regionHandling) && (NULL != imp))
   {
      uint8 cmaStatus = _regionHandling->getStatus(SystemEnquiryInfo(regionId, SYSTEM_CATEGORY_CMA, STATUS_CATEGORY_SYSTEM));
      switch (cmaStatus)
      {
         case SYSTEM_STATUS_PNM_DISCONNECTED:
         case SYSTEM_STATUS_HOURLOGIC_DISCONNECTED:
         {
            ETG_TRACE_USR4(("CabinConnectionHandler: setSysMuteStatusForMic1: RegionId: %d, Status: %d", regionId, status));
            imp->updateSetSystemMuteReq((!status), regionId);
            break;
         }
         default:
            break;
      }
   }
}


void CabinConnectionHandler::sendPluginMsgForMICBeep(const int16 sinkId, const uint8 beepOutput)
{
   ETG_TRACE_USR4(("CabinConnectionHandler: sendPluginMsgForMICBeep: SinkId: %d, BeepOutput: %d", sinkId, beepOutput));
   boost::shared_ptr<EventDataUtility> pluginData(EventDataUtility::newEventDataUtility());
   if (NULL != pluginData.get())
   {
      pluginData->addEventData(static_cast<uint8>(sinkId));
      pluginData->addEventData(static_cast<uint8>(beepOutput));
      POST_MSG_NOTRACE((COURIER_MESSAGE_NEW(PluginCommandReqMsg)(PLUGIN_NAME_MASTER, CTRBLOCK_NAME_BEEP_OUTPUT, pluginData)));
   }
}


void CabinConnectionHandler::updateMicSourceRestoration(const RegionsIdInfo& info)
{
   bool isMIC1SwitchActive = (getMic1Status()) ? true : false;
   if ((isMIC1SwitchActive) && (NULL != _hmiStateHandler) && (isLastHmiStateValidForMic1Restoration(_hmiStateHandler->getLastHmiState(), _hmiStateHandler->getPNMGrp10Status())))
   {
      RegionsIdInfo tRequiredRegionsInfo;
      SupportedRegionsInfo tSupportedRegionsInfo = _regionHandling->getSupportedRegionsInfo();
      for (SupportedRegionsInfo::const_iterator itr = tSupportedRegionsInfo.begin(); (itr != tSupportedRegionsInfo.end()); ++itr)
      {
         if ((_regionHandling->isCabinRegion((*itr).first)) && ((info.empty()) || (::std::find(info.begin(), info.end(), (*itr).first) != info.end())))
         {
            tRequiredRegionsInfo.push_back((*itr).first);
         }
      }
      handleMic1Status(true, tRequiredRegionsInfo);
   }

   bool isMIC2SwitchActive = ((NULL != _nanoMsgClientHandler) && (_nanoMsgClientHandler->getMic2Status())) ? true : false;
   if ((isMIC2SwitchActive) && (NULL != _hmiStateHandler) && (isLastHmiStateValidForMixedSourceRestoration(_hmiStateHandler->getLastHmiState())))
   {
      RegionsIdInfo tRequiredRegionsInfo;
      SupportedRegionsInfo tSupportedRegionsInfo = _regionHandling->getSupportedRegionsInfo();
      for (SupportedRegionsInfo::const_iterator itr = tSupportedRegionsInfo.begin(); (itr != tSupportedRegionsInfo.end()); ++itr)
      {
         if ((_regionHandling->isCabinRegion((*itr).first)) && ((info.empty()) || (::std::find(info.begin(), info.end(), (*itr).first) != info.end())))
         {
            tRequiredRegionsInfo.push_back((*itr).first);
         }
      }
      handleMic2Status(true, tRequiredRegionsInfo);
   }

   bool isMIC3SwitchActive = ((NULL != _nanoMsgClientHandler) && (_nanoMsgClientHandler->getMic3Status())) ? true : false;
   if ((isMIC3SwitchActive) && (NULL != _hmiStateHandler) && (isLastHmiStateValidForMixedSourceRestoration(_hmiStateHandler->getLastHmiState())))
   {
      RegionsIdInfo tRequiredRegionsInfo;
      SupportedRegionsInfo tSupportedRegionsInfo = _regionHandling->getSupportedRegionsInfo();
      for (SupportedRegionsInfo::const_iterator itr = tSupportedRegionsInfo.begin(); (itr != tSupportedRegionsInfo.end()); ++itr)
      {
         if ((_regionHandling->isCabinRegion((*itr).first)) && ((info.empty()) || (::std::find(info.begin(), info.end(), (*itr).first) != info.end())))
         {
            tRequiredRegionsInfo.push_back((*itr).first);
         }
      }
      handleMic3Status(true, tRequiredRegionsInfo);
   }
}


void CabinConnectionHandler::updateMicSourceDisconnection(const SystemStatusInfo& info)
{
   switch (info.getStatus())
   {
      case SYSTEM_STATUS_UNAVAILABLE:
      case SYSTEM_STATUS_COCKPIT_DISCONNECTED:
      case SYSTEM_STATUS_HMI_OFF_DISCONNECTED:
      case SYSTEM_STATUS_INACTIVE_DISCONNECTED:
      case SYSTEM_STATUS_USER_OFF_DISCONNECTED:
      case SYSTEM_STATUS_SWUPDATE_DISCONNECTED:
      case SYSTEM_STATUS_DIAGNOSIS_DISCONNECTED:
      {
         if (getMic1Status())
         {
            handleMic1Status(false, info.getRegionsIdInfo());
         }
         if (NULL != _nanoMsgClientHandler)
         {
            if (_nanoMsgClientHandler->getMic2Status())
            {
               handleMic2Status(false, info.getRegionsIdInfo());
            }
            if (_nanoMsgClientHandler->getMic3Status())
            {
               handleMic3Status(false, info.getRegionsIdInfo());
            }
         }
         break;
      }
      case SYSTEM_STATUS_PNM_DISCONNECTED:
      {
         if (NULL != _nanoMsgClientHandler)
         {
            if (_nanoMsgClientHandler->getMic2Status())
            {
               handleMic2Status(false, info.getRegionsIdInfo());
            }
            if (_nanoMsgClientHandler->getMic3Status())
            {
               handleMic3Status(false, info.getRegionsIdInfo());
            }
         }
         break;
      }
      default:
         break;
   }
}


void CabinConnectionHandler::handleMic1Status(const bool status, const RegionsIdInfo& info)
{
   ETG_TRACE_USR4(("CabinConnectionController: handleMic1Status: Status: %d", status));
   if (NULL != _regionHandling)
   {
      bool isConnectionQueueEmpty = (_connReqQueue.empty()) ? true : false;
      SupportedRegionsInfo tRequiredRegionsInfo;
      SupportedRegionsInfo tSupportedRegionsInfo = _regionHandling->getSupportedRegionsInfo();
      for (SupportedRegionsInfo::const_iterator itr = tSupportedRegionsInfo.begin(); (itr != tSupportedRegionsInfo.end()); ++itr)
      {
         if ((_regionHandling->isCabinRegion((*itr).first)) && ((info.empty()) || (::std::find(info.begin(), info.end(), (*itr).first) != info.end())))
         {
            tRequiredRegionsInfo.push_back((*itr));
         }
      }
      if (status)
      {
         for (SupportedRegionsInfo::const_iterator itr = tRequiredRegionsInfo.begin(); (itr != tRequiredRegionsInfo.end()); ++itr)
         {
            CabinConnectionHandlerInfo tInfo = getCabinConnectionHandlerInfo(stSourceInfo(SRC_MIC1_ANNOUNCEMENT, -1, STATE_UNKNOWN, _regionHandling->getAudioSinkIdForRegion((*itr).first)));
            if ((isValidToActivateMIC1((*itr).first)) && (NULL != _cabinConnectionUtility) && (!_cabinConnectionUtility->isConnectionAvailable(tInfo)))
            {
               _connReqQueue.push_back(stSourceInfo(SRC_MIC1_ANNOUNCEMENT, -1, STATE_UNKNOWN, _regionHandling->getAudioSinkIdForRegion((*itr).first)));
            }
         }
      }
      else
      {
         updateMIC1ConnectionStatus(false);
         for (SupportedRegionsInfo::const_iterator itr = tRequiredRegionsInfo.begin(); (itr != tRequiredRegionsInfo.end()); ++itr)
         {
            int16 audioSinkId = _regionHandling->getAudioSinkIdForRegion((*itr).first);
            if (NULL != _cabinConnectionUtility)
            {
               CabinConnectionHandlerInfo tInfo = getCabinConnectionHandlerInfo(stSourceInfo(SRC_MIC1_ANNOUNCEMENT, -1, STATE_DISCONNECT, audioSinkId));
               if (_cabinConnectionUtility->isConnectionAvailable(tInfo))
               {
                  _connReqQueue.push_back(stSourceInfo(SRC_MIC1_ANNOUNCEMENT, -1, STATE_DISCONNECT, audioSinkId));
               }
               else
               {
                  bool isMIC1disconnReqAvail = false;
                  for (RequestsQueueInfo::iterator tItr = _connReqQueue.begin(); (tItr != _connReqQueue.end());)
                  {
                     if (((*tItr).srcId == SRC_MIC1_ANNOUNCEMENT) && ((*tItr).sinkId == audioSinkId) && ((*tItr).connectionState == STATE_UNKNOWN))
                     {
                        tItr = _connReqQueue.erase(tItr);
                     }
                     else if (((*tItr).srcId == SRC_MIC1_ANNOUNCEMENT) && ((*tItr).sinkId == audioSinkId) && ((*tItr).connectionState == STATE_DISCONNECT))
                     {
                        isMIC1disconnReqAvail = true;
                        ++tItr;
                     }
                     else
                     {
                        ++tItr;
                     }
                  }
                  if (!_connReqQueue.empty())
                  {
                     const stSourceInfo& tInfo = _connReqQueue.front();
                     bool isVaildToPlaceMIC1DisconnReq   = ((tInfo.srcId == SRC_MIC1_ANNOUNCEMENT) && (tInfo.sinkId == audioSinkId));
                     isVaildToPlaceMIC1DisconnReq        = isVaildToPlaceMIC1DisconnReq && (tInfo.connectionState == STATE_CONNECTING);
                     isVaildToPlaceMIC1DisconnReq        = isVaildToPlaceMIC1DisconnReq && (isMIC1disconnReqAvail == false);
                     if (isVaildToPlaceMIC1DisconnReq)
                     {
                        _connReqQueue.push_back(stSourceInfo(SRC_MIC1_ANNOUNCEMENT, -1, STATE_DISCONNECT, audioSinkId));
                     }
                  }
               }
            }
         }
      }
      if (isConnectionQueueEmpty)
      {
         executeConnectionRequestsQueue();
      }
   }
}


void CabinConnectionHandler::handleMic2Status(const bool status, const RegionsIdInfo& info)
{
   ETG_TRACE_USR4(("CabinConnectionHandler: handleMic2Status: Status: %d", status));
   if (NULL != _regionHandling)
   {
      bool isConnectionQueueEmpty = (_mixSrcConnReqQueue.empty()) ? true : false;
      SupportedRegionsInfo tRequiredRegionsInfo;
      SupportedRegionsInfo tSupportedRegionsInfo = _regionHandling->getSupportedRegionsInfo();
      for (SupportedRegionsInfo::const_iterator itr = tSupportedRegionsInfo.begin(); (itr != tSupportedRegionsInfo.end()); ++itr)
      {
         if ((_regionHandling->isCabinRegion((*itr).first)) && ((info.empty()) || (::std::find(info.begin(), info.end(), (*itr).first) != info.end())))
         {
            if (_regionHandling->isSourceIdSupported(SRC_MIC2_ANNOUNCEMENT, SYSTEM_CATEGORY_CMA, (*itr).first))
            {
               tRequiredRegionsInfo.push_back((*itr));
            }
         }
      }
      if (status)
      {
         for (SupportedRegionsInfo::const_iterator itr = tRequiredRegionsInfo.begin(); (itr != tRequiredRegionsInfo.end()); ++itr)
         {
            stSourceInfo tInfo = stSourceInfo(SRC_MIC2_ANNOUNCEMENT, -1, STATE_UNKNOWN, _regionHandling->getAudioSinkIdForRegion((*itr).first));
            if ((isValidToActivateMICMixSource((*itr).first, SRC_MIC2_ANNOUNCEMENT)) && (!isMixSourceConnectionAvailable(tInfo)))
            {
               _mixSrcConnReqQueue.push_back(tInfo);
            }
         }
      }
      else
      {
         if (isMixSourceActive(SRC_MIC2_ANNOUNCEMENT))
         {
            POST_MSG_NOTRACE((COURIER_MESSAGE_NEW(MixSourceIndexConnectionStatusReqMsg)(MixSource_MIC2, false)));
         }
         for (SupportedRegionsInfo::const_iterator itr = tRequiredRegionsInfo.begin(); (itr != tRequiredRegionsInfo.end()); ++itr)
         {
            stSourceInfo tInfo = stSourceInfo(SRC_MIC2_ANNOUNCEMENT, -1, STATE_DISCONNECT, _regionHandling->getAudioSinkIdForRegion((*itr).first));
            if (isMixSourceConnectionAvailable(tInfo))
            {
               _mixSrcConnReqQueue.push_back(tInfo);
            }
            else
            {
               if (NULL != _cabinConnectionUtility)
               {
                  _cabinConnectionUtility->sendSourceDisconnect(SRC_MIC2_ANNOUNCEMENT, -1, _regionHandling->getAudioSinkIdForRegion((*itr).first));
               }
            }
         }
      }
      if (isConnectionQueueEmpty)
      {
         executeMixSourceConnectionRequestsQueue();
      }
   }
}


void CabinConnectionHandler::handleMic3Status(const bool status, const RegionsIdInfo& info)
{
   ETG_TRACE_USR4(("CabinConnectionHandler: handleMic3Status: Status: %d", status));
   if (NULL != _regionHandling)
   {
      bool isConnectionQueueEmpty = (_mixSrcConnReqQueue.empty()) ? true : false;
      SupportedRegionsInfo tRequiredRegionsInfo;
      SupportedRegionsInfo tSupportedRegionsInfo = _regionHandling->getSupportedRegionsInfo();
      for (SupportedRegionsInfo::const_iterator itr = tSupportedRegionsInfo.begin(); (itr != tSupportedRegionsInfo.end()); ++itr)
      {
         if ((_regionHandling->isCabinRegion((*itr).first)) && ((info.empty()) || (::std::find(info.begin(), info.end(), (*itr).first) != info.end())))
         {
            if (_regionHandling->isSourceIdSupported(SRC_MIC3_ANNOUNCEMENT, SYSTEM_CATEGORY_CMA, (*itr).first))
            {
               tRequiredRegionsInfo.push_back((*itr));
            }
         }
      }
      if (status)
      {
         for (SupportedRegionsInfo::const_iterator itr = tRequiredRegionsInfo.begin(); (itr != tRequiredRegionsInfo.end()); ++itr)
         {
            stSourceInfo info = stSourceInfo(SRC_MIC3_ANNOUNCEMENT, -1, STATE_UNKNOWN, _regionHandling->getAudioSinkIdForRegion((*itr).first));
            if ((isValidToActivateMICMixSource((*itr).first, SRC_MIC3_ANNOUNCEMENT)) && (!isMixSourceConnectionAvailable(info)))
            {
               _mixSrcConnReqQueue.push_back(info);
            }
         }
      }
      else
      {
         if (isMixSourceActive(SRC_MIC3_ANNOUNCEMENT))
         {
            POST_MSG_NOTRACE((COURIER_MESSAGE_NEW(MixSourceIndexConnectionStatusReqMsg)(MixSource_MIC3, false)));
         }
         for (SupportedRegionsInfo::const_iterator itr = tRequiredRegionsInfo.begin(); (itr != tRequiredRegionsInfo.end()); ++itr)
         {
            stSourceInfo info = stSourceInfo(SRC_MIC3_ANNOUNCEMENT, -1, STATE_DISCONNECT, _regionHandling->getAudioSinkIdForRegion((*itr).first));
            if (isMixSourceConnectionAvailable(info))
            {
               _mixSrcConnReqQueue.push_back(info);
            }
            else
            {
               if (NULL != _cabinConnectionUtility)
               {
                  _cabinConnectionUtility->sendSourceDisconnect(SRC_MIC3_ANNOUNCEMENT, -1, _regionHandling->getAudioSinkIdForRegion((*itr).first));
               }
            }
         }
      }
      if (isConnectionQueueEmpty)
      {
         executeMixSourceConnectionRequestsQueue();
      }
   }
}


bool CabinConnectionHandler::isValidToActivateMIC1(const uint32 regionId)
{
   bool isValid = false;
   if (NULL != _regionHandling)
   {
      uint8 systemStatus = _regionHandling->getStatus(SystemEnquiryInfo(regionId, SYSTEM_CATEGORY_CMA, STATUS_CATEGORY_SYSTEM));
      switch (systemStatus)
      {
         case SYSTEM_STATUS_CONNECTED:
         case SYSTEM_STATUS_PNM_DISCONNECTED:
         case SYSTEM_STATUS_HOURLOGIC_DISCONNECTED:
         {
            if (NULL != _resourceController)
            {
               isValid = (_resourceController->isSourceAvailable(SRC_MIC1_ANNOUNCEMENT)) ? true : false;
               isValid = ((isValid) && (_resourceController->isSinkAdded(_regionHandling->getAudioSinkIdForRegion(regionId)))) ? true : false;
            }
            break;
         }
         default:
            break;
      }
   }
   ETG_TRACE_USR4(("CabinConnectionHandler isValidToActivateMIC1: RegionId: %d, IsValid: %d", regionId, isValid));
   return isValid;
}


bool CabinConnectionHandler::isValidToActivateMICMixSource(const uint32 regionId, const uint32 sourceId)
{
   bool isValid = false;
   if (NULL != _regionHandling)
   {
      uint8 systemStatus = _regionHandling->getStatus(SystemEnquiryInfo(regionId, SYSTEM_CATEGORY_CMA, STATUS_CATEGORY_SYSTEM));
      switch (systemStatus)
      {
         case SYSTEM_STATUS_CONNECTED:
         {
            if (NULL != _resourceController)
            {
               isValid = (_resourceController->isSourceAvailable(sourceId)) ? true : false;
               isValid = ((isValid) && (_resourceController->isSinkAdded(_regionHandling->getAudioSinkIdForRegion(regionId)))) ? true : false;
            }
            break;
         }
         default:
            break;
      }
   }
   ETG_TRACE_USR4(("CabinConnectionHandler: isValidToActivateMICMixSource: RegionId: %d, SourceId: %d, IsValid: %d", regionId, sourceId, isValid));
   return isValid;
}


bool CabinConnectionHandler::getMic1Status()
{
   return _mic1Status;
}


#endif /* VARIANT_S_FTR_ENABLE_MIC_ACTIVATION */


#ifdef VARIANT_S_FTR_ENABLE_MAP_STREAMING

void CabinConnectionHandler::onToggleMapOutStatus(const uint32 regionId)
{
   if (fnGetVariantType() ==  VARIANT_TYPE_NAVI)
   {
      SinkIdsInfo tSupportedSinksInfo;
      if ((NULL != _hmiInfoServiceServerComponent) && (NULL != _cabinConnectionUtility))
      {
         uint8 groupCategory = (_hmiInfoServiceServerComponent->getCabinVideoMixIndex(regionId) == CABIN_VIDEO_MIX_INDEX_NONE) ? \
                               SYSTEM_GROUP_CATEGORY_NONE : SYSTEM_GROUP_CATEGORY_CMG_CABIN;
         tSupportedSinksInfo = _cabinConnectionUtility->getGlassSinksInfo(regionId, groupCategory);
      }
      if (!tSupportedSinksInfo.empty() && (fnGetVariantType() ==  VARIANT_TYPE_NAVI))
      {
         const RegionMapOutInfo* imp = getRegionMapOutInfo(regionId);
         if (NULL == imp)
         {
            activateMapOut(regionId, tSupportedSinksInfo);
         }
         else
         {
            const SinkMapOutInfo* tSinkMapOutInfo = imp->getSinkMapOutInfo(tSupportedSinksInfo[0]);
            if (NULL != tSinkMapOutInfo)
            {
               if ((tSinkMapOutInfo->getState() == STATE_CONNECTED) || (tSinkMapOutInfo->getState() == STATE_CONNECTING))
               {
                  deactivateMapOut(regionId, tSupportedSinksInfo);
               }
               else
               {
                  activateMapOut(regionId, tSupportedSinksInfo);
               }
            }
            else
            {
               activateMapOut(regionId, tSupportedSinksInfo);
            }
         }
      }
   }
   else
   {
      ETG_TRACE_ERR(("CabinConnectionHandler: onToggleMapOutStatus: Feature NA in NonNavi Variant"));
   }
}


void CabinConnectionHandler::onMapOutStatusUpdMsg(const uint32 regionId, const bool status)
{
   RegionMapOutInfo* imp = getMutableRegionMapOutInfo(regionId);
   if (NULL != imp)
   {
      uint8 state = imp->getState();
      ETG_TRACE_USR4(("CabinConnectionHandler: onMapOutStatusUpdMsg: RegionId: %d, Status: %d, State: %d", regionId, status, state));
      if (state == STATE_CONNECTING)
      {
         if (status)
         {
            imp->setState(STATE_CONNECTED);
            onMapOutActivation(imp);
         }
      }
      else
      {
         removeFromRegionsMapOutInfo(regionId);
      }
   }
}


void CabinConnectionHandler::onCMGStatusUpdate(const SystemStatusInfo& info)
{
   if ((info.getStatus() != SYSTEM_STATUS_CONNECTED) && (!info.getRegionsIdInfo().empty()))
   {
      uint32 regionId = info.getRegionsIdInfo()[0]; //INFO: Force reading because CMG is mounted to a region / sink
      SinkIdsInfo tSinkIdsInfo;
      if ((NULL != _regionHandling) && (_regionHandling->fetchSinkIdsInfoForGroup(tSinkIdsInfo, info.getSystemGroup(), SYSTEM_CATEGORY_CMG, regionId)) && (!tSinkIdsInfo.empty()))
      {
         int16 sinkId = tSinkIdsInfo[0]; //INFO: Force reading because CMG is mounted to a region / sink
         SystemEnquiryInfo tSystemEnquiryInfo(regionId, SYSTEM_CATEGORY_CMG, STATUS_CATEGORY_SYSTEM);
         if (!_regionHandling->isAnyStatusSame(tSystemEnquiryInfo, SYSTEM_STATUS_CONNECTED, sinkId))
         {
            deactivateMapOut(regionId, tSinkIdsInfo);
         }
      }
   }
}


void CabinConnectionHandler::clearMapOutInfoOnCMAError(const SystemStatusInfo& info)
{
   if ((info.getStatus() != SYSTEM_STATUS_CONNECTED) && (fnGetVariantType() ==  VARIANT_TYPE_NAVI))
   {
      for (RegionsIdInfo::const_iterator itr = info.getRegionsIdInfo().begin(); (itr != info.getRegionsIdInfo().end()); ++itr)
      {
         deactivateMapOut((*itr), SinkIdsInfo());
         removeFromRegionsMapOutInfo((*itr)); //INFO: Clear the entry forcefully due to CMA error
      }
   }
}


void CabinConnectionHandler::activateMapOut(const uint32 regionId, const SinkIdsInfo& info)
{
   if (fnGetVariantType() ==  VARIANT_TYPE_NAVI)
   {
      ETG_TRACE_USR4(("CabinConnectionHandler: activateMapOut: RegionId: %d", regionId));
      if ((NULL != _regionHandling) && (_regionHandling->isAllStatusSame(SystemEnquiryInfo(regionId, SYSTEM_CATEGORY_CMA, STATUS_CATEGORY_SYSTEM), SYSTEM_STATUS_CONNECTED)))
      {
         SinkIdsInfo tValidSinksInfo;
         for (SinkIdsInfo::const_iterator itr = info.begin(); (itr != info.end()); ++itr)
         {
            SystemEnquiryInfo tSystemEnquiryInfo(regionId, SYSTEM_CATEGORY_CMG, STATUS_CATEGORY_SYSTEM);
            bool isValid = (::std::find(tValidSinksInfo.begin(), tValidSinksInfo.end(), (*itr)) == tValidSinksInfo.end()) ? true : false;
            isValid      = ((isValid) && (_regionHandling->isAnyStatusSame(tSystemEnquiryInfo, SYSTEM_STATUS_CONNECTED, (*itr)))) ? true : false;
            if (isValid)
            {
               tValidSinksInfo.push_back((*itr));
               ETG_TRACE_USR4(("CabinConnectionHandler: activateMapOut: Valid Sink: %d", (*itr)));
            }
         }
         if (!tValidSinksInfo.empty())
         {
            RegionMapOutInfo* imp = getMutableRegionMapOutInfo(regionId);
            if (NULL == imp)
            {
               addToRegionsMapOutInfo(regionId);
               imp = getMutableRegionMapOutInfo(regionId);
            }
            if (NULL != imp)
            {
               ETG_TRACE_USR4(("CabinConnectionHandler: activateMapOut: Region State: %d", imp->getState()));
               switch (imp->getState())
               {
                  case STATE_UNKNOWN:
                  case STATE_DISCONNECTING:
                  {
                     imp->setState(STATE_CONNECTING);
                     sendMapOutRequestToNavi(regionId, true);
                     for (SinkIdsInfo::const_iterator itr = tValidSinksInfo.begin(); (itr != tValidSinksInfo.end()); ++itr)
                     {
                        imp->addToSinksMapOutInfo((*itr));
                        imp->updateSinkState(STATE_CONNECTING, (*itr));
                        ETG_TRACE_USR4(("CabinConnectionHandler: activateMapOut: Added Sink: %d", (*itr)));
                     }
                     sendMapOutUpdate(regionId, imp->getSinksMapOutInfo()); //INFO: To communicate state "CONNECTING"
                     break;
                  }
                  case STATE_CONNECTED:
                  case STATE_CONNECTING:
                  {
                     bool isModified = false;
                     for (SinkIdsInfo::const_iterator itr = tValidSinksInfo.begin(); (itr != tValidSinksInfo.end()); ++itr)
                     {
                        SinkMapOutInfo* tInfo = imp->getMutableSinkMapOutInfo((*itr));
                        if (NULL == tInfo)
                        {
                           imp->addToSinksMapOutInfo((*itr));
                           tInfo = imp->getMutableSinkMapOutInfo((*itr));
                        }
                        if ((tInfo->getState() != STATE_CONNECTED) && (tInfo->getState() != STATE_CONNECTING))
                        {
                           isModified = true;
                           tInfo->setState(STATE_CONNECTING);
                           ETG_TRACE_USR4(("CabinConnectionHandler: activateMapOut: Modified Sink: %d", (*itr)));
                        }
                     }
                     if (isModified)
                     {
                        sendMapOutUpdate(regionId, imp->getSinksMapOutInfo()); //INFO: To communicate state "CONNECTING"
                        if (imp->getState() == STATE_CONNECTED)
                        {
                           onMapOutActivation(imp);
                        }
                     }
                     break;
                  }
                  default:
                     break;
               }
            }
         }
      }
      else
      {
         ETG_TRACE_USR4(("CabinConnectionHandler: activateMapOut: CMA is not connected"));
      }
   }
   else
   {
      ETG_TRACE_ERR(("CabinConnectionHandler: activateMapOut: Feature NA in NonNavi Variant"));
   }
}


void CabinConnectionHandler::deactivateMapOut(const uint32 regionId, const SinkIdsInfo& info)
{
   if (fnGetVariantType() ==  VARIANT_TYPE_NAVI)
   {
      ETG_TRACE_USR4(("CabinConnectionHandler: deactivateMapOut: RegionId: %d", regionId));
      RegionMapOutInfo* imp = getMutableRegionMapOutInfo(regionId);
      if (NULL != imp)
      {
         ETG_TRACE_USR4(("CabinConnectionHandler: deactivateMapOut: Region State: %d", imp->getState()));
         if (imp->getState() != STATE_DISCONNECTING)
         {
            bool isModified = false;
            if (info.empty())
            {
               SinksMapOutInfo* tInfo = &(imp->getMutableSinksMapOutInfo());
               if (NULL != tInfo)
               {
                  for (SinksMapOutInfo::iterator itr = tInfo->begin(); (itr != tInfo->end()); ++itr)
                  {
                     if ((*itr).getState() != STATE_DISCONNECTING)
                     {
                        (*itr).setState(STATE_DISCONNECTING);
                        isModified = true;
                        ETG_TRACE_USR4(("CabinConnectionHandler: deactivateMapOut: Modified Sink: %d", (*itr).getSinkId()));
                     }
                  }
               }
            }
            else
            {
               for (SinkIdsInfo::const_iterator itr = info.begin(); (itr != info.end()); ++itr)
               {
                  SinkMapOutInfo* tInfo = imp->getMutableSinkMapOutInfo((*itr));
                  if ((tInfo != NULL) && (tInfo->getState() != STATE_DISCONNECTING))
                  {
                     isModified = true;
                     tInfo->setState(STATE_DISCONNECTING);
                     ETG_TRACE_USR4(("CabinConnectionHandler: deactivateMapOut: Modified Sink: %d", (*itr)));
                  }
               }
            }
            if (isModified)
            {
               onMapOutDeactivation(imp);
               if (imp->getState() == STATE_UNKNOWN)
               {
                  removeFromRegionsMapOutInfo(regionId);
               }
               else if (!imp->isSinksMapOutInfoValid())
               {
                  imp->setState(STATE_DISCONNECTING);
                  sendMapOutRequestToNavi(regionId, false);
               }
               else
               {}
            }
         }
      }
   }
   else
   {
      ETG_TRACE_ERR(("CabinConnectionHandler: deactivateMapOut: Feature NA in NonNavi Variant"));
   }
}


void CabinConnectionHandler::restoreMapOut(const RestoreRegionsInfo& info)
{
   if (fnGetVariantType() ==  VARIANT_TYPE_NAVI)
   {
      ::std::map< uint32, SinkIdsInfo > tAbortRestoreSinksInfo;
      for (RegionsMapOutInfo::const_iterator itr = _regionsMapOutInfo.begin(); (itr != _regionsMapOutInfo.end()); ++itr)
      {
         SinkIdsInfo tRestoringSinks;
         for (SinksMapOutInfo::const_iterator sItr = (*itr).getSinksMapOutInfo().begin(); (sItr != (*itr).getSinksMapOutInfo().end()); ++sItr)
         {
            if ((*sItr).getState() == STATE_RESTORE)
            {
               tRestoringSinks.push_back((*sItr).getSinkId());
            }
         }
         if (!tRestoringSinks.empty())
         {
            tAbortRestoreSinksInfo.insert(::std::pair< uint32, SinkIdsInfo >((*itr).getRegionId(), tRestoringSinks));
         }
      }
      for (RestoreRegionsInfo::const_iterator itr = info.begin(); (itr != info.end()); ++itr)
      {
         for (ActivitiesRestoreSinksInfo::const_iterator sItr = (*itr).getActivitiesRestoreSinksInfo().begin(); sItr != (*itr).getActivitiesRestoreSinksInfo().end(); ++sItr)
         {
            if (CABIN_ACTIVITY_MAPOUT == (*sItr).getActivityType())
            {
               SinkIdsInfo* tAbortSinksInfo = NULL;
               ::std::map< uint32, SinkIdsInfo >::iterator tRegionItr = tAbortRestoreSinksInfo.find((*itr).getRegionId());
               if (tRegionItr != tAbortRestoreSinksInfo.end())
               {
                  tAbortSinksInfo = (&(tRegionItr->second));
               }
               RegionMapOutInfo* imp = getMutableRegionMapOutInfo((*itr).getRegionId());
               if (NULL == imp)
               {
                  ETG_TRACE_USR4(("CabinConnectionHandler: restoreMapOut: Added Region: %d", (*itr).getRegionId()));
                  addToRegionsMapOutInfo((*itr).getRegionId());
                  imp = getMutableRegionMapOutInfo((*itr).getRegionId());
               }
               if (NULL != imp)
               {
                  SinkIdsInfo tRequiredSinks;
                  for (RestoreSinksInfoGroup::const_iterator groupItr = (*sItr).getRestoreSinksInfoGroup().begin(); (groupItr != (*sItr).getRestoreSinksInfoGroup().end()); ++groupItr)
                  {
                     for (RestoreSinksInfo::const_iterator tItr = (*groupItr).second.begin(); (tItr != (*groupItr).second.end()); ++tItr)
                     {
                        bool isValidToRestore = false;
                        switch ((*tItr).getRestoreStatus())
                        {
                           case CABIN_RESTORE_STATUS_EVALUATING:
                           {
                              isValidToRestore = true;
                              imp->addToSinksMapOutInfo((*tItr).getSinkId());
                              imp->updateSinkState(STATE_RESTORE, (*tItr).getSinkId());
                              sendMapOutUpdate((*itr).getRegionId(), imp->getSinksMapOutInfo()); //INFO: To communicate state "RESTORE"
                              ETG_TRACE_USR4(("CabinConnectionHandler: restoreMapOut: Added Sink: %d", (*tItr).getSinkId()));
                              break;
                           }
                           case CABIN_RESTORE_STATUS_RESTORE:
                           {
                              isValidToRestore = true;
                              tRequiredSinks.push_back((*tItr).getSinkId());
                              break;
                           }
                           case CABIN_RESTORE_STATUS_PAUSE_MIC1:
                           {
                              isValidToRestore = true;
                           }
                           default:
                              break;
                        }
                        if ((isValidToRestore) && (tAbortSinksInfo != NULL))
                        {
                           SinkIdsInfo::iterator tSinksItr = ::std::find(tAbortSinksInfo->begin(), tAbortSinksInfo->end(), (*tItr).getSinkId());
                           if (tSinksItr != tAbortSinksInfo->end())
                           {
                              tAbortSinksInfo->erase(tSinksItr);
                           }
                        }
                     }
                  }
                  if (!tRequiredSinks.empty())
                  {
                     activateMapOut((*itr).getRegionId(), tRequiredSinks);
                  }
               }
            }
         }
      }
      for (::std::map< uint32, SinkIdsInfo >::const_iterator itr = tAbortRestoreSinksInfo.begin(); (itr != tAbortRestoreSinksInfo.end()); ++itr)
      {
         if (!((itr->second).empty()))
         {
            deactivateMapOut(itr->first, itr->second);
         }
      }
   }
   else
   {
      ETG_TRACE_ERR(("CabinConnectionHandler: restoreMapOut: Feature NA in NonNavi Variant"));
   }
}


void CabinConnectionHandler::onMapOutActivation(RegionMapOutInfo* imp)
{
   if (NULL != imp)
   {
      ETG_TRACE_USR4(("CabinConnectionHandler: onMapOutActivation: RegionId: %d", imp->getRegionId()));
      bool isConnectionQueueEmpty = (_connReqQueue.empty()) ? true : false;
      CabinConnectionHandlerInfo tInfo;
      tInfo.setRegionMapOutInfo((*imp));
      tInfo.setSourceRequestsQueueInfo(_srcReqQueue);
      tInfo.setActiveSinksInfo(_activeConnectionsInfo);
      tInfo.setConnectionRequestsQueueInfo(_connReqQueue);
      for (CabinConnectionControllersInfo::iterator itr = _cabinConnectionControllers.begin(); (itr != _cabinConnectionControllers.end()); ++itr)
      {
         if ((*itr) != NULL)
         {
            (*itr)->onMapOutActivation(tInfo);
         }
      }
      if (tInfo.isModifiedSinksInfoValid())
      {
         sendMapOutUpdate(imp->getRegionId(), imp->getSinksMapOutInfo());
         if (NULL != _cabinConnectionUtility)
         {
            _cabinConnectionUtility->updateSinksMapOutStatusAndStream(imp->getRegionId(), tInfo.getModifiedSinksInfo(), true);
         }
         if (isConnectionQueueEmpty)
         {
            executeConnectionRequestsQueue();
         }
         if (NULL != _cabinRestorationHandler)
         {
            _cabinRestorationHandler->onUserInteraction(imp->getRegionId(), tInfo.getModifiedSinksInfo());
         }
         updateCabinsActivityInfoDp();
      }
   }
}


void CabinConnectionHandler::onMapOutDeactivation(RegionMapOutInfo* imp)
{
   if (NULL != imp)
   {
      ETG_TRACE_USR4(("CabinConnectionHandler: onMapOutDeactivation: RegionId: %d", imp->getRegionId()));
      bool isConnectionQueueEmpty = (_connReqQueue.empty()) ? true : false;
      CabinConnectionHandlerInfo tInfo;
      tInfo.setRegionMapOutInfo((*imp));
      tInfo.setSourceRequestsQueueInfo(_srcReqQueue);
      tInfo.setActiveSinksInfo(_activeConnectionsInfo);
      tInfo.setConnectionRequestsQueueInfo(_connReqQueue);
      for (CabinConnectionControllersInfo::iterator itr = _cabinConnectionControllers.begin(); (itr != _cabinConnectionControllers.end()); ++itr)
      {
         if ((*itr) != NULL)
         {
            (*itr)->onMapOutDeactivation(tInfo);
         }
      }
      if (tInfo.isModifiedSinksInfoValid())
      {
         sendMapOutUpdate(imp->getRegionId(), imp->getSinksMapOutInfo());
         if (NULL != _cabinConnectionUtility)
         {
            _cabinConnectionUtility->updateSinksMapOutStatusAndStream(imp->getRegionId(), tInfo.getModifiedSinksInfo(), false);
         }
         if (isConnectionQueueEmpty)
         {
            executeConnectionRequestsQueue();
         }
         updateCabinsActivityInfoDp();
      }
   }
}


void CabinConnectionHandler::sendMapOutRequestToNavi(const uint32 regionId, const bool status)
{
   if (NULL != _cabinConnectionUtility)
   {
      _cabinConnectionUtility->sendMapOutRequestToNavi(regionId, status);
   }
}


void CabinConnectionHandler::sendMapOutUpdate(const uint32 regionId, const SinksMapOutInfo& info)
{
   for (::std::vector< IMapOutStatusListenerCB* >::const_iterator itr = _mapOutStatusListeners.begin(); (itr != _mapOutStatusListeners.end()); ++itr)
   {
      if (NULL != (*itr))
      {
         (*itr)->onMapOutStatusUpdate(regionId, info);
      }
   }
}


#endif /* VARIANT_S_FTR_ENABLE_MAP_STREAMING */


void CabinConnectionHandler::onCMAStatusUpdate(const SystemStatusInfo& info)
{
   ETG_TRACE_USR4(("CabinConnectionHandler: onCMAStatusUpdate: Status: %d, SystemId: %d", info.getStatus(), info.getSystemId()));
   for (CabinConnectionControllersInfo::const_iterator itr = _cabinConnectionControllers.begin(); (itr != _cabinConnectionControllers.end()); ++itr)
   {
      if (NULL != (*itr))
      {
         (*itr)->onCMAStatusUpdate(info);
      }
   }
   clearRequestQueuesOnCMAError(info);
   clearMixSourceRequestQueuesOnCMAError(info);
#ifdef VARIANT_S_FTR_ENABLE_MAP_STREAMING
   clearMapOutInfoOnCMAError(info);
#endif
#ifdef VARIANT_S_FTR_ENABLE_MIC_ACTIVATION
   updateMicSourceDisconnection(info);
   updateMicSourceRestoration(info.getRegionsIdInfo());
#endif
   updateCabinVideoMixInfoForCMAStatus(info);
}


void CabinConnectionHandler::updateCabinVideoMixInfoForCMAStatus(const SystemStatusInfo& info)
{
   if ((info.getStatus() != SYSTEM_STATUS_CONNECTED) && (NULL != _regionHandling) && (NULL != _hmiInfoServiceServerComponent) && (NULL != _cabinConnectionUtility))
   {
      const RegionsIdInfo& tRegionsIdInfo = info.getRegionsIdInfo();
      CabinsVideoMixInfo cabinsVideoMixInfoUpdate;
      for (RegionsIdInfo::const_iterator itr = tRegionsIdInfo.begin(); (itr != tRegionsIdInfo.end()); ++itr)
      {
         SystemsGroupInfo systemGroupInfo;
         if ((_regionHandling->fetchSystemsGroupInfo(systemGroupInfo, SYSTEM_CATEGORY_CMG, (*itr))) && (systemGroupInfo.size() > CABIN_MIN_SYSTEM_GROUP))
         {
            SinkIdsInfo sinksInfo = _cabinConnectionUtility->getGlassSinksInfo((*itr), SYSTEM_GROUP_CATEGORY_CMG_CABIN_MIX);
            const CabinVideoMixInfo* tInfo = _hmiInfoServiceServerComponent->getVideoMixInfoForRegion((*itr));
            if (((NULL != tInfo) && ((tInfo->getReason() != CABIN_VIDEO_MIX_CHANGE_REASON_RESTORE))) ||
                  (_cabinRestorationHandler->getSinkRestoreStatus(sinksInfo[0], (*itr), SINKGROUP_VIDEOMIX) == CABIN_RESTORE_STATUS_RESTORE))
            {
               CabinVideoMixInfo cabinVideoMixInfo;
               cabinVideoMixInfo.setCabinId(*itr);
               cabinVideoMixInfo.setVideoMixIndex(CABIN_VIDEO_MIX_INDEX_NONE);
               cabinVideoMixInfo.setReason(CABIN_VIDEO_MIX_CHANGE_REASON_CMA_ERROR);
               cabinsVideoMixInfoUpdate.push_back(cabinVideoMixInfo);
            }
         }
      }
      if (!cabinsVideoMixInfoUpdate.empty())
      {
         updateCabinVideoMixInfoProperty(cabinsVideoMixInfoUpdate);
      }
   }
}


void CabinConnectionHandler::updateCabinsActivityInfoDp()
{
   ETG_TRACE_USR4(("CabinConnectionHandler: updateCabinsActivityInfoDp"));
   bool isValid = (NULL != _cabinsActivityInfoDpHandler) ? true : false;
   isValid      = ((isValid) && (NULL != _hmiStateHandler)) ? true : false;
#ifdef VARIANT_S_FTR_ENABLE_MIC_ACTIVATION
   isValid      = ((isValid) && (!getMic1Status())) ? true : false;
#endif
   isValid      = ((isValid) && (isSPMHMIStateShowHMI(_hmiStateHandler->getHmiState()))) ? true : false;
   if (isValid)
   {
      _cabinsActivityInfoDpHandler->prepareDPUpdate(*this);
#ifdef VARIANT_S_FTR_ENABLE_MAP_STREAMING
      updateMapoutConnectionDP();
#endif
      updateActiveConnectionDP();
      _cabinsActivityInfoDpHandler->finalizeDPUpdate(*this);
   }
}


void CabinConnectionHandler::updateActiveConnectionDP()
{
   if ((!_activeConnectionsInfo.empty()) && (NULL != _cabinsActivityInfoDpHandler))
   {
      for (SinkConnectionMap::iterator iter = _activeConnectionsInfo.begin(); (iter != _activeConnectionsInfo.end()); ++iter)
      {
         if ((STATE_CONNECTED == iter->second.connectionState) || (STATE_SUSPENDED == iter->second.connectionState))
         {
            uint32 regionId = REGION_INVALID;
            if ((NULL != _regionHandling) && (_regionHandling->fetchRegionIdForSinkId(iter->second.sinkId, regionId)))
            {
               CabinActivityInfoDPHolder cabinInfo;
               cabinInfo.setCabinId(regionId);
               cabinInfo.setSinkId(iter->second.sinkId);
               cabinInfo.setSourceId(iter->second.srcId);
               cabinInfo.setSubSourceId(iter->second.deviceId);
               cabinInfo.setState(iter->second.connectionState);
               cabinInfo.setActivityType(CabinSrcInfo::getActivityTypeForSource(iter->second.srcId));
               CabinSrcInfo::SrcType newSrcType = CabinSrcInfo::getSrcType(iter->second.srcId);
               if ((CabinSrcInfo::CABIN_VIDEO_SOURCE == newSrcType) && (_regionHandling->isSinkGroupCategorySame(iter->second.sinkId, SYSTEM_GROUP_CATEGORY_CMG_CABIN)))
               {
                  IMonoGlassController* monoGlassController = getControllerInstance<IMonoGlassController>();
                  if (monoGlassController != NULL)
                  {
                     stSourceInfo restoreSrcInfo;
                     if (monoGlassController->fetchAVRestoreInfo(_regionHandling->getAudioSinkIdForRegion(regionId), restoreSrcInfo))
                     {
                        cabinInfo.setLastSourceId(restoreSrcInfo.srcId);
                        cabinInfo.setLastState(restoreSrcInfo.connectionState);
                        cabinInfo.setLastSubSourceId(restoreSrcInfo.deviceId);
                     }
                  }
               }
               _cabinsActivityInfoDpHandler->updateDPData(*this, cabinInfo);
            }
         }
      }
   }
}


void CabinConnectionHandler::updateMapoutConnectionDP()
{
#ifdef VARIANT_S_FTR_ENABLE_MAP_STREAMING
   if (NULL != _cabinsActivityInfoDpHandler && (fnGetVariantType() ==  VARIANT_TYPE_NAVI))
   {
      for (RegionsMapOutInfo::const_iterator itr = _regionsMapOutInfo.begin(); (itr != _regionsMapOutInfo.end()); itr++)
      {
         if ((*itr).getState() == STATE_CONNECTED)
         {
            const SinksMapOutInfo& tSinksMapOutInfo = (*itr).getSinksMapOutInfo();
            for (SinksMapOutInfo::const_iterator sItr = tSinksMapOutInfo.begin(); (sItr != tSinksMapOutInfo.end()); sItr++)
            {
               if ((*sItr).getState() == STATE_CONNECTED)
               {
                  CabinActivityInfoDPHolder cabinInfo;
                  cabinInfo.setCabinId((*itr).getRegionId());
                  cabinInfo.setState((*sItr).getState());
                  cabinInfo.setSinkId((*sItr).getSinkId());
                  cabinInfo.setActivityType(CABIN_ACTIVITY_MAPOUT);
                  if ((NULL != _regionHandling) && (_regionHandling->isSinkGroupCategorySame((*sItr).getSinkId(), SYSTEM_GROUP_CATEGORY_CMG_CABIN)))
                  {
                     IMonoGlassController* monoGlassController = getControllerInstance<IMonoGlassController>();
                     if (monoGlassController != NULL)
                     {
                        stSourceInfo restoreSrcInfo;
                        if (monoGlassController->fetchAVRestoreInfo(_regionHandling->getAudioSinkIdForRegion((*itr).getRegionId()), restoreSrcInfo))
                        {
                           cabinInfo.setLastSourceId(restoreSrcInfo.srcId);
                           cabinInfo.setLastState(restoreSrcInfo.connectionState);
                           cabinInfo.setLastSubSourceId(restoreSrcInfo.deviceId);
                        }
                     }
                  }
                  _cabinsActivityInfoDpHandler->updateDPData(*this, cabinInfo);
               }
            }
         }
      }
   }
#endif /* VARIANT_S_FTR_ENABLE_MAP_STREAMING */
}


void CabinConnectionHandler::onSetCabinVideoMixInfo(const uint32 regionId, const uint8 videoMixIndex, const uint8 reason)
{
   CabinVideoMixInfo cabinVideoMixInfoUpdate;
   cabinVideoMixInfoUpdate.setCabinId(regionId);
   cabinVideoMixInfoUpdate.setVideoMixIndex(videoMixIndex);
   cabinVideoMixInfoUpdate.setReason(reason);
   onSetCabinVideoMixInfo(cabinVideoMixInfoUpdate);
}


void CabinConnectionHandler::onSetCabinVideoMixInfo(const CabinVideoMixInfo cabinVideoMixInfoUpdate)
{
   uint32 cabinId = cabinVideoMixInfoUpdate.getCabinId();
   ETG_TRACE_USR4(("CabinConnectionHandler: onSetCabinVideoMixInfo: RegionId: %d,  VideoMixIndex: %d, Reason:%d", cabinId, cabinVideoMixInfoUpdate.getVideoMixIndex(), cabinVideoMixInfoUpdate.getReason()));
   if ((NULL != _regionHandling) && (NULL != _cabinsVideoMixInfoDpHandler))
   {
      SystemsGroupInfo systemGroupInfo;
      bool isValid = _regionHandling->fetchSystemsGroupInfo(systemGroupInfo, SYSTEM_CATEGORY_CMG, cabinId);
      isValid = (isValid && (systemGroupInfo.size() > CABIN_MIN_SYSTEM_GROUP)) ? true : false;
      isValid = (isValid && ((cabinVideoMixInfoUpdate.getReason() == CABIN_VIDEO_MIX_CHANGE_REASON_RESTORE) ||
                             (_regionHandling->getStatus(SystemEnquiryInfo(cabinId, SYSTEM_CATEGORY_CMA, STATUS_CATEGORY_SYSTEM)) == SYSTEM_STATUS_CONNECTED))) ? true : false;
      if (isValid)
      {
         CabinsVideoMixInfo cabinsVideoMixInfo;
         cabinsVideoMixInfo.push_back(cabinVideoMixInfoUpdate);
         CabinVideoMixInfoHolder cabinVideoMixInfoHolder;
         cabinVideoMixInfoHolder.setCabinId(cabinId);
         cabinVideoMixInfoHolder.setVideoMixIndex(cabinVideoMixInfoUpdate.getVideoMixIndex());
         _cabinsVideoMixInfoDpHandler->updateCabinVideoMixDPData(cabinVideoMixInfoHolder);
         updateCabinVideoMixInfoProperty(cabinsVideoMixInfo);
      }
   }
}


void CabinConnectionHandler::evalAndUpdateMonoModeStatus()
{
   MonoGlassController* imp = getControllerInstance<MonoGlassController>();
   if ((NULL != imp) && (NULL != _hmiInfoServiceServerComponent))
   {
      const CabinsVideoMixInfo& tCabinsVideoMixInfo = _hmiInfoServiceServerComponent->getCabinsVideoMixInfo();
      for (CabinsVideoMixInfo::const_iterator itr = tCabinsVideoMixInfo.begin(); (itr != tCabinsVideoMixInfo.end()); ++itr)
      {
         bool tStatus = ((*itr).getVideoMixIndex() == CABIN_VIDEO_MIX_INDEX_NONE) ? true : false;
         imp->updateMonoModeStatus((*itr).getCabinId(), tStatus);
      }
   }
}


void CabinConnectionHandler::evalAndUpdateVideoMixController()
{
   if (NULL != _hmiInfoServiceServerComponent)
   {
      bool isVideoMixConfigured = false;
      const CabinsVideoMixInfo& tCabinsVideoMixInfo = _hmiInfoServiceServerComponent->getCabinsVideoMixInfo();
      for (CabinsVideoMixInfo::const_iterator itr = tCabinsVideoMixInfo.begin(); (itr != tCabinsVideoMixInfo.end()); ++itr)
      {
         if ((*itr).getVideoMixIndex() != CABIN_VIDEO_MIX_INDEX_NONE)
         {
            isVideoMixConfigured = true;
            break;
         }
      }
      if (isVideoMixConfigured)
      {
         VideoMixController* imp = getControllerInstance<VideoMixController>();
         if (NULL == imp)
         {
            ICabinConnectionController* videoMixController = new VideoMixController();
            videoMixController->setCabinConnectionUtility(*_cabinConnectionUtility);
            _cabinConnectionControllers.push_back(videoMixController);
         }
      }
      else
      {
         removeControllerInstance<VideoMixController>();
      }
   }
}


void CabinConnectionHandler::evalAndSyncMixGlassesConnections(const uint32 regionId, const uint8 reason, const CabinsVideoMixInfo& lastInfo)
{
   if ((NULL != _cabinConnectionUtility) && (reason == CABIN_VIDEO_MIX_CHANGE_REASON_USER))
   {
      uint8 lastMixIndex = CABIN_VIDEO_MIX_INDEX_NONE;
      for (CabinsVideoMixInfo::const_iterator itr = lastInfo.begin(); (itr != lastInfo.end()); ++itr)
      {
         if ((*itr).getCabinId() == regionId)
         {
            lastMixIndex = (*itr).getVideoMixIndex();
            break;
         }
      }
      SinkIdsInfo tCabinGlassSinksInfo = _cabinConnectionUtility->getGlassSinksInfo(regionId, SYSTEM_GROUP_CATEGORY_CMG_CABIN);
      SinkIdsInfo tMixGlassSinksInfo   = _cabinConnectionUtility->getGlassSinksInfo(regionId, SYSTEM_GROUP_CATEGORY_CMG_CABIN_MIX);
      if ((!tCabinGlassSinksInfo.empty()) && (!tMixGlassSinksInfo.empty()))
      {
         int16 cabinGlassSrcId       = -1;
         bool cabinGlassMapOutStatus = false;
         SinkConnectionMap::const_iterator itr = _activeConnectionsInfo.find(tCabinGlassSinksInfo[0]);
         if (itr != _activeConnectionsInfo.end())
         {
            cabinGlassSrcId = itr->second.srcId;
         }
#ifdef VARIANT_S_FTR_ENABLE_MAP_STREAMING
         const RegionMapOutInfo* imp = getRegionMapOutInfo(regionId);
         const SinkMapOutInfo* tInfo = (NULL != imp) ? imp->getSinkMapOutInfo(tCabinGlassSinksInfo[0]) : NULL;
         if ((NULL != tInfo) && ((tInfo->getState() == STATE_CONNECTED) || (tInfo->getState() == STATE_CONNECTING)))
         {
            cabinGlassMapOutStatus = true;
         }
         if (lastMixIndex == CABIN_VIDEO_MIX_INDEX_MAP_OUT)
         {
            if (fnGetVariantType() ==  VARIANT_TYPE_NAVI)
            {
               if (cabinGlassSrcId != -1)
               {
                  deactivateMapOut(regionId, tMixGlassSinksInfo);
                  if (_connReqQueue.empty())
                  {
                     _connReqQueue.push_back(stSourceInfo(cabinGlassSrcId, -1, STATE_UNKNOWN,  tMixGlassSinksInfo[0]));
                     executeConnectionRequestsQueue();
                  }
                  else
                  {
                     _srcReqQueue.push_back(stSourceInfo(cabinGlassSrcId, -1, STATE_UNKNOWN,  tMixGlassSinksInfo[0]));
                  }
               }
               else if (!cabinGlassMapOutStatus)
               {
                  deactivateMapOut(regionId, tMixGlassSinksInfo);
               }
               else
               {}
            }
            else
            {
               ETG_TRACE_ERR(("CabinConnectionHandler: evalAndSyncMixGlassesConnections : CABIN_VIDEO_MIX_INDEX_MAP_OUT Feature NA in NonNavi Variant"));
            }
         }
         else if (lastMixIndex != CABIN_VIDEO_MIX_INDEX_NONE) //INFO: Case for camera sources
#else
         if (lastMixIndex != CABIN_VIDEO_MIX_INDEX_NONE)
#endif /* VARIANT_S_FTR_ENABLE_MAP_STREAMING */
         {
            SinkConnectionMap::const_iterator tItr = _activeConnectionsInfo.find(tMixGlassSinksInfo[0]);
            if (tItr != _activeConnectionsInfo.end())
            {
               if ((cabinGlassSrcId == -1) && (!cabinGlassMapOutStatus))
               {
                  if (_connReqQueue.empty())
                  {
                     _connReqQueue.push_back(stSourceInfo(tItr->second.srcId, tItr->second.deviceId, STATE_DISCONNECT, tItr->second.sinkId));
                     executeConnectionRequestsQueue();
                  }
                  else
                  {
                     _srcReqQueue.push_back(stSourceInfo(tItr->second.srcId, tItr->second.deviceId, STATE_DISCONNECT, tItr->second.sinkId));
                  }
               }
               else if ((cabinGlassSrcId != -1) && (cabinGlassSrcId != tItr->second.srcId))
               {
                  if (_connReqQueue.empty())
                  {
                     _connReqQueue.push_back(stSourceInfo(cabinGlassSrcId, -1, STATE_UNKNOWN,  tItr->second.sinkId));
                     executeConnectionRequestsQueue();
                  }
                  else
                  {
                     _srcReqQueue.push_back(stSourceInfo(cabinGlassSrcId, -1, STATE_UNKNOWN,  tItr->second.sinkId));
                  }
               }
               else if (cabinGlassMapOutStatus)
               {
#ifdef VARIANT_S_FTR_ENABLE_MAP_STREAMING
                  activateMapOut(regionId, tMixGlassSinksInfo);
#endif /* VARIANT_S_FTR_ENABLE_MAP_STREAMING */
               }
               else
               {}
            }
         }
         else
         {}
      }
   }
}


void CabinConnectionHandler::updateCabinVideoMixInfoProperty(const CabinsVideoMixInfo& info)
{
   if (_hmiInfoServiceServerComponent != NULL)
   {
      CabinsVideoMixInfo tLastCabinsVideoMixInfo(_hmiInfoServiceServerComponent->getCabinsVideoMixInfo());
      _hmiInfoServiceServerComponent->setCabinVideoMixInfo(info);
      evalAndUpdateMonoModeStatus();
      evalAndUpdateVideoMixController();
      for (CabinsVideoMixInfo::const_iterator itr = info.begin(); (itr != info.end()); ++itr)
      {
         if ((*itr).getReason() != CABIN_VIDEO_MIX_CHANGE_REASON_RESTORE)
         {
            switch ((*itr).getVideoMixIndex())
            {
               case CABIN_VIDEO_MIX_INDEX_NONE:
               {
                  evalAndSyncMixGlassesConnections((*itr).getCabinId(), (*itr).getReason(), tLastCabinsVideoMixInfo);
                  break;
               }
#ifdef VARIANT_S_FTR_ENABLE_MAP_STREAMING
               case CABIN_VIDEO_MIX_INDEX_MAP_OUT:
               {
                  if (fnGetVariantType() ==  VARIANT_TYPE_NAVI)
                  {
                     if (NULL != _cabinConnectionUtility)
                     {
                        SinkIdsInfo tSinksInfo;
                        tSinksInfo = _cabinConnectionUtility->getGlassSinksInfo((*itr).getCabinId(), SYSTEM_GROUP_CATEGORY_CMG_CABIN_MIX);
                        if (!tSinksInfo.empty())
                        {
                           activateMapOut((*itr).getCabinId(), tSinksInfo);
                        }
                     }
                  }
                  else
                  {
                     ETG_TRACE_ERR(("CabinConnectionHandler: updateCabinVideoMixInfoProperty: Feature NA in NonNavi Variant"));
                  }
                  break;
               }
#endif /* VARIANT_S_FTR_ENABLE_MAP_STREAMING */
               default:
                  break;
            }
         }
      }
   }
}


void CabinConnectionHandler::removeSinkIdfromConnectionQueue(const int16 sinkId)
{
   for (RequestsQueueInfo::iterator tItr = _connReqQueue.begin(); (tItr != _connReqQueue.end());)
   {
      if ((*tItr).sinkId == sinkId)
      {
         tItr = _connReqQueue.erase(tItr);
      }
      else
      {
         ++tItr;
      }
   }
}


void CabinConnectionHandler::removeSinkIdfromSourceQueue(const int16 sinkId)
{
   for (RequestsQueueInfo::iterator tItr = _srcReqQueue.begin(); (tItr != _srcReqQueue.end());)
   {
      if ((*tItr).sinkId == sinkId)
      {
         tItr = _srcReqQueue.erase(tItr);
      }
      else
      {
         ++tItr;
      }
   }
}


void CabinConnectionHandler::evalAndResetRequestQueues()
{
   if (_srcReqQueue.size() >= (MAX_CONNECTION_REQUESTS - 1))
   {
      ETG_TRACE_USR4(("CabinConnectionHandler: evalAndResetRequestQueues: Request queue is reset"));
      _srcReqQueue.clear();
      _connReqQueue.clear();
   }
}


CabinConnectionHandlerInfo CabinConnectionHandler::getCabinConnectionHandlerInfo(const stSourceInfo& info)
{
   CabinConnectionHandlerInfo tInfo;
   tInfo.setSourceRequestsQueueInfo(_srcReqQueue);
   tInfo.setActiveSinksInfo(_activeConnectionsInfo);
   tInfo.setConnectionRequestsQueueInfo(_connReqQueue);
   tInfo.setSourceInfo(info);
#ifdef VARIANT_S_FTR_ENABLE_MAP_STREAMING
   RegionMapOutInfo* imp  = getMutableRegionMapOutInfo(getRegionIdForSinkId(info.sinkId));
   if (NULL != imp)
   {
      tInfo.setRegionMapOutInfo((*imp));
   }
#endif /* VARIANT_S_FTR_ENABLE_MAP_STREAMING */
   return tInfo;
}


uint32 CabinConnectionHandler::getRegionIdForSinkId(const int16 sinkId) const
{
   uint32 regionId = REGION_INVALID;
   if (NULL != _cabinConnectionUtility)
   {
      regionId = _cabinConnectionUtility->getRegionIdForSinkId(sinkId);
   }
   return regionId;
}


uint8 CabinConnectionHandler::getCabinMapOutStatus(const uint32 regionId) const
{
   uint8 mapOutStatus           = STATE_DISCONNECTED;
   const RegionMapOutInfo* imp  = getRegionMapOutInfo(regionId);
   if ((NULL != imp) && (NULL != _cabinConnectionUtility) && (NULL != _hmiInfoServiceServerComponent))
   {
      SinkIdsInfo tSupportedSinksInfo;
      uint8 groupCategory = (_hmiInfoServiceServerComponent->getCabinVideoMixIndex(regionId) == CABIN_VIDEO_MIX_INDEX_NONE) ? SYSTEM_GROUP_CATEGORY_NONE : SYSTEM_GROUP_CATEGORY_CMG_CABIN;
      tSupportedSinksInfo = _cabinConnectionUtility->getGlassSinksInfo(regionId, groupCategory);
      for (SinkIdsInfo::iterator itr = tSupportedSinksInfo.begin(); (itr != tSupportedSinksInfo.end()); ++itr)
      {
         const SinkMapOutInfo* tInfo = imp->getSinkMapOutInfo((*itr));
         if ((NULL != tInfo) && ((tInfo->getState() == STATE_CONNECTED) || (tInfo->getState() == STATE_CONNECTING)))
         {
            mapOutStatus = tInfo->getState();
            if (tInfo->getState() == STATE_CONNECTED)
            {
               break;
            }
         }
      }
      if (mapOutStatus == STATE_DISCONNECTED)
      {
         for (SinkIdsInfo::iterator itr = tSupportedSinksInfo.begin(); (itr != tSupportedSinksInfo.end()); ++itr)
         {
            const SinkMapOutInfo* tInfo = imp->getSinkMapOutInfo((*itr));
            if ((NULL != tInfo) && (tInfo->getState() == STATE_RESTORE))
            {
               mapOutStatus = tInfo->getState();
               break;
            }
         }
      }
   }
   return mapOutStatus;
}


void CabinConnectionHandler::startConnectionReqTimer()
{
   if (NULL == _connReqTimer)
   {
      _connReqTimer = new ::asf::core::Timer(*this, 4000);
   }
   if (NULL != _connReqTimer)
   {
      ETG_TRACE_USR4(("CabinConnectionHandler: startConnectionReqTimer"));
      _connReqTimer->start();
   }
}


void CabinConnectionHandler::stopConnectionReqTimer(const bool flush)
{
   if (NULL != _connReqTimer)
   {
      ETG_TRACE_USR4(("CabinConnectionHandler: stopConnectionReqTimer: Flush: %d", flush));
      _connReqTimer->stop();
      if (flush)
      {
         delete _connReqTimer;
         _connReqTimer = NULL;
      }
   }
}


void CabinConnectionHandler::startMixConnectionReqTimer()
{
   if (NULL == _mixConnReqTimer)
   {
      _mixConnReqTimer = new ::asf::core::Timer(*this, 4000);
   }
   if (NULL != _mixConnReqTimer)
   {
      ETG_TRACE_USR4(("CabinConnectionHandler: startMixConnectionReqTimer"));
      _mixConnReqTimer->start();
   }
}


void CabinConnectionHandler::stopMixConnectionReqTimer(const bool flush)
{
   if (NULL != _mixConnReqTimer)
   {
      ETG_TRACE_USR4(("CabinConnectionHandler: stopMixConnectionReqTimer: Flush: %d", flush));
      _mixConnReqTimer->stop();
      if (flush)
      {
         delete _mixConnReqTimer;
         _mixConnReqTimer = NULL;
      }
   }
}


void CabinConnectionHandler::onExpired(asf::core::Timer& timer, boost::shared_ptr<asf::core::TimerPayload> data)
{
   if ((NULL != _connReqTimer) && ((&timer) == (_connReqTimer)))
   {
      stopConnectionReqTimer(false);
      if ((!_connReqQueue.empty()) && (_connReqQueue.front().connectionState == STATE_CONNECTING))
      {
         (void)_connReqQueue.erase(_connReqQueue.begin());
         executeConnectionRequestsQueue();
         executeSourceRequestsQueue();
      }
   }
   else if ((NULL != _mixConnReqTimer) && ((&timer) == (_mixConnReqTimer)))
   {
      stopMixConnectionReqTimer(false);
      if ((!_mixSrcConnReqQueue.empty()) && (_mixSrcConnReqQueue.front().connectionState == STATE_CONNECTING))
      {
         (void)_mixSrcConnReqQueue.erase(_mixSrcConnReqQueue.begin());
         executeMixSourceConnectionRequestsQueue();
      }
   }
   else
   {
   }
}


#ifdef VARIANT_S_FTR_ENABLE_MIC_ACTIVATION
void CabinConnectionHandler::onHmiStateUpdate(const uint8 currentHmiState)
{
   updateMicSourceRestoration(RegionsIdInfo());
}


#endif
} //namespace Core
} //namespace App
